/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include "vmelilo.h"
#include "mvmebug.h"

#include "mvmeipl.data"
#include "mvmeloader.data"

/*--------------------------------------------------------------------------*/
/* IPL fragment list element
 */
typedef struct {
	u_short		count;			/* number of sectors						*/
	u_long		block;			/* sector address or zero for hole			*/

} IPLFRAG;

/*--------------------------------------------------------------------------*/

void
mvme_init_boot_block
(	void			*buf
)
{	BOOTBLOCK		*bb = (BOOTBLOCK *) buf;
	const FILEMAP	*map;
	IPLFRAG			*frp;
	int				blksper;
	int				i;
	u_long			maxfrags;
	u_long			n1;
	u_long			n2;

	/* zero initialise boot block */
	memset(bb, 0, sizeof(BOOTBLOCK));

	/* set boot block id */
	strcpy(bb->boot_id, BOOT_BLOCK_ID);

	/* load and run IPL code at address 0x20000, stack at 0x21000 */
	bb->sp               = 0x21000;
	bb->pc               = 0x22;	/* offset 0x22 in boot sector		*/
	bb->boot_start_block = 0;		/* boot sector is also bootfile		*/
	bb->boot_block_count = 2;		/* 1 sector = 2 x 256 byte blocks	*/
	bb->boot_load_addr   = 0x20000;	/* boot sector actual load address	*/
	bb->cfg_block        = 0;		/* no configuration data			*/

	/* get maximum number of fragments allowed in boot block */
	maxfrags = (sizeof(bb->frag_list) - 2) / sizeof(IPLFRAG);

	if (MAP_NUMFRAGS(loader_map) > maxfrags)
	{
		n1 = MAP_NUMFRAGS(loader_map) - maxfrags;

		die("Boot loader file is too fragmented for boot block "
			"(%ld fragment%s too many)\n",
					n1, (n1 == 1) ? "" : "s");
	}

	if (f_verbose)
	{
		n1 = MAP_NUMFRAGS(loader_map);
		n2 = maxfrags - MAP_NUMFRAGS(loader_map);

		printf(	"Boot loader file has %ld fragment%s, "
				"enough space for %lu more\n",
				  	n1, (n1 == 1) ? "" : "s", n2);
	}

	/* check that IPL code fits in bootblock */
	if (sizeof(mvmeipl_code) > sizeof(bb->boot_code))
	{
		die("Too much IPL code to fit in boot block\n");
	}

	/* put IPL code into boot block */
	memcpy(bb->boot_code, mvmeipl_code, sizeof(mvmeipl_code));

	/* get pointer to fragment list for the IPL */
	frp = (IPLFRAG *) bb->frag_list;

	/* get pointer to first file fragment of the loader */
	map = MAP_FIRSTFRAG(loader_map);

	/* get number of 256 byte MVMEBug blocks per sector */
	blksper = SECTOR_SIZE / 256;

	/* put file map of loader into boot block */
	i = MAP_NUMFRAGS(loader_map);
	while (i--)
	{
		frp->count = map->length * blksper;
		frp->block = map->offset * blksper;
		frp++;
		map++;
	}
}

/*--------------------------------------------------------------------------*/

int
mvme_valid_boot_block
(	void		*buf
)
{	BOOTBLOCK	*bb = (BOOTBLOCK *) buf;
	
	return (strcmp(bb->boot_id, BOOT_BLOCK_ID) == 0);
}

/*--------------------------------------------------------------------------*/

SUBARCH mvme_subarch = {
	"MVMEBug",
	mvmeloader_code,
	sizeof(mvmeloader_code),
	sizeof(BOOTBLOCK),
	mvme_init_boot_block,
	mvme_valid_boot_block
};

/*-----------------------------< end of file >------------------------------*/
