/*
   logcmd.c

   $Id: logcmd.c,v 1.1.2.2 1997/09/30 17:47:00 andy Exp $

   A version of script(1) which takes arguments specifying a program to run,
   for use by the dpkg-mountable method to log the input and output of a dpkg
   run.

   Derived from the BSD sources for script, the copyright for which is
   included below.

   Changes are:
     * Returns the exit value of the child, rather than always returning zero.
     * The script file is given via the -f option, rather than directly.
     * Any command can be passed on the commandline, along with its arguments,
       rather than just using the contents of the SHELL variable.
     * New -q option, doesn't add any extra text to the logfiles or the screen
       display.

   */

/*
 * Copyright (c) 1980, 1992, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE. */

#define _BSD_SOURCE

#ifndef lint
static char copyright[] =
"@(#) Copyright (c) 1980, 1992, 1993\n\
	The Regents of the University of California.  All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char sccsid[] = "@(#)script.c	8.1 (Berkeley) 6/6/93";
#endif /* not lint */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include <errno.h>
#include <fcntl.h>
#include <paths.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>

#ifdef __linux__
#define SECSPERMIN		60
#define __dead
#endif /* __linux__ */

FILE	*fscript;
int	master, slave;
int	child, subchild;
int	outcc;
char	*fname;
int	qflg;

struct	termios tt;

__dead	void done __P((int status));
	void dooutput __P((void));
	void doshell __P((int argc, char *argv[]));
	void err __P((const char *, ...));
	void fail __P((void));
	void finish __P((int));
	void scriptflush __P((int));

int
main(argc, argv)
	int argc;
	char *argv[];
{
	register int cc;
	struct termios rtt;
	struct winsize win;
	int aflg, ch;
	char ibuf[BUFSIZ];

	aflg = qflg = 0;
	fname="commandlog";
	while ((ch = getopt(argc, argv, "aqf:")) != EOF)
		switch(ch) {
		case 'a':
			aflg = 1;
			break;
		case 'q':
			qflg = 1;
			break;
		case 'f':
			fname=optarg;
			break;
		case '?':
		default:
			(void)fprintf(stderr, "usage: script [-aq] [-f file] command [arg ...]\n");
			exit(1);
		}
	argc -= optind;
	argv += optind;

	if ((fscript = fopen(fname, aflg ? "a" : "w")) == NULL)
		err("%s: %s", fname, strerror(errno));

	(void)tcgetattr(STDIN_FILENO, &tt);
	(void)ioctl(STDIN_FILENO, TIOCGWINSZ, &win);
	if (openpty(&master, &slave, NULL, &tt, &win) == -1)
		err("openpty: %s", strerror(errno));

	if (!qflg) (void)printf("Logging %s to file %s\n", argv[0], fname);
	rtt = tt;
	cfmakeraw(&rtt);
	rtt.c_lflag &= ~ECHO;
	(void)tcsetattr(STDIN_FILENO, TCSAFLUSH, &rtt);

	(void)signal(SIGCHLD, finish);
	child = fork();
	if (child < 0) {
		perror("fork");
		fail();
	}
	if (child == 0) {
		subchild = child = fork();
		if (child < 0) {
			perror("fork");
			fail();
		}
		if (child)
		  dooutput();
		else
		  doshell(argc, argv);
	}

	(void)fclose(fscript);
	while ((cc = read(STDIN_FILENO, ibuf, BUFSIZ)) > 0)
		(void)write(master, ibuf, cc);
	done(0);
}

void
finish(signo)
	int signo;
{
	register int die, pid, exst;
	union wait status;
  
	die = 0;
	while ((pid = wait3((int *)&status, WNOHANG, 0)) > 0)
		if (pid == child) {
			die = 1;
			exst = WIFSIGNALED(status)?WTERMSIG(status)+128:WEXITSTATUS(status);
		}
	if (die)
		done(exst);
}

void
dooutput()
{
	struct itimerval value;
	register int cc;
	time_t tvec;
	char obuf[BUFSIZ];

	(void)close(STDIN_FILENO);
	tvec = time(NULL);
	if (!qflg) (void)fprintf(fscript, "Log started on %s", ctime(&tvec));

	(void)signal(SIGALRM, scriptflush);
	value.it_interval.tv_sec = SECSPERMIN / 2;
	value.it_interval.tv_usec = 0;
	value.it_value = value.it_interval;
	(void)setitimer(ITIMER_REAL, &value, NULL);
	for (;;) {
		cc = read(master, obuf, sizeof (obuf));
		if (cc <= 0)
			break;
		(void)write(1, obuf, cc);
		(void)fwrite(obuf, 1, cc, fscript);
		outcc += cc;
	}
	done(0);
}

void
scriptflush(signo)
	int signo;
{
	if (outcc) {
		(void)fflush(fscript);
		outcc = 0;
	}
}

void
doshell(int argc, char *argv[])
{
  char *name,*lsl,*c;

  name=argv[0];
  for (c=lsl=name;*c;c++)
    if (*c=='/') lsl=c+1;
  argv[0]=strdup(lsl);
  (void)close(master);
  (void)fclose(fscript);
  login_tty(slave);
  execvp(name, argv);
  perror(name);
  fail();
}

void
fail()
{

	(void)kill(0, SIGTERM);
	done(0);
}

void
done(int stat)
{
	time_t tvec;
	register int pid, exst;
	union wait status;
	
	exst=stat;
	
	while ((pid = wait3((int *)&status, WNOHANG, 0)) > 0)
		if (pid == child) {
			exst=status.w_retcode;
		}
	
	if (subchild) {
		tvec = time(NULL);
		if (!qflg) (void)fprintf(fscript,"\nLog done on %s", ctime(&tvec));
		(void)fclose(fscript);
		(void)close(master);
	} else {
		(void)tcsetattr(STDIN_FILENO, TCSAFLUSH, &tt);
		/* (void)printf("Script done, output file is %s\n", fname); */
	}
	exit(exst);
}

#if __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

void
#if __STDC__
err(const char *fmt, ...)
#else
err(fmt, va_alist)
	char *fmt;
	va_dcl
#endif
{
	va_list ap;
#if __STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	(void)fprintf(stderr, "script: ");
	(void)vfprintf(stderr, fmt, ap);
	va_end(ap);
	(void)fprintf(stderr, "\n");
	exit(1);
	/* NOTREACHED */
}
