# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: blt2d.tcl,v 2.2 1999/08/28 10:55:13 jfontain Exp $}

class blt2DViewer {

    # use pie colors for consistency
    set ::blt2DViewer::(colors) {#7FFFFF #7FFF7F #FF7F7F #FFFF7F #7F7FFF #FFBF00 #BFBFBF #FF7FFF #FFFFFF}

    proc blt2DViewer {this path} viewer {} {
        $path configure -cursor {} -plotpadx 2 -plotpady 2   ;# use minimum padding for extreme values, flat zero line to be visible
        $path yaxis configure -tickshadow {} -title {} -tickfont $font::(smallNormal)
        $path legend configure -borderwidth 1 -font $font::(mediumNormal) -activebackground white

        viewer::setupDropSite $this $path                                                            ;# allow dropping of data cells

        set ($this,elements) {}                                                          ;# initialize list of data line identifiers
        set ($this,colorIndex) 0
        set ($this,path) $path
    }

    proc ~blt2DViewer {this} {
        if {[info exists ($this,drag)]} {
            delete $($this,drag)
        }
        eval delete $($this,elements)                                                                    ;# delete existing elements
        if {[info exists ($this,selector)]} {
            delete $($this,selector)
        }
    }

    proc supportedTypes {this} {
        return {integer real}
    }

    proc dragData {this format} {
        set selectedElements [selector::selected $($this,selector)]
        switch $format {
            OBJECTS {
                if {[llength $selectedElements]>0} {
                    return $selectedElements                                            ;# return selected elements if there are any
                } elseif {[llength $($this,elements)]==0} {
                    return $this                                                   ;# return graph itself if it contains no elements
                } else {
                    return {}                                                                            ;# return nothing otherwise
                }
            }
            DATACELLS {
                return [cellsFromElements $this $selectedElements]
            }
        }
    }

    proc validateDrag {this x y} {
        if {[llength $($this,elements)]==0} {
            return 1                                                                                   ;# allow drag of empty viewer
        }
        # allow dragging if only from a selected cell
        return [expr {[lsearch -exact [selector::selected $($this,selector)] [$($this,path) legend get @$x,$y]]>=0}]
    }

    proc monitorCell {this array row column} {
        viewer::registerTrace $this $array
        set cell ${array}($row,$column)
        if {[lsearch -exact [cellsFromElements $this $($this,elements)] $cell]>=0} return                  ;# already charted, abort
        set element [newElement $this $($this,path)\
            -label [viewer::label $array $row $column] -color [lindex $(colors) $($this,colorIndex)]\
        ]
        # keep track of element existence
        switched::configure $element -deletecommand "blt2DViewer::deletedElement $this $array $element"
        set ($this,colorIndex) [expr {($($this,colorIndex)+1)%[llength $(colors)]}]                         ;# circle through colors
        lappend ($this,elements) $element                                                                  ;# register new data line
        set ($this,cell,$element) $cell
        if {[info exists ($this,selector)]} {                                       ;# selector may not exist if dragging disallowed
            selector::add $($this,selector) $element
        }
    }

    proc cells {this} {
        return [cellsFromElements $this $($this,elements)]
    }

    proc deletedElement {this array element} {
        viewer::unregisterTrace $this $array                                          ;# trace may no longer be needed on this array
        ldelete ($this,elements) $element
        if {[info exists ($this,selector)]} {                                       ;# selector may not exist if dragging disallowed
            selector::remove $($this,selector) $element
        }
        unset ($this,cell,$element)
    }

    proc update {this array args} {                                                               ;# update display using cells data
        updateTimeDisplay $this [set seconds [clock seconds]]
        foreach element $($this,elements) {
            set cell $($this,cell,$element)
            if {[string first $array $cell]<0} continue                                  ;# check that cell belongs to updated array
            if {[info exists $cell]} {
                updateElement $this $element $seconds [set $cell]
            } else {
                updateElement $this $element $seconds {}                                                    ;# data no longer exists
            }
        }
    }

    virtual proc newElement {this path args}                                       ;# let derived class create an element of its own

    virtual proc updateElement {this element seconds value}      ;# let derived class (such as graph, bar chart, ...) update element

    virtual proc updateTimeDisplay {this seconds} {}        ;# eventually let derived class (such as graph) update axis, for example

    proc cellsFromElements {this elements} {
        set cells {}
        foreach element $elements {
            lappend cells $($this,cell,$element)
        }
        return $cells
    }

    proc setElementsState {this elements select} {
        if {$select} {
            set action activate
        } else {
            set action deactivate
        }
        set path $($this,path)
        foreach element $elements {
            $path legend $action $element
        }
    }

    proc setSelection {this x y} {
        if {[string length [set element [$($this,path) legend get @$x,$y]]]>0} {                                      ;# in a legend
            selector::select $($this,selector) $element
        }
    }

    proc toggleSelection {this x y} {
        if {[string length [set element [$($this,path) legend get @$x,$y]]]>0} {                                      ;# in a legend
            selector::toggle $($this,selector) $element
        }
    }

    proc extendSelection {this x y} {
        if {[string length [set element [$($this,path) legend get @$x,$y]]]>0} {                                      ;# in a legend
            selector::extend $($this,selector) $element
        }
    }

    proc allowDrag {this} {
        set path $($this,path)

        set ($this,drag) [new dragSite -path $path -validcommand "blt2DViewer::validateDrag $this"]
        dragSite::provide $($this,drag) OBJECTS "blt2DViewer::dragData $this"                             ;# provide list of objects
        dragSite::provide $($this,drag) DATACELLS "blt2DViewer::dragData $this"

        set ($this,selector) [new objectSelector -selectcommand "blt2DViewer::setElementsState $this"]
        bind $path <ButtonRelease-1> "blt2DViewer::setSelection $this %x %y"
        bind $path <Control-ButtonRelease-1> "blt2DViewer::toggleSelection $this %x %y"
        bind $path <Shift-ButtonRelease-1> "blt2DViewer::extendSelection $this %x %y"
    }

}
