/* a line in /etc/route.conf have following entries:
   dest what mask iface

   strategy: the following pseudo code extracted from /etc/rc.d/route:
   
   IF   what == "0.0.0.0" || what == "*"
   THEN IF   netmask == "" || iface == ""
	THEN ERROR
        ELSE IF   mask == "255.255.255.255"
	     THEN ROUTEADD -host dest dev iface
	     ELSE ROUTEADD -net  dest netmask mask dev iface

   ELSE IF   mask == "0.0.0.0" || mask == ""
        THEN ROUTEADD -net dest gw what <dev iface>
	ELSIF mask == "255.255.255.255"
	THEN IF   dest == what
	     THEN ROUTEADD -host dest <dev iface>
	     ELSE ROUTEADD -host dest gw what <dev iface>
	ELSE ROUTEADD -net dest netmask mask gw what <dev iface>

	
   For reading, host routes leading to alias-interfaces are silently ignored.
   For writing, ipalias-routes are read and appended to the config file.
   The ipalias-stuff takes care of adding/deleting these entries.
 */

#include <string.h>
#include <ctype.h>
#include <misc.h>
#include "suse.h"
#include <netconf.h>
#include <subsys.h>

/*
static HELP_FILE help_routes ("suse","static-routes");
*/
static CONFIG_FILE f_staticroutes ("/etc/route.conf"
	,help_nil
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,"root","root",0644
	,subsys_netclient);

#define DEBUG_ROUTE_SAVE 0

#if DEBUG_ROUTE_SAVE
static CONFIG_FILE f_savestaticroutes ("/etc/route.conf.new"
	,help_nil
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,"root","root",0644
	,subsys_netclient);
#endif

int routes_load_ipalias(ROUTES &rt,bool readAliasRoute)
{
	int ret = 0;
	int lno = 0;
	bool defaultgtw = false;
	bool isAliasRoute = false;
	FILE *fin = f_staticroutes.fopen ("r");
	if (fin != NULL){
		char buf[1000];
		while (fgets_strip(buf,sizeof(buf)-1,fin,&lno)!=NULL){
		        if(buf[0] != '\0' && buf[0] != ' '){
			        isAliasRoute = false;
			        char buf2[1000];
				str_exptab ((const char *) buf,1,buf2); //remove tabs
			        char word[4][200];
				strcpy(word[2],"");
				strcpy(word[3],"");
				if(sscanf (buf2,"%s %s %s %s",word[0],word[1],word[2],word[3]) < 2) {
				       ret = -1;
				       break;
				}
				const char *dest  = word[0];
				const char *what  = word[1];
				const char *mask  = word[2];
				const char *iface = word[3];
				const char *flags = NULL;
				const char *gtw   = NULL;
				
				if (strcmp(what,"0.0.0.0") == 0 || strcmp(what,"*") == 0){
				        if (strcmp(mask,"") == 0 || strcmp(iface,"") == 0){ 
					        ret = -1;
						break;
					}else {
					        if (strcmp(mask,"255.255.255.255") == 0){
						        if(strstr(iface, ":") != NULL) {
							        isAliasRoute = true;
							}
						        flags = "UH"; /* ROUTEADD -host dest dev iface */
							mask  = "";
						}else {
						        flags = "U";  /* ROUTEADD -net  dest netmask mask dev iface */
							gtw = "*";
						}
					}
				}else {
				        if (strcmp(mask,"0.0.0.0") == 0 || strcmp(mask,"") == 0){
					        /* check, if there does exist a default route */
					        if (strcmp(dest,"default") == 0){
						        defaultgtw = true;
						}
					        flags = "UG"; /* ROUTEADD -net dest gw what <dev iface> */
						gtw   = what;
					}else if (strcmp(mask,"255.255.255.255") == 0){
					        mask  = "";
					        if(strcmp(dest,what) == 0) {
						        flags = "UH"; /* ROUTEADD -host dest <dev iface> */
						}else {
						        flags = "UGH";/* ROUTEADD -host dest gw what <dev iface> */
							gtw   = what;
						}
					}else {
					        flags = "UG"; /* ROUTEADD -net dest netmask mask gw what <dev iface> */
						gtw = what;
					}
				}

				if ((readAliasRoute && isAliasRoute) ||
				    (!readAliasRoute && !isAliasRoute)){
				  rt.add (new ROUTE (dest,((gtw == NULL) ? "" : gtw), mask,flags,iface));
				}
			}
		}
		fclose (fin);
	}
	
	if(defaultgtw == false) {
	        /* we did not find a default gateway in /etc/rc.config, so we kindly ask linuxconf... */
	        const char *defgtw = linuxconf_getval ("routing","gateway");
		if (defgtw != NULL && defgtw[0] != '\0'){
		        rt.add (new ROUTE ("default",defgtw,"","UG",""));
		}
	}

	if(ret == -1) {
	  ; // print some error information! line number is written in lno
	}
	return ret;
}

int routes_load (ROUTES &rt)
{
        return routes_load_ipalias(rt,false);
}


int routes_save_append(
	SSTRING &defroute,
	ROUTES &rts,
	bool append)
{
	int ret = -1;
	char device[20];
	const char *mode = append ? "a" : "w";
#if DEBUG_ROUTE_SAVE
	FILE *fout = f_savestaticroutes.fopen (mode);
#else
	FILE *fout = f_staticroutes.fopen (mode);
#endif
	if (fout != NULL){
	        for (int i=0; i<rts.getnb(); i++){
		        ROUTE *r = rts.getitem(i);
			netconf_finddevfromhost (r->getgateway(),device);
			
			const char *mask = r->getmask();
			if (r->dst_is_host()){
			        mask = "255.255.255.255";
			}else if (mask[0] == '\0'){
			        char defmask[20];
			        device_setstdnetmask(r->getdst(),defmask);
				mask = defmask;
			}

			const char *gtw = NULL;
			if(r->isdevice()) {
			        gtw = "0.0.0.0";
			} else{
			        gtw = r->getgateway();
			}
			  
			if(strcmp(gtw,"") == 0) {
			        gtw = "0.0.0.0";
			}
			fprintf (fout,"%-15.15s %-15.15s %-15.15s %s\n"
				 ,r->getdst()
				 ,gtw
				 ,mask
				 ,r->getiface());
		}

		const char *defr = defroute.get();
		if(strcmp(defr,"") != 0) {
		        fprintf (fout,"%-15.15s %s\n","default",defr);
			linuxconf_removeall ("routing","gateway");
			linuxconf_removeall ("routing","gatewaydev");
		}
		ret = fclose (fout);
	}
	return ret;
}

#define min(x,y) (x<y ? x : y)
int routes_save(
	SSTRING &defroute,
	ROUTES &rts)
{
        // we normally overwrite the routing file
        SSTRING myDef;
	myDef.setfrom("");
	ROUTES myRts;
        int ret = routes_load_ipalias(myRts,true);
	int ret2 = routes_save_append(myDef,myRts,false);
        int ret3 = routes_save_append(defroute,rts,true);
	return min(ret,min(ret2,ret3));
}


















