/* Marcelo Tosatti <marcelo@conectiva.com.br> :
   unset LANG and LC_ALL before running chkconfig to avoid parse
   problems with traslations */

#include <string.h>
#include <stdlib.h>
#include "redhat.h"
#include <popen.h>
#include <misc.h>
#include <netconf.h>
#include "redhat.m"
#include "../module_apis/status_api.h"
#include "../module_apis/package_api.h"
#include "../module_apis/servicectl_apidef.h"
#include <userconf.h>

static HELP_FILE help_sysv ("redhat","sysv");

class SERVICE_SYSV_REDHAT: public SERVICE{
	/*~PROTOBEG~ SERVICE_SYSV_REDHAT */
public:
	SERVICE_SYSV_REDHAT (const char *_name,
		 const char *_desc,
		 int _state);
	int control (SERVICE_OPER oper);
	int edit (void);
	const char *getconfstatus (void);
	const char *getrunstatus (void);
	void showstatus (void);
	/*~PROTOEND~ SERVICE_SYSV_REDHAT */
};

PUBLIC SERVICE_SYSV_REDHAT::SERVICE_SYSV_REDHAT(
	const char *_name,
	const char *_desc,
	int _state)
	: SERVICE (_name,_desc,_state,OWNER_DISTRIB)
{
}
static const char initd[] = "/etc/rc.d/init.d";
static const char redhat_sysv[]="redhat/sysv.c";
PUBLIC int SERVICE_SYSV_REDHAT::control (SERVICE_OPER oper)
{
	int ret = -1;
	if (perm_rootaccess(MSG_U(P_CTRLSERV,"control service activity"))){
		char startcmd[100];
		snprintf (startcmd,sizeof(startcmd)-1,"%s/%s start",initd,name.get());
		char stopcmd[100];
		snprintf (stopcmd,sizeof(stopcmd)-1,"%s/%s stop",initd,name.get());
		if (oper == SERVICE_START){
			ret = netconf_system (15,startcmd);
		}else if (oper == SERVICE_STOP){
			ret = netconf_system (15,stopcmd);
		}else if (oper == SERVICE_RESTART){
			// We do a stop/start as restart is not always reliable
			netconf_system (15,stopcmd);
			ret = netconf_system (15,startcmd);
		}
	}
	return ret;
}

/*
	Return a string indicating if the service is running
*/
PUBLIC const char *SERVICE_SYSV_REDHAT::getrunstatus()
{
	char lockfile[PATH_MAX];
	sprintf (lockfile,"/var/lock/subsys/%s",name.get());
	return file_exist(lockfile) ? MSG_U(I_RUNNING,"Running") : "";
}
/*
	Return a string indicating if the service is enabled
*/
PUBLIC const char *SERVICE_SYSV_REDHAT::getconfstatus()
{
	return state == 0 ? MSG_U(I_SYSVENABLED,"Automatic")
			: MSG_U(I_SYSVDISABLED,"Manual");
}

PUBLIC void SERVICE_SYSV_REDHAT::showstatus()
{
	STATUS_API *api = status_api_init(redhat_sysv);
	if (api != NULL){
		char cmd[100],title[100];
		snprintf (cmd,sizeof(cmd)-1,"%s/%s",initd,name.get());
		snprintf (title,sizeof(title)-1,MSG_U(T_SERVNAME,"Service %s"),name.get());
		api->showcommand (title,cmd,"status");
		status_api_end(api);
	}
}

PUBLIC int SERVICE_SYSV_REDHAT::edit()
{
	int ret = 0;
	int nof = 0;
	char path[PATH_MAX];
	snprintf (path,sizeof(path)-1,"%s/%s",initd,name.get());
	SSTRINGS desc;
	{
		FILE *fin = fopen (path,"r");
		if (fin != NULL){
			char buf[1000];
			while (fgets(buf,sizeof(buf)-1,fin)!=NULL){
				if (buf[0] != '#') break;
				strip_end (buf);
				// Check for a continuation character 
				int last = strlen (buf) - 1;
				bool cont = false;
				if (buf[last] == '\\'){
					buf[last] = '\0';
					cont = true;
				}
				if (strncmp(buf,"# description:",14)==0){
					desc.add (new SSTRING(str_skip(buf+14)));
					if (!cont) break;
				}else if (desc.getnb() > 0){
					desc.add (new SSTRING(str_skip(buf+1)));
					if (!cont) break;
				}
			}
			fclose (fin);
		}
	}
	char enabled = state==0;
	PACKAGE_API *pkg_api = package_api_init ("redhat/sysv.cc");
	SSTRING pkg;
	PACKAGE_VERSION ver;
	bool pkgfound = false;
	if (pkg_api != NULL){
		// Get the package from the sysv script
		pkgfound = pkg_api->path2pkg (pkg_api,path,pkg,ver) != -1;
	}
	DIALOG dia;
	dia.newf_chk (MSG_U(F_CONFIGURED,"Startup")
		,enabled,MSG_R(I_SYSVENABLED));
	SSTRING statusstr(getrunstatus());
	int status_field = dia.getnb();
	dia.newf_str (MSG_U(F_STATUS,"Status"),statusstr);
	// dia.set_lastreadonly();
	for (int i=0; i<desc.getnb(); i++){
		dia.newf_info (i==0 ? MSG_U(F_DESCRIPTION,"Description") : ""
			,desc.getitem(i)->get());
	}
	if (pkgfound){
		dia.newf_info (MSG_U(F_PACKAGE,"Package name"),pkg.get());
		dia.newf_info (MSG_U(F_PKGVER,"Package version"),ver.str);
	}else if (pkg_api == NULL){
		// Tell the user that managerpm is not installed
		dia.newf_info (MSG_R(F_PACKAGE)
			,MSG_U(I_NOMANAGERPM,"(No package manager available)"));
	}else{
		dia.newf_info (MSG_R(F_PACKAGE),"");
	}
	dia.setbutinfo (MENU_USR1,MSG_U(B_START,"Start"),MSG_R(B_START));
	dia.setbutinfo (MENU_USR2,MSG_U(B_STOP,"Stop"),MSG_R(B_STOP));
	dia.setbutinfo (MENU_USR3,MSG_U(B_RESTART,"Restart"),MSG_R(B_RESTART));
	int butopt = MENUBUT_USR1|MENUBUT_USR2|MENUBUT_USR3;
	if (status_api_available(redhat_sysv)){
		dia.setbutinfo (MENU_USR4,MSG_U(B_STATUS,"Status"),MSG_R(B_STATUS));
		butopt |= MENUBUT_USR4;
	}
	if (pkg_api != NULL && pkgfound){
		dia.setbutinfo (MENU_USR5,MSG_U(B_PKGINFO,"Pkg info"),MSG_R(B_PKGINFO));
		butopt |= MENUBUT_USR5;
	}
	char title[100];
	snprintf (title,sizeof(title)-1,MSG_U(T_ONESERVICE,"Service %s")
		,name.get());
	while (1){
		MENU_STATUS code = dia.edit (title
			,MSG_U(I_ONESERVICE,"You can enable/disable a service\n"
				"or you can start and stop it manually")
			,help_sysv,nof
			,MENUBUT_CANCEL|MENUBUT_ACCEPT|butopt);
		if (code == MENU_ESCAPE || code == MENU_CANCEL){
			break;
		}else if (code == MENU_USR1
			|| code == MENU_USR2
			|| code == MENU_USR3){
			control (code == MENU_USR1 ? SERVICE_START
				: (code == MENU_USR2 ? SERVICE_STOP : SERVICE_RESTART));
			statusstr.setfrom (getrunstatus());
			dia.reload (status_field);
			ret = 1;
		}else if (code == MENU_USR4){
			showstatus();
		}else if (code == MENU_USR5){
			pkg_api->showinfo (pkg_api,pkg.get());
		}else if (code == MENU_ACCEPT){
			state = enabled ? 0 : 1;
			if (state != previous){
				char tmp[100];
				sprintf (tmp,"%s %s",name.get(),state ? "off" : "on");
				ret |= netconf_system_if ("chkconfig",tmp);
			}
			break;
		}
	}
	package_api_end (pkg_api);
	return ret || state != previous;
}


static int sysv_collect (SERVICES &tb)
{
	int ret = -1;
	bool lc = false, langb = false;
	char *lc_all = NULL, *lang = NULL;

	if(getenv("LC_ALL") != NULL)  {
		lc = true;
		lc_all = strdup(getenv("LC_ALL"));
	}

	if(getenv("LANG") != NULL) {
		langb = true;
		lang = strdup(getenv("LANG"));
	}

	setenv("LC_ALL","",1);
	setenv("LANG","",1);

	POPEN pop ("chkconfig","--list");

	if(langb == true) {
		setenv("LANG",lang,1);
		free(lang);
	}

	if(lc == true) {
		setenv("LC_ALL",lc_all,1);
		free(lc_all);
	}

	if (pop.isok()){
		ret = 0;
		while (pop.wait(10)!=-1){
			char buf[1000];
			while (pop.readout(buf,sizeof(buf)-1)!=-1){
				strip_end (buf);
				if (buf[0] != '\0'){
					ret++;
					char name[100];
					int state = 2;
					const char *pt = str_copyword(name,buf,sizeof(name)-1);
					while (1){
						char word[100];
						pt = str_copyword (word,pt,sizeof(word)-1);
						if (word[0] == '\0') break;
						int level = atoi(word);
						if (level != 2 && strstr(word,":on")!=NULL){
							state = 0;
							break;
						}
					}
					tb.add (new SERVICE_SYSV_REDHAT (name,"",state));
				}
			}
		}
	}
	return ret;
}

void *sysv_api_get ()
{
	SERVICECTL_API *api = new SERVICECTL_API;
	api->collect = sysv_collect;
	return api;
}

void sysv_api_release (void *obj)
{
	SERVICECTL_API *api = (SERVICECTL_API*)obj;
	delete api;
}

