/*

    GPS - Graphical Process Statistics
    Copyright (C) 1999 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/* The CPU and memory polling were deduced from the kernel sources and
   from wmmon.app's source code (although gPS contains no code from
   wmmon).

   wmmon.app was developed by Martijn Pieterse and Antoine Nulle
                                (http://windowmaker.mezaway.org)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "diefast.h"
#include "history.h"

#include "hred.xpm"
#include "hgreen.xpm"

struct historic { /* circular buffer */
  float *values;
  long length;
  long cursor; /* "now" position */
} cpuh,memh,swph;

GtkWidget *hdlg=NULL;
GtkWidget *G1,*G2,*H1,*H2; /* gauges and history */
GtkWidget *lg1,*ml[12];
GdkPixmap *g1canvas=NULL,*h1canvas=NULL,
  *g2canvas=NULL,*h2canvas=NULL;

double cpu_usage=0.0;
double mem_usage=0.0;
double swp_usage=0.0;
long memtot=1,memused=1,swptot=1,swpused=1;

gint to_history=-1;

void pop_history(GtkWidget *widget, gpointer data) {
  GtkWidget *v1,*h1,*hs,*dis,*cput;
  char title[512],bu[256];

  if (hdlg!=NULL)
    return;

  /* I shall make the history larger later, but if you have enough
     screen to make a history larger than 1024 pixels you're an
     exception. */

  cpuh.values=(float *)g_malloc0(sizeof(float)*1024);
  cpuh.length=1024L;
  cpuh.cursor=0L;

  memh.values=(float *)g_malloc0(sizeof(float)*1024);
  memh.length=1024L;
  memh.cursor=0L;

  swph.values=(float *)g_malloc0(sizeof(float)*1024);
  swph.length=1024L;
  swph.cursor=0L;

  hdlg=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_realize(hdlg);
  gtk_window_set_policy(GTK_WINDOW(hdlg),TRUE,TRUE,TRUE);
  gethostname(bu,256);
  sprintf(title,"CPU and Memory (%s)",bu);
  gtk_window_set_title(GTK_WINDOW(hdlg),title);
  gtk_container_set_border_width(GTK_CONTAINER(hdlg),4);
  
  v1=gtk_vbox_new(FALSE,2);
  gtk_container_add(GTK_CONTAINER(hdlg),v1);

  cput=gtk_table_new(6,3,FALSE);
  gtk_box_pack_start(GTK_BOX(v1),cput,TRUE,TRUE,4);

  mk_cput(cput);

  hs=gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(v1),hs,FALSE,FALSE,4);

  h1=gtk_hbox_new(FALSE,2);
  gtk_box_pack_start(GTK_BOX(v1),h1,FALSE,TRUE,2);

  dis=gtk_button_new_with_label("   Close   ");
  gtk_box_pack_end(GTK_BOX(h1),dis,FALSE,FALSE,2);

  gtk_signal_connect(GTK_OBJECT(dis),"clicked",
		     GTK_SIGNAL_FUNC(close_history),NULL);
  gtk_signal_connect (GTK_OBJECT (hdlg), "destroy",
		      GTK_SIGNAL_FUNC (destroy_history), NULL);

  gtk_widget_show(v1);
  gtk_widget_show(h1);
  gtk_widget_show(hs);
  gtk_widget_show(dis);
  gtk_widget_show(cput);
  gtk_widget_show(hdlg);

  to_history=gtk_timeout_add(250,refresh_history,NULL);
}

void close_history(GtkWidget *w, gpointer data) {
  gtk_widget_destroy(hdlg);
  hdlg=NULL;
}

void destroy_history(GtkWidget *w,gpointer data) {
  if (to_history!=-1)
    gtk_timeout_remove(to_history);
  to_history=-1;
  g_free(cpuh.values);
  g_free(memh.values);
  g_free(swph.values);
  cpuh.values=NULL;
  memh.values=NULL;
  swph.values=NULL;
  gdk_pixmap_unref(g1canvas); g1canvas=NULL;
  gdk_pixmap_unref(h1canvas); h1canvas=NULL;
  gdk_pixmap_unref(g2canvas); g2canvas=NULL;
  gdk_pixmap_unref(h2canvas); h2canvas=NULL;
  hdlg=NULL;
}

void mk_cput(GtkWidget *t) {
  GtkTable *cput;
  GtkWidget *l[4];
  GtkWidget *rt,*hm,*hs;
  GtkStyle *style;
  GdkPixmap *p;
  GdkBitmap *mask;
  GtkWidget *wp[2];
  int i;

  cput=GTK_TABLE(t);

  l[0]=gtk_label_new("CPU");
  l[1]=gtk_label_new("History");
  l[2]=gtk_label_new("Memory");
  l[3]=gtk_label_new("History");

  gtk_table_attach_defaults(cput,l[0],0,1,0,1);
  gtk_table_attach_defaults(cput,l[1],2,3,0,1);
  gtk_table_attach_defaults(cput,l[2],0,1,3,4);
  gtk_table_attach_defaults(cput,l[3],2,3,3,4);

  /* mem/swap  labels */
  rt=gtk_table_new(3,4,TRUE);
  gtk_table_attach_defaults(cput,rt,0,3,5,6);

  hm=gtk_hbox_new(FALSE,0);
  hs=gtk_hbox_new(FALSE,0);

  style=gtk_widget_get_style(hdlg);
  p=gdk_pixmap_create_from_xpm_d(hdlg->window,&mask,
				 &style->bg[GTK_STATE_NORMAL],
				 (gchar **)hgreen_xpm);
  wp[0]=gtk_pixmap_new(p,mask);
  p=gdk_pixmap_create_from_xpm_d(hdlg->window,&mask,
				 &style->bg[GTK_STATE_NORMAL],
				 (gchar **)hred_xpm);
  wp[1]=gtk_pixmap_new(p,mask);

  ml[1]=gtk_label_new("Used"); 
  ml[2]=gtk_label_new("Free");
  ml[3]=gtk_label_new("Total");

  ml[4]=gtk_label_new("Memory"); 
  ml[5]=gtk_label_new("0");
  ml[6]=gtk_label_new("0");
  ml[7]=gtk_label_new("0");

  ml[8]=gtk_label_new("Swap");
  ml[9]=gtk_label_new("0");
  ml[10]=gtk_label_new("0");
  ml[11]=gtk_label_new("0");

  gtk_box_pack_start(GTK_BOX(hm),wp[0],FALSE,FALSE,4);
  gtk_box_pack_start(GTK_BOX(hm),ml[4],FALSE,FALSE,0);

  gtk_box_pack_start(GTK_BOX(hs),wp[1],FALSE,FALSE,4);
  gtk_box_pack_start(GTK_BOX(hs),ml[8],FALSE,FALSE,0);

  gtk_table_attach_defaults(GTK_TABLE(rt),ml[1],1,2,0,1);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[2],2,3,0,1);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[3],3,4,0,1);

  gtk_table_attach_defaults(GTK_TABLE(rt),hm,0,1,1,2);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[5],1,2,1,2);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[6],2,3,1,2);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[7],3,4,1,2);

  gtk_table_attach_defaults(GTK_TABLE(rt),hs,0,1,2,3);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[9],1,2,2,3);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[10],2,3,2,3);
  gtk_table_attach_defaults(GTK_TABLE(rt),ml[11],3,4,2,3);

  /* cpu gauge */
  G1=gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(G1),64,96);

  gtk_signal_connect (GTK_OBJECT (G1), "expose_event",
		      (GtkSignalFunc) g1_expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT (G1), "configure_event",
		      (GtkSignalFunc) g1_configure_event, NULL);
  gtk_widget_set_events (G1, GDK_EXPOSURE_MASK);

  gtk_table_attach_defaults(cput,G1,0,1,1,2);

  lg1=gtk_label_new(" ");
  gtk_table_attach_defaults(cput,lg1,0,1,2,3);

  /* cpu history */
  H1=gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(H1),350,96);

  gtk_signal_connect (GTK_OBJECT (H1), "expose_event",
		      (GtkSignalFunc) h1_expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT (H1), "configure_event",
		      (GtkSignalFunc) h1_configure_event, NULL);
  gtk_widget_set_events (H1, GDK_EXPOSURE_MASK);

  gtk_table_attach_defaults(cput,H1,2,3,1,2);

  /* mem gauge */
  G2=gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(G2),64,96);

  gtk_signal_connect (GTK_OBJECT (G2), "expose_event",
		      (GtkSignalFunc) g2_expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT (G2), "configure_event",
		      (GtkSignalFunc) g2_configure_event, NULL);
  gtk_widget_set_events (G2, GDK_EXPOSURE_MASK);

  gtk_table_attach_defaults(cput,G2,0,1,4,5);

  /* mem history */
  H2=gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(H2),350,96);

  gtk_signal_connect (GTK_OBJECT (H2), "expose_event",
		      (GtkSignalFunc) h2_expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT (H2), "configure_event",
		      (GtkSignalFunc) h2_configure_event, NULL);
  gtk_widget_set_events (H2, GDK_EXPOSURE_MASK);

  gtk_table_attach_defaults(cput,H2,2,3,4,5);

  for(i=0;i<4;i++)
    gtk_widget_show(l[i]);

  for(i=1;i<12;i++)
    gtk_widget_show(ml[i]);

  gtk_widget_show(G1);
  gtk_widget_show(H1);
  gtk_widget_show(G2);
  gtk_widget_show(H2);
  gtk_widget_show(lg1);
  gtk_widget_show(wp[0]);
  gtk_widget_show(wp[1]);
  gtk_widget_show(hm);
  gtk_widget_show(hs);
  gtk_widget_show(rt);
}

/* gauge 1 (cpu) */

gboolean g1_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
			    gpointer data) {
  GdkGC *mygc;
  int ia,i,x,y,d,c;
  double fa;

  if (g1canvas!=NULL)
    gdk_pixmap_unref(g1canvas);

  mygc=gdk_gc_new(widget->window);
  
  g1canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(g1canvas,widget->style->black_gc,TRUE,0,0,x,y);

  fa=(1.0-cpu_usage)*((double)(y-5));
  ia=3+(int)fa;

  gdk_rgb_gc_set_foreground(mygc,0x008000);
  d=3+(y-5)/2;
  gdk_draw_line(g1canvas,mygc,1,d,x-2,d);
  d=3+(y-5)/4;
  gdk_draw_line(g1canvas,mygc,1,d,x-2,d);
  d=3+3*(y-5)/4;
  gdk_draw_line(g1canvas,mygc,1,d,x-2,d);

  for(i=y-2,c=0;i>=ia;i--,c=1-c) {
    if (c)
      gdk_rgb_gc_set_foreground(mygc,0x00ff00);
    else
      gdk_rgb_gc_set_foreground(mygc,0x008000);
    gdk_draw_line(g1canvas,mygc,10,i,x-10,i);
  }

  gdk_rgb_gc_set_foreground(mygc,0x00ff00);
  gdk_draw_rectangle(g1canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

gboolean g1_expose_event(GtkWidget *widget,GdkEventExpose *ee,
			    gpointer data) {
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  g1canvas,
		  ee->area.x, ee->area.y,
		  ee->area.x, ee->area.y,
		  ee->area.width, ee->area.height);
  return FALSE;
}

/* history 1 (cpu) */

gboolean h1_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
			    gpointer data) {
  GdkGC *mygc;
  int ia,ib,i,x,y,d,c;
  float fa,fb;
  long la;

  if (h1canvas!=NULL)
    gdk_pixmap_unref(h1canvas);

  mygc=gdk_gc_new(widget->window);
  
  h1canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(h1canvas,widget->style->black_gc,TRUE,0,0,x,y);

  /* should I fill or should I go ? */
  gdk_rgb_gc_set_foreground(mygc,0x004000);
  la=cpuh.cursor+2*cpuh.length;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-cpuh.values[la%cpuh.length])*((float)(y-5));
    ia=3+(int)fa;
    gdk_draw_line(h1canvas,mygc,i,ia,i,y-5+3);
  }

  /* 25%, 50%, 75% */
  gdk_rgb_gc_set_foreground(mygc,0x008000);
  d=3+(y-5)/2;
  gdk_draw_line(h1canvas,mygc,1,d,x-2,d);
  d=3+(y-5)/4;
  gdk_draw_line(h1canvas,mygc,1,d,x-2,d);
  d=3+3*(y-5)/4;
  gdk_draw_line(h1canvas,mygc,1,d,x-2,d);

  /* minute bars */
  gdk_rgb_gc_set_foreground(mygc,0x008000);
  for(i=x-2,c=0;i>1;i--,c++)
    if (((c%60)==0)&&(c!=0)) 
      gdk_draw_line(h1canvas,mygc,i,1,i,y-2);
  
  gdk_rgb_gc_set_foreground(mygc,0x00ff00);
  la=cpuh.cursor+2*cpuh.length;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-cpuh.values[la%cpuh.length])*((float)(y-5));
    ia=3+(int)fa;
    fb=(1.0-cpuh.values[(la-1)%cpuh.length])*((float)(y-5));
    ib=3+(int)fb;
    gdk_draw_line(h1canvas,mygc,i,ia,i-1,ib);
  }

  gdk_rgb_gc_set_foreground(mygc,0x00ff00);
  gdk_draw_rectangle(h1canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

gboolean h1_expose_event(GtkWidget *widget,GdkEventExpose *ee,
			    gpointer data) {
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  h1canvas,
		  ee->area.x, ee->area.y,
		  ee->area.x, ee->area.y,
		  ee->area.width, ee->area.height);
  return FALSE;
}

/* gauge 2 (memory & swap) */

gboolean g2_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
			    gpointer data) {
  GdkGC *mygc;
  int ia,i,x,y,hx,d,c;
  double fa;

  if (g2canvas!=NULL)
    gdk_pixmap_unref(g2canvas);

  mygc=gdk_gc_new(widget->window);
  
  g2canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(g2canvas,widget->style->black_gc,TRUE,0,0,x,y);

  hx=x/2;

  /* MEMORY */
  fa=(1.0-mem_usage)*((double)(y-5));
  ia=3+(int)fa;

  gdk_rgb_gc_set_foreground(mygc,0x008000);
  d=3+(y-5)/2;
  gdk_draw_line(g2canvas,mygc,1,d,x-2,d);
  d=3+(y-5)/4;
  gdk_draw_line(g2canvas,mygc,1,d,x-2,d);
  d=3+3*(y-5)/4;
  gdk_draw_line(g2canvas,mygc,1,d,x-2,d);

  for(i=y-2,c=0;i>=ia;i--,c=1-c) {
    if (c)
      gdk_rgb_gc_set_foreground(mygc,0x00ff00);
    else
      gdk_rgb_gc_set_foreground(mygc,0x008000);
    gdk_draw_line(g2canvas,mygc,10,i,((x-10)+hx)/2,i);
  }

  /* SWAP */
  fa=(1.0-swp_usage)*((double)(y-5));
  ia=3+(int)fa;

  for(i=y-2,c=0;i>=ia;i--,c=1-c) {
    if (c)
      gdk_rgb_gc_set_foreground(mygc,0xff0000);
    else
      gdk_rgb_gc_set_foreground(mygc,0x800000);
    gdk_draw_line(g2canvas,mygc,hx+1,i,x-10,i);
  }

  gdk_rgb_gc_set_foreground(mygc,0x00ff00);
  gdk_draw_rectangle(g2canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

gboolean g2_expose_event(GtkWidget *widget,GdkEventExpose *ee,
			    gpointer data) {
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  g2canvas,
		  ee->area.x, ee->area.y,
		  ee->area.x, ee->area.y,
		  ee->area.width, ee->area.height);
  return FALSE;
}

/* history 2 (mem/swap) */

gboolean h2_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
			    gpointer data) {
  GdkGC *mygc;
  int ia,ib,i,x,y,d,c;
  float fa,fb;
  long la;

  if (h2canvas!=NULL)
    gdk_pixmap_unref(h2canvas);

  mygc=gdk_gc_new(widget->window);
  
  h2canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			  y=widget->allocation.height,-1);
  gdk_draw_rectangle(h2canvas,widget->style->black_gc,TRUE,0,0,x,y);

  /* should I fill or should I go ? */
  gdk_rgb_gc_set_foreground(mygc,0x004000);
  la=memh.cursor+2*memh.length;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-memh.values[la%memh.length])*((float)(y-5));
    ia=3+(int)fa;
    gdk_draw_line(h2canvas,mygc,i,ia,i,y-5+3);
  }

  /* swap fill (filled in front of memory, as 
     memory usage should be higher than swap) */
  gdk_rgb_gc_set_foreground(mygc,0x800000);
  la=swph.cursor+2*swph.length;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-swph.values[la%swph.length])*((float)(y-5));
    ia=3+(int)fa;
    gdk_draw_line(h2canvas,mygc,i,ia,i,y-5+3);
  }

  /* 25%, 50%, 75% */
  gdk_rgb_gc_set_foreground(mygc,0x008000);
  d=3+(y-5)/2;
  gdk_draw_line(h2canvas,mygc,1,d,x-2,d);
  d=3+(y-5)/4;
  gdk_draw_line(h2canvas,mygc,1,d,x-2,d);
  d=3+3*(y-5)/4;
  gdk_draw_line(h2canvas,mygc,1,d,x-2,d);

  /* minute bars */
  gdk_rgb_gc_set_foreground(mygc,0x008000);
  for(i=x-2,c=0;i>1;i--,c++)
    if (((c%60)==0)&&(c!=0)) 
      gdk_draw_line(h2canvas,mygc,i,1,i,y-2);
  
  /* mem */
  gdk_rgb_gc_set_foreground(mygc,0x00ff00);
  la=memh.cursor+2*memh.length;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-memh.values[la%memh.length])*((float)(y-5));
    ia=3+(int)fa;
    fb=(1.0-memh.values[(la-1)%memh.length])*((float)(y-5));
    ib=3+(int)fb;
    gdk_draw_line(h2canvas,mygc,i,ia,i-1,ib);
  }

  /* swap */
  gdk_rgb_gc_set_foreground(mygc,0xff0000);
  la=swph.cursor+2*swph.length;
  for(i=x-2;i>1;i--,la--) {
    fa=(1.0-swph.values[la%swph.length])*((float)(y-5));
    ia=3+(int)fa;
    fb=(1.0-swph.values[(la-1)%swph.length])*((float)(y-5));
    ib=3+(int)fb;
    gdk_draw_line(h2canvas,mygc,i,ia,i-1,ib);
  }

  gdk_rgb_gc_set_foreground(mygc,0x00ff00);
  gdk_draw_rectangle(h2canvas,mygc,FALSE,0,0,x-1,y-1);

  gdk_gc_destroy(mygc);
  return FALSE;
}

gboolean h2_expose_event(GtkWidget *widget,GdkEventExpose *ee,
			    gpointer data) {
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  h2canvas,
		  ee->area.x, ee->area.y,
		  ee->area.x, ee->area.y,
		  ee->area.width, ee->area.height);
  return FALSE;
}

gint refresh_history(gpointer data) {
  static gint whereby=0;
  static float va=0.0,vb=0.0,vc=0.0,vd=0.0;
  char b[32];

  if (hdlg==NULL)
    return FALSE;

  cpu_usage=read_cpu_usage();

  switch(whereby) {
  case 0:
    cpuh.values[cpuh.cursor]=((va+vb+vc+vd)/4.0);
    va=(float)cpu_usage;
    read_mem_usage();
    memh.values[memh.cursor]=(float)mem_usage; 
    swph.values[swph.cursor]=(float)swp_usage; 
    break;
  case 1: vb=(float)cpu_usage; break;
  case 2: 
    vc=(float)cpu_usage;
    break;
  case 3: vd=(float)cpu_usage; break;
  }
  whereby=(++whereby)%4;

  g1_configure_event(G1,NULL,NULL);
  if (whereby==1)
    h1_configure_event(H1,NULL,NULL);

  if (whereby==1) {
    g2_configure_event(G2,NULL,NULL);
    gtk_widget_queue_draw(G2);
    h2_configure_event(H2,NULL,NULL);
    gtk_widget_queue_draw(H2);
    memh.cursor=(++memh.cursor)%memh.length;
    swph.cursor=(++swph.cursor)%swph.length;

    sprintf(b,"%lu KB",(memused>>10));
    gtk_label_set_text(GTK_LABEL(ml[5]),b);
    sprintf(b,"%lu KB",(memtot-memused)>>10);
    gtk_label_set_text(GTK_LABEL(ml[6]),b);
    sprintf(b,"%lu KB",(memtot>>10));
    gtk_label_set_text(GTK_LABEL(ml[7]),b);

    sprintf(b,"%lu KB",(swpused>>10));
    gtk_label_set_text(GTK_LABEL(ml[9]),b);
    sprintf(b,"%lu KB",(swptot-swpused)>>10);
    gtk_label_set_text(GTK_LABEL(ml[10]),b);
    sprintf(b,"%lu KB",(swptot>>10));
    gtk_label_set_text(GTK_LABEL(ml[11]),b);

  }
  gtk_widget_queue_draw(G1);
  if (whereby==1) {
    gtk_widget_queue_draw(H1);
    cpuh.cursor=(++cpuh.cursor)%cpuh.length;
  }
  return TRUE;
}

/*
 *
 * mutex'ed wrapper interface to the /proc pollers
 *
 */

double wrapped_read_cpu_usage() {
  double a;
  while(!enter_mutex()) usleep(17);
  a=read_cpu_usage();
  exit_mutex();
  return(a);
}

void wrapped_read_mem_usage() {
  while(!enter_mutex()) usleep(11);
  read_mem_usage();
  exit_mutex();  
}

/*
 *
 * non-GUI /proc reading functions, don't call directly.
 *
 */

double read_cpu_usage() {
  static long last_idle=-1,last_busy=-1;
  FILE *f;
  long idle,busy,da,db;
  double fa,fb;
  char buffer[1024];
  char *p;
  int i;

  f=fopen("/proc/stat","r");
  if (f==NULL)
    return 0.0;

  do 
    p=fgets(buffer,1024,f);
  while((strstr(buffer,"cpu")==NULL)&&(p!=NULL));

  fclose(f);

  if (p==NULL)
    return 0.0;

  p=strtok(buffer," \n\t");

  busy=0;
  for(i=0;i<3;i++) {
    p=strtok(NULL," \n\t");
    busy+=atol(p);
  }
  idle=atol(strtok(NULL," \n\t"));

  if (last_busy==-1) {
    last_busy=busy;
    last_idle=idle;
    return 0.0;
  }

  if ((idle==last_idle)&&(busy==last_busy)) {
    last_busy=busy;
    last_idle=idle;
    return 0.0;
  }

  da=busy-last_busy;
  db=idle-last_idle;
  fa=(double)da;
  fb=(double)db;
  fa/=fa+fb;
  if (fa>1.0) fa=1.0;
  
  last_busy=busy;
  last_idle=idle;
  return(fa);
}

void read_mem_usage() {
  FILE *f;
  char buffer[1024];
  char *p;

  f=fopen("/proc/meminfo","r");

  if (f==NULL) {
    mem_usage=swp_usage=1.0;
    memtot=swptot=memused=swpused=1;
    return;
  }

  /* MEMORY */
  do 
    p=fgets(buffer,1024,f);
  while((strstr(buffer,"Mem:")==NULL)&&(p!=NULL));

  if (p==NULL) {
    mem_usage=swp_usage=1.0;
    memtot=swptot=memused=swpused=1;
    return;
  }

  strtok(buffer," \n\t");

  memtot=atol(strtok(NULL," \n\t"));
  memused=atol(strtok(NULL," \n\t"));
  strtok(NULL," \n\t");
  strtok(NULL," \n\t");
  memused-=atol(strtok(NULL," \n\t")); // buffers
  memused-=atol(strtok(NULL," \n\t")); // cached

  mem_usage=((double)(memused))/((double)(memtot));

  /* SWAP */
  do 
    p=fgets(buffer,1024,f);
  while((strstr(buffer,"Swap:")==NULL)&&(p!=NULL));

  if (p==NULL) {
    swp_usage=1.0;
    swptot=swpused=1;
    return;
  }

  strtok(buffer," \n\t");
  swptot=atol(strtok(NULL," \n\t"));
  swpused=atol(strtok(NULL," \n\t"));

  swp_usage=((double)(swpused))/((double)(swptot));

  fclose(f);
}

/* NOT USED YET

int get_cpu_count() {
  FILE *f;
  char buffer[512];
  char *p;
  int pcount;

  f=fopen("/proc/stat","r");
  if (f==NULL) return 1; // until the closing of this gPS version 
                         // Linux still wasn't able to run on a 0-CPU
                         // machine.
  pcount=0;
  while((p=fgets(buffer,512,f))!=NULL)
    if (strstr(buffer,"cpu")!=NULL)
      pcount++;
  fclose(f);
  if (pcount>1) pcount--;

  return(pcount)
}

*/
