
#include <std.h>
#include "packages.h"
#include "screens.h"
#include "colors.h"
#include <interface/coreui.h>
#include <interface/formatting.h>
#include <interface/dialogs.h>
#include <func/apt.h>

// -- Main list of filters for the main package list
FilterList Filters;
//

FilterList::FilterList(void)
{
	Head = Tail = NULL;
	nodecount = 0;
}

FilterList::Node * FilterList::Push(char *str, FilterType Foo)
{
	if (str == NULL)
		return NULL;

	if (*str == 0)
		return NULL;

	nodecount += 1;

	if (Head == NULL && Tail == NULL)
	{
		Head = Tail = (Node *) malloc(sizeof(struct Node));
		Tail->next = NULL;
	}
	else
	{
		Tail->next = (Node *) malloc(sizeof(struct Node));
		Tail->next->next = NULL;
		Tail = Tail->next;
	}

	Tail->String = strdup(str);
	Tail->FilterID = nodecount;
	Tail->type = Foo;
	Tail->filtercount = 0;
	regcomp(&Tail->preg, str, REG_EXTENDED | REG_ICASE);

	return Tail;
}

void FilterList::Pop(void)
{
	struct Node *p;

	nodecount -= 1;

	regfree(&Tail->preg);

	if (Head == Tail)
	{
		free(Head);
		Head = Tail = NULL;
	}
	else
	{
		for (p = Head; p->next != Tail; p = p->next);
		p->next = p->next->next;
		free(Tail);
		Tail = p;
	}

}

bool FilterList::Match(pkgCache::PkgIterator I, FilterMode Mode)
{
	string s;

	switch (Mode)
	{
	case None:
	case Upright:
		for (Node * N = Head; N != NULL; N = N->next)
		{
			switch (N->type)
			{
			case PkgName:
				s = I.Name();
				break;

			case PkgShortDesc:
				s = Recs->Lookup(((*Cache)[I].CandidateVerIter(*Cache)).FileList()).ShortDesc();
				break;

			case PkgLongDesc:
				s = Recs->Lookup(((*Cache)[I].CandidateVerIter(*Cache)).FileList()).LongDesc();
				break;
			}

			if (regexec(&N->preg, s.c_str(), 0, NULL, 0) == 0)
			{
				N->filtercount++;
				return true;
			}
		}
		break;

	case Inverted:
		for (Node * N = Head; N != NULL; N = N->next)
		{
			switch (N->type)
			{
			case PkgName:
				s = I.Name();
				break;

			case PkgShortDesc:
				s = Recs->Lookup(((*Cache)[I].CandidateVerIter(*Cache)).FileList()).ShortDesc();
				break;

			case PkgLongDesc:
				s = Recs->Lookup(((*Cache)[I].CandidateVerIter(*Cache)).FileList()).LongDesc();
				break;
			}

			if (regexec(&N->preg, s.c_str(), 0, NULL, 0) != 0)
			{
				N->filtercount++;
				return true;
			}
		}
		break;
	}

	return false;
}

bool FilterList::SimulateApply()
{
	long fc = 0, dc = 0;
	ScreenList::Screen * S = screen.Head();

	if (S->next != NULL)
		S = S->next->next;

	for (PackageList::Package * P = S->Packages->Head(); P != NULL; P = P->next)
		if (screen->Packages->IsDivider(P))
			dc++;
		else if (Match(P->pkg, S->Packages->GetFilterMode()))
			fc++;

	for (Node * N = Head; N != NULL; N = N->next)
		N->filtercount = 0;

	if (fc == 0 || screen->Packages->Length() - dc == fc)
		return false;
	else
		return true;
}

void FilterList::Audit(void)
{
	while (nodecount > 0)
		Pop();
}

void FilterList::Display(void)
{
	char lbuf[COLS];
	char buf[COLS * 500];
	struct Node *p;

	strcpy(buf, "");

	for (p = Head; p != NULL; p = p->next)
	{
		sprintf(lbuf,
			" $2#type of filter:      $7#%s \n"
			" $2#identifying number:  $7#%03ld \n"
			" $2#regular expression:  $7#%s \n"
			" $2#number of packages:  $7#%04ld \n",
			p->type == PkgName ? "Package Name" :
			(p->type == PkgShortDesc ? "Package Short Description" : (p->type == PkgLongDesc ? "Package Long Description" : "Unknown")), p->FilterID, p->String, p->filtercount);

		strcat(buf, lbuf);

		if (nodecount > 1 && p->next != NULL)
			strcat(buf, "\n");
	}

	ui_dialog(Pair(COLOR_YELLOW, COLOR_BLUE) | A_BOLD, "Active Filters", "\n%s", buf);
	ui_refresh_current();
}

void FilterList::RedoLists()
{
	if (Length() == 0)
		return;

	delete screen.Head()->next->Packages;
	screen.Head()->next->Packages = new PackageList(FilterList::Upright);;

	delete screen.Head()->next->next->Packages;
	screen.Head()->next->next->Packages = new PackageList(FilterList::Inverted);

	screen.boundreset(screen.Head()->next);
	screen.boundreset(screen.Head()->next->next);
}

void filter_add_single()
{
	char *buf = (char *) malloc(COLS);

	ui_textentry(" filter package names by regular expression ", &buf);

	if (*buf == 0)
	{
		free(buf);
		paint_status();
		return;
	}

	FilterList::Node * New = Filters.Push(buf, FilterList::PkgName);

	free(buf);

	if (Filters.SimulateApply() == false)
	{
		ui_dialog(Pair(COLOR_YELLOW, COLOR_BLUE) | A_BOLD, "Filter Error",
			  "\n"
			  " The effect of your list of filters either: \n"
			  "\n" " (i)  filters every package \n" " (ii) filters no packages \n" "\n" " Try and refine or expand your regular expression \n");

		Filters.Pop();

		ui_refresh_current();

		return;
	}

	if (Filters.Length() == 1)
	{
		screen.filtnorm = screen.add(ScreenList::noPkgInit);
		screen.Head()->next->Packages = new PackageList(FilterList::Upright);

		screen.filtinv = screen.add(ScreenList::noPkgInit);
		screen.Head()->next->next->Packages = new PackageList(FilterList::Inverted);

		screen.boundreset(screen.Head()->next);
		screen.boundreset(screen.Head()->next->next);
	}
	else
		Filters.RedoLists();

	ui_redraw_current();
}

void filter_revoke_all()
{
	if (Filters.Length() == 0)
		return;

	Filters.Audit();

	screen.remove(screen.Head()->next->next);
	screen.remove(screen.Head()->next);

	screen.filtnorm = NULL;
	screen.filtinv = NULL;

	screen = screen.Head();

	ui_redraw_current();
}

void filter_display_all()
{
	if (Filters.Length() == 0)
		return;

	Filters.Display();

	ui_redraw_current();
}

void filter_change_view()
{
	if (Filters.Length() == 0)
		return;

	if (screen.Current() == screen.Head()->next)
		screen = screen.Head()->next->next;
	else
		screen = screen.Head()->prev;

	ui_redraw_current();
}

void filter_invert()
{
	if (Filters.Length() == 0)
		return;

	if (screen.Current() == screen.Head()->next->next)
		screen = screen->prev;
	else if (screen.Current() == screen.Head()->next)
		screen = screen->next;

	ui_redraw_current();
}
