/*
 * installer/image-linux.c
 *
 * Linux image stuff.
 * 
 * Copyright (c) Tuomo Valkonen 1996-1998.
 */
 
#include<stdio.h>
#include<strings.h>
#include<sys/types.h>
#include<sys/stat.h>
#include<fcntl.h>
#include<errno.h>
#include<linux/fs.h>
 
#include<chos/chos.h>
#include<chos/main.h>
#include<chos/image.h>
#include<chos/install.h>
#include<chos/map.h>
#include<chos/linux.h>
#include<chos/bin.h>

#define LIN1SECT(x)	((MF_LinuxFirstSector*)(map[(x)->addr]))

#define LI_SIGNATURE	"HdrS"
#define LI_SIG_LEN	4

#define LI_HDR_VERSION	0x0201	// minimum versions supported

#define _BIG		(img->flags&LINF_BIG)

// Hopefully GCC does no alignments on this 
//
// code32_start should really be set to 0x100000 at kernel compilation time
// or some flag set so that we could really know it is a bzImage, but for now
// we just have to trust the user...

typedef struct{	
	short		jump_code;
	char		signature[4];	// HdrS
	unsigned short	version;
	short		rm_swtch,rm_seg;
	short		start_sys_seg;
	short		kernel_version_ptr;
	char		type_of_loader;	
	char		load_flags;
	short		setup_move_size;
	long		code32_start;
	long		ramdisk_image;
	long		ramdisk_size;
	short		bs_kludge,bs_seg;
	short		heap_end_ptr;
} LinuxHeader;

static int check_header(int fd,char* name)
{
	LinuxHeader	hdr;
	
	if(lseek(fd,512,SEEK_SET)!=512)		// skip boot sector
		die(errno,name);

	if(read(fd,&hdr,sizeof(LinuxHeader))!=sizeof(LinuxHeader))
		die(errno,name);

	if(memcmp(hdr.signature,LI_SIGNATURE,LI_SIG_LEN) ||
	   hdr.version<LI_HDR_VERSION)
	   	return -1;
	
	if(hdr.load_flags&1){
		// We allow a load address >1M but we will, however,
		// always load it at 1M.
		if(hdr.code32_start>=0x100000)
			return 1;
		die(-1,"%s: Header claims the kernel is a bzImage but "
			"at the same time default load address is below 1M!\n",name);
	}

	// We only support zImages loaded at default 0x10000
	if(hdr.start_sys_seg!=0x1000)
		die(-1,"%s: Invalid system segment for zImage 0x%x"
		       " (should be 0x1000)\n",name,hdr.start_sys_seg);
	return 0;
}

static void do_linux_image(MF_ImageDes*img,char*name)
{
	int 		pos=0,pos2=0;
	int 		fd,size,addr;
	struct	stat 	st;
	GEOMETRY 	geo;
	MF_LinuxFirstSector*sect1;
	MF_KernelSector*tmpsect;
	char		setupsects;

	sect1=LIN1SECT(img);
	
	if(sect1->device!=0xff)
		cfgerror("Duplicate image=");
		
	verbose("\tUsing %s as kernel image...\n",name);
	
	// Open the kernel file,check it's normal 'n count the size in sectors
	//
	if( (fd=open(name,O_RDONLY))<0)
		die(errno,name);

	if(fstat(fd,&st)<0)
		die(errno,name);
		
	if(!S_ISREG(st.st_mode))
		die(-1,"%s:not a valid regular file!\n",name);

	switch(check_header(fd,name)){
	case -1:
		if(img->flags&LINF_BIG)
			die(-1,"%s: Old kernel - cannot load high.\n",name);
		break;
	case 1:
		img->flags|=LINF_NEW;
		if(!(img->flags&LINF_BIG)){
			img->flags|=LINF_BIG;
			verbose("\tSeems to be a bzImage...\n");
		}
		break;
	case 0:
		if(img->flags&LINF_BIG)
			die(-1,"%s: Not a bzImage. Use \"linux\" instead of \"big_linux\"\n",name);
		img->flags|=LINF_NEW;
	}

	if( (size=lseek(fd,0,SEEK_END))<0)
		die(errno,name);

	size=size/SECTORSIZE+(size%SECTORSIZE==0?0:1);
	
	if(size>MAX_LINUX_SIZE){
		if(!_BIG)
			die(-1,	"%s: Kernel too big !\n"
				"Use \'big_linux\' in case of bzImage (see the README).");
		if(!(img->flags&LINF_NEW))
			die(-1,"%s: Unknown header format - cannot load high.",name);
	}
	
	
	// Get the geometry of the device the file is on
	//
	if(!get_device(fd,&geo))
		die(errno,name);

	sect1->device=geo.device;
	sect1->floppybsect=get_addr(fd,0,&geo);

	VER_BIOS(&geo,sect1->floppybsect,name);
	
	map_get_sector(&tmpsect);
	
	// Count the addresses of kernel setup sectors
	//
	if( lseek(fd,LINUX_SETUPSECT_OFF,SEEK_SET)<LINUX_SETUPSECT_OFF )
		die(errno,name);

	if(read(fd,&setupsects,1)<1)
		die(errno,name);
	
	for(pos=1,pos2=0;pos<=setupsects;pos++,pos2++){
		if(addr=get_addr(fd,pos,&geo)){
			if(addr<0)
				die(errno,name);
			VER_BIOS(&geo,addr,name);
			tmpsect->sect[pos2]=addr;
		}else
			tmpsect->sect[pos2]=MF_NULLSECTOR;
	}

	// Put a MF_ENDSECTOR there so the loader knows where the setup ends
	//
	tmpsect->sect[pos2++]=MF_ENDSECTOR;
	
	// Now, start the loop in which we count the addresses of each sector
	// of the kernel.
	//
	for(;pos<size;pos++){
		if(addr=get_addr(fd,pos,&geo)){
			if(addr<0)
				die(errno,name);
			VER_BIOS(&geo,addr,name);
			tmpsect->sect[pos2]=addr;
		}else
			tmpsect->sect[pos2]=MF_NULLSECTOR;
		pos2++;
		if( pos2>=MF_NENTS ){
			map_get_sector(&tmpsect);
			pos2=0;
		}
	}
	tmpsect->sect[pos2]=MF_ENDSECTOR;

	close(fd);
}

#define MEM_FILE	"/proc/meminfo"

#ifdef SUPPORT_INITRD

extern char tmpstr[SECTORSIZE];

// I hope this works and all kernels use the same format with /proc/meminfo.
static ulong get_memory_size()
{
	int	fd;
	ulong 	i=0;
	char*	p;
	
	if((fd=open(MEM_FILE,O_RDONLY))<0)
		die(errno,MEM_FILE);
	
	while(1){
		if(read(fd,&tmpstr[i],1)!=1)
			die(errno,MEM_FILE);
		switch(tmpstr[i]){
		case '\n':
			i=0;
			if(strncmp(tmpstr,"MemTotal:",9))
				break;
			// Found it 
			p=tmpstr+9;
			while(*p==' ' || *p=='\t'){
				if(*p=='\0')
					die(-1,MEM_FILE" is bugging!");
				p++;
			}
			i=strtoul(p,&p,0);
			if(*p!=' ')
				die(-1,MEM_FILE" is bugging!");
				
			return i*1024;	// It should be in kB...
			
		case EOF:
			die(-1,"Couldn't find MemTotal in "MEM_FILE"!");
		default:
			i++;
		}
	}
}
	
// Damn... wish there was an easy way to find out the size of
// the uncompressed kernel - say, it was in the kernel header.
// For now we just assume something.

static char* initrd_top_warn=
	"This system may/does not have enough memory for initrd!\n"
	"If creating a boot disk for another system you may continue, else don't.\n";

static void check_if_ramdisk_fits(ulong addr,ulong size,int big)
{
	ulong	mem_size=get_memory_size();
	int	top=(addr==0xffffffff);
	
	if(addr==0xffffffff)
		addr=mem_size-size;

	if(addr+size>mem_size){
		warn(initrd_top_warn);
		return;
	}
		
	if(!big){
		if(addr<0x200000){
			if(top)
				warn(initrd_top_warn);
			else			
				die(-1,"Initrd must be loaded above 2M.\n");
		}else if(addr<0x400000){
			if(top)
				warn(initrd_top_warn);
			else
				warn("It is safer to load initrd above 4M.\n");
		}
		return;
	}
	
	// bzImage
	if(addr<0x600000){
		if(top)
			warn(initrd_top_warn);
		else
			die(-1,"For bzImages, initrd must be loaded above 6M.\n");
	}else if(addr<0x800000){
		if(top)
			warn(initrd_top_warn);
		else
			warn("It is safer to load initrd above 8M.\n");
	}
	
}

#endif

static void do_initrd(MF_ImageDes*img,char*file,int big)
{
#ifndef SUPPORT_INITRD
	cfgerror("Initrd support not compiled in. Please recompile.");
#else
	int		fd,size;
	GEOMETRY	geo;
	struct stat	st;
	char*		p,*p2;
	ulong		load_addr,pos,pos2,addr;
	MF_LinuxFirstSector*s1=LIN1SECT(img);
	MF_KernelSector*tmpsect;
	
	if(img->flags&LINF_INITRD)
		cfgerror("Number of initial ramdisks is restricted to one.");

	if(s1->device==0xff)
		cfgerror("You must specify kernel image before initrd, sorry.\n");
		
	if(p=strchr(file,',')){
		*p='\0';
		load_addr=strtoul(p+1,&p2,0);
		if(*p2 || p2==p){
			if(p2==p || strcmp(p,"top"))
				cfgerror("Invalid ramdisk load address");
			load_addr=0xffffffff;
		}
	}else
		load_addr=0xffffffff;
	
	verbose("\tUsing %s as initial ramdisk...\n",file);
	
	if((fd=open(file,O_RDONLY))<0)
		die(errno,file);

	if(fstat(fd,&st)<0)
		die(errno,file);
		
	if(!S_ISREG(st.st_mode))
		die(-1,"%s:not a regular file!\n",file);

	if((size=lseek(fd,0,SEEK_END))<0)
		die(errno,file);

	check_if_ramdisk_fits(load_addr,size,big);
	
	if(load_addr==0xffffffff)
		verbose("\t ramdisk image will be loaded on top of memory.\n");
	else
		verbose("\t ramdisk image will be loaded at 0x%08x\n",load_addr);
		
	size=size/SECTORSIZE+(((size%SECTORSIZE)==0)?0:1);
		
	if(!get_device(fd,&geo))
		die(errno,file);
	
	img->flags|=LINF_INITRD;
	s1->rd_device=geo.device;
	s1->rd_size=size;
	s1->rd_load_addr=load_addr;
	s1->rd_map=map_get_sector(&tmpsect);

	for(pos=pos2=0;pos<size;pos++){
		if(addr=get_addr(fd,pos,&geo)){
			if(addr<0)
				die(errno,file);
			VER_BIOS(&geo,addr,file);
			tmpsect->sect[pos2]=addr;
		}else
			tmpsect->sect[pos2]=MF_NULLSECTOR;
		pos2++;
		if( pos2>=MF_NENTS ){
			map_get_sector(&tmpsect);
			pos2=0;
		}
	}

	tmpsect->sect[pos2]=MF_ENDSECTOR;

	close(fd);
#endif
}

static void do_linux_initrd(MF_ImageDes*img,char*file)
{
	if(!(img->flags&LINF_NEW))
		cfgerror("Image has unknown header format (old kernel?) - cannot load initrd!\n");

	do_initrd(img,file,_BIG);
}

static void do_linux_cmdline(MF_ImageDes*img,char*cmd)
{
	int	i;
	MF_LinuxFirstSector*sect1=LIN1SECT(img);
	
	if(sect1->cmdline[0])
		cfgerror("Duplicate cmdline=");
		
	for(i=0;;i++,cmd++){
		if(*cmd=='\t')
			sect1->cmdline[i]=' ';
		else 
			sect1->cmdline[i]=*cmd;
			
		if(*cmd=='\0')
			break;
			
		if(i==LINUX_CMDLINE_LEN-1)
			cfgerror("Command line too long!");
			
	}
		
	verbose("\tCommand line set to: %s\n",sect1->cmdline);
}

static void do_linux_init(MF_ImageDes*img)
{
	MF_HeaderSector*	__mh=mh;
	MF_LinuxFirstSector*	sect1;
	
	img->addr=map_get_sector((MF_KernelSector**)&sect1);
	img->device=0xff;
	
	sect1->device=0xff;
}

static void do_biglinux_init(MF_ImageDes*img)
{
#ifdef SUPPORT_BZIMAGE
	do_linux_init(img);
	img->flags|=LINF_BIG;
#else
	cfgerror("bzImage support not compiled in. Please recompile.\n");
#endif
}

static void do_linux_end(MF_ImageDes*img)
{
	MF_LinuxFirstSector*s=LIN1SECT(img);
	
	if(s->device==0xff)
		die(-1,"no kernel image specified for \"%s\"\n",img->name);
		
	if(img->flags&LINF_INITRD && !(img->flags&LINF_NEW))
		die(-1,"Cannot support initrd - kernel header has unknown format (old kernel?)\n");

	if(img->loader==0xff)
		img->loader=get_loader("linux",CHOS_MODULE(BIT_LINUX),"Linux loader");

}

static struct icfgstr_struct linux_cfgs[]={
	{"image=",do_linux_image},
	{"cmdline=",do_linux_cmdline},
	{"initrd=",do_linux_initrd},
	{NULL,}
};


struct imagestr_struct imgstr_linux=
	{"linux",	"Linux kernel",   	BIT_LINUX,   linux_cfgs,do_linux_init,do_linux_end};


struct imagestr_struct imgstr_biglinux=
	{"big_linux",	"big Linux kernel",	BIT_LINUX,   linux_cfgs,do_biglinux_init,do_linux_end};
