/* 
 * installer/device.c
 * 
 * devices stuff.
 *
 * Copyright (c) Tuomo Valkonen 1996-1998.
 */
 
#include<stdio.h>
#include<strings.h>
#include<sys/types.h>
#include<sys/stat.h>
#include<fcntl.h>
#include<errno.h>
#include<linux/fs.h>
#include<linux/fd.h>
#include<linux/hdreg.h>
#include<dirent.h>
#include<ctype.h>

#include<chos/chos.h>
#include<chos/main.h>
#include<chos/image.h>
#include<chos/install.h>
#include<chos/map.h>
#include<chos/mapfile.h>

#define DEVPATH 	"/dev"	

int last_dev(char* dirname,int lmaj,int incr)
{
	DIR		*dp;
	struct dirent	*dir;
	int		max,this,fd;
	struct stat	st;
	struct hd_geometry dummy;

	if( (dp=opendir(dirname))==NULL)
		die(errno,dirname);

	max=0;
	
	while( dir=readdir(dp) ){
		// Find devices which have the same major
		sprintf(tmpstr,"%s/%s",dirname,dir->d_name);
		if( stat(tmpstr,&st)>=0 ){
			if( S_ISBLK(st.st_mode) && MAJOR(st.st_rdev)==lmaj &&
			(MINOR(st.st_rdev) & (incr-1)) == 0){
				this=MINOR(st.st_rdev)/incr+1;
				if(this>max){
				 /* Can we read it ?*/
				 if( (fd=open(tmpstr,O_RDWR))>=0){
				  if( read(fd,&dummy,1)==1 &&
				      ioctl(fd,HDIO_GETGEO,&dummy)>=0)
					max=this;
				  close(fd);
				 }
				}
			}
		}
	}
	return max;
}
							
struct biosdrive_map{
	dev_t dev;
	char  bios;
};
#define MAX_DRIVEMAPS 32
struct biosdrive_map drivemaps[MAX_DRIVEMAPS]; // I guess that is enough...
int ndrivemaps=0;

void do_add_biosdrive_map(char*str)
{
	struct stat st;
	int dev;
	ulong biosid;
	char*name;

	biosid=strtoul(str,&name,16);
	while(isspace(*name))
		name++;
	if(*name!=',')
		cfgerror("Expected: drive_number,device_name");
	name++;
	while(isspace(*name))
		name++;
	if(*name=='\0')
		cfgerror("Expected: drive_number,device_name");
		
	if(biosid>0xfe)
		cfgerror("Max. bios drive number is 0xfe\n");

	// ---
	
	if(stat(name,&st)<0)
		die(errno,name);

	if(!S_ISBLK(st.st_mode))
		die(ENOTBLK,name);
		
	switch(MAJOR(st.st_rdev)){
	case MAJOR_FD:
		if(MINOR(st.st_rdev)>3)
			cfgerror("%s: Please use fd[0-3]\n",name);
		break;
	case MAJOR_HD:
	case MAJOR_IDE2:
	case MAJOR_XT:
		if(MINOR(st.st_rdev)&63)
			cfgerror("%s: Please use hd[a-d]\n",name);
		break;
	case MAJOR_SD:
		if(MINOR(st.st_rdev)&15)
			cfgerror("%s: Please use sd[a-g]\n",name);
		break;
	default:
		cfgerror("%s: unknown device\n");
	}
	if(ndrivemaps==MAX_DRIVEMAPS-1)
		die(-1,"Max number of linux->bios drive mappings exceeded\n");
	
	drivemaps[ndrivemaps].dev=st.st_rdev;
	drivemaps[ndrivemaps].bios=biosid;
	ndrivemaps++;
	
	verbose("Device %s mapped to bios number 0x%02x\n",name,biosid);
}

static int search_biosdrive_map(dev_t dev)
{
	int minor;
	int i;
	
	if(!ndrivemaps)
		return 0xff;
	minor=MINOR(dev);
	switch(MAJOR(dev)){
	case MAJOR_FD:
		minor&=3;
		break;
	case MAJOR_HD:
	case MAJOR_IDE2:
	case MAJOR_XT:
		minor&=~63;
		break;
	case MAJOR_SD:
		minor&=~15;
		break;
	default:
		return 0xff;
	}
	dev=MKDEV(MAJOR(dev),minor);
	for(i=0;i<ndrivemaps;i++){
		if(drivemaps[i].dev==dev)
			return drivemaps[i].bios;
	}
	return 0xff;
}

	
// Get geometry of a device where fdp is on or fdp is
//
int get_device( int fdp, GEOMETRY *geo)
{
	int 			fd;
	struct 	floppy_struct 	fdgeo;
	struct 	hd_geometry 	hdgeo;
        struct 	stat 		st;
        int			device;
        int			tmp_flags;
        int			biosdrive_set=0;
        
        if( fstat(fdp,&st)<0 )
        	return 0;
        
        if( S_ISBLK(st.st_mode) ){
        	device=st.st_rdev;
        	fd=fdp;
        	// assume block size is sector size
        	geo->spb=1;
        }else{	// It's a file...
        	device=st.st_dev;
        	// Get block size
        	if( ioctl(fdp,FIGETBSZ,&(geo->spb))<0)
        		return 0;
     
        	geo->spb=geo->spb/SECTORSIZE;
        	fd=open_dev(device);
	}
	if((geo->device=search_biosdrive_map(device))!=0xff){
		biosdrive_set=1;
		//verbose("Remapped 0x%02x\n",geo->device);
	}
	switch(MAJOR(device)){
	case MAJOR_FD:
		if(ioctl(fd,FDGETPRM,&fdgeo)<0)
			die(errno,dev_name(device));
			geo->heads=fdgeo.head;
		geo->cylinders=fdgeo.track;
		geo->sectors=fdgeo.sect;
		geo->start=0;
		if(!biosdrive_set)
			geo->device=MINOR(device)&3;
		break;

	case MAJOR_HD:
		if(ioctl(fd,HDIO_GETGEO,&hdgeo)<0)
			die(errno,dev_name(device));
			
		geo->heads=hdgeo.heads;
		geo->cylinders=hdgeo.cylinders;
		geo->sectors=hdgeo.sectors;
		geo->start=hdgeo.start;
		if(!biosdrive_set)
			geo->device=0x80+(MINOR(device)>>6);
		break;
	case MAJOR_XT:
	case MAJOR_IDE2:
		if(ioctl(fd,HDIO_GETGEO,&hdgeo)<0)
			die(errno,dev_name(device));
		
		geo->heads=hdgeo.heads;
		geo->cylinders=hdgeo.cylinders;
		geo->sectors=hdgeo.sectors;
		geo->start=hdgeo.start;
		if(!biosdrive_set)
			geo->device=0x80+(MINOR(device)>>6)+
				last_dev(DEVPATH,MAJOR_HD,64);
		break;
	case MAJOR_SD:
		if(ioctl(fd,HDIO_GETGEO,&hdgeo)<0)
			die(errno,dev_name(device));
			
		geo->heads=hdgeo.heads;
		geo->cylinders=hdgeo.cylinders;
		geo->sectors=hdgeo.sectors;
		geo->start=hdgeo.start;
			
		// LILO does it something like this...
		// I don't know if this works as I don't have SCSI
		// disks...
		if(!hdgeo.sectors){
			die(-1,"Invalid harddisk geometry gotten\n"
			       " (some1 should add the option to manually tell geometry!)\n");
		}
		if(!biosdrive_set)
			geo->device=0x80+(MINOR(device)>>4)+
				last_dev(DEVPATH,MAJOR_HD,64);
		break;
		
	default:
		die(-1,"Dunno how to handle device %s.",dev_name(device));
	}

	return 1;					
}
 
 
// Scan /dev directory for a device.
//
char* dev_name( dev_t device )
{
	DIR 		*ds;
	struct dirent 	*dir;
	struct stat 	st;
	int 		fd;
	
	if( (ds=opendir(DEVPATH))==NULL)
		die(errno,DEVPATH);
	
	while( dir=readdir(ds) ){
		sprintf(tmpstr,DEVPATH"/%s",dir->d_name);
		if(stat(tmpstr,&st)>=0){
			if(S_ISBLK(st.st_mode) && st.st_rdev==device){
				closedir(ds);
				return tmpstr;
			}
		}		
	}

	die(-1,"(%02x:%02x):No such device in "DEVPATH" !!!\n",
		MAJOR(device),MINOR(device));
}			

int open_dev(dev_t device)
{
	int	fd;
	if((fd=open(dev_name(device),3))<0)
		die(errno,tmpstr);
		
	return fd;
}
		
// Gets linear sector address of sector sect of file fd on device geo.
//
int get_addr( int fd, int sect, GEOMETRY *geo )
{
	struct 	stat 	st;
	int		block,sector;
	
	if( fstat(fd,&st)<0 )
		return -1;
	
	/* Is it a block special file ? - return start address in geo	*/
	if( S_ISBLK(st.st_mode) )
		return geo->start+sect;
	
	/* Count block address	*/
	block=sect/geo->spb;
	
	if( ioctl(fd,FIBMAP,&block)<0 )
		return -1;

	if(block==0)
		return	0;
	
	/* Count the sector address	*/
	sector=(block*geo->spb)+(sect%geo->spb);
	sector+=geo->start;
	
	return sector;
}	

// Verifies that BIOS can read sector address addr with geometry geo.
//
int verify_bios_read(GEOMETRY *geo,ulong addr)
{
	unsigned char 	head,sector;
	unsigned short 	cylinder;
	
	sector   = addr % geo->sectors + 1;
	head     = addr / geo->sectors % geo->heads;
	cylinder = addr / geo->sectors / geo->heads;
	
	if( sector > geo->sectors || head > geo->heads )
		die(-1,"Geometry calculation error !!!\n");
	if( cylinder >= geo->cylinders && geo->cylinders!=1)
		die(-1,"Cylinder beyond end of media (%d>%d)\n",cylinder,geo->cylinders-1);
	if( cylinder > 1023 )
		return 0;
	
	return 1;
}	
