// vs_tree.cc
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Implementation of stuff in vs_tree.h

#include "vs_tree.h"
#include "vs_statusedit.h"
#include "config/keybindings.h"
#include "config/colors.h"

class vs_getsearchterm:public vs_statusedit
{
public:
  vs_getsearchterm(vs_tree *_owner):vs_statusedit(_owner, "Search for: ") {}

  void entered(string s)
  {
    get_tree()->search_for(s);
  }
};

vs_tree::vs_tree()
  :vscreen(), root(NULL), begin(new vs_tree_root_iterator(NULL)), end(begin), top(begin), selected(top)
{
}

vs_tree::vs_tree(vs_treeitem *_root, bool showroot)
  :vscreen(), root(NULL), begin(new vs_tree_root_iterator(NULL)), end(begin), top(begin), selected(top)
{
  set_root(_root);
}

void vs_tree::set_root(vs_treeitem *_root, bool showroot)
{
  if(root)
    delete root;

  root=_root;

  if(root)
    {
      if(showroot)
	{
	  begin=new vs_tree_root_iterator(_root);
	  end=_root->end();
	  if(end==NULL)
	    {
	      end=begin;
	      end++;
	    }
	}
      else
	{
	  begin=_root->begin();
	  end=_root->end();
	}

      top=begin;
    }
  else
    {
      top=begin=end=new vs_tree_root_iterator(NULL);
    }

  selected=top;
  while(selected!=end && !selected->get_selectable())
    selected++;
  if(selected!=end)
    selected->highlighted(this);
}

void vs_tree::sync_bounds()
  // As I said: yuck!
{
  begin=root->begin();
  if(top==end)
    top=begin;
  if(selected==end)
    selected=begin;
  end=root->end();
}

int vs_tree::line_of(vs_treeiterator item)
  // Returns the Y coordinate of the given item.  (so we have to count
  // from 1)
{
  int j;
  vs_treeiterator i=top;
  if(item==top)
    return 1;

  j=1;
  do {
    i++;
    j++;
    if(i==item)
      return j;
  } while(i!=end);

  i=top;
  j=1;
  do {
    i--;
    j--;
    if(i==item)
      return j;
  } while(i!=begin);

  assert(0);
}

bool vs_tree::item_visible(vs_treeiterator pkg)
{
  int width,height;
  vs_treeiterator i=top;

  getmaxyx(height,width);
  height-=2;
  // Argh.  I really need to implement subwindows for this stuff.

  while(height>0 && i!=pkg && i!=end)
    {
      height--;
      i++;
    }

  return height>0 && i!=end;
}

void vs_tree::move_selection(vs_treeiterator from, vs_treeiterator to)
{
  if(from!=end)
    {
      attrset(from->get_normal_attr());
      (*from).display(this, line_of(from));
    }

  if(to!=end && to->get_selectable())
    {
      attrset(to->get_highlight_attr());
      (*to).display(this, line_of(to));
    }
}

void vs_tree::set_selection(vs_treeiterator to)
{
  if(item_visible(to))
    {
      if(selected!=end)
	selected->unhighlighted(this);
      move_selection(selected, to);
      selected=to;
      if(selected!=end)
	selected->highlighted(this);
      refresh();
    }
  else
    {
      if(selected!=end)
	selected->unhighlighted(this);
      selected=top=to;
      if(selected!=end)
	selected->highlighted(this);
      repaint();
    }
}

void vs_tree::dispatch_char(chtype ch)
{
  while(!delete_list.empty()) // As good a time as any..
    {
      delete delete_list.front();
      delete_list.pop_front();
    }

  int width,height;

  getmaxyx(height,width);
  height-=2;	// Because of the status and header lines
  width++;	// Because we *actually* get the maximum X value (ie, x-1) --
		// also the reason that height is -=1, not -=2.

  if(selected!=vs_treeiterator(NULL))
    {
      if(global_bindings.key_matches(ch, "SelectParent"))
	{
	  if(!selected.is_root())
	    set_selection(selected.get_up());
	}
      else if(global_bindings.key_matches(ch, "SelectNext"))
	{
	  if(selected!=end)
	    {
	      vs_treeiterator orig=selected;
	      bool do_repaint=false;
	      selected->unhighlighted(this);
	      int newline=line_of(selected);
	      do
		{
		  ++newline;
		  ++selected;
		  if(newline>=height-1)
		    {
		      int i;
		      do_repaint=true;

		      for(i=0; i<height/2 && top!=end; i++)
			++top;

		      if(top==end)
			--top;
		    }
		} while(selected!=end && !do_repaint && !selected->get_selectable());

	      if(selected!=end && !selected->get_selectable())
		{
		  newline=line_of(selected);
		  while(newline<height-2 && selected!=end && !selected->get_selectable())
		    {
		      ++newline;
		      ++selected;
		    }
		}

	      if(selected==end)
		--selected;
	      if(selected!=end)
		selected->highlighted(this);
	      if(do_repaint)
		repaint();
	      else
		{
		  move_selection(orig, selected);
		  refresh();
		}
	    }
	}
      else if(global_bindings.key_matches(ch, "SelectPrev"))
	{
	  vs_treeiterator orig=selected;
	  bool do_repaint=false;
	  if(selected!=end)
	    selected->unhighlighted(this);

	  do
	    {
	      vs_treeiterator prev=selected;
	      if(selected!=begin)
		--selected;

	      if(prev==top && selected!=top)
		{
		  int i;

		  if(do_repaint)
		    break;

		  do_repaint=true;
		  for(i=0; i<height/2 && top!=begin; i++)
		    --top;
		}
	    } while(selected!=begin && !do_repaint && !selected->get_selectable());

	  if(selected!=begin && !selected->get_selectable())
	    {
	      while(selected!=top && !selected->get_selectable())
		--selected;
	    }

	  if(orig==end && !do_repaint && selected!=end && !selected->get_selectable())
	    selected=end;
	  /* If everything is hidden and we didn't move, continue to hide
	   * the selection.
	   */

	  if(selected!=end)
	    selected->highlighted(this);

	  if(do_repaint)
	    repaint();
	  else
	    {
	      move_selection(orig, selected);
	      refresh();
	    }
	}
      else if(global_bindings.key_matches(ch, "NextPage"))
	{
	  int count=height;
	  vs_treeiterator newtop=top;
	  while(count>0 && newtop!=end)
	    {
	      ++newtop;
	      count--;
	    }

	  if(count==0 && newtop!=end)
	    {
	      int l=0;
	      (*selected).unhighlighted(this);
	      selected=top=newtop;
	      while(l<height && selected!=end && !selected->get_selectable())
		++selected;
	      if(l==height || selected==end)
		selected=top;
	      (*selected).highlighted(this);
	      repaint();
	    }
	}
      else if(global_bindings.key_matches(ch, "PrevPage"))
	{
	  int count=height;
	  vs_treeiterator newtop=top;
	  while(count>0 && newtop!=begin)
	    {
	      --newtop;
	      count--;
	    }

	  if(newtop!=top)
	    {
	      int l=0;
	      selected=top=newtop;
	      while(l<height && selected!=end && !selected->get_selectable())
		++selected;
	      if(l==height || selected==end)
		selected=top;
	      repaint();
	    }
	}
      else if(global_bindings.key_matches(ch, "ListTop"))
	{
	  int l=0;
	  vs_treeiterator prev=selected;

	  if(selected!=end)
	    selected->unhighlighted(this);

	  selected=begin;
	  while(l<height && selected!=end && !selected->get_selectable())
	    ++selected;
	  if(l==height || selected==end)
	    selected=begin;

	  if(selected!=end)
	    selected->highlighted(this);

	  if(top!=begin)
	    {
	      top=begin;
	      repaint();
	    }
	  else
	    {
	      move_selection(prev,selected);
	      refresh();
	    }
	}
      else if(global_bindings.key_matches(ch, "ListBottom"))
	{
	  int l=-1;
	  vs_treeiterator last=end,newtop=end,prev=selected;
	  --last;
	  while(newtop!=begin && newtop!=top && height>0)
	    {
	      --newtop;
	      --height;
	      l++;
	    }

	  if(selected!=end)
	    selected->unhighlighted(this);

	  selected=last;
	  while(l>=0 && selected!=end && !selected->get_selectable())
	    {
	      --selected;
	      l--;
	    }
	  if(selected==end && l<0)
	    selected=last;

	  if(selected!=end)
	    selected->highlighted(this);

	  if(newtop!=top)
	    {
	      top=newtop;
	      repaint();
	    }
	  else
	    {
	      move_selection(prev,selected);
	      refresh();
	    }
	}
      else if(global_bindings.key_matches(ch, "ListSearch"))
	{
	  add_status_widget(new vs_getsearchterm(this));
	  refresh();
	}
      else if(global_bindings.key_matches(ch, "Refresh"))
	{
	  vscreen_refresh();
	}
      else
	{
	  if(selected!=end && selected->get_selectable() && selected->dispatch_char(ch))
	    repaint();
	}
    }
}

void vs_tree::search_for(string s)
{
  if(s.size()==0 && last_search_term.size()==0)
    {
      beep();
      refresh();
    }
  else
    {
      if(s.size()>0)
	last_search_term=s;
      vs_treeiterator curr((selected==vs_treeiterator(NULL))?begin:selected, true);
      // Make an iterator that ignores all the rules >=)

      if(curr!=end && curr->matches(last_search_term))
	curr++;
      while(curr!=end && !curr->matches(last_search_term))
	curr++;

      if(curr==end)
	{
	  beep();
	  refresh();
	}
      else
	{
	  selected=top=vs_treeiterator(curr,false);
	  while(!curr.is_root())
	    {
	      curr=curr.get_up();
	      curr.expand();
	    }
	  repaint();
	}
    }
}

void vs_tree::repaint()
{
  int width,height;
  int selectedln=line_of(selected);

  if(!winavail())
    return;
 
  leaveok(true);
  erase();

  getmaxyx(height,width);

  while(selected!=begin && selectedln>height-2)
    {
      selected--;
      selectedln--;
    }

  bkgd(COLOR_PAIR(get_background_color()));

  show_status();
  show_header();

  vs_treeiterator i=top;
  int y=1;
  while(y<height-1 && i!=end)
    // We have to adjust for the status and header lines, so we count from 1
    // to height-1.  [ FIXME: use subwindows to eliminate this hack ]
    {
      vs_treeitem *curr=&*i;

      if(i==selected && i->get_selectable())
	attrset(curr->get_highlight_attr());
      else
	attrset(curr->get_normal_attr());

      curr->display(this,y);

      i++;
      y++;
    }
  refresh();
}

void vs_tree::show_header()
{
  paint_header();
}

void vs_tree::show_status()
{
  if(status_stack.empty())
    paint_status();
  else
    status_stack.front()->display();
}

void vs_tree::add_status_widget(vs_tree_status *widget, bool front)
{
  if(front)
    status_stack.push_front(widget);
  else
    status_stack.push_back(widget);
  show_status();
}

void vs_tree::remove_status_widget(vs_tree_status *widget)
{
  status_stack.remove(widget);
  show_status();
}

void vs_tree::delete_status_widget(vs_tree_status *widget)
{
  remove_status_widget(widget);
  delete_list.push_front(widget);
}

vscreen_widget *vs_tree::get_focus()
{
  if(status_stack.empty())
    return NULL;
  else
    return status_stack.front();
}

void vs_tree::paint_header()
{
  display_header(header, COLOR_PAIR(get_header_color())|A_BOLD);
}

void vs_tree::paint_status()
{
  display_status(status, COLOR_PAIR(get_status_color())|A_BOLD);
}

void vs_tree::set_header(string new_header)
{
  header=new_header;
  show_header();
}

void vs_tree::set_status(string new_status)
{
  status=new_status;
  show_status();
}

bool vs_tree_msg::dispatch_char(chtype ch)
{
  removed();
  get_tree()->delete_status_widget(this);
  return true;
}

void vs_tree_msg::display()
{
  get_owner()->display_status(msg, attr);
}
