/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl.functions;

import com.kvisco.xsl.*;
import com.kvisco.util.List;
import org.w3c.dom.Node;
import com.kvisco.xml.XMLUtil;

/**
 * A implementation of the "system-property" function call
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class SystemFunctionCall extends FunctionCall {
    
    public static final short SYSTEM_PROPERTY    = 1;
    public static final short FUNCTION_AVAILABLE = 2;
    
    private short fType = SYSTEM_PROPERTY;
    
    /**
     * Creates a new SystemFunctionCall using the default
     * function type (system-property())
    **/
    public SystemFunctionCall() {
        super(Names.SYSTEM_PROPERTY_FN);
    } //-- SystemFunctionCall
    
    /**
     * Creates a new SystemFunctionCall 
    **/
    public SystemFunctionCall(short type) {
        super(getFunctionName(type));
        this.fType = type;
    } //-- SystemFunctionCall
    
    /**
     * Evalutes this FunctionCall using the given Node as
     * the context.
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        List params = getParameterList();
        
        if (params.size() != 1)
            throw new InvalidExprException(INVALID_NUMBER_PARAMS+this);
            
        Expr expr = (Expr)params.get(0);
        
        StringResult strResult = 
            StringExpr.toStringResult(expr.evaluate(context, ps));
            
            
        String paramStr = strResult.getValue();
        
        if (fType == FUNCTION_AVAILABLE) {
            String ns = XMLUtil.getNameSpace(paramStr);
            if (ns.length() == 0) ns = null;
            String name = XMLUtil.getLocalPart(paramStr);
            return new BooleanResult(ps.getScriptHandler(name, ns) != null);
        }
        else //-- SystemProperty 
            return new StringResult(ps.getProperty(paramStr));
        
    } //-- evaluate
    
    
    private static String getFunctionName(short type) {
        if (type == FUNCTION_AVAILABLE)
            return Names.FUNCTION_AVAILABLE_FN;
        else 
            return Names.SYSTEM_PROPERTY_FN;
    } //-- getFunctionName
    
} //-- SystemFunctionCall

