/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The Software is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Software. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import org.w3c.dom.*;

/**
 * Represents the xsl:value-of element.
 * Handles string expressions 
 * Section 2.7.13.2 of the W3C XSL Working Draft 1.0 (19981216)
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
class ValueOf extends EmptyXSLObject {
    
    /**
     * the StringExpr for this ValueOf
    **/
    private StringExpr stringExpr = null;
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates a new ValueOf 
    **/
    protected ValueOf(XSLStylesheet parentStylesheet)
        throws XSLException
    {
        super(parentStylesheet, XSLObject.VALUE_OF);
    } //--ValueOf
    
      //------------------/
     //- Public Methods -/
    //------------------/
    
    public void setAttribute(String name, String value)
        throws XSLException 
    {
        if (Names.SELECT_ATTR.equals(name)) {
            super.setAttribute(name, value);
            try {
                stringExpr = ExpressionParser.createStringExpr(value);
            }
            catch(InvalidExprException iee) {
                String err = "Invalid Select Expression: " + iee.getMessage();
                err += "\n -- expression: " + value;
                throw new XSLException(err);
            }
        }
    }
      //---------------------/
     //- Protected Methods -/
    //---------------------/
    
    protected StringExpr getStringExpr() {
        return this.stringExpr;
    } //-- getStringExpr
    
    /**
     * Retrieves the value of the node (Element or Attribute) 
     * matched by the pattern of this ValueOf Object.
     * @param context the node to find "relative" matches from.
     * @param rpState the ProcessorState for accessing the current processing
     * environment
     * @return the String value of the selected node.
    **/
    protected String getValue(Node context, ProcessorState rpState) {
        if (stringExpr == null) return "";
        else return stringExpr.getValue(context,rpState);
        
    } //-- getValue
    

    /**
     * Sets the StringExpr used with this ValueOf.
     * @param stringExpr the StringExpr to use with this 
     * ValueOf
    **/
    protected void setStringExpr(StringExpr stringExpr) {
        this.stringExpr = stringExpr;
        try {
            setAttribute(Names.SELECT_ATTR,stringExpr.toString());
        }
        catch(XSLException xslException) {};
    } //-- setStringExpr
    
} //-- ValueOf
