/*
 * (C) Copyright Keith Visco 1998, 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by 
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or 
 * lost profits even if the Copyright owner has been advised of the 
 * possibility of their occurrence.  
 */
package com.kvisco.xsl;

import com.kvisco.xml.*;
import com.kvisco.xml.parser.DOMPackage;
import com.kvisco.xsl.util.*;
import com.kvisco.util.*;
import org.mitre.tjt.xsl.XslNumberFormat;

//-- 3rd Party Classes
import org.w3c.dom.*;

//-- JDK Classes
import java.io.PrintWriter;
import java.io.Writer;
import java.io.OutputStream;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Properties;


/**
 * This class is the "template" rule processor which processes an XML document 
 * using it's XSLStylesheet. It's primary method #process starts with the 
 * "document" rule ("/"), specified or default, and runs through the XML 
 * Source tree processing all rules until there is nothing left to process. 
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class RuleProcessor {

   
    /**
     * The XSL version property name
    **/
    public static final String XSL_VERSION = "version";

    /**
     * The XSL vendor property name
    **/
    public static final String XSL_VENDOR = "vendor";

    /**
     * The XSL vendor-url property name
    **/
    public static final String XSL_VENDOR_URL = "vendor-url";
    
    /**
     * Rule Conflict error message
    **/
    private static final String RULE_CONFLICT 
        = "There is a rule conflict between the following two " + 
          "template match expressions:";

    /**
     * the name of the result tree document element
    **/
    private final String NS_SEP = ":";

    /**
     * PrintWriter to printer errors to
    **/
    private PrintWriter errorWriter      = new PrintWriter(System.out,true);
        
    /** 
     * The stylesheet that this RuleProcessor will process
    **/
    private XSLStylesheet stylesheet     = null;
    
    /**
     * Handle to all the rules of the stylesheet
    **/
    private TemplateRule[] allRules      = null;
    
    /**
     * The default rule
    **/
    private TemplateRule defaultRule     = null;
    
    /**
     * The default action of the default rule
    **/
    private Selection selectChildren     = null;
    
    /**
     * The DOM Package to use for creating the Result Tree
    **/
    private DOMPackage domPackage          = null;
    
    
    private Properties xslSysProps         = null;
    
    /**
     * StringBuffer used to create error messages
    **/
    private StringBuffer errMsgBuffer      =  null;
    
    
    /**
     * The list of message observers
    **/
    List msgObservers = null;
    
    
    /**
     * Create a RuleProcessor for the given XSL stylesheet
     * @param xsl the XSLStylesheet to process
     * @param domPackage the DOMPackage to use when creating
     * the result tree
    **/
	public RuleProcessor(XSLStylesheet xsl, DOMPackage domPackage) {
	    super();
	    this.stylesheet = xsl;
	    this.domPackage = domPackage;
	    errMsgBuffer = new StringBuffer();
	    msgObservers = new List(1);
	    
	    // Create DefaultRule
	    defaultRule = new TemplateRule(xsl);
	    selectChildren = new Selection(xsl,XSLObject.APPLY_TEMPLATES);
	    defaultRule.appendAction(selectChildren);
	    
	    //-- set xsl propertites
	    xslSysProps = new Properties();
	    xslSysProps.put(XSL_VERSION, "1.0");
	    xslSysProps.put(XSL_VENDOR, "XSL:P Open Software Group");
	    xslSysProps.put(XSL_VENDOR_URL, "http://xslp.kvisco.com");
	    
	    
        // assign rules
        allRules = xsl.getTemplates();
    } //-- RuleProcessor
    
    /**
     * Adds the given MessageObserver to this processors list
     * of MessageObservers
     * @param msgObserver the MessageObserver to add to this processors
     * list of MessageObservers
    **/
    public void addMessageObserver(MessageObserver msgObserver) {
        msgObservers.add(msgObserver);
    } //-- addMessageObserver
    
    /**
     * Returns the property value that is associated with the given name.
     * @return the property value that is associated with the given name.
    **/
    public String getProperty(String name) {
        return xslSysProps.getProperty(name, "");
    } //-- getProperty
    
    /**
     * Processes the given XML Document using this processors stylesheet
     * @param xmlDocument the Document to process
     * @return the result tree as a DOM NodeList
     * @see process
    **/
    public Document process(Document xmlDocument) {
        
        if (xmlDocument == null) {
            errorWriter.println("Null document, returning empty result");
            return domPackage.createDocument();
        }
        Hashtable idRefs = new Hashtable(200, (float)0.5);
        Element root = xmlDocument.getDocumentElement();
        if (root != null) {
            addIDReferences(root,stylesheet.getIds(), idRefs);
        }
        return process(xmlDocument, idRefs);
        
    } //-- process
    
    /**
     * Removes the given MessageObserver from this processors list
     * of MessageObservers
     * @param msgObserver the MessageObserver to remove from this processors
     * list of MessageObservers
     * @return the given MessageObserver if it was removed from the list,
     * otherwise return null
    **/
    public MessageObserver removeMessageObserver
        (MessageObserver msgObserver) 
    {
        if (msgObservers.remove(msgObserver)) return msgObserver;
        else return null;
    } //-- addMessageObserver
    
	/**
	 * Sets the Writer for this RuleProcessor to print error
	 * messages to
	 * @param stream the OutputStream to print error messages to.
	**/
	public void setErrorWriter(OutputStream stream) {
	    this.errorWriter = new PrintWriter(stream, true);
	} //-- setErrorStream
	
	/**
	 * Sets the Writer for this RuleProcessor to print error
	 * messages to
	 * @param writer the Writer to print error messages to.
	**/
	public void setErrorWriter(Writer writer) {
	    this.errorWriter = new PrintWriter(writer, true);
	} //-- setErrorStream
	
	/**
	 * Retrieves the PrintWriter that this RuleProcessor prints
	 * it's error messages to.
	 * @return the PrintWriter that this RuleProcessor prints
	 * it's error messages to.
	**/
	public PrintWriter getErrorWriter() {
	    return this.errorWriter;
	} //-- getErrorWriter
	
      //---------------------/
     //- Protected Methods -/
    //---------------------/

    /**
     * Binds the given xsl:variable to the top most variable set scope
     * within ProcessorState using the variable name attribute
     * @param the xsl variable to bind
     * @param context the current context node for evaluating the variable's
     * template
     * @param ps the current ProcessorState
    **/
    protected void bindVariable 
        (Variable var, Node context, ProcessorState ps) 
    {
        VariableSet varSet = (VariableSet)ps.getVariableSets().peek();
        
        ExprResult result = null;
        
        // check for param variable 
        if (var.getType() == XSLObject.PARAM_VARIABLE) {
            VariableSet params = (VariableSet)ps.getParameterStack().peek();
            if (params != null) {
                result = params.get(var.getName());
            }
        }
        if (result == null) result = processVariable(var, context, ps);
        //-- check for duplicate variable names
            /* add later */
        //-- add variable to VariableSet
        if (result != null) varSet.put(var.getName(), result);
        
    } //-- bindVariable

    
    /**
     * Processes the given XML Document using this processors stylesheet
     * and the given ID reference table
     * @param xmlDocument the Document to process
     * @param idRefs the ID references of the given xmlDocument
     * @return the result tree as a NodeList
    **/
    protected Document process(Document xmlDocument, Hashtable idRefs) {
        
        if (xmlDocument == null) 
            return domPackage.createDocument();
        
        //-- Performance Testing Setup
        /* *
        Runtime rt = Runtime.getRuntime();
        rt.gc();
        System.out.println("Free Memory: " + rt.freeMemory());
        long stime = System.currentTimeMillis();
        /* */
        
        // initialize...
        ProcessorState rpState 
            = new ProcessorState(this, xmlDocument, stylesheet,domPackage);
            
        // add IDRefs to the processorEnv
        rpState.setIDReferences(idRefs);
            
        Document resultDoc = rpState.getResultDocument();
        QuickStack nodeStack = rpState.getNodeStack();
        
        // Process global scripts
        List scripts = stylesheet.getScripts();
        Object obj;
        ScriptHandler scriptHandler = null;
        String language = null;
        String scriptNS = null;
        XSLScript xslScript;
        for (int i = 0; i < scripts.size(); i++) {
            xslScript = (XSLScript)scripts.get(i);
            language = xslScript.getAttribute(Names.LANGUAGE_ATTR);
            scriptHandler = rpState.getScriptHandler(language);
            if (scriptHandler != null) {
                
                //-- create namespace
                scriptNS = xslScript.getAttribute(Names.NS_ATTR);
                if (scriptNS != null) 
                    scriptHandler.createNamespace(scriptNS);
                    
                //-- evaluate script
                obj = scriptHandler.eval(xslScript, xmlDocument, scriptNS);
                if (obj != null) errorWriter.print(obj);
            }
        }
        
    	// Process the Root rule
    	TemplateRule rootRule = null;
    	try {
    	    rootRule = findTemplate(allRules, xmlDocument, null, rpState);
    	}
    	catch (Exception iee) {
    	    errorWriter.println(iee.getMessage());
    	    errorWriter.flush();
    	}
    	try {
        	if (rootRule != null) {
        		processTemplate(xmlDocument, rootRule.getActions(), rpState);
        	}
        	else { // use default root rule
        		processTemplate(xmlDocument, defaultRule.getActions(), rpState);
        	}
    	}
    	catch(Exception ex) {
    		ex.printStackTrace(errorWriter);
    	}
    	//-- Performance Testing
    	/* *
    	long etime = System.currentTimeMillis();
    	System.out.println("RuleProcessor Time: " + (etime - stime) + "(ms)");
    	/* */
        return resultDoc;  
        
    } //-- process

    
    
    /** 
     * Finds a TemplateRule that matches the given xml node
     * @param rules the set of TemplateRules to search
     * @param node the XML Node to find a rule for.
     * @param mode the current processing mode for template rules
     * @param rpState the current ProcessorState
     * @return the matching TemplateRule, or null if no match is found
    **/
    protected TemplateRule findTemplate
        (TemplateRule[] rules, Node node, String mode, ProcessorState rpState)
        throws InvalidExprException, XSLException
    {
        
		TemplateRule matchingRule = null;
		
		// Find the matching rule with the hightest priority
		for (int i = 0; i < rules.length; i++) {
		    TemplateRule rule = rules[i];
			if (!rule.matches( node, node, rpState)) continue;
			    
			// compare modes
			String rMode = rule.getMode();
			if ((rMode == null) && (mode != null)) continue;
			if ((rMode != null) && (mode == null)) continue;
			if ((rMode != null) && (mode != null)) {
			    if (! rMode.equals(mode)) continue;
			}
			    
			// if passed modes look for highest priority
			    
			if (matchingRule == null) matchingRule = rule;
			else {
			    
			    float oldp = matchingRule.calculatePriority(node, rpState);
			    float newp = rule.calculatePriority(node, rpState);
			    
			    if (oldp < newp) matchingRule = rule;
			    else if (oldp == newp) {
					    
					if (matchingRule.getParentStylesheet() == 
					        rule.getParentStylesheet()) 
					{
						StringBuffer err = new StringBuffer(RULE_CONFLICT);
						err.append("\n 1." + matchingRule.getMatchExpr());
						err.append(" mode='" + matchingRule.getMode() + "' {");
						err.append(matchingRule.getParentStylesheet().getHref());
						err.append("}\n 2." + rule.getMatchExpr());
						err.append(" mode='" + rule.getMode() + "' {");
						err.append(rule.getParentStylesheet().getHref());
						err.append("}\n");
						throw new XSLException(err.toString());
					}
					else matchingRule = rule;
				}
			}
		}
		
		/* <debug>
		if (matchingRule != null)
		    System.out.println("Match: " + matchingRule.getMatchExpr());
		else System.out.println("no match");
		/* </debug> */
		
		return matchingRule;   
    } //-- findTemplate
    
      //-------------------/
     //- Private Methods -/
    //-------------------/
    
    /**
     * Binds the given xsl:variable by Processing it's Expr or Template
     * with the given context and state     
     * @param the xsl variable to bind
     * @param context the current context node for evaluating the variable's
     * template
     * @param ps the current ProcessorState
     * @return the new ExprResult
    **/
    protected ExprResult processVariable
        (Variable var, Node context, ProcessorState ps) 
    {
        QuickStack nodeStack = ps.getNodeStack();
        Document resultDoc = ps.getResultDocument();
        
        String exprAtt = var.getAttribute(Names.EXPR_ATTR);
        ExprResult exprResult = null;
        if ((exprAtt != null) && (exprAtt.length() > 0)) {
            try {
                Expr expr = ExpressionParser.createExpr(exprAtt);
                exprResult = expr.evaluate(context, ps);
            }
            catch(InvalidExprException iee) {
                errMsgBuffer.setLength(0); 
                errMsgBuffer.append("variable binding error: '");
                errMsgBuffer.append(var.getName());
                errMsgBuffer.append("' - ");
                errMsgBuffer.append(iee.getMessage());
                errorWriter.println(errMsgBuffer.toString());
            }
        }
        else {
            Node node = resultDoc.createDocumentFragment();
            nodeStack.push(node);
            List list = var.getActions();
            processTemplate(context, var.getActions(), ps);
            nodeStack.pop();
            NodeList nl = node.getChildNodes();
            if (nl.getLength() == 1)
                exprResult = new TreeFragmentResult(nl.item(0));
            else
                exprResult = new TreeFragmentResult(node);
        }
        //-- check for duplicate variable names
            /* add later */
        //-- add variable to VariableSet
        
        return exprResult;
    } //-- processVariable
     
    /**
     * processes the list of attributes for the given literal 
     * element and adds the attributes to the element currently
     * being processed
    **/
    private void processAttributes
        (XSLObject xslObject, Node context, ProcessorState rpState) 
    {
        Node node = (Node)rpState.getNodeStack().peek();
        if (node.getNodeType() != Node.ELEMENT_NODE) return;
        
        Element element = (Element)node;
        
        Enumeration attNames = xslObject.getAttributeNames();
        
	    AttributeValueTemplate avt = null;
	    String name, value;
	    
	    while (attNames.hasMoreElements()) {
	        name = (String)attNames.nextElement();
	        try {
                avt = xslObject.getAttributeAsAVT(name);
            }
            catch (XSLException ex) {
                avt = null;
            }
            if (avt != null) {
                value = avt.evaluate(context, rpState);
                value = Whitespace.stripSpace(value, true, true);
            }
            else value = "";
            element.setAttribute(name,value);
        }

    } //--processAttributes
    
    
    /**
     * Processes the Parameters for the given XSLObject
     * @param xslObject the XSLObject to process the parameters of
     * @param context, the current Node context in which to process the
     * parameters
     * @param ps the current ProcessorState
     * @return the VariableSet represting the parameters of the given
     * XSLObject
    **/
    private VariableSet processParameters
        (XSLObject xslObject, Node context, ProcessorState ps) {
        
        VariableSet varSet = new VariableSet();
        
        List actions = xslObject.getActions();
        for (int i = 0; i < actions.size(); i++) {
            XSLObject action = (XSLObject) actions.get(i);
            if (action.getType() == XSLObject.PARAM) {
                Variable var = (Variable)action;
                varSet.put(var.getName(), processVariable(var,context, ps));
            }
        }
        return varSet;
        
    } //-- processParameters
    
    /**
     * processes the given XML Node using the desired template 
     * (NodeList).
     * @param xmlNode the XML Node to process with the given rule
     * @param template the desired NodeList to use for processing
    **/
    private void processTemplate
        (Node xmlNode, List template, ProcessorState rpState) 
    {
        
        
        if (xmlNode.getNodeType() == Node.ELEMENT_NODE) {
            //-- check whitespace att
            String xmlSpace 
                = ((Element)xmlNode).getAttribute(Names.XMLSPACE_ATTR);
            if ((xmlSpace != null) && (xmlSpace.length() > 0))
                rpState.getXMLSpaceModes().push(xmlSpace.intern());
        }
        
        if (template != null) {
            
            //-- push a new variableSet onto stack
            QuickStack vars = rpState.getVariableSets();
            vars.push(new VariableSet(5));
            for (int i = 0; i < template.size(); i++) {
                XSLObject xslObject = (XSLObject) template.get(i);
                processAction(xslObject, xmlNode, null, rpState);
            }
            vars.pop();
        }
        else {
            // Do Built-in Rules
            if (xmlNode == null) return;
            
            switch (xmlNode.getNodeType()) {
                case Node.ELEMENT_NODE:
                    //...simply process all children
    			    processAction(selectChildren,xmlNode,null, rpState);
    			    break;
    		    case Node.CDATA_SECTION_NODE:
    		        // add CDATASection 
    		        String data = ((Text)xmlNode).getData();
    		        Document resultDoc = rpState.getResultDocument();
    		        CDATASection cdata = resultDoc.createCDATASection(data);
    		        rpState.addToResultTree(cdata);
    		        break;
    		    case Node.TEXT_NODE:
    		        // add Text Node to Tree
    		        if (isStripSpaceAllowed(xmlNode, rpState))
    		            data = stripSpace((Text)xmlNode);
    		        else data = ((Text)xmlNode).getData();
    		            
    		        if ((data != null) && (data.length() > 0)) {
    		            resultDoc = rpState.getResultDocument();
    		            Text text = resultDoc.createTextNode(data);
        		        rpState.addToResultTree(text);
        		    }
			        break;
			    default:
			        // do nothing;
			        break;
			}
		}
        
    } //-- processTemplate
        
    
    /**
     * Processes the given XSLNumber for the current element
    **/
    private void processXSLNumber
        (Element current, XSLNumber xslNumber, ProcessorState rpState) 
    {
        String result = xslNumber.getFormattedNumber(current, rpState);
        rpState.addToResultTree(rpState.getResultDocument().createTextNode(result));
    } //-- processXSLNumber
    

    /**
     * processes the given Action, for the current XML Node using the
     * specified template (NodeList of actions)
    **/
	private void processAction
	    (XSLObject xslObject, Node xmlNode, 
	     NodeList template, ProcessorState rpState ) 
	{
	    
	    //-- <performance-testing>
	    //long stime = System.currentTimeMillis();
	    //-- </performance-testing>
	    
	    // get variables from ProcessorState
	    Document resultDoc     = rpState.getResultDocument();
	    QuickStack nodeStack   = rpState.getNodeStack();
	    Hashtable idRefs       = rpState.getIDReferences();
	    
	    Element result           = null;
	    Node node                = null;
	    XSLObject child          = null;
	    String name              = null;
	    String value             = null;
	    List childActions        = null;
	    TemplateRule[] templates = allRules;
	    
	    NodeSet matches = null;
	    Selection selection;
	    
        switch (xslObject.getType()) {
            // <xsl:apply-templates>
            case XSLObject.APPLY_IMPORTS:
                templates = xslObject.getParentStylesheet().getTemplates();
                /* do not break here */
            case XSLObject.APPLY_TEMPLATES:
            { //-- add local scoping 
            
                selection = (Selection)xslObject;
                
                //-- process parameters
                VariableSet params = 
                    processParameters(xslObject, xmlNode, rpState);
                    
                try {
                    matches = selection.selectNodes(xmlNode, rpState);
                }
                catch(InvalidExprException iee) {
                    errorWriter.println(iee.getMessage());
                    errorWriter.flush();
                    break;
                }
                // sort nodes if necessary
                if (matches.size() > 0) {
                    XSLSort[] sortKeys = selection.getSortElements();
                    if (sortKeys.length > 0) {
                        matches = NodeSorter.sort(matches,sortKeys,xmlNode,rpState);
                    }
                }
                String mode = selection.getAttribute(Names.MODE_ATTR);
                
                rpState.getParameterStack().push(params);
                rpState.getNodeSetStack().push(matches);
                for (int j = 0; j < matches.size(); j++) {
                    node = matches.get(j);
                    List actions = null;
                    TemplateRule tr = null;
                    try {
                        tr = findTemplate(templates,node,mode,rpState);
                    }
                    catch (Exception iee) {
                        errorWriter.print("error finding template: "); 
                        errorWriter.println(iee);
                        errorWriter.flush();
                    }
                    if (tr != null) actions = tr.getActions();
                    if (node != xmlNode) {
                        processTemplate(node,actions,rpState);
                    }
                    else {
                        // make sure we do not use same rule: otherwise
                        // error...
                        // find current rule
                        XSLObject xsle = 
                            selection.getNearestAncestor(XSLObject.TEMPLATE);
                        if (tr != xsle) 
                            processTemplate(node, tr.getActions(),rpState);
                    }
                }
                rpState.getParameterStack().pop();
                rpState.getNodeSetStack().pop();
                break;
            }
            
			// <xsl:call-template>
			// Peter Ciuffetti
			// Added for WD-xslt-19990421
			case XSLObject.CALL_TEMPLATE:
			{ 
			
			
			    //-- added parameter support (kav)
			    VariableSet params = 
			        processParameters(xslObject, xmlNode, rpState);
			        
			    rpState.getParameterStack().push(params);
			    //-- process named template
			    name = xslObject.getAttribute(Names.NAME_ATTR);
			    TemplateRule tr = stylesheet.getNamedTemplate(name);
			    if (tr != null) {
			        processTemplate(xmlNode, tr.getActions(), rpState);
                } 
                else {
			        errorWriter.print("No named template found: ");
			        errorWriter.println(name);
			    }
			    //-- remove parameters from stack
			    rpState.getParameterStack().pop();
			    break;
			}
            // <xsl:for-each>
			case XSLObject.FOR_EACH:
                selection = (Selection)xslObject;
                //process each select pattern
                try {
                    matches = selection.selectNodes(xmlNode, rpState);
                }
                catch(InvalidExprException iee) {
                    errorWriter.print(iee.getMessage());
                    break;
                }
                // sort nodes if necessary
                if (matches.size() > 0) {
                    XSLSort[] sortKeys = selection.getSortElements();
                    if (sortKeys.length > 0) {
                        matches = NodeSorter.sort(matches, sortKeys, xmlNode, rpState);
                    }
                }
                rpState.getNodeSetStack().push(matches);
                for (int j = 0; j < matches.size();j++) {
                    node = matches.get(j);
                    if (node != xmlNode) 
                        processTemplate(node, selection.getActions(), rpState);
                }
                rpState.getNodeSetStack().pop();
                break;
            // <xsl:if>
			case XSLObject.IF:
			{
                Conditional conditional = (Conditional)xslObject;
                BooleanResult br = null;
                try {
                    br = conditional.evaluate(xmlNode, rpState);
                }
                catch(InvalidExprException iee) {
                    errorWriter.print(iee.getMessage());
                    break;
                }
                if (br.booleanValue()) {                        
                    processTemplate(xmlNode, xslObject.getActions(), rpState);
                }
                break;
            }
            // <xsl:choose>
            case XSLObject.CHOOSE:
            {
		        // find first matching xsl:when or xsl:otherwise
		        List actions = xslObject.getActions();
		        BooleanResult br = null;
		        for (int i = 0; i < actions.size(); i++) {
		            XSLObject xslWhen = (XSLObject)actions.get(i);
		            Conditional conditional = (Conditional)xslWhen;
                    try {
                        br = conditional.evaluate(xmlNode, rpState);
                    }
                    catch(InvalidExprException iee) {
                        errorWriter.print(iee.getMessage());
                        break;
                    }
                    if (br.booleanValue()) {                        
                        processTemplate(xmlNode, xslWhen.getActions(), rpState);
                        break;
                    }
		        } //-- end for each child selection
                break;
            }
            // <xsl:message>
            case XSLObject.MESSAGE:
            {
                DocumentFragment dfrag = resultDoc.createDocumentFragment();
                nodeStack.push(dfrag);
                processTemplate(xmlNode, xslObject.getActions(), rpState);
                nodeStack.pop();
                value = XSLObject.getNodeValue(dfrag);
                for (int i = 0; i < msgObservers.size(); i++) {
                    ((MessageObserver)msgObservers.get(i)).recieveMessage(value);
                }
                break;
            }
		    // <xsl:number>
		    case XSLObject.NUMBER: 
	    	    processXSLNumber( (Element)xmlNode, (XSLNumber)xslObject, rpState);
    		    break;
	    	// <xsl:value-of>
    		case XSLObject.VALUE_OF:
		        value = ((ValueOf)xslObject).getValue(xmlNode, rpState);
		        
		        if (isStripSpaceAllowed(xmlNode, rpState))
		            value = Whitespace.stripSpace(value, true, true);
		            
		        if ((value != null) && (value.length() > 0))
		            rpState.addToResultTree(resultDoc.createTextNode(value));
		        //-- clean up value for gc
		        value = null;
			    break;
			// <xsl:variable>
			case XSLObject.VARIABLE:
			case XSLObject.PARAM_VARIABLE:
			    bindVariable((Variable)xslObject, xmlNode, rpState);
			    break;
			
			// <xsl:attribute>
			// <xsl:comment>
			// <xsl:copy>
			// <xsl:entity-ref> PROPRIETARY EntityReference creation
			// <xsl:element>
			// <xsl:copy>
			// <xsl:pi>
			case XSLObject.ATTRIBUTE:
			case XSLObject.COMMENT:
			case XSLObject.COPY:
			case XSLObject.ELEMENT:
			case XSLObject.PI:
			case XSLObject.ENTITY_REF:
			    try {
    			    node = createNode(xmlNode, xslObject, rpState);
    			    if (node != null) rpState.addToResultTree(node);
			    }
			    catch(XSLException ex) {
			        errorWriter.println(ex);
			    }
			    break;
			// <xsl:copy-of>
		    case XSLObject.COPY_OF:
		        CopyOf copyOf = (CopyOf)xslObject;
			    try {
			        NodeSet nodes = copyOf.selectNodes(xmlNode, rpState);
			        for (int i = 0; i < nodes.size(); i++) {
			            node = copyNode(resultDoc, nodes.get(i), true);
			            rpState.addToResultTree(node);
			        }
			    }
			    catch(InvalidExprException ex) {
			        errorWriter.println(ex);
			    }
			    break;
			// <xsl:text>
            case XSLObject.TEXT:
            { // local scope data
                String data = ((XSLText)xslObject).getData();
                rpState.addToResultTree(resultDoc.createTextNode(data));
                break;
            }
            case XSLObject.CDATA:
            { // local scope data
                String data = ((XSLCData)xslObject).getData();
                rpState.addToResultTree(resultDoc.createCDATASection(data));
                break;
            }
			// <xsl:use>
			case XSLObject.USE:
			    name = xslObject.getAttribute(Names.ATTRIBUTE_SET_ATTR);
			    if ((name != null) && (name.length() > 0)) {
			        AttributeSet attSet = stylesheet.getAttributeSet(name);
			        if (attSet != null) {
			            processTemplate((Node)nodeStack.peek(),
			                attSet.getActions(), rpState);
			        }
			    }
			    else {
			        //-- error: missing attribute-set attribute
			    }
			    break;
    		// <xsl:script>
    		// Scripting is proprietary
    		case XSLObject.SCRIPT:
    		    XSLScript xslScript = (XSLScript)xslObject;
    		    String language = xslScript.getAttribute(Names.LANGUAGE_ATTR);
    		    ScriptHandler scriptHandler = rpState.getScriptHandler(language);
    		    if (scriptHandler != null) {
    			    Object obj = scriptHandler.evalAsFunction(xslScript, xmlNode);
    			    if (obj != null)
                        rpState.addToResultTree(resultDoc.createCDATASection(obj.toString()));
                }
                else {
                    errorWriter.print("Warning no ScriptHandler found for language: ");
                    errorWriter.println(language);
                    errorWriter.flush();
                }
    			break;
		    // XML Literals, Formatting Objects
    		case XSLObject.LITERAL:
    		    name = getResolvedName(xslObject.getTypeName());
		    	result = resultDoc.createElement(name);
        		    	    
		    	rpState.addToResultTree(result);
		    	nodeStack.push(result);
		    	processAttributes(xslObject, xmlNode, rpState);
		    	processTemplate(xmlNode, xslObject.getActions(), rpState);
			    // Process the children
			    //childActions = xslObject.getActions();
			    //for (int i = 0; i < childActions.size(); i++) {
			    //    XSLObject action = (XSLObject)childActions.get(i);
				//    processAction(action, xmlNode,template, rpState );
			    //}
			    nodeStack.pop();
				break;
		    default:
		        break;
		} //-- end switch
	
	} //-- processAction
	
	
	/**
	 * Looks for element ids in the given element and adds the element to the
	 * reference table if necessary.
	**/
	private void addIDReferences(Element element, Hashtable idNames, Hashtable idRefs) {
    	
    	if (element == null) return;
    	
	    NodeList nl = element.getChildNodes();
	    Node node;
	    Element el;
	    Enumeration keys;
	    for (int i = 0; i < nl.getLength(); i++) {
	        node = nl.item(i);
	        if (node.getNodeType() == Node.ELEMENT_NODE) {
	            el = (Element) node;
	            keys = idNames.keys();
	            String type;
	            String id;
	            String idValue;
	            while (keys.hasMoreElements()) {
	                id = (String) keys.nextElement();
	                type = (String) idNames.get(id);
	                if ((type.equals(Names.WILD_CARD)) || 
	                    (type.equals(el.getNodeName()))) {
                        // Check for empty string to solve changes from 
                        // XML4J 1.1.9 to 1.1.14 as suggested by Domagoj Cosic.
	                    // old version: if ((idValue = el.getAttribute(id)) != null)
	                    idValue = el.getAttribute(id);
	                    if ((idValue != null) && (idValue.length() > 0)) {
	                        if (idRefs.get(idValue) != null) {
	                            errorWriter.print("warning: multiple elements with the same id value: ");
	                            errorWriter.println(idValue);
	                            errorWriter.println(" -- processing will continue with first occurance only.");	                            
	                        }
	                        else idRefs.put(idValue, el);
	                    }
	                }
	            }
	            // add Child refs
	            addIDReferences(el, idNames, idRefs);
	        } // end if Element Node
	    } // end for (each child)
	    
	} //-- AddIDReferences
    
    /**
     * Copies the Node based on the rules in the XSL WD for xsl:copy
     * and xsl:copy-of
    **/
    private Node copyNode(Document ownerDoc, Node node, boolean deep) {
        
        Node newNode = null;
        String data;
        CharacterData charData;
        switch (node.getNodeType()) {
            case Node.DOCUMENT_NODE:
                break;
            case Node.ATTRIBUTE_NODE:
                Attr srcAttr = (Attr)node;
                newNode = ownerDoc.createAttribute(srcAttr.getName());
                ((Attr)newNode).setValue(srcAttr.getValue());
                break;
            case Node.CDATA_SECTION_NODE:
                data = ((CharacterData)node).getData();
                newNode = ownerDoc.createCDATASection(data);
                break;
            case Node.COMMENT_NODE:
                data = ((CharacterData)node).getData();
                newNode = ownerDoc.createComment(data);
                break;
            case Node.DOCUMENT_FRAGMENT_NODE:
                newNode = ownerDoc.createDocumentFragment();
                if (deep) {
                    NodeList nl = node.getChildNodes();
                    for (int i = 0; i < nl.getLength(); i++) {
                        Node child = copyNode(ownerDoc, nl.item(i), deep);
                        newNode.appendChild(child);
                    }
                }
                break;
            case Node.ELEMENT_NODE:
                Element srcElement = (Element)node;
                newNode = ownerDoc.createElement(srcElement.getNodeName());
                if (deep) {
                    //-- copy attribute nodes
                    Element el = (Element)newNode;
                    NamedNodeMap atts = srcElement.getAttributes();
                    if (atts != null) {
                        for (int i = 0; i < atts.getLength(); i++) {
                            Attr attr = (Attr)atts.item(i);
                            el.setAttribute(attr.getName(), attr.getValue());
                        }
                    }
                    //-- copy child nodes
                    NodeList nl = node.getChildNodes();
                    for (int i = 0; i < nl.getLength(); i++) {
                        Node child = copyNode(ownerDoc, nl.item(i), deep);
                        newNode.appendChild(child);
                    }
                }
                break;
            case Node.ENTITY_REFERENCE_NODE:
                newNode = ownerDoc.createEntityReference(node.getNodeName());
                break;
            case Node.TEXT_NODE:
                data = ((CharacterData)node).getData();
                newNode = ownerDoc.createTextNode(data);
                break;
            case Node.PROCESSING_INSTRUCTION_NODE:
                ProcessingInstruction srcPI = (ProcessingInstruction)node;
                newNode = ownerDoc.createProcessingInstruction (
                                srcPI.getTarget(), srcPI.getData());
                break;
            default:
                break;
        }
        return newNode;
    }
    
    /**
     * Creates a new Node based on the type of XSLObject
     * @param context the DOM Node to use as the current context for
     * evaluating any expressions or scripts
     * @return the new Node
    **/
    private Node createNode
        (Node context, XSLObject xslObject, ProcessorState rpState) 
        throws XSLException 
    {
        
        // ProcessorState variables
        Document resultDoc = rpState.getResultDocument();
        QuickStack nodeStack = rpState.getNodeStack();
        
        String  name    = null;
        String  value   = null;
        Node    node    = null;
        Element element;
        
        
        AttributeValueTemplate nodeName = null;
        
        nodeName = xslObject.getAttributeAsAVT(Names.NAME_ATTR);
        
        if (nodeName != null)
            name = nodeName.evaluate(context,rpState);
        else name = "";
            
        
        
        // get value
        int type = xslObject.getType();
        if ((type != XSLObject.COPY) ||
            (type != XSLObject.ELEMENT)) {
            element = resultDoc.createElement(xslObject.getTypeName());
            nodeStack.push(element);
            processTemplate(context,xslObject.getActions(), rpState);
            nodeStack.pop();
            value = XSLObject.getText(element);
        }
        
        switch (type) {
            // xsl:attribute
            case XSLObject.ATTRIBUTE:
                node = (Node)nodeStack.peek();
                if (node.getNodeType() != Node.ELEMENT_NODE) break;
                element = (Element) node;
                if (element.hasChildNodes()) return null;
                value = Whitespace.stripSpace(value);
                node = resultDoc.createAttribute(name);
                ((Attr)node).setValue(value);
                break;
            // xsl:comment
            case XSLObject.COMMENT:
                node = resultDoc.createComment(value);
                break;
            // xsl:copy
            case XSLObject.COPY:
                node = copyNode(resultDoc, context, false);
                if (node.getNodeType() == Node.ELEMENT_NODE) {
                    nodeStack.push(node);
                    processTemplate(context,xslObject.getActions(), rpState);
                    nodeStack.pop();
                }
                break;
            // xsl:element
            case XSLObject.ELEMENT:
                node = resultDoc.createElement(name);
                nodeStack.push(node);
                processTemplate(context,xslObject.getActions(), rpState);
                nodeStack.pop();
                break;
            // xsl:entity-ref
            case XSLObject.ENTITY_REF: // Proprietary to XSL:P
                node = resultDoc.createEntityReference(name);
                break;
            // xsl:pi
            case XSLObject.PI:
                node = resultDoc.createProcessingInstruction(name,value);
                break;
            default:
                break;
        }
        return node;
    } //-- createNode

    /**
     * Returns the name with any quoted namespaces resolved
     * @param elementName the element name to resolve
     * @return the resolved name
    **/
    private String getResolvedName(String elementName) {
	    int idx = 0;
	    if ((idx = elementName.indexOf(NS_SEP)) > 0) {
	        String ns = stylesheet.getQuotedNamespace(elementName.substring(0,idx));
	        return ns + elementName.substring(idx);
	    }
	    return elementName;
    } //-- getResolvedName
    
    /**
     * Strips Whitespace from the Text Node 
    **/
    private String stripSpace(Text textNode) {
        //-- set strip leading space flag
        boolean stripLWS = (textNode.getPreviousSibling() == null);
        //-- set strip trailing space flag
        boolean stripTWS = (textNode.getNextSibling() == null);
        return Whitespace.stripSpace(textNode.getData(), stripLWS, stripTWS);
    } //-- stripSpace
    
    private boolean isStripSpaceAllowed(Node context, ProcessorState ps) {
        if (context == null) return true;
        if (ps.getXMLSpaceModes().peek() == Names.PRESERVE_VALUE)
            return false;
            
        Node realContext = context;
        switch (context.getNodeType()) {
            case Node.DOCUMENT_NODE:
                return true;
            case Node.ELEMENT_NODE:
                break;
            default:
                realContext = context.getParentNode();
                //-- check for null parent (Ray Powell)
                if (realContext == null) return false;
                if (realContext.getNodeType() != Node.ELEMENT_NODE)
                    return true;
                break;
        }
        return stylesheet.isStripSpaceAllowed(realContext.getNodeName());
    } //-- isStripSpaceAllowed
    
} //-- RuleProcessor

