/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The contents of this file are released under an Open Source 
 * Definition (OSD) compliant license; you may not use this file 
 * execpt in compliance with the license. Please see license.txt, 
 * distributed with this file. You may also obtain a copy of the
 * license at http://www.clc-marketing.com/xslp/license.txt
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * 
 */

package com.kvisco.xsl;

/**
 * Represents a Boolean result
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class BooleanResult implements ExprResult {

    public static final BooleanResult FALSE_RESULT 
        = new BooleanResult(false);
        
    public static final BooleanResult TRUE_RESULT 
        = new BooleanResult(true);
        
    private boolean value = false;
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates a new BooleanResult with the default value
    **/
    public BooleanResult() {
        super();        
    } //-- BooleanResult
    
    /**
     * Creates a new BooleanResult with the given value
     * @param value the boolean to initialize the
     * new BooleanResult to
    **/
    public BooleanResult(boolean value) {
        this.value = value;
    } //-- BooleanResult
    
    /**
     * Creates a new BooleanResult by converting the given
     * ExprResult to a boolean value
     * @param exprResult the ExprResult that will get
     * converted to the boolean with which to initialize
     * the new BooleanResult to.
    **/
    public BooleanResult(ExprResult exprResult) {
        this.value = exprResult.booleanValue();
    } //-- BooleanResult
    
      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Returns the value of this ExprResult as a boolean
     * @return the value of this ExprResult as a boolean
    **/
    public boolean booleanValue() {
        return value;
    } //-- booleanValue
    
    /**
     * Returns true if the the given ExprResult is of the same type and
     * has the same value as this ExprResult
    **/
    public boolean equals(ExprResult exprResult) {
        if (exprResult.getResultType() != ExprResult.BOOLEAN)
            return false;
        return (value  == exprResult.booleanValue());
    } //-- equals
    
    /**
     * Returns the value of this BooleanResult
     * @return the value of this BooleanResult
    **/
    public boolean getValue() {
        return value;
    } //-- getValue
    
    /**
     * Returns the type of this ExprResult
     * @return the type of this ExprResult
     * @see ExprResult
    **/
    public short getResultType() {
        return ExprResult.BOOLEAN;
    } //-- getResultType
    
    /**
     * Returns the value of this ExprResult as a double
     * @return the value of this ExprResult as a double
    **/
    public double numberValue() {
        if (value) return 1.0;
        else return 0.0;
    } //-- numberValue
    
    /**
     * Sets the value of this BooleanResult
     * @param value the boolean value to use as the result value
    **/
    public void setValue(boolean value) {
        this.value = value;
    } //-- setValue
    
    /**
     * Returns the Java Object this ExprResult represents
     * @return the Java Object this ExprResult represents
    **/
    public Object toJavaObject() {
        return new Boolean(value);
    } //-- toJavaObject
    
    /**
     * Returns the value of this ExprResult as a String
     * @return the value of this ExprResult as a String
    **/
    public String toString() {
        return String.valueOf(value);
    } //--toString
    
} //-- BooleanResult