/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The contents of this file are released under an Open Source 
 * Definition (OSD) compliant license; you may not use this file 
 * execpt in compliance with the license. Please see license.txt, 
 * distributed with this file. You may also obtain a copy of the
 * license at http://www.clc-marketing.com/xslp/license.txt
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * 
 */
package com.kvisco.xsl;

import java.util.Hashtable;

import org.w3c.dom.*;

/**
 * Represents an AdditiveExpr
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class AdditiveExpr implements Expr {

    public static final short ADD             = 0;
    public static final short SUBTRACT        = 1;
    
    private Expr leftExpr = null;
    private Expr rightExpr = null;
    
    private short additiveOp = ADD;
    
    private static Hashtable additiveOps = null;
    static {
        additiveOps = new Hashtable(2);
        additiveOps.put(Names.ADDITION_OP,       new Short(ADD));
        additiveOps.put(Names.SUBTRACTION_OP,    new Short(SUBTRACT));
    }
    
      //---------------/
     //- Constructor -/
    //---------------/
    
    /**
     * Creates a new AdditiveExpr using the default operator
     * @param leftSideExpr the Expr that is to be evaluated as
     * the left side of this AdditiveExpr
     * @param rightSideExpr the Expr that is to be evaluated as
     * the right side of this AdditiveExpr
     * <BR><B>Note:</B> the default operator is AdditiveExpr.ADD
    **/
    public AdditiveExpr(Expr leftSideExpr, Expr rightSideExpr) {
        leftExpr = leftSideExpr;
        rightExpr = rightSideExpr;
    } //-- AdditiveExpr 
    
    /**
     * Creates a new AdditiveExpr
     * @param leftSideExpr the Expr that is to be evaluated as
     * the left side of this AdditiveExpr
     * @param rightSideExpr the Expr that is to be evaluated as
     * the right side of this AdditiveExpr
     * @param additiveOp the additive operator for this AdditiveExpr
     * @exception InvalidExprException when the additive operator is 
     * invalid 
    **/
    public AdditiveExpr
        (Expr leftSideExpr, Expr rightSideExpr, short additiveOp) 
        throws InvalidExprException
    {
        leftExpr = leftSideExpr;
        rightExpr = rightSideExpr;
        if ((additiveOp < 0) || (additiveOp > additiveOps.size()))
            throw new InvalidExprException("invalid operator for additive expression");
        this.additiveOp = additiveOp;
    } //-- AdditiveExpr 
    
    /**
     * Creates a new AdditiveExpr
     * @param leftSideExpr the Expr that is to be evaluated as
     * the left side of this AdditiveExpr
     * @param rightSideExpr the Expr that is to be evaluated as
     * the right side of this AdditiveExpr
     * @param additiveOp the additive operator for this AdditiveExpr
     * @exception InvalidExprException when the additive operator is 
     * invalid 
    **/
    public AdditiveExpr
        (Expr leftSideExpr, Expr rightSideExpr, String additiveOp) 
        throws InvalidExprException
    {
        this.additiveOp = -1;
        if (additiveOp != null) {
            Short sval = (Short)additiveOps.get(additiveOp);
            if (sval != null) this.additiveOp = sval.shortValue();
        }
        if ((this.additiveOp < 0) || (this.additiveOp > additiveOps.size()))
            throw new InvalidExprException("invalid operator for additive expression");
        leftExpr = leftSideExpr;
        rightExpr = rightSideExpr;
    } //-- AdditiveExpr
        
      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Returns the type of Expr this Expr represents
     * @return the type of Expr this Expr represents
    **/
    public short getExprType() {
        return Expr.NUMBER;
    } //-- getExprType
    
    /**
     * Evaluates this Expr using the given context Node and ProcessorState
     * @param context the current context Node
     * @param ps the ProcessorState that contains the current processing 
     * environment
     * @return the ExprResult
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        double value = 0;
        
        if ((leftExpr == null) || (rightExpr == null))
            return new NumberResult(Double.NaN);

        ExprResult rResult = null;
        ExprResult lResult = null;
            
        lResult = leftExpr.evaluate(context, ps);
        rResult = rightExpr.evaluate(context, ps);
        
        boolean evalResult = false;
        
        value = rResult.numberValue();
        if (additiveOp == SUBTRACT) value = -value;
        
        value = lResult.numberValue() + value;
        
        return new NumberResult(value);
    } //-- evaluate
    
    public static boolean isAdditiveOperator(String operator) {
        if (operator == null) return false;
        return (additiveOps.get(operator) != null);
    } //-- isAdditiveOperator
    
    /**
     * Returns the String representation of this EqualityExpr
     * @return the String representation of this EqualityExpr
    **/
    public String toString() {
        
        
        StringBuffer sb = new StringBuffer();
        if (leftExpr != null)
            sb.append(leftExpr.toString());
        else
            sb.append("null");
        sb.append(" ");
        
        if (additiveOp == SUBTRACT)
            sb.append(Names.SUBTRACTION_OP);
        else 
            sb.append(Names.ADDITION_OP);
        
        sb.append(" ");
        if (rightExpr != null)
            sb.append(rightExpr.toString());
        else sb.append("null");
        
        return sb.toString();
    } //-- toString
    
} //-- AdditiveExpr