#if !defined(lint)
static char rcs_id[] = "$Id: http-analyze.c,v 1.9.1.16 1996/09/22 13:14:18 stefan Exp $";
#endif

/*
** http-analyze
**	analyzes HTTPD logfiles and creates 2D and 3D summaries
**	by year, month, and day.
**
** Copyright 1996 by Stefan Stapelberg, <stefan@rent-a-guru.de>
**
** ------------------------------------------------------------------
** Since nobody likes to read those very long license agreements,
** I'll try to keep mine as short as possible:
**
** You may use or abuse this software and run the program on any
** (even commercial) WWW installation at your own risk as long as:
**
** 	o you don't claim you have written it
** 	o you don't make money by selling or using this software
**	o you leave all copyright notes intact in all source
**	  files and in all HTML pages which this program produces
**	o you don't remove the link to the homepage of http-analyze
**	  which is created by the program in the HTML files.
**
** ------------------------------------------------------------------
** IT IS STRICTLY FORBIDDEN TO SELL THIS SOFTWARE IN WHOLE OR IN PART
** OR TO INCLUDE IT IN WHOLE OR IN PART IN A COMMERCIAL PRODUCT.
**
** If you plan to run http-analyze on a commercial WWW installation
** and you need support, or if you plan to bundle the program with
** your WWW server product, send an email to office@rent-a-guru.de.
** ------------------------------------------------------------------
**
**
** $Log: http-analyze.c,v $
** Revision 1.9.1.16  1996/09/22  13:14:18  stefan
** Fixed a bug which caused wrong calculation of amount of data
** transferred by the web server. Previous versions did account
** for HEAD requests which do only generate header responses by
** the server (but why the hell do servers then fill in the
** reqsize field in logfiles entries for HEAD requests?!?).
** Removed the background colors for table headings on behalf
** of popular request. The text in table headings will now be
** displayed in black even if the bg color of the cell is dark.
** This may be changed back again when enough browsers supply
** background colors in table cells.
**
** Revision 1.9.1.15  1996/06/27  11:43:44  stefan
** Removed unused variable to keep lint happy.
**
** Revision 1.9.1.14  1996/06/26  10:16:43  stefan
** Introduced again code for performance measurement. If compiled
** with -DTIME_STATS, the undocumented option -y forces http-analyze
** to print the time needed for processing the logfile entries, the
** time needed to create the summary, the total elapsed time, and
** the performance in hits/sec. Fixed some bugs which caused
** http-analyze to dump core on really huge logfiles. The parser
** and other time-consuming parts of the code have been improved
** somewhat again, especially for RISC-based platforms. http-analyze
** does not remove trailing slashes from URLs anymore, since an
** incomplete URL causes a redirect with the full qualified URL
** sent back by the server. Since only the URLs for Code 200 (OK)
** and Code 304 (NotModified) responses are saved, we do not need
** to remove any trailing slashes. Now, warnings about empty lines
** in the logfile are output in verbose mode only.
**
** Revision 1.9.1.13  1996/06/18  16:21:08  stefan
** Although this is unlikely to occur we check for a NULL
** ptr when dereferencing elements in the hidden_items
** tab by using an index saved in the ishidden element
** of NLIST structures.
**
** Revision 1.9.1.12  1996/06/18  14:47:53  stefan
** Now prints list of hidden items sorted also by their main topic
** and summarizes the hits/304's/bytes for all hidden URLs if more
** than one URL are hidden under an item.
**
** Revision 1.9.1.11  1996/06/18  13:33:34  stefan
** Checked for memory leaks. Moved VERSION macro into Makefile.
**
** Revision 1.9.1.10  1996/06/18  06:57:45  stefan
** Added explizit text color in colored table cells and rows
** for statistic pages with dark background and white colors.
**
** Revision 1.9.1.9  1996/06/17  19:52:05  stefan
** Now http-analyze generates a detailed list of all URLs
** which are hidden under some item. All URLs in the list
** of URLs are now hot links. The default mode of operation
** (-d or -m) can now be defined in the configuration file.
**
** Revision 1.9.1.8  1996/06/15  11:04:10  stefan
** Added a counter for the total size of all responses other
** than Code 200 (OK) and Code 304 (Not Modified) responses
** in the RespCode array. Removed adjustement of the size of
** Code 403 (Forbidden) and Code 404 (Not Found) responses
** since the http daemon should account for the extra overhead
** already. Removed code which truncated the trailing `/' from
** URLs since incomplete URLs (for example, `/dir') will result
** in a Code 302 (Redirect) response to the complete URL (`/dir/').
** Because the incomplete URLs are not shown in the URL lists,
** there is no need to merge the complete and incomplete URLs,
** and therefore no need to truncate the trailing slash from
** complete URLs.
**
** Revision 1.9.1.7  1996/06/13  10:45:25  stefan
** Changed table "Daily Summary" into "Hits by day",
** which is somewhat less confusing.
**
** Revision 1.9.1.6  1996/06/13  10:12:50  stefan
** Fixed a bug in calculation of the totals on
** last 12 month page again. Removed PR_ROW
** macros because there were used only once.
** Added colors to table cells with headers.
** Introduced counter for empty requests,
** suppress warning about empty requests unless
** running in verbose mode.
**
** Revision 1.9.1.5  1996/06/07  12:47:46  stefan
** Added counter for corrupted logfile entries.
**
** Revision 1.9.1.4  1996/06/05  14:08:50  stefan
** Fixed bug with wrong calculation of totals in last 12 month
** page (thanks Linda ;-)
**
** Revision 1.9.1.3  1996/06/05  08:13:52  stefan
** Fixed a bug in clearCounter() which caused the program
** to dump core because of dereferencing pointer with an
** invalid text address in function prMonStats().
**
** Revision 1.9.1.2  1996/05/29  06:00:24  stefan
** Added a hashing mechanism to the list of hidden items/sites
** to remove duplicates from the list. Cleaned up handling of
** some options, fixed some bugs.
**
** Revision 1.9.1.1  1996/05/25  16:31:32  stefan
** Semantic change: number of sites in the "Daily Summary" list
** are now the total number of sites per day rather than the
** total number of new sites per day as it was before. Thus,
** the sum of sites in the "Daily Summary" table is not equal
** to the total number of unique sites per month. Fixed a bug
** in the HTML output when suppressing the top sec/min/hour
** lists. Added code to generate a list of the N most boring
** URLs, where N can be modified via cmd line option or config
** file entry. Now allow 0 value to `-N' to suppress the lists.
**
** Revision 1.9  1996/05/25  02:01:47  stefan
** Enhanced the program for the sake of speed while providing
** the greatest possible ANSI C compatibility. Cleaned up data
** types. Introduced new options. Fixed lot of bugs. Changed
** order of computations (especially for hidden items) to
** generate more accurate results. Replaced excessive calls
** to stdlib functions in time-critical sections. Removed
** code for time stats because of the processing overhead it
** created. Fixed a bug in calculation of total kilobytes
** saved by cache. Fixed a bug in calculation of unique URLs
** and unique sites. Fixed a bug in computation of total
** number of hits for unresolved hosts. Revised the whole
** program and made it totally lint-clean. Added code to
** conditionally account for user agents. Replaced the
** call to gethostname(3) by the more common uname(2)
** function. Cleaned up command line syntax.
**
** Revision 1.8  1996/05/09  09:15:15  stefan
** Fixed a lot of minor bugs. http-analyze now uses uname(2) by default
** to determine the hostname (define USE_GETHOSTNAME to force usage of
** gethostname(2)). Added code to measure cpu time spent in analyzing
** data. Added code for integration of 3Dstats. Modified the command
** line interface to fully conform to the System V command line syntax
** rules. Added values for average number of hits per day and per hour
** to the full statistics. Fixed bug which caused the list of countries
** to disappear when analyzing more than one month in the same run.
**
** Revision 1.7  1996/02/07  14:27:11  stefan
** Added format specification of last column in top lists.
**
** Revision 1.6  1996/02/07  11:01:16  stefan
** Fixed some bugs with top hour/min/sec counters.
**
** Revision 1.5  1996/01/28  14:34:25  stefan
** Made it lint-clean, fixed some bugs.
**
** Revision 1.4  1996/01/28  13:44:06  stefan
** Corrected display of program version in trailer.
**
** Revision 1.3  1996/01/28  13:16:35  stefan
** Subdomains are now being stripped from the FQDN in top 5 site list
** if option -s (suppress full sitename listing) has been specified.
**
** Revision 1.2  1996/01/21  16:42:26  stefan
** Removed unused doc_root variable. Fixed bug where hidden sites
** appeared in top 5 sitelist if response was anything other than
** a Code 200 or Code 304 request. Commented out response sizes
** from RespCode[] while in beta test to allow easy cross-checking
** with results from other Log Analyzers, which don't take into
** account response sizes (define BETA_TEST). Introduced top secs,
** mins, hours counters. Added option to suppress these counters
** for speed comparisons.
**
** Revision 1.1  1996/01/17  00:00:00  stefan
** Initial revision
**
**
*/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <limits.h>
#include <assert.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>

#include "defs.h"
#include "cntrycode.h"

#if !defined(USE_GETHOSTNAME)
# include <sys/utsname.h>
#endif

#if defined(TIME_STATS)
# define TICKS_PER_MSEC  1000L
# include <sys/time.h>		/* for gettimeofday() */
#endif

/*
** If you don't use fgets, BEST_IO_SIZE defines the optimum buffer
** size per read(2) system call. On my SGI Indy running IRIX 5.3
** the optimum I/O buffer seems to be 64KB. Your mileage may vary.
*/
#if !defined(USE_FGETS) && !defined(BEST_IO_SIZE)
# define BEST_IO_SIZE	(1024*64)
#endif

/* global variables */
static char *progname;		/* name of program */
static char *last_update;	/* time of last update */

char *srv_name = NULL;		/* name of this server */
char *log_file = NULL;		/* name of the logfile */
char *out_dir = NULL;		/* directory for HTML output files */
char *priv_dir = NULL;		/* private HTML directory */
char *doc_title = NULL;		/* document title */
char *html_str[5] = { NULL, NULL, NULL, NULL, NULL };	/* HTML strings */
int monthly = -1;		/* if set create monthly summary */

/* We allow for up to MAX_HPNAMES alternate homepages */
char *home_page[MAX_HPNAMES];
int hpnum = 0;			/* total # of homepage names */
static size_t hplen[4];		/* string length */

LOGTIME tstart, tend;		/* time of first and last entry */
static LOGTIME tcur;		/* current time */

/*
** counters
*/

/* totals for one month */
u_long total_hits = 0L;			/* total # of hits */

static u_long total_files = 0L;		/* total # of files transferred */
static u_long total_nomod = 0L;		/* total # of 304 (NoMod) responses */
static u_long total_sites = 0L;		/* total # of sites */
static u_long total_kbytes = 0L;	/* total kbytes sent */
static u_long total_kbsaved = 0L;	/* total kbytes saved by cache */
static u_long this_hits = 0L;		/* total # of hits per run */
static u_long uniq_urls = 0L;		/* total # of unique URLs */
static u_long corrupt = 0L;		/* total # of invalid requests */
static u_long empty = 0L;		/* total # of empty requests */
static double total_bytes = 0.0;	/* total bytes transferred */
static size_t num_cntry;		/* total # of countries */


/* counters for one month */
u_long daily_hits[31];
u_long daily_files[31];
u_long daily_nomod[31];
u_long daily_sites[31];
u_long daily_kbytes[31];

/* counters for last 12 month */
static u_long monthly_hits[12];
static u_long monthly_files[12];
static u_long monthly_nomod[12];
static u_long monthly_sites[12];
static u_long monthly_kbytes[12];

u_long mn_hits[13];
u_long mn_files[13];
/*u_long mn_nomod[13]; *** unused by now */
u_long mn_sites[13];
u_long mn_kbytes[13];
char *mn_str[13];

/* counters for sum of last 12 month */
static u_long sum_hits   = 0L;
static u_long sum_files  = 0L;
static u_long sum_nomod  = 0L;
static u_long sum_sites  = 0L;
static u_long sum_kbytes = 0L;

/* counters for 24 hours */
u_long hr_hits[24];		/* per day */
u_long whr_hits[7][24];		/* per weekday and hour */

/* top URL/site counters */
int topn_sites = -1;
int topn_urls  = -1;
int lastn_urls = -1;
static NLIST **top_sites = NULL;
static NLIST **top_urls  = NULL;
static NLIST **last_urls = NULL;

/* top sec/min/hour counters */
static TOP_COUNTER top_sec[5];	/* top 5 seconds */
static TOP_COUNTER top_min[5];	/* top 5 minutes */
static TOP_COUNTER top_hrs[24];	/* top 24 hours */
static TOP_COUNTER csec;	/* current second */
static TOP_COUNTER cmin;	/* current minute */
static TOP_COUNTER chrs;	/* current hour */

/* common used messages */
static char enoent[] = "Can\'t open file `%s\' (%s)\n";
static char enomem[] = "Not enough memory -- need %u more bytes\n";
char *monnam[] = {
	"January", "February", "March", "April", "May", "June", "July",
	"August", "September",	"October", "November", "December" };

/* static variables */
static int hflg = 0;		/* if set print help only */
static int nohist = 0;		/* if set ignore history file */
static int noitemlist = 0;	/* if set suppress details about hidden items */
static int nofilelist = 0;	/* if set suppress details about URLs */
static int notoplist = 0;	/* if set suppress top sec/min/hours lists */
static int nositelist = 0;	/* if set suppress details about hosts  */
static int nocharts = 0;	/* if set suppress charts in summary reports */
static int timestats = 0;	/* if set print elapsed time */
/*static int vrml = 0;		/* create a VRML model of average load */
static int embed3D = 0;		/* include embedded OpenInventor 3D model */
static int verbose = 0;		/* be verbose */
static int nodefhid = 0;	/* don't hide default items (images) */
static int wdtab[31];		/* list of weekdays for this month */

#define HASHSIZE	4001
static NLIST *sitetab[HASHSIZE];
static NLIST *urltab[HASHSIZE];
NLIST *hstab[101];		/* hash lists to collect identical items */
NLIST *hitab[101];

#if defined(USER_AGENT)
static size_t uag = 0;		/* number of unique user agents in uatab[] */
static UAGENT uatab[100];	/* user agent info */
#endif

/*
 * HTTP response codes.
 * Note: the order of the table entries below is important.
 */
#define IDX_UNKNOWN		0	/* unknown response */
#define IDX_OK			1	/* 200 */
#define IDX_NO_RESPONSE		2	/* 204 */
#define IDX_REDIRECT		3	/* 302 */
#define IDX_NOT_MODIFIED	4	/* 304 */
#define IDX_BAD_REQUEST		5	/* 400 */
#define IDX_UNAUTHORIZED	6	/* 401 */
#define IDX_FORBIDDEN		7	/* 403 */
#define IDX_NOT_FOUND		8	/* 404 */
#define IDX_PROXY_UNAUTHORIZED	9	/* 407 */
#define IDX_SERVER_ERROR	10	/* 500 */
#define IDX_NOT_IMPLEMENTED	11	/* 501 */

static RESPONSE RespCode[] = {
    {   0,   0, 0L, 0.0, "Unknown response" },
    { 200, 180, 0L, 0.0, NULL },
    { 204,  98, 0L, 0.0, "Code 204 No Response" },
    { 302, 255, 0L, 0.0, "Code 302 Redirected Requests" },
    { 304,  98, 0L, 0.0, "Code 304 Not Modified Requests" },
    { 400, 313, 0L, 0.0, "Code 400 Bad Requests" },
    { 401, 281, 0L, 0.0, "Code 401 Unauthorized Requests" },
    { 403, 268, 0L, 0.0, "Code 403 Forbidden Requests" },
    { 404, 250, 0L, 0.0, "Code 404 Not Found Requests" },
    { 407, 281, 0L, 0.0, "Code 407 Proxy Unauthorized" },
    { 500, 485, 0L, 0.0, "Code 500 Server Errors" },
    { 501, 482, 0L, 0.0, "Code 501 Not Implemented Requests" }
};

/* static functions */
static int readLog(FILE * const, LOGENT * const, LOGTIME * const);
static char *prSiteStats(char * const, char * const, char * const, NLIST ** const);
static char *prURLStats(char * const, char * const, char * const, NLIST ** const);
static void prDayStats(void), prMonStats(void), prIndex(void);
static void clearItems(NLIST ** const, int const);
static void clearCounter(int const);
static void writeHistory(int const);
static int readHistory(int const, LOGTIME * const);
static u_char hextochar(char *);
static void insertTop(TOP_COUNTER * const, TOP_COUNTER [], int const);
static void mkdtab(LOGTIME * const);

#if defined(USER_AGENT)
static void userAgent(char * const);
static void clearUATab(void);
#endif

/*
 * Print an usage message.
 */
static char usage[] = \
	"Usage:\n   %s [-{d|m|h}] [-nrstuvxz] [-c cfgfile] [-i logfile] [-o outdir]\n" \
	"\t[-p privdir] [-N #s|u|l] [-H homepage] [-S srvname] [-T title] [file]\n\n";

static void pusage(int const help) {
	(void) fprintf(stderr, usage, progname);
	if (!help) {
		(void) fprintf(stderr, "Use `%s -h' for a help list.\n\n", progname);
		return;
	}
	(void) fprintf(stderr, \
		"\t-h\t\tprint this help list\n" \
		"\t-d\t\tgenerate short statistics (default)\n" \
		"\t-m\t\tgenerate full statistics (includes -d)\n" \
		"\t-n\t\tdon't use the history file\n" \
		"\t-r\t\tsuppress detailed list of hidden items\n" \
		"\t-s\t\tsuppress detailed list of hosts\n" \
		"\t-t\t\tsuppress top sec/min/hour lists\n" \
		"\t-u\t\tsuppress detailed list of URLs (implies -r)\n" \
		"\t-v\t\tverbose mode: comment ongoing processing\n" \
		"\t-x\t\tdon't comprise images by default\n" \
		"\t-z\t\tdon't create graphics in summaries\n" \
		"\t-c cfgfile\tname of the configuration file\n" \
		"\t-i logfile\tname of the web server's logfile (`-' for stdin)\n" \
		"\t-o outdir\tname of the directory for HTML output files\n" \
		"\t-p privdir\tname of a private directory for site/URL lists\n" \
		"\t-N 10s\t\tnumber of entries in top site list (10)\n" \
		"\t-N 30u\t\tnumber of entries in top URL list (30)\n" \
		"\t-N 10l\t\tnumber of entries in last URL list (10)\n" \
		"\t-H homepage\tname of additional homepage other than \"/index.html\"\n" \
		"\t-S srvname\tset server name (default: system name)\n" \
		"\t-T title\ttitle of the HTML documents\n" \
		"\tfile\t\tname of the logfile (`-' for stdin)\n\n");
	return;
}


int main(int const argc, char ** const argv) {
	extern char *optarg;
	extern int optind;
	char *cfg_file = NULL;
	char *ep, dbuf[SMALLSIZE];
	int cc, errflg = 0;
	time_t now;
	u_long val;
	u_long cur_tick = 0L;
	u_long last_tick[3] = { 0L, 0L, 0L };
	struct tm *tp;
	LOGENT entry;
	NLIST *np = NULL;
	FILE *lfp = NULL;

#if defined(TIME_STATS)
	struct timeval tvs, tve;
	u_long asec = 0L, rsec = 0L;

	(void) gettimeofday(&tvs);		/* measure execution time */
#endif
	now = time(NULL);			/* get current time */

	if ((progname = strrchr(argv[0], '/')) != NULL)
		progname++;			/* save program name */
	else	progname = argv[0];

	home_page[hpnum++] = "/index.html";	/* name of default homepage */

	/* process options */
	while ((cc = getopt(argc, argv, "hdmnrstuvxyzXc:i:o:p:N:H:S:T:")) != EOF) {
		switch (cc) {
		  case 'h':	hflg = ++errflg; break;	/* print help only */
		  case 'd':	monthly = 0;	break;	/* daily statistics */
		  case 'm':	monthly = 1;	break;	/* monthly statistics */
		  case 'n':	nohist++;	break;	/* don't use history */
		  case 'r':	noitemlist++;	break;	/* don't create hidden item list */
		  case 's':	nositelist++;	break;	/* don't create hostname list */
		  case 't':	notoplist++;	break;	/* don't create top sec/min/hours lists */
		  case 'u':	nofilelist++;	break;	/* don't create filename list */
		  case 'v':	verbose++;	break;	/* be verbose */
		  case 'x':	nodefhid++;	break;	/* don't hide default items */
		  case 'y':	timestats++;	break;	/* print elapsed time */
		  case 'z':	nocharts++;	break;	/* don't create charts in reports */
		  case 'X':	embed3D++;	break;	/* include embedded OpenInventor 3D model */
		  case 'c':	cfg_file=optarg; break;	/* configuration file */
		  case 'i':	log_file=optarg; break;	/* input file */
		  case 'o':	out_dir=optarg; break;	/* directory for output files */
		  case 'p':	priv_dir=optarg; break; /* private dir for site/URL lists */
		  case 'N':	val = strtoul(optarg, &ep, 10);
				if (ep == optarg) {
					prmsg(2, "invalid value: %s\n", optarg);
					++errflg;
					break;
				}
				if (*ep == 's' || *ep == 'S')
					topn_sites = (int)val;
				else if (*ep == 'u' || *ep == 'U')
					topn_urls = (int)val;
				else if (*ep == 'l' || *ep == 'L')
					lastn_urls = (int)val;
				break;
		  case 'H':	if (hpnum < TABSIZE(home_page))	/* name of the optional homepage */
					home_page[hpnum++]=optarg;
				break;
		  case 'S':	srv_name=optarg; break; /* server name */
		  case 'T':	doc_title=optarg; break; /* document title */
		  case '?':	errflg++;	break;	/* invalid option */
		}
	}

	if (argc-optind != 0 && !log_file) {
		log_file = argv[optind++];	/* process given logfile */
		if (*log_file == '-' && *(log_file+1) == '\0')
			*log_file = '\0';	/* process data from stdin */
	}

	if (errflg || optind != argc) {		/* print usage and exit */
		pusage(hflg);
		exit(!hflg);
	}
#if !defined(TIME_STATS)
	if (timestats)
		prmsg(0, "Timing code not compiled in, -y ignored\n");
#endif

	if (cfg_file && !readcfg(cfg_file)) {	/* read config file */
		prmsg(2, enoent, cfg_file, strerror(errno));
		exit(1);
	}

	if (!nodefhid)
		defHiddenItem();		/* add default items to hide */

	if (monthly < 0)			/* set default mode of operation */
		monthly = 0;

	if (topn_sites < 0)			/* add default dimensions for top lists */
		topn_sites = 10;
	if (topn_urls < 0)
		topn_urls = 30;
	if (lastn_urls < 0)
		lastn_urls = 10;

	/* allocate space for top lists */
	if (topn_sites != 0)
		top_sites = calloc(topn_sites, sizeof(NLIST *));
	if (topn_urls != 0)
		top_urls = calloc(topn_urls, sizeof(NLIST *));
	if (lastn_urls != 0)
		last_urls = calloc(lastn_urls, sizeof(NLIST *));

	if ((topn_sites && top_sites == NULL) ||
	    (topn_urls && top_urls == NULL) ||
	    (lastn_urls && last_urls == NULL)) {
		prmsg(2, enomem, topn_sites+topn_urls+lastn_urls);
		exit(1);
	}

	/* set defaults for some parameters not specified */
	if (log_file == NULL)			/* none given, not on invocation nor in config file */
		log_file = "";			/* process stdin */

	for (cc=0; cc < hpnum; cc++) {
		if (home_page[cc] != NULL && *home_page[cc] != '/') {
			prmsg(2, "The server's homepage URL must start with a slash.\n" \
				"Please change the name `%s' according to this rule.\n",
				home_page[cc]);
			exit(1);
		}
		hplen[cc] = strlen(home_page[cc]);
	}

	if (!srv_name) {
#if !defined(USE_GETHOSTNAME)
		static struct utsname nbuf;

		if (uname(&nbuf) != -1)
			srv_name = nbuf.nodename;
#else
		static char hbuf[SMALLSIZE];
		if (gethostname(hbuf, sizeof hbuf) == 0) {
			hbuf[SMALLSIZE-1] = '\0';
			srv_name = hbuf;
		}
#endif
		if (!srv_name) {
			srv_name = "www.un.known";
			prmsg(1, "Couldn't determine hostname, use `%s'\n", srv_name);
		}
	}
	if (!doc_title)
		doc_title = "Access statistics for";

	clearCounter(1);		/* clear all counters */

	tp = localtime(&now);		/* start processing */
	(void) strftime(dbuf, sizeof dbuf, "%d/%b/%Y:%T", tp);
	if (!setDate(&tcur, dbuf)) {  /* set the current date */
		prmsg(2, "Can't parse start date: %s\n", dbuf);
		exit(1);
	}
	tend = tstart = tcur;		/* set start and end time to current time */

	(void) strftime(dbuf, sizeof dbuf, "%d/%b/%Y %H:%M", tp);
	last_update = strsave(dbuf);	/* save string for display in HTML pages */

	if (monthly)			/* reset start time for monthly summary */
		tstart.mday = tstart.mon = tstart.year = 0;

	/* open the logfile (may reside in current dir) */
	if (*log_file && (lfp = fopen(log_file, "r")) == NULL) {
		prmsg(2, enoent, log_file, strerror(errno));
		exit(1);
	}

	/* now change into the HTML directory */
	if (!out_dir) {
		if (verbose)
			prmsg(0, "Output directory defaults to current dir\n");
	} else if (chdir(out_dir) != 0) {
		prmsg(2, "Can't change into directory `%s'\n", out_dir);
		exit(1);
	} else if (verbose)
		prmsg(0, "Creating output files in directory %s\n", out_dir);

	if (verbose && priv_dir && !(nositelist || nofilelist))
		prmsg(0, "Creating site/URL lists in private directory %s/%s\n",
			out_dir ? out_dir : ".", priv_dir);

	/*
	 * Now read the history to speed up processing for daily summaries.
	 * If doing a monthly summary, delay reading of the history file
	 * until we could determine the start time.
	 */
	if (!monthly) {	/* read history to speed up processing for daily summaries */
		if (!readHistory(0, &tstart)) {	/* if no valid history ... */
			tstart.mday = 1;	/* tick back to first day of month */
			if (verbose)
				prmsg(0, "No history file, start time is " TIME_FMT "\n",
					tstart.mday, monnam[tstart.mon], EPOCH(tstart.year));
		} else if (verbose)
			prmsg(0, "Skip logfile entries until " TIME_FMT "\n",
				tstart.mday, monnam[tstart.mon], EPOCH(tstart.year));
	}

	if (verbose)
		prmsg(0, "Reading data from %s ...\n", (*log_file ? log_file : "stdin"));

	/*
	 * Read the logfile. Remember the last day done in tend.
	 * For monthly stats, get summary period from first logfile
	 * entry read. Save start time of period in tstart.
	 */
	while (readLog(*log_file ? lfp : stdin, &entry, &tstart)) {
		if (tstart.mday == 0) {			/* we have no history so we use the */
			tstart = tend = entry.tm;	/* first entry to determine the month */
			tstart.mday = 1;		/* tick back to beginning of month */
			(void) readHistory(1, &tstart);
			if (verbose)
				prmsg(0, "First logfile entry sets start time " TIME_FMT "\n",
					tstart.mday, monnam[tstart.mon], EPOCH(tstart.year));
			mkdtab(&tstart);
		} else if (tstart.year != entry.tm.year || tstart.mon != entry.tm.mon) {
			if (monthly && !notoplist) {	/* flush top counter */
				insertTop(&chrs, top_hrs, TABSIZE(top_hrs));
				insertTop(&cmin, top_min, TABSIZE(top_min));
				insertTop(&csec, top_sec, TABSIZE(top_sec));
				last_tick[0] = last_tick[1] = last_tick[2] = 0L;
			}
#if defined(TIME_STATS)
			if (timestats) {
				(void) gettimeofday(&tve);
				rsec += (tve.tv_sec-tvs.tv_sec) * TICKS_PER_MSEC;
				rsec -= tvs.tv_usec/TICKS_PER_MSEC;
				rsec += tve.tv_usec/TICKS_PER_MSEC;
			}
#endif
			total_hits += empty;	/* account for empty hits */

			if (verbose)
				prmsg(0, "Analyzing data ...\n");

			prMonStats();	/* year or month wrap */
			prIndex();

			if (!nohist)
				writeHistory(1);

			/* prepare for new period */
			clearCounter(0);
			clearItems(sitetab, TABSIZE(sitetab));
			clearItems(urltab, TABSIZE(urltab));
			initHiddenItems();
			clearCountry();
#if defined(USER_AGENT)
			clearUATab();
#endif

			tstart = tend = entry.tm;
			tstart.mday = 1; /* tick back to beginning of month */
			if (verbose)
				prmsg(0, "Clear mostly all counters until "TIME_FMT " ;-)\n",
					tstart.mday, monnam[tstart.mon], EPOCH(tstart.year));
			mkdtab(&tstart);
#if defined(TIME_STATS)
			if (timestats) {
				(void) gettimeofday(&tvs);
				asec += (tvs.tv_sec-tve.tv_sec) * TICKS_PER_MSEC;
				asec -= tve.tv_usec/TICKS_PER_MSEC;
				asec += tvs.tv_usec/TICKS_PER_MSEC;
			}
#endif
		} else if (tend.mday != entry.tm.mday) {
			if (monthly && !notoplist) {	/* flush top counter */
				insertTop(&chrs, top_hrs, TABSIZE(top_hrs));
				insertTop(&cmin, top_min, TABSIZE(top_min));
				insertTop(&csec, top_sec, TABSIZE(top_sec));
				last_tick[0] = last_tick[1] = last_tick[2] = 0L;
			}
			tend.mday = entry.tm.mday;	/* remember last day done */
		}

		/*
		 * Count the totals.
		 */
		total_hits++;
		daily_hits[entry.tm.mday-1]++;
		hr_hits[entry.tm.hour]++;
		if (entry.respidx == IDX_OK) {
			total_files++;
			daily_files[entry.tm.mday-1]++;
		} else if (entry.respidx == IDX_NOT_MODIFIED) {
			total_nomod++;
			daily_nomod[entry.tm.mday-1]++;
		} else
			RespCode[entry.respidx].count++;

		if (monthly) {		/* check for known URL, save it */
			if ((entry.respidx == IDX_OK || entry.respidx == IDX_NOT_MODIFIED)) {
				np = lookupItem(urltab, TABSIZE(urltab), entry.request);
				if (np == NULL || !np->count)	/* no mem or brand new */
					uniq_urls++;

				if (np != NULL) {		/* update counters */
					np->count++;
					np->bytes += (double)entry.reqsize;
					if (entry.respidx == IDX_NOT_MODIFIED)
						np->nomod++;
					else if (!np->size && entry.respidx == IDX_OK)
						np->size = entry.reqsize;
				}
			} else
				RespCode[entry.respidx].bytes += (double)entry.reqsize;
#if defined(USER_AGENT)
			/* save user agent */
			if (entry.uagent)
			    userAgent(entry.uagent);
#endif
		}
		/* check for known sitename, save it */
		np = lookupItem(sitetab, TABSIZE(sitetab), entry.sitename);
		if (np == NULL) {		/* no more memory, count as new */
			total_sites++;
			daily_sites[entry.tm.mday-1]++;
		} else {
			if (!np->count)				/* brand new */
				total_sites++;
			if (np->ishidden != entry.tm.mday) {	/* new for today */
				daily_sites[entry.tm.mday-1]++;
				np->ishidden = entry.tm.mday;	/* last modified timestamp */
			}
			np->count++;				/* update counters */
			np->bytes += (double)entry.reqsize;
			if (entry.respidx == IDX_NOT_MODIFIED)
				np->nomod++;
		}

		total_bytes += (double)entry.reqsize;
		total_kbytes += (entry.reqsize+1023L)/1024L;
		daily_kbytes[entry.tm.mday-1] += (entry.reqsize+1023L)/1024L;

		if (monthly) {			/* compute average hits per hour & day */
			int wd = wdtab[entry.tm.mday-1];
			whr_hits[wd][entry.tm.hour]++;

			if (!notoplist) {	/* count top seconds, minutes, and hours */
				/* compute "ticks". Note: seconds after the minute -> [0, 61] */
				cur_tick = (u_long)((entry.tm.hour*60*62)+(entry.tm.min*62)+entry.tm.sec);

				if (cur_tick-last_tick[0] >= 60L*62L) {		/* hour has changed */
					insertTop(&chrs, top_hrs, TABSIZE(top_hrs));
					last_tick[0] = ((u_long)entry.tm.hour*60LU*62LU);
				}
				chrs.count++;
				chrs.nomod += (u_long)(entry.respidx == IDX_NOT_MODIFIED ? 1 : 0);
				chrs.bytes += (double)entry.reqsize;
				chrs.tm = entry.tm;

				if (cur_tick-last_tick[1] >= 62L) {		/* minute has changed */
					insertTop(&cmin, top_min, TABSIZE(top_min));
					last_tick[1] = ((u_long)entry.tm.hour*60LU*62LU)+((u_long)entry.tm.min*62LU);
				}
				cmin.count++;
				cmin.nomod += (u_long)(entry.respidx == IDX_NOT_MODIFIED ? 1 : 0);
				cmin.bytes += (double)entry.reqsize;
				cmin.tm = entry.tm;

				if (cur_tick-last_tick[2] != 0L) {		/* second has changed */
					insertTop(&csec, top_sec, TABSIZE(top_sec));
					last_tick[2] = cur_tick;
				}
				csec.count++;
				csec.nomod += (u_long)(entry.respidx == IDX_NOT_MODIFIED ? 1 : 0);
				csec.bytes += (double)entry.reqsize;
				csec.tm = entry.tm;
			}
		}
	}
	if (lfp != NULL)
		(void) fclose(lfp);

	if (monthly && !notoplist) {
		insertTop(&chrs, top_hrs, TABSIZE(top_hrs));
		insertTop(&cmin, top_min, TABSIZE(top_min));
		insertTop(&csec, top_sec, TABSIZE(top_sec));
		last_tick[0] = last_tick[1] = last_tick[2] = 0L;
	}

	if (this_hits == 0L && total_hits == 0L) {
		prmsg(2, "No hits at all?!?\n");
		exit(1);
	}
	total_hits += empty;	/* account for empty hits after the validation above */

#if defined(TIME_STATS)
	if (timestats) {
		(void) gettimeofday(&tve);
		rsec += (tve.tv_sec-tvs.tv_sec) * TICKS_PER_MSEC;
		rsec -= tvs.tv_usec/TICKS_PER_MSEC;
		rsec += tve.tv_usec/TICKS_PER_MSEC;
	}
#endif
	if (this_hits > 150 && total_hits <= 15) {
		if (verbose)
			prmsg(0, "Let the dust settle down: ignore %lu hits since " TIME_FMT "\n",
				total_hits, tend.mday, monnam[tend.mon], EPOCH(tend.year));
	} else {
		if (verbose)
			prmsg(0, "Analyzing data ...\n");

		if (tend.year == tcur.year && tend.mon == tcur.mon) {	/* current month */
			if (tend.mday < tcur.mday) {
				if (!nohist && verbose)
					prmsg(0, "No more hits since " TIME_FMT "\n",
						tend.mday, monnam[tend.mon], EPOCH(tend.year));
				tend.mday = tcur.mday;	 /* in case there were no hits since then */
			}
			if (monthly && tend.mday > 1)	/* first monthly stats allowed after 1 day */
				prMonStats();

			prDayStats();
			prIndex();
			if (!nohist)
				writeHistory(0);
		} else if (monthly) {		/* monthly processing for previous months */
			prMonStats();
			prIndex();
			if (!nohist)
				writeHistory(1);
		}
	}

#if defined(TIME_STATS)
	if (timestats) {
		(void) gettimeofday(&tvs);		/* measure execution time */
		asec += (tvs.tv_sec-tve.tv_sec) * TICKS_PER_MSEC;
		asec -= tve.tv_usec/TICKS_PER_MSEC;
		asec += tvs.tv_usec/TICKS_PER_MSEC;
		this_hits += total_hits;		/* sum up this month' hits */

		prmsg(0, "\nTime to process %lu entries: %lu.%03lu sec (%lu hits/sec)\n" \
			"Time to create the summary: %lu.%03lu sec\n",
			this_hits, rsec/TICKS_PER_MSEC, rsec%TICKS_PER_MSEC,
			(u_long) ((double)this_hits / (((double)rsec/TICKS_PER_MSEC))),
			asec/TICKS_PER_MSEC, asec%TICKS_PER_MSEC);

		rsec += asec;
		prmsg(0, "Total time elapsed: %lu.%03lu sec (%lu hits/sec)\n",
			rsec/TICKS_PER_MSEC, rsec%TICKS_PER_MSEC,
			(u_long) ((double)this_hits / (((double)rsec/TICKS_PER_MSEC))));
	}
#endif
	/*
	** Clean up. Although technically not necessary here,
	** freeing allocated memory enables us to detect memory
	** leaks elsewhere.
	*/
	free(last_update);
	clearItems(sitetab, TABSIZE(sitetab));
	clearItems(urltab, TABSIZE(urltab));
	clearItems(hstab, TABSIZE(hstab));
	clearItems(hitab, TABSIZE(hitab));
#if defined(USER_AGENT)
	clearUATab();
#endif
	if (top_sites != NULL)
		free(top_sites);
	if (top_urls != NULL)
		free(top_urls);
	if (last_urls != NULL)
		free(last_urls);

	return 0;
}

static void insertTop(TOP_COUNTER * const cp, TOP_COUNTER tp[], int const max) {
	int idx, minidx = 0;
	u_long mincnt = ~0UL;
	double minbyt = 0.0;

	if (!cp->count && !cp->nomod && !cp->bytes)
		return;

	for (idx=0; idx < max; idx++) {	/* find lowest value */
		if (tp[idx].count < mincnt || tp[idx].count == mincnt &&
		    (minbyt == 0.0 || (double)tp[idx].bytes < minbyt)) {
			minidx = idx;
			mincnt = tp[idx].count;
			minbyt = (double)tp[idx].bytes;
		}
	}
	if (cp->count > tp[minidx].count ||
	    cp->count == tp[minidx].count && cp->bytes > tp[minidx].bytes) {
		tp[minidx].count = cp->count;
		tp[minidx].nomod = cp->nomod;
		tp[minidx].bytes = cp->bytes;
		tp[minidx].tm = cp->tm;
	}
	cp->count = 0L;
	cp->nomod = 0L;
	cp->bytes = 0.0;
	return;
}

/*
 * Print an HTML header.
 */
static void html_header(FILE * const ofp, char * const period, int const lup) {
	(void) fprintf(ofp, "<HTML><HEAD>\n" \
		"<TITLE>%s %s (%s)</TITLE>\n</HEAD>\n", doc_title, srv_name, period);

	if (html_str[HTML_HEADPFX])
		(void) fprintf(ofp, "%s\n", html_str[HTML_HEADPFX]);
	else	(void) fprintf(ofp, "<BODY BGCOLOR=\"#EFEFEF\">\n");

	(void) fprintf(ofp, "<H2>%s %s</H2>\n" \
		"<FONT SIZE=\"2\"><B>Summary period: %s\n",
		doc_title, srv_name, period);
	if (lup)
		(void) fprintf(ofp, "<BR>Last updated: %s\n", last_update);
	(void) fprintf(ofp, "</B></FONT>\n");

	if (html_str[HTML_HEADSFX])
		(void) fprintf(ofp, "%s\n", html_str[HTML_HEADSFX]);
	else	(void) fprintf(ofp, "<HR SIZE=\"4\"><P>\n");
	return;
}

/*
 * Print an HTML trailer.
 */
static void html_trailer(FILE * const ofp, int const  back) {
	if (back)
		(void) fprintf(ofp, "<BR><FONT SIZE=\"-1\"><A HREF=\"index.html\">" \
			"Back</A> to the monthly summary</FONT>\n");

	if (html_str[HTML_TRAILER])
		(void) fprintf(ofp, "%s\n", html_str[HTML_TRAILER]);

	(void) fprintf(ofp, "<P>\n<HR SIZE=\"4\">\n<TABLE WIDTH=\"100%%\">\n" \
		"<TR><TD ALIGN=\"LEFT\"><FONT SIZE=\"-1\"><B>" \
		"Statistics generated by <A HREF=\"http://www.netstore.de/Supply/http-analyze/\">" \
		"http-analyze %s</A></B></FONT></TD>\n" \
		"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>Copyright &#169; 1996 by " \
		"RENT-A-GURU&#174;</B></FONT></TD></TR></TABLE>\n", VERSION);
	return;
}

#define PERCENT(val, max)	((val) ? ((double)(val)*100.0)/(double)(max) : 0.0)

#define PR_TOP(ofp, label, c1, c2, c3, c4, c5) \
    (void) fprintf((ofp), \
	"<TR><TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n" \
	"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
	"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-2\">%2.1f%%</FONT></TD>\n" \
	"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
	"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-2\">%2.1f%%</FONT></TD>\n" \
	"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n", \
	(label), (c1), (c2), (c3), (c4), (c5))

#define FMT_HEAD(cspan, label)	\
    "<TR><TH COLSPAN=\"" cspan "\" BGCOLOR=\"#CCCCCC\">" \
    "<FONT COLOR=\"#000000\"><B>" label "</B></FONT></TH></TR>\n"

#define FMT_SPACE(ht)	"<TR><TD HEIGHT=\"" ht "\"></TD></TR>\n"

#define FMT_SUM(label)	\
    "<TR><TD WIDTH=\"62%%\" ALIGN=\"RIGHT\">" \
    "<FONT SIZE=\"-1\">" label "</FONT></TD>\n" \
    "<TD COLSPAN=\"2\" ALIGN=\"RIGHT\"><FONT SIZE=\"-1\">" \
    "<B>%lu</B></FONT></TD></TR>\n"

/*
 * Print daily table entries.
 */
static void prDayTab(FILE * const ofp, int const stop) {
	char cell_C[] = "<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n";
	char cell_R[] = "<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>" \
			"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-2\">%2.1f%%</FONT></TD>\n";
	int idx;

	(void) fprintf(ofp, FMT_SPACE("4") \
			FMT_HEAD("11", "Hits by day") FMT_SPACE("4"));

	(void) fprintf(ofp, "<TR><TH BGCOLOR=\"#333333\"><FONT SIZE=\"-1\">Day</FONT></TH>" \
		"<TH COLSPAN=\"2\" BGCOLOR=\"#00AA00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hits</FONT></TH>" \
		"<TH COLSPAN=\"2\" BGCOLOR=\"#0000DC\"><FONT SIZE=\"-1\">Files</FONT></TH>" \
		"<TH COLSPAN=\"2\" BGCOLOR=\"#FFFF00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">304's</FONT></TH>" \
		"<TH COLSPAN=\"2\" BGCOLOR=\"#DC0000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Sites</FONT></TH>" \
		"<TH COLSPAN=\"2\" BGCOLOR=\"#FF5000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">KBytes&nbsp;sent</FONT></TD></TR>\n");
	(void) fprintf(ofp, FMT_SPACE("4"));

	for (idx=0; idx < stop; idx++) {
		(void) fputs("<TR>", ofp);
		(void) fprintf(ofp, cell_C, idx+1);
		(void) fprintf(ofp, cell_R, daily_hits[idx], PERCENT(daily_hits[idx], total_hits));
		(void) fprintf(ofp, cell_R, daily_files[idx], PERCENT(daily_files[idx], total_files));
		(void) fprintf(ofp, cell_R, daily_nomod[idx], PERCENT(daily_nomod[idx], total_nomod));
		(void) fprintf(ofp, cell_R, daily_sites[idx], PERCENT(daily_sites[idx], total_sites));
		(void) fprintf(ofp, cell_R, daily_kbytes[idx], PERCENT(daily_kbytes[idx], total_kbytes));
		(void) fputs("</TR>\n", ofp);
	}
	return;
}


/*
 * Print daily statistics.
 */
static void prDayStats(void) {
	char fname[FILENAME_MAX];
	char mname[FILENAME_MAX];
	char td100[] = "<TD ALIGN=\"CENTER\"><FONT SIZE=\"-2\">100.0%</FONT></TD>";
	FILE *ofp;

	(void) sprintf(fname, "stats.html");
	(void) sprintf(mname, "stats.gif");

	if ((ofp=fopen(fname, "w")) == NULL) {
		prmsg(2, enoent, fname, strerror(errno));
		exit(1);
	}

	(void) sprintf(fname, TIME_FMT " to " TIME_FMT,
		1, monnam[tstart.mon], EPOCH(tstart.year),
		tend.mday, monnam[tend.mon], EPOCH(tend.year));

	if (verbose)
		prmsg(0, "Creating daily stats for %s ...\n", fname);

	html_header(ofp, fname, 1);

	if (!nocharts)
		(void) fprintf(ofp, "<P><CENTER><IMG SRC=\"%s\" ALT=\"diagram\" " \
				"WIDTH=\"492\" HEIGHT=\"316\"></CENTER><P>\n", mname);

	(void) fprintf(ofp, "<CENTER><TABLE WIDTH=\"492\" BORDER=\"2\" CELLSPACING=\"1\" CELLPADDING=\"1\">\n");

	prDayTab(ofp, (int)tcur.mday);
	(void) fprintf(ofp, FMT_SPACE("4"));
	(void) fprintf(ofp, "<TR BGCOLOR=\"#CCCCCC\">" \
		"<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\"><B>Total</B></FONT></TD>\n" \
		"<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n%s\n" \
		"<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n%s\n" \
		"<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n%s\n" \
		"<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n%s\n" \
		"<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n%s\n",
		total_hits, td100, total_files, td100, total_nomod, td100,
		total_sites, td100, total_kbytes, td100);

	(void) fprintf(ofp, FMT_SPACE("4") \
		"</TABLE></CENTER>\n<P><HR SIZE=\"4\">\n");

	(void) sprintf(fname, "stats%02d%02d.html", tend.mon+1, EPOCH(tend.year));
	if (access(fname, 0) == 0)
		(void) fprintf(ofp, "<BR><FONT SIZE=\"-1\"><A HREF=\"%s\">" \
			"Full Statistics</A> for %s %d\n</FONT>\n",
			fname, monnam[tcur.mon], EPOCH(tend.year));

	html_trailer(ofp, 1);
	(void) fclose(ofp);

	(void) mn_bars(492, 317, 28, mname);
	return;
}

/*
 * Print an index file.
 */
static void prIndex(void) {
	static int mdays[2][12] = {
		{ 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 },
		{ 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 }
	};
	char fname[FILENAME_MAX];
	char iname[FILENAME_MAX];
	char gname[FILENAME_MAX];
	
	int current = (tend.year == tcur.year && tend.mon == tcur.mon);
	int dst = TABSIZE(mn_hits)-1;
	int idx, rc;
	FILE *ofp;

	(void) sprintf(iname, "gr-icon.gif");
	(void) sprintf(gname, "graph%04d.gif", tstart.year);

	if (current)		/* current update of tcur.mon */
		(void) sprintf(fname, "index.html");
	else		/* monthly update until tend.mon */
		(void) sprintf(fname, "stats%04d.html", tstart.year);

	if ((ofp=fopen(fname, "w")) == NULL) {
		prmsg(2, enoent, fname, strerror(errno));
		exit(1);
	}
	(void) sprintf(fname, current ? "last 12 month": "%d", tend.year);
	html_header(ofp, fname, current);

	if (!nocharts)
		(void) fprintf(ofp, "<CENTER><IMG SRC=\"%s\" ALT=\"diagram\" " \
				"WIDTH=\"490\" HEIGHT=\"317\"></CENTER><P>\n", gname);

	(void) fprintf(ofp, "<CENTER><TABLE WIDTH=\"490\" BORDER=\"2\" CELLSPACING=\"1\" CELLPADDING=\"1\">\n");
	if (current && access("stats.html", 0) == 0) {
		(void) fprintf(ofp, FMT_SPACE("4") \
			"<TR><TH COLSPAN=\"5\" BGCOLOR=\"#CCCCCC\"><FONT SIZE=\"-1\" COLOR=\"#000000\">" \
			"<A HREF=\"stats.html\">Current statistic for %s %d (updated more frequently)</A>\n" \
			"</FONT></TH></TR>\n", monnam[tend.mon], EPOCH(tend.year));
	}

	(void) fprintf(ofp, FMT_SPACE("4") \
		"<TR><TH BGCOLOR=\"#333333\"><FONT SIZE=\"-1\">Month</FONT></TH>" \
		"<TH WIDTH=\"20%%\" BGCOLOR=\"#00AA00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hits</FONT></TH>" \
		"<TH WIDTH=\"20%%\" BGCOLOR=\"#0000DC\"><FONT SIZE=\"-1\">Files</FONT></TH>" \
		"<TH WIDTH=\"20%%\" BGCOLOR=\"#DC0000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Sites</FONT></TH>" \
		"<TH WIDTH=\"28%%\" BGCOLOR=\"#FF5000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">KBytes&nbsp;sent</FONT></TH></TR>\n");

	(void) fprintf(ofp, FMT_SPACE("4") \
		"<TR><TD ALIGN=\"LEFT\"><FONT SIZE=\"-1\"><B>");
	if (!current || tend.mday > 1)
		(void) fprintf(ofp, "<A HREF=\"stats%02d%02d.html\">%s %d</A>\n",
			tend.mon+1, EPOCH(tend.year), monnam[tend.mon], EPOCH(tend.year));
	else
		(void) fprintf(ofp, "%s %d", monnam[tend.mon], EPOCH(tend.year));
	(void) fprintf(ofp, "</B></FONT></TD>\n");

	(void) fprintf(ofp, "<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD> " \
		"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
		"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
		"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD></TR>\n", \
		total_hits, total_files, total_sites, total_kbytes);

	sum_hits = mn_hits[dst] = total_hits;
	sum_files = mn_files[dst] = total_files;
	sum_sites = mn_sites[dst] = total_sites;
	sum_kbytes = mn_kbytes[dst] = total_kbytes;

	if (current) {		/* interpolate values based on average values */
		int leap = tend.year%4 == 0 && tend.year%100 != 0 || tend.year%400 == 0;
		int ddiff = mdays[leap][tend.mon] - tend.mday;

		if (ddiff > 0) {
			mn_hits[dst] += (mn_hits[dst] * (u_long)ddiff) / (u_long)tcur.mday;
			mn_files[dst] += (mn_files[dst] * (u_long)ddiff) / (u_long)tcur.mday;
			mn_sites[dst] += (mn_sites[dst] * (u_long)ddiff) / (u_long)tcur.mday;
			mn_kbytes[dst] += (mn_kbytes[dst] * (u_long)ddiff) / (u_long)tcur.mday;
		}
	}
	mn_str[dst--] = monnam[tend.mon];

	for (idx=(int)tend.mon; --idx >= 0; ) {
		(void) fprintf((ofp), "<TR><TD><FONT SIZE=\"-1\"><B>");
		if (monthly_hits[idx] > 0L)
			(void) fprintf(ofp, "<A HREF=\"stats%02d%02d.html\">%s %d</A>",
				idx+1, EPOCH(tend.year), monnam[idx], EPOCH(tend.year));
		else
			(void) fprintf(ofp, "%s %d", monnam[idx], EPOCH(tend.year));

		(void) fprintf(ofp, "</B></FONT></TD>\n" \
			"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
			"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
			"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
			"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD></TR>\n", \
			monthly_hits[idx], monthly_files[idx], monthly_sites[idx], monthly_kbytes[idx]);

		sum_hits += (mn_hits[dst] = monthly_hits[idx]);
		sum_files += (mn_files[dst] = monthly_files[idx]);
		sum_sites += (mn_sites[dst] = monthly_sites[idx]);
		sum_kbytes += (mn_kbytes[dst] = monthly_kbytes[idx]);
		mn_str[dst--] = monnam[idx];
	}
	for (idx=11; idx > (int)tend.mon; idx--) {
		(void) fprintf((ofp), "<TR><TD><FONT SIZE=\"-1\"><B>");
		if (monthly_hits[idx] > 0L)
			(void) fprintf(ofp, "<A HREF=\"stats%02d%02d.html\">%s %d</A>",
				idx+1, EPOCH(tend.year)-1, monnam[idx], EPOCH(tend.year)-1);
		else
			(void) fprintf(ofp, "%s %d", monnam[idx], EPOCH(tend.year)-1);

		(void) fprintf(ofp, "</B></FONT></TD>\n" \
			"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
			"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
			"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
			"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD></TR>\n", \
			monthly_hits[idx], monthly_files[idx], monthly_sites[idx], monthly_kbytes[idx]);

		sum_hits += (mn_hits[dst] = monthly_hits[idx]);
		sum_files += (mn_files[dst] = monthly_files[idx]);
		sum_sites += (mn_sites[dst] = monthly_sites[idx]);
		sum_kbytes += (mn_kbytes[dst] = monthly_kbytes[idx]);
		mn_str[dst--] = monnam[idx];
	}
	mn_hits[dst] = monthly_hits[idx];	/* previous year */
	mn_files[dst] = monthly_files[idx];
	mn_sites[dst] = monthly_sites[idx];
	mn_kbytes[dst] = monthly_kbytes[idx];
	mn_str[dst] = monnam[idx];

	(void) fprintf((ofp), FMT_SPACE("4") \
		"<TR BGCOLOR=\"#CCCCCC\">" \
		"<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\" COLOR=\"#000000\"><B>Total</B></FONT></TD> " \
		"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\" COLOR=\"#000000\"><B>%lu</B></FONT></TD> " \
		"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\" COLOR=\"#000000\"><B>%lu</B></FONT></TD> " \
		"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\" COLOR=\"#000000\"><B>%lu</B></FONT></TD> " \
		"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\" COLOR=\"#000000\"><B>%lu</B></FONT></TD></TR>\n", \
		sum_hits, sum_files, sum_sites, sum_kbytes);

	(void) fprintf(ofp, FMT_SPACE("4") \
		"</TABLE></CENTER>\n<P><HR SIZE=\"4\">\n");

	rc = 0;
	idx = (int)tend.year;
	while (!rc) {		/* create links for previous summary periods */
		(void) sprintf(fname, "stats%04d.html", --idx);
		if ((rc=access(fname, 0)) == 0)
			(void) fprintf(ofp, "<BR><FONT SIZE=\"-1\">" \
				"<A HREF=\"%s\">Full statistics</A> for %d\n</FONT>\n",
				fname, idx);
	}

	if (!current)
		(void) fprintf(ofp, "<BR><FONT SIZE=\"-1\"><A HREF=\"index.html\">" \
			"Back to the monthly summary</A>");

	html_trailer(ofp, 0);
	(void) fclose(ofp);

	(void) graph(490, 317, 28, gname);
	(void) graph(50, 32, 0, iname);
	return;
}

/*
 * Sort list by hits, hidden item, and country.
 */

static int sort_by_hits(const void *e1, const void *e2) {
	if ((*(NLIST **)e2)->count == (*(NLIST **)e1)->count)
		return 0;
	return ((*(NLIST **)e2)->count < (*(NLIST **)e1)->count) ? -1 : 1;
}

static int sort_by_item(const void *e1, const void *e2) {
	size_t stamp1 = (*(NLIST **)e1)->ishidden;
	size_t stamp2 = (*(NLIST **)e2)->ishidden;

	if (stamp1 == stamp2 ||
	    !strcmp(hidden_items[stamp1].col->str, hidden_items[stamp2].col->str)) {
		if ((*(NLIST **)e2)->count == (*(NLIST **)e1)->count)
			return 0;
		return ((*(NLIST **)e2)->count < (*(NLIST **)e1)->count) ? -1 : 1;
	}
	if (hidden_items[stamp1].col->count == hidden_items[stamp2].col->count)
		return 0;
	return (hidden_items[stamp2].col->count < hidden_items[stamp1].col->count) ? -1 : 1;
}

static int sort_by_cntry(const void *e1, const void *e2) {
	if ((*(COUNTRY **)e2)->count == (*(COUNTRY **)e1)->count)
		return 0;
	return ((*(COUNTRY **)e2)->count < (*(COUNTRY **)e1)->count) ? -1 : 1;
}

/*
 * Sort top list by hits and data sent.
 */

static int sort_top(const void *e1, const void *e2) {
	if (((TOP_COUNTER *)e2)->count == ((TOP_COUNTER *)e1)->count)
		return (int)(((TOP_COUNTER *)e2)->bytes - ((TOP_COUNTER *)e1)->bytes);

	return ((TOP_COUNTER *)e2)->count < ((TOP_COUNTER *)e1)->count ? -1 : 1;
}
/*
 * Print monthly summary.
 */

static int tsites = 0;		/* total number of sites & files */
static int tfiles = 0;		/* set by prSiteStats and prURLStats */
static int lfiles = 0;
static int hfiles = 0;

static void prMonStats(void) {
	static char *authmsg = "(authorization required)";
	char *pgsites = NULL;
	char *pgfiles = NULL;
	char *cp, tbuf[SMALLSIZE];
	char fname[FILENAME_MAX];	/* name of output file */
	char mname[FILENAME_MAX];	/* name of GIF image for daily stats */
	char hname[FILENAME_MAX];	/* name of GIF image for hourly stats */
	char cname[FILENAME_MAX];	/* name of GIF image for country stats */
	/*char vrml_file[FILENAME_MAX];	/* name of VRML 3D model */
	u_long av_hits = 0L;		/* average hits per day */
	u_long max_dhits = 0L;		/* max hits per day */
	u_long max_hhits = 0L;		/* max hits per hour */
	int current = (tend.year == tcur.year && tend.mon == tcur.mon);
	int idx, cday, hour;		/* temp */
	COUNTRY *ccp, **clist = NULL;	/* ptr into (sorted) country list */
	FILE *ofp;

	if (verbose)
		prmsg(0, "Creating monthly stats for %3.3s %d ...\n", monnam[tstart.mon], tstart.year);

	checkForIcons();

	if (!current)
		(void) sprintf(tbuf, "%s %d", monnam[tend.mon], tend.year);
	else
		(void) sprintf(tbuf, TIME_FMT " to " TIME_FMT,
			1, monnam[tstart.mon], EPOCH(tstart.year),
			tend.mday, monnam[tend.mon], EPOCH(tend.year));

	(void) sprintf(fname,
		priv_dir ? "../stats%02d%02d.html" : "stats%02d%02d.html",
		tstart.mon+1, EPOCH(tstart.year));

	(void) sprintf(mname, "%s/sites%02d%02d.html",
		!priv_dir ? "." : priv_dir, tstart.mon+1, EPOCH(tstart.year));
	pgsites = prSiteStats(mname, fname, tbuf, sitetab);

	(void) sprintf(mname, "%s/files%02d%02d.html",
		!priv_dir ? "." : priv_dir, tstart.mon+1, EPOCH(tstart.year));
	pgfiles = prURLStats(mname, fname, tbuf, urltab);

	(void) sprintf(mname, "stats%02d%02d.gif", tstart.mon+1, EPOCH(tstart.year));
	(void) sprintf(hname, "avload%02d%02d.gif", tstart.mon+1, EPOCH(tstart.year));
	(void) sprintf(cname, "cntry%02d%02d.gif", tstart.mon+1, EPOCH(tstart.year));

	/* compute average hits per day and hour */
	av_hits = total_hits / (u_long)tend.mday;

	/* find highest values per day and hour */
	for (cday=0; cday < (int)tend.mday; cday++) {
		if (daily_hits[cday] > max_dhits)
			max_dhits = daily_hits[cday];
	}
	for (cday=0; cday < 7; cday++) {
		for (hour=0; hour < 24; hour++) {
			if (whr_hits[cday][hour] > max_hhits)
				max_hhits = whr_hits[cday][hour];
		}
	}

	(void) sprintf(fname, "stats%02d%02d.html", tstart.mon+1, EPOCH(tstart.year));
	if ((ofp=fopen(fname, "w")) == NULL) {
		prmsg(2, enoent, fname, strerror(errno));
		exit(1);
	}

	html_header(ofp, tbuf, current);
	(void) fprintf(ofp, "<P><CENTER><TABLE WIDTH=\"492\" BORDER=\"2\" " \
			"CELLSPACING=\"1\" CELLPADDING=\"1\">\n");
	(void) fprintf(ofp, FMT_SPACE("4") \
			FMT_HEAD("3", "Totals for %s %d") FMT_SPACE("4"),
			monnam[tstart.mon], tstart.year);

#define SQ_WID(max, val) \
	(!(val) || !(max) ? 1 : (((val)*100)/(max))+1)

	(void) fprintf(ofp, "<TR><TD WIDTH=\"62%%\" ALIGN=\"RIGHT\">\n" \
		"<FONT SIZE=\"-1\">Total hits</FONT> " \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"4\"><BR>\n" \
		"<FONT SIZE=\"-1\">Total files sent</FONT> " \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"4\"><BR>\n" \
		"<FONT SIZE=\"-1\">Total 304 (NoMod) responses</FONT> " \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"4\"><BR>\n" \
		"<FONT SIZE=\"-1\">Other responses (see below)</FONT> " \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"4\"></TD>\n",
		SQICON_GREEN, SQICON_BLUE, SQICON_YELLOW, SQICON_RED);

	(void) fprintf(ofp, "<TD VALIGN=\"CENTER\">" \
		"<IMG SRC=\"%s\" VSPACE=\"2\" HEIGHT=\"8\" WIDTH=\"%d\"><BR>\n" \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"%lu\">" \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"%lu\">" \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"%lu\"></TD>\n",
		SQICON_GREEN, 102,
		SQICON_BLUE, SQ_WID(total_hits, total_files),
		SQICON_RED, SQ_WID(total_hits, total_hits-(total_files+total_nomod)),
		SQICON_YELLOW, SQ_WID(total_hits, total_nomod));

	(void) fprintf(ofp, "<TD WIDTH=\"20%%\" ALIGN=\"RIGHT\">" \
		"<FONT SIZE=\"-1\"><B>%lu</B></FONT><BR>\n" \
		"<FONT SIZE=\"-1\"><B>%lu</B></FONT><BR>\n" \
		"<FONT SIZE=\"-1\"><B>%lu</B></FONT><BR>\n" \
		"<FONT SIZE=\"-1\"><B>%lu</B></FONT></TD></TR>\n",
		total_hits, total_files, total_nomod,
		total_hits-(total_files+total_nomod));

	(void) fprintf(ofp, FMT_SPACE("4") \
		"<TR><TD WIDTH=\"60%%\" ALIGN=\"RIGHT\">" \
		"<FONT SIZE=\"-1\">Total KB requested</FONT> \n" \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"4\"><BR>\n" \
		"<FONT SIZE=\"-1\">Total KB transferred</FONT> " \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"4\"><BR>\n" \
		"<FONT SIZE=\"-1\">Total KB saved by cache</FONT> " \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"4\"></TD>\n", \
		SQICON_GREEN, SQICON_ORANGE, SQICON_YELLOW);

	(void) fprintf(ofp, "<TD VALIGN=\"CENTER\">\n" \
		"<IMG SRC=\"%s\" VSPACE=\"2\" HEIGHT=\"8\" WIDTH=\"%d\"><BR>\n" \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"%lu\">" \
		"<IMG SRC=\"%s\" HEIGHT=\"8\" WIDTH=\"%lu\"></TD>\n",
		SQICON_GREEN, 102,
		SQICON_ORANGE, SQ_WID(total_kbytes+total_kbsaved, total_kbytes),
		SQICON_YELLOW, SQ_WID(total_kbytes+total_kbsaved, total_kbsaved));

	(void) fprintf(ofp, "<TD WIDTH=\"20%%\" ALIGN=\"RIGHT\">" \
		"<FONT SIZE=\"-1\"><B>%lu</B></FONT><BR>\n" \
		"<FONT SIZE=\"-1\"><B>%lu</B></FONT><BR>\n" \
		"<FONT SIZE=\"-1\"><B>%lu</B></FONT></TD></TR>\n",
		total_kbytes+total_kbsaved, total_kbytes, total_kbsaved);

	(void) fprintf(ofp, FMT_SPACE("4") \
		FMT_SUM("Total unique URLs") \
		FMT_SUM("Total unique sites") \
		FMT_SPACE("4"), uniq_urls, total_sites);

	if (corrupt != 0)
		(void) fprintf(ofp, FMT_SUM("Corrupted logfile entries"), corrupt);
	if (empty != 0)
		(void) fprintf(ofp, FMT_SUM("Empty requests"), empty);

	for (idx=0; idx < (int)TABSIZE(RespCode); idx++) {
		if (idx == IDX_OK || idx == IDX_NOT_MODIFIED || !RespCode[idx].count)
			continue;
		(void) fprintf(ofp, FMT_SUM("%s"),
			RespCode[idx].msg, RespCode[idx].count);
	}

	(void) fprintf(ofp, FMT_SPACE("4") \
			FMT_SUM("Max hits per day") \
			FMT_SUM("Average hits/day"), max_dhits, av_hits);

	(void) fprintf(ofp, FMT_SUM("Max hits per hour") \
			FMT_SUM("Average hits/hour"), max_hhits, av_hits/24L);

	(void) fprintf(ofp, FMT_SPACE("4") \
		"<TR><TD ALIGN=\"CENTER\" COLSPAN=\"3\"><FONT SIZE=\"-1\"><B>" \
		"<A HREF=\"#daysum\">Hits by day</A>" \
		" / <A HREF=\"#avload\">Average load</A>");
	if (tsites)
		(void) fprintf(ofp, " / <A HREF=\"#topsites\">Top %d sites</A>\n", tsites);
	if (tfiles)
		(void) fprintf(ofp, " / <A HREF=\"#topurls\">Top %d URLs</A>\n", tfiles);
	(void) fprintf(ofp, " / <A HREF=\"#country\">Hits by Country</A>\n");

	(void) fprintf(ofp, "</B></FONT></TD></TR>\n" \
		FMT_SPACE("4") "</TABLE></CENTER>\n");

	/* the bar chart */
	if (!nocharts)
		(void) fprintf(ofp, "<P><CENTER><IMG SRC=\"%s\" ALT=\"diagram\" " \
				"WIDTH=\"492\" HEIGHT=\"316\"></CENTER><P>\n", mname);

	if (embed3D) {
		(void) fprintf(ofp, "<CENTER>\n<FONT SIZE=\"2\"><B>" \
			"3D-View (requires Netscape 2.01 with SGI OpenInventor plugin)" \
			"</B></FONT>\n</CENTER>\n<P>\n" \
			"<EMBED SRC=\"3Dstats%02d%02d.iv\" WIDTH=\"640\" HEIGHT=\"320\"><P>\n",
			tstart.mon+1, EPOCH(tstart.year));
	}

	/* the daily values */
	(void) fprintf(ofp, "<CENTER><A NAME=\"daysum\">" \
		"<TABLE WIDTH=\"492\" BORDER=\"2\" CELLSPACING=\"1\" CELLPADDING=\"1\"></A>\n");
	prDayTab(ofp, (int)tend.mday);
	(void) fprintf(ofp, FMT_SPACE("4") "</TABLE></CENTER>\n");

	if (!notoplist) {	/* the following diagram is considered part of the top lists */
		if (nocharts)
			(void) fprintf(ofp, "<P>\n<A NAME=\"avload\"><HR SIZE=\"4\"></A>\n<P>\n");
		else
			(void) fprintf(ofp, "<P><CENTER><A NAME=\"avload\"><IMG SRC=\"%s\" ALT=\"diagram\" " \
				"WIDTH=\"492\" HEIGHT=\"190\"></A></CENTER><P>\n", hname);

		/* the top lists */
		(void) fprintf(ofp, "<CENTER>" \
			"<TABLE WIDTH=\"492\" BORDER=\"2\" CELLSPACING=\"1\" CELLPADDING=\"1\">\n");
		(void) fprintf(ofp, FMT_SPACE("4") \
			FMT_HEAD("7", "The top 5 seconds of the period") \
			FMT_SPACE("4"));

	(void) fprintf(ofp, "<TR><TH BGCOLOR=\"#333333\"><FONT SIZE=\"-1\">No.</FONT></TH>" \
		"<TH COLSPAN=\"2\" BGCOLOR=\"#00AA00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hits</FONT></TH>" \
		"<TH COLSPAN=\"2\" BGCOLOR=\"#FFFF00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">304's</FONT></TH>" \
		"<TH BGCOLOR=\"#FF5000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">KBytes&nbsp;sent</FONT></TH> " \
		"<TH BGCOLOR=\"#00CCFF\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Date/Time</FONT></TH></TR>\n" \
		FMT_SPACE("4"));

		/* sort the top lists by accesses */
		qsort((void *)top_sec, TABSIZE(top_sec), sizeof(top_sec[0]), sort_top);
		qsort((void *)top_min, TABSIZE(top_min), sizeof(top_min[0]), sort_top);
		qsort((void *)top_hrs, TABSIZE(top_hrs), sizeof(top_hrs[0]), sort_top);

		for (idx=0; idx < (int)TABSIZE(top_sec); idx++)
			if (top_sec[idx].count > 0L) {
				PR_TOP(ofp, idx+1,
					top_sec[idx].count, PERCENT(top_sec[idx].count, total_hits),
					top_sec[idx].nomod, PERCENT(top_sec[idx].nomod, total_nomod),
					((u_long)top_sec[idx].bytes+1023L)/1024L);
				(void) fprintf(ofp, "<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\">" \
					"%02d/%3.3s/%d %02d:%02d:%02d</FONT></TD></TR>\n",
					top_sec[idx].tm.mday, monnam[top_sec[idx].tm.mon], top_sec[idx].tm.year,
					top_sec[idx].tm.hour, top_sec[idx].tm.min, top_sec[idx].tm.sec);
			}

		(void) fprintf(ofp, FMT_SPACE("4") FMT_HEAD("7", "The top 5 minutes of the period") FMT_SPACE("4"));
#if 0
		(void) fprintf(ofp, "<TR><TH BGCOLOR=\"#333333\"><FONT SIZE=\"-1\">No.</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#00AA00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hits</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#FFFF00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">304's</FONT></TH>" \
			"<TH BGCOLOR=\"#FF5000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">KBytes&nbsp;sent</FONT></TH> " \
			"<TH BGCOLOR=\"#00CCFF\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Date/Time</FONT></TH></TR>\n" FMT_SPACE("4"));
#endif

		for (idx=0; idx < (int)TABSIZE(top_min); idx++)
			if (top_min[idx].count > 0L) {
				PR_TOP(ofp, idx+1,
					top_min[idx].count, PERCENT(top_min[idx].count, total_hits),
					top_min[idx].nomod, PERCENT(top_min[idx].nomod, total_nomod),
					((u_long)top_min[idx].bytes+1023L)/1024L);
				(void) fprintf(ofp, "<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\">" \
					"%02d/%3.3s/%d %02d:%02d:SS</FONT></TD></TR>\n",
					top_min[idx].tm.mday, monnam[top_min[idx].tm.mon], top_min[idx].tm.year,
					top_min[idx].tm.hour, top_min[idx].tm.min);
			}

		(void) fprintf(ofp, FMT_SPACE("4") FMT_HEAD("7", "The top 24 hours of the period") FMT_SPACE("4"));
#if 0
		(void) fprintf(ofp, "<TR><TH BGCOLOR=\"#333333\"><FONT SIZE=\"-1\">No.</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#00AA00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hits</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#FFFF00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">304's</FONT></TH>" \
			"<TH BGCOLOR=\"#FF5000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">KBytes&nbsp;sent</FONT></TH> " \
			"<TH BGCOLOR=\"#00CCFF\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Date/Time</FONT></TH></TR>\n" FMT_SPACE("4"));
#endif

		for (idx=0; idx < (int)TABSIZE(top_hrs); idx++)
			if (top_hrs[idx].count > 0L) {
				PR_TOP(ofp, idx+1,
					top_hrs[idx].count, PERCENT(top_hrs[idx].count, total_hits),
					top_hrs[idx].nomod, PERCENT(top_hrs[idx].nomod, total_nomod),
					((u_long)top_hrs[idx].bytes+1023L)/1024L);
				(void) fprintf(ofp, "<TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\">" \
					"%02d/%3.3s/%d %02d:MM:SS</FONT></TD></TR>\n",
					top_hrs[idx].tm.mday, monnam[top_hrs[idx].tm.mon],
					top_hrs[idx].tm.year, top_hrs[idx].tm.hour);
			}
		(void) fprintf(ofp, FMT_SPACE("4") "</TABLE></CENTER>\n<P>\n");

	}

	if (tsites) {
		(void) fprintf(ofp, "<CENTER><A NAME=\"topsites\">" \
			"<TABLE WIDTH=\"492\" BORDER=\"2\" CELLSPACING=\"1\" CELLPADDING=\"1\"></A>\n");
		(void) fprintf(ofp, FMT_SPACE("4") \
			FMT_HEAD("7", "The top %d sites most often accessing your server") \
			FMT_SPACE("4"), tsites);

		(void) fprintf(ofp, "<TR><TH BGCOLOR=\"#333333\"><FONT SIZE=\"-1\">No.</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#00AA00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hits</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#FFFF00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">304's</FONT></TH>" \
			"<TH BGCOLOR=\"#FF5000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">KBytes&nbsp;sent</FONT></TH> " \
			"<TH BGCOLOR=\"#00CCFF\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hostname</FONT></TH></TR>\n");

		(void) fprintf(ofp, FMT_SPACE("4"));
			for (idx=0; idx < topn_sites; idx++) {
				if (top_sites[idx] == NULL)
					break;

				PR_TOP(ofp, idx+1,
					top_sites[idx]->count, PERCENT(top_sites[idx]->count, total_hits),
					top_sites[idx]->nomod, PERCENT(top_sites[idx]->nomod, total_nomod),
					((u_long)top_sites[idx]->bytes+1023L)/1024L);

				if ((nositelist || priv_dir) &&		/* strip sitename */
				    (cp=strchr(top_sites[idx]->str, '.')) && strchr(cp+1, '.'))
					(void) fprintf(ofp, "<TD ALIGN=\"LEFT\"><FONT SIZE=\"-1\">" \
						"<STRIKE>xxxx</STRIKE>%s</FONT></TD></TR>\n", cp);
				else
					(void) fprintf(ofp, "<TD ALIGN=\"LEFT\"><FONT SIZE=\"-1\">" \
						"%s</FONT></TD></TR>\n", top_sites[idx]->str);
			}
		(void) fprintf(ofp, FMT_SPACE("4") "</TABLE></CENTER>\n<P>\n");
	}

	if (tfiles) {
		(void) fprintf(ofp, "<CENTER><A NAME=\"topurls\">" \
			"<TABLE WIDTH=\"492\" BORDER=\"2\" CELLSPACING=\"1\" CELLPADDING=\"1\"></A>\n");
		(void) fprintf(ofp, FMT_SPACE("4") \
			FMT_HEAD("7", "The %d most commonly accessed URLs") FMT_SPACE("4"), tfiles);

		(void) fprintf(ofp, "<TR><TH BGCOLOR=\"#333333\"><FONT SIZE=\"-1\">No.</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#00AA00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hits</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#FFFF00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">304's</FONT></TH>" \
			"<TH BGCOLOR=\"#FF5000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">KBytes&nbsp;sent</FONT></TH> " \
			"<TH BGCOLOR=\"#00CCFF\"><FONT SIZE=\"-1\" COLOR=\"#000000\">URL</FONT></TH></TR>\n");

		(void) fprintf(ofp, FMT_SPACE("4"));
			for (idx=0; idx < topn_urls; idx++) {
				if (top_urls[idx] == NULL)
					break;

				PR_TOP(ofp, idx+1,
					top_urls[idx]->count, PERCENT(top_urls[idx]->count, total_hits),
					top_urls[idx]->nomod, PERCENT(top_urls[idx]->nomod, total_nomod),
					((u_long)top_urls[idx]->bytes+1023L)/1024L);

				if (*top_urls[idx]->str == '/')
					(void) fprintf(ofp, "<TD ALIGN=\"LEFT\"><FONT SIZE=\"-1\">" \
						"<A HREF=\"%s\">%s</A></FONT></TD></TR>\n",
						top_urls[idx]->str, top_urls[idx]->str);
				else
					(void) fprintf(ofp, "<TD ALIGN=\"LEFT\"><FONT SIZE=\"-1\">" \
						"%s</FONT></TD></TR>\n", top_urls[idx]->str);
			}
		(void) fprintf(ofp, FMT_SPACE("4") "</TABLE></CENTER>\n<P>\n");
	}

	if (lfiles) {
		(void) fprintf(ofp, "<CENTER>" \
			"<TABLE WIDTH=\"492\" BORDER=\"2\" CELLSPACING=\"1\" CELLPADDING=\"1\"></A>\n");
		(void) fprintf(ofp, FMT_SPACE("4") \
			FMT_HEAD("7", "The %d last frequently accessed URLs") FMT_SPACE("4"), lfiles);

		(void) fprintf(ofp, "<TR><TH BGCOLOR=\"#333333\"><FONT SIZE=\"-1\">No.</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#00AA00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hits</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#FFFF00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">304's</FONT></TH>" \
			"<TH BGCOLOR=\"#FF5000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">KBytes&nbsp;sent</FONT></TD> " \
			"<TH BGCOLOR=\"#00CCFF\"><FONT SIZE=\"-1\" COLOR=\"#000000\">URL</FONT></TD></TR>\n");

		(void) fprintf(ofp, FMT_SPACE("4"));
			for (idx=lastn_urls-1; idx >= 0; idx--) {
				if (last_urls[idx] == NULL)
					continue;

				PR_TOP(ofp, idx+1,
					last_urls[idx]->count, PERCENT(last_urls[idx]->count, total_hits),
					last_urls[idx]->nomod, PERCENT(last_urls[idx]->nomod, total_nomod),
					((u_long)last_urls[idx]->bytes+1023L)/1024L);

				if (*last_urls[idx]->str == '/')
					(void) fprintf(ofp, "<TD ALIGN=\"LEFT\"><FONT SIZE=\"-1\">" \
						"<A HREF=\"%s\">%s</A></FONT></TD></TR>\n",
						last_urls[idx]->str, last_urls[idx]->str);
				else
					(void) fprintf(ofp, "<TD ALIGN=\"LEFT\"><FONT SIZE=\"-1\">" \
						"%s</FONT></TD></TR>\n", last_urls[idx]->str);
			}
		(void) fprintf(ofp, FMT_SPACE("4") "</TABLE></CENTER>\n<P>\n");
	}

	/* generate country list */
	if ((clist = calloc(num_cntry, sizeof(COUNTRY *))) == NULL) {
		prmsg(1, enomem, num_cntry*sizeof(NLIST *));
		num_cntry = 0;
	} else {		/* create a linear list */
		for (idx=0; ccp = nextCountry(0); idx++)
			clist[idx] = ccp;

		/* save number of countries, sort the list by hits/cntry */
		if ((num_cntry = (size_t)idx) > 1)
			qsort((void *)clist, num_cntry, sizeof(COUNTRY *), sort_by_cntry);
		if (nocharts)
			(void) fprintf(ofp, "<P>\n<A NAME=\"country\"><HR SIZE=\"4\"></A>\n<P>\n");
		else
			(void) fprintf(ofp, "<P><CENTER>\n<A NAME=\"country\">" \
				"<IMG SRC=\"%s\" ALT=\"country pie chart\" " \
				"WIDTH=\"492\" HEIGHT=\"320\"></A>\n" \
				"</CENTER><P>\n", cname);

		(void) fprintf(ofp, "<CENTER>\n" \
			"<TABLE WIDTH=\"492\" BORDER=\"2\" CELLSPACING=\"1\" CELLPADDING=\"1\">\n");
		(void) fprintf(ofp, FMT_SPACE("4") \
			FMT_HEAD("7", "Total transfers by Country") FMT_SPACE("4"));

		(void) fprintf(ofp, "<TR><TH BGCOLOR=\"#333333\"><FONT SIZE=\"-1\">No.</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#00AA00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hits</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#FFFF00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">304's</FONT></TH>" \
			"<TH BGCOLOR=\"#FF5000\"><FONT SIZE=\"-1\" COLOR=\"#000000\">KBytes&nbsp;sent</FONT></TD> " \
			"<TH BGCOLOR=\"#00CCFF\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Country</FONT></TD></TR>\n");

		(void) fprintf(ofp, FMT_SPACE("4"));
		for (idx=0; idx < (int)num_cntry; idx++) {
			PR_TOP(ofp, idx+1,
				clist[idx]->count, PERCENT(clist[idx]->count, total_hits),
				clist[idx]->nomod, PERCENT(clist[idx]->nomod, total_nomod),
				((u_long)clist[idx]->bytes+1023L)/1024L);
			(void) fprintf(ofp, "<TD ALIGN=\"LEFT\"><FONT SIZE=\"-1\">" \
				"%s</FONT></TD></TR>\n", clist[idx]->name);
		}
		(void) fprintf(ofp, FMT_SPACE("4") "</TABLE></CENTER>\n<P>\n");
	}

#if defined(USER_AGENT)
	if (uag > 1) {
		(void) fprintf(ofp, "<CENTER>\n" \
			"<TABLE WIDTH=\"492\" BORDER=\"2\" CELLSPACING=\"1\" CELLPADDING=\"1\">\n");
		(void) fprintf(ofp, FMT_SPACE("4") \
			FMT_HEAD("4", "Most common used browsers") FMT_SPACE("4"));

		(void) fprintf(ofp, "<TR><TH BGCOLOR=\"#333333\"><FONT SIZE=\"-1\">No.</FONT></TH>" \
			"<TH COLSPAN=\"2\" BGCOLOR=\"#00AA00\"><FONT SIZE=\"-1\" COLOR=\"#000000\">Hits</FONT></TH>" \
			"<TH BGCOLOR=\"#00CCFF\"><FONT SIZE=\"-1\" COLOR=\"#000000\">User Agent</FONT></TD></TR>\n");

		(void) fprintf(ofp, FMT_SPACE("4"));
		for (idx=0; idx < uag; idx++) {
			(void) fprintf((ofp), \
				"<TR><TD ALIGN=\"CENTER\"><FONT SIZE=\"-1\"><B>%d</B></FONT></TD> " \
				"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD> " \
				"<TD ALIGN=\"RIGHT\"><FONT SIZE=\"-2\">%2.1f%%</FONT></TD> " \
				"<TD ALIGN=\"LEFT\"><FONT SIZE=\"-1\">%s</FONT></TD></TR>\n",
				idx+1, uatab[idx].count, PERCENT(uatab[idx].count, total_hits),
				uatab[idx].name);
		}
		(void) fprintf(ofp, FMT_SPACE("4") "</TABLE></CENTER>\n<P>\n");
	}
#endif

	(void) fprintf(ofp, "<P><HR SIZE=\"4\">\n");
	if (!nofilelist && pgfiles != NULL) {
		(void) fprintf(ofp, "<BR><FONT SIZE=\"-1\"><A HREF=\"%s\">" \
			"Total Transfers by Files</A> %s\n</FONT>\n",
			pgfiles, priv_dir ? authmsg : "");
		if (hfiles)
			(void) fprintf(ofp, "<BR><FONT SIZE=\"-1\"><A HREF=\"%s#hidden\">" \
				"Total Transfers by Hidden Items</A> %s\n</FONT>\n",
				pgfiles, priv_dir ? authmsg : "");

		free(pgfiles);
	}
	if (!nositelist && pgsites != NULL) {
		(void) fprintf(ofp, "<BR><FONT SIZE=\"-1\"><A HREF=\"%s\">" \
			"Total Transfers by Client Domain</A> %s\n</FONT>\n",
			pgsites, priv_dir ? authmsg : "");
		free(pgsites);
	}

	html_trailer(ofp, 1);
	(void) fclose(ofp);

	(void) mn_bars(492, 317, 28, mname);
	(void) hr_bars(492, 190, 28, hname);

	if (num_cntry)		/* create pie chart for countries */
		(void) c_chart(492, 320, 28, cname, clist, num_cntry);

	if (clist != NULL)			/* conditional cleanup */
		free(clist);

	return;
}

#define PR_SITE(ofp, c1, c2, c3, c4, c5, c6, c7)	\
    (void) fprintf((ofp), "%7lu %5.1f%% %6lu %5.1f%% %12.0f %5.1f%%   | %s\n", \
	(c1), (c2), (c3), (c4), (c5), (c6), (c7))

#define PR_URL(ofp, c1, c2, c3, c4, c5, c6, c7, c8)	\
    (void) fprintf((ofp), "%7lu %5.1f%% %6lu %5.1f%% %12.0f %5.1f%% %10lu  | %s\n", \
	(c1), (c2), (c3), (c4), (c5), (c6), (c7), (c8))

static char enosite[] = "The monthly list of sites will be omitted from the report.";
static char enofile[] = "The monthly list of files will be omitted from the report.";

/*
 * Print detailed statistics about all unique sites.
 */
static char *prSiteStats(char * const file, char * const back,
			 char * const period, NLIST ** const htab) {
	NLIST unres, *np, **list;
	size_t cnt = total_sites;
	int idx, ndx;
	FILE *ofp;

	unres.str = "Unresolved";
	unres.len = 10;
	unres.count = unres.nomod = unres.size = 0L;
	unres.bytes = 0.0;

	if ((list = calloc(cnt, sizeof(NLIST *))) == NULL) {
		prmsg(1, "Failed to allocate %u more bytes.\n%s\n",
			cnt*sizeof(NLIST *), enosite);
		return NULL;
	}

	/* create a linear list of hidden sites */
	for (idx=0, cnt=0; idx < HASHSIZE; idx++) {
		if ((np = htab[idx]) == NULL)
			continue;

		do {
			if (isdigit(*np->str)) {	/* unresolved */
				unres.count += np->count;
				unres.nomod += np->nomod;
				unres.bytes += np->bytes;
				unres.size++;		/* we use size for the sum of all unresolved hosts */
				continue;
			}
			/* check for hidden sitename, collect values */
			if (!isHiddenItem(HIDDEN_SITES, np))
				list[cnt++] = np;
		} while ((np=np->next) != NULL && cnt < total_sites);
	}

	/* add collected hidden sites to the list */
	for (idx=0; idx < TABSIZE(hstab); idx++) {
		if ((np=hstab[idx]) == NULL)
			continue;
		do {
			if (np->count)
				list[cnt++] = np;
		} while ((np=np->next) != NULL && cnt < total_sites);
	}

	/* now sort the list by accesses */
	qsort((void *)list, cnt, sizeof(NLIST *), sort_by_hits);

	if (topn_sites) {	/* create top site list, skip hidden sites */
		for (idx=ndx=0; idx < topn_sites && ndx < (int)cnt; ndx++) {
			if (*list[ndx]->str == '[')
				continue;

			top_sites[idx++] = list[ndx];
		}
		if (idx > 1)
			tsites = idx;
	}

	/* create list of countries */
	num_cntry = (size_t)addCountry(&unres, list, cnt);
	assert(num_cntry != 0);

	if (nofilelist) {		/* done already */
		free(list);
		return NULL;
	}

	if ((ofp=fopen(file, "w")) == NULL) {
		prmsg(1, "Can't open file `%s'.\n%s\n", file, enosite);
		return NULL;
	}

	html_header(ofp, period, 0);

	(void) fprintf(ofp, "<H3>Total Transfers by Client Domain</H3>\n<PRE>\n");
	(void) fprintf((ofp),
		"      Total      Total 304's\n" \
		"      Requests   (NoMod Req)          Bytes sent   | Domain\n" \
		"--------------------------------------------------------------------------------\n");
	if (unres.count != 0)
		PR_SITE(ofp, unres.count, PERCENT(unres.count, total_hits),
			unres.nomod, PERCENT(unres.nomod, total_nomod),
			unres.bytes, PERCENT(unres.bytes, total_bytes), unres.str);

	for (idx=0; idx < (int)cnt; idx++) {
		PR_SITE(ofp, list[idx]->count, PERCENT(list[idx]->count, total_hits),
			list[idx]->nomod, PERCENT(list[idx]->nomod, total_nomod),
			list[idx]->bytes, PERCENT(list[idx]->bytes, total_bytes), list[idx]->str);
	}
	(void) fprintf(ofp,
		"--------------------------------------------------------------------------------\n" \
		"%7lu 100.0%% %6lu 100.0%% %12.0f 100.0%%\n" \
		"================================================================================\n",
		total_hits, total_nomod, total_bytes);

	(void) fprintf(ofp, "</PRE>\n");

	if (back != NULL)
		(void) fprintf(ofp, "<P><HR SIZE=\"4\"><P>\n" \
			"<FONT SIZE=\"-1\"><A HREF=\"%s\">Back</A> " \
			"to the monthly summary of %s %d</FONT>\n",
			back, monnam[tstart.mon], tstart.year);

	html_trailer(ofp, 0);
	(void) fclose(ofp);	/* cleanup */
	free(list);
	return strsave(file);
}

/*
 * Print detailed statistics for all requested URLS.
 */
static char hrule1[] = \
	"--------------------------------------------------------------------------------";
static char hrule2[] = \
	"================================================================================";

static char *prURLStats(char * const file, char * const back,
			char * const period, NLIST ** const htab) {
	NLIST *np, **list;
	size_t cnt = uniq_urls;
	int idx, ndx, num;
	FILE *ofp;

	if ((list = calloc(cnt, sizeof(NLIST *))) == NULL) {
		prmsg(1, "Failed to allocate %d more bytes.\n%s\n",
			cnt*sizeof(NLIST *), enofile);
		return NULL;
	}

	/* create a linear list of hidden items */
	for (idx=0, cnt=0; idx < HASHSIZE && cnt < uniq_urls; idx++) {
		if ((np = htab[idx]) == NULL)
			continue;

		do {
			/* compute total bytes saved, check for hidden item */
			total_kbsaved += ((np->nomod*np->size)+1023L)/1024L;
			if (!isHiddenItem(HIDDEN_ITEMS, np))
				list[cnt++] = np;
		} while ((np=np->next) != NULL && cnt < uniq_urls);
	}

	/* add collected hidden items to the list */
	for (idx=0; idx < TABSIZE(hitab); idx++) {
		if ((np=hitab[idx]) == NULL)
			continue;
		do {
			if (np->count)
				list[cnt++] = np;
		} while ((np=np->next) != NULL && cnt < uniq_urls);
	}

	/* now sort the list by accesses */
	qsort((void *)list, cnt, sizeof(NLIST *), sort_by_hits);

	if (topn_urls) {	/* create top URL list, skip hidden items */
		for (idx=ndx=0; idx < topn_urls && ndx < (int)cnt; ndx++) {
			if (*list[ndx]->str == '[')
				continue;

			top_urls[idx++] = list[ndx];
		}
		if (idx > 1)
			tfiles = idx;
	}

	if (lastn_urls &&	/* conditionally create list of most boring URLs */
	    (lastn_urls+topn_urls < (int)cnt)) {
		for (idx=0, ndx=(int)cnt-1; idx < lastn_urls && ndx >= 0; ndx--) {
			if (*list[ndx]->str == '[')
				continue;

			last_urls[idx++] = list[ndx];
		}
		if (idx > 1)
			lfiles = idx;
	}

	if (nofilelist) {	/* we already have the list of the top URLs */
		free(list);
		return NULL;
	}

	/* print the statistics */
	if ((ofp=fopen(file, "w")) == NULL) {
		prmsg(1, "Can't open file `%s'.\n%s\n", file, enofile);
		free(list);
		return NULL;
	}

	html_header(ofp, period, 0);
	(void) fprintf(ofp, "<H3>Total Transfers by Files</H3>\n<PRE>\n");
	(void) fprintf(ofp,
		"      Total      Total 304's\n" \
		"      Requests   (NoMod Req)          Bytes sent       Size  | URL\n" \
		"%s\n", hrule1);

	for (idx=0; idx < (int)cnt; idx++) {
		PR_URL(ofp, list[idx]->count, PERCENT(list[idx]->count, total_hits),
			list[idx]->nomod, PERCENT(list[idx]->nomod, total_nomod),
			list[idx]->bytes, PERCENT(list[idx]->bytes, total_bytes),
			list[idx]->size, list[idx]->str);

	}
	for (idx=0; idx < (int)TABSIZE(RespCode); idx++) {
		if (idx == IDX_OK || idx == IDX_NOT_MODIFIED || RespCode[idx].bytes == 0.0)
			continue;
		PR_URL(ofp, RespCode[idx].count, PERCENT(RespCode[idx].count, total_hits),
			0L, 0.0, 
			RespCode[idx].bytes, PERCENT(RespCode[idx].bytes, total_bytes),
			0L, RespCode[idx].msg);
	}
	(void) fprintf(ofp,
		"%s\n%7lu 100.0%% %6lu 100.0%% %12.0f 100.0%%\n%s\n",
		hrule1, total_hits, total_nomod, total_bytes, hrule2);

	(void) fprintf(ofp, "\n</PRE>\n");

	if (!noitemlist) {
		/* create a linear list of all hidden items */
		for (idx=0, cnt=0; idx < HASHSIZE && cnt < uniq_urls; idx++) {
			if ((np = htab[idx]) == NULL)
				continue;

			do {
				if (np->ishidden >= 0 &&
				    np->ishidden < MAX_HIDDEN_ITEMS &&
				    hidden_items[np->ishidden].pfx != NULL &&
				    *hidden_items[np->ishidden].col->str != '[')
					list[cnt++] = np;
			} while ((np=np->next) != NULL && cnt < uniq_urls);
		}

		if ((hfiles = cnt) != 0) {
			qsort((void *)list, cnt, sizeof(NLIST *), sort_by_item);

			(void) fprintf(ofp, "<P><A NAME=\"hidden\"><HR SIZE=\"4\"></A>\n" \
				"<H3>Total Transfers by Hidden Items</H3>\n<PRE>\n");
			(void) fprintf(ofp,
"      Total      Total 304's\n" \
"      Requests   (NoMod Req)          Bytes sent       Size  | URL\n%s\n", hrule1);

			num = 0;
			ndx = -1;
			for (idx=0; idx < (int)cnt; idx++) {
				if (ndx != (int)list[idx]->ishidden) {
					if (ndx >= 0 &&
	strcmp(hidden_items[list[idx]->ishidden].col->str, hidden_items[ndx].col->str)) {
						(void) fprintf(ofp, "%s\n", hrule1);
						if (num > 1)
							(void) fprintf(ofp,
"%7lu %5.1f%% %6lu %5.1f%% %12.0f %5.1f%%\n%s\n",
hidden_items[ndx].col->count, PERCENT(hidden_items[ndx].col->count, total_hits),
hidden_items[ndx].col->nomod, PERCENT(hidden_items[ndx].col->nomod, total_nomod),
hidden_items[ndx].col->bytes, PERCENT(hidden_items[ndx].col->bytes, total_bytes), hrule2);
						num = 0;
					}
					ndx = (int)list[idx]->ishidden;
					(void) fprintf(ofp, "\n<B>%s</B> <FONT SIZE=\"-1\">(%s)</FONT>\n",
						hidden_items[ndx].col->str, hidden_items[ndx].pfx);
				}
				num++;
				PR_URL(ofp, list[idx]->count, PERCENT(list[idx]->count, total_hits),
					list[idx]->nomod, PERCENT(list[idx]->nomod, total_nomod),
					list[idx]->bytes, PERCENT(list[idx]->bytes, total_bytes),
					list[idx]->size, list[idx]->str);
			}
			if (num > 1)
				(void) fprintf(ofp,
"%s\n%7lu %5.1f%% %6lu %5.1f%% %12.0f %5.1f%%\n%s\n\n",
hrule1, hidden_items[ndx].col->count, PERCENT(hidden_items[ndx].col->count, total_hits),
hidden_items[ndx].col->nomod, PERCENT(hidden_items[ndx].col->nomod, total_nomod),
hidden_items[ndx].col->bytes, PERCENT(hidden_items[ndx].col->bytes, total_bytes), hrule2);
			else
				(void) fprintf(ofp, "%s\n\n", hrule1);
			(void) fprintf(ofp, "</PRE>\n");
		}
	}

	if (back != NULL)
		(void) fprintf(ofp, "<P><HR SIZE=\"4\"><P>\n" \
			"<FONT SIZE=\"-1\"><A HREF=\"%s\">Back</A> " \
			"to the monthly summary of %s %d</FONT>\n",
			back, monnam[tstart.mon], tstart.year);

	html_trailer(ofp, 0);

	(void) fclose(ofp);	/* cleanup */
	free(list);
	return strsave(file);
}


#if !defined(USE_FGETS)
#define MIN(a,b)	((a) < (b) ? (a) : (b))		/* WARNING: evaluates it's args more than once! */

/*
** Read a line from given file.
** This function may be more
*/
static int readLine(char *bp, size_t maxsize, FILE * const lfp) {
	static char fbuf[BEST_IO_SIZE];	/* I/O buffer */
	static char *next = NULL;	/* ptr to next char to read */
	static int saweof = 0;		/* set if read past EOF */
	static int nleft = 0;		/* numbers of chars left in fbuf */
	register int cnt = 0;		/* counter */
	register char *np;
	char *obp = bp;

	if (saweof)
		return -1;

	maxsize--;			/* preserve space for '\0' byte below */
	do {
		if (nleft == 0) {
			if ((nleft = read(fileno(lfp), fbuf, sizeof fbuf)) <= 0) {
				if (cnt > 0) {
					saweof = 1;
					nleft = 0;
					break;
				}
				return -1;
			}
			next = fbuf;
		}
		np = (char *)memccpy((void *)bp, (void *)next, '\n', MIN(maxsize, (size_t)nleft));
		if (np != NULL) {		/* line copied successfully */
			nleft -= np-bp;
			next += np-bp;
			*--np = '\0';		/* strip newline */
			cnt = np-obp;
			break;
		}
		if (maxsize < (size_t)nleft) {		/* buffer overflow */
			nleft -= (int)maxsize;
			break;
		}
		cnt = nleft;		/* read continuation line */
		maxsize -= (size_t)nleft;
		bp += nleft;
		nleft = 0;
	/*CONSTCOND*/
	} while (1);

	obp[cnt] = '\0';		/* terminate string in case of overflow */
	return cnt;
}
#endif

/*
** Read the logfile, parse the entries, fill in the
** elements of a LOGENT structure. Returns TRUE if
** an entry could be parsed, FALSE on EOF.
*/
#define SKIPMSG(msg,ln,lb)	prmsg(1, (msg), (ln), (lb))

static int readLog(FILE * const lfp, LOGENT * const ep, LOGTIME * const stp) {
	static char lbuf[BIGSIZE];	/* line buffer */
	static int lnum = 0;		/* line number */
	static u_long dtstart = 0L;	/* dayticks start time */
	register int len;		/* string length */
	register char *cp, *tm;		/* temp */
	char *sitename, *request;	/* temp sitename, request */
#if defined(USER_AGENT)
	char *uagent;			/* print stats for user agents */
#endif
	int reqmethod;			/* type of request (method) */
	u_int resp;			/* response code */
	size_t idx;			/* temp */
	size_t reqlen, sitelen;		/* length of sitename/request */
	u_long reqsize;
	LOGTIME *tp;

	if (lfp == NULL)
		return 0;

	if (!dtstart && stp->mday)
		dtstart = (u_long)(stp->mday+(stp->mon*31)+(stp->year*512));

#if defined(USE_FGETS)
# if defined(NS_FASTTRACK)	/* skip first line for Netscape fasttrack servers */
	if (fgets(lbuf, sizeof lbuf, lfp) == NULL)
		return 0;	/* got unexpected EOF */
	lnum++;
# endif
	while ((cp=fgets(lbuf, sizeof lbuf, lfp)) != NULL) {
		lnum++;
		for (len=0, tm=cp; *tm != '\0'; tm++)
			len++;
		if (lbuf[len-1] == '\n')		/* strip trailing newline */
			lbuf[--len] = '\0';
		if (*cp == '-' || *cp == '\0') {	/* skip empty lines */
			empty++;
			continue;
		}
#else	/* use a highly optimized version */
# if defined(NS_FASTTRACK)	/* skip first line for Netscape fasttrack servers */
	if (readLine(lbuf, sizeof lbuf, lfp) <= 0)
		return 0;	/* got unexpected EOF */
	lnum++;
# endif
	while ((len=readLine(lbuf, sizeof lbuf, lfp)) >= 0) {
		lnum++;
		cp = lbuf;
		if (len == 0 || *cp == '-') {		/* skip empty lines */
			empty++;
			continue;
		}
#endif
		request = cp+len;			/* save ptr to end of line */

		/* find the sitename, convert it to lower case */
		for (sitename=cp; *cp != '\0'; cp++, --len) {
			if (*cp == ' ' || *cp == '\t')
				break;
			MKLOWER(cp);		
		}

		if (*cp != ' ' && *cp != '\t') {
			SKIPMSG("Couldn't find the sitename, skip line %d: %s\n", lnum, lbuf);
			corrupt++;
			continue;
		}
		sitelen = (size_t)(cp-sitename);	/* remember length of sitename */
		*cp++ = '\0';

		for (--len; *cp && *cp != '['; cp++)	/* isolate date string */
			--len;

		if (*cp != '[' || *++cp == '\0' || --len < 27 || cp[26] != ']' ||
		    cp[2] != '/' || cp[6] != '/' || cp[11] != ':' || cp[14] != ':' || cp[17] != ':') {
			SKIPMSG("Couldn't find the date, skip line %d: %s\n", lnum, lbuf);
			corrupt++;
			continue;
		}
		tm = cp;
		cp += 26;
		*cp++ = '\0';

		/*
		 * Parse a date string in format DD/MMM/YYYY:HH:MM:SS
		 * and fill in the elements of the LOGTIME structure.
		 */
		tp = &ep->tm;

		tp->hour = (u_short) ((tm[12]-'0') * 10 + (tm[13]-'0'));
		tp->min  = (u_short) ((tm[15]-'0') * 10 + (tm[16]-'0'));
		tp->sec  = (u_short) ((tm[18]-'0') * 10 + (tm[19]-'0'));
		tp->mday = (u_short) ((tm[0]-'0') * 10 + (tm[1]-'0'));
		tp->year = (u_short) ((tm[7]-'0') * 1000 + (tm[8]-'0') * 100 +
				      (tm[9]-'0') * 10   + (tm[10]-'0'));

		switch (tm[4]) {
		  case 'a':		/* jan, mar, may */
			switch (tm[5]) {
			  case 'n':	tp->mon = 0;	break;
			  case 'r':	tp->mon = 2;	break;
			  default:	tp->mon = 4;	break;
			}
			break;

		  case 'u':		/* jun, jul, aug */
			switch (tm[5]) {
			  case 'n':	tp->mon = 5;	break;
			  case 'l':	tp->mon = 6;	break;
			  default:	tp->mon = 7;	break;
			}
			break;

		  case 'e':		/* feb, sec, dec */
			switch (tm[3]) {
			  case 'F':	tp->mon = 1;	break;
			  case 'S':	tp->mon = 8;	break;
			  default:	tp->mon = 11;	break;
			}
			break;

		  default:		/* apr, oct, nov */
			switch (tm[3]) {
			  case 'A':	tp->mon = 3;	break;
			  case 'O':	tp->mon = 9;	break;
			  default:	tp->mon = 10;	break;
			}
			break;
		}

		if (dtstart != 0L) {	/* skip entries which have been processed already */
			u_long drq = (u_long)(ep->tm.mday+(ep->tm.mon*31)+(ep->tm.year*512));
			if (drq < dtstart)
				continue;
		}

		while (*cp && *cp != '"')		/* check request */
			cp++;

		if (*cp != '"' || *++cp == '\0') {
			SKIPMSG("Couldn't find start of request, skip line %d: %s\n", lnum, lbuf);
			corrupt++;
			continue;
		}

		/*
		** Determine the request method and isolate the URL.
		** Take care for embedded `"' characters.
		*/
		reqmethod = METHOD_UNKNOWN;
		tm = cp;
		switch (*cp) {
		  case 'G':	cp++;
				if (*cp++ != 'E' || *cp++ != 'T')
					cp = tm;
				reqmethod = METHOD_GET;
				break;
		  case 'P':	cp++;
				if (*cp++ != 'O' || *cp++ != 'S' || *cp++ != 'T')
					cp = tm;
				reqmethod = METHOD_POST;
				break;
		  case 'H':	cp++;
				if (*cp++ != 'E' || *cp++ != 'A' || *cp++ != 'D')
					cp = tm;
				reqmethod = METHOD_HEAD;
				break;
		}
		if (reqmethod == METHOD_UNKNOWN)
			prmsg(1, "Unknown request method: %s\n", cp);

		if (*cp == ' ' || *cp == 't')
			cp++;

		tm = cp;	/* save start of request */
		cp = request;	/* set to end of request */

		while (cp > tm && *--cp != '"')
			/* noop */ ;

		if (*cp != '"') {
			SKIPMSG("Couldn't find end of request, skip line %d: %s\n", lnum, lbuf);
			corrupt++;
			continue;
		}
		*cp++ = '\0';

		/* now skip arguments, target names, and HTTP headers */
		for (request=tm; *tm != '\0'; tm++)
			if (*tm == '?' || *tm == '#' || *tm == ' ' || *tm == '\t')
				break;

		if (tm > request)
			*tm = '\0';
		if (!(reqlen = (size_t)(tm-request))) {	/* remember length of request */
			if (verbose)			/* print warning only in verbose mode */
				SKIPMSG("Empty request?!? skip line %d: %s\n", lnum, lbuf);
			empty++;
			continue;
		}

		/* parse response code */
		if ((*cp != ' ' && *cp != '\t') || *++cp == '\0') {
			SKIPMSG("Couldn't find the repsonse code, skip line %d: %s\n", lnum, lbuf);
			corrupt++;
			continue;
		}
		resp = 0;
		if (*cp != '-') {
			while (*cp >= '0' && *cp <= '9')
				resp = 10*resp + (u_int)(*cp++ - '0');
		} else
			cp++;

		/* parse request size */
		if ((*cp != ' ' && *cp != '\t') || *++cp == '\0') {
			SKIPMSG("Couldn't find the request size, skip line %d: %s\n", lnum, lbuf);
			corrupt++;
			continue;
		}
		reqsize = 0;
		if (*cp != '-') {
			while (*cp >= '0' && *cp <= '9')
				reqsize = 10*reqsize + (u_long)(*cp++ - '0');
		} else
			cp++;

		/* don't account for document size if request method was HEAD */
		if (reqmethod == METHOD_HEAD)
			reqsize = 0;

#if defined(USER_AGENT)
		uagent = NULL;			/* parse optional user agent */
		if ((*cp == ' ' || *cp == '\t') && *++cp != '\0') {
			for (uagent=cp; *cp && *cp != ' ' && *cp != '\t'; cp++)
				/* noop */ ;
			if (*cp == ' ' || *cp == '\t') {
				if (*(cp+1) == '(') {
					for (cp++; *cp && *cp != ')'; cp++)
						/* noop */ ;
					if (*cp == ')')
						cp++;
				}
				*cp++ = '\0';
			}
		}
#endif

		/* Truncate the name of the homepage and it's variations so they merge with `/' */
		for (idx=0; idx < hpnum; idx++) {
			if (reqlen >= (len = (int)hplen[idx]) &&
			    home_page[idx][1] == *(request+reqlen-len+1)) {
				tm = home_page[idx]+len;
				cp = request+reqlen;
				while (len-- > 0 && *--cp == *--tm)
					/* noop */ ;
				if (len < 0) {
					reqlen -= hplen[idx]+1;
					*++cp = '\0';	/* preserve trailing slash */
					break;
				}
			}
		}

		/*
		 * Decode hex sequences into character codes.
		 * Since the decoded string must be always shorter than the encoded
		 * version, we can safely use the same array for the resulting string.
		 */
		for (cp=tm=request; *cp != '\0'; cp++, tm++) {
			if (*cp == '%' && isxdigit(cp[1]) && isxdigit(cp[2])) {
				*tm = hextochar(cp+1);
				cp += 2;
				reqlen -= 2;
				continue;
			}
			*tm = *cp;
		}
		*tm = *cp;

		ep->sitename = sitename;
		ep->sitelen = sitelen;
		ep->request = request;
		ep->reqlen = reqlen;
		ep->reqsize = reqsize;

#if defined(USER_AGENT)
		ep->uagent = uagent;
#endif

		switch (resp) {
		  case 200:	ep->respidx = IDX_OK;		break;
		  case 204:	ep->respidx = IDX_NO_RESPONSE;	break;
		  case 302:	ep->respidx = IDX_REDIRECT;	break;
		  case 304:	ep->respidx = IDX_NOT_MODIFIED;	break;
		  case 400:	ep->respidx = IDX_BAD_REQUEST;	break;
		  case 401:	ep->respidx = IDX_UNAUTHORIZED;	break;
		  case 403:	ep->respidx = IDX_FORBIDDEN;	break;
		  case 404:	ep->respidx = IDX_NOT_FOUND;	break;
		  case 407:	ep->respidx = IDX_PROXY_UNAUTHORIZED; break;
		  case 500:	ep->respidx = IDX_SERVER_ERROR;	break;
		  case 501:	ep->respidx = IDX_NOT_IMPLEMENTED; break;
		  default:	ep->respidx = IDX_UNKNOWN;	break;
		}

#if defined(COUNT_EXTRA_BYTES)
		/* adjust sizes not counted for in the logfile */
		ep->reqsize += RespCode[ep->respidx].size;
#endif

#if 0
fprintf(stderr, "->> PROCESS %s[%u], %s[%u], %d (%s), %lu\n",
	ep->sitename, ep->sitelen, ep->request, ep->reqlen, ep->respidx,
	RespCode[ep->respidx].msg ? RespCode[ep->respidx].msg : "OK",
	ep->reqsize);
#endif

		return 1;		/* found an entry */
	}
	return 0;
}

/*
** Convert a hex sequence into a character.
*/
static u_char hextochar(char *cp) {
	u_int rc = 0;
	int i = 2;

	while (--i >= 0) {
		MKUPPER(cp);
		if (*cp >= 'A' && *cp <= 'Z')
			rc += (u_int)(*cp++ - 'A')+10;
		else	rc += (u_int)(*cp++ - '0');
		rc <<= i*4;
	}
	return (u_char)rc;
}

/*
** Lookup item in hash list.
*/
static int nomem = 0;

NLIST *lookupItem(NLIST ** const htab, size_t const hsize, char * const str) {
	register u_int hval = 0;
	register char *cp = str;
	register NLIST *np;
	size_t len = 0;

	while (*cp != '\0')	/* compute a hash value */
		hval += *cp++;	/* simple, but very fast */

	hval %= (u_int)hsize;
	len = cp - str;

	/* lookup the item in the given list */
	for (np=htab[hval]; np != NULL; np = np->next)
		if (len == np->len && !strcmp(np->str, str))
			break;

	if (np != NULL)		/* known already */
		return np;

	if (nomem)		/* new entry, but no more memory available */
		return NULL;	/* we can't do anything useful */

	/* create new entry */
	if ((np = (NLIST *)calloc(1, sizeof(NLIST))) == NULL ||
	    (np->str = (char *)malloc(len+1)) == NULL) {
		prmsg(1, enomem, sizeof(NLIST));
		nomem = 1;
		return NULL;
	}
	(void) strcpy(np->str, str);	/* initialize values */
	np->len = len;
	np->count = np->nomod = np->size = 0L;
        np->ishidden = -1;
        np->bytes = 0.0;
	np->next = htab[hval];		/* insert element into hash list */
	htab[hval] = np;
	return np;
}

/*
** Clear all items from the given list.
*/
static void clearItems(NLIST ** const htab, int const max) {
	NLIST *node, *np;
	size_t idx;

	for (idx=0; idx < max; idx++) {
		if ((node = htab[idx]) == NULL)
			continue;
			
		do {	np = node->next;
			free((void *)node->str);
			free((void *)node);
		} while ((node = np) != NULL);
		htab[idx] = NULL;
	}
	nomem = 0;	/* try again */
	return;
}

#if defined(USER_AGENT)
/*
** Count number of different user agents.
*/
static void userAgent(char * const name) {
	register char *tm, *cp;
	register u_int len = 0;
	register u_int sum = 0;
	register size_t idx;
	static int warned = 0;

	for (cp=name; *cp != '\0'; cp++, len++)	/* compute string characteristic */
		sum += *cp;

	for (idx=0; idx < uag; idx++) {		/* search list */
		if (uatab[idx].len == len && uatab[idx].sum == sum) {
			cp = uatab[idx].name;	/* this replaces a call to strcmp() */
			tm = name;
			while (*cp == *tm) {
				if (*cp == '\0') {	/* found it! */
					uatab[idx].count++;
					return;
				}
				cp++, tm++;
			}
		}
	}
	if (uag == TABSIZE(uatab)) {
		if (!warned) {
			warned = 1;
			prmsg(1, "Too few table entries for user agents (%d).\n" \
				"Resulting statistics for user agents may be incorrect.\n", uag);
		}
		return;
	}
	if ((uatab[uag].name = strsave(name)) != NULL) { /* insert new name */
		uatab[uag].len = len;
		uatab[uag].sum = sum;
		uatab[uag++].count = 1;
	}
	return;
}

/*
** Clear user agent table
*/
static void clearUATab(void) {
	size_t idx;

	for (idx=0; idx < uag; idx++)		/* free memory */
		if (uatab[idx].name != NULL)
			free(uatab[idx].name);

	uag = 0;				/* reset count, clear table */
	(void) memset((void *)uatab, '\0', sizeof uatab);
}
#endif

/*
** Clear all or some counters.
*/
static void clearCounter(int const all) {
	size_t idx;

	this_hits += total_hits;	/* add total hits to hits per session */

	total_bytes = 0.0;
	total_hits = total_files = total_nomod = total_sites = 0L;
	total_kbytes = uniq_urls = corrupt = empty = 0L;

	if (all) {
		(void) memset((void *)monthly_hits, '\0', sizeof monthly_hits);
		(void) memset((void *)monthly_files, '\0', sizeof monthly_files);
		(void) memset((void *)monthly_sites, '\0', sizeof monthly_sites);
		(void) memset((void *)monthly_nomod, '\0', sizeof monthly_nomod);
		(void) memset((void *)monthly_kbytes, '\0', sizeof monthly_kbytes);
		sum_hits = sum_files = sum_nomod = sum_sites = sum_kbytes = 0L;
	}

	(void) memset((void *)daily_hits, '\0', sizeof daily_hits);
	(void) memset((void *)daily_files, '\0', sizeof daily_files);
	(void) memset((void *)daily_sites, '\0', sizeof daily_sites);
	(void) memset((void *)daily_nomod, '\0', sizeof daily_nomod);
	(void) memset((void *)daily_kbytes, '\0', sizeof daily_kbytes);
	(void) memset((void *)hr_hits, '\0', sizeof hr_hits);
	(void) memset((void *)whr_hits, '\0', sizeof whr_hits);
	(void) memset((void *)top_sec, '\0', sizeof top_sec);
	(void) memset((void *)top_min, '\0', sizeof top_min);
	(void) memset((void *)top_hrs, '\0', sizeof top_hrs);

	if (top_sites != NULL)
		(void) memset((void *)top_sites, '\0', topn_sites*sizeof(NLIST *));
	if (top_urls != NULL)
		(void) memset((void *)top_urls, '\0', topn_urls*sizeof(NLIST *));
	if (last_urls != NULL)
		(void) memset((void *)last_urls, '\0', lastn_urls*sizeof(NLIST *));

	for (idx=0; idx < TABSIZE(RespCode); idx++) {
		RespCode[idx].count = 0L;
		RespCode[idx].bytes = 0.0;
	}

	return;
}

/*
** Read values from history file, initialize counters.
*/
static int readHistory(int const mflag, LOGTIME *const stp) {
	char hname[FILENAME_MAX];
	char lbuf[256];
	u_long hits, files, nomod, sites, kbytes;
	u_short md, mn, yr;
	FILE *hfp;

	(void) sprintf(hname, "stats.hist");
	if ((hfp = fopen(hname, "r")) == NULL)
		return mflag;

	while (fgets(lbuf, sizeof lbuf, hfp) != NULL) {
		if (sscanf(lbuf, "MON %hu/%hu: %lu %lu %lu %lu %lu\n",
			&mn, &yr, &hits, &files, &nomod, &sites, &kbytes) == 7) {
			mn--;
			assert(mn < (u_short)TABSIZE(monthly_hits));

			if ((yr == tstart.year && mn == tstart.mon) ||	/* skip current month */
			    (mn < tstart.mon && yr != tstart.year) ||	/* skip older/newer periods */
			    (mn > tstart.mon && yr != tstart.year-1))
				continue;

			monthly_hits[mn] = hits;
			monthly_files[mn] = files;
			monthly_nomod[mn] = nomod;
			monthly_sites[mn] = sites;
			monthly_kbytes[mn] = kbytes;
			continue;
		}
		if (!mflag &&			/* skip daily stats for monthly summary */
		    sscanf(lbuf, "DAY %hu/%hu/%hu: %lu %lu %lu %lu %lu",
			&md, &mn, &yr, &hits, &files, &nomod, &sites, &kbytes) == 8) {
			md--;
			assert(md < (u_short)TABSIZE(daily_hits));

			mn--;
			assert(mn < (u_short)TABSIZE(monthly_hits));

			if (mn != tcur.mon || yr != tcur.year)	/* skip expired data */
				continue;

			total_hits += (daily_hits[md] = hits);
			total_files += (daily_files[md] = files);
			total_nomod += (daily_nomod[md] = nomod);
			total_sites += (daily_sites[md] = sites);
			total_kbytes += (daily_kbytes[md] = kbytes);
			stp->mday = md+2;
			stp->mon = mn;
			stp->year = yr;
		}
	}
	(void) fclose(hfp);
	return 1;
}

/*
** Create or update the history file.
*/

static void writeHistory(int const mflag) {
	char hname[FILENAME_MAX];
	u_short sti;
	size_t idx;
	FILE *hfp;

	(void) sprintf(hname, "stats.hist");
	if ((hfp = fopen(hname, "w")) == NULL) {
		prmsg(1, enoent, hname, strerror(errno));
		return;
	}

	if (!mflag)
		sti = tend.mon;
	else {
		sti = tend.mon+1;	/* update counter */
		monthly_hits[tstart.mon] = total_hits;
		monthly_files[tstart.mon] = total_files;
		monthly_nomod[tstart.mon] = total_nomod;
		monthly_sites[tstart.mon] = total_sites;
		monthly_kbytes[tstart.mon] = total_kbytes;
	}

	if (!mflag && tend.mday > 1) {	/* save daily counters until last day done */
		(void) fprintf(hfp, "# DAILY COUNTERS for summary period 1-%d %3.3s %d\n",
			tend.mday-1, monnam[tend.mon], EPOCH(tend.year));
		for (idx=0; idx < (u_int)tend.mday-1; idx++) {
			(void) fprintf(hfp, "DAY %02d/%02d/%04d:\t%7lu %7lu %7lu %7lu %7lu\n",
				idx+1, tend.mon+1, tend.year,
				daily_hits[idx], daily_files[idx],
				daily_nomod[idx], daily_sites[idx], daily_kbytes[idx]);
		}
		(void) fprintf(hfp, "checksum\t%7lu %7lu %7lu %7lu %7lu\n\n",
			total_hits, total_files, total_nomod, total_sites, total_kbytes);
	}

	sum_hits = sum_files = sum_nomod = sum_sites = sum_kbytes = 0L;
	(void) fprintf(hfp, "# MONTHLY COUNTERS for last 12 month\n");
	for (idx=0; idx < 12; idx++) {
		(void) fprintf(hfp, "MON %02d/%04d:\t%7lu %7lu %7lu %7lu %7lu\n",
			idx+1, idx >= (size_t)sti ? (int)tend.year-1 : (int)tend.year,
			monthly_hits[idx], monthly_files[idx],
			monthly_nomod[idx], monthly_sites[idx],
			monthly_kbytes[idx]);
		sum_hits += monthly_hits[idx];
		sum_files += monthly_files[idx];
		sum_nomod += monthly_nomod[idx];
		sum_sites += monthly_sites[idx];
		sum_kbytes += monthly_kbytes[idx];
	}
	(void) fprintf(hfp, "checksum\t%7lu %7lu %7lu %7lu %7lu\n\n",
		sum_hits, sum_files, sum_nomod, sum_sites, sum_kbytes);

	(void) fclose(hfp);
	return;
}


/*
** Create a table with the weekdays for this month.
** Note that in Germany a week starts at Monday, so
** we have to adjust the table entries for the values
** used by Unix time functions.
*/
static void mkdtab(LOGTIME * const lt) {
	int wday;
	size_t idx;
	time_t now;
	struct tm *tp;

	tp = localtime(&now);		/* get current time to compensate for timezone */
	tp->tm_mday = 1;		/* tick back to first day of month, 00:00:00 */
	tp->tm_mon = (int)lt->mon;
	tp->tm_year = (int)lt->year-1900;
	tp->tm_hour = tp->tm_min = tp->tm_sec = 0;

	(void) mktime(tp);		/* adjust tm_wday */

	if ((wday = tp->tm_wday) == 0)	/* get first day of week, create table */
		wday = 6;
	else				/* localization */
		wday--;
	for (idx=0; idx < (int)TABSIZE(wdtab); idx++, wday++)
		wdtab[idx] = wday % 7;
	return;
}

/*
** Print a message.
** Level is 0 for informational messages,
** 1 for warnings, and 2 for fatal errors.
*/

static char *msg_type[3] = { "[INFO]", "[WARNING]", "[FATAL]" };

void prmsg(int const level, char * const fmt, ...) {
	va_list ap;

	if (level || !isatty(2))
		(void) fprintf(stderr, "%s %s: ", progname, msg_type[level%3]);

	va_start(ap, fmt);
	(void) vfprintf(stderr, fmt, ap);
	va_end(ap);
	return;
}
