/*
 * printers.c
 *
 * Recording information about the printers (named outputs)
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97, 98 Akim Demaille, Miguel Santana
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * $Id: printers.c,v 1.43 1998/03/02 08:57:09 demaille Exp $
 */
#define printer_table hash_table

#include "a2ps.h"
#include "jobs.h"
#include "printers.h"
#include "routines.h"
#include "xstrrpl.h"
#include "hashtab.h"
#include "printers.h"
#include "msg.h"
#include "metaseq.h"
#include "stream.h"
#include "ppd.h"
#include "pathwalk.h"

#define DEFAULT_PRINTER _("Default Printer")
#define UNKNOWN_PRINTER _("Unknown Printer")

/* Road map of this file
  1. struct printer and functions.
  2. struct printer_table and functions.
  3. interface for struct a2ps_job
*/
struct a2ps_printers_s {
  struct a2ps_common_s * common;/* Shared mem			*/

  struct printer_table * printers;/* User defined printers		*/

  /* PPD handling */
  char * default_ppdkey;	/* Key of the ppd to use as default	*/
  char * request_ppdkey;	/* User has specified			*/
  struct ppd * ppd;		/* Name of the ppd file asked		*/

  /* Selected destination */
  boolean output_is_printer;	/* output_name is about printer?	*/
  				/* Otherwise, it's a file name		*/
  char *output_name;		/* if is printer, NULL -> default printer
				   if not, NULL-> stdout		*/

};


/************************************************************************
 * hash tables for the printers						*
 ************************************************************************/
/*
 * Description of a printer
 */
struct printer {
  char * key;
  char * ppdkey;	/* Key to the ppd file which describe it */
  char * command;	/* Command to run to run the printer	*/
  boolean is_file;	/* File or shell command?		*/
};

/*
 * Basic routines
 */
static unsigned long
printer_hash_1 (struct printer *printer)
{
  return_STRING_HASH_1 (printer->key);
}

static unsigned long
printer_hash_2 (struct printer *printer)
{
  return_STRING_HASH_2 (printer->key);
}

static int
printer_hash_cmp (struct printer *x, struct printer *y)
{
  return_STRING_COMPARE (x->key, y->key);
}

/*
 * For sorting them in alpha order
 */
static int
printer_hash_qcmp (struct printer **x, struct printer **y)
{
  return strcoll ((*x)->key, (*y)->key);
}

/*
 * Kill
 */
static void
printer_free (struct printer * printer)
{
  FREE (printer->key);
  XFREE (printer->ppdkey);
  FREE (printer->command);
}

/*
 * Format the presentation of a printer and its command for
 * --list-printers.
 */
static void
printer_self_print (struct printer * printer, FILE * stream)
{
  fputs (printer->key, stream);
  if (printer->ppdkey)
    fprintf (stream, " (PPD: %s)\n\t", printer->ppdkey);
  else
    fputs ("\n\t", stream);

  switch (printer->is_file) {
  case false:
    fprintf (stream, _("pipe in %s\n"), printer->command);
    break;
    
  case true:
    fprintf (stream, _("save in %s\n"), printer->command);
    break;
  }
}

/*
 * TRUE iff not the default or unknwon printer
 */
static int
printer_is_named_p (const char * name)
{
  return (!strequ (name, DEFAULT_PRINTER)
          && !strequ (name, UNKNOWN_PRINTER));
}

/************************************************************************/
/* 2. Printer_table functions						*/
/************************************************************************/
/*
 * Create the structure that stores the list of printers
 */
static inline struct printer_table *
printer_table_new (void)
{
  NEW (struct hash_table, res);
  
  hash_init (res, 8,
	     (hash_func_t) printer_hash_1,
	     (hash_func_t) printer_hash_2,
	     (hash_cmp_func_t) printer_hash_cmp);

  return (printer_table *) res;
}

/*
 * Free the whole structure
 */
static inline void
printer_table_free (struct printer_table * table)
{
  hash_free (table, (hash_map_func_t) printer_free);
  FREE (table);
}

/*
 *  Add a pair, with your own allocation for them.
 * It KEY is yet used, override its value with VALUE
 */
static inline void
printer_table_add (struct printer_table * table,
		   const char * key, const char * ppdkey, 
		   const char * command, boolean is_file)
{
  struct printer * item, token;
  
  token.key = (char *) key;
  item = (struct printer *) hash_find_item (table, &token);

  if (item) {
    XFREE (item->command);
    XFREE (item->ppdkey);
  } else {
    item = ALLOC (struct printer, 1);
    item->key = xstrdup(key);
  }
  
  item->ppdkey = ppdkey ? xstrdup (ppdkey) : NULL;
  item->command = command ? xstrdup (command) : NULL;
  item->is_file = is_file;

  hash_insert (table, item);
}

static inline struct printer *
printer_table_get_internal (struct printer_table * table, const char * key)
{
  struct printer * item, token;
  
  token.key = (char *) key;
  item = (struct printer *) hash_find_item (table, &token);

  /* Returns NULL if not in the table */
  return item;
}

/*
 * Resoves Default and Unknown printers
 */
static inline struct printer *
printer_table_get (struct printer_table * table, const char * key)
{
  struct printer * item, token;

  /* Is this the default printer? */
  if (IS_EMPTY (key))
    {
      item = printer_table_get_internal (table, DEFAULT_PRINTER);
      if (!item)
	error (1, 0, _("no default command for option `-d'"));
      return item;
    }


  /* Is this a known printer? */
  token.key = (char *) key;
  item = (struct printer *) hash_find_item (table, &token);
  if (item)
    return item;

  /* Is a unknown printer */
  item = printer_table_get_internal (table, UNKNOWN_PRINTER);
  if (!item)
    error (1, 0, _("no default command for unknown printer `%s'"), key);
  return item;
}


/*
 * Return its ppd key
 */
static const char *
a2ps_printer_table_get_ppd (struct a2ps_printers_s * printers,
			    const char * key)
{
  struct printer * printer;
  
  printer = printer_table_get (printers->printers, key);
  if (printer)
    return printer->ppdkey;
  else
    return NULL;
}

/*
 * Report content (short form)
 */
static void
printer_table_short_self_print (struct printer_table * table, FILE * stream)
{
  int i, tab = 0;
  struct printer ** entries;

  entries = (struct printer **) 
    hash_dump (table, NULL,
	       (hash_cmp_func_t) printer_hash_qcmp);

  for (i = 0 ; entries[i] ; i++) {
    if (!printer_is_named_p (entries[i]->key))
      continue;
    if (!(tab++ % 7))
      fputs ("\n  ", stream);
    fprintf (stream, "%-10s", entries[i]->key);
  }
  putc ('\n', stream);
  FREE (entries);
}

/*
 * Report content (long form)
 */
static void
printer_table_self_print (struct printer_table * table, FILE * stream)
{
  hash_maparg (table, (hash_maparg_func_t) printer_self_print,
	       stream, (qsort_cmp_t) printer_hash_qcmp);
}


/*
 * Open a stream on the specified output in JOB
 */
void
a2ps_open_output_stream (struct a2ps_job * job)
{
  /* Open the destination */
  if (job->printers->output_is_printer) 
    {
      /* Open an output stream onto PRINTER */
      const char * command;
      struct printer * printer;
      printer = printer_table_get (job->printers->printers, job->printers->output_name);
      
      /* Expand the metaseq before */
      command = (char *) expand_user_string (job, FIRST_FILE (job),
					     (uchar *) "output command", 
					     (uchar *) printer->command);
      job->output_stream = stream_wopen (command, printer->is_file);
    } 
  else
    {
      /* Back up the file before */
      if (! IS_EMPTY (job->printers->output_name))
	xbackup_file (job->printers->output_name);
      job->output_stream = stream_wopen (job->printers->output_name, TRUE);
    }
}

/*
 * Close the output stream with fclose or pclose
 */
void
a2ps_close_output_stream (struct a2ps_job * job)
{
  stream_close (job->output_stream);
}

/*
 * Send a file as it is to stdout.
 * This is used for pass_trough files (e.g. PS files)
 */
void
file_into_stdout (const void * arg)
{
  const char * filename = (const char *) arg;
  FILE * fp;

  message (msg_file, (stderr, "Dumping `%s' onto stdout.\n", filename));
  fp = xrfopen (filename);
  streams_copy (fp, stdout);
  fclose (fp);
}

/*
 * Make a standard message upon the destination
 */
static void
destination_to_string_internal (const char * name, boolean is_file,
				uchar * buf)
{
  if (is_file) {
    if (IS_EMPTY (name))
      sprintf ((char *) buf, _("sent to the standard output"));
    else
      sprintf ((char *) buf, _("saved into the file `%s'"), name);
  } else {
    /* NAME is about a printer */
    if (IS_EMPTY (name) || strequ (name, DEFAULT_PRINTER))
      sprintf ((char *) buf, _("sent to the default printer"));
    else
      sprintf ((char *) buf, _("sent to the printer `%s'"), name);
  }
}

/*
 * Return a string that describes the destination of the
 * file.
 */
void
destination_to_string (a2ps_job * job, uchar * buf)
{
  /* Make a nice message to tell where the output is sent */
  if (job->printers->output_is_printer)
    destination_to_string_internal (job->printers->output_name, false, buf);
  else
    destination_to_string_internal (job->printers->output_name, true, buf);
}

/*
 * The same, but after evaluation
 */
void
final_destination_to_string (a2ps_job * job, uchar * buf)
{
  /* The main difference is when sending to a file, in which
   * case we want to have its real name */
  struct printer * printer;

  if (job->printers->output_is_printer)
    {
      /* -P or -d */
      printer = printer_table_get (job->printers->printers, 
				   job->printers->output_name);

      if (printer->is_file) {
	char * filename;
	/* Its a file: report its real name (i.e. expand the command) */
	filename = (char *) expand_user_string (job, FIRST_FILE (job),
						(uchar *) "output command", 
						(uchar *) printer->command);
	destination_to_string_internal (filename, true, buf);
      } else {
	/* Its a `printer': report its name/key.
	 * We give job->command_name because it means default printer
	 * when empty, though printer->key is the default printer
	 * when equal to DEFAULT_PRINTER */
	destination_to_string_internal (job->printers->output_name, false, buf);
      }
    } 
  else
    {
      /* -o (can be stdout) */
      destination_to_string_internal (job->printers->output_name, true, buf) ;
    }
}

/************************************************************************/
/*	Handling the printers module					*/
/************************************************************************/
/*
 * Create the mem of the printers module
 */
struct a2ps_printers_s *
a2ps_printers_new (struct a2ps_common_s * common)
{
  NEW (struct a2ps_printers_s, res);

  /* Shared mem */
  res->common = common;

  /* Available printers (outputs) */
  res->printers = printer_table_new ();  
  
  /* PPD */
  res->request_ppdkey = NULL;
  res->default_ppdkey = xstrdup ("level1"); /* By default, level1 PS */
  res->ppd = NULL;		/* Printer's ppd are not read yet */

  /* Output */
  res->output_is_printer = true;/* Default is to send to default printer */
  res->output_name = NULL;

  return res;
}

/*
 * Release the mem used by a PRINTERS module
 * The module is freed
 */
void
a2ps_printers_free (struct a2ps_printers_s * printers)
{
  /* Don't free common, a2ps_job is in charge */

  printer_table_free (printers->printers);

  /* PPD */
  XFREE (printers->request_ppdkey);
  XFREE (printers->default_ppdkey);
  ppd_free (printers->ppd);

  /* Output */
  XFREE (printers->output_name);

  FREE (printers);
}

/*
 * Finalize the printers module.
 * This is called once the configuration/options have been totally
 * parsed/treated
 */
void
a2ps_printers_finalize (struct a2ps_printers_s * printers)
{
  const char * ppdkey;

  /* 1. Get the right ppd key */
  if ((ppdkey = printers->request_ppdkey))
    /* Nothing */;
  else if (printers->output_is_printer)
    ppdkey = a2ps_printer_table_get_ppd (printers, 
					 printers->output_name);
  if (!ppdkey)
    ppdkey = printers->default_ppdkey;

  /* 2. Get the struct ppd */
  printers->ppd = _a2ps_ppd_get (printers->common->path, ppdkey);
  /* FIXME: Check for errors */
}


/*
 * Record a printer defined by the config line "Printer:"
 */
boolean
a2ps_printers_add (struct a2ps_printers_s * printers,
		   const char * key, char * definition) 
{
  char * ppdkey = NULL;
  char * token = NULL;
  char * command = NULL;
  boolean is_file;

  /* Skip the blanks */
  token = definition + strspn (definition, " \t");

  /* PPD given ? */
  if ((*token != '>') && (*token != '|'))
    {
      /* If the first token does not start by `|' or `>', then ppdkey
       * is defined */
      ppdkey = strtok (token, " \t");
      token = strtok (NULL, "\n");
    }

  /* Skip the blanks */
  token += strspn (token, " \t");

  switch (*token) {
  case '|':
    is_file = false;
    break;

  case '>':
    is_file = true;
    break;

  default:
    return false;
  }

  /* Remains the command itself */
  command = token + 1 + strspn (token + 1, " \t");

  printer_table_add (printers->printers, key, ppdkey, command, is_file);
  return TRUE;
}

/*
 * Accessing the PPD fields
 */
const char *
a2ps_printers_default_ppdkey_get (struct a2ps_printers_s * printers)
{
  return printers->default_ppdkey;
}

void
a2ps_printers_default_ppdkey_set (struct a2ps_printers_s * printers,
				  const char * ppdkey)
{
  xstrcpy (printers->default_ppdkey, ppdkey);
}

const char *
a2ps_printers_request_ppdkey_get (struct a2ps_printers_s * printers)
{
  return printers->request_ppdkey;
}

void
a2ps_printers_request_ppdkey_set (struct a2ps_printers_s * printers,
				  const char * ppdkey)
{
  xstrcpy (printers->request_ppdkey, ppdkey);
}

/*
 * Accessing the output fields
 */
void
a2ps_printers_output_set (struct a2ps_printers_s * printers,
			  const char * output_name,
			  boolean is_printer)
{
  printers->output_is_printer = is_printer;

  if (!is_printer && output_name && strequ (output_name, "-"))
    {
      /* Request for stdin */
      XFREE (printers->output_name);
      printers->output_name = NULL;
    }
  else
    xstrcpy (printers->output_name, output_name);
}

const char *
a2ps_printers_output_name_get (struct a2ps_printers_s * printers)
{
  return (const char *) printers->output_name;
}

boolean
a2ps_printers_output_is_printer_get (struct a2ps_printers_s * printers)
{
  return printers->output_is_printer;
}

/*
 * Questioning the printers module
 */
int
a2ps_printers_font_known_p (struct a2ps_printers_s * printers, 
			    const char * name)
{
  return ppd_font_known_p (printers->ppd, name);
}

/*
 * Interface to job
 */
void
a2ps_printers_list_short (struct a2ps_job * job, FILE * stream)
{
  fputs (_("Known output destination (printers, etc.)"), stream);
  printer_table_short_self_print (job->printers->printers, stream);
}

void
a2ps_printers_list_long (struct a2ps_job * job, FILE * stream)
{
  fputs (_("Known output destination (printers, etc.)"), stream);
  putc ('\n', stream);

  printer_table_self_print (job->printers->printers, stream);
}

void
a2ps_ppd_list_short (struct a2ps_job * job, FILE * stream)
{
  _a2ps_ppd_list_short (job->common.path, stream);
}

void
a2ps_ppd_list_long (struct a2ps_job * job, FILE * stream)
{
  _a2ps_ppd_list_long (job->common.path, stream);
}

