/*
 * confg.c
 *
 * Read and understanding everything about the options 
 * & (dynamic) configuration of a2ps.
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97, 98 Akim Demaille, Miguel Santana
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * $Id: options.c,v 1.45 1998/03/03 11:46:23 demaille Exp $
 */


/************************************************************************/
/*									*/
/*			I n c l u d e   f i l e s			*/
/*                                                                      */
/************************************************************************/
#include "a2ps.h"
#include "routines.h"
#include "options.h"
#include "argmatch.h"
#include "getopt.h"
#include "caret.h"
#include "msg.h"
#include "jobs.h"
#include "useropt.h"
#include "prange.h"
#include "metaseq.h"

#define MAN_LINES               66	/* no lines for a man */
extern char * program_name;

/*
 * Hooks used
 */
option_hook handle_option_hook = NULL;



static void
print_argv (const char * string, char * argv[])
{
  int i;
  fprintf (stderr, "%2d %-10s ", optind, string);
  for (i = 0 ; argv [i] ; i++)
    fprintf (stderr, "%-10s", argv[i]);
  fputs ("\n", stderr);
}

static struct option long_options[] =
{
  /* Tasks */
  {"help",		no_argument,		0, 'h'},
  {"guess",		no_argument,		0, 138},
  {"list", 		required_argument,	0, 145},
  {"copyright", 	no_argument,		0, 174},
  {"version", 		no_argument,		0, 'V'},
  /* these ones are a courtesy for those who are used to enscript */
  {"list-options", 	no_argument,		0, 139},
  {"list-media", 	no_argument,		0, 161},
  {"list-style-sheets",	no_argument,		0, 162},
  {"help-languages",	no_argument,		0, 162},
  {"help-pretty-print",	no_argument,		0, 162},

  /* Global */
  {"macro-meta-sequence",required_argument,	0, 137},
  {"user-option",	required_argument,	0, '='},

  /* Pretty Print */
  {"pretty-print",	optional_argument,	0, 'E'},

  /* PostScript */
  {"ppd",		optional_argument,	0, 163},
  {"prologue",		required_argument,	0, 134},
  {"include",		required_argument,	0, 134},
  {"sides", 		required_argument,	0, 's'},
  {"statusdict",	required_argument,	0, 'S'},
  {"setpagedevice",	required_argument,      0, 'D'},

  /* Not sorted yet */
  {"columns",		required_argument,	0, 132},
  {"rows",		required_argument,	0, 133},
  {"compact",		required_argument,      0, 150}, /* -A */
  {"header",		optional_argument,	0, 'b'},
  {"no-header", 	no_argument, 		0, 'B'},
  {"truncate-lines",	required_argument,	0, 151}, /* -c */
  {"line-numbers",	required_argument,      0, 152}, /* -C */


  {"font-size",		required_argument, 	0, 'f'},

  /* Obsolete, replaced by the following */
  {"graphic-symbols", 	required_argument, 	0, 154}, /* -g */
  {"highlight-level", 	required_argument, 	0, 173},

  {"interpret", 	required_argument,	0, 155}, /* -i */
  {"end-of-line", 	required_argument,	0, 169},
  {"borders", 		required_argument, 	0, 156}, /* -j */
  {"page-prefeed",	no_argument,		0, 'k'},
  {"no-page-prefeed",	no_argument,		0, 'K'},  
  {"lines-per-page",	required_argument,	0, 'L'},
  {"chars-per-line",	required_argument,	0, 'l'},
  {"catman",		no_argument,		0, 'm'},
  {"medium", 		required_argument,	0, 'M'},
  /* Convenience for enscript users */
  {"media", 		required_argument,	0, 'M'},
  {"copies", 		required_argument,	0, 'n'},
  {"output", 		required_argument,	0, 'o'},
  {"printer", 		optional_argument,	0, 'P'},
  {"quiet", 		no_argument,		0, 'q'},
  {"silent",		no_argument,		0, 'q'},
  {"landscape", 	no_argument,		0, 'r'},
  {"portrait", 		no_argument,		0, 'R'},
  {"title", 		optional_argument,	0, 't'},
  {"tabsize", 		required_argument,	0, 'T'},
  {"underlay", 		required_argument,	0, 'u'},
  {"verbose", 		optional_argument,	0, 'v'},
  {"encoding", 		required_argument,	0, 'X'},


  {"non-printable-format",required_argument,	0, 135},
  {"print-anyway",	required_argument,	0, 136},

  {"center-title",	optional_argument,	0, 149},
  {"left-title",	optional_argument,	0, 140},
  {"right-title",	optional_argument,	0, 141},
  {"left-footer",	optional_argument,	0, 142},
  {"footer",		optional_argument,	0, 143},
  {"right-footer",	optional_argument,	0, 144},
  {"stdin",		required_argument,	0, 166},

  {"margin",		optional_argument,	0, 147},
  {"strip-level",	required_argument,	0, 148},
  {"major",		required_argument,	0, 157},
  {"version-control",	required_argument,	0, 158},
  {"suffix",		required_argument,	0, 159},

  {"debug",	 	no_argument,		0, 146},

  {"delegate",	 	required_argument,	0, 160},
  /* Courtesy for enscript */
  {"pass-through", 	required_argument,	0, 160},

  {"toc",		optional_argument,	0, 167},
  {"pages", 		optional_argument,	0, 'a'},

  /* Free: 164, 165, 168, 170, 171, 172 Next to use is: 175 */
  {NULL, 0, 0, 0}
};

#define OPT_STRING \
"123456789=:Aa:b::BcCdD:E::f:gGhijkKl:L:mM:n:o:P:qrRs:S:t::T:u::v::VX:Z"

/************************************************************************/
/*				arguments				*/
/************************************************************************/
/*
 * Return the index of the right answer
 */
int
a2ps_get_symbolic_value (const char * const args[], 
			 const void * values,
			 const char * opt,
			 const char * arg)
{
  int i, last_value = 0;
  
  i = argmatch (arg, args);

  if (i >= 0)
    return i;

  /* There is an error */
  if (i == -1)
    error (0, 0, _("invalid argument `%s' for option `%s'"),
	   arg, opt);
  else				/* Assume -2.  */
    error (0, 0, _("ambiguous argument `%s' for option `%s'"),
	   arg, opt);

  /* We try to put synonyms on the same line.
   * The assumption is that they follow each other */
  fprintf (stderr, _("Valid arguments are:"));
  for (i = 0 ; args[i] ; i++)
    if ((i == 0)
	|| (last_value != ((const int *) values)[i])) {
      fprintf (stderr, "\n  * `%s'", args[i]);
      last_value = ((const int *) values)[i];
    } else {
      fprintf (stderr, ", `%s'", args[i]);
    }
  putc ('\n', stderr);
  exit (EXIT_BADARG);
  return -1; 	/* To please some compilers */
}

static void
bad_arg (const char * opt, char * arg)
{
  error (0, 0, _("wrong value for option %s: `%s'"), opt, arg);
  fprintf (stderr, _("Try `%s --help' for more information.\n"), program_name);
  exit (EXIT_BADARG);
}

/************************************************************************/
/* Helping routines for the options' arguments				*/
/************************************************************************/
/*
 * What about the backup machinery
 * (inspired by "getversion.c" by David MacKenzie <djm@gnu.ai.mit.edu>
 */
static const char *const backup_args[] =
{
  "none", "off", 
  "never", "simple", 
  "nil", "existing", 
  "t", "numbered", 
  0
};

static const enum backup_type backup_types[] =
{
  none, none, 
  simple, simple, 
  numbered_existing, numbered_existing, 
  numbered, numbered
};

/* Return the type of backup indicated by VERSION.
   Unique abbreviations are accepted. */
enum backup_type
a2ps_get_backup_type (const char *version)
{
  int i;
  
  if (version == 0 || *version == 0)
    return numbered_existing;
  i = a2ps_get_symbolic_value (backup_args, backup_types, 
			       "--version-control", version);
  return backup_types[i];
}

/*
 * What about the booleans
 */
static const char *const boolean_args[] =
{
  "yes", "on", "1",
  "no", "off", "0",
  0
};

static const boolean boolean_types[] =
{
  true, true, true,
  false, false, false
};

/*
 * Return the boolean value
 */
boolean
a2ps_get_boolean (const char * option, const char * arg)
{
  int i = a2ps_get_symbolic_value (boolean_args, boolean_types,
			      option, arg);
  return boolean_types[i];
}

int
a2ps_get_integer_in_range (const char * option, const char * arg,
			   int min, int max)
{
  char buf[256];
  int res;

  if ((sscanf (optarg, "%d%255s", &res, buf) != 1)
      || ((min >= 0) && (res < min))
      || ((max >= 0) && (res > max)))
    {
      error (0, 0, _("invalid argument `%s' for option `%s'"),
	     arg, option);
      if (min >= 0 && max >= 0)
	fprintf (stderr, _("Valid arguments are integers between %d and %d\n"),
		 min, max);
      else if (min >= 0)
	fprintf (stderr, _("Valid arguments are integers greater than %d\n"),
		 min);
      else
	fprintf (stderr, _("Valid arguments are integers smaller than %d\n"),
		 max);
      exit (EXIT_BADARG);
    }
  return res;
}

/*
 * Return the non printable format
 */
static const char *const non_printable_args[] =
{
  "octal",
  "hexa",
  "emacs",
  "questionmark",
  "space", "white", "blank",
  "caret",
  0
};

static const enum unprintable_format non_printable_types[] =
{
  octal,
  hexa,
  Emacs,
  question_mark,
  space, space, space,
  caret
};

static enum unprintable_format
get_non_printable_format (const char * option, const char * arg)
{
  int i = a2ps_get_symbolic_value (non_printable_args, non_printable_types,
				  option, arg);
  return non_printable_types[i];
}

/*
 * Return a major mode
 */  
static const char *const major_args[] =
{
  "rows", "columns", 0
};

static MAJOR major_types[] =
{
  major_rows, major_columns
};

static MAJOR
get_major (const char * option, const char * arg)
{
  int i = a2ps_get_symbolic_value (major_args, major_types,
				  option, arg);
  return major_types[i];
}

/*
 * Return the ratio to inch
 */  
static const char *const length_args[] =
{
  "inchs",
  "points",
  "cm", "centimeters",
  0
};

static float length_types[] =
{
  1.0,
  72.0,
  2.54, 2.54
};

/*
 * Return a value in inches.
 */
static float
get_length (const char * option, const char * arg, float min, float max)
{
  float res;
  int i;
  char buf[256];

  switch (sscanf (arg, "%f%255s", &res, buf)) {
  case 2:
    fprintf (stderr, "%f  %s\n", res, buf);
    i = a2ps_get_symbolic_value (length_args, length_types,
				 option, buf);
    res *= length_types [i];
    break;

  case 1:
    break;

  default:
    error (1, 0, "scanf failed");
  }
  
  if (((min >= 0) && (res < min))
      || ((max >= 0) && (res > max)))
    {
      error (0, 0, _("invalid argument `%s' for option `%s'"),
	     arg, option);
      fprintf (stderr,
	       _("Valid arguments are lengths between %.1fin and %.1fin\n"),
	       min, max);
      exit (EXIT_BADARG);
    }
  
  return res;
}

/************************************************************************/
/*	Handle the options						*/
/************************************************************************/
int
a2ps_handle_option (a2ps_job * job, int argc, char *argv[])
{
  int c;

  /* Reset optind so that getopt is reinitialized. */
  optind = 0;
  
  while (1)
    {
      /* getopt_long stores the index of the option here. */
      int option_index = 0;

      c = getopt_long (argc, argv, OPT_STRING, long_options, &option_index);

      if (c == EOF)
        break;

      /* 
       * First the option is passed to the application, in case
       * the application wants to redefine some options
       */
      if (handle_option_hook && handle_option_hook (c, optarg))
	/* The option has been handled, continue processing */
	continue;

      /* The option has not been recognized by the application,
       * let the lib handle it */
      switch (c) {
      case '1':				/* 1 logical page per sheet */
	job->columns = 1;
	job->rows = 1;
	job->orientation = portrait;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '2':				/* twin pages */
	job->columns = 2;
	job->rows = 1;
	job->orientation = landscape;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '3':				/* 3 virtual pages */
	job->columns = 3;
	job->rows = 1;
	job->orientation = landscape;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '4':				/* 4 virtual pages */
	job->columns = 2;
	job->rows = 2;
	job->orientation = portrait;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '5':				/* 5 virtual pages */
	job->columns = 5;
	job->rows = 1;
	job->orientation = landscape;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '6':				/* 6 virtual pages */
	job->columns = 3;
	job->rows = 2;
	job->orientation = landscape;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '7':				/* 7 virtual pages */
	job->columns = 7;
	job->rows = 1;
	job->orientation = landscape;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '8':				/* 8 virtual pages */
	job->columns = 4;
	job->rows = 2;
	job->orientation = landscape;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '9':				/* 9 virtual pages */
	job->columns = 3;
	job->rows = 3;
	job->orientation = portrait;
	job->columns_requested = 80;
	job->lines_requested = 0;
	job->Major = major_rows;
	break;

      case '=':				/* A user option */
	{
	  /* Caution with recursive calls to getopt.
	   * Actually, this is not enough: there is an error if the
	   * the user option if after the files.  I don't understand
	   * why (getopt internals) nor see what to do. */
	  int saved_optind = optind;
	  a2ps_handle_string_options (job, user_option_get (job, optarg));
	  optind = saved_optind;
	}
	break;

      case 'a':			/* --pages= pages to print */
	a2ps_page_range_set_string (job, optarg);
	break;

      case 'A':				/* allow two files per sheet */
	job->compact_mode = TRUE;
	break;

      case 'b':				/* sheet header value */
	xustrcpy (job->header, optarg);
	break;

      case 'B':				/* No headers at all */
	/* Free them if they were allocated */
	XFREE (job->header);
	XFREE (job->left_footer);
	XFREE (job->footer);
	XFREE (job->right_footer);
	XFREE (job->left_title);
	XFREE (job->center_title);
	XFREE (job->right_title);
	XFREE (job->water);

	job->header = UNULL;
	job->left_footer = UNULL;
	job->footer = UNULL;
	job->right_footer = UNULL;
	job->left_title = UNULL;
	job->center_title = UNULL;
	job->right_title = UNULL;
	job->water = UNULL;
	break;

      case 'c':				/* cut lines too large */
	job->folding = FALSE;
	break;

      case 151:				/* cut lines too large */
	job->folding = (a2ps_get_boolean ("--truncate-lines", optarg)
			 ? FALSE : TRUE);
	break;

      case 'C':				/* line numbering */
	job->numbering = 5; /* Default is 5 by 5 */
	break;

      case 152:				/* line numbering */
	job->numbering = 
	  a2ps_get_integer_in_range ("--line-numbers", optarg, 0, -2);
	break;

      case 'd':				/* fork a process to print */ 
	a2ps_printers_output_set (job->printers, NULL, true);
	break;

      case 'D': 			/* --setpagedevice		*/
	{
	  char *value;
	  value = strchr (optarg, ':');
	  if (IS_EMPTY (value)) {
	    delpagedevice (job, optarg);
	  } else {
	    *value = NUL;
	    value ++;
	    setpagedevice (job, optarg, value);
	  }
	}
      break;

      case 'f': 
	{
	  char * cp;
	  /* This is for compatibility with the previous scheme */
	  cp = strchr (optarg, '@');
	  if (cp)
	    cp ++;
	  else
	    cp = optarg;
	  /* A font size is given */
	  job->fontsize = get_length ("--font-size", cp, 2.0, 200.0);
	  job->columns_requested = 0;
	  job->lines_requested = 0;	  
	}
      break;

      case 'i':				/* interpret control chars */
	job->interpret = TRUE;
	break;

      case 155:				/* interpret control chars */
	job->interpret = a2ps_get_boolean ("--interpret", optarg);
	break;

      case 'j':				/* surrounding border */
	job->border = TRUE;
	break;

      case 156:				/* surrounding border */
	job->border = a2ps_get_boolean ("--border", optarg);
	break;

      case 'k':				/* page prefeed */
	job->page_prefeed = TRUE;
	break;

      case 'K':				/* no page prefeed */
	job->page_prefeed = FALSE;
	break;

      case 'l':
	/* set columns per line, useful for most cases */
	job->columns_requested = 
	  a2ps_get_integer_in_range ("--chars-per-line", optarg, 1, -2);
	job->lines_requested = 0;
	break;

      case 'L':
	/* set lines per page.  Useful with preformatted files. Scaling is
	 * automatically done when necessary.  */
	job->lines_requested = 
	  a2ps_get_integer_in_range ("--lines-per-page", optarg, 1, -2);
	/* Unset value given to columns-per-page, so that this one
	 * is not hidden */
	job->columns_requested = 0;
	break;

      case 'm':				/* Process file as a man */
	job->lines_requested = MAN_LINES;
	job->columns_requested = 0;
	break;

      case 'M':                 		/* select a medium */
	job->medium = a2ps_get_medium (job, optarg);
	break;

      case 'n':				/* n copies */
	job->copies = 
	  a2ps_get_integer_in_range ("--copies", optarg, 0, -2);
	break;

      case 'o':			/* output goes into a file */
	a2ps_printers_output_set (job->printers, optarg, false);
	break;

      case 'P':					/* fork a process to print */ 
	a2ps_printers_output_set (job->printers, optarg, true);
	break;

      case 'q':			       /* don't say anything but errors */
	message_verbosity = msg_null;
	break;

      case 'r':
	job->orientation = landscape;  		/* landscape format */
	break;

      case 'R':
	job->orientation = portrait;  		/* portrait format */
	break;

      case 's':
	switch (a2ps_get_integer_in_range ("--sides", optarg, 1, 2)) {
	case 1:
	  job->duplex = 0;
	  delpagedevice (job, "Duplex");
	  break;
	  
	case 2:
	  job->duplex = 1;
	  setpagedevice (job, "Duplex", "true");
	  break;
	}
	break;

      case 'S':				/* statusdict definitions */
	{
	  char *value;
	  value = strchr (optarg, ':');
	  if (IS_EMPTY (value)) {
	    delstatusdict (job, optarg);
	  } else {
	    *value = NUL;
	    value ++;
	    if (*value == ':')
	       setstatusdict (job, optarg, value + 1, TRUE);
	     else
	       setstatusdict (job, optarg, value, FALSE);
	  }
	}
	break;

      case 't':				/* Job title		*/
	xustrcpy (job->title, optarg);
	break;

      case 'T':
	job->tabsize = 
	  a2ps_get_integer_in_range ("--tabsize", optarg, 1, -2);
	break;

      case 'u':				/* water mark (under lay) */
	xustrcpy (job->water, optarg);
	break;

      case 'v':					    /* verbosity */
	if (!IS_EMPTY(optarg)) {
	  message_verbosity = 
	    a2ps_get_integer_in_range ("--verbose", optarg, 0, -2);
	} else
	  message_verbosity = msg_report1;
	break;

      case 'X': 			/* change the encoding scheme */
	/* Since there can be -X in the config files, and because
	 * the encoding.map has not been read yet (because to read
	 * encoding.map, one has to know the lib path, and to know
	 * the lib path, one has to read the config files...), we
	 * can't store the requested encoding as an encoding.
	 *
	 * Nevertheless, if encoding.map has been read. which means
	 * that this is actually a real command line option,
	 * do store the correct encoding */
	XFREE (job->requested_encoding_name);
	job->requested_encoding_name = xstrdup (optarg);
	break;

      case 132:				/* Number of columns */
	job->columns = 
	  a2ps_get_integer_in_range ("--columns", optarg, 1, -2);
	break;

      case 133:				/* Number of rows */
	job->rows =
	  a2ps_get_integer_in_range ("--rows", optarg, 1, -2);
	break;

      case 134:				/* --include ps prologue */
	xstrcpy (job->prolog, optarg);
	break;

      case 135:				/* --non-printable-format */
	job->unprintable_format = 
	  get_non_printable_format ("--non-printable", optarg);
	break;

      case 136:				/* --print-anyway=bool */
	job->print_binaries = a2ps_get_boolean ("--print-anyway", optarg);
	break;

      case 137:			/* --macro-meta-sequence=key:value */
	{
	  char *value;
	  value = strchr (optarg, ':');
	  if (value) {
	    *value = NUL;
	    value ++;
	    if (!macro_meta_sequence_add (job, optarg, value))
	      error (1, 0, 
		     _("invalid macro identifier `%s'"), optarg);
	  } else {
	    macro_meta_sequence_delete (job, optarg);
	  } 
	}
	break;

      case 140:
	xustrcpy (job->left_title, optarg);
	break;

      case 141:
	xustrcpy (job->right_title, optarg);
	break;

      case 149:
	xustrcpy (job->center_title, optarg);
	break;

      case 142:
	xustrcpy (job->left_footer, optarg);
	break;

      case 143:
	xustrcpy (job->footer, optarg);
	break;

      case 144:
	xustrcpy (job->right_footer, optarg);
	break;

      case 146:				/* --debug */
	job->debug = TRUE;
	break;

      case 147:				/* --margin */
	if (optarg)
	  job->margin =
	    a2ps_get_integer_in_range ("--margin", optarg, 0, -2);
	else
	  job->margin = 12;
	break;

      case 150:				/* allow two files per sheet */
	job->compact_mode = a2ps_get_boolean ("--compact", optarg);
	break;

      case 157: 			/* --major= */
	job->Major = get_major ("--major", optarg);
	break;

      case 158:				/* --version-control */
	backup_type = a2ps_get_backup_type (optarg);
	break;

      case 159:				/* --suffix	*/
	simple_backup_suffix = xstrdup (optarg);
	break;

      case 163:				/* --ppd[=FILE] */
	a2ps_printers_request_ppdkey_set (job->printers, optarg);
	break;

      case 166:			/* Set the name of file give by stdin */
	xustrcpy (job->stdin_filename, optarg);
	break;

      case '?':				/* Unknown option */
	/* Error message is done by getopt */
	fprintf (stderr,
		 _("Try `%s --help' for more information.\n"), program_name);
	exit (EXIT_BADARG);
	break;
      }
    }
  return optind;
}

int
a2ps_handle_string_options (a2ps_job * job, const char * string)
{
  int argc;
  char **argv;
  char *str;
  int i, res;

  if (string == NULL)
    return 0;

  message (msg_opt, (stderr, "handle_string_options(%s)", string));

  /* Alloca-copy string so we can modify it in place. */
  astrcpy (str, string);

  /*
   * We can count this, each option takes at least 1 character and one 
   * space.  We also need one for program's name and one for the 
   * trailing NULL. 
   */
  argc = (strlen (str) + 1) / 2 + 2;
  argv = (char **) CALLOC (char *, argc);	/* FIXME: alloca? */
  
  /* Set program name. */
  argc = 0;
  argv[argc++] = program_name;

  /* Split string and set arguments to argv array. */
  i = 0;
  while (str[i])
    {
      /* Skip leading whitespace. */
      for (; str[i] && isspace ((int) str[i]); i++)
	;
      if (!str[i])
	break;

      /* Check for quoted arguments. */
      if (str[i] == '"' || str[i] == '\'')
	{
	  int endch = str[i++];

	  argv[argc++] = str + i;

	  /* Skip until we found the end of the quotation. */
	  for (; str[i] && str[i] != endch; i++)
	    ;
	  if (!str[i])
	    error (1, 0, _("syntax error in option string `%s':\n\
missing end of quotation: %c"), string, endch);

	  str[i++] = '\0';
	}
      else
	{
	  argv[argc++] = str + i;

	  /* Skip until whitespace if found. */
	  for (; str[i] && !isspace ((int) str[i]); i++)
	    ;
	  if (str[i])
	    str[i++] = '\0';
	}
    }
  
  /* argv[argc] must be NULL. */
  argv[argc] = NULL;

  message (msg_opt, (stderr, " (argc=%d):\n", argc));
  for (i = 0; i < argc; i++)
    message (msg_opt, (stderr, "   %3d = `%s'\n", i, argv[i]));

  /* Process options. */
  res = a2ps_handle_option (job, argc, argv);

  /* Cleanup. */
  XFREE (argv);

  return res;
}
