/* filename: rlprd.c
 * project: rlpr
 * author: meem  --  meem@sherilyn.wustl.edu
 * version: $Id: rlprd.c,v 1.13 1997/01/16 16:44:11 meem Exp meem $
 * contents: daemon which does "reflecting" of messages from non-privileged
 *           to privileged ports
 *
 * Time-stamp: <1997/04/27 01:21:33 cdt -- meem@sherilyn.wustl.edu>
 */

/* copyright (c) 1996, 1997 meem, meem@gnu.ai.mit.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include "config.h"

#ifndef STDC_HEADERS
#error there is currently no support for compiling on machines without ANSI headers. \
       write meem@gnu.ai.mit.edu if you are interested in adding it.
#endif

#include <sys/types.h>
#include <signal.h>
#include <unistd.h>		       /* for geteuid(), ...       */
#include <stdlib.h>		       /* for EXIT_* constants, .. */
#include <sys/time.h>		       /* for select()             */
#include <netdb.h>		       /* for gethostbyaddr()      */
#include <sys/stat.h>		       /* for umask()              */

#ifdef SYSLOG_USABLE
#include <syslog.h>
#endif /* SYSLOG_USABLE */

#include <stdio.h>

#include "getopt.h"
#include "wait.h"
#include "rlpr-rfc1179.h"
#include "rlprd.h"
#include "rlpr-msg.h"
#include "rlpr-util.h"
       
static void         register_sigchld(void);
static void         register_sigalrm(void);
static RETSIGTYPE   sigchld(int unused);
static RETSIGTYPE   sigalrm(int unused);
static const char * hostname(struct sockaddr_in *sin);
static ssize_t      readstr(int fd, char * buf, size_t bufsz);
static void         converse(int client, int server);

static rlpr_rlprd_info rlpr_rlprd_props;
extern struct option   rlpr_long_opts[]; 
extern const char *    rlpr_opts;

char * program_name;		      /* program name */

int main(int argc, char *argv[]) {
  struct sockaddr_in sin_in;          /* where we will listen */
  struct sockaddr_in sin_out;         /* our projected identity (to lpd) */
  struct sockaddr_in sin_lpd;         /* the connection we will send to lpd */  
  struct sockaddr_in sin_from;        /* the connection that came in */
  int    inc_rlprd, out_lpd;          /* socket descriptors */
  int    listen_fd;                   /* passive descriptor listening connxns */
  int    sin_fromlen = sizeof sin_from; /* the length of the incoming desc */
  char   printhost[MAXHOSTNAMELEN + 1]; /* name of printhost */
  char   localhost[MAXHOSTNAMELEN + 1]; /* for finding out our name */
  int    orig_argc = argc;            /* original # args */

  program_name = argv[0];

  argc -= rlpr_rlprd_args(orig_argc, argv);
  argv += orig_argc - argc;

  if (geteuid() != 0)
    rlpr_msg(FATAL, NO_ERRNO, "must be run root or setuid root!");

  toggle_euid();		       /* lose root */

  if (rlpr_rlprd_props.daemon) 
    daemon_init();		       /* if daemon, do the magic */

  if (get_local_hostname(localhost, sizeof(localhost)) == -1)
    rlpr_msg(FATAL, NO_ERRNO, "unable to resolve your local hostname!");

  if ((listen_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) 
    rlpr_msg(FATAL, ERRNO, "socket");
  
  /* initialize and fill in what we can up front */
  
  init_sockaddr(&sin_in,   NULL,      rlpr_rlprd_props.port); 
  init_sockaddr(&sin_out,  localhost, 0); 
  init_sockaddr(&sin_from, NULL,      0);

  sin_in.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(listen_fd,  (struct sockaddr *) &sin_in,  sizeof(sin_in)) < 0)
    rlpr_msg(FATAL, ERRNO, "bind to port %hi failed", rlpr_rlprd_props.port);
  
  register_sigchld();               /* register the reaper */
  register_sigalrm();               /* register the alarm */

  rlpr_msg(DEBUG, NO_ERRNO, "listening on port %hi...", rlpr_rlprd_props.port);
  listen(listen_fd, 5);             /* don't rely on more than 5 */

  for (;;) {

    inc_rlprd = accept(listen_fd, (struct sockaddr *) &sin_from, &sin_fromlen);
    if (inc_rlprd < 0) {
#ifndef SA_RESTART
      if (errno == EINTR) continue;
#endif  /* NOT SA_RESTART */
      rlpr_msg(WARNING, ERRNO, "accept");
      continue;
    }

    rlpr_msg(DEBUG, NO_ERRNO, "accepted incoming connection");

    switch (fork()) {
    case  0: break;
    case -1: rlpr_msg(FATAL, ERRNO, "fork");
    default: close(inc_rlprd);                           /* parent */
             continue;
    }

    /* CHILD */

    alarm(rlpr_rlprd_props.timeout);   /* set timeout */
    
    if (readstr(inc_rlprd, printhost, MAXHOSTNAMELEN) < 0)
      rlpr_msg(FATAL, NO_ERRNO, "unable to read proxy hostname");     

    rlpr_msg(INFO, NO_ERRNO,
             "proxy from %s to %s", hostname(&sin_from), printhost);

    /* bind our local socket so we come from a privileged port */

    toggle_euid();                            /* gain root */

    if ((out_lpd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
      rlpr_msg(FATAL, ERRNO, "socket");

    if (bind_try_range(&sin_out, LO_LPD_DST_PORT, HI_LPD_DST_PORT, out_lpd) < 0)
      rlpr_msg(FATAL, ERRNO,
               "bind to ports %hi-%hi", LO_LPD_DST_PORT, HI_LPD_DST_PORT);

    toggle_euid();                            /* lose root */
      
    /* initialize sin_lpd and connect to the host running lpd */

    init_sockaddr(&sin_lpd, printhost, LPD_DST_PORT);

    if (connect(out_lpd, (struct sockaddr *) &sin_lpd, sizeof(sin_lpd)) < 0)
      rlpr_msg(FATAL, ERRNO, "connect");

    /* converse */
    converse(inc_rlprd, out_lpd);
  
    exit(EXIT_SUCCESS);
  }

  exit(EXIT_SUCCESS);
}

void converse(int client, int server) {
  int         maxfd = (client > server ? client : server) + 1;
  static char buf[BUFSIZ * 4];        /* could be a more optimal size */  
  int         nc, nfds;               /* data moved in read/write */
  fd_set      readfds;                /* read file descriptor set */

  for (;;) {                  

    FD_ZERO(&readfds);                      /* initialize fds structure */
    FD_SET(client, &readfds);
    FD_SET(server, &readfds);

    if ((nfds = select(maxfd, &readfds, NULL, NULL, NULL)) < 0)
      rlpr_msg(FATAL, ERRNO, "select");
        
    if (nfds)
      if (FD_ISSET(client, &readfds)) { 
        /* data from rlpr client */
        
        if ((nc = read(client, buf, sizeof buf)) < 0)
          rlpr_msg(FATAL, ERRNO, "read from rlpr client");
        else if (!nc) break;
        else writen(server, buf, nc);

      } else {
        /* data from lpd server */

        if ((nc = read(server, buf, sizeof buf)) < 0)
          rlpr_msg(FATAL, ERRNO, "write to lpd server");
        else if (!nc) break;
        else writen(client, buf, nc);
      }
  }
}

/* daemon_init()
 *
 * purpose: initialize the server to run in daemon mode.
 *   input: void
 *  output: void
 */ 

void daemon_init(void) {
  int syslog_on = 1;

  if (getppid() > 1) {  
    /* we weren't started by init.  let's fork ourselves silly */
    
    /* turn on the logger */
#ifdef SYSLOG_USABLE
    openlog(program_name, LOG_PID, LOG_LPR);
#endif /* SYSLOG_USABLE */
    
    rlpr_msg_set(RLPR_MSG_SYSLOG, &syslog_on);

    switch (fork()) {
      case  0: break;
      case -1: rlpr_msg(FATAL, ERRNO, "fork");
      default: exit(EXIT_SUCCESS);
    }

    /* child */

    setsid();                   /* now unique session, PGID, no ctty */

    /* fork again to make sure we don't reacquire a ctty */

    switch (fork()) {
      case  0: break;
      case -1: rlpr_msg(FATAL, ERRNO, "fork");
      default: exit(EXIT_SUCCESS);
    }

    /* second child */
  }

  chdir("/");                   /* in case we were on a mounted partition */
  umask(0);                     /* clear file mode creation mask */
}

static const char * hostname(struct sockaddr_in *sin) {
  struct hostent *hp;
  hp = gethostbyaddr((char *) &sin->sin_addr, sizeof(struct in_addr), AF_INET);
  if (!hp) rlpr_msg(FATAL, NO_ERRNO, "gethostbyaddr failed: %s", h_strerror());
  return hp->h_name;
}
     
static RETSIGTYPE sigchld(int unused) {
  int olderrno = errno;
  while (waitpid( -1, NULL, WNOHANG) > 0) /* NULL BODY */;
  errno = olderrno;
}

static RETSIGTYPE sigalrm(int unused) {
  rlpr_msg(FATAL, NO_ERRNO, "connection timed out");
}

static void register_sigchld(void) {
  struct sigaction sa;
  sa.sa_handler = sigchld;
  sa.sa_flags   = 0;
#ifdef SA_RESTART
  sa.sa_flags  |= SA_RESTART;
#endif /* SA_RESTART */
  sigemptyset(&sa.sa_mask);

  if (sigaction(SIGCHLD, &sa, NULL) == -1)
    rlpr_msg(FATAL, ERRNO, "sigaction");
}

static void register_sigalrm(void) {
  if (signal(SIGALRM, sigalrm) == SIG_ERR)
    rlpr_msg(FATAL, ERRNO, "signal");
}

/* read at most bufsz characters into buf, stopping at the first \n */

static ssize_t readstr(int fd, char * buf, size_t bufsz) {
  size_t i;
  
  for (i = 0; i < bufsz; i++) {
    if (read(fd, &buf[i], 1) < 0) return -1;
    if (buf[i] == '\n') break;
  }
  buf[i] = '\0';
  return i;
}

static void rlpr_rlprd_init(void) {
  rlpr_rlprd_props.port    = RLPRD_DST_PORT;
  rlpr_rlprd_props.timeout = RLPRD_TIMEOUT;
  rlpr_rlprd_props.daemon  = 1;          /* daemon on */
}

int rlpr_rlprd_args(int argc, char *argv[]) {
  int c, num_args = 1;
  extern int optind;

  rlpr_rlprd_init();
  rlpr_msg_args(argc, argv);	/* initialize the messaging subsystem */

  optind = 0;
  while ((c = getopt_long(argc, argv, rlpr_opts, rlpr_long_opts, NULL)) != EOF) {
    num_args++;
    switch(c) {
    case 'n':
      rlpr_rlprd_props.daemon  = 0;
      break;
    case 'p':
      rlpr_rlprd_props.port    = atoi(optarg);
      break;
    case 't':
      rlpr_rlprd_props.timeout = atoi(optarg);
      break;
    case 'V':
      fprintf(stdout, "%s: version "VERSION" from "__DATE__" "__TIME__ 
              " -- meem@gnu.ai.mit.edu\n", program_name);
      exit(EXIT_SUCCESS);
    }
  }

  return num_args;
}

struct option rlpr_long_opts[] = {
  { "no-daemon", 0, NULL, 'n' },
  { "port"    ,  1, NULL, 'p' },
  { "timeout" ,  1, NULL, 't' },
  { "version" ,  0, NULL, 'V' },
  { "debug"   ,  0, NULL, -101 },
  { "quiet"   ,  0, NULL, 'q'  },
  { "silent"  ,  0, NULL, 'q'  },
  { "verbose" ,  0, NULL, -102 },
  { 0, 0, 0, 0 }
};

const char * rlpr_opts = "np:tVq";
