/*
 * mknbi.h  -  mknbi constants and type definitions for DOS loader
 *
 * Copyright (C) 1996-1998 Gero Kuhlmann   <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef _MKNBI_H_DOS_
#define _MKNBI_H_DOS_


/*
 * Include version information
 */
#include "version.h"


/*
 * Everything in this file has to be in packed structures.
 */
#ifdef USE_PRAGMA
#pragma pack(1)
#endif
#ifdef USE_PRAGMA_ALIGN
#pragma options align=packed
#endif



/*
 * The bootrom loads the load header to 0x10000, the primary boot
 * loader to 0x10400, and the ram disk image at the first address
 * above the HMA int extended memory.
 */
#define HEADERSEG	0x1000		/* segment for primary load header */
#define HEADERLSIZE	512L		/* maximum load size of load header */
#define HEADERMSIZE	512L		/* maximum memory size of load header */

#define LOADERSEG	0x1040		/* segment for primary boot loader */
#define LOADERLSIZE	64512L		/* maximum load size of boot loader */
#define LOADERMSIZE	64512L		/* maximum memory size of boot loader */

#define RDADDR		0x110000L	/* start of ext memory at top of HMA */

#define MAX_CYLS	255			/* max number of cylinders */
#define MAX_SECTS	63			/* max number of sectors */
#define MAX_RDSIZE	(MAX_CYLS * MAX_SECTS)	/* max size of ram disk in kB */
#define MIN_RDSIZE	360			/* min size of ram disk in kB */
#define MAX_IOSYS	58			/* max sects for IO.SYS */



/*
 * Defines for handling all the boot sector stuff.
 */
#define SECTSIZE	512			/* Size of one sector */

struct boot_record {
  __u8          jmp_opcode[3]	PACKED;		/* jump opcode */
  __u8          oem_name[8]	PACKED;		/* OEM name */
  __u16         bytes_per_sect	PACKED;		/* bytes per sector */
  __u8          sect_per_cluster PACKED;	/* sectors per cluster */
  __u16         reserved_sect	PACKED;		/* reserved sectors */
  __u8          fat_num		PACKED;		/* number of FATs */
  __u16         dir_num		PACKED;		/* dir entries */
  __u16         sect_num	PACKED;		/* number of sectors */
  __u8          media_id	PACKED;		/* media ID */
  __u16         sect_per_fat	PACKED;		/* sectors per FAT */
  __u16         sect_per_track	PACKED;		/* sectors per track */
  __u16         head_num	PACKED;		/* number of heads */
  struct i_long hidden_num	PACKED;		/* hidden sectors */
  struct i_long sect_num_32	PACKED;		/* sects if part > 32MB */
  __u8          boot_id		PACKED;		/* id of boot disk */
  __u16         dev_attr	PACKED;		/* device attribute */
  struct i_long vol_num		PACKED;		/* volume number */
  __u8          vol_name[11]	PACKED;		/* volume name */
  __u8          fat_name[8]	PACKED;		/* FAT name */
};


/* Private extensions to boot record */
struct priv_boot {
  struct boot_record boot_rec		PACKED;	/* official boot record */
  __u16              io_sys_sects	PACKED;	/* no of sectors in IO.SYS */
};

#define BOOT_FAT12_MAX	0x0fef			/* max clust # for 12 bit FAT */
#define BOOT_FAT12_NAME	"FAT12   "		/* FAT id for 12 bit FATs */
#define BOOT_FAT16_NAME	"FAT16   "		/* FAT id for 16 bit FATs */

#define BOOT_SIGNATURE	0xaa55			/* boot signature */
#define BOOT_PART_OFF	0x01be			/* partition table offset */
#define BOOT_SIG_OFF	0x01fe			/* boot signature offset */

#define BOOT_ACTIVE	0x80			/* marks active partition */
#define BOOT_TYPE_12	0x01			/* partition type (FAT12) */
#define BOOT_TYPE_16	0x04			/* partition type (FAT16) */

#define BOOT_DEV_FD	0x00			/* BIOS id of floppy drive */
#define BOOT_DEV_HD	0x80			/* BIOS id of hard disk drive */
#define BOOT_MEDIA_ID	0xf8			/* media id for hard disk */

struct partition {
  __u8          status		PACKED;		/* status of entry */
  __u8          first_head	PACKED;		/* start head for partition */
  __u8          first_sector	PACKED;		/* start sector of partition */
  __u8          first_track	PACKED;		/* start track of partition */
  __u8          type		PACKED;		/* partition type */
  __u8          last_head	PACKED;		/* end head for partition */
  __u8          last_sector	PACKED;		/* end sector of partition */
  __u8          last_track	PACKED;		/* end track of partition */
  struct i_long first_abs_sector PACKED;	/* first absolute sector num */
  struct i_long number_sectors	PACKED;		/* number of sectors */
};



/*
 * Definitions for DOS directory entries
 */
struct dos_dir {
  __u8          name[8]		PACKED;		/* name of file */
  __u8          ext[3]		PACKED;		/* file extension */
  __u8          attrib		PACKED;		/* file attributes */
  __u8          res[10]		PACKED;		/* reserved */
  __u16         time		PACKED;		/* time of last access */
  __u16         date		PACKED;		/* date of last access */
  __u16         cluster		PACKED;		/* number of start cluster */
  struct i_long size		PACKED;		/* size of file in bytes */
};

#define ATTR_READONLY	0x01		/* read only file */
#define ATTR_HIDDEN	0x02		/* hidden file */
#define ATTR_SYSTEM	0x04		/* system file */
#define ATTR_LABEL	0x08		/* filesystem label */
#define ATTR_DIR	0x10		/* directory */



/*
 * The boot image has the following header in it's first sector
 */
struct load_header {
  struct i_long magic		PACKED;		/* magic number */
  __u8          hlength		PACKED;		/* length of header */
  __u8          hflags1		PACKED;		/* header flags */
  __u8          hflags2		PACKED;
  __u8          hflags3		PACKED;
  struct i_addr locn		PACKED;		/* location of this header */
  struct i_addr execute		PACKED;		/* execution address */
  __u8          dummy[494];			/* up to full sector */
  __u16         bootsig		PACKED;		/* boot signature */
};

#define HEADER_MAGIC	0x1B031336		/* Magic no for load header */



/*
 * In the header sector are the load records located, which actually tell
 * which part of the following image has to go where in memory
 */
struct disk_geometry {
  __u16 num_sectors		PACKED;		/* total # of sectors */
  __u16 sect_per_track		PACKED;		/* # of sectors per track */
  __u16 cylinders		PACKED;		/* # of cylinders */
  __u8  boot_drive		PACKED;		/* ID of boot drive */
  __u8  no_hd_flag		PACKED;		/* non-zero if no hard disk */
};

union vendor_data {
  struct disk_geometry geometry;	/* disk geometry */
};

struct load_record {
  __u8              rlength	PACKED;		/* length of record */
  __u8              rtag1	PACKED;		/* record tags */
  __u8              rtag2	PACKED;
  __u8              rflags	PACKED;		/* record flags */
  struct i_long     address	PACKED;		/* abs addr for part in mem */
  struct i_long     ilength	PACKED;		/* len of part in boot image */
  struct i_long     mlength	PACKED;		/* memory needed for part */
  union vendor_data vendor_data	PACKED;		/* optional vendor data */
};

#define FLAG_B0		1
#define FLAG_B1		2
#define FLAG_EOF	4



/*
 * Number of each load record.
 */
#define LOADERNUM	0		/* primary boot loader */
#define RAMDISKNUM	1		/* ramdisk image */

#define NUM_RECORDS	2

#define VENDOR_ID	"GK-mknbi-DOS"	/* Vendor ID */
#define VENDOR_OFF	16		/* Offset for vendor tags */



/*
 * Turnoff structure packing and return to normal data alignment.
 */
#ifdef USE_PRAGMA
#pragma pack()
#endif
#ifdef USE_PRAGMA_ALIGN
#pragma options align=reset
#endif



/*
 * Global variables
 */
extern char *progname;			/* program name */
extern int verbose;			/* verbose flag */
extern int usehd;			/* use hard disk for ramdisk */
extern int nohd;			/* dont allow hard disk access */
extern int skipchk;			/* skip OEM name check */



/*
 * External routines
 */
extern int openrd __P((char *name, int rdsize));   /* defined in openrd.c */



/*
 * Data area which contains the boot loader module
 */
extern unsigned char first_data[];
extern unsigned int first_data_size;



/*
 * Data area which contains the floppy boot sector
 */
extern unsigned char boot_data[];
extern unsigned int boot_data_size;


#endif
