/*
 * Handle all the more complicated user interactions with an entry list menu.
 * Whenever the user begins editing a row, the entry in that row is copied
 * to the edit struct. All editing takes place, until it is confirmed and
 * put back into the main list. The reason for this is that entering a new
 * date or time doesn't make the row disappear from the current list because
 * it now belongs to a different day before editing is finished.
 *
 *	got_entry_press(w,lw,x,y,on)	The user pressed on one of the
 *					buttons in the list menu button array.
 *	confirm_new_entry()		Puts the currently edited row back
 *					into the mainlist, and re-sorts.
 *	undo_new_entry()		Kill the edit buffer, undoing the
 *					changes to the current row.
 *	got_entry_text(lw, x, y, text)	The user entered text into the text
 *					button. There is never more than one.
 *	sensitize_edit_buttons()	Desensitize all buttons in the
 *					bottom button row that don't make
 *					sense at the moment.
 */

#ifndef MIPS
#include <stdlib.h>
#endif
#include <stdio.h>
#include <time.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Xm/LabelP.h>
#include <Xm/LabelG.h>
#include <Xm/PushBP.h>
#include <Xm/PushBG.h>
#include <Xm/ToggleB.h>
#include <Xm/Protocols.h>
#include "cal.h"


extern void		destroy_user_sel_popup();
extern void		create_user_sel_popup();
static void		user_sel_callback(), abort_callback();

extern Display		*display;	/* everybody uses the same server */
extern Pixel		color[NCOLS];	/* colors: COL_* */
extern struct config	config;		/* global configuration data */
extern struct plist	*mainlist;	/* list of all schedule entries */
extern Widget		mainwindow;	/* popup menus hang off main window */
extern struct user	*user;		/* user list (from file_r.c) */
extern int		curr_month;	/* month being displayed, 0..11 */
extern int		curr_year;	/* year being displayed, since 1900 */

struct edit		edit;		/* info about entry being edited */


/*
 * user selection popup is up, and the user has made a selection. Store the
 * new user number in the edited appointment. The user selection popup had
 * been installed by got_entry_press.
 */

static void got_new_user(newuser)
	int	newuser;
{
	int	olduser;

	olduser = name_to_user(edit.entry.user);
	if (!prepare_for_modify(user[olduser].name) ||
	    !prepare_for_modify(user[newuser].name))
		return;
	if (edit.entry.user)
		free(edit.entry.user);
	edit.entry.user = newuser<0 ? 0 : mystrdup(user[newuser].name);
	edit.changed = TRUE;
	edit.editing = TRUE;
	confirm_new_entry();
	update_all_listmenus();
	sensitize_edit_buttons();
}


/*
 * user on one of the buttons in the appointment entry menu
 */

void got_entry_press(lw, x, y, on)
	struct listmenu		*lw;		/* which menu */
	int			x, y;		/* which column/row */
	int			on;		/* enable radio but: on/off */
{
	register struct entry	*ep;		/* affected entry */
	int			num;		/* number of entries so far */
	time_t			now;		/* current time */

	destroy_user_sel_popup();
	num = lw->sublist ? lw->sublist->nentries : 0;
	if (y-1 > num)					/* illegal entry? */
		return;
	if (y-1 < num && !prepare_for_modify(lw->sublist->entry[y-1]->user)) {
		if (x == SC_ENABLE)
			XtVaSetValues(lw->entry[y][x], XmNset, !on, NULL);
		return;
	}
	if (edit.editing && (edit.y != y || edit.menu != lw))
		confirm_new_entry();
	num = lw->sublist ? lw->sublist->nentries : 0;
	if (!edit.editing) {
		if (y-1 < num) {
			clone_entry(&edit.entry, lw->sublist->entry[y-1]);
			if (!server_entry_op(&edit.entry, 'l')) {
				create_error_popup(lw->shell, 0, "%s\n%s\n",
					"Somebody else is editing this entry.",
					"Please try again later.");
				return;
			}
		} else {
			if (x != SC_DATE && x != SC_TIME) {
				create_error_popup(lw->shell, 0,
					"Please enter time or date first");
				return;
			}
			clone_entry(&edit.entry, (struct entry *)0);
			now = get_time();
			edit.entry.time = lw->time ? lw->time : now - now%60;
		}
		edit.editing	= TRUE;
		edit.changed	= FALSE;
		edit.y		= y;
		edit.menu	= lw;
		sensitize_edit_buttons();
	}
	ep = &edit.entry;
	switch(x) {
	  case SC_ENABLE:
		ep->suspended = !on;
		edit.changed = TRUE;
		sensitize_edit_buttons();
		update_all_listmenus();
		break;

	  case SC_USER:
		create_user_sel_popup(lw->entry[y][x],
				name_to_user(ep->user), got_new_user);
		break;

	  case SC_TIME:
		if (lw->time && !lw->period &&
				(!lw->sublist || y > lw->sublist->nentries))
			print_button(lw->entry[y][SC_DATE],
				mkdatestring(lw->time ? lw->time : now));
		edit_list_button(TRUE, lw, x, y);
		break;

	  case SC_LENGTH:
		if (ep->notime)
			break;
	  case SC_DATE:
	  case SC_NOTE:
		edit_list_button(TRUE, lw, x, y);
		break;

	  case SC_ADVANCE:
		if (ep->notime)
			break;
		if (config.bigwarning)
			edit_list_button(TRUE, lw, x, y);
		else
			create_advance_popup();
		break;

	  case SC_RECYCLE:
		create_recycle_popup();
		break;

	  case SC_MESSAGE:
		create_text_popup("Message", &edit.entry.message, x);
		break;

	  case SC_SCRIPT:
		if (ep->notime)
			break;
		create_text_popup("Shell Script", &edit.entry.script, x);
		break;

	  case SC_EXCEPT:
		create_except_popup();
		break;

	  case SC_PRIVATE:
		ep->private ^= TRUE;
		got_new_user(0);
		break;
	}
}


/*
 * confirm new entry, put it in the list. If the user has changed, mark
 * the old and the new user modified (if non-null), or reclassify as own
 * if old or new user is read-only. If it ends up with a different user,
 * tell the old server to delete and the new server to add.
 */

void confirm_new_entry()
{
	BOOL		changed;
	unsigned int	new_file;
	struct tm	new;
	struct tm	old;

	destroy_user_sel_popup();
	edit_list_button(FALSE, edit.menu, 0, 0);
	changed = edit.editing && edit.changed;
	destroy_text_popup();
	destroy_recycle_popup();
	destroy_advance_popup();
	destroy_except_popup();
	new = *time_to_tm(edit.entry.time);
	old = new;
	if (changed) {
		struct entry *ep = 0;
		mainlist->locked++;
		if (edit.menu->sublist &&
		    edit.y-1 < edit.menu->sublist->nentries)
			ep = edit.menu->sublist->entry[edit.y-1];
		if (!edit.entry.user && user[0].name)
			edit.entry.user = mystrdup(user[0].name);
		new_file = user[name_to_user(edit.entry.user)].file_id;
		if (edit.entry.private && ep && !ep->private) {	/* private */
			server_entry_op(&edit.entry, 'u');
			server_entry_op(&edit.entry, 'd');
			edit.entry.file = 0;
			edit.entry.id   = 0;

		} else if (!edit.entry.id) {			/* new */
			edit.entry.file = new_file;
			server_entry_op(&edit.entry, 'w');

		} else if (new_file == edit.entry.file) {	/* same file */
			server_entry_op(&edit.entry, 'w');
			server_entry_op(&edit.entry, 'u');
		} else {					/* diff file */
			server_entry_op(&edit.entry, 'u');
			server_entry_op(&edit.entry, 'd');
			edit.entry.file = new_file;
			edit.entry.id   = 0;
			server_entry_op(&edit.entry, 'w');
		}
		if (ep) {
			old = *time_to_tm(ep->time);
			delete_entry(mainlist, ep - mainlist->entry);
		}
		(void)add_entry(&mainlist, &edit.entry);
		rebuild_repeat_chain(mainlist);
		mainlist->locked--;
	}
	destroy_entry(&edit.entry);
	edit.editing = FALSE;
	edit.changed = FALSE;
	sensitize_edit_buttons();
	if (changed) {
		update_all_listmenus();
		redraw_all_views();
		mainlist->modified = TRUE;
	}
}


/*
 * undo all editing, restore the entry before we started editing
 */

void undo_new_entry()
{
	if (edit.editing) {
		server_entry_op(&edit.entry, 'u');
		destroy_text_popup();
		destroy_recycle_popup();
		destroy_advance_popup();
		destroy_entry(&edit.entry);
		edit.editing = FALSE;
		edit.changed = FALSE;
		sensitize_edit_buttons();
		draw_list(edit.menu);
	}
}


/*
 * user entered text into one of the buttons in the entry list
 */

void got_entry_text(lw, x, y, text)
	struct listmenu		*lw;		/* which menu */
	int			x, y;		/* which column/row */
	char			*text;		/* text input by user */
{
	register struct entry	*ep;		/* affected entry */
	register char		*p;		/* to remove trailing \n's */

	while (*text == ' ' || *text == '\t')
		text++;
	ep = &edit.entry;
	switch(x) {
	  case SC_DATE:
		if (*text) {
			time_t temp = parse_datestring(text, 0);
			if (temp > EON)
				ep->time = ep->time % 86400 + temp;
		}
		edit_list_button(TRUE, lw, SC_TIME, y);
		print_button(lw->entry[y][SC_DATE],
					mkbuttonstring(lw, SC_DATE, y));
		break;

	  case SC_TIME:
		if (*text == '-') {
			ep->notime = TRUE;
			ep->time -= ep->time % 86400;
		} else if (*text) {
			ep->notime = FALSE;
			ep->time -= ep->time % 86400;
			ep->time += parse_timestring(text, FALSE);
		}
		edit_list_button(TRUE,lw, ep->notime ? SC_NOTE : SC_LENGTH, y);
		print_button(lw->entry[y][SC_TIME],
					mkbuttonstring(lw, SC_TIME, y));
		break;

	  case SC_LENGTH: {
		time_t length;
		length = *text==0   ? 0 :
			 *text=='-' ? parse_timestring(text+1, FALSE) -
								 ep->time%86400
				    : parse_timestring(text,   FALSE);
		if (length < 0)
			length = 0;
		if (length > 86400)
			length = 86400;
		ep->length = length;
		edit_list_button(TRUE, lw, config.bigwarning && !ep->notime ?
					SC_ADVANCE : SC_NOTE, y);
		print_button(lw->entry[y][SC_LENGTH],
					mkbuttonstring(lw, SC_LENGTH, y));
		break; }

	  case SC_ADVANCE:
		parse_warnstring(text, &ep->early_warn, &ep->late_warn,
							&ep->noalarm);
		edit_list_button(TRUE, lw, SC_NOTE, y);
		print_button(lw->entry[y][SC_ADVANCE],
					mkbuttonstring(lw, SC_ADVANCE, y));
		break;

	  case SC_NOTE:
		if (ep->note)
			free(ep->note);
		ep->note = 0;
		for (p=text+strlen(text)-1; p != text &&
			(*p == '\n' || *p == '\t' || *p == ' '); *p-- = 0);
		for (p=text; *p; p++)
			if (*p == '\n') *p = ' ';
		if (*text)
			ep->note = mystrdup(text);
		edit_list_button(FALSE, lw, 0, 0);
		print_button(lw->entry[y][SC_NOTE], "%s",
					mkbuttonstring(lw, SC_NOTE, y));
		XmProcessTraversal(lw->done, XmTRAVERSE_CURRENT);
	}
	if (x == SC_DATE || x == SC_TIME)
		if (ep->rep_last && ep->rep_last < ep->time) {
			create_error_popup(lw->shell, 0,
				"The new date is after the last-trigger date,\n"
				"%s. The last-trigger date has been\n"
				"disabled. Press the cycle button to re-enter.",
						mkdatestring(ep->rep_last));
			ep->rep_every = 0;
			ep->rep_last  = 0;
		}
	edit.changed = TRUE;
	edit.entry.suspended = FALSE;
#if 1						/* 1=Return in note confirms */
	if (x == SC_NOTE) /* eoln */
		confirm_new_entry();
#endif
	sensitize_edit_buttons();
}


/*
 * while a new entry is being edited, it is kept in the <edit> struct,
 * which overrides the appropriate entry (if any) in the mainlist. If
 * it is being edited, the Confirm button should be sensitive. If it is
 * being edited and has been changed, the Undo button should be sensitive.
 */

void sensitize_edit_buttons()
{
	Arg			args;

	if (edit.menu) {
		XtSetArg(args, XmNsensitive, edit.changed);
		XtSetValues(edit.menu->confirm, &args, 1);
		XtSetArg(args, XmNsensitive, edit.changed);
		XtSetValues(edit.menu->undo,    &args, 1);
		XtSetArg(args, XmNsensitive, edit.editing);
		XtSetValues(edit.menu->dup,     &args, 1);
		XtSetArg(args, XmNsensitive, edit.editing);
		XtSetValues(edit.menu->del,     &args, 1);
	}
}
