/*
   Siag, Scheme In A Grid
   Copyright (C) 1996-1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *      cmds.c
 *
 *      This rather bulky module contains all the functions that implement
 *      commands.  It also handles initialization of the interface to those
 *      functions.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/stat.h>

#include "../siod/siod.h"
#include "../common/cmalloc.h"
#include "../common/fonts.h"
#include "calc.h"
#include "types.h"

extern int input_warp_pointer;	/* from input.c */

typedef struct {
	char *name;
	 LISP(*function) ();
} s_fn_table;

/* moving around */

int recalc = 1;

static LISP auto_recalc(LISP n)
{
	if (NFLONUMP(n))
		err("wta(1st) to auto-recalc", n);
	recalc = FLONM(n);
	return NIL;
}


static LISP spawn(LISP command)
{
	char *argv[20], *p, cmd[1024];
	int argc = 0;

	strncpy(cmd, get_c_string(command), 1000);
	for (p = strtok(cmd, " \t\r\n");
	     p && argc < 20;
	     p = strtok(NULL, " \t\r\n")) {
		argv[argc++] = p;
	}
	argv[argc] = NULL;
	if (!fork()) {
		/* this is the child */
		execvp(argv[0], argv);
		exit(0);
	}
	return NIL;
}

/* Windows and buffers */

/*X
   static void load_buffer()

   Load a buffer from file.
   X */
static LISP load_buffer()
{
	static char path[1024], name[1024];
	char fn[1024];
	char fmt[80];
	buffer *b;
	static int need_init = 1;
	char *startup;

TRACEME((f,"load_buffer()"))

	if (need_init) {
		getcwd(path, 1024);
		need_init = 0;
	}
	strncpy(name, "", 1020);
	fn[0] = '\0';
	if (select_file(path, name, patterns, fmt)) {
		sprintf(fn, "%s/%s", path, name);

		b = new_buffer(buffer_name(fn), fn);
		llpr("Loading");

TRACEME((f,"loading '%s' using format %s",fn,fmt))

		if (loadmatrix(fn, b, fmt))
			llpr("New file");
	/* this will fail if load returns 1 because the string pool is full */

	/* execute startup code, if any */
		if ((startup = get_property(b, "startup"))) {
			FILE *fp;
			char fn[256];
			char cmd[256];
			char *buttons[] = {"Yes", "No"};
			int n = alert_box("Load auto startup code?", buttons, 2);
			if (n != 1) {
				sprintf(fn, "/tmp/siag%d", getpid());
				sprintf(cmd, "(load \"%s\")", fn);
				fp = fopen(fn, "w");
				fwrite(startup, strlen(startup), 1, fp);
				fputc('\n', fp);
				fclose(fp);
				execute(cmd);
				remove(fn);
			}
		}

		calc_matrix(b);
		b->change = FALSE;
		set_window_buffer(w_list, b);
		pr_scr_flag = TRUE;
	}
	activate_window(w_list);
	return NIL;
}

/* Load using an external program

   1. Ask for the program to use
   2. Run the program and save output to file in /tmp
   3. Load the file using NULL format (i.e. ask for type)
*/
static LISP load_external()
{
	static int loaders = 0;
	static char *loadname[20], *loadprog[20];
	static int need_init = 1;
	char program[256], param[256], fn[80], cmd[256];
	buffer *b;
	int i;

	if (need_init) {
		FILE *fp;
		char fnl[1024];
		char *p, *q, b[256];

		sprintf(fnl, "%s/siag/external.load", siaghome);
		if ((fp = fopen(fnl, "r")) == NULL) {
			llpr("Can't open loader file");
			return NIL;
		}
		while (fgets(b, 250, fp) != NULL && loaders < 20) {
			if ((p = strtok(b, ":")) && (q = strtok(NULL, "\n"))) {
				loadname[loaders] = cstrdup(p);
				loadprog[loaders] = cstrdup(q);
				loaders++;
			}
		}
		fclose(fp);
		need_init = 0;
	}
	program[0] = param[0] = '\0';
	i = select_from_list("External Program:", loadname, loaders);

	if (i >= 0 && ask_for_str("Parameters:", param)) {
		sprintf(fn, "/tmp/siag%d", getpid());
		sprintf(cmd, "%s %s > %s", loadprog[i], param, fn);
		if (system(cmd)) {
			llpr("External program failed");
			return NIL;
		}

		b = new_buffer(buffer_name(fn), fn);
		llpr("Loading");

		if (loadmatrix(fn, b, NULL))
			llpr("New file");
	/* this will fail if load returns 1 because the string pool is full */

		calc_matrix(b);
		b->change = FALSE;
		set_window_buffer(w_list, b);
		pr_scr_flag = TRUE;
	}
	activate_window(w_list);
	return NIL;
}

static LISP
delete_window()
{
	if (!remove_window(w_list))
		llpr("Attempt to delete sole ordinary window");
	else
		pr_scr_flag = TRUE;
	activate_window(w_list);
	return NIL;
}

static LISP
delete_other_windows()
{
	while (remove_window(next_window(w_list)));
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP
split_window_vertically()
{
	if (!split_window(w_list))
		llpr("This window is too small to split");
	else
		pr_scr_flag = TRUE;
	return NIL;
}

static LISP
other_window()
{
	activate_window(next_window(w_list));
	pr_scr_flag = TRUE;
	return NIL;
}

/*X
   static void save_buffer()

   Save the buffer in the currently active window to file.
   X */
static LISP save_buffer()
{
	buffer *buf = buffer_of_window(w_list);
	llpr("Saving");

	if (savematrix(buf->path,
			buf,
			guess_file_format(buf->path)))
		error_box("Couldn't save");
	else {
		buf->change = FALSE;
		llpr("File saved");
	}
	return NIL;
}

static LISP lsavematrix(LISP path, LISP bname, LISP format)
{
	char *p, *fmt;
	buffer *b;

TRACEME((stderr, "lsavematrix()\n"));

	if (NULLP(bname)) b = buffer_of_window(w_list);
	else b = find_buffer_by_name(get_c_string(bname));
	if (!b) {
		llpr("No such buffer");
		return NIL;
	}

	p = get_c_string(path);
	if (NULLP(format)) fmt = guess_file_format(p);
	else fmt = get_c_string(format);

	if (savematrix(p, b, fmt)) {
		llpr("File saved");
		return NIL;
	} else {
		return a_true_value();
	}
}

static int confirm_overwrite = 0;

static LISP lconfirm_overwrite(LISP x)
{
	confirm_overwrite = get_c_long(x);
	return NIL;
}

/*X
   static void save_buffer_as()

   Save the buffer in the currently active window to a named file.
   X */
static LISP save_buffer_as()
{
	static char path[1024], name[1024];
	char fn[1024];
	char fmt[80];
	char *p;
	static int need_init = 1;
	struct stat sb;

	if (need_init) {
		getcwd(path, 1024);
		need_init = 0;
	}
	p = strrchr(buffer_of_window(w_list)->path, '/');
	if (p) strncpy(name, p+1, 1020);
	else strncpy(name, buffer_of_window(w_list)->path, 1020);
	fn[0] = '\0';
	if (select_file(path, name, patterns, fmt)) {
		sprintf(fn, "%s/%s", path, name);
		if (confirm_overwrite && !stat(fn, &sb)) {
			/* file exists */
			char q[1024];
			char *btn[] = {"Yes", "No"};
			int n;
			sprintf(q, "Overwrite existing %s?", fn);
			n = alert_box(q, btn, 2);
			if (n == 1) return NIL;
		}
		llpr("Saving");
		if (savematrix(fn, buffer_of_window(w_list), fmt))
			error_box("Couldn't save");
		else {
			buffer_of_window(w_list)->change = FALSE;
			llpr("File saved");
			strncpy(buffer_of_window(w_list)->path, fn, 1020);
		}
	/*	pr_scr_flag = TRUE;*/
	}
	return NIL;
}

/* Save using an external program

   1. Ask for the program to use
   2. Save to a file in /tmp using NULL format (i.e. ask for type)
   3. Run the program and read the file as input
*/
static LISP save_external()
{
	static int savers = 0;
	static char *savename[20], *saveprog[20];
	static int need_init = 1;
	char program[256], param[256], fn[80], cmd[256];
	int i;

	if (need_init) {
		FILE *fp;
		char fnl[1024];
		char *p, *q, b[256];

		sprintf(fnl, "%s/siag/external.save", siaghome);
		if ((fp = fopen(fnl, "r")) == NULL) {
			error_box("Can't open saver file");
			return NIL;
		}
		while (fgets(b, 250, fp) != NULL && savers < 20) {
			if ((p = strtok(b, ":")) && (q = strtok(NULL, "\n"))) {
				savename[savers] = cstrdup(p);
				saveprog[savers] = cstrdup(q);
				savers++;
			}
		}
		fclose(fp);
		need_init = 0;
	}

	program[0] = param[0] = '\0';
	i = select_from_list("External Program:", savename, savers);

	if (i >= 0 && ask_for_str("Parameters:", param)) {
		llpr("Saving");

		sprintf(fn, "/tmp/siag%d", getpid());
		if (savematrix(fn, buffer_of_window(w_list), NULL)) {
			error_box("Couldn't save");
			return NIL;
		}
		sprintf(cmd, "%s %s < %s", program, param, fn);
		if (system(cmd)) {
			error_box("External program failed");
			return NIL;
		}
	}
	llpr("File saved");
	return NIL;
}

#if USE_COMPLETION
/*X
   static void complete_name(char *name)
   This function takes a partially completed buffer name
   and returns the first buffer name that matches it.
   X */
static int complete_name(char *name)
{
	buffer *b;
	int len;

	b = w_list->buf;	/* start with the next buffer */
	do {
		b = b->next;
		if ((len = strlen(name)) == 0 
				|| !strncmp(b->name, name, len)) {
			strncpy(name, b->name, 1020);
			return FALSE;
		}
	} while (b != w_list->buf);
	return FALSE;
}
#endif

static LISP switch_to_buffer()
{
	buffer *b;
	char *blist[100];
	int nblist = 0, n;

	b = buffer_of_window(w_list);
	do {
		b = b->next;
		blist[nblist++] = b->name;
	} while (b != buffer_of_window(w_list));
	if ((n = select_from_list("Change Buffer:", blist, nblist)) >= 0)
		set_window_buffer(w_list, find_buffer_by_name(blist[n]));
	pr_scr_flag = TRUE;
	activate_window(w_list);
	return NIL;
}

static LISP kill_buffer()
{
	buffer *b, *next_b;
	window * w;
	char *blist[100];
	int nblist = 0, n;

	b = buffer_of_window(w_list);
	do {
		b = b->next;
		blist[nblist++] = b->name;
	} while (b != buffer_of_window(w_list));
	if ((n = select_from_list("Kill Buffer:", blist, nblist)) >= 0) {
		if ((b = find_buffer_by_name(blist[n])) != NULL) {
			if (b != b->next) {
				next_b = free_buffer(b);
				w = w_list;
				do {
					if (buffer_of_window(w) == b)
						set_window_buffer(w, next_b);
					w = next_window(w);
				} while (w != NULL && w != w_list);
				pr_scr_flag = TRUE;
			}
			else llpr("Couldn't kill last buffer");
		}
	}
	activate_window(w_list);
	return NIL;
}

static void
copy_from_start(buffer *b, position blku, position blkl, int nr, int nk,
								int smart)
{
	int i, j;
	char *old, *new;

	for (i = 0; i < nr; i++) {
		for (j = 0; j < nk; j++) {
			if (inblock(w_list, get_point(w_list).row+i,
					get_point(w_list).col+j) &&
				ins_data(b,
					ret_interpreter(b, blku.row+i,
						blku.col+j),
				     ret_text(b, blku.row+i, blku.col+j),
				     ret_val(b, blku.row+i, blku.col+j),
				     ret_type(b, blku.row+i, blku.col+j),
				     get_point(w_list).row+i,
				     get_point(w_list).col+j)) {
				buffer_of_window(w_list)->change = TRUE;
				if(smart && (ret_type(b, get_point(w_list).row+i,
															get_point(w_list).col+j)==EXPRESSION)) {
					cval value;
					old = ret_text(b, get_point(w_list).row+i, get_point(w_list).col+j);
					new = update_references(ret_interpreter(b, get_point(w_list).row+i,
																									get_point(w_list).col+j),
																	old, 1, 1, BUFFER_ROWS, BUFFER_COLS,
																	get_point(w_list).row - blku.row,
																	get_point(w_list).col - blku.col);
																	if (old != new) {
																		value = ret_val(b, get_point(w_list).row+i,
																										get_point(w_list).col+j);
						 ins_data(b, ret_interpreter(b, get_point(w_list).row+i,
						 get_point(w_list).col+j), new, value,
						 ret_type(b, get_point(w_list).row+i,get_point(w_list).col+j),
						 get_point(w_list).row+i, get_point(w_list).col+j);
																	}
				}

			}
		}
	}
}

/* Name changed from copy_block to fill_block */
/* Also only fills within block */
static LISP fill_block()
{
	int nr, nk;
	position blku, blkl;

	blku = block_upper(w_list);
	blkl = block_lower(w_list);
	nr = blkl.row - blku.row + 1;
	nk = blkl.col - blku.col + 1;

	copy_from_start(buffer_of_window(w_list), blku, blkl, nr, nk, 0);
	calc_matrix(buffer_of_window(w_list));
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP smart_fill_block()
{
       int nr, nk;
       position blku, blkl;

       blku = block_upper(w_list);
       blkl = block_lower(w_list);
       nr = blkl.row - blku.row + 1;
       nk = blkl.col - blku.col + 1;

       copy_from_start(buffer_of_window(w_list), blku, blkl, nr, nk,1);
       calc_matrix(buffer_of_window(w_list));
       pr_scr_flag = TRUE;
       return NIL;
}

static LISP
block_borders(LISP style)
{
	int sty, r, c, fmt;

	if (NFLONUMP(style))
		err("wta(1st) to block-borders", style);
	sty = FLONM(style);
	if  (block_upper(w_list).row < 1 || block_upper(w_list).col < 1)
		return NIL;

	for (r = block_upper(w_list).row; r <= block_lower(w_list).row; r++)
		for (c = block_upper(w_list).col; c <= block_lower(w_list).col; c++) {
			fmt = ret_format(buffer_of_window(w_list), r, c);
			switch (sty) {
			case 1:		/* borders */
				if (r == block_upper(w_list).row)
					fmt |= BORDER_TOP;
				if (r == block_lower(w_list).row)
					fmt |= BORDER_BOTTOM;
				if (c == block_upper(w_list).col)
					fmt |= BORDER_LEFT;
				if (c == block_lower(w_list).col)
					fmt |= BORDER_RIGHT;
				break;
			case 2:		/* grid */
				fmt |= BORDER_MASK;
				break;
			case 3:		/* underline */
				if (r == block_lower(w_list).row)
					fmt |= BORDER_BOTTOM;
				break;
			default:	/* none */
				fmt &= ~BORDER_MASK;
			}
			ins_format(buffer_of_window(w_list), r, c, fmt);
		}
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP execute_extended_command()
{
	char b[256];

	b[0] = '\0';
	if (ask_for_str("Command: ", b))
		execute(b);
	return NIL;
}

static LISP execute_interpreter_command(LISP intpr)
{
	char prompt[80];
	char b[256];
	char *intname = get_c_string(intpr);
	int intp = name2interpreter(intname);
	if (intp < 0) return NIL;
	sprintf(prompt, "%s command: ", intname);
	b[0] = '\0';
	if (ask_for_str(prompt, b))
		exec_expr(intp, b);
	return NIL;
}

static char *quit_buttons[] = {"Yes", "No", "Cancel"};

static LISP lalertbox(LISP prompt, LISP buttons)
{
	char *btext[10], *p = get_c_string(prompt);
	int bno = 0, n;

	while (bno < 10 && NNULLP(buttons)) {
		char *c = get_c_string(car(buttons));
		if (c) btext[bno] = cstrdup(c);
		else btext[bno] = cstrdup("button");
		buttons = cdr(buttons);
		bno++;
	}
	if (p == NULL) p = "prompt";
	n = alert_box(p, btext, bno);
	while (bno) cfree(btext[--bno]);
	return flocons(n);
}

static LISP
quit_siag()
{
	char prompt[256];
	buffer *b = b_list;
	do {
		if (b->change) {
			sprintf(prompt, "Save %s?", b->name);
			switch (alert_box(prompt, quit_buttons, 3)) {
			case 0:
				savematrix(b->path,
						b,
						guess_file_format(b->path));
				break;
			case 2:
				return NIL;
			default:
				break;
			}
		}
		b = b->next;
	} while (b != b_list);
	exit(0);
	return NIL;
}

static LISP go_to()
{
	char b[256];
	int tr = 0, tc = 0;

	b[0] = '\0';
	if (ask_for_str("Go to: ", b))
		sscanf(b, "%d %d", &tr, &tc);

	if ((tr > 0) && (tr <= BUFFER_ROWS))
		set_point_row(w_list, tr);
	if ((tc > 0) && (tc <= BUFFER_COLS))
		set_point_col(w_list, tc);
	return NIL;
}

static LISP set_cell_width()
{
	char b[256];
	int width = 0;

	b[0] = 0;
	if (ask_for_str("Width: ", b))
		sscanf(b, "%d", &width);

	if (width > 5 && width < 500) {
		set_width(buffer_of_window(w_list), get_point(w_list).col, width);
		pr_scr_flag = 1;
	}
	return NIL;
}

static LISP get_cell_width(LISP col)
{
	int c = FLONM(col);
	return flocons(cell_width(buffer_of_window(w_list), c));
}


static LISP set_cell_height()
{
	char b[256];
	int height = 0;

	b[0] = 0;
	if (ask_for_str("Height: ", b))
		sscanf(b, "%d", &height);

	if (height > 5 && height < 500) {
		set_height(buffer_of_window(w_list), get_point(w_list).row, height);
		pr_scr_flag = 1;
	}
	return NIL;
}

static LISP get_cell_height(LISP row)
{
	int r = FLONM(row);
	return flocons(cell_height(buffer_of_window(w_list), r));
}


static LISP set_block_format()
{
	int format = 0, mask = 0;

	format = ret_format(buffer_of_window(w_list),
		block_upper(w_list).row, block_upper(w_list).col);
	if (font_input(&format, &mask)) {
		int r, c, r2, c2, oldfmt;
		buffer *b = buffer_of_window(w_list);

		r2 = b->alloc_lines;
		for (r = block_upper(w_list).row; r <= block_lower(w_list).row; r++) {
			if ((format & BORDER_MASK) == 0 && r > r2)
				break;
			c2 = b->alloc_cols[r];
			for (c = block_upper(w_list).col; c <= block_lower(w_list).col; c++) {
				if ((format & BORDER_MASK) == 0 && c > c2)
					break;
				oldfmt = FMT_MASK &
					ret_format(buffer_of_window(w_list), r, c);
				ins_format(buffer_of_window(w_list), r, c, oldfmt | format);
			}
		}
		pr_scr_flag = 1;
	}
	return NIL;
}

static LISP set_cell_format()
{
	int format = 0, mask = 0;

	format = ret_format(buffer_of_window(w_list), get_point(w_list).row,
			    get_point(w_list).col);
	if (font_input(&format, &mask)) {
		int oldfmt = FMT_MASK & ret_format(buffer_of_window(w_list),
			get_point(w_list).row, get_point(w_list).col);
		ins_format(buffer_of_window(w_list), get_point(w_list).row,
			get_point(w_list).col, oldfmt | format);
		pr_scr_flag = 1;
	}
	return NIL;
}

static char *stylelist[16] = {
	"Default", "Invisible", "Integer", "Scientific", "Fixed",
	"Date", "Time", "Comma", "Percent", "Hex", "Currency",
	"User 1", "User 2", "User 3", "User 4", "User 5"};

static int fmtstyles[16] = {
	FMT_DEFAULT, FMT_INVISIBLE, FMT_INTEGER, FMT_SCIENTIFIC, FMT_FIXED,
	FMT_DATE, FMT_TIME, FMT_COMMA, FMT_PERCENT, FMT_HEX, FMT_CURRENCY,
	FMT_USER1, FMT_USER2, FMT_USER3, FMT_USER4, FMT_USER5};

static LISP set_block_style()
{
	int i, style, oldfmt;
	int r, c, r2, c2;
	buffer *b = buffer_of_window(w_list);

	i = select_from_list("Expression style:", stylelist, 16);
	if (i < 0) return NIL;

	style = fmtstyles[i];

	r2 = b->alloc_lines;
	for (r = block_upper(w_list).row; r <= block_lower(w_list).row; r++)
		if (r > r2) break;
		c2 = b->alloc_cols[r];
		for (c = block_upper(w_list).col;
			c <= block_lower(w_list).col; c++) {
			if (c > c2) break;
			oldfmt = ~FMT_MASK & ret_format(buffer_of_window(w_list), r, c);
			ins_format(buffer_of_window(w_list), r, c, oldfmt | style);
		}
	pr_scr_flag = 1;
	return NIL;
}

static LISP set_cell_style()
{
	int i, style, oldfmt;

	i = select_from_list("Expression style:", stylelist, 16);
	if (i < 0) return NIL;

	style = fmtstyles[i];

	oldfmt = ~FMT_MASK & ret_format(buffer_of_window(w_list),
			get_point(w_list).row, get_point(w_list).col);
	ins_format(buffer_of_window(w_list),
		get_point(w_list).row, get_point(w_list).col,
		oldfmt | style);
	pr_scr_flag = 1;
	return NIL;
}

static LISP define_style()
{
	int i, style;
	char *p;

	i = select_from_list("Expression style:", stylelist, 16);
	if (i < 0) return NIL;

	style = fmtstyles[i];
	p = fmt_get(NULL, style);
	if (p == NULL) {
		llpr("Can't change style");
		return NIL;
	}

	if (ask_for_str(stylelist[i], p))
		fmt_put(p, style);

	pr_scr_flag = 1;
	return NIL;
}

static LISP print_version()
{
	llpr(VERSION);
	return NIL;
}

static LISP buffer_changed(LISP bname)
{
	buffer *buf;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));

	if (buf) buf->change = TRUE;
	return NIL;
}

/* Returns value if successful, otherwise NIL */
static LISP set_data(LISP bname, LISP text, LISP value, LISP type, LISP pos)
{
	buffer *buf;
	char *tx;
	cval val;
	short t;
	char *i;
	int r, c;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;

	tx = get_c_string(text);
	if ((i = tx) == NULL) return NIL;
	t = get_c_long(type);
	switch (t) {
	case EXPRESSION:
		val.number = get_c_double(value);
		break;
	case STRING:
		val.text = i;
		break;
	default:
		val.number = 0;
	}
	r = POSITION_ROW(pos);
	c = POSITION_COL(pos);

	if (ins_data(buf, siod_interpreter, i, val, t, r, c)) {
		buf->change = TRUE;
		return value;
	}
	return NIL;
}

static LISP set_format(LISP bname, LISP pos, LISP format)
{
	buffer *buf;
	int fmt, r, c;
	double retval;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;

	r = POSITION_ROW(pos);
	c = POSITION_COL(pos);
	fmt = get_c_long(format);

	buf->change = TRUE;
	retval = ins_format(buf, r, c, fmt);
	return flocons(retval);
}

static LISP get_format(LISP bname, LISP pos)
{
	buffer *buf;
	int r, c;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;

	r = POSITION_ROW(pos);
	c = POSITION_COL(pos);
	return flocons(ret_format(buf, r, c));
}


static LISP ldownshift_matrix(LISP bname, LISP row)
{
	buffer *buf;
	int r;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;
	r = get_c_long(row);

	downshift_matrix(buf, r);
	return NIL;
}

static LISP lupshift_matrix(LISP bname, LISP row)
{
	buffer *buf;
	int r;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;
	r = get_c_long(row);

	upshift_matrix(buf, r);
	return NIL;
}

static LISP lrightshift_matrix(LISP bname, LISP col)
{
	buffer *buf;
	int c;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;
	c = get_c_long(col);

	rightshift_matrix(buf, c);
	return NIL;
}

static LISP lleftshift_matrix(LISP bname, LISP col)
{
	buffer *buf;
	int c;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;
	c = get_c_long(col);

	leftshift_matrix(buf, c);
	return NIL;
}

static LISP lcalc_matrix(LISP bname)
{
	buffer *buf;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;

	buf->recalc = 1;	/* mark for recalculation */
	return NIL;
}

static LISP lask_for_str(LISP prompt, LISP buf)
{
	char *p, b[256];

	strncpy(b, get_c_string(buf), 255);
	p = get_c_string(prompt);
	if (ask_for_str(p, b))
		return strcons(strlen(b), b);
	else
		return NIL;
}

extern char *psformat;	/* in fileio_ps.c */

static LISP lpsformat()
{
	return strcons(strlen(psformat), psformat);
}

extern int grid_lines;	/* window.c */

static LISP lgrid_lines(LISP n)
{
	grid_lines = get_c_long(n);
	pr_scr_flag = TRUE;
	return NIL;
}

static LISP lget_color(LISP bname, LISP pos)
{
	buffer *buf;
	int row, col;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;

	row = POSITION_ROW(pos);
	col = POSITION_COL(pos);

	return flocons(ret_color(buf, row, col));
}

static LISP lset_color(LISP bname, LISP pos, LISP color)
{
	buffer *buf;
	int row, col, c;

	if (NULLP(bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));
	if (!buf) return NIL;

	row = POSITION_ROW(pos);
	col = POSITION_COL(pos);

	c = get_c_long(color);

	set_color(buf, row, col, c);
	return NIL;
}

static LISP lexecute(LISP cmd)
{
TRACEME((stderr, "lexecute()\n"));

	execute(get_c_string(cmd));
	return NIL;
}

static LISP linput_warp_pointer(LISP value)
{
	input_warp_pointer = get_c_long(value);
	return NIL;
}

extern int interpreter_count;
extern char* intnames[];

static LISP interpreter_test(LISP intnam)
{
	buffer *buf = buffer_of_window(w_list);
	position pos = get_point(w_list);
	int interpreter, type = ERROR;
	char prompt[80], b[256], *p;
	char *i;
	cval val;
	val.number = 0;

	if (NULLP(intnam))
		interpreter = select_from_list("Which interpreter?",
					intnames, interpreter_count);
	else interpreter = name2interpreter(get_c_string(intnam));
	if (interpreter < 0) return NIL;

	sprintf(prompt, "%s expression:", interpreter2name(interpreter));
	p = ret_text(buf, pos.row, pos.col);
	b[0] = 0;
	if (p) strncpy(b, p, 255);
	if (ask_for_str(prompt, b)) {
		int r = pos.row;
		int c = pos.col;
		if ((i = b) == 0) return NIL;
		undo_save(buf, r, c, r, c);
		ins_data(buf, interpreter, i, val, type, r, c);
		buf->change = TRUE;
		calc_matrix(buf);
		pr_scr_flag = TRUE;
	}
	return NIL;
}

static LISP lundo_save(LISP r1, LISP c1, LISP r2, LISP c2)
{
	double retval = undo_save(buffer_of_window(w_list),
				get_c_long(r1), get_c_long(c1),
				get_c_long(r2), get_c_long(c2));
	return flocons(retval);
}

static LISP lundo_restore()
{
	buffer *b = buffer_of_window(w_list);
	double retval = undo_restore(b);
	b->change = TRUE;
	calc_matrix(b);
	pr_scr_flag = TRUE;
	return flocons(retval);
}

/* Set up the table of functions and names */

/* Commands that take no arguments */
static s_fn_table fn_table[] =
{
	/* moving around */
	{"go-to", go_to},
	{"set-cell-width", set_cell_width},
	{"set-cell-height", set_cell_height},
	{"set-cell-format", set_cell_format},
	{"set-block-format", set_block_format},
	{"set-block-style", set_block_style},
	{"set-cell-style", set_cell_style},
	{"define-style", define_style},

	/* block commands */
	{"fill-block", fill_block},
	{"smart-fill-block", smart_fill_block},
/*	{"save-block-file", save_block_file},
*/
	/* new window */
	{"delete-window", delete_window},
	{"delete-other-windows", delete_other_windows},
	{"split-window-vertically", split_window_vertically},
	{"other-window", other_window},

	/* buffers and windows */
	{"switch-to-buffer", switch_to_buffer},
	{"kill-buffer", kill_buffer},
	{"load-buffer", load_buffer},
	{"save-buffer", save_buffer},
	{"save-buffer-as", save_buffer_as},
	{"load-external", load_external},
	{"save-external", save_external},

	/* help commands */
	{"print-version", print_version},

	/* misc. */
	{"execute-extended-command", execute_extended_command},
	{"quit-siag", quit_siag},
	{"undo-restore", lundo_restore},

	/* keyboard macros */

	/* low level functions */
	{"psformat", lpsformat},
	{(char *) 0, (LISP(*)())0}
};

/* Commands that take 1 argument */

static s_fn_table fn_table1[] = {
	{"get-cell-width", get_cell_width},
	{"get-cell-height", get_cell_height},
	{"auto-recalc", auto_recalc},
	{"block-borders", block_borders},

	{"spawn", spawn},
	{"interpreter-test", interpreter_test},
	{"execute-interpreter-command", execute_interpreter_command},

	/* low level functions */
	{"buffer-changed", buffer_changed},
	{"calc-matrix", lcalc_matrix},
	{"grid-lines", lgrid_lines},
	{"execute", lexecute},
	{"input-warp-pointer", linput_warp_pointer},

	{"confirm-overwrite", lconfirm_overwrite},
	{NULL, NULL}
};

void init_cmds()
{
	int i;

	for (i = 0; fn_table[i].name; i++)
		init_subr_0(fn_table[i].name, fn_table[i].function);
	for (i = 0; fn_table1[i].name; i++)
		init_subr_1(fn_table1[i].name, fn_table1[i].function);
	init_subr_2("downshift-matrix", ldownshift_matrix);
	init_subr_2("upshift-matrix", lupshift_matrix);
	init_subr_2("rightshift-matrix", lrightshift_matrix);
	init_subr_2("leftshift-matrix", lleftshift_matrix);
	init_subr_2("ask-for-str", lask_for_str);
	init_subr_2("alertbox", lalertbox);
	init_subr_3("set-format", set_format);
	init_subr_2("get-format", get_format);
	init_subr_5("set-data", set_data);
	init_subr_3("savematrix", lsavematrix);
	init_subr_2("get-color", lget_color);
	init_subr_3("set-color", lset_color);
	init_subr_4("undo-save", lundo_save);
}

/* begin stuff copied from calc-cmds.c */

#define ESC 27
#define DEL 0x7f
#define C_BIT 31
#define CTRL(x) ((x)&C_BIT)
#define M_BIT 0x80
#define ALT(x) ((x)|M_BIT)

static int is_prefix;   /* indicate if lastc is part of a prefix */
static int prefix_length;
static char prefix_keys[256];
int lastc;	/* remove from window.c */

typedef struct s_kbd_table {
  int length;
  char *keys;
  char *function;       /* the name of the function */
} kbd_table;

static kbd_table keymap[1000];

static int argument = 0, keep_argument;
/* argument is reset after each command that doesn't set keep_argument */

static void install(char *k, int l, char *s)
{
  int i, j;
TRACEME((f, "install(%d,%d,%s)\n", (int)k[0], l, s));

  for (i = 0; keymap[i].keys != NULL; i++) {
    if (keymap[i].length == l)
      if (!memcmp(k, keymap[i].keys, l)) {
	if (keymap[i].function != NULL) free(keymap[i].function);
	keymap[i].function = strdup(s);
	return;
      }
  }
  keymap[i].length = l;
  keymap[i].keys = cmalloc(l);
  for (j = 0; j < l; j++) keymap[i].keys[j] = k[j];
  keymap[i].function = strdup(s);
}

/*	char *k;   the keypresses */
/*	int l;     the # of keypresses */

static char *get_command(char *k, int l)
{
  int i;

  for (i = 0; keymap[i].keys != NULL; i++) {
    if (keymap[i].length == l)
      if (!memcmp(keymap[i].keys, k, l)) break;
  }
  return keymap[i].function;
}

static int keycode(char *p)
{
  static struct {
    char *t;
    int c;
  } spec[] = {
    {"LFD",     '\n'},
    {"RET",     '\r'},
    {"TAB",     '\t'},
    {"ESC",     ESC},
    {"SPC",     ' '},
    {"DEL",     DEL},
    {(char *)0, 0}};
  
  int i;
  
  if (p == NULL) return -1;
  
  switch (strlen(p)) {
  case 1:
    if (isprint(p[0])) return p[0];
    break;
  case 3:
    for (i = 0; spec[i].t != NULL; i++)
      if (!strcmp(p, spec[i].t)) return spec[i].c;
    if (p[0] == 'C' && p[1] == '-' && isprint(p[2])) return CTRL(p[2]);
    if (p[0] == 'M' && p[1] == '-' && isprint(p[2])) return ALT(p[2]);
  default:
    break;
  }
  return -1;
}

static void decode(char *kbd)
{
  int i, is_comment;
  char *keys, *cmd, p[256], *q;
  char kseq[256];

  if (kbd[0] == '\t') return;
  strcpy(p, kbd);

  is_comment = FALSE;
  if ((keys = strtok(p, "\t")) != NULL) {
    cmd = strtok((char *)0, "\t\n");
    for (q = strtok(keys, " "), i = 0;
	 q != NULL;
	 q = strtok((char *)0, " "), i++)
      is_comment |= ((kseq[i] = keycode(q)) == -1);
    if (!is_comment) install(kseq, i, cmd);
  }
}

/*X
static void start_kbd_macro()

Starts recording a keyboard macro.  From now on, everything that is
typed is also recorded into a buffer.  There is a limit to the size of
the buffer, but the limit is big enough for any reasonable use.
X*/
static LISP start_kbd_macro()
{
  if (macro_flag) {
    macro_flag = FALSE;
    llpr("Already defining kbd macro!");
  }
  else {
    macro_flag = TRUE;
    kbd_macro.size = 0;
    llpr("Defining kbd macro...");
  }
  return NIL;
}

/*X
static void end_kbd_macro()

Stops recording the macro.  This command must be bound to a key sequence,
because it actually gets recorded before it is invoked.  It removes itself
from the macro by deleting as many charactesr as the key sequence that
invoked it.  Typing "M-x end-kbd-macro" will only delete the last two
characters from the buffer.
X*/
static LISP end_kbd_macro()
{
  if (!macro_flag) llpr("Not defining kbd macro");
  else {
    macro_flag = FALSE;
    kbd_macro.size -= prefix_length; /* strip off the closing C-x ) */
    llpr("Keyboard macro defined");
  }
  return NIL;
}

extern int add_str_to_input_queue(textbuf);	/* FIXME */

/*X
static void call_last_kbd_macro()

Executes the last macro that was recorded with C-x (.
There can only be one macro defined at a time.
X*/
static LISP call_last_kbd_macro()
{
  if (macro_flag) {
    llpr("Can't execute anonymous macro while defining one");
    macro_flag = FALSE;
  }
  do {
    if (!add_str_to_input_queue(kbd_macro))
      llpr("Input buffer overflow; macro execution terminated");
  } while (--argument > 0);
  return NIL;
}

static LISP prefix_command()
{
  is_prefix = keep_argument = TRUE;
  return NIL;
}

static LISP universal_argument(int argc, char **argv)
{
  if (argc > 1) argument = atoi(argv[1]);
  else if (argument) argument *= 4;
  else argument = 4;
  keep_argument = TRUE;
  return NIL;
}

#define sign(x) ((x)<0?-1:1)
#define todigit(c) (isdigit(c)?(c)-'0':0)

static LISP argument_digit()
{
  argument = 10*argument+sign(argument)*todigit(lastc);
  keep_argument = TRUE;
  return NIL;
}

static LISP argument_sign()
{
  argument = -argument;
  keep_argument = TRUE;
  return NIL;
}

static LISP add_keybinding(LISP keys, LISP cmd)
{
	char *k = get_c_string(keys);
	char *c = get_c_string(cmd);
	int i = 0;
	char kseq[256], p[256], *q;
	strcpy(p, k);
TRACEME((f, "add_keybinding(%s,%s) p='%s'\n", k, c, p));
	for (q = strtok(p, " "); q; q = strtok(NULL, " "))
		if ((kseq[i++] = keycode(q)) == -1) return NIL;
	install(kseq, i, c);
	return NIL;
}

static s_fn_table calc_fn_table[] = {
  {"start-kbd-macro", 		start_kbd_macro},
  {"end-kbd-macro",		end_kbd_macro},
  {"call-last-kbd-macro",	call_last_kbd_macro},
  {"universal-argument",        universal_argument},
  {"argument-digit",            argument_digit},
  {"argument-sign",             argument_sign},

  /* misc. */
  {"prefix",                    prefix_command},
  {"execute-extended-command",  execute_extended_command},
  {"quit",                      quit},
  {NULL,                        NULL}
};

/* Set up the table of keycodes and functions */


void init_calc_cmds()
{
  	int i;

    	decode("C-x\t(prefix)\n");
    	decode("C-x C-c\t(quit-siag)\n");

	for (i = 0; calc_fn_table[i].name; i++)
                init_subr_0(calc_fn_table[i].name, calc_fn_table[i].function);
	init_subr_2("add-keybinding", add_keybinding);
}

void do_cmd(int c)
{
  char *cmd;

TRACEME((f, "do_cmd(%d)\n", c));

  if (!isascii(c)) {
    do_cmd(ESC);
    c = toascii(c);
  }
  if (!is_prefix) prefix_length = 0;
  prefix_keys[prefix_length++] = c;
  is_prefix = keep_argument = FALSE;
  if ((cmd = get_command(prefix_keys, prefix_length)))
	  execute(cmd);
  if (!keep_argument) argument = 0;
}

