/*
 * bltVector.c --
 *
 *	This module implements vector data objects.
 *
 * Copyright 1995-1997 Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no event
 * shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or performance
 * of this software.
 */

#include "bltInt.h"
#include <ctype.h>

#define VECTOR_VERSION "1.1"

#define TRACE_ALL	(TCL_TRACE_WRITES | TCL_TRACE_READS | TCL_TRACE_UNSETS)
#define VECTOR_MAGIC	((unsigned int) 0x46170277)
#define DEF_ARRAY_SIZE  64

#define SPECIAL_INDEX	-2

/* These defines allow parsing of different types of indices */
#define ALLOW_SPECIAL	(1<<0)	/* Recognize "min", "max", and "++end" as
				 * valid indices */
#define ALLOW_COLON	(1<<1)	/* Also recognize a range of indices separated
				 * by a colon */
#define CHECK_RANGE	(1<<2)	/* Verify that the specified index or range
				 * of indices are within limits */

#define USE_VARIABLE	(1<<0)	/* Use Tcl variable API with vector */
#define USE_COMMAND	(1<<1)	/* Use Tcl command API with vector */

/*
 * Vector --
 *
 *	A vector is simple array of double precision values.  It can
 *	be queried or modified by either a Tcl command and array
 *	variable which are created when the array is created.  The
 *	memory storage for the array of values is initially static,
 *	but space is malloc-ed if more is necessary.
 *
 *	Vectors can be modified from C code too. Furthermore, the same
 *	vector can be used in multiple instances (for example, by two
 *	different graph widgets).  All the clients of the vector will
 *	share the data associated with the vector.  Therefore, when a
 *	client wants to use a vector, it allocates a vector
 *	identifier.  A vector identifier uniquely identifies the
 *	client.  The client then uses this id to specify a callback
 *	routine to be invoked whenever the vector is modified or
 *	destroyed.  Whenever the vector is updated or destroyed, each
 *	client is notified of the change by their callback routine.
 *
 */
typedef struct {
    /*
     * If you change these fields, make sure you change the definition
     * of Blt_Vector in bltInt.h and blt.h too.
     */
    double *valueArr;		/* Array of values (malloc-ed) */
    int numValues;		/* Number of values in the array */
    int arraySize;		/* Size of the allocated space */
    double min, max;		/* Minimum and maximum values in the vector */
    int dirty;			/* Indicates if the vector has been updated */
    int reserved;

    /* The following fields are local to this module  */

    Tk_Uid nameId;		/* Name of the vector. This and the
				 * interpreter key the search for
				 * vectors. */
    Tcl_Interp *interp;		/* Interpreter associated with the
				 * vector */

    Tcl_FreeProc *freeProc;	/* Address of procedure to call to
				 * release storage for the value
				 * array, Optionally can be one of the
				 * following: TCL_STATIC, TCL_DYNAMIC,
				 * or TCL_VOLATILE. */
    char *arrayName;		/* The name of the Tcl array variable
				 * mapped to the vector
				 * (malloc'ed). If NULL, indicates
				 * that the vector isn't mapped to any
				 * variable */
    int global;			/* Indicates if the mapped Tcl array
				 * variable is global or not. Will be
				 * set to TCL_GLOBAL_ONLY if variable
				 * is global or 0 otherwise. */
    int offset;			/* Offset from zero of the vector's
				 * starting index */

    Tcl_Command cmdToken;	/* Token for vector's Tcl command. */

    double staticSpace[DEF_ARRAY_SIZE];
    Blt_List clientLst;		/* List of clients using this vector */
    int flags;			/* Notification flags. See definitions
				 * below */

} Vector;


#define NOTIFY_UPDATED		((int)BLT_VECTOR_NOTIFY_UPDATE)
#define NOTIFY_DESTROYED	((int)BLT_VECTOR_NOTIFY_DESTROY)

#define NOTIFY_NEVER		(1<<3)	/* Never notify clients of updates to
					* the vector */
#define NOTIFY_ALWAYS		(1<<4)	/* Notify clients after each update
					* of the vector is made */
#define NOTIFY_WHENIDLE		(1<<5)	/* Notify clients at the next idle point
					* that the vector has been updated. */

#define NOTIFY_PENDING		(1<<6)	/* A do-when-idle notification of the
					* vector's clients is pending. */
#define NOTIFY_NOW		(1<<7)	/* Notify clients of changes once
					 * immediately */

#define NOTIFY_WHEN_MASK	(NOTIFY_NEVER|NOTIFY_ALWAYS|NOTIFY_WHENIDLE)

#define UPDATE_LIMITS		(1<<9) /* The data of the vector has changed.
					* Update the min and max limits when
					* they are needed */
/*
 * ClientInfo --
 *
 *	A vector can be shared by several applications.  Each client
 *	of the vector allocates memory for this structure.  It
 *	contains a pointer to the master information pertaining to the
 *	vector.  It also contains pointers to a routine the call when
 *	the vector changes and data to pass with this routine.
 */
typedef struct {
    unsigned int magic;		/* Magic value designating whether this
				 * really is a vector token or not */
    Vector *master;		/* Pointer to the master record of the vector
				 * If NULL, indicates that vector has been
				 * destroyed. */
    Blt_VectorChangedProc *proc;/* Routine to call when the contents of
				 * the vector change */
    ClientData clientData;	/* Data to pass on ChangedProc call */

} ClientInfo;

/* 
 * VectorIndex
 *
 *	Vectors are keyed by their name and their interpreter. This
 *	lets us use vectors in multiple interpreters. Eventually this
 *	should be extended to namespaces as well.
 */

typedef struct VectorIndex {
    int first, last;
    Blt_VectorIndexProc *readProc;
} VectorIndex;


/* 
 * VectorKey
 *
 *	Vectors are keyed by their name and their interpreter. This
 *	lets us use vectors in multiple interpreters. Eventually this
 *	should be extended to namespaces as well.
 */
typedef struct VectorKey {
    Tk_Uid id;
    Tcl_Interp *interp;
} VectorKey;

static Tcl_HashTable vectorTable; /* Table of vectors */
static int initialized = 0;

static Tcl_HashTable indexProcTable;

static char *VariableProc _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, char *part1, char *part2, int flags));
static int VectorInstCmd _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp,
	int argc, char **argv));



/*
 * ----------------------------------------------------------------------
 *
 * GlobalVariable --
 *
 *	Determines if a variable is global or not.  This is needed
 *	to determine how to set, unset and reset traces on a variable.
 *	There may be a local variable by the same name.  In that case
 *	we want to reset the trace on the global variable only.
 *
 *      Note: This routine uses private information internal to the
 *	      interpreter.  The following structure must be kept
 *	      up-to-date with the structure Interp in tclInt.h. This
 *	      ought to be a library-interface routine. I really don't
 *	      want to do something like the following:
 *
 *            result = Tcl_VarEval(interp, "info globals ", arrayName,
 * 			 (char *)NULL);
 *            if (result == NULL) {
 * 	          return NULL;
 *            }
 *            if (strcmp(interp->result, "1") == 0) {
 * 	          global = TCL_GLOBAL_ONLY;
 *            }
 *            Tcl_ResetResult(interp);
 *
 * Results:
 *	Return 1 if the variable is global and 0 if it is not.  Note
 *	that the variable need not exist first.
 *
 * ----------------------------------------------------------------------
 */

#ifdef ITCL_NAMESPACES

static int
GlobalVariable(interp, name)
    Tcl_Interp *interp;
    char *name;
{
    Tcl_Var varToken;

    if (Itcl_FindVariable(interp, name, 0, &varToken) != TCL_OK) {
	return 0;
    }
    return (varToken != (Tcl_Var) 0);
}

#else

#if (TCL_MAJOR_VERSION < 8) 
typedef struct TclInterpInfo {
    char *result;
    Tcl_FreeProc *freeProc;
    int errorLine;
    Tcl_HashTable commandTable;
    Tcl_HashTable mathFuncTable;

    Tcl_HashTable globalTable;	/* This is the only field we care about */

    int numLevels;
    int maxNestingDepth;
} TclInterpInfo;

static int
GlobalVariable(interp, name)
    Tcl_Interp *interp;
    char *name;
{
    TclInterpInfo *iPtr = (TclInterpInfo *) interp;

    return (Tcl_FindHashEntry(&(iPtr->globalTable), name) != NULL);
}

#else

static int
GlobalVariable(interp, name)
    Tcl_Interp *interp;
    char *name;
{
    int result;

    if (Tcl_VarEval(interp, "info globals ", name, (char *)NULL) != TCL_OK) {
	return 0;
    }
    result = (interp->result[0] != '\0');
    if ((result) && (strcmp(interp->result, name) != 0)) {
	fprintf(stderr, "Fix the API! result=(%s)\n", interp->result);
    }
    Tcl_ResetResult(interp);
    return result;
}

#endif /* TCL_MAJOR_VERSION < 8 */
#endif /* ITCL_NAMESPACES */

/*
 * ----------------------------------------------------------------------
 *
 * FindLimits --
 *
 *	Determines the minimum and maximum values in the vector.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The min and max fields of the vector are updated.
 *
 * ----------------------------------------------------------------------
 */
static void
FindLimits(vPtr)
    Vector *vPtr;
{
    register double min, max;

    min = max = 0.0;
    if (vPtr->numValues > 0) {
	register int i;
	register double *valuePtr;

	min = max = vPtr->valueArr[0];
	valuePtr = vPtr->valueArr + 1;
	for (i = 1; i < vPtr->numValues; i++) {
	    if (min > *valuePtr) {
		min = *valuePtr;
	    } else if (max < *valuePtr) {
		max = *valuePtr;
	    }
	    valuePtr++;
	}
    }
    vPtr->min = min, vPtr->max = max;
    vPtr->flags &= ~UPDATE_LIMITS;
}

static double 
MinIndexProc (vecPtr)
    Blt_Vector *vecPtr;
{
    Vector *vPtr = (Vector *)vecPtr;

    if (vPtr->flags & UPDATE_LIMITS) {
	FindLimits(vPtr);
    }
    return vPtr->min;
}

static double 
MaxIndexProc (vecPtr)
    Blt_Vector *vecPtr;
{
    Vector *vPtr = (Vector *)vecPtr;

    if (vPtr->flags & UPDATE_LIMITS) {
	FindLimits(vPtr);
    }
    return vPtr->max;
}

static double 
MeanIndexProc (vecPtr)
    Blt_Vector *vecPtr;
{
    Vector *vPtr = (Vector *)vecPtr;
    register int i;
    double sum;

    sum = 0.0;
    for (i = 0; i < vPtr->numValues; i++) {
	sum += vPtr->valueArr[i];
    }
    return sum / (double)vPtr->numValues;
}

static double 
SumIndexProc (vecPtr)
    Blt_Vector *vecPtr;
{
    Vector *vPtr = (Vector *)vecPtr;
    register int i;
    double sum;

    sum = 0.0;
    for (i = 0; i < vPtr->numValues; i++) {
	sum += vPtr->valueArr[i];
    }
    return sum;
}

/*
 * ----------------------------------------------------------------------
 *
 * GetIndex --
 *
 *	Converts the string representing an index in the vector, to
 *	its numeric value.  A valid index may be an numeric string of
 *	the string "end" (indicating the last element in the string).
 *
 * Results:
 *	A standard Tcl result.  If the string is a valid index, TCL_OK
 *	is returned.  Otherwise TCL_ERROR is returned and interp->result
 *	will contain an error message.
 *
 * ----------------------------------------------------------------------
 */
static int
GetIndex(vPtr, string, indexPtr, procPtrPtr, flags)
    Vector *vPtr;
    char *string;
    int *indexPtr;
    Blt_VectorIndexProc **procPtrPtr;
    int flags;
{
    char c;
    long int value;

    c = string[0];

    /* Treat the index "end" like a numeric index.  */

    if ((c == 'e') && (strcmp(string, "end") == 0)) {
	if (vPtr->numValues < 1) {
	    Tcl_AppendResult(vPtr->interp, 
		     "invalid index \"end\": vector is empty", (char *)NULL);
	    return TCL_ERROR;
	} 
	*indexPtr = vPtr->numValues - 1;
	return TCL_OK;
    } else if ((c == '+') && (strcmp(string, "++end") == 0)) {
	*indexPtr = vPtr->numValues;
	return TCL_OK;
    } 

    if (procPtrPtr != NULL) {
	Tcl_HashEntry *hPtr;

	hPtr = Tcl_FindHashEntry(&indexProcTable, string);
	if (hPtr != NULL) {
	    *indexPtr = SPECIAL_INDEX;
	    *procPtrPtr = (Blt_VectorIndexProc *) Tcl_GetHashValue(hPtr);
	    return TCL_OK;
	} 
    }
    if (Blt_ExprLong(vPtr->interp, string, &value) != TCL_OK) {
	return TCL_ERROR;
    }

    /* 
     * Correct the index by the current value of the offset. This makes
     * all the numeric indices non-negative, which is how we distinguish
     * the special non-numeric indices.
     */
    value -= vPtr->offset;	

    if ((value < 0) || ((flags & CHECK_RANGE) && (value >= vPtr->numValues))) {
	Tcl_AppendResult(vPtr->interp, "index \"", string,
			 "\" is out of range", (char *)NULL);
	return TCL_ERROR;
    }
    *indexPtr = (int)value;
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * GetIndex2 --
 *
 *	Converts the string representing an index in the vector, to
 *	its numeric value.  A valid index may be an numeric string of
 *	the string "end" (indicating the last element in the string).
 *
 * Results:
 *	A standard Tcl result.  If the string is a valid index, TCL_OK
 *	is returned.  Otherwise TCL_ERROR is returned and interp->result
 *	will contain an error message.
 *
 * ----------------------------------------------------------------------
 */
static int
GetIndex2(vPtr, string, indexPtr, flags)
    Vector *vPtr;
    char *string;
    VectorIndex *indexPtr;
    int flags;
{
    int newIndex;
    char *colon;
    Blt_VectorIndexProc *procPtr;

    colon = NULL;
    if (flags & ALLOW_COLON) {
	colon = strchr(string, ':');
    }
    if (colon != NULL) {
	if (string == colon) {
	    indexPtr->first = 0; /* Default to the first index */
	} else {
	    int result;
	    
	    *colon = '\0';
	    result = GetIndex(vPtr, string, &newIndex, 
		      (Blt_VectorIndexProc **)NULL, flags);
	    *colon = ':';
	    if (result != TCL_OK) {
		return TCL_ERROR;
	    }
	    indexPtr->first = newIndex;
	}
	if (*(colon+1) == '\0') {
	    /* Default to the last index */
	    indexPtr->last = (vPtr->numValues > 0) ? vPtr->numValues - 1 : 0;
	} else {
	    if(GetIndex(vPtr, colon+1, &newIndex, (Blt_VectorIndexProc **)NULL, 
			flags) != TCL_OK) {
		return TCL_ERROR;
	    }
	    indexPtr->last = newIndex;
	}
	if (indexPtr->first > indexPtr->last) {
	    Tcl_AppendResult(vPtr->interp, "invalid range \"",
		     string, "\" (first > last)", (char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	if(GetIndex(vPtr, string, &newIndex, &procPtr, flags) != TCL_OK) {
	    return TCL_ERROR;
	}
	indexPtr->last = indexPtr->first = newIndex;
	indexPtr->readProc = procPtr;
    }
    return TCL_OK;
}


/*
 * ----------------------------------------------------------------------
 *
 * NotifyClients --
 *
 *	Notifies each client of the vector that the vector has changed
 *	(updated or destroyed) by calling the provided function back.
 *	The function pointer may be NULL, in that case the client is
 *	not notified.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The results depend upon what actions the client callbacks
 *	take.
 *
 * ----------------------------------------------------------------------
 */
static void
NotifyClients(clientData)
    ClientData clientData;
{
    Vector *vPtr = (Vector *)clientData;
    Blt_ListItem *iPtr;
    ClientInfo *clientPtr;
    Blt_VectorNotify notify;

    notify = (vPtr->flags & NOTIFY_DESTROYED)
	? BLT_VECTOR_NOTIFY_DESTROY : BLT_VECTOR_NOTIFY_UPDATE;
    vPtr->flags &= ~(NOTIFY_UPDATED | NOTIFY_DESTROYED | NOTIFY_PENDING);

    if (vPtr->flags & UPDATE_LIMITS) {
	FindLimits(vPtr);
    }
    for (iPtr = Blt_ListFirstItem(&(vPtr->clientLst)); iPtr != NULL;
	iPtr = Blt_ListNextItem(iPtr)) {
	clientPtr = (ClientInfo *)Blt_ListGetValue(iPtr);
	if (clientPtr->proc != NULL) {
	    (*clientPtr->proc) (vPtr->interp, clientPtr->clientData, notify);
	}
    }
    /*
     * Some clients may not handle the "destroy" callback properly
     * (they should call Blt_FreeVectorId to release the client
     * identifier), so we need to mark any remaining clients to
     * indicate that the master vector has gone away.
     */
    if (notify == BLT_VECTOR_NOTIFY_DESTROY) {
	for (iPtr = Blt_ListFirstItem(&(vPtr->clientLst)); iPtr != NULL;
	    iPtr = Blt_ListNextItem(iPtr)) {
	    clientPtr = (ClientInfo *)Blt_ListGetValue(iPtr);
	    clientPtr->master = NULL;
	}
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * UpdateClients --
 *
 *	Notifies each client of the vector that the vector has changed
 *	(updated or destroyed) by calling the provided function back.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The results depend upon what actions the client callbacks
 *	take.
 *
 * ----------------------------------------------------------------------
 */
static void
UpdateClients(vPtr)
    Vector *vPtr;
{
    vPtr->dirty++;
    if (vPtr->flags & NOTIFY_NEVER) {
	return;
    }
    vPtr->flags |= NOTIFY_UPDATED;
    if (vPtr->flags & NOTIFY_ALWAYS) {
	NotifyClients((ClientData)vPtr);
	return;
    }
    if (!(vPtr->flags & NOTIFY_PENDING)) {
	vPtr->flags |= NOTIFY_PENDING;
	Tk_DoWhenIdle(NotifyClients, (ClientData)vPtr);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * FlushCache --
 *
 *	Unsets all the elements of the Tcl array variable associated
 *	with the vector, freeing memory associated with the variable.
 *	This includes both the hash table and the hash keys.  The down
 *	side is that this effectively flushes the caching of vector
 *	elements in the array.  This means that the subsequent reads
 *	of the array will require a decimal to string conversion.
 *
 *	This is needed when the vector changes its values, making
 *	the array variable out-of-sync.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	All elements of array variable (except one) are unset, freeing
 *	the memory associated with the variable.
 *
 * ---------------------------------------------------------------------- 
 */
static void
FlushCache(vPtr)
    Vector *vPtr;
{
    if (vPtr->arrayName == NULL) {
	return;			/* Doesn't use the variable API */
    }
    /*
     * Turn off the trace temporarily so that we can unset all the
     * elements in the array.
     */
    Tcl_UntraceVar2(vPtr->interp, vPtr->arrayName, (char *)NULL,
	TRACE_ALL | vPtr->global, VariableProc, (ClientData)vPtr);

    /* Unset the entire array */
    Tcl_UnsetVar2(vPtr->interp, vPtr->arrayName, (char *)NULL, vPtr->global);

    /* Restore the "end" index by default and the trace on the entire array */
    Tcl_SetVar2(vPtr->interp, vPtr->arrayName, "end", "", vPtr->global);
    Tcl_TraceVar2(vPtr->interp, vPtr->arrayName, (char *)NULL,
	TRACE_ALL | vPtr->global, VariableProc, (ClientData)vPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * FindVector --
 *
 *	Searches for the vector associated with the name given.
 *
 * Results:
 *	Returns a pointer to the vector if found, otherwise NULL.
 *	If the name is not associated with a vector and the
 *	TCL_LEAVE_ERR_MSG flag is set, and interp->result will
 *	contain an error message.
 *
 * ----------------------------------------------------------------------
 */
static Vector *
FindVector(interp, vecName, flags)
    Tcl_Interp *interp;
    char *vecName;
    unsigned int flags;
{
    Tcl_HashEntry *hPtr;
    VectorKey key;

    key.id = Tk_GetUid(vecName);
    key.interp = interp;

    hPtr = Tcl_FindHashEntry(&vectorTable, (char *)&key);
    if (hPtr == NULL) {
	if (flags & TCL_LEAVE_ERR_MSG) {
	    Tcl_AppendResult(interp, "can't find a vector \"", vecName, "\"",
		(char *)NULL);
	}
	return NULL;
    }
    return (Vector *)Tcl_GetHashValue(hPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * ResizeVector --
 *
 *	Resizes the vector to the new size.
 *
 *	The new size of the vector is computed by doubling the
 *	size of the vector until it fits number of slots needed
 *	(designated by *length*).  
 *
 *	If the new size is the same as the old, simply adjust the
 *	length of the vector.  Otherwise we're copying the data from
 *	one memory location to another.  Either the new size fits the
 *	static storage of the vector *staticSpace* or it's malloc'ed.
 *	The trailing elements of the vector need to be reset to zero.
 *
 *	If the storage changed memory locations, free up the old
 *	location if it was dynamically allocated.
 *
 * Results:
 *	A standard Tcl result.  If the reallocation is successful,
 *	TCL_OK is returned, otherwise TCL_ERROR.
 *
 * Side effects:
 *	Memory for the array is reallocated.
 *
 * ---------------------------------------------------------------------- */
static int
ResizeVector(vPtr, length)
    Vector *vPtr;
    int length;
{
    int newSize;
    double *newArr;

    /* Compute the new array size by doubling the size until its big enough */
    newSize = DEF_ARRAY_SIZE;
    if (length > DEF_ARRAY_SIZE) {
	while (newSize < length) {
	    newSize += newSize;
	}
    }
    if (newSize == vPtr->arraySize) {
	/* Size is the same, use the current array. */
	newArr = vPtr->valueArr; 
    } else {
	if (newSize > DEF_ARRAY_SIZE) {
	    /* 
	     * The new size of too big for the static array, so
	     * dynamically allocate a new array
	     */
	    newArr = (double *)calloc(newSize, sizeof(double));
	    if (newArr == NULL) {
		return TCL_ERROR;
	    }
	} else {
	    /* The only way we can get here is if the new size is smaller
	     * and we cross the threshold back to static space */
	    assert(vPtr->valueArr != vPtr->staticSpace);
	    assert(newSize < vPtr->arraySize);
	    
	    /* Otherwise, we'll use the static space. */
	    newArr = vPtr->staticSpace;
	}

	/* Copy any previous data */
	if (vPtr->numValues > 0) {
	    memcpy((char *)newArr, (char *)vPtr->valueArr,
		   sizeof(double) * MIN(newSize, vPtr->numValues));
	}
    }

    /* Clear any unused trailing slots in the array */
    if (newSize < vPtr->numValues) {
	memset((char *)newArr + newSize, 0, 
	       sizeof(double) * (vPtr->numValues - newSize));
    }

    if (newArr != vPtr->valueArr) {

	/*  
	 * We're not using the old storage anymore, so free it if
	 * it's not static.  It can be static either because it's
	 * using the default staticspace field of the vector, or
	 * because the user previously reset the vector with a
	 * statically allocated array (setting freeProc to
	 * TCL_STATIC).  
	 */

	if ((vPtr->valueArr != vPtr->staticSpace) &&
	    (vPtr->freeProc != TCL_STATIC)) {
	    if (vPtr->freeProc == TCL_DYNAMIC) {
		free((char *)vPtr->valueArr);
	    } else {
		(*vPtr->freeProc) ((char *)vPtr->valueArr);
	    }
	}
	/* Set the type of the new storage */
	vPtr->freeProc = (newArr == vPtr->staticSpace) 
	    ? TCL_STATIC : TCL_DYNAMIC;
    }

    vPtr->valueArr = newArr;
    vPtr->arraySize = newSize;
    vPtr->numValues = length;
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * DestroyVector --
 *
 *	Removes the memory and frees resources associated with the
 *	vector.
 *
 *	o Removes the trace and the Tcl array variable and unsets
 *	  the variable.
 *	o Notifies clients of the vector that the vector is being
 *	  destroyed.
 *	o Removes any clients that are left after notification.
 *	o Frees the memory (if necessary) allocated for the array.
 *	o Removes the entry from the hash table of vectors.
 *	o Frees the memory allocated for the name.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *
 * ----------------------------------------------------------------------
 */
static void
DestroyVector(clientData)
    ClientData clientData;
{
    Vector *vPtr = (Vector *)clientData;
    Tcl_HashEntry *hPtr;
    Blt_ListItem *iPtr;
    ClientInfo *clientPtr;
    VectorKey key;

    if (vPtr->arrayName != NULL) {
	Tcl_UntraceVar2(vPtr->interp, vPtr->arrayName, (char *)NULL,
			TRACE_ALL | vPtr->global, VariableProc, (ClientData)vPtr);
	Tcl_UnsetVar2(vPtr->interp, vPtr->arrayName, (char *)NULL, vPtr->global);
    }
    vPtr->numValues = 0;

    /* Immediately notify clients that vector is going away */
    if (vPtr->flags & NOTIFY_PENDING) {
	vPtr->flags &= ~NOTIFY_PENDING;
	Tk_CancelIdleCall(NotifyClients, (ClientData)vPtr);
    }
    vPtr->flags |= NOTIFY_DESTROYED;
    NotifyClients((ClientData)vPtr);

    for (iPtr = Blt_ListFirstItem(&(vPtr->clientLst)); iPtr != NULL;
	iPtr = Blt_ListNextItem(iPtr)) {
	clientPtr = (ClientInfo *)Blt_ListGetValue(iPtr);
	free((char *)clientPtr);
    }
    Blt_ListReset(&(vPtr->clientLst));
    if ((vPtr->valueArr != vPtr->staticSpace) &&
	(vPtr->freeProc != TCL_STATIC)) {
	if (vPtr->freeProc == TCL_DYNAMIC) {
	    free((char *)vPtr->valueArr);
	} else {
	    (*vPtr->freeProc) ((char *)vPtr->valueArr);
	}
    }

    key.id = vPtr->nameId;
    key.interp = vPtr->interp;

    hPtr = Tcl_FindHashEntry(&vectorTable, (char *)&key);
    if (hPtr != NULL) {
	Tcl_DeleteHashEntry(hPtr);
    }
    if (vPtr->arrayName != NULL) {
	free((char *)vPtr->arrayName);
    }
    free((char *)vPtr);
}


/*
 * ----------------------------------------------------------------------
 *
 * MapVariableToVector --
 *
 *	Sets up traces on a Tcl variable to access the vector.
 *
 *	If another variable is already mapped, it is first untraced
 *	and removed.  Don't do anything else for variables named ""
 *	(even though Tcl allows this pathology). Saves the name of
 *	the new array variable.
 *
 * Results:
 *	A standard Tcl result. If an error occurs setting the variable
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter.
 *
 * Side effects:
 *	Traces are set for the new variable. The new variable
 *	name is saved in a malloc'ed string in vPtr->arrayName.
 *	If this variable is non-NULL, it indicates that a Tcl variable
 *	has been mapped to this vector.
 *
 * ---------------------------------------------------------------------- 
 */
static int
MapVariableToVector(vPtr, varName)
    Vector *vPtr;
    char *varName;
{
    Tcl_DString varNameStr;	/* Holds the fully qualified variable name */

    if (vPtr->arrayName != NULL) {
	Tcl_UntraceVar2(vPtr->interp, vPtr->arrayName, (char *)NULL,
		TRACE_ALL | vPtr->global, VariableProc, (ClientData)vPtr);
	/* Unset the entire array */
	Tcl_UnsetVar2(vPtr->interp, vPtr->arrayName, (char *)NULL, vPtr->global);

	free(vPtr->arrayName);
	vPtr->arrayName = NULL;
    }
    /*  
     * 
     */

    if ((varName == NULL) || (varName[0] == '\0')) {
	return TCL_OK;		/* If the variable name is the empty string,
				 * simply return after destroying any existing
				 * variable. */
    }
    Tcl_DStringInit(&varNameStr);
#if HAVE_NAMESPACES
    /* 
     * We need the fully qualified name of the variable, starting
     * from the global namespace.  We're going to be destroying and
     * recreating this array variable as we flush the cache, so we
     * need to remember where we got it from.
     */
    if ((varName[0] != ':') && (varName[1] != ':')) {
#if (TCL_MAJOR_VERSION >= 8)
	Tcl_Namespace *Tcl_GetCurrentNamespace _ANSI_ARGS_((Tcl_Interp *interp));
	Tcl_Namespace *spacePtr;

	spacePtr = Tcl_GetCurrentNamespace(vPtr->interp);
	Tcl_DStringAppend(&varNameStr, spacePtr->fullName, -1);
	if (spacePtr->parentPtr != NULL) {
	    Tcl_DStringAppend(&varNameStr, "::", -1);
	}
	Tcl_DStringAppend(&varNameStr, varName, -1);
#endif /* TCL_MAJOR_VERSION >= 8 */
#ifdef ITCL_NAMESPACES
	char *curName;

	curName = Itcl_GetNamespPath(Itcl_GetActiveNamesp(vPtr->interp));
	Tcl_DStringAppend(&varNameStr, curName, -1);
	if ((curName[0] != ':') || (curName[1] != ':') || (curName[2] != '\0')) {
	    Tcl_DStringAppend(&varNameStr, "::", -1);
	}
	Tcl_DStringAppend(&varNameStr, varName, -1);
#endif /* ITCL_NAMESPACES */
	varName = Tcl_DStringValue(&varNameStr);
    }
#endif /* HAVE_NAMESPACES */

    /* Try to unset the new variable name, in case one already exists */
    Tcl_UnsetVar2(vPtr->interp, varName, (char *)NULL, 0);

    /*
     * The index "end" designates the last element in the array. We need
     * to set it here (before checking if the variable is global) in case
     * the variable doesn't already exist.
     */
    if (Tcl_SetVar2(vPtr->interp, varName, "end", "", TCL_LEAVE_ERR_MSG) == NULL) {
	Tcl_DStringFree(&varNameStr);
	return TCL_ERROR;
    }
    
    /* Determine if the variable is global or not */
    if (GlobalVariable(vPtr->interp, varName)) {
	vPtr->global = TCL_GLOBAL_ONLY;
    }
    /* Trace the array on reads, writes, and unsets */
    Tcl_TraceVar2(vPtr->interp, varName, (char *)NULL, (TRACE_ALL | vPtr->global),
        VariableProc, (ClientData)vPtr);

    vPtr->arrayName = strdup(varName);
    Tcl_DStringFree(&varNameStr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * FreeVector --
 *
 *	Deletes the command associated with the vector.  As a side
 *	effect, a Tcl_DeleteCmdProc routine is called to free up
 *	resource with the vector.
 *
 * Results:
 *	A standard Tcl result.  If the reallocation is successful,
 *	TCL_OK is returned, otherwise TCL_ERROR.
 *
 * Side effects:
 *	Memory for the array is reallocated.
 *
 * ----------------------------------------------------------------------
 */
static void
FreeVector(vPtr)
    Vector *vPtr;
{
    /*
     * Deleting the command associated with the vector will trigger a
     * cleanup of the resources used by it.
     */
    Tcl_DeleteCommand(vPtr->interp, vPtr->nameId);
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateVector --
 *
 *	Creates a vector structure and the following items:
 *
 *	o Tcl command
 *	o Tcl array variable and establishes traces on the variable
 *	o Adds a  new entry in the vector hash table
 *
 * Results:
 *	A pointer to the new vector structure.  If an error occurred
 *	NULL is returned and an error message is left in
 *	interp->result.
 *
 * Side effects:
 *	A new Tcl command and array variable is added to the
 *	interpreter.
 *
 * ----------------------------------------------------------------------
 */

static Vector *
CreateVector(interp, vecName, cmdName, varName, newPtr)
    Tcl_Interp *interp;
    char *vecName;		/* Name of the vector */
    char *cmdName;		/* Name of the Tcl command mapped to
				 * the vector */
    char *varName;		/* Name of the Tcl array mapped to the
				 * vector */
    int *newPtr;
{
    Tcl_CmdInfo info;
    Tcl_HashEntry *hPtr;
    Vector *vPtr;
    VectorKey key;
    int isNew;

    vPtr = NULL;		/* Initialize vector pointer */
    *newPtr = 1;

    key.id = Tk_GetUid(vecName);
    key.interp = interp;

    hPtr = Tcl_CreateHashEntry(&vectorTable, (char *)&key, &isNew);
    if (isNew) {
	vPtr = (Vector *)calloc(1, sizeof(Vector));
	assert (vPtr);
	vPtr->flags = NOTIFY_WHENIDLE;
	vPtr->freeProc = TCL_STATIC;
	vPtr->nameId = key.id;
	vPtr->valueArr = vPtr->staticSpace;
	vPtr->arraySize = DEF_ARRAY_SIZE;
	vPtr->interp = interp;
	Blt_InitList(&(vPtr->clientLst), TCL_ONE_WORD_KEYS);
	Tcl_SetHashValue(hPtr, (char *)vPtr);
    } else {
	vPtr = (Vector *)Tcl_GetHashValue(hPtr);
    }
    *newPtr = isNew;
    if ((cmdName != NULL) && (Tcl_GetCommandInfo(interp, cmdName, &info))) {
	if ((ClientData)vPtr != info.clientData) {
	    Tcl_AppendResult(interp, "command \"", cmdName, 
			     "\" already exists", (char *)NULL);
	    goto error;
	}
	/* We get here only if the old is the same as the new name. */
	goto checkVariable;
    } 
    if (vPtr->cmdToken != 0) {	
	char *oldName;
	Tcl_CmdInfo oldInfo;

	/* 
	 * The vector was already mapped to a different command name. 
	 * We need to disable the deleteProc, to remove the command 
	 * entry so that the vector cleanup procedure doesn't get 
	 * invoked inadvertently.
	 */
	oldName = Tcl_GetCommandName(interp, vPtr->cmdToken);
	if (Tcl_GetCommandInfo(interp, oldName, &oldInfo)) {
	    oldInfo.deleteProc = NULL;
	    Tcl_SetCommandInfo(interp, oldName, &oldInfo);
	    Tcl_DeleteCommand(interp, oldName);
	}
	vPtr->cmdToken = 0;
    }
    if (cmdName != NULL) {
	vPtr->cmdToken = Tcl_CreateCommand(interp, vecName, VectorInstCmd,
	       (ClientData)vPtr, DestroyVector);
    } 

checkVariable:
    if (varName != NULL) {
	if (MapVariableToVector(vPtr, varName) != TCL_OK) {
	    goto error;
	}
    }
    return (vPtr);

error:
    DestroyVector(vPtr);
    return NULL;
}

static void
GetValues(vPtr, indexPtr, dsPtr)
    Vector *vPtr;
    VectorIndex *indexPtr;
    Tcl_DString *dsPtr;
{
    register int i;
    char string[TCL_DOUBLE_SPACE + 1];

    for (i = indexPtr->first; i <= indexPtr->last; i++) {
	Tcl_PrintDouble(vPtr->interp, vPtr->valueArr[i], string);
	Tcl_DStringAppendElement(dsPtr, string);
    }
}

static void
SetValues(vPtr, indexPtr, value)
    Vector *vPtr;
    VectorIndex *indexPtr;
    double value;
{
    register int i;

    /* Set possibly an entire range of values */
    for(i = indexPtr->first; i <= indexPtr->last; i++) {
	vPtr->valueArr[i] = value;
    }
    vPtr->flags |= UPDATE_LIMITS;
}


/*
 * ----------------------------------------------------------------------
 *
 * VariableProc --
 *
 * Results:
 *	Always returns NULL.  Only called from a variable trace.
 *
 * Side effects:
 *
 * ----------------------------------------------------------------------
 */
static char *
VariableProc(clientData, interp, part1, part2, flags)
    ClientData clientData;	/* File output information. */
    Tcl_Interp *interp;
    char *part1, *part2;
    int flags;
{
    Vector *vPtr = (Vector *)clientData;
    char string[TCL_DOUBLE_SPACE + 1];
    VectorIndex vecIndex;
    static char errorMesg[200];

    if (part2 == NULL) {
	if (flags & TCL_TRACE_UNSETS) {
	    FreeVector(vPtr);	/* Unsetting the entire array. */
	}
	return NULL;
    }
    if (GetIndex2(vPtr, part2, &vecIndex, 
		  (ALLOW_SPECIAL | CHECK_RANGE | ALLOW_COLON)) != TCL_OK) {
	static char errMsg[200];

	strcpy(errMsg, vPtr->interp->result);
	return (errMsg);
    }

    if (flags & TCL_TRACE_WRITES) {
	double value;
	char *newValue;

	if (vecIndex.first == SPECIAL_INDEX) {
	    return NULL;	/* Tried to set "min" or "max" */
	}
	newValue = Tcl_GetVar2(interp, part1, part2, 0);
	if (newValue == NULL) {
	    return "can't read current vector value";
	}
	if (Blt_ExprDouble(interp, newValue, &value) != TCL_OK) {
	    if ((vecIndex.last == vecIndex.first) && (vecIndex.first >= 0)) {
		/* Single numeric index. Reset the array element to
                   its old value on errors */
		Tcl_PrintDouble(interp, vPtr->valueArr[vecIndex.first], string);
		Tcl_SetVar2(interp, part1, part2, string, 0);
	    }
	    return "bad value for vector element";
	}
	if (vecIndex.first == vPtr->numValues) {
	    if (ResizeVector(vPtr, vPtr->numValues + 1) != TCL_OK) {
		return "error resizing vector";
	    }
	}
	/* Set possibly an entire range of values */
	SetValues(vPtr, &vecIndex, value);
    } else if (flags & TCL_TRACE_READS) {
	double value;

	if (vecIndex.first == vPtr->numValues) {
	    return NULL;	/* Can't read from index "++end" */
	}
	if (vecIndex.first == vecIndex.last) {
	    if (vecIndex.first >= 0) {
		value = vPtr->valueArr[vecIndex.first];
	    } else {
		value = (*vecIndex.readProc)((Blt_Vector *)vPtr);
	    }
	    Tcl_PrintDouble(interp, value, string);
	    if (Tcl_SetVar2(interp, part1, part2, string, 0) == NULL) {
		sprintf(errorMesg, "error setting \"%s(%s)\" on read",
			part1, part2);
		return errorMesg;
	    }
	} else {
	    Tcl_DString dStr;
	    char *result;

	    Tcl_DStringInit(&dStr);
	    GetValues(vPtr, &vecIndex, &dStr);
	    result = Tcl_SetVar2(interp, part1, part2, Tcl_DStringValue(&dStr), 0);
	    Tcl_DStringFree(&dStr);
	    if (result == NULL) {
		sprintf(errorMesg, "error setting \"%s(%s)\" on read",
			part1, part2);
		return errorMesg;
	    }
	}
    } else if (flags & TCL_TRACE_UNSETS) {
	register int i, j;

	if ((vecIndex.first == vPtr->numValues) || 
	    (vecIndex.first == SPECIAL_INDEX)) {
		sprintf(errorMesg, "can't unset \"%s(%s)\": special vector index",
			part1, part2);
		return errorMesg;
	} 
	/*
	 * Collapse the vector from the point of the first unset element.
	 * Also flush any array variable entries so that the shift is
	 * reflected when the array variable is read.
	 */
	for (i = vecIndex.first, j = vecIndex.last + 1; j < vPtr->numValues; 
	     i++, j++) {
	    vPtr->valueArr[i] = vPtr->valueArr[j];
	}
	vPtr->numValues -= ((vecIndex.last - vecIndex.first) + 1);
	FlushCache(vPtr);
	vPtr->flags |= UPDATE_LIMITS;
    } else {
	return "unknown variable flags";
    }
    if (flags & (TCL_TRACE_UNSETS | TCL_TRACE_WRITES)) {
	UpdateClients(vPtr);
    }
    return NULL;
}

/*ARGSUSED*/
static int
SetList(vPtr, numElem, elemArr)
    Vector *vPtr;
    int numElem;
    char **elemArr;
{
    register int i;
    double value;

    if (ResizeVector(vPtr, numElem) != TCL_OK) {
	Tcl_AppendResult(vPtr->interp, "can't resize vector \"",
	    vPtr->nameId, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    for (i = 0; i < numElem; i++) {
	if (Blt_ExprDouble(vPtr->interp, elemArr[i], &value) != TCL_OK) {
	    vPtr->numValues = i;
	    return TCL_ERROR;
	}
	vPtr->valueArr[i] = value;
    }
    return TCL_OK;
}

static int
SetVector(destPtr, srcPtr)
    Vector *destPtr, *srcPtr;
{
    int numBytes;

    if (ResizeVector(destPtr, srcPtr->numValues) != TCL_OK) {
	Tcl_AppendResult(destPtr->interp, "can't resize vector \"",
	    destPtr->nameId, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    numBytes = srcPtr->numValues * sizeof(double);
    memcpy(destPtr->valueArr, srcPtr->valueArr, numBytes);
    if (srcPtr->flags & UPDATE_LIMITS) {
	FindLimits(srcPtr);
    }
    destPtr->min = srcPtr->min;
    destPtr->max = srcPtr->max;
    destPtr->offset = srcPtr->offset;
    return TCL_OK;
}

static int
AppendVector(destPtr, srcPtr)
    Vector *destPtr, *srcPtr;
{
    int numBytes;
    int oldSize, newSize;

    oldSize = destPtr->numValues;
    newSize = oldSize + srcPtr->numValues;
    if (ResizeVector(destPtr, newSize) != TCL_OK) {
	Tcl_AppendResult(destPtr->interp, "can't resize vector \"",
	    destPtr->nameId, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    numBytes = (newSize - oldSize) * sizeof(double);
    memcpy((char *)&(destPtr->valueArr[oldSize]), srcPtr->valueArr, numBytes);
    destPtr->flags |= UPDATE_LIMITS;
    return TCL_OK;
}

static int
AppendList(vPtr, numElem, elemArr)
    Vector *vPtr;
    int numElem;
    char **elemArr;
{
    int count;
    register int i;
    double value;
    int oldSize;

    oldSize = vPtr->numValues;
    if (ResizeVector(vPtr, vPtr->numValues + numElem) != TCL_OK) {
	Tcl_AppendResult(vPtr->interp, "can't resize vector \"",
	    vPtr->nameId, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    count = oldSize;
    for (i = 0; i < numElem; i++) {
	if (Blt_ExprDouble(vPtr->interp, elemArr[i], &value) != TCL_OK) {
	    vPtr->numValues = count;
	    return TCL_ERROR;
	}
	vPtr->valueArr[count++] = value;
    }
    vPtr->flags |= UPDATE_LIMITS;
    return TCL_OK;
}

/* Vector instance option commands */

/*
 * -----------------------------------------------------------------------
 *
 * AppendOp --
 *
 *	Appends one of more Tcl lists of values, or vector objects
 *	onto the end of the current vector object.
 *
 * Results:
 *	A standard Tcl result.  If a current vector can't be created,
 *      resized, any of the named vectors can't be found, or one of
 *	lists of values is invalid, TCL_ERROR is returned.
 *
 * Side Effects:
 *	Clients of current vector will be notified of the change.
 *
 * -----------------------------------------------------------------------
 */
static int
AppendOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    register int i;
    int result;

    for (i = 2; i < argc; i++) {
	if (isalpha(argv[i][0])) {
	    Vector *v2Ptr;

	    v2Ptr = FindVector(interp, argv[i], TCL_LEAVE_ERR_MSG);
	    if (v2Ptr == NULL) {
		return TCL_ERROR;
	    }
	    result = AppendVector(vPtr, v2Ptr);
	} else {
	    int numElem;
	    char **elemArr;

	    if (Tcl_SplitList(interp, argv[i], &numElem, &elemArr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    result = AppendList(vPtr, numElem, elemArr);
	    free((char *)elemArr);
	}
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    if (argc > 2) {
	FlushCache(vPtr);
	UpdateClients(vPtr);
    }
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * ClearOp --
 *
 *	Deletes all the accumulated array indices for the Tcl array
 *	associated will the vector.  This routine can be used to
 *	free excess memory from a large vector.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 * Side Effects:
 *	Memory used for the entries of the Tcl array variable is freed.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ClearOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    FlushCache(vPtr);
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * DeleteOp --
 *
 *	Deletes the given indices from the vector.  If no indices are
 *	provided the entire vector is deleted.
 *
 * Results:
 *	A standard Tcl result.  If any of the given indices is invalid,
 *	interp->result will an error message and TCL_ERROR is returned.
 *
 * Side Effects:
 *	The clients of the vector will be notified of the vector
 *	deletions.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
DeleteOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    unsigned char *unsetArr;
    register int i, j;
    register int count;
    VectorIndex vecIndex;

    if (argc == 2) {
	FreeVector(vPtr);
	return TCL_OK;
    }
    /*
     * Allocate an "unset" bitmap the size of the vector.  We should
     * try to use bit fields instead of a character array, since
     * memory may be an issue if the vector is large.
     */
    unsetArr = (unsigned char *)calloc(sizeof(unsigned char), vPtr->numValues);
    assert(unsetArr);
    for (i = 2; i < argc; i++) {
	if (GetIndex2(vPtr, argv[i], &vecIndex, 
		      (ALLOW_COLON | CHECK_RANGE)) != TCL_OK) {
	    free((char *)unsetArr);
	    return TCL_ERROR;
	}
	for(j = vecIndex.first; j <= vecIndex.last; j++) {
	    unsetArr[j] = TRUE;
	}
    }
    count = 0;
    for (i = 0; i < vPtr->numValues; i++) {
	if (unsetArr[i]) {
	    continue;
	}
	if (count < i) {
	    vPtr->valueArr[count] = vPtr->valueArr[i];
	}
	count++;
    }
    free((char *)unsetArr);
    vPtr->numValues = count;
    vPtr->flags |= UPDATE_LIMITS;
    FlushCache(vPtr);
    UpdateClients(vPtr);
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * DupOp --
 *
 *	Creates one or more duplicates of the vector object.
 *
 * Results:
 *	A standard Tcl result.  If a new vector can't be created,
 *      or and existing vector resized, TCL_ERROR is returned.
 *
 * Side Effects:
 *	Clients of existing vectors will be notified of the change.
 *
 * -----------------------------------------------------------------------
 */
static int
DupOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Vector *v2Ptr;
    int isNew;
    register int i;

    for (i = 2; i < argc; i++) {
	v2Ptr = FindVector(interp, argv[i], 0);
	isNew = 0;
	if (v2Ptr == NULL) {
	    v2Ptr = CreateVector(interp, argv[i], argv[i], argv[i], &isNew);
	}
	if (v2Ptr == NULL) {
	    return TCL_ERROR;
	}
	if (v2Ptr == vPtr) {
	    continue;
	}
	if (SetVector(v2Ptr, vPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (!isNew) {
	    FlushCache(vPtr);
	    UpdateClients(v2Ptr);
	}
    }
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * IndexOp --
 *
 *	Returns the length of the vector.  If a new size is given, the
 *	vector is resized to the new vector.
 *
 * Results:
 *	A standard Tcl result.  If the new length is invalid,
 *	interp->result will an error message and TCL_ERROR is returned.
 *	Otherwise interp->result will contain the length of the vector.
 *
 * -----------------------------------------------------------------------
 */
static int
IndexOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    VectorIndex vecIndex;

    if (GetIndex2(vPtr, argv[2], &vecIndex, 
		  (ALLOW_SPECIAL | CHECK_RANGE | ALLOW_COLON)) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc == 3) {
	Tcl_DString dStr;

	if (vecIndex.first == vPtr->numValues) {
	    Tcl_AppendResult(interp, "can't get index \"", argv[2], "\"",
			     (char *)NULL);
	    return TCL_ERROR;	/* Can't read from index "++end" */
	}
	Tcl_DStringInit(&dStr);
	GetValues(vPtr, &vecIndex, &dStr);
	Tcl_DStringResult(interp, &dStr);
	Tcl_DStringFree(&dStr);
    } else {
	char string[TCL_DOUBLE_SPACE + 1];
	double value;

	if (vecIndex.first == SPECIAL_INDEX) {
	    Tcl_AppendResult(interp, "can't set index \"", argv[2], "\"",
			     (char *)NULL);
	    return TCL_ERROR;	/* Tried to set "min" or "max" */
	}
	if (Blt_ExprDouble(interp, argv[3], &value) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (vecIndex.first == vPtr->numValues) {
	    if (ResizeVector(vPtr, vPtr->numValues + 1) != TCL_OK) {
		Tcl_AppendResult(interp, "error resizing vector", (char *)NULL);
		return TCL_ERROR;
	    }
	}
	SetValues(vPtr, &vecIndex, value);
	Tcl_PrintDouble(interp, value, string);
	Tcl_SetResult(interp, string, TCL_VOLATILE);
	FlushCache(vPtr);
	UpdateClients(vPtr);
    }
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * LengthOp --
 *
 *	Returns the length of the vector.  If a new size is given, the
 *	vector is resized to the new vector.
 *
 * Results:
 *	A standard Tcl result.  If the new length is invalid,
 *	interp->result will an error message and TCL_ERROR is returned.
 *	Otherwise interp->result will contain the length of the vector.
 *
 * -----------------------------------------------------------------------
 */
static int
LengthOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc == 3) {
	int size;

	if (Tcl_GetInt(interp, argv[2], &size) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (size < 0) {
	    Tcl_AppendResult(interp, "bad vector size \"", argv[3], "\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
	if (ResizeVector(vPtr, size) != TCL_OK) {
	    Tcl_AppendResult(vPtr->interp, "can't resize vector \"",
		vPtr->nameId, "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	FlushCache(vPtr);
	UpdateClients(vPtr);
    }
    sprintf(interp->result, "%d", vPtr->numValues);
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * MapOp --
 *
 *	Queries or sets the offset of the array index from the base
 *	address of the data array of values.
 *
 * Results:
 *	A standard Tcl result.  If the source vector doesn't exist
 *	or the source list is not a valid list of numbers, TCL_ERROR
 *	returned.  Otherwise TCL_OK is returned.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
MapOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;			/* not used */
    char **argv;
{

    if (MapVariableToVector(vPtr, argv[2]) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SetResult (interp, vPtr->arrayName, TCL_STATIC);
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * MergeOp --
 *
 *	Merges the values from the given vectors to the current vector.
 *
 * Results:
 *	A standard Tcl result.  If any of the given vectors differ in size,
 *	TCL_ERROR is returned.  Otherwise TCL_OK is returned and the
 *	vector data will contain merged values of the given vectors.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
MergeOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Vector *v2Ptr;
    Vector **vecArr;
    register Vector **vPtrPtr;
    int refSize;
    register int i;

    /*
     * Allocate an array of vector pointers of each vector
     * to be merged in the current vector.
     */
    vecArr = (Vector **)malloc(sizeof(Vector *) * argc);
    assert(vecArr);
    vPtrPtr = vecArr;
    *vPtrPtr = vPtr;		/* Initialize the list with the first vector */
    vPtrPtr++;

    refSize = vPtr->numValues;
    for (i = 2; i < argc; i++) {
	v2Ptr = FindVector(interp, argv[i], TCL_LEAVE_ERR_MSG);
	if (v2Ptr == NULL) {
	    free((char *)vecArr);
	    return TCL_ERROR;
	}
	/* Check that all the vectors are the same length */

	if (v2Ptr->numValues != refSize) {
	    Tcl_AppendResult(vPtr->interp, "vectors \"", vPtr->nameId,
		"\" and \"", v2Ptr->nameId, "\" differ in length",
		(char *)NULL);
	    free((char *)vecArr);
	    return TCL_ERROR;
	}
	*vPtrPtr = v2Ptr;
	vPtrPtr++;
    }
    *vPtrPtr = NULL;

    /* Merge the values from each of the vectors into the current vector */
    for (i = 0; i < refSize; i++) {
	for (vPtrPtr = vecArr; *vPtrPtr != NULL; vPtrPtr++) {
	    Blt_AppendDoubleElement(interp, (*vPtrPtr)->valueArr[i]);
	}
    }
    free((char *)vecArr);
    return TCL_OK;
}
/*
 * -----------------------------------------------------------------------
 *
 * NormalizeOp --
 *
 *	Normalizes the vector.
 *
 * Results:
 *	A standard Tcl result.  If the density is invalid, TCL_ERROR
 *	is returned.  Otherwise TCL_OK is returned.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
NormalizeOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    register int i;
    double range;

    if (vPtr->flags & UPDATE_LIMITS) {
	FindLimits(vPtr);
    }
    range = vPtr->max - vPtr->min;
    if (argc > 2) {
	Vector *v2Ptr;
	int isNew;

	v2Ptr = FindVector(interp, argv[2], 0);
	isNew = 0;
	if (v2Ptr == NULL) {
	    v2Ptr = CreateVector(interp, argv[2], argv[2], argv[2], &isNew);
	}
	if (v2Ptr == NULL) {
	    return TCL_ERROR;
	}
	if (ResizeVector(v2Ptr, vPtr->numValues) != TCL_OK) {
	    Tcl_AppendResult(v2Ptr->interp, "can't resize vector \"",
			     v2Ptr->nameId, "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	for (i = 0; i < vPtr->numValues; i++) {
	    v2Ptr->valueArr[i] = (vPtr->valueArr[i] - vPtr->min) / range;
	}
	v2Ptr->flags |= UPDATE_LIMITS;
	if (!isNew) {
	    FlushCache(vPtr);
	    UpdateClients(v2Ptr);
	}
    } else {
	double norm;

	for (i = 0; i < vPtr->numValues; i++) {
	    norm = (vPtr->valueArr[i] - vPtr->min) / range;
	    Blt_AppendDoubleElement(interp, norm);
	}
    }	
    return TCL_OK;
}


/*
 * -----------------------------------------------------------------------
 *
 * NotifyOp --
 *
 *	Notify clients of vector.
 *
 * Results:
 *	A standard Tcl result.  If any of the given vectors differ in size,
 *	TCL_ERROR is returned.  Otherwise TCL_OK is returned and the
 *	vector data will contain merged values of the given vectors.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
NotifyOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    char c;
    int length;

    c = argv[2][0];
    length = strlen(argv[2]);
    if ((c == 'a') && (length > 1)
	&& (strncmp(argv[2], "always", length) == 0)) {
	vPtr->flags &= ~NOTIFY_WHEN_MASK;
	vPtr->flags |= NOTIFY_ALWAYS;
    } else if ((c == 'n') && (length > 2)
	&& (strncmp(argv[2], "never", length) == 0)) {
	vPtr->flags &= ~NOTIFY_WHEN_MASK;
	vPtr->flags |= NOTIFY_NEVER;
    } else if ((c == 'w') && (length > 1)
	&& (strncmp(argv[2], "whenidle", length) == 0)) {
	vPtr->flags &= ~NOTIFY_WHEN_MASK;
	vPtr->flags |= NOTIFY_WHENIDLE;
    } else if ((c == 'n') && (length > 2)
	&& (strncmp(argv[2], "now", length) == 0)) {
	/* How does this play when an update is pending? */
	NotifyClients(vPtr);
    } else if ((c == 'c') && (length > 1)
	&& (strncmp(argv[2], "cancel", length) == 0)) {
	if (vPtr->flags & NOTIFY_PENDING) {
	    vPtr->flags &= ~NOTIFY_PENDING;
	    Tk_CancelIdleCall(NotifyClients, (ClientData)vPtr);
	}
    } else if ((c == 'p') && (length > 1)
	&& (strncmp(argv[2], "pending", length) == 0)) {
	interp->result = (vPtr->flags & NOTIFY_PENDING) ? "1" : "0";
    } else {
	Tcl_AppendResult(interp, "bad qualifier \"", argv[2], "\": should be \
\"always\", \"never\", \"whenidle\", \"now\", \"cancel\", or \"pending\"",
	    (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * PopulateOp --
 *
 *	Creates or resizes a new vector based upon the density specified.
 *
 * Results:
 *	A standard Tcl result.  If the density is invalid, TCL_ERROR
 *	is returned.  Otherwise TCL_OK is returned.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
PopulateOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Vector *v2Ptr;
    int size, density;
    int isNew;
    register int i, j;
    double slice, range;
    register double *valuePtr;
    int count;

    v2Ptr = FindVector(interp, argv[2], 0);
    isNew = 0;
    if (v2Ptr == NULL) {
	v2Ptr = CreateVector(interp, argv[2], argv[2], argv[2], &isNew);
    }
    if (v2Ptr == NULL) {
	return TCL_ERROR;
    }
    if (Tcl_GetInt(interp, argv[3], &density) != TCL_OK) {
	return TCL_ERROR;
    }
    if (density < 1) {
	Tcl_AppendResult(interp, "invalid density \"", argv[3], "\"",
	    (char *)NULL);
	return TCL_ERROR;
    }
    size = (vPtr->numValues - 1) * (density + 1) + 1;
    if (ResizeVector(v2Ptr, size) != TCL_OK) {
	Tcl_AppendResult(v2Ptr->interp, "can't resize vector \"",
	    v2Ptr->nameId, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    count = 0;
    valuePtr = v2Ptr->valueArr;
    for (i = 0; i < (vPtr->numValues - 1); i++) {
	range = vPtr->valueArr[i + 1] - vPtr->valueArr[i];
	slice = range / (double)(density + 1);
	for (j = 0; j <= density; j++) {
	    *valuePtr = vPtr->valueArr[i] + (slice * (double)j);
	    valuePtr++;
	    count++;
	}
    }
    count++;
    *valuePtr = vPtr->valueArr[i];
    assert(count == v2Ptr->numValues);
    v2Ptr->flags |= UPDATE_LIMITS;
    if (!isNew) {
	FlushCache(v2Ptr);
	UpdateClients(v2Ptr);
    }
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * RangeOp --
 *
 *	Returns a Tcl list of the range of vector values specified.
 *
 * Results:
 *	A standard Tcl result.  If the given range is invalid, TCL_ERROR
 *	is returned.  Otherwise TCL_OK is returned and interp->result
 *	will contain the list of values.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
RangeOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;			/* not used */
    char **argv;
{
    int first, last;
    register int i;

    if ((GetIndex(vPtr, argv[2], &first, (Blt_VectorIndexProc **)NULL, 
		  CHECK_RANGE) != TCL_OK) ||
	(GetIndex(vPtr, argv[3], &last, (Blt_VectorIndexProc **)NULL, 
		  CHECK_RANGE) != TCL_OK)) {
	return TCL_ERROR;
    }
    if (first > last) {
	/* Return the list reversed */
	for (i = last; i <= first; i++) {
	    Blt_AppendDoubleElement(interp, vPtr->valueArr[i]);
	}
    } else {
	for (i = first; i <= last; i++) {
	    Blt_AppendDoubleElement(interp, vPtr->valueArr[i]);
	}
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * OutOfRange --
 *
 *	Determines if a value does not lie within a given range.  
 *
 *	The value is normalized and compared against the interval 
 *	[0..1], where 0.0 is the minimum and 1.0 is the maximum.
 *	DBL_EPSILON is the smallest number that can be represented
 *	on the host machine, such that (1.0 + epsilon) != 1.0.
 *
 *	Please note, *max* can't equal *min*.
 *
 * Results:
 *	Returns whether the value lies outside of the given range.
 *	If value is outside of the interval [min..max], 1 is returned; 
 *	0 otherwise.
 *
 * ---------------------------------------------------------------------- 
 */
INLINE static int
OutOfRange(value, min, max)
    register double value, min, max;
{
    register double norm;

    norm = (value - min) / (max - min);
    return (((norm - 1.0) > DBL_EPSILON) || (((1.0 - norm) - 1.0) > DBL_EPSILON));
}

/*
 * -----------------------------------------------------------------------
 *
 * SearchOp --
 *
 *	Searchs for a value in the vector. Returns the indices of all
 *	vector elements matching a particular value.
 *
 * Results:
 *	Always returns TCL_OK.  interp->result will contain a list
 *	of the indices of array elements matching value. If no elements
 *	match, interp->result will contain the empty string.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
SearchOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;			/* not used */
    char **argv;
{
    double min, max;
    register int i;

    if (Blt_ExprDouble(interp, argv[2], &min) != TCL_OK) {
	return TCL_ERROR;
    }
    max = min;
    if ((argc > 3) && (Blt_ExprDouble(interp, argv[3], &max) != TCL_OK)) {
	return TCL_ERROR;
    }
#ifdef notdef
    if (min > max) {
	double temp;

	temp = max, max = min, min = temp;
    }
#endif
    if (min != max) {
	for (i = 0; i < vPtr->numValues; i++) {
	    if (!OutOfRange(vPtr->valueArr[i], min, max)) {
		Blt_AppendIntElement(interp, i + vPtr->offset);
	    }
	}
    } else {
	for (i = 0; i < vPtr->numValues; i++) {
	    if (vPtr->valueArr[i] == min) {
		Blt_AppendIntElement(interp, i + vPtr->offset);
	    }
	}
    } 
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * OffsetOp --
 *
 *	Queries or sets the offset of the array index from the base
 *	address of the data array of values.
 *
 * Results:
 *	A standard Tcl result.  If the source vector doesn't exist
 *	or the source list is not a valid list of numbers, TCL_ERROR
 *	returned.  Otherwise TCL_OK is returned.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
OffsetOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;			/* not used */
    char **argv;
{
    if (argc == 3) {
	int newOffset;

	if (Tcl_GetInt(interp, argv[2], &newOffset) != TCL_OK) {
	    return TCL_ERROR;
	}
	vPtr->offset = newOffset;
    }
    sprintf(interp->result, "%d", vPtr->offset);
    return TCL_OK;
}


#ifdef HAVE_DRAND48

/*
 * -----------------------------------------------------------------------
 *
 * RandomOp --
 *
 *	Generates random values for the length of the vector.
 *
 * Results:
 *	A standard Tcl result.  
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
RandomOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;			/* not used */
    char **argv;
{
    register int i;
    extern double drand48();

    for (i = 0; i < vPtr->numValues; i++) {
	vPtr->valueArr[i] = drand48();
    }
    FlushCache(vPtr);
    vPtr->flags |= UPDATE_LIMITS;
    UpdateClients(vPtr);
    return TCL_OK;
}

#endif /* HAVE_DRAND48 */

/*
 * -----------------------------------------------------------------------
 *
 * SequenceOp --
 *
 *	Generates a sequence of values in the vector.
 *
 * Results:
 *	A standard Tcl result.  
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
SequenceOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;			/* not used */
    char **argv;
{
    register int i;
    double start, step;

    if (Blt_ExprDouble(interp, argv[2], &start) != TCL_OK) {
	return TCL_ERROR;
    }
    step = 1.0;
    if ((argc > 3) && (Blt_ExprDouble(interp, argv[3], &step) != TCL_OK)) {
	return TCL_ERROR;
    }
    for (i = 0; i < vPtr->numValues; i++) {
	vPtr->valueArr[i] = start + (step * (double)i);
    }
    vPtr->flags |= UPDATE_LIMITS;
    FlushCache(vPtr);
    UpdateClients(vPtr);
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * SetOp --
 *
 *	Sets the data of the vector object from a list of values.
 *
 * Results:
 *	A standard Tcl result.  If the source vector doesn't exist
 *	or the source list is not a valid list of numbers, TCL_ERROR
 *	returned.  Otherwise TCL_OK is returned.
 *
 * Side Effects:
 *	The vector data is reset.  Clients of the vector are notified.
 *	Any cached array indices are flushed.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
SetOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;			/* not used */
    char **argv;
{
    int result;
    Vector *v2Ptr;
    int numElem;
    char **elemArr;

    /* 
     * The source can be either a list of expressions of another vector.
     */
    if (Tcl_SplitList(interp, argv[2], &numElem, &elemArr) != TCL_OK) {
	return TCL_ERROR;
    }
    /* 
     * If there's only one element, check to see whether it's the name
     * of a vector.  Otherwise, treat it as a single numeric expression.  
     */
    if ((numElem == 1) && ((v2Ptr = FindVector(interp, argv[2], 0)) != NULL)) {
	result = SetVector(vPtr, v2Ptr);
    } else {
	result = SetList(vPtr, numElem, elemArr);
    }
    free((char *)elemArr);

    if (result == TCL_OK) {
	/* 
	 * The vector has changed; so flush the array indices (they're
	 * wrong now), find the new limits of the data, and notify
	 * the vector's clients that it's been modified.
	 */
	FlushCache(vPtr);
	vPtr->flags |= UPDATE_LIMITS;
	UpdateClients(vPtr);
    }
    return (result);
}

/*
 * -----------------------------------------------------------------------
 *
 * SortOp --
 *
 *	Sorts the vector object and any other vectors according to
 *	sorting order of the vector object.
 *
 * Results:
 *	A standard Tcl result.  If any of the auxiliary vectors are
 *	a different size than the sorted vector object, TCL_ERROR is
 *	returned.  Otherwise TCL_OK is returned.
 *
 * Side Effects:
 *	The vectors are sorted.
 *
 * -----------------------------------------------------------------------
 */
static double *sortArr;		/* Pointer to the array of values currently
				 * being sorted. */
static int reverse;		/* Indicates the ordering of the sort. If
				 * non-zero, the vectors are sorted in
				 * decreasing order */

static int
CompareVector(a, b)
    void *a;
    void *b;
{
    double delta;
    int result;

    delta = sortArr[*(int *)a] - sortArr[*(int *)b];
    if (delta < 0.0) {
	result = -1;
    } else if (delta > 0.0) {
	result = 1;
    } else {
	return 0;
    }
    if (reverse) {
	result = -result;
    }
    return (result);
}

static int
SortOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int *indexArr;
    double *mergeArr;
    Vector *v2Ptr;
    int refSize, numBytes;
    int result;
    register int i, n;

    reverse = FALSE;
    if ((argc > 2) && (argv[2][0] == '-')) {
	int length;

	length = strlen(argv[2]);
	if ((length > 1) && (strncmp(argv[2], "-reverse", length) == 0)) {
	    reverse = TRUE;
	} else {
	    Tcl_AppendResult(interp, "unknown flag \"", argv[2],
		"\": should be \"-reverse\"", (char *)NULL);
	    return TCL_ERROR;
	}
	argc--, argv++;
    }
    refSize = vPtr->numValues;
    /*
     * Create and initialize an array of indices.  This array will be
     * then sorted based upon the current values in the vector in
     * ascending order.  We'll use this array as a guide for sorting
     * the actual values in the vector and any other vectors listed.
     */
    indexArr = (int *)malloc(sizeof(int) * refSize);
    assert(indexArr);
    for (i = 0; i < refSize; i++) {
	indexArr[i] = i;
    }
    sortArr = vPtr->valueArr;
    qsort((char *)indexArr, refSize, sizeof(int), 
	  (QSortCompareProc *)CompareVector);

    /*
     * Create an array to store a copy of the current values of the
     * vector. We'll merge the values back into the vector based upon
     * the indices found in the index array.
     */
    numBytes = sizeof(double) * refSize;
    mergeArr = (double *)malloc(numBytes);
    assert(mergeArr);
    memcpy((char *)mergeArr, (char *)vPtr->valueArr, numBytes);
    for (n = 0; n < refSize; n++) {
	vPtr->valueArr[n] = mergeArr[indexArr[n]];
    }
    FlushCache(vPtr);
    UpdateClients(vPtr);

    /*
     * Now sort any other vectors in the same fashion.  The vectors
     * must be the same size as the indexArr though.
     */
    result = TCL_ERROR;
    for (i = 2; i < argc; i++) {
	v2Ptr = FindVector(interp, argv[i], TCL_LEAVE_ERR_MSG);
	if (v2Ptr == NULL) {
	    goto error;
	}
	if (v2Ptr->numValues != refSize) {
	    Tcl_AppendResult(interp, "vector \"", v2Ptr->nameId,
		"\" is not the same size as \"", vPtr->nameId, "\"",
		(char *)NULL);
	    goto error;
	}
	memcpy((char *)mergeArr, (char *)v2Ptr->valueArr, numBytes);
	for (n = 0; n < refSize; n++) {
	    v2Ptr->valueArr[n] = mergeArr[indexArr[n]];
	}
	UpdateClients(v2Ptr);
	FlushCache(v2Ptr);
    }
    result = TCL_OK;
  error:
    free((char *)mergeArr);
    free((char *)indexArr);
    return result;
}

/*
 * -----------------------------------------------------------------------
 *
 * ArithOp --
 *
 * Results:
 *	A standard Tcl result.  If the source vector doesn't exist
 *	or the source list is not a valid list of numbers, TCL_ERROR
 *	returned.  Otherwise TCL_OK is returned.
 *
 * Side Effects:
 *	The vector data is reset.  Clients of the vector are notified.
 *	Any cached array indices are flushed.
 *
 * -----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ArithOp(vPtr, interp, argc, argv)
    Vector *vPtr;
    Tcl_Interp *interp;
    int argc;			/* not used */
    char **argv;
{
    register double value;
    register int i;

    if (isalpha(argv[2][0])) {
	Vector *v2Ptr;

	v2Ptr = FindVector(interp, argv[2], TCL_LEAVE_ERR_MSG);
	if (v2Ptr == NULL) {
	    return TCL_ERROR;
	}
	if (v2Ptr->numValues != vPtr->numValues) {
	    Tcl_AppendResult(interp, "vectors \"", argv[0], "\" and \"",
		     argv[2], "\" are not the same length", (char *)NULL);
	    return TCL_ERROR;
	}
	switch (argv[1][0]) {
	case '*':
	    for (i = 0; i < vPtr->numValues; i++) {
		value = vPtr->valueArr[i] * v2Ptr->valueArr[i];
		Blt_AppendDoubleElement(interp, value);
	    }
	    break;

	case '/':
	    for (i = 0; i < vPtr->numValues; i++) {
		value = vPtr->valueArr[i] / v2Ptr->valueArr[i];
		Blt_AppendDoubleElement(interp, value);
	    }
	    break;

	case '-':
	    for (i = 0; i < vPtr->numValues; i++) {
		value = vPtr->valueArr[i] - v2Ptr->valueArr[i];
		Blt_AppendDoubleElement(interp, value);
	    }
	    break;

	case '+':
	    for (i = 0; i < vPtr->numValues; i++) {
		value = vPtr->valueArr[i] + v2Ptr->valueArr[i];
		Blt_AppendDoubleElement(interp, value);
	    }
	    break;
	}
    } else {
	double scalar;

	if (Blt_ExprDouble(interp, argv[2], &scalar) != TCL_OK) {
	    return TCL_ERROR;
	}
	switch (argv[1][0]) {
	case '*':
	    for (i = 0; i < vPtr->numValues; i++) {
		value = vPtr->valueArr[i] * scalar;
		Blt_AppendDoubleElement(interp, value);
	    }
	    break;

	case '/':
	    for (i = 0; i < vPtr->numValues; i++) {
		value = vPtr->valueArr[i] / scalar;
		Blt_AppendDoubleElement(interp, value);
	    }
	    break;

	case '-':
	    for (i = 0; i < vPtr->numValues; i++) {
		value = vPtr->valueArr[i] - scalar;
		Blt_AppendDoubleElement(interp, value);
	    }
	    break;

	case '+':
	    for (i = 0; i < vPtr->numValues; i++) {
		value = vPtr->valueArr[i] + scalar;
		Blt_AppendDoubleElement(interp, value);
	    }
	    break;
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * VectorInstCmd --
 *
 *	Parses and invokes the appropriate vector instance command
 *	option.
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
static Blt_OpSpec vectorOps[] =
{
    {"*", 1, (Blt_Operation) ArithOp,  3, 3, "item",},
    {"+", 1, (Blt_Operation) ArithOp,  3, 3, "item",},
    {"-", 1, (Blt_Operation) ArithOp,  3, 3, "item",},
    {"/", 1, (Blt_Operation) ArithOp,  3, 3, "item",},
    {"append", 1, (Blt_Operation) AppendOp, 3, 0, "item ?item...?",},
    {"clear", 1, (Blt_Operation) ClearOp, 2, 2, "",},
    {"delete", 2, (Blt_Operation) DeleteOp, 2, 0, "index ?index...?",},
    {"dup", 2, (Blt_Operation) DupOp, 3, 0, "vecName",},
    {"index", 1, (Blt_Operation) IndexOp, 3, 4, "index ?value?",},
    {"length", 1, (Blt_Operation) LengthOp, 2, 3, "?newSize?",},
    {"merge", 1, (Blt_Operation) MergeOp, 3, 0, "vecName ?vecName...?",},
    {"normalize", 3, (Blt_Operation) NormalizeOp, 2, 3, "?vecName?",},
    {"notify", 3, (Blt_Operation) NotifyOp, 3, 3, "keyword",},
    {"offset", 2, (Blt_Operation) OffsetOp, 2, 3, "?offset?",},
    {"populate", 1, (Blt_Operation) PopulateOp, 4, 4, "vecName density",},
#ifdef HAVE_DRAND48
    {"random", 4, (Blt_Operation) RandomOp, 2, 2, "",},
#endif
    {"range", 4, (Blt_Operation) RangeOp, 4, 4, "firstIndex lastIndex",},
    {"search", 3, (Blt_Operation) SearchOp, 3, 4, "value ?value?",},
    {"seq", 3, (Blt_Operation) SequenceOp, 3, 4, "start ?step?",},
    {"set", 3, (Blt_Operation) SetOp, 3, 3, "list",},
    {"sort", 2, (Blt_Operation) SortOp, 2, 0, "?-reverse? ?vecName...?",},
    {"variable", 1, (Blt_Operation) MapOp, 3, 3, "varName",},
};
static int numVectorOps = sizeof(vectorOps) / sizeof(Blt_OpSpec);

static int
VectorInstCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Blt_Operation proc;
    Vector *vPtr = (Vector *)clientData;

    proc = Blt_LookupOperation(interp, numVectorOps, vectorOps, BLT_OPER_ARG1,
	argc, argv);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    return ((*proc) (vPtr, interp, argc, argv));
}

/*
 *----------------------------------------------------------------------
 *
 * VectorCmd --
 *
 *	Creates a Tcl command, and array variable representing an
 *	instance of a vector.
 *
 *	vector a 
 *	vector b(20)
 *	vector c(-5:14)
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
VectorCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Unused */
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Vector *vPtr;
    char *leftParen, *rightParen;
    register int i;
    int isNew, size, first, last;
    char *cmdName, *varName;
    int length;

    if (argc == 1) {
	Tcl_HashEntry *hPtr;
	Tcl_HashSearch cursor;

	/* No arguments. List the names of all vectors */

	for (hPtr = Tcl_FirstHashEntry(&vectorTable, &cursor);
	    hPtr != NULL; hPtr = Tcl_NextHashEntry(&cursor)) {
	    vPtr = (Vector *)Tcl_GetHashValue(hPtr);
	    Tcl_AppendElement(interp, vPtr->nameId);
	}
	return TCL_OK;
    }

    /* 
     * Handle switches to the vector command
     */
    varName = cmdName = NULL;
    for (i = 1; i < argc; i++) {
	length = strlen(argv[i]);
	if (argv[i][0] != '-') {
	    break;		/* Not a switch, must be a vector name */
	}
	if ((length > 1) && (strncmp(argv[i], "-variable",  length) == 0)) {
	    if ((i+1) == argc) {
		Tcl_AppendResult(interp, "no variable name supplied with \"", 
				 argv[i], "\" switch", (char *)NULL);
		return TCL_ERROR;
	    }
	    i++;
	    varName = argv[i];
	} else if ((length > 1) && (strncmp(argv[i], "-command",  length) == 0)) {
	    if ((i+1) == argc) {
		Tcl_AppendResult(interp, "no command name supplied with \"", 
				 argv[i], "\" switch", (char *)NULL);
		return TCL_ERROR;
	    }
	    i++;
	    cmdName = argv[i];
	} else if ((length > 1) && (argv[i][1] == '-') && (argv[i][0] == '\0')) {
	    i++;
	    break;		/* Allow vector names to start with - */
	} else {
	    Tcl_AppendResult(interp, "bad vector switch \"", argv[i], "\"", 
			     (char *)NULL);
	    return TCL_ERROR;
	}
    }

    if (i == argc) {
	Tcl_AppendResult(interp, "no vector names supplied", (char *)NULL);
	return TCL_ERROR;
    }
    if ((argc - i) > 1) {
	if ((cmdName != NULL) && (cmdName[0] != '\0')) {
	    Tcl_AppendResult(interp, 
		"can't specify more than one vector with \"-command\" switch", 
  	         (char *)NULL);
	    return TCL_ERROR;
	}
	if ((varName != NULL) && (varName[0] != '\0')) {
	    Tcl_AppendResult(interp, 
		"can't specify more than one vector with \"-variable\" switch", 
  	         (char *)NULL);
	    return TCL_ERROR;
	}
    }
	
    for (/* empty */; i < argc; i++) {
	size = first = last = 0;
	leftParen = strchr(argv[i], '(');
	rightParen = strchr(argv[i], ')');
	if (((leftParen != NULL) && (rightParen == NULL)) ||
	    ((leftParen == NULL) && (rightParen != NULL)) ||
	    (leftParen > rightParen)) {
	    Tcl_AppendResult(interp, "bad vector specification \"", argv[i],
		"\"", (char *)NULL);
	    return TCL_ERROR;
	}
	if (leftParen != NULL) {
	    int result;
	    char *colon;

	    *rightParen = '\0';
	    colon = strchr(leftParen + 1, ':');
	    if (colon != NULL) {

		/* Specification is in the form vecName(first:last) */
		*colon = '\0';
		result = Tcl_GetInt(interp, leftParen + 1, &first);
		if ((*(colon+1) != '\0') && (result == TCL_OK)) {
		    result = Tcl_GetInt(interp, colon + 1, &last);
		    if (first > last) {
			Tcl_AppendResult(interp, "bad vector range \"", argv[i], 
				"\"", (char *)NULL);
			result = TCL_ERROR;
		    }
		    size = (last - first) + 1;
		}
		*colon = ':';
	    } else {
		/* Specification is in the form vecName(size) */
		result = Tcl_GetInt(interp, leftParen + 1, &size);
	    }
	    *rightParen = ')';
	    if (result != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (size < 0) {
		Tcl_AppendResult(interp, "bad vector size \"", argv[i], "\"",
		    (char *)NULL);
		return TCL_ERROR;
	    }
	}
	if (leftParen != NULL) {
	    *leftParen = '\0';
	}

	/*  
	 * By default, we create a Tcl array and a Tcl command by the
	 * name of the vector. This probably will change in a future
	 * release so that variables are *NOT* created unless you
	 * specify a variable name.  
	 */
	vPtr = CreateVector(interp, argv[i], 
			    (cmdName == NULL) ? argv[i] : cmdName, 
			    (varName == NULL) ? argv[i] : varName, 
			    &isNew);
	if (leftParen != NULL) {
	    *leftParen = '(';
	}
	if (vPtr == NULL) {
	    return TCL_ERROR;
	}
	vPtr->offset = first;
	if (size > 0) {
	    if (ResizeVector(vPtr, size) != TCL_OK) {
		Tcl_AppendResult(vPtr->interp, "can't resize vector \"",
		    vPtr->nameId, "\"", (char *)NULL);
		return TCL_ERROR;
	    }
	}
	if (!isNew) {
	    FlushCache(vPtr);
	    UpdateClients(vPtr);
	}

    }
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * VectorDeleteCmd --
 * 
 *	This is called when the "vector" command is deleted from the
 *	interpreter.  It will delete all vectors without a Tcl command
 *	or variable associated with them.  Most vectors will be
 *	destroyed along with the interpreter.  However, if the vector
 *	is "anonymous" then there's nothing to trigger the clean up.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Destroys all the vectors not connected to a variable or command. 
 *
 * ------------------------------------------------------------------------ */
/* ARGSUSED */
static void
VectorDeleteCmd(clientData)
    ClientData clientData;	/* Unused */
{
    Tcl_HashEntry *hPtr;
    Tcl_HashSearch cursor;
    Vector *vPtr;
    Blt_List removeLst;
    Blt_ListItem *iPtr;

    /*
     * First save the vectors in a list. We can't delete any hashtable
     * entries while we're doing a walk of the hash table itself.
     */
    Blt_InitList(&removeLst, TCL_ONE_WORD_KEYS);
    for (hPtr = Tcl_FirstHashEntry(&vectorTable, &cursor);
	 hPtr != NULL; hPtr = Tcl_NextHashEntry(&cursor)) {
	vPtr = (Vector *)Tcl_GetHashValue(hPtr);
	if ((vPtr->arrayName == NULL) && (vPtr->cmdToken == 0)) {
	    Blt_ListAppend(&removeLst, (char *)vPtr, (ClientData)vPtr);
	}
    }
    for (iPtr = Blt_ListFirstItem(&removeLst); iPtr != NULL; 
	 iPtr = Blt_ListNextItem(iPtr)) {
	vPtr = (Vector *)Blt_ListGetValue(iPtr);
	DestroyVector(vPtr);
    }
    Blt_ListReset(&removeLst);
}

/*LINTLIBRARY*/
void
Blt_InstallIndexProc(indexName, procPtr)
    char *indexName;
    Blt_VectorIndexProc *procPtr; /* Pointer to function to be called
				   * when the vector finds the named index.
				   * If NULL, this indicates to remove
				   * the index from the table.
				   */
{
    Tcl_HashEntry *hPtr;
    int dummy;

    hPtr = Tcl_CreateHashEntry(&indexProcTable, indexName, &dummy);
    if (procPtr == NULL) {
	Tcl_DeleteHashEntry(hPtr);
    } else {
	Tcl_SetHashValue(hPtr, (ClientData)procPtr);
    }
}


/*
 * -----------------------------------------------------------------------
 *
 * Blt_VectorInit --
 *
 *	This procedure is invoked to initialize the "vector" command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Creates the new command and adds a new entry into a global Tcl
 *	associative array.
 *
 * ------------------------------------------------------------------------
 */
int
Blt_VectorInit(interp)
    Tcl_Interp *interp;
{
    static Blt_CmdSpec cmdSpec = { "vector", VectorCmd, VectorDeleteCmd, };

    /* Flag this so this routine can be run more than once */
    if (!initialized) {
	Tcl_InitHashTable(&vectorTable, sizeof(VectorKey)/sizeof(int));
	Tcl_InitHashTable(&indexProcTable, TCL_STRING_KEYS);
	initialized = TRUE;
#ifdef HAVE_SRAND48
	srand48(1234L);
#endif
    }
    Blt_InstallIndexProc("min", MinIndexProc);
    Blt_InstallIndexProc("max", MaxIndexProc);
    Blt_InstallIndexProc("mean", MeanIndexProc);
    Blt_InstallIndexProc("sum", SumIndexProc);
#ifdef notdef
    Blt_InstallIndexProc("median", MedianIndexProc);
    Blt_InstallIndexProc("stddev", StddevIndexProc);
    Blt_InstallIndexProc("q1", Q1IndexProc);
    Blt_InstallIndexProc("q2", Q1IndexProc);
#endif
    return (Blt_InitCmd(interp, "blt", &cmdSpec));
}


/* Public C interface to vectors */

/*
 * -----------------------------------------------------------------------
 *
 * Blt_CreateVector --
 *
 *	Creates a new vector by the name and size.
 *
 * Results:
 *	A standard Tcl result.  If the new array size is invalid or a
 *	vector already exists by that name, TCL_ERROR is returned.
 *	Otherwise TCL_OK is returned and the new vector is created.
 *
 * Side Effects:
 *	Memory will be allocated for the new vector.  A new Tcl command
 *	and Tcl array variable will be created.
 *
 * -----------------------------------------------------------------------
 */

/*LINTLIBRARY*/
int
Blt_CreateVector2(interp, vecName, cmdName, varName, initialSize, vecPtrPtr)
    Tcl_Interp *interp;
    char *vecName;
    char *cmdName, *varName;
    int initialSize;
    Blt_Vector **vecPtrPtr;
{
    Vector *vPtr;
    int isNew;

    if (!initialized) {
	Tcl_InitHashTable(&vectorTable, sizeof(VectorKey)/sizeof(int));
	initialized = TRUE;
    }
    if (initialSize < 0) {
	sprintf(interp->result, "bad vector size \"%d\"", initialSize);
	return TCL_ERROR;
    }

    vPtr = CreateVector(interp, vecName, cmdName, varName, &isNew);
    if (vPtr == NULL) {
	return TCL_ERROR;
    }

    if (initialSize > 0) {
	if (ResizeVector(vPtr, initialSize) != TCL_OK) {
	    Tcl_AppendResult(vPtr->interp, "can't resize vector \"",
		vPtr->nameId, "\"", (char *)NULL);
	    return TCL_ERROR;
	}
    }
    if (vecPtrPtr != NULL) {
	*vecPtrPtr = (Blt_Vector *)vPtr;
    }
    return TCL_OK;
}

int
Blt_CreateVector(interp, name, size, vecPtrPtr)
    Tcl_Interp *interp;
    char *name;
    int size;
    Blt_Vector **vecPtrPtr;
{
    return Blt_CreateVector2(interp, name, name, name, size, vecPtrPtr);
}

/*
 * -----------------------------------------------------------------------
 *
 * Blt_DeleteVector --
 *
 *	Deletes the vector of the given name.  All clients with
 *	designated callback routines will be notified.
 *
 * Results:
 *	A standard Tcl result.  If no vector exists by that name,
 *	TCL_ERROR is returned.  Otherwise TCL_OK is returned and
 *	vector is deleted.
 *
 * Side Effects:
 *	Memory will be released for the new vector.  Both the Tcl
 *	command and array variable will be deleted.  All clients which
 *	set call back procedures will be notified.
 *
 * -----------------------------------------------------------------------
 */
/*LINTLIBRARY*/
int
Blt_DeleteVector(vecPtr)
    Blt_Vector *vecPtr;
{
    Vector *vPtr = (Vector *)vecPtr;

    FreeVector(vPtr);
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * Blt_DeleteVectorByName --
 *
 *	Deletes the vector of the given name.  All clients with
 *	designated callback routines will be notified.
 *
 * Results:
 *	A standard Tcl result.  If no vector exists by that name,
 *	TCL_ERROR is returned.  Otherwise TCL_OK is returned and
 *	vector is deleted.
 *
 * Side Effects:
 *	Memory will be released for the new vector.  Both the Tcl
 *	command and array variable will be deleted.  All clients which
 *	set call back procedures will be notified.
 *
 * -----------------------------------------------------------------------
 */
/*LINTLIBRARY*/
int
Blt_DeleteVectorByName(interp, vecName)
    Tcl_Interp *interp;
    char *vecName;
{
    Vector *vPtr;

    if (!initialized) {
	Tcl_InitHashTable(&vectorTable, sizeof(VectorKey)/sizeof(int));
	initialized = TRUE;
    }
    vPtr = FindVector(interp, vecName, TCL_LEAVE_ERR_MSG);
    if (vPtr == NULL) {
	return TCL_ERROR;
    }
    FreeVector(vPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_VectorExists --
 *
 *	Returns whether the vector associated with the client token
 *	still exists.
 *
 * Results:
 *	Returns 1 is the vector still exists, 0 otherwise.
 *
 * ----------------------------------------------------------------------
 */
int
Blt_VectorExists(interp, vecName)
    Tcl_Interp *interp;
    char *vecName;
{
    Vector *vPtr;

    if (!initialized) {
	Tcl_InitHashTable(&vectorTable, sizeof(VectorKey)/sizeof(int));
	initialized = TRUE;
    }
    vPtr = FindVector(interp, vecName, 0);
    return (vPtr != NULL);
}

/*
 * -----------------------------------------------------------------------
 *
 * Blt_GetVector --
 *
 *	Returns a pointer to the vector associated with the given name.
 *
 * Results:
 *	A standard Tcl result.  If there is no vector "name", TCL_ERROR
 *	is returned.  Otherwise TCL_OK is returned and vecPtrPtr will
 *	point to the vector.
 *
 * -----------------------------------------------------------------------
 */
int
Blt_GetVector(interp, vecName, vecPtrPtr)
    Tcl_Interp *interp;
    char *vecName;
    Blt_Vector **vecPtrPtr;
{
    Vector *vPtr;

    if (!initialized) {
	Tcl_InitHashTable(&vectorTable, sizeof(VectorKey)/sizeof(int));
	initialized = TRUE;
    }
    vPtr = FindVector(interp, vecName, TCL_LEAVE_ERR_MSG);
    if (vPtr == NULL) {
	return TCL_ERROR;
    }
    if (vPtr->flags & UPDATE_LIMITS) {
	FindLimits(vPtr);
    }
    *vecPtrPtr = (Blt_Vector *)vPtr;
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * Blt_ResetVector --
 *
 *	Resets the vector data.  This is called by a client to
 *	indicate that the vector data has changed.  The vector does
 *	not need to point to different memory.  Any clients of the
 *	vector will be notified of the change.
 *
 * Results:
 *	A standard Tcl result.  If the new array size is invalid,
 *	TCL_ERROR is returned.  Otherwise TCL_OK is returned and the
 *	new vector data is recorded.
 *
 * Side Effects:
 *	Any client designated callbacks will be posted.  Memory may
 *	be changed for the vector array.
 *
 * -----------------------------------------------------------------------
 */
int
Blt_ResetVector(vecPtr, dataArr, numValues, arraySize, freeProc)
    Blt_Vector *vecPtr;	
    double *dataArr;
    int numValues;
    int arraySize;
    Tcl_FreeProc *freeProc;	/* Address of memory deallocation routine
				 * for the array of values.  Can also be
				 * TCL_STATIC, TCL_DYNAMIC, or TCL_VOLATILE. */
{
    Vector *vPtr;

    vPtr = (Vector *)vecPtr;

    if (arraySize < 0) {
	vPtr->interp->result = "invalid array size";
	return TCL_ERROR;
    }
    if (vPtr->valueArr != dataArr) {

	/*
	 * New array of values is in different memory than the current
	 * vector.
	 */

	if ((dataArr == NULL) || (arraySize == 0)) {
	    /* Empty array. Set up default values */
	    freeProc = TCL_STATIC;
	    dataArr = vPtr->staticSpace;
	    arraySize = DEF_ARRAY_SIZE;
	    numValues = 0;
	} else if (freeProc == TCL_VOLATILE) {
	    double *newArr;
	    
	    /*
	     * Data is volatile. Make a copy of the value array.
	     */
	    newArr = (double *)malloc(sizeof(double) * arraySize);
	    memcpy((char *)newArr, (char *)dataArr, sizeof(double)*numValues);
	    
	    dataArr = newArr;
	    freeProc = TCL_DYNAMIC;
	}
	/* 
	 * Old data was dynamically allocated. Free it before attaching
	 * new data.
	 */

	if ((vPtr->valueArr != vPtr->staticSpace) && 
	    (vPtr->freeProc != TCL_STATIC)) {
	    if (vPtr->freeProc == TCL_DYNAMIC) {
		free((char *)vPtr->valueArr);
	    } else {
		(*freeProc) ((char *)vPtr->valueArr);
	    }
	}
	vPtr->freeProc = freeProc;
	vPtr->valueArr = dataArr;
	vPtr->arraySize = arraySize;
    }
    vPtr->numValues = numValues;
    FlushCache(vPtr);
    FindLimits(vPtr);
    UpdateClients(vPtr);
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * Blt_ResizeVector --
 *
 *	Changes the size of the vector.  All clients with designated
 *	callback routines will be notified of the size change.
 *
 * Results:
 *	A standard Tcl result.  If no vector exists by that name,
 *	TCL_ERROR is returned.  Otherwise TCL_OK is returned and
 *	vector is resized.
 *
 * Side Effects:
 *	Memory may be reallocated for the new vector size.  All clients
 *	which set call back procedures will be notified.
 *
 * -----------------------------------------------------------------------
 */
int
Blt_ResizeVector(vecPtr, length)
    Blt_Vector *vecPtr;
    int length;
{
    Vector *vPtr = (Vector *)vecPtr;

    if (ResizeVector(vPtr, length) != TCL_OK) {
	Tcl_AppendResult(vPtr->interp, "can't resize vector \"", vPtr->nameId,
	    "\"", (char *)NULL);
	return TCL_ERROR;
    }
    FlushCache(vPtr);
    FindLimits(vPtr);
    UpdateClients(vPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Blt_AllocVectorId --
 *
 *	Creates an identifier token for an existing vector.  The
 *	identifier is used by the client routines to get call backs
 *	when (and if) the vector changes.
 *
 * Results:
 *	A standard Tcl result.  If "vecName" is not associated with
 *	a vector, TCL_ERROR is returned and interp->result is filled
 *	with an error message.
 *
 *--------------------------------------------------------------
 */
Blt_VectorId
Blt_AllocVectorId(interp, vecName)
    Tcl_Interp *interp;
    char *vecName;
{
    Vector *vPtr;
    ClientInfo *clientPtr;
    Blt_VectorId clientId;

    if (!initialized) {
	Tcl_InitHashTable(&vectorTable, sizeof(VectorKey)/sizeof(int));
	initialized = TRUE;
    }
    vPtr = FindVector(interp, vecName, TCL_LEAVE_ERR_MSG);
    if (vPtr == NULL) {
	return (Blt_VectorId)0;
    }
    /* Allocate a new client structure */
    clientPtr = (ClientInfo *)calloc(1, sizeof(ClientInfo));
    assert (clientPtr);
    clientPtr->magic = VECTOR_MAGIC;

    /*
     * Add the pointer to the master list of clients
     */
    Blt_ListAppend(&(vPtr->clientLst), (char *)clientPtr, (ClientData)clientPtr);

    clientPtr->master = vPtr;
    clientId = (Blt_VectorId)clientPtr;
    return (clientId);
}

/*
 * -----------------------------------------------------------------------
 *
 * Blt_SetVectorChangedProc --
 *
 *	Sets the routine to be called back when the vector is changed
 *	or deleted.  *clientData* will be provided as an argument. If
 *	*proc* is NULL, no callback will be made.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The designated routine will be called when the vector is changed
 *	or deleted.
 *
 * -----------------------------------------------------------------------
 */
void
Blt_SetVectorChangedProc(clientId, proc, clientData)
    Blt_VectorId clientId;	/* Client token identifying the vector */
    Blt_VectorChangedProc *proc;/* Address of routine to call when the contents
				 * of the vector change. If NULL, no routine
				 * will be called */
    ClientData clientData;	/* One word of information to pass along when
				 * the above routine is called */
{
    ClientInfo *clientPtr = (ClientInfo *)clientId;

    if (clientPtr->magic != VECTOR_MAGIC) {
	return;			/* Not a valid token */
    }
    clientPtr->clientData = clientData;
    clientPtr->proc = proc;
}

/*
 *--------------------------------------------------------------
 *
 * Blt_FreeVectorId --
 *
 *	Releases the token for an existing vector.  This indicates
 *	that the client is no longer interested the vector.  Any
 *	previously specified callback routine will no longer be
 *	invoked when (and if) the vector changes.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Any previously specified callback routine will no longer be
 *	invoked when (and if) the vector changes.
 *
 *--------------------------------------------------------------
 */
void
Blt_FreeVectorId(clientId)
    Blt_VectorId clientId;	/* Client token identifying the vector */
{
    ClientInfo *clientPtr = (ClientInfo *)clientId;

    if (clientPtr->magic != VECTOR_MAGIC) {
	return;			/* Not a valid token */
    }
    if (clientPtr->master != NULL) {
	Vector *vPtr = clientPtr->master;

	/* Remove the client from the master list */
	Blt_ListDelete(&(vPtr->clientLst), (char *)clientPtr);
    }
    free((char *)clientPtr);
}

/*
 *--------------------------------------------------------------
 *
 * Blt_NameOfVectorId --
 *
 *	Returns the name of the vector (and array variable).
 *
 * Results:
 *	The name of the array variable is returned.
 *
 *--------------------------------------------------------------
 */
char *
Blt_NameOfVectorId(clientId)
    Blt_VectorId clientId;	/* Client token identifying the vector */
{
    ClientInfo *clientPtr = (ClientInfo *)clientId;

    if ((clientPtr->magic != VECTOR_MAGIC) || (clientPtr->master == NULL)) {
	return NULL;
    }
    return (clientPtr->master->nameId);
}

/*
 *--------------------------------------------------------------
 *
 * Blt_VectorNotifyPending --
 *
 *	Returns the name of the vector (and array variable).
 *
 * Results:
 *	The name of the array variable is returned.
 *
 *--------------------------------------------------------------
 */
int
Blt_VectorNotifyPending(clientId)
    Blt_VectorId clientId;	/* Client token identifying the vector */
{
    ClientInfo *clientPtr = (ClientInfo *)clientId;

    if ((clientPtr->magic != VECTOR_MAGIC) || (clientPtr->master == NULL)) {
	return 0;
    }
    return (clientPtr->master->flags & NOTIFY_PENDING);
}

/*
 * -----------------------------------------------------------------------
 *
 * Blt_GetVectorById --
 *
 *	Returns a pointer to the vector associated with the client
 *	token.
 *
 * Results:
 *	A standard Tcl result.  If the client token is not associated
 *	with a vector any longer, TCL_ERROR is returned. Otherwise,
 *	TCL_OK is returned and vecPtrPtr will point to vector.
 *
 * -----------------------------------------------------------------------
 */
int
Blt_GetVectorById(interp, clientId, vecPtrPtr)
    Tcl_Interp *interp;
    Blt_VectorId clientId;	/* Client token identifying the vector */
    Blt_Vector **vecPtrPtr;
{
    ClientInfo *clientPtr = (ClientInfo *)clientId;

    if (clientPtr->magic != VECTOR_MAGIC) {
	interp->result = "invalid vector token";
	return TCL_ERROR;
    }
    if (clientPtr->master == NULL) {
	interp->result = "vector no longer exists";
	return TCL_ERROR;
    }
    if (clientPtr->master->flags & UPDATE_LIMITS) {
	FindLimits(clientPtr->master);
    }
    *vecPtrPtr = (Blt_Vector *)clientPtr->master;
    return TCL_OK;
}
