/***************************************************************************
 * SANE - Scanner Access Now Easy.

   microtek.c 

   This file (C) 1997 Matthew Marjanovic

   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an executable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.

 ***************************************************************************

   This file implements a SANE backend for Microtek scanners.

   (feedback to:  mtek-bugs@mir.com)
   (for latest info:  http://www.mir.com/mtek/)

 ***************************************************************************/


#define MICROTEK_MAJOR 0
#define MICROTEK_MINOR 6

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>

#include <math.h>

#include <sane/sane.h>
#include <sane/sanei.h>
#include <sane/sanei_config.h>
#include <sane/sanei_scsi.h>
#include <sane/saneopts.h>

#define BACKEND_NAME microtek
#include <sane/sanei_backend.h>

#include "microtek.h"


#define MICROTEK_CONFIG_FILE "microtek.conf"

#ifndef PATH_MAX
# define PATH_MAX	1024
#endif

#define MM_PER_INCH	25.4

#define SCSI_BUFF_SIZE sanei_scsi_max_request_size /* XXXXXXXXXXx */


#define MIN(a,b) (((a) < (b)) ? (a) : (b))      /* XXXXXXXXX */
#define MAX(a,b) (((a) > (b)) ? (a) : (b))      /* XXXXXXXXX */

static int num_devices = 0;
static Microtek_Device *first_dev = NULL;     /* list of known devices */
static Microtek_Scanner *first_handle = NULL; /* list of open scanners */


#define M_LINEART  "LineArt"
#define M_HALFTONE "Halftone"
#define M_GRAY     "Gray"
#define M_COLOR    "Color"

#define M_OPAQUE   "Opaque/Normal"
#define M_TRANS    "Transparency"

#define M_NONE "None"
#define M_SCALAR "Scalar"
#define M_TABLE "Table"

static SANE_String_Const gamma_mode_list[4] = {
  M_NONE,
  M_SCALAR,
  M_TABLE,
  NULL
};


/* These are for the E6.  Does this hold for other models? */
static SANE_String_Const halftone_mode_list[13] = {
  " 1 53-dot screen (53 gray levels)",
  " 2 Horiz. screen (65 gray levels)",
  " 3 Vert. screen (65 gray levels)",
  " 4 Mixed page (33 gray levels)",
  " 5 71-dot screen (29 gray levels)",
  " 6 60-dot #1 (26 gray levels)",
  " 7 60-dot #2 (26 gray levels)",
  " 8 Fine detail #1 (17 gray levels)",
  " 9 Fine detail #2 (17 gray levels)",
  "10 Slant line (17 gray levels)",
  "11 Posterizing (10 gray levels)",
  "12 High Contrast (5 gray levels)",
  NULL
};



static SANE_Range geo_range50 =
{ SANE_FIX(0), SANE_FIX(5.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range80 =
{ SANE_FIX(0), SANE_FIX(8.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range83 =
{ SANE_FIX(0), SANE_FIX(8.3 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range85 =
{ SANE_FIX(0), SANE_FIX(/*XXXXX8.5*/ 8.45 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range100 =
{ SANE_FIX(0), SANE_FIX(10.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range110 =
{ SANE_FIX(0), SANE_FIX(11.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range1169 =
{ SANE_FIX(0), SANE_FIX(11.69 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range130 =
{ SANE_FIX(0), SANE_FIX(13.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range140 =
{ SANE_FIX(0), SANE_FIX(14.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range35M =
{ SANE_FIX(0), SANE_FIX(35.0), SANE_FIX(0) };
static SANE_Range geo_range36M =
{ SANE_FIX(0), SANE_FIX(36.0), SANE_FIX(0) };
static SANE_Range geo_range000 = 
{ SANE_FIX(0), SANE_FIX(0), SANE_FIX(0) };

static SANE_Range speed_range = {1, 7, 1};

/*static SANE_Range brightness_range = {-100, 100, 1};*/
static SANE_Range brightness_range = {0, 255, 1};
static SANE_Range exposure_range = {-18, 21, 3};
static SANE_Range contrast_range = {-42, 49, 7};
static SANE_Range u8_range = {0, 255, 1};
static SANE_Range analog_gamma_range = 
{ SANE_FIX(0.1), SANE_FIX(4.0), SANE_FIX(0) };



/***************************************************************************
 * Utility Functions
 ***************************************************************************/

static size_t max_string_size (const SANE_String_Const strings[])
{
  size_t size, max_size = 0;
  int i;

  for (i = 0; strings[i]; ++i) {
    size = strlen(strings[i]) + 1;
    if (size > max_size) max_size = size;
  }
  return max_size;
}



/********************************************************************/
/********************************************************************/
/*** Basic SCSI Commands ********************************************/
/********************************************************************/
/********************************************************************/


static SANE_Status
sense_handler (int scsi_fd, u_char *sense, void *arg) 
{
  int *sense_flags = (int *)arg;
  SANE_Status stat;
  
  switch(sense[0]) {
  case 0x00:
    return SANE_STATUS_GOOD;
  case 0x81:           /* COMMAND/DATA ERROR */
    stat = SANE_STATUS_GOOD;
    if (sense[1] & 0x01) {
      if ((sense_flags != NULL) && (*sense_flags & MS_SENSE_IGNORE))
	DBG(10, "sense:  ERR_SCSICMD -- ignored\n");
      else {
	DBG(10, "sense:  ERR_SCSICMD\n");
	stat = SANE_STATUS_IO_ERROR;
      }
    }	
    if (sense[1] & 0x02) {
      DBG(10, "sense:  ERR_TOOMANY\n");
      stat = SANE_STATUS_IO_ERROR;
    }
    return stat;
  case 0x82 :           /* SCANNER HARDWARE ERROR */
    if (sense[1] & 0x01) DBG(10, "sense:  ERR_CPURAMFAIL\n");
    if (sense[1] & 0x02) DBG(10, "sense:  ERR_SYSRAMFAIL\n");
    if (sense[1] & 0x04) DBG(10, "sense:  ERR_IMGRAMFAIL\n");
    if (sense[1] & 0x10) DBG(10, "sense:  ERR_CALIBRATE\n");
    if (sense[1] & 0x20) DBG(10, "sense:  ERR_LAMPFAIL\n");
    if (sense[1] & 0x40) DBG(10, "sense:  ERR_MOTORFAIL\n");
    if (sense[1] & 0x80) DBG(10, "sense:  ERR_FEEDERFAIL\n");
    if (sense[2] & 0x01) DBG(10, "sense:  ERR_POWERFAIL\n");
    if (sense[2] & 0x02) DBG(10, "sense:  ERR_ILAMPFAIL\n");
    if (sense[2] & 0x04) DBG(10, "sense:  ERR_IMOTORFAIL\n");
    if (sense[2] & 0x08) DBG(10, "sense:  ERR_PAPERFAIL\n");
    if (sense[2] & 0x10) DBG(10, "sense:  ERR_FILTERFAIL\n");
    return SANE_STATUS_IO_ERROR;
  case 0x83 :           /* OPERATION ERROR */
    if (sense[1] & 0x01) DBG(10, "sense:  ERR_ILLGRAIN\n");
    if (sense[1] & 0x02) DBG(10, "sense:  ERR_ILLRES\n");
    if (sense[1] & 0x04) DBG(10, "sense:  ERR_ILLCOORD\n");
    if (sense[1] & 0x10) DBG(10, "sense:  ERR_ILLCNTR\n");
    if (sense[1] & 0x20) DBG(10, "sense:  ERR_ILLLENGTH\n");
    if (sense[1] & 0x40) DBG(10, "sense:  ERR_ILLADJUST\n");
    if (sense[1] & 0x80) DBG(10, "sense:  ERR_ILLEXPOSE\n");
    if (sense[2] & 0x01) DBG(10, "sense:  ERR_ILLFILTER\n");
    if (sense[2] & 0x02) DBG(10, "sense:  ERR_NOPAPER\n");
    if (sense[2] & 0x04) DBG(10, "sense:  ERR_ILLTABLE\n");
    if (sense[2] & 0x08) DBG(10, "sense:  ERR_ILLOFFSET\n");
    if (sense[2] & 0x10) DBG(10, "sense:  ERR_ILLBPP\n");
    return SANE_STATUS_IO_ERROR;
  default :
    DBG(10, "sense: unknown error\n");
    DBG(10, "sense = %02x %02x %02x %02x.\n", 
            sense[0], sense[1], sense[2], sense[3]);
    return SANE_STATUS_IO_ERROR;
  }
  return SANE_STATUS_GOOD;
}



static SANE_Status
wait_ready(Microtek_Scanner *ms)
{
  u_int8_t comm[6] = { 0, 0, 0, 0, 0, 0 };
  SANE_Status status;
  int retry = 0;

  DBG(23, ".wait_ready %d...\n", ms->sfd);
  while ((status = sanei_scsi_cmd(ms->sfd, comm, 6, 0, 0)) 
	 != SANE_STATUS_GOOD) {
    DBG(23, "wait_ready failed (%d)\n", retry);
    if (retry > 5) return SANE_STATUS_IO_ERROR; /* XXXXXXXX */
    retry++;
    sleep(3);
  }
  return SANE_STATUS_GOOD;
}
  

static SANE_Status
scanning_frame(Microtek_Scanner *ms)
{
  u_int8_t *data, comm[15] = { 0x04, 0, 0, 0, 0x09, 0 };
  int x1, y1, x2, y2;

  DBG(23, ".scanning_frame %d...\n", ms->sfd);

  if (ms->x1 > ms->x2) {
    x1 = ms->x2;
    x2 = ms->x1;
  } else {
    x1 = ms->x1;
    x2 = ms->x2;
  }
  if (ms->y1 > ms->y2) {
    y1 = ms->y2;
    y2 = ms->y1;
  } else {
    y1 = ms->y1;
    y2 = ms->y2;
  }

  /* E6 weirdness XXXXXXXXXX */
  if (ms->unit_type == MS_UNIT_18INCH) {
    x1 /= 2;
    x2 /= 2;
    y1 /= 2;
    y2 /= 2;
  }

  DBG(23, ".scanning_frame:  in- %d,%d  %d,%d\n",
      ms->x1, ms->y1, ms->x2, ms->y2);
  DBG(23, ".scanning_frame: out- %d,%d  %d,%d\n", x1, y1, x2, y2);
  data = comm + 6;
  data[0] = 
    ((ms->unit_type == MS_UNIT_PIXELS) ? 0x08 : 0 ) |
    ((ms->mode == MS_MODE_HALFTONE) ? 0x01 : 0 );
  data[1] = x1 & 0xFF;
  data[2] = (x1 >> 8) & 0xFF;
  data[3] = y1 & 0xFF;
  data[4] = (y1 >> 8) & 0xFF;
  data[5] = x2 & 0xFF;
  data[6] = (x2 >> 8) & 0xFF;
  data[7] = y2 & 0xFF;
  data[8] = (y2 >> 8) & 0xFF;
  /* FIX_FRAMESIZE ????  XXXXXXXXXXXX */
  /* also, check bounds on x,y vs. max x,y XXXXXX */
  if (DBG_LEVEL >= 192) {
    int i;  
    fprintf(stderr, "SF: ");
    for (i=0;i<6+0x09;i++) fprintf(stderr, "%2x ", comm[i]);
    fprintf(stderr, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6 + 0x09, 0, 0);
}



static SANE_Status
mode_select(Microtek_Scanner *ms)
{
  u_int8_t *data, comm[19] = { 0x15, 0, 0, 0, 0, 0 };

  DBG(23, ".mode_select %d...\n", ms->sfd);
  data = comm + 6; 
  data[0] = 
    0x81 |
    ((ms->unit_type == MS_UNIT_18INCH) ? 0 : 0x08) |
    ((ms->res_type == MS_RES_5PER) ? 0 : 0x02);
  data[1] = ms->resolution_code;
  data[2] = ms->exposure;
  data[3] = ms->contrast;
  data[4] = ms->pattern;
  data[5] = ms->velocity;
  data[6] = ms->shadow;
  data[7] = ms->highlight;
  DBG(23, ".mode_select:  pap_len: %d\n", ms->paper_length);
  data[8] = ms->paper_length & 0xFF;
  data[9] = (ms->paper_length >> 8) & 0xFF;
  data[10] = ms->midtone;
  /* set command/data length */
  comm[4] = (ms->midtone_support) ? 0x0B : 0x0A;

  if (DBG_LEVEL >= 192) {
    int i;  
    fprintf(stderr, "MSL: ");
    for (i=0;i<6+comm[4];i++) fprintf(stderr, "%2x ", comm[i]);
    fprintf(stderr, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6 + comm[4], 0, 0); 
  /* check sense?? XXXXXX */
}


static SANE_Status
mode_select_1(Microtek_Scanner *ms)
{
  u_int8_t *data, comm[16] = { 0x16, 0, 0, 0, 0x0A, 0,
                               0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

  DBG(23, ".mode_select_1 %d...\n", ms->sfd);
  data = comm + 6;
  data[1] = ms->bright_r;
  data[3] = ((ms->allow_calibrate) ? 0 : 0x02) | 0x40; /* QQQQQQQ */

  if (DBG_LEVEL >= 192) {
    int i;  
    fprintf(stderr, "MSL1: ");
    for (i=0;i<6+0x0A;i++) fprintf(stderr, "%2x ", comm[i]);
    fprintf(stderr, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6 + 0x0A, 0, 0);
}


static SANE_Status
mode_sense(Microtek_Scanner *ms)
{
  u_int8_t data[20], comm[6] = { 0x1A, 0, 0, 0, 0, 0};
  size_t lenp;
  SANE_Status status;
  int i;

  DBG(23, ".mode_sense %d...\n", ms->sfd);
  if (ms->onepass) {
    comm[4] = 0x13;
  } else if (ms->midtone_support) {
    comm[4] = 0x0B;
  } else {
    comm[4] = 0x0A;
  }
  lenp = comm[4];

  for (i=0; i<20; i++) data[i] = 0xAB;
  status = sanei_scsi_cmd(ms->sfd, comm, 6, data, &lenp);

  if (DBG_LEVEL >= 192) {
    int i;  
    fprintf(stderr, "MS: ");
    for (i=0;i<lenp;i++) fprintf(stderr, "%2x ", data[i]);
    fprintf(stderr, "\n");
  }

  return status;
}
  
  
static SANE_Status
mode_sense_1(Microtek_Scanner *ms)
{
  u_int8_t *data, comm[36] = { 0x19, 0, 0, 0, 0x1E, 0 };

  DBG(23, ".mode_sense_1...\n");
  data = comm + 6;
  memset(data, 0, 30);
  /*  data[1] = ms->bright_r;   XXXXXXXXYYYYYYYYY
  data[2] = ms->bright_g;
  data[3] = ms->bright_b;
  */
  if (DBG_LEVEL >= 192) {
    int i;  
    fprintf(stderr, "MS1: ");
    for (i=0;i<6+0x1E;i++) fprintf(stderr, "%2x ", comm[i]);
    fprintf(stderr, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6 + 0x1E, 0, 0);
}



static SANE_Status
accessory(Microtek_Scanner *ms)
{
  u_int8_t comm[6] = { 0x10, 0, 0, 0, 0, 0 };

  DBG(23, ".accessory...\n");
  comm[4] = 
    ((ms->useADF) ? 0x41 : 0x40) |
    ((ms->prescan) ? 0x18 : 0x10) |
    /*    ((ms->transparency) ? 0x24 : 0) |*/
    ((ms->transparency) ? 0x24 : 0x20) |
    ((ms->allowbacktrack) ? 0x82 : 0x80);
  /*    ((ms->allowbacktrack) ? 0x81 : 0); QQQQQQQQ */

  if (DBG_LEVEL >= 192) {
    int i;  
    fprintf(stderr, "AC: ");
    for (i=0;i<6;i++) fprintf(stderr, "%2x ", comm[i]);
    fprintf(stderr, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6, 0, 0);
}



static SANE_Status
start_scan(Microtek_Scanner *ms)
{
  u_int8_t comm[6] = { 0x1B, 0, 0, 0, 0, 0 };

  DBG(23, ".start_scan...\n");
  comm[4] = 
    0x01 |  /* "start" */
    ((ms->expandedresolution) ? 0x80 : 0) |
    ((ms->multibit) ? 0x40 : 0) |
    ((ms->onepasscolor) ? 0x20 : 0) |
    ((ms->reversecolors) ? 0x04 : 0) |
    ((ms->fastprescan) ? 0x02 : 0) |
    ((ms->filter == MS_FILT_RED) ? 0x08 : 0) |
    ((ms->filter == MS_FILT_GREEN) ? 0x10 : 0) |
    ((ms->filter == MS_FILT_BLUE) ? 0x18 : 0) ;

  if (DBG_LEVEL >= 192) {
    int i;  
    fprintf(stderr, "SS: ");
    for (i=0;i<6;i++) fprintf(stderr, "%2x ", comm[i]);
    fprintf(stderr, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6, 0, 0);
}


static SANE_Status
stop_scan(Microtek_Scanner *ms)
{
  u_int8_t comm[6] = { 0x1B, 0, 0, 0, 0, 0 };

  DBG(23, ".stop_scan...\n");
  if (DBG_LEVEL >= 192) {
    int i;  
    fprintf(stderr, "SPS:");
    for (i=0;i<6;i++) fprintf(stderr, "%2x ", comm[i]);
    fprintf(stderr, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6, 0, 0);
  /* do some wait crap?  XXXXXXXXXXXX */
}

  
static SANE_Status
get_scan_status(Microtek_Scanner *ms,
		SANE_Int *busy,
		SANE_Int *bytes_per_line,
		SANE_Int *lines)
{
  u_int8_t data[6], comm[6] = { 0x0F, 0, 0, 0, 0x06, 0 };
  SANE_Status status;
  size_t lenp;
  int retry = 0;

  DBG(23, ".get_scan_status %d...\n", ms->sfd);

  do {
    lenp = 6;
    /* do some retry stuff in here too XXXXXXXXXXXXX */
    status = sanei_scsi_cmd(ms->sfd, comm, 6, data, &lenp);
    *busy = data[0];
    *bytes_per_line = (data[1]) + (data[2] << 8);
    *lines = (data[3]) + (data[4] << 8) + (data[5] << 16);
    
    DBG(20, "get_scan_status(%lu): %d, %d, %d\n", 
	(u_long) lenp, *busy, *bytes_per_line, *lines);
    DBG(20, "> %2x %2x %2x %2x %2x %2x\n",
	    data[0], data[1], data[2], data[3], data[4], data[5]);
    if (*busy != 0) {
      retry++;
      sleep(1);
    }
  } while ((*busy != 0) && (retry < 10));
    
  return status;
}


static SANE_Status
read_scan_data(Microtek_Scanner *ms,
	       int lines,
	       u_int8_t *buffer, 
	       size_t *bufsize)
{
  u_int8_t comm[6] = { 0x08, 0, 0, 0, 0, 0 };

  DBG(23, ".read_scan_data...\n");
  comm[2] = (lines >> 16) & 0xFF;
  comm[3] = (lines >> 8) & 0xFF;
  comm[4] = (lines) & 0xFF;
  
  return sanei_scsi_cmd(ms->sfd, comm, 6, buffer, bufsize);
}
  
  
static SANE_Status
download_gamma(Microtek_Scanner *ms)
{
  u_int8_t *data, comm[266] = { 0x55, 0, 0x27, 0, 0,
				   0, 0,    0, 0, 0 };
  int i, pl;
  SANE_Status status;

  DBG(23, ".download_gamma...\n");
  data = comm + 10;
  
  comm[7] = 1;
  comm[8] = 0;
  comm[9] = 0;

  if (!(strcmp(ms->val[OPT_CUSTOM_GAMMA].s, M_TABLE))) {
    if (ms->val[OPT_GAMMA_BIND].w == SANE_TRUE) {
      for (i=0; i<256; i++) 
	data[i] = (u_int8_t) ms->gamma_table[0][i];
      return sanei_scsi_cmd(ms->sfd, comm, 266, 0, 0);
    } else {
      pl = 1;
      do {
	for (i=0; i<256; i++) 
	  data[i] = (u_int8_t) ms->gamma_table[pl][i];
	comm[9] = pl << 6;
	status = sanei_scsi_cmd(ms->sfd, comm, 266, 0, 0);
	pl++;
      } while ((pl < 4) && (status == SANE_STATUS_GOOD));
      return status;
    }
  } else if (!(strcmp(ms->val[OPT_CUSTOM_GAMMA].s, M_SCALAR))) {
    if (ms->val[OPT_GAMMA_BIND].w == SANE_TRUE) {
      double gamma = SANE_UNFIX(ms->val[OPT_ANALOG_GAMMA].w);
      for (i=0; i<256; i++)
	data[i] = (u_int8_t) (255 * pow((double) i / 255.0, 1.0 / gamma));
      return sanei_scsi_cmd(ms->sfd, comm, 266, 0, 0);
    } else {
      double gamma;
      pl = 1;
      do {
	switch (pl) {
	case 1: gamma = SANE_UNFIX(ms->val[OPT_ANALOG_GAMMA_R].w); break;
	case 2: gamma = SANE_UNFIX(ms->val[OPT_ANALOG_GAMMA_G].w); break;
	case 3: gamma = SANE_UNFIX(ms->val[OPT_ANALOG_GAMMA_B].w); break;
	default: gamma = 1.0; break; /*XXXXXXXX*/
	}
	for (i=0; i<256; i++) 
	  data[i] = (u_int8_t) (255 * pow((double) i / 255.0, 1.0 / gamma));
	comm[9] = pl << 6;
	status = sanei_scsi_cmd(ms->sfd, comm, 266, 0, 0);
	pl++;
      } while ((pl < 4) && (status == SANE_STATUS_GOOD));
      return status;
    }
  } else {  /* no custom gamma tables */
    for (i=0; i<256; i++) 
      data[i] = (u_int8_t) i;
    return sanei_scsi_cmd(ms->sfd, comm, 266, 0, 0);
  }
}



/***************************************************************************
 *
 * Myriad of internal functions
 *
 ***************************************************************************/



/***************************************************************************/
/* Initialize the options registry                                         */
/***************************************************************************/
static SANE_Status 
init_options(Microtek_Scanner *ms)
{
  int i;
  SANE_Option_Descriptor *sod = ms->sod;
  Microtek_Option_Value *val = ms->val;

  DBG(15, "init_options...\n");

  memset(ms->sod, 0, sizeof(ms->sod));
  memset(ms->val, 0, sizeof(ms->val));
  /* default:  software selectable word options... */
  for (i=0; i<NUM_OPTIONS; i++) {
    sod[i].size = sizeof(SANE_Word);
    sod[i].cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
  }

  sod[OPT_NUM_OPTS].name =   SANE_NAME_NUM_OPTIONS;
  sod[OPT_NUM_OPTS].title =  SANE_TITLE_NUM_OPTIONS;
  sod[OPT_NUM_OPTS].desc =   SANE_DESC_NUM_OPTIONS;
  sod[OPT_NUM_OPTS].type =   SANE_TYPE_INT;
  sod[OPT_NUM_OPTS].unit =   SANE_UNIT_NONE;
  sod[OPT_NUM_OPTS].size =   sizeof (SANE_Word);
  sod[OPT_NUM_OPTS].cap =    SANE_CAP_SOFT_DETECT;
  sod[OPT_NUM_OPTS].constraint_type = SANE_CONSTRAINT_NONE;

  /* The Scan Mode Group */
  sod[OPT_MODE_GROUP].name  = "";
  sod[OPT_MODE_GROUP].title = "Scan Mode";
  sod[OPT_MODE_GROUP].desc  = "";
  sod[OPT_MODE_GROUP].type  = SANE_TYPE_GROUP;
  sod[OPT_MODE_GROUP].cap   = 0;
  sod[OPT_MODE_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  sod[OPT_MODE].name = SANE_NAME_SCAN_MODE;
  sod[OPT_MODE].title = SANE_TITLE_SCAN_MODE;
  sod[OPT_MODE].desc = SANE_DESC_SCAN_MODE;
  sod[OPT_MODE].type = SANE_TYPE_STRING;
  sod[OPT_MODE].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  {
    SANE_String_Const *mode_list;
    mode_list = (SANE_String_Const *) malloc(5 * sizeof(SANE_String_Const));
    if (mode_list == NULL) return SANE_STATUS_NO_MEM;
    i = 0;
    if (ms->dev->info.modes & MI_MODES_COLOR)    mode_list[i++] = M_COLOR;
    if (ms->dev->info.modes & MI_MODES_GRAY)     mode_list[i++] = M_GRAY;
    if (ms->dev->info.modes & MI_MODES_HALFTONE) mode_list[i++] = M_HALFTONE;
    if (ms->dev->info.modes & MI_MODES_LINEART)  mode_list[i++] = M_LINEART;
    mode_list[i] = NULL;
    sod[OPT_MODE].constraint.string_list = mode_list;
    sod[OPT_MODE].size                   = max_string_size(mode_list);
    val[OPT_MODE].s = strdup(mode_list[0]);
  }

  sod[OPT_RESOLUTION].name  = SANE_NAME_SCAN_RESOLUTION;
  sod[OPT_RESOLUTION].title = SANE_TITLE_SCAN_RESOLUTION;
  sod[OPT_RESOLUTION].desc  = SANE_DESC_SCAN_RESOLUTION;
  sod[OPT_RESOLUTION].type  = SANE_TYPE_FIXED;
  sod[OPT_RESOLUTION].unit  = SANE_UNIT_DPI;
  sod[OPT_RESOLUTION].constraint_type  = SANE_CONSTRAINT_RANGE;
  {
    SANE_Int maxres = ms->dev->info.base_resolution;

    ms->res_range.max = SANE_FIX(maxres);
    ms->exp_res_range.max = SANE_FIX(2 * maxres);
    if (ms->dev->info.res_step & MI_RESSTEP_1PER) {
      DBG(23, "init_options:  quant yes\n");
      ms->res_range.min = SANE_FIX( maxres / 100 );
      ms->res_range.quant = ms->res_range.min;
      ms->exp_res_range.min = SANE_FIX(2 * maxres / 100);
      ms->exp_res_range.quant = ms->exp_res_range.min;
    } else {
      /* XXXXXXXXXXX */
      DBG(23, "init_options:  quant no\n");
      ms->res_range.quant = SANE_FIX( 0 );
    }
    sod[OPT_RESOLUTION].constraint.range = &(ms->res_range);
  }
  val[OPT_RESOLUTION].w     = SANE_FIX(100);

  sod[OPT_HALFTONE_PATTERN].name = SANE_NAME_HALFTONE;
  sod[OPT_HALFTONE_PATTERN].title = SANE_TITLE_HALFTONE;
  sod[OPT_HALFTONE_PATTERN].desc = SANE_DESC_HALFTONE_PATTERN;
  sod[OPT_HALFTONE_PATTERN].type = SANE_TYPE_STRING;
  sod[OPT_HALFTONE_PATTERN].size = max_string_size(halftone_mode_list);
  sod[OPT_HALFTONE_PATTERN].cap  |= SANE_CAP_INACTIVE;
  sod[OPT_HALFTONE_PATTERN].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  sod[OPT_HALFTONE_PATTERN].constraint.string_list = halftone_mode_list;
  val[OPT_HALFTONE_PATTERN].s = strdup(halftone_mode_list[0]);

  sod[OPT_NEGATIVE].name  = SANE_NAME_NEGATIVE;
  sod[OPT_NEGATIVE].title = SANE_TITLE_NEGATIVE;
  sod[OPT_NEGATIVE].desc  = SANE_DESC_NEGATIVE;
  sod[OPT_NEGATIVE].type  = SANE_TYPE_BOOL;
  sod[OPT_NEGATIVE].cap   |= 
    (ms->dev->info.modes & MI_MODES_NEGATIVE) ? 0 : SANE_CAP_INACTIVE;
  val[OPT_NEGATIVE].w     = SANE_FALSE;

  sod[OPT_SPEED].name  = SANE_NAME_SCAN_SPEED;
  sod[OPT_SPEED].title = SANE_TITLE_SCAN_SPEED;
  sod[OPT_SPEED].desc  = SANE_DESC_SCAN_SPEED;
  sod[OPT_SPEED].type  = SANE_TYPE_INT;
  sod[OPT_SPEED].cap   |= SANE_CAP_ADVANCED;
  sod[OPT_SPEED].unit  = SANE_UNIT_NONE;
  sod[OPT_SPEED].size  = sizeof(SANE_Word);
  sod[OPT_SPEED].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_SPEED].constraint.range = &speed_range;
  val[OPT_SPEED].w     = 1;

  sod[OPT_SOURCE].name  = SANE_NAME_SCAN_SOURCE;
  sod[OPT_SOURCE].title = SANE_TITLE_SCAN_SOURCE;
  sod[OPT_SOURCE].desc  = SANE_DESC_SCAN_SOURCE;
  sod[OPT_SOURCE].type  = SANE_TYPE_STRING;
  sod[OPT_SOURCE].unit  = SANE_UNIT_NONE;
  sod[OPT_SOURCE].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  {
    SANE_String_Const *source_list;
    source_list = (SANE_String_Const *) malloc(3 * sizeof(SANE_String_Const));
    if (source_list == NULL) return SANE_STATUS_NO_MEM;
    i = 0;
    source_list[i++] = M_OPAQUE;
    if (ms->dev->info.source_options & MI_SRC_HAS_TRANS)
      source_list[i++] = M_TRANS;
    source_list[i] = NULL;
    sod[OPT_SOURCE].constraint.string_list = source_list;
    sod[OPT_SOURCE].size                   = max_string_size(source_list);
    if (i < 2) 
      sod[OPT_SOURCE].cap |= SANE_CAP_INACTIVE;
    val[OPT_SOURCE].s = strdup(source_list[0]);
  }

  sod[OPT_PREVIEW].name  = SANE_NAME_PREVIEW;
  sod[OPT_PREVIEW].title = SANE_TITLE_PREVIEW;
  sod[OPT_PREVIEW].desc  = SANE_DESC_PREVIEW;
  sod[OPT_PREVIEW].type  = SANE_TYPE_BOOL; 
  sod[OPT_PREVIEW].unit  = SANE_UNIT_NONE;
  sod[OPT_PREVIEW].size  = sizeof(SANE_Word);
  val[OPT_PREVIEW].w     = SANE_FALSE;


  sod[OPT_GEOMETRY_GROUP].name  = "";
  sod[OPT_GEOMETRY_GROUP].title = "Geometry";
  sod[OPT_GEOMETRY_GROUP].desc  = "";
  sod[OPT_GEOMETRY_GROUP].type  = SANE_TYPE_GROUP;
  sod[OPT_GEOMETRY_GROUP].cap   = SANE_CAP_ADVANCED;
  sod[OPT_GEOMETRY_GROUP].constraint_type = SANE_CONSTRAINT_NONE;


  sod[OPT_TL_X].name  = SANE_NAME_SCAN_TL_X;
  sod[OPT_TL_X].title = SANE_TITLE_SCAN_TL_X;
  sod[OPT_TL_X].desc  = SANE_DESC_SCAN_TL_X;
  sod[OPT_TL_X].type  = SANE_TYPE_FIXED;
  sod[OPT_TL_X].unit  = SANE_UNIT_MM;
  sod[OPT_TL_X].size  = sizeof(SANE_Word);
  sod[OPT_TL_X].constraint_type = SANE_CONSTRAINT_RANGE;

  sod[OPT_TL_Y].name  = SANE_NAME_SCAN_TL_Y;
  sod[OPT_TL_Y].title = SANE_TITLE_SCAN_TL_Y;
  sod[OPT_TL_Y].desc  = SANE_DESC_SCAN_TL_Y;
  sod[OPT_TL_Y].type  = SANE_TYPE_FIXED;
  sod[OPT_TL_Y].unit  = SANE_UNIT_MM;
  sod[OPT_TL_Y].size  = sizeof(SANE_Word);
  sod[OPT_TL_Y].constraint_type = SANE_CONSTRAINT_RANGE;

  sod[OPT_BR_X].name  = SANE_NAME_SCAN_BR_X;
  sod[OPT_BR_X].title = SANE_TITLE_SCAN_BR_X;
  sod[OPT_BR_X].desc  = SANE_DESC_SCAN_BR_X;
  sod[OPT_BR_X].type  = SANE_TYPE_FIXED;
  sod[OPT_BR_X].unit  = SANE_UNIT_MM;
  sod[OPT_BR_X].size  = sizeof(SANE_Word);
  sod[OPT_BR_X].constraint_type = SANE_CONSTRAINT_RANGE;

  sod[OPT_BR_Y].name  = SANE_NAME_SCAN_BR_Y;
  sod[OPT_BR_Y].title = SANE_TITLE_SCAN_BR_Y;
  sod[OPT_BR_Y].desc  = SANE_DESC_SCAN_BR_Y;
  sod[OPT_BR_Y].type  = SANE_TYPE_FIXED;
  sod[OPT_BR_Y].unit  = SANE_UNIT_MM;
  sod[OPT_BR_Y].size  = sizeof(SANE_Word);
  sod[OPT_BR_Y].constraint_type = SANE_CONSTRAINT_RANGE;

  switch (ms->dev->info.doc_size_code) {
  case 0x00:
    sod[OPT_TL_X].constraint.range = 
      sod[OPT_BR_X].constraint.range = &geo_range85;
    sod[OPT_TL_Y].constraint.range = 
      sod[OPT_BR_Y].constraint.range = &geo_range140;
    break;
  case 0x01:
    sod[OPT_TL_X].constraint.range = 
      sod[OPT_BR_X].constraint.range = &geo_range85;
    sod[OPT_TL_Y].constraint.range =
      sod[OPT_BR_Y].constraint.range = &geo_range110;
    break;
  case 0x02:
    sod[OPT_TL_X].constraint.range = 
      sod[OPT_BR_X].constraint.range = &geo_range85;
    sod[OPT_TL_Y].constraint.range = 
      sod[OPT_BR_Y].constraint.range = &geo_range1169;
    break;
  case 0x03:
    sod[OPT_TL_X].constraint.range = 
      sod[OPT_BR_X].constraint.range = &geo_range85;
    sod[OPT_TL_Y].constraint.range = 
      sod[OPT_BR_Y].constraint.range = &geo_range130;
    break;
  case 0x04:
    sod[OPT_TL_X].constraint.range = 
      sod[OPT_BR_X].constraint.range = &geo_range80;
    sod[OPT_TL_Y].constraint.range = 
      sod[OPT_BR_Y].constraint.range = &geo_range100;
    break;
  case 0x05:
    sod[OPT_TL_X].constraint.range = 
      sod[OPT_BR_X].constraint.range = &geo_range83;
    sod[OPT_TL_Y].constraint.range = 
      sod[OPT_BR_Y].constraint.range = &geo_range140;
    break;
  case 0x80:
    sod[OPT_TL_X].constraint.range = 
      sod[OPT_BR_X].constraint.range = &geo_range35M;
    sod[OPT_TL_Y].constraint.range = 
      sod[OPT_BR_Y].constraint.range = &geo_range35M;
    break;
  case 0x81:
    sod[OPT_TL_X].constraint.range = 
      sod[OPT_BR_X].constraint.range = &geo_range50;
    sod[OPT_TL_Y].constraint.range = 
      sod[OPT_BR_Y].constraint.range = &geo_range50;
    break;
  case 0x82:
    sod[OPT_TL_X].constraint.range = 
      sod[OPT_BR_X].constraint.range = &geo_range36M;
    sod[OPT_TL_Y].constraint.range = 
      sod[OPT_BR_Y].constraint.range = &geo_range36M;
    break;
  default:
    sod[OPT_TL_X].constraint.range =
      sod[OPT_BR_X].constraint.range =
      sod[OPT_TL_Y].constraint.range =
      sod[OPT_BR_Y].constraint.range = &geo_range000;
    DBG(15, "init_options:  Unknown doc size code!  0x%x\n",
	ms->dev->info.doc_size_code);    
  }

  /* set default scan region to be maximum size */
  val[OPT_TL_X].w     = sod[OPT_TL_X].constraint.range->min;
  val[OPT_TL_Y].w     = sod[OPT_TL_Y].constraint.range->min;
  val[OPT_BR_X].w     = sod[OPT_BR_X].constraint.range->max;
  val[OPT_BR_Y].w     = sod[OPT_BR_Y].constraint.range->max;

  sod[OPT_ENHANCEMENT_GROUP].name  = "";
  sod[OPT_ENHANCEMENT_GROUP].title = "Enhancement";
  sod[OPT_ENHANCEMENT_GROUP].desc  = "";
  sod[OPT_ENHANCEMENT_GROUP].type  = SANE_TYPE_GROUP;
  sod[OPT_ENHANCEMENT_GROUP].cap   = 0;
  sod[OPT_ENHANCEMENT_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  /* XXXXXXXXXX */
  sod[OPT_EXPOSURE].name  = "exposure";
  sod[OPT_EXPOSURE].title = "Exposure";
  sod[OPT_EXPOSURE].desc  = "Analog exposure control";
  sod[OPT_EXPOSURE].type  = SANE_TYPE_INT;
  sod[OPT_EXPOSURE].unit  = SANE_UNIT_PERCENT;
  sod[OPT_EXPOSURE].size  = sizeof(SANE_Word);
  sod[OPT_EXPOSURE].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_EXPOSURE].constraint.range      = &exposure_range;
  val[OPT_EXPOSURE].w     = 0;

  sod[OPT_BRIGHTNESS].name  = SANE_NAME_BRIGHTNESS;
  sod[OPT_BRIGHTNESS].title = SANE_TITLE_BRIGHTNESS;
  sod[OPT_BRIGHTNESS].desc  = SANE_DESC_BRIGHTNESS;
  sod[OPT_BRIGHTNESS].type  = SANE_TYPE_INT;
  sod[OPT_BRIGHTNESS].unit  = SANE_UNIT_PERCENT;
  sod[OPT_BRIGHTNESS].size  = sizeof(SANE_Word);
  sod[OPT_BRIGHTNESS].cap   |= 
    ((ms->dev->info.extra_cap & MI_EXCAP_OFF_CTL) ? 0: SANE_CAP_INACTIVE);
  sod[OPT_BRIGHTNESS].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_BRIGHTNESS].constraint.range      = &brightness_range;
  val[OPT_BRIGHTNESS].w     = 0;

  sod[OPT_CONTRAST].name  = SANE_NAME_CONTRAST;
  sod[OPT_CONTRAST].title = SANE_TITLE_CONTRAST;
  sod[OPT_CONTRAST].desc  = SANE_DESC_CONTRAST;
  sod[OPT_CONTRAST].type  = SANE_TYPE_INT;
  sod[OPT_CONTRAST].unit  = SANE_UNIT_PERCENT;
  sod[OPT_CONTRAST].size  = sizeof(SANE_Word);
  sod[OPT_CONTRAST].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_CONTRAST].constraint.range      = &contrast_range;
  val[OPT_CONTRAST].w     = 0;


  sod[OPT_HIGHLIGHT].name  = SANE_NAME_WHITE_LEVEL;
  sod[OPT_HIGHLIGHT].title = SANE_TITLE_WHITE_LEVEL;
  sod[OPT_HIGHLIGHT].desc  = SANE_DESC_WHITE_LEVEL;
  sod[OPT_HIGHLIGHT].type  = SANE_TYPE_INT;
  sod[OPT_HIGHLIGHT].unit  = SANE_UNIT_NONE;
  sod[OPT_HIGHLIGHT].size  = sizeof(SANE_Word);
  sod[OPT_HIGHLIGHT].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_HIGHLIGHT].constraint.range      = &u8_range;
  val[OPT_HIGHLIGHT].w     = 255;

  sod[OPT_SHADOW].name  = SANE_NAME_BLACK_LEVEL;
  sod[OPT_SHADOW].title = SANE_TITLE_BLACK_LEVEL;
  sod[OPT_SHADOW].desc  = SANE_DESC_BLACK_LEVEL;
  sod[OPT_SHADOW].type  = SANE_TYPE_INT;
  sod[OPT_SHADOW].unit  = SANE_UNIT_NONE;
  sod[OPT_SHADOW].size  = sizeof(SANE_Word);
  sod[OPT_SHADOW].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_SHADOW].constraint.range      = &u8_range;
  val[OPT_SHADOW].w     = 0;

  if (ms->dev->info.enhance_cap & MI_ENH_CAP_SHADOW) {
    sod[OPT_HIGHLIGHT].cap |= SANE_CAP_ADVANCED;
    sod[OPT_SHADOW].cap    |= SANE_CAP_ADVANCED;
  } else {
    sod[OPT_HIGHLIGHT].cap |= SANE_CAP_INACTIVE;
    sod[OPT_SHADOW].cap    |= SANE_CAP_INACTIVE;
  } 
   
  sod[OPT_MIDTONE].name  = "midtone";
  sod[OPT_MIDTONE].title = "Midtone Level";
  sod[OPT_MIDTONE].desc  = "Midtone Level";
  sod[OPT_MIDTONE].type  = SANE_TYPE_INT;
  sod[OPT_MIDTONE].unit  = SANE_UNIT_NONE;
  sod[OPT_MIDTONE].size  = sizeof(SANE_Word);
  sod[OPT_MIDTONE].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_MIDTONE].constraint.range      = &u8_range;
  val[OPT_MIDTONE].w     = 128;
  if (ms->dev->info.enhance_cap & MI_ENH_CAP_MIDTONE) {
    sod[OPT_MIDTONE].cap   |= SANE_CAP_ADVANCED;
  } else {
    sod[OPT_MIDTONE].cap   |= SANE_CAP_INACTIVE;
  }

  if ((strcmp(M_COLOR, val[OPT_MODE].s)) &&
      (strcmp(M_GRAY, val[OPT_MODE].s))) {
    sod[OPT_HIGHLIGHT].cap |= SANE_CAP_INACTIVE;
    sod[OPT_SHADOW].cap    |= SANE_CAP_INACTIVE;
    sod[OPT_MIDTONE].cap   |= SANE_CAP_INACTIVE;
  }

  sod[OPT_HALFTONE_PATTERN].name = SANE_NAME_HALFTONE;
  sod[OPT_HALFTONE_PATTERN].title = SANE_TITLE_HALFTONE;
  sod[OPT_HALFTONE_PATTERN].desc = SANE_DESC_HALFTONE_PATTERN;
  sod[OPT_HALFTONE_PATTERN].type = SANE_TYPE_STRING;
  sod[OPT_HALFTONE_PATTERN].size = max_string_size(halftone_mode_list);
  sod[OPT_HALFTONE_PATTERN].cap  |= SANE_CAP_INACTIVE;
  sod[OPT_HALFTONE_PATTERN].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  sod[OPT_HALFTONE_PATTERN].constraint.string_list = halftone_mode_list;
  val[OPT_HALFTONE_PATTERN].s = strdup(halftone_mode_list[0]);

  sod[OPT_GAMMA_GROUP].name  = "";
  sod[OPT_GAMMA_GROUP].title = "Gamma Control";
  sod[OPT_GAMMA_GROUP].desc  = "";
  sod[OPT_GAMMA_GROUP].type  = SANE_TYPE_GROUP;
  sod[OPT_GAMMA_GROUP].cap   = 0;
  sod[OPT_GAMMA_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  sod[OPT_CUSTOM_GAMMA].name  = "gamma-mode";
  sod[OPT_CUSTOM_GAMMA].title = "Gamma Control Mode";
  sod[OPT_CUSTOM_GAMMA].desc  = "How to specify gamma correction, if at all";
  sod[OPT_CUSTOM_GAMMA].type  = SANE_TYPE_STRING;
  sod[OPT_CUSTOM_GAMMA].size  = max_string_size(gamma_mode_list);
  sod[OPT_CUSTOM_GAMMA].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  sod[OPT_CUSTOM_GAMMA].constraint.string_list = gamma_mode_list;
  val[OPT_CUSTOM_GAMMA].s = strdup(gamma_mode_list[0]);

  sod[OPT_GAMMA_BIND].name  = SANE_NAME_ANALOG_GAMMA_BIND;
  sod[OPT_GAMMA_BIND].title = SANE_TITLE_ANALOG_GAMMA_BIND;
  sod[OPT_GAMMA_BIND].desc  = SANE_DESC_ANALOG_GAMMA_BIND;
  sod[OPT_GAMMA_BIND].type  = SANE_TYPE_BOOL;
  sod[OPT_GAMMA_BIND].cap   |= SANE_CAP_INACTIVE;
  val[OPT_GAMMA_BIND].w     = SANE_TRUE;

  sod[OPT_ANALOG_GAMMA].name  = SANE_NAME_ANALOG_GAMMA;
  sod[OPT_ANALOG_GAMMA].title = SANE_TITLE_ANALOG_GAMMA;
  sod[OPT_ANALOG_GAMMA].desc  = SANE_DESC_ANALOG_GAMMA;
  sod[OPT_ANALOG_GAMMA].type  = SANE_TYPE_FIXED;
  sod[OPT_ANALOG_GAMMA].unit  = SANE_UNIT_NONE;
  sod[OPT_ANALOG_GAMMA].size  = sizeof(SANE_Word);
  sod[OPT_ANALOG_GAMMA].cap   |= SANE_CAP_INACTIVE;
  sod[OPT_ANALOG_GAMMA].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_ANALOG_GAMMA].constraint.range      = &analog_gamma_range;
  val[OPT_ANALOG_GAMMA].w     = SANE_FIX(1.0);

  sod[OPT_ANALOG_GAMMA_R].name  = SANE_NAME_ANALOG_GAMMA_R;
  sod[OPT_ANALOG_GAMMA_R].title = SANE_TITLE_ANALOG_GAMMA_R;
  sod[OPT_ANALOG_GAMMA_R].desc  = SANE_DESC_ANALOG_GAMMA_R;
  sod[OPT_ANALOG_GAMMA_R].type  = SANE_TYPE_FIXED;
  sod[OPT_ANALOG_GAMMA_R].unit  = SANE_UNIT_NONE;
  sod[OPT_ANALOG_GAMMA_R].size  = sizeof(SANE_Word);
  sod[OPT_ANALOG_GAMMA_R].cap   |= SANE_CAP_INACTIVE;
  sod[OPT_ANALOG_GAMMA_R].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_ANALOG_GAMMA_R].constraint.range      = &analog_gamma_range;
  val[OPT_ANALOG_GAMMA_R].w     = SANE_FIX(1.0);

  sod[OPT_ANALOG_GAMMA_G].name  = SANE_NAME_ANALOG_GAMMA_G;
  sod[OPT_ANALOG_GAMMA_G].title = SANE_TITLE_ANALOG_GAMMA_G;
  sod[OPT_ANALOG_GAMMA_G].desc  = SANE_DESC_ANALOG_GAMMA_G;
  sod[OPT_ANALOG_GAMMA_G].type  = SANE_TYPE_FIXED;
  sod[OPT_ANALOG_GAMMA_G].unit  = SANE_UNIT_NONE;
  sod[OPT_ANALOG_GAMMA_G].size  = sizeof(SANE_Word);
  sod[OPT_ANALOG_GAMMA_G].cap   |= SANE_CAP_INACTIVE;
  sod[OPT_ANALOG_GAMMA_G].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_ANALOG_GAMMA_G].constraint.range      = &analog_gamma_range;
  val[OPT_ANALOG_GAMMA_G].w     = SANE_FIX(1.0);

  sod[OPT_ANALOG_GAMMA_B].name  = SANE_NAME_ANALOG_GAMMA_B;
  sod[OPT_ANALOG_GAMMA_B].title = SANE_TITLE_ANALOG_GAMMA_B;
  sod[OPT_ANALOG_GAMMA_B].desc  = SANE_DESC_ANALOG_GAMMA_B;
  sod[OPT_ANALOG_GAMMA_B].type  = SANE_TYPE_FIXED;
  sod[OPT_ANALOG_GAMMA_B].unit  = SANE_UNIT_NONE;
  sod[OPT_ANALOG_GAMMA_B].size  = sizeof(SANE_Word);
  sod[OPT_ANALOG_GAMMA_B].cap   |= SANE_CAP_INACTIVE;
  sod[OPT_ANALOG_GAMMA_B].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_ANALOG_GAMMA_B].constraint.range      = &analog_gamma_range;
  val[OPT_ANALOG_GAMMA_B].w     = SANE_FIX(1.0);

  sod[OPT_GAMMA_VECTOR].name  = SANE_NAME_GAMMA_VECTOR;
  sod[OPT_GAMMA_VECTOR].title = SANE_TITLE_GAMMA_VECTOR;
  sod[OPT_GAMMA_VECTOR].desc  = SANE_DESC_GAMMA_VECTOR;
  sod[OPT_GAMMA_VECTOR].type  = SANE_TYPE_INT;
  sod[OPT_GAMMA_VECTOR].unit  = SANE_UNIT_NONE;
  sod[OPT_GAMMA_VECTOR].size  = 256 * sizeof(SANE_Word);
  sod[OPT_GAMMA_VECTOR].cap   |= SANE_CAP_INACTIVE;
  sod[OPT_GAMMA_VECTOR].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_GAMMA_VECTOR].constraint.range      = &u8_range;
  val[OPT_GAMMA_VECTOR].wa     = &(ms->gamma_table[0][0]);

  sod[OPT_GAMMA_VECTOR_R].name  = SANE_NAME_GAMMA_VECTOR_R;
  sod[OPT_GAMMA_VECTOR_R].title = SANE_TITLE_GAMMA_VECTOR_R;
  sod[OPT_GAMMA_VECTOR_R].desc  = SANE_DESC_GAMMA_VECTOR_R;
  sod[OPT_GAMMA_VECTOR_R].type  = SANE_TYPE_INT;
  sod[OPT_GAMMA_VECTOR_R].unit  = SANE_UNIT_NONE;
  sod[OPT_GAMMA_VECTOR_R].size  = 256 * sizeof(SANE_Word);
  sod[OPT_GAMMA_VECTOR_R].cap   |= SANE_CAP_INACTIVE;
  sod[OPT_GAMMA_VECTOR_R].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_GAMMA_VECTOR_R].constraint.range      = &u8_range;
  val[OPT_GAMMA_VECTOR_R].wa     = &(ms->gamma_table[1][0]);

  sod[OPT_GAMMA_VECTOR_G].name  = SANE_NAME_GAMMA_VECTOR_G;
  sod[OPT_GAMMA_VECTOR_G].title = SANE_TITLE_GAMMA_VECTOR_G;
  sod[OPT_GAMMA_VECTOR_G].desc  = SANE_DESC_GAMMA_VECTOR_G;
  sod[OPT_GAMMA_VECTOR_G].type  = SANE_TYPE_INT;
  sod[OPT_GAMMA_VECTOR_G].unit  = SANE_UNIT_NONE;
  sod[OPT_GAMMA_VECTOR_G].size  = 256 * sizeof(SANE_Word);
  sod[OPT_GAMMA_VECTOR_G].cap   |= SANE_CAP_INACTIVE;
  sod[OPT_GAMMA_VECTOR_G].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_GAMMA_VECTOR_G].constraint.range      = &u8_range;
  val[OPT_GAMMA_VECTOR_G].wa     = &(ms->gamma_table[2][0]);

  sod[OPT_GAMMA_VECTOR_B].name  = SANE_NAME_GAMMA_VECTOR_B;
  sod[OPT_GAMMA_VECTOR_B].title = SANE_TITLE_GAMMA_VECTOR_B;
  sod[OPT_GAMMA_VECTOR_B].desc  = SANE_DESC_GAMMA_VECTOR_B;
  sod[OPT_GAMMA_VECTOR_B].type  = SANE_TYPE_INT;
  sod[OPT_GAMMA_VECTOR_B].unit  = SANE_UNIT_NONE;
  sod[OPT_GAMMA_VECTOR_B].size  = 256 * sizeof(SANE_Word);
  sod[OPT_GAMMA_VECTOR_B].cap   |= SANE_CAP_INACTIVE;
  sod[OPT_GAMMA_VECTOR_B].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_GAMMA_VECTOR_B].constraint.range      = &u8_range;
  val[OPT_GAMMA_VECTOR_B].wa     = &(ms->gamma_table[3][0]);

  sod[OPT_EXP_RES].name  = "exp_res";
  sod[OPT_EXP_RES].title = "Expanded Resolution";
  sod[OPT_EXP_RES].desc  = "Enable double-resolution scans";
  sod[OPT_EXP_RES].type  = SANE_TYPE_BOOL;
  sod[OPT_EXP_RES].cap   |= SANE_CAP_ADVANCED;
  if (!(ms->dev->info.expanded_resolution))
    sod[OPT_EXP_RES].cap |= SANE_CAP_INACTIVE;
  val[OPT_EXP_RES].w     = SANE_FALSE;

  sod[OPT_FORCE_3PASS].name  = "force_3pass";
  sod[OPT_FORCE_3PASS].title = "Force 3-pass";
  sod[OPT_FORCE_3PASS].desc  = 
    "Force 3-pass color scans on single-pass scanners";
  sod[OPT_FORCE_3PASS].type  = SANE_TYPE_BOOL;
  sod[OPT_FORCE_3PASS].cap   |= SANE_CAP_ADVANCED;
  /*  if (!(ms->dev->info.modes & MI_MODES_ONEPASS) ||
      (strcmp(val[OPT_MODE].s, M_COLOR)*/ /* mode != color */ /*))*/ /* || 
      !(ms->dev->info.does_3pass)) XXXXXXX*/
    sod[OPT_FORCE_3PASS].cap   |= SANE_CAP_INACTIVE;
  val[OPT_FORCE_3PASS].w     = SANE_FALSE;

  /*
  sod[OPT_].name  = SANE_NAME_;
  sod[OPT_].title = SANE_TITLE_;
  sod[OPT_].desc  = SANE_DESC_;
  sod[OPT_].type  = SANE_TYPE_;
  sod[OPT_].unit  = SANE_UNIT_NONE;
  sod[OPT_].size  = sizeof(SANE_Word);
  sod[OPT_].cap   = 0;
  sod[OPT_].constraint_type = SANE_CONSTRAINT_NONE;
  sod[OPT_].constraint      = ;
  val[OPT_].w     = ;
  */

  DBG(15, "init_options:  done.\n");
  return SANE_STATUS_GOOD;
}


/***************************************************************************/
/* Parse an INQUIRY information block, as returned by scanner              */
/***************************************************************************/
static SANE_Status
parse_inquiry(Microtek_Info *mi, unsigned char *result)
{
  DBG(15, "parse_inquiry...\n");
  strncpy(mi->vendor_id, &result[8], 8);
  strncpy(mi->model_name, &result[16], 16);
  strncpy(mi->revision_num, &result[32], 4);
  mi->vendor_id[8]   = 0;
  mi->model_name[16] = 0;
  mi->revision_num[5] = 0;
  
  mi->device_type                = (SANE_Byte)(result[0] & 0x1f);
  mi->SCSI_firmware_ver_major    = (SANE_Byte)((result[1] & 0xf0) >> 4);
  mi->SCSI_firmware_ver_minor    = (SANE_Byte)(result[1] & 0x0f);
  mi->scanner_firmware_ver_major = (SANE_Byte)((result[2] & 0xf0) >> 4);
  mi->scanner_firmware_ver_minor = (SANE_Byte)(result[2] & 0x0f);
  mi->response_data_format       = (SANE_Byte)(result[3]);
  
  mi->res_step                   = (SANE_Byte)(result[56] & 0x03);
  mi->modes                      = (SANE_Byte)(result[57]);
  mi->pattern_count              = (SANE_Int)(result[58] & 0x7f);
  mi->pattern_dwnld              = (SANE_Byte)(result[58] & 0x80) > 0;
  mi->feed_type                  = (SANE_Byte)(result[59] & 0x0F);
  mi->compress_type              = (SANE_Byte)(result[59] & 0x30);
  mi->unit_type                  = (SANE_Byte)(result[59] & 0xC0);
  
  mi->doc_size_code              = (SANE_Byte)result[60];
  /* Our max_x,y is in inches.  Some scanners think in mm, though. */
  switch (mi->doc_size_code) {
  case 0x00:
    mi->max_x = SANE_FIX(8.5); mi->max_y = SANE_FIX(14.0); break;
  case 0x01:   
    mi->max_x = SANE_FIX(8.5); mi->max_y = SANE_FIX(11.0); break;
  case 0x02:
    mi->max_x = SANE_FIX(8.5); mi->max_y = SANE_FIX(11.69); break;
  case 0x03:
    mi->max_x = SANE_FIX(8.5); mi->max_y = SANE_FIX(13.0); break;
  case 0x04:
    mi->max_x = SANE_FIX(8.0); mi->max_y = SANE_FIX(10.0); break;
  case 0x05:
    mi->max_x = SANE_FIX(8.3); mi->max_y = SANE_FIX(14.0); break;
  case 0x80:
    /* Slide format, size is mm */
    mi->max_x = SANE_FIX(35.0 / MM_PER_INCH);
    mi->max_y = SANE_FIX(35.0 / MM_PER_INCH);  
      break;
  case 0x81:
    mi->max_x = SANE_FIX(5.0); mi->max_y = SANE_FIX(5.0); break;
  case 0x82:
    /* Slide format, size is mm */
    mi->max_x = SANE_FIX(36.0 / MM_PER_INCH);
    mi->max_y = SANE_FIX(36.0 / MM_PER_INCH); 
    break;
  default:
    /* Undefined document format code */
      mi->max_x =  SANE_FIX(0.0); mi->max_y =  SANE_FIX(0.0);
      DBG(15, "parse_inquiry:  Unknown doc_size_code!  0x%x\n",
	  mi->doc_size_code);
  }
  
  mi->cont_settings              = (SANE_Int)((result[61] & 0xf0) >> 4);
  mi->exp_settings               = (SANE_Int)(result[61] & 0x0f);
  mi->model_code                 = (SANE_Byte)(result[62]);
  switch (mi->model_code) {
  case 0x50: /* ScanMaker II/IIXE */
  case 0x54: /* ScanMaker IISP    */
  case 0x55: /* ScanMaker IIER    */
  case 0x57: /* ScanMaker IIHR    */
  case 0x58: /* ScanMaker IIG     */
  case 0x5f: /* ScanMaker E3      */
  case 0x56: /* ScanMaker A3t     */
  case 0x64: /* ScanMaker E2 (,Vobis RealScan) */
    mi->base_resolution = 300;
    break;
  case 0x59: /* ScanMaker III     */
  case 0x63: /* ScanMaker E6      */
  case 0x66: /* ScanMaker E6 (new)*/
    mi->base_resolution = 600;
    break;
  case 0x51: /* ScanMaker 45t     */
  case 0x52: /* ScanMaker 35t     */
  case 0x62: /* ScanMaker 35t+     */
    mi->base_resolution = 1950;
    break;
  default:
    mi->base_resolution = 300;
    DBG(15, "parse_inquiry:  Unknown model code!  0x%x\n",
	mi->model_code);
    break;
  }

  mi->source_options             = (SANE_Byte)(result[63]); 
  
  mi->expanded_resolution        = (result[64] & 0x01);
  /* my E6 reports incorrectly */
  if (mi->model_code == 0x66) mi->expanded_resolution = 0xFF;

  mi->enhance_cap                = (result[65] & 0x03);
  
  switch (result[66] & 0x0F) {
  case 0x00: mi->max_lookup_size =     0; break;
  case 0x01: mi->max_lookup_size =   256; break;
  case 0x03: mi->max_lookup_size =  1024; break;
  case 0x05: mi->max_lookup_size =  4096; break;
  case 0x09: mi->max_lookup_size = 65536; break;
  default: 
    mi->max_lookup_size = 0;
    DBG(15, "parse_inquiry:  Unknown gamma LUT size!  0x%x\n",
	result[66]);
  }
  
  switch (result[66] >> 5) {
  case 0x00: mi->max_gamma_val =   255;  mi->gamma_size = 1;  break;
  case 0x01: mi->max_gamma_val =  1023;  mi->gamma_size = 2;  break;
  case 0x02: mi->max_gamma_val =  4095;  mi->gamma_size = 2;  break;
  case 0x03: mi->max_gamma_val = 65535;  mi->gamma_size = 2;  break;
  default:
    mi->max_gamma_val =     0;  mi->gamma_size = 0;
    DBG(15, "parse_inquiry:  Unknown gamma max val!  0x%x\n",
	result[66]);
  }
  
  mi->fast_color_preview         = (SANE_Byte)(result[67] & 0x01);
  mi->xfer_format_select         = (SANE_Byte)(result[68] & 0x01);
  mi->color_sequence             = (SANE_Byte)(result[69] & 0x7f);
  mi->does_3pass                 = (SANE_Byte)(result[69] & 0x80);
  mi->does_mode1                 = (SANE_Byte)(result[71] & 0x01);
  
  /* XXXXXXXX */
  mi->contrast_vals              = (SANE_Int)(result[72]);
  mi->min_contrast = -42;
  mi->max_contrast =  49;
  if (mi->contrast_vals)
    mi->max_contrast = (mi->contrast_vals * 7) - 49;
  
  mi->exposure_vals              = (SANE_Int)(result[73]);
  mi->min_exposure  = -18;
  mi->max_exposure  =  21;
  if (mi->exposure_vals)
    mi->max_exposure  = (mi->exposure_vals * 3) - 21;
  
  mi->bit_formats                = (SANE_Byte)(result[74] & 0x0F);
  mi->extra_cap                  = (SANE_Byte)(result[75] & 0x07);

  /* The E2 lies... */
  if (mi->model_code == 0x64) {
    DBG(4, "parse_inquiry:  The E2 lies about it's 3-pass heritage.\n");
    mi->does_3pass = 1;
    mi->modes &= ~MI_MODES_ONEPASS;
  }
  
  return SANE_STATUS_GOOD;
}



/***************************************************************************/
/* Dump all we know about scanner to stderr                                */
/***************************************************************************/
static SANE_Status 
dump_inquiry(Microtek_Info *mi, unsigned char *result)
{
  int i;

  DBG(15, "dump_inquiry...\n");
  fprintf(stderr, "  === SANE/Microtek backend v%d.%d ===\n",
	  MICROTEK_MAJOR, MICROTEK_MINOR);
  fprintf(stderr, "========== Scanner Inquiry Block ========mm\n");
  for (i=0; i<96; i++) {
    if (!(i % 16) && (i)) fprintf(stderr, "\n");
    fprintf(stderr, "%02x ", (int)result[i]);
  }
  fprintf(stderr, "\n\n");

  fprintf(stderr, "========== Scanner Inquiry Report ==========\n");
  fprintf(stderr, "===== Scanner ID...\n");
  fprintf(stderr, "Device Type Code: 0x%02x\n", mi->device_type);
  fprintf(stderr, "Model Code: 0x%02x\n", mi->model_code);
  fprintf(stderr, "Vendor Name: '%s'   Model Name: '%s'\n",
	  mi->vendor_id, mi->model_name);
  fprintf(stderr, "Firmware Rev: '%s'\n", mi->revision_num);
  fprintf(stderr, 
	  "SCSI F/W version: %1d.%1d     Scanner F/W version: %1d.%1d\n",
	  mi->SCSI_firmware_ver_major, mi->SCSI_firmware_ver_minor,
	  mi->scanner_firmware_ver_major, mi->scanner_firmware_ver_minor);
  fprintf(stderr, "Response data format: 0x%02x\n", mi->response_data_format);
  
  fprintf(stderr, "===== Imaging Capabilities...\n");
  fprintf(stderr, "Modes:  %s%s%s%s%s%s%s\n",
	  (mi->modes & MI_MODES_LINEART) ? "Lineart " : "",
	  (mi->modes & MI_MODES_HALFTONE) ? "Halftone " : "",
	  (mi->modes & MI_MODES_GRAY) ? "Gray " : "",
	  (mi->modes & MI_MODES_COLOR) ? "Color " : "",
	  (mi->modes & MI_MODES_TRANSMSV) ? "(X-msv) " : "",
	  (mi->modes & MI_MODES_ONEPASS) ? "(OnePass) " : "",
	  (mi->modes & MI_MODES_NEGATIVE) ? "(Negative) " : "");
  fprintf(stderr, 
	  "Resolution Step Sizes: %s%s    Expanded Resolution Support? %s%s\n",
	  (mi->res_step & MI_RESSTEP_1PER) ? "1% " : "",
	  (mi->res_step & MI_RESSTEP_5PER) ? "5%" : "",
	  (mi->expanded_resolution) ? "yes" : "no",
	  (mi->expanded_resolution == 0xFF) ? "(but says no)" : "");
  fprintf(stderr, "Supported Bits Per Sample: %s8 %s%s%s\n",
	  (mi->bit_formats & MI_FMT_CAP_4BPP) ? "4 " : "",
	  (mi->bit_formats & MI_FMT_CAP_10BPP) ? "10 " : "",
	  (mi->bit_formats & MI_FMT_CAP_12BPP) ? "12 " : "",
	  (mi->bit_formats & MI_FMT_CAP_16BPP) ? "16 " : "");
  fprintf(stderr, "Max. document size code: 0x%02x\n",
	  mi->doc_size_code);
  fprintf(stderr, "Max. document size:  %f x %f inches\n",
	  SANE_UNFIX(mi->max_x), SANE_UNFIX(mi->max_y));
  fprintf(stderr, "Frame units:  %s%s\n",
	  (mi->unit_type & MI_UNIT_PIXELS) ? "pixels  " : "",
	  (mi->unit_type & MI_UNIT_8TH_INCH) ? "1/8\"'s " : "");
  fprintf(stderr, "# of built-in halftones: %d   Downloadable patterns? %s\n",
	  mi->pattern_count, (mi->pattern_dwnld) ? "Yes" : "No");

  fprintf(stderr, "Data Compression: %s%s\n",
	  (mi->compress_type & MI_COMPRSS_HUFF) ? "huffman " : "",
	  (mi->compress_type & MI_COMPRSS_RD) ? "read-data " : "");
  fprintf(stderr, "Contrast Settings: %d   Exposure Settings: %d\n",
	  mi->cont_settings, mi->exp_settings);
  fprintf(stderr, "Adjustable Shadow/Highlight? %s   Adjustable Midtone? %s\n",
	  (mi->enhance_cap & MI_ENH_CAP_SHADOW) ? "yes" : "no ",
	  (mi->enhance_cap & MI_ENH_CAP_MIDTONE) ? "yes" : "no ");
  fprintf(stderr, "Digital brightness/offset? %s\n",
	  (mi->extra_cap & MI_EXCAP_OFF_CTL) ? "yes" : "no");
  fprintf(stderr, 
	  "Gamma Table Size: %d entries of %d bytes (max. value: %d)\n",
	  mi->max_lookup_size, mi->gamma_size, mi->max_gamma_val);

  fprintf(stderr, "===== Source Options...\n");
  fprintf(stderr, "Feed type:  %s%s   ADF support? %s\n",
	  (mi->feed_type & MI_FEED_FLATBED) ? "flatbed " : "",
	  (mi->feed_type & MI_FEED_EDGEFEED) ? "edge-feed " : "",
	  (mi->feed_type & MI_FEED_AUTOSUPP) ? "yes" : "no");  
  fprintf(stderr, "Document Feeder Support? %s   Feeder Backtracking? %s\n",
	  (mi->source_options & MI_SRC_FEED_SUPP) ? "yes" : "no ",
	  (mi->source_options & MI_SRC_FEED_BT) ? "yes" : "no ");
  fprintf(stderr, "Feeder Installed? %s          Feeder Ready? %s\n",
	  (mi->source_options & MI_SRC_HAS_FEED) ? "yes" : "no ",
	  (mi->source_options & MI_SRC_FEED_RDY) ? "yes" : "no ");
  fprintf(stderr, "Transparency Adapter Installed? %s\n",
	  (mi->source_options & MI_SRC_HAS_TRANS) ? "yes" : "no ");
  /* GET_TRANS GET_FEED XXXXXXXXX */
  /* mt_SWslct ???? XXXXXXXXXXX */
  /*#define DOC_ON_FLATBED 0x00
    #define DOC_IN_FEEDER  0x01
    #define TRANSPARENCY   0x10
    */
  fprintf(stderr, "Fast Color Prescan? %s\n",
	  (mi->fast_color_preview) ? "yes" : "no");
  fprintf(stderr, "Selectable Transfer Format? %s\n",
	  (mi->xfer_format_select) ? "yes" : "no");
  fprintf(stderr, "Color Transfer Sequence: ");
  switch (mi->color_sequence) {
  case MI_COLSEQ_PLANE: 
    fprintf(stderr, "plane-by-plane (3-pass)\n"); break;
  case MI_COLSEQ_PIXEL: 
    fprintf(stderr, "pixel-by-pixel RGB\n"); break;
  case MI_COLSEQ_RGB:
    fprintf(stderr, "line-by-line, R-G-B sequence\n"); break;
  case MI_COLSEQ_NONRGB:
    fprintf(stderr, "line-by-line, non-sequential with headers\n"); break;
  default:
    fprintf(stderr, "UNKNOWN CODE (0x%02x)\n", mi->color_sequence);
  }
  if (mi->modes & MI_MODES_ONEPASS)
    fprintf(stderr, "Three pass scan support? %s\n",
	    (mi->does_3pass ? "yes" : "no"));
  fprintf(stderr, "ModeSelect-1 and ModeSense-1 Support? %s\n",
	  (mi->does_mode1) ? "yes" : "no");
  fprintf(stderr, "Can Disable Linearization Table? %s\n",
	  (mi->extra_cap & MI_EXCAP_DIS_LNTBL) ? "yes" : "no");
  fprintf(stderr, "Can Disable Start-of-Scan Recalibration? %s\n",
	  (mi->extra_cap & MI_EXCAP_DIS_RECAL) ? "yes" : "no");


  /*
    fprintf(stderr, "cntr_vals = %d, min_cntr = %d, max_cntr = %d\n",
    cntr_vals, min_cntr, max_cntr);
    fprintf(stderr, "exp_vals = %d, min_exp = %d, max_exp = %d\n",
    exp_vals, min_exp, max_exp);
    */
  fprintf(stderr, "\n\n");  
  return SANE_STATUS_GOOD;
}


/***************************************************************************/
/* Dump all we know about some unknown scanner to stderr                   */
/***************************************************************************/
static SANE_Status 
dump_suspect_inquiry(unsigned char *result)
{
  int i;
  char vendor_id[64], model_name[64], revision_num[16];
  SANE_Byte device_type, model_code;
  SANE_Byte SCSI_firmware_ver_major, SCSI_firmware_ver_minor;
  SANE_Byte scanner_firmware_ver_major, scanner_firmware_ver_minor;
  SANE_Byte response_data_format;

  DBG(15, "dump_suspect_inquiry...\n");
  fprintf(stderr, "  === SANE/Microtek backend v%d.%d ===\n",
	  MICROTEK_MAJOR, MICROTEK_MINOR);
  fprintf(stderr, "========== Scanner Inquiry Block ========mm\n");
  for (i=0; i<96; i++) {
    if (!(i % 16) && (i)) fprintf(stderr, "\n");
    fprintf(stderr, "%02x ", (int)result[i]);
  }
  fprintf(stderr, "\n\n");

  strncpy(vendor_id, &result[8], 8);
  strncpy(model_name, &result[16], 16);
  strncpy(revision_num, &result[32], 4);
  vendor_id[8]    = 0;
  model_name[16]  = 0;
  revision_num[5] = 0;  
  device_type                = (SANE_Byte)(result[0] & 0x1f);
  SCSI_firmware_ver_major    = (SANE_Byte)((result[1] & 0xf0) >> 4);
  SCSI_firmware_ver_minor    = (SANE_Byte)(result[1] & 0x0f);
  scanner_firmware_ver_major = (SANE_Byte)((result[2] & 0xf0) >> 4);
  scanner_firmware_ver_minor = (SANE_Byte)(result[2] & 0x0f);
  response_data_format       = (SANE_Byte)(result[3]);
  model_code                 = (SANE_Byte)(result[62]);

  fprintf(stderr, "========== Scanner Inquiry Report ==========\n");
  fprintf(stderr, "===== Scanner ID...\n");
  fprintf(stderr, "Device Type Code: 0x%02x\n", device_type);
  fprintf(stderr, "Model Code: 0x%02x\n", model_code);
  fprintf(stderr, "Vendor Name: '%s'   Model Name: '%s'\n",
	  vendor_id, model_name);
  fprintf(stderr, "Firmware Rev: '%s'\n", revision_num);
  fprintf(stderr, 
	  "SCSI F/W version: %1d.%1d     Scanner F/W version: %1d.%1d\n",
	  SCSI_firmware_ver_major, SCSI_firmware_ver_minor,
	  scanner_firmware_ver_major, scanner_firmware_ver_minor);
  fprintf(stderr, "Response data format: 0x%02x\n", response_data_format);
  fprintf(stderr, "\n\n");

  return SANE_STATUS_GOOD;
}


/***************************************************************************/
/* Try to determine if device is a Microtek Scanner (from INQUIRY info)    */
/***************************************************************************/
static SANE_Status
id_microtek(u_int8_t *result, char **model_string)
{
  SANE_Byte device_type, response_data_format;

  DBG(15, "id_microtek...\n");
  /* check device type first... */
  device_type = (SANE_Byte)(result[0] & 0x1f);
  if (device_type != 0x06) {
    DBG(15, "id_microtek:  not even a scanner:  dev_type = %d\n",
	device_type);
    return SANE_STATUS_INVAL;
  }
  if (!(strncmp("MICROTEK", &(result[8]), 8)) ||
      !(strncmp("MII SC31", &(result[8]), 8)) ||  /* for the IISP */
      !(strncmp("        ", &(result[8]), 8)) ) {
    switch (result[62]) {
    case 0x50 :
      *model_string = "ScanMaker II/IIXE";  break;
    case 0x51 :
      *model_string = "ScanMaker 45t";      break;
    case 0x52 :
      *model_string = "ScanMaker 35t";      break;
    case 0x54 :
      *model_string = "ScanMaker IISP";     break;
    case 0x55 :
      *model_string = "ScanMaker IIER";     break;
    case 0x56 :
      *model_string = "ScanMaker A3t";      break;
    case 0x57 :
      *model_string = "ScanMaker IIHR";     break;
    case 0x58 :
      *model_string = "ScanMaker IIG";      break;
    case 0x59 :
      *model_string = "ScanMaker III";      break;
    case 0x5f :
      *model_string = "ScanMaker E3";       break;
    case 0x62 :
      *model_string = "ScanMaker 35t+";	    break;
    case 0x63 :
    case 0x66 :
      *model_string = "ScanMaker E6";       break;
    case 0x64 : /* and "Vobis RealScan" */
      *model_string = "ScanMaker E2"; break;
    default :
      /* this might be a newer scanner, which uses the SCSI II command set. */
      /* that's unfortunate, but we'll warn the user anyway....             */
      response_data_format = (SANE_Byte)(result[3]);
      if (response_data_format == 0x02) {
	DBG(15, "id_microtek:  (uses new SCSI II command set)\n");
	if (DBG_LEVEL >= 15) {
	  fprintf(stderr, "========== Congratulations! ==========\n");
	  fprintf(stderr, "You appear to be the proud owner of a ");
	  fprintf(stderr, "brand-new Microtek scanner, which uses");
	  fprintf(stderr, "a new SCSI II command set.            ");
	  fprintf(stderr, "Unfortunately, this command set is not");
	  fprintf(stderr, "supported by this backend.            ");
	}
      }
      return SANE_STATUS_INVAL;
    }
    return SANE_STATUS_GOOD;
  }
  DBG(15, "id_microtek:  not microtek:  %d, %d, %d\n",
      strncmp("MICROTEK", &(result[8]), 8),
      strncmp("        ", &(result[8]), 8),
      result[62]);
  return SANE_STATUS_INVAL;
}



/***************************************************************************/
/* Try to attach a device as a Microtek scanner                            */
/***************************************************************************/
static SANE_Status 
attach_scanner(const char *devicename, Microtek_Device **devp)
{
  Microtek_Device *dev;
  int sfd;
  size_t size;
  unsigned char result[0x60];
  SANE_Status status;
  char *model_string;
  u_int8_t inquiry[] = { 0x12, 0, 0, 0, 0x60, 0 };

  DBG(15,"attach_scanner:  %s\n", devicename);
  /* check if device is already known... */
  for (dev = first_dev; dev; dev = dev->next) {
    if (strcmp(dev->sane.name, devicename) == 0) {
      if (devp) *devp = dev;
      return SANE_STATUS_GOOD;
    }
  }

  /* open scsi device... */
  DBG(20, "attach_scanner:  opening %s\n", devicename);
  if (sanei_scsi_open(devicename, &sfd, sense_handler, NULL) != 0) {
    DBG(20, "attach_scanner:  open failed\n");
    return SANE_STATUS_INVAL;
  }

  /* say hello... */
  DBG(20, "attach_scanner:  sending INQUIRY\n");
  size = sizeof(result);
  status = sanei_scsi_cmd(sfd, inquiry, sizeof(inquiry), result, &size);
  if (status != SANE_STATUS_GOOD || size != 0x60) {
    DBG(20, "attach_scanner:  inquiry failed (%s)\n", 
	sane_strstatus (status));
    sanei_scsi_close (sfd);
    return status;
  }

  /* wait, and close. XXXXXXXXXXXX */
  /*  status = wait_ready(sfd);*/
  sanei_scsi_close(sfd);
  if (status != SANE_STATUS_GOOD)
    return status;

  if (id_microtek(result, &model_string) != SANE_STATUS_GOOD) {
      DBG(15, "attach_scanner:  device doesn't look like a Microtek scanner.");
      if (DBG_LEVEL >= 5) dump_suspect_inquiry(result);
      return SANE_STATUS_INVAL;
  }
  
  dev=malloc(sizeof(*dev));
  if (!dev) return SANE_STATUS_NO_MEM;
  memset(dev, 0, sizeof(*dev));

  parse_inquiry(&(dev->info), result);
  if (DBG_LEVEL > 0) dump_inquiry(&(dev->info), result);

  /* initialize dev structure */
  dev->sane.name   = strdup(devicename);
  dev->sane.vendor = "Microtek";
  dev->sane.model  = strdup(model_string);
  dev->sane.type   = "flatbed scanner";

  /* link into device list... */
  ++num_devices;
  dev->next = first_dev;
  first_dev = dev;
  if (devp) *devp = dev;

  DBG(15, "attach_scanner:  happy.\n");

  return SANE_STATUS_GOOD;
}


/***************************************************************************/
/* End a scan, and clean up afterwards                                     */
/***************************************************************************/
static SANE_Status end_scan(Microtek_Scanner *s)
{
  SANE_Status status;

  DBG(15, "end_scan...\n");
  status = stop_scan(s);
  if (status != SANE_STATUS_GOOD) 
    DBG(23, "OY!");
  s->scanning = SANE_FALSE;
  /* free the buffers we malloc'ed */
  if (s->buffer != NULL) free(s->buffer);
  s->buffer = NULL;
  if (s->pre_hold != NULL) free(s->pre_hold);
  s->pre_hold = NULL;
  if (s->post_hold != NULL) free(s->post_hold);
  s->post_hold = NULL;
  /* close the SCSI device */
  sanei_scsi_close(s->sfd);
  s->sfd = -1;
  return SANE_STATUS_GOOD;
}




SANE_Status process_flat_data(Microtek_Scanner *s,
			      size_t nlines,
			      SANE_Byte *dest_buffer,
			      size_t buffsize,
			      SANE_Int *ret_length,
			      SANE_Int *ret_lines,
			      SANE_Bool multibit)
{
  /* easy... just copy data to destination buffer */
  /*         and return actual size               */
  if (s->expandedresolution) {
    int i;
    double x, j, z;
    unsigned int line, bit;
    SANE_Byte *sb, *db, byte;

    sb = s->buffer;
    db = dest_buffer;
    if (!multibit) {
      for (line=0; line<nlines; line++) {
	for (i=0; i < s->dest_pixel_bpl; i++) {
	  byte = 0;
	  for (bit=0; bit < 8; bit++) {
	    x = s->exp_aspect * ((i*8) + bit + 0.5);
	    j = floor(x - 0.5);
	    z = x - (j + 0.5);
	    /*
	      #define getbit(byte, index) (((byte)>>(index))&1)
	      */
	    byte |= 
	      ((
	       (((sb[(int) j / 8 + (s->pixel_bpl * line)])>>(7 - (int) j % 8))&1) *
	       (1.0 - z) +
	       (((sb[((int) j + 1) / 8 + (s->pixel_bpl * line)])>>
		 (7 - ((int) j + 1) % 8))&1) * z
	       ) > 0.5) << (7 - bit);
	       /*
	    byte |=
	      (((sb[(int) j / 8 + (s->pixel_bpl * line)])>>(7 - (int) j % 8))&1)
	      << (7 - bit);
	      */
	    if (line==190) 
	      DBG(23, 
		  "i,b: %d,%d  x,j,z: %f,%f,%f  by,idx: 0x%x,%d  sft: 0x%x\n",
		  i, bit, x, j, z,
		  sb[(int) j / 8 + (s->pixel_bpl * line)], ((int) j % 8),
		  byte);
	    
	  }
	  db[i + (s->dest_pixel_bpl * line)] = byte;
	}
      }
    } else { /* multibit scan */
      for (line=0; line<nlines; line++) { 
	for (i=0; i < s->dest_pixel_bpl; i++) {
	  x = s->exp_aspect * (i + 0.5);
	  j = floor(x - 0.5);
	  z = x - (j + 0.5);
	  db[i + (s->dest_pixel_bpl * line)] =
	    (int) ((double)sb[(int)j + (s->pixel_bpl * line)] * (1.0 - z) +
		   (double)sb[(int)j + 1 + (s->pixel_bpl * line)] * z);
	}
      }
    }
    *ret_length = s->dest_pixel_bpl * nlines;
    *ret_lines = nlines;

  } else {
    memcpy(dest_buffer, s->buffer, buffsize);
    *ret_length = buffsize;
    *ret_lines = nlines;
  }
  return SANE_STATUS_GOOD;
}

  
SANE_Status process_seqrgb_data(Microtek_Scanner *s, 
				size_t nlines,
				SANE_Byte *dest_buffer,
				size_t buffsize,
				SANE_Int *ret_length,
				SANE_Int *ret_lines)
{
  /* one-pass color, with line-by-line RGB data */
  SANE_Byte *r, *g, *b;
  SANE_Byte *d = dest_buffer;
  int i, line;
  SANE_Byte *buffo = s->buffer;
  
  for (line=0; line < nlines; line++) {
    r = buffo;
    g = r + s->ppl; /*linewidth;*/
    b = g + s->ppl; /*linewidth;*/
    /* will dest_buffer buff be big enough? XXXXXXXXx */
    for (i=0; i < s->ppl; i++) {
      *(d++) = *(r++);
      *(d++) = *(g++);
      *(d++) = *(b++);
    }
    buffo += s->pixel_bpl + s->header_bpl;
  }
  *ret_length = (s->pixel_bpl * nlines);
  *ret_lines = nlines;
  return SANE_STATUS_GOOD;
}


size_t process_goofyrgb_data(Microtek_Scanner *s, 
			     size_t nlines,
			     SANE_Byte *dest_buffer,
			     size_t max_dest_lines,
			     size_t buffsize,
			     SANE_Int *ret_length,
			     SANE_Int *ret_lines)
{
  /* hard... the RGB data is not by pixel, and not necessarily R-G-B! */
  int seg, i;
  SANE_Byte *d;
  SANE_Byte *buffo = s->buffer;
  int r_seg, g_seg, b_seg;
  int total, complete, delivered;
  int doffset, sss;
  
  if (s->pre_lines > 0) {
    DBG(20, "sane_read:  copy pre_hold (%d)\n", s->pre_lines);
    memcpy(dest_buffer, 
	   s->pre_hold, s->pre_lines * s->dest_pixel_bpl);
  }
  if (s->post_lines > 0) {
    if (s->post_lines <= (max_dest_lines - s->pre_lines)) {
      DBG(20, "sane_read:  copy post_hold (%d)\n", s->post_lines);
      memcpy(dest_buffer + (s->pre_lines * s->dest_pixel_bpl), 
	     s->post_hold, s->post_lines * s->dest_pixel_bpl);
    } else {
      /* This is for the exceptional (hi-res) case in which the
	 number of leading, not-full lines is greater than the 
	 size of the frontend's destination buffer.
	 It only works assuming that all the pre_lines did fit,
	 which seems to be the case.  */
      int fits = max_dest_lines - s->pre_lines;
      DBG(20, "sane_read:  big post_hold (%d)\n", s->post_lines);
      memcpy(dest_buffer + (s->pre_lines * s->dest_pixel_bpl),
	     s->post_hold, fits * s->dest_pixel_bpl);
      memmove(s->post_hold, s->post_hold + (fits * s->dest_pixel_bpl),
	      (s->post_lines - fits) * s->dest_pixel_bpl);
    }
  }
  r_seg = s->extra_r_seg;
  g_seg = s->extra_g_seg;
  b_seg = s->extra_b_seg;
  
  for (seg = 0; seg < nlines * 3; seg++) {
    buffo++;
    switch (*buffo) {
    case 'R': doffset = 0; sss = r_seg; r_seg++; break;
    case 'G': doffset = 1; sss = g_seg; g_seg++; break;
    case 'B': doffset = 2; sss = b_seg; b_seg++; break;
    default:
      DBG(18, "sane_read:  missing scanline RGB header!\n");
      end_scan(s);
      return SANE_STATUS_IO_ERROR;
    }
    
    if (sss < max_dest_lines) {
      d = dest_buffer + doffset + (sss * s->dest_pixel_bpl);
    } else {
      d = s->post_hold + doffset + ((sss - max_dest_lines) * s->dest_pixel_bpl);
      DBG(18, "sane_read:  extra:  seg-%d  doffset-%d  sss-%d\n",
	  seg, doffset, sss);
    }

    buffo++;
    if (s->expandedresolution) {
      int i;
      double x=0.0;
      double j=0.0;
      double z=0.0;
      SANE_Byte *sb, *db;
      
      sb = buffo;
      db = d;

      for (i=0; i < s->dest_ppl; i++) {
	x = s->exp_aspect * (i + 0.5);
	j = floor(x - 0.5);
	z = x - (j + 0.5);
	db[i*3] =
	  (int) ((double)sb[(int)j] * (1.0 - z) +
		 (double)sb[(int)j + 1] * z);
      }
      DBG(123, "process:  x,j,z = %f,%f,%f\n", x, j, z);
      buffo += s->ppl;
      
    } else {
      for (i=0; i < s->ppl; i++) {
	*d = *buffo;
	buffo++;
	d += 3;
      }
    }
 
  }
  
  total = MAX(r_seg, MAX(g_seg, b_seg));    /* # of lines we are holding */
  complete = MIN(r_seg, MIN(g_seg, b_seg)); /* # of complete lines only  */
  delivered = MIN(complete, max_dest_lines); /* # of lines we send back  */
  
  if (total > max_dest_lines) {
    if (max_dest_lines > complete)
      s->pre_lines = max_dest_lines - complete;
    else
      s->pre_lines = 0;
    s->post_lines = total - max_dest_lines;
  } else {
    s->pre_lines = total - complete;
    s->post_lines = 0;
  }
  
  /*
    s->pre_lines = MAX(0, MIN(max_dest_lines - complete, total - complete));
    s->post_lines = MAX(0, total - max_dest_lines);
    */
  
  s->extra_r_seg = r_seg - delivered;
  s->extra_g_seg = g_seg - delivered;
  s->extra_b_seg = b_seg - delivered;
  
  if (s->pre_lines > 0) 
    memcpy(s->pre_hold, dest_buffer + (complete * s->dest_pixel_bpl), 
	   s->pre_lines * s->dest_pixel_bpl);
  
  DBG(18, "sane_read:  extra r: %d  g: %d  b: %d  pre: %d  post: %d\n",
      s->extra_r_seg, s->extra_g_seg, s->extra_b_seg,
      s->pre_lines, s->post_lines);
  DBG(18, "sane_read:  total: %d  complete: %d  delivered: %d\n",
      total, complete, delivered);

  *ret_length = (delivered * s->dest_pixel_bpl); /*s->pixel_bpl*/
  *ret_lines = delivered;
  return SANE_STATUS_GOOD;
}




/***************************************************************************
 *
 * "Registered" SANE API Functions 
 *
 ***************************************************************************/


/***** sane_init() *****************************************/
SANE_Status
sane_init(SANE_Int *version_code, SANE_Auth_Callback authorize)
{
  char dev_name[PATH_MAX];
  size_t len;
  FILE *fp;

  DBG_INIT();

  DBG(1, "sane_init:  MICROTEK says hello! (v%d.%d)\n",
      MICROTEK_MAJOR, MICROTEK_MINOR);

  if (version_code)
    *version_code = SANE_VERSION_CODE (V_MAJOR, V_MINOR, 0);

  fp = sanei_config_open (MICROTEK_CONFIG_FILE);
  if (!fp) {
    /* default to /dev/scanner instead of insisting on config file */
    DBG(1, "sane_init:  missing config file '%s'\n",
	MICROTEK_CONFIG_FILE);
    attach_scanner("/dev/scanner", 0);
    return SANE_STATUS_GOOD;
  }
  
  while (fgets (dev_name, sizeof (dev_name), fp)) {
    DBG(23, "sane_init:  config- %s", dev_name);
    if (dev_name[0] == '#') continue;	/* ignore line comments */
    len = strlen (dev_name);
    if (dev_name[len - 1] == '\n') dev_name[--len] = '\0';
    if (!len) continue;			/* ignore empty lines */
    attach_scanner(dev_name, 0);
    }
  fclose (fp);
  return SANE_STATUS_GOOD;
}


/***** sane_get_devices **************************************/
SANE_Status
sane_get_devices(const SANE_Device ***device_list, 
		 SANE_Bool local_only)
{
  static const SANE_Device **devlist = 0;
  Microtek_Device *dev;
  int i;

  DBG(10, "sane_get_devices\n");

  /* we keep an internal copy */
  if (devlist)
    free(devlist);  /* hmm, free it if we want a new one, I guess.  YYYYY*/

  devlist = malloc((num_devices + 1) * sizeof(devlist[0]));
  if (!devlist) return SANE_STATUS_NO_MEM;

  for (i=0, dev=first_dev; i < num_devices; dev = dev->next)
    devlist[i++] = &dev->sane;
  devlist[i++] = 0;

  *device_list = devlist;
  return SANE_STATUS_GOOD;
}



/***** sane_open ***********************************************************/
SANE_Status
sane_open(SANE_String_Const devicename,
	  SANE_Handle *handle)
{
  Microtek_Scanner *scanner;
  Microtek_Device *dev;
  SANE_Status status;

  DBG(10, "sane_open\n");

  /* find device... */
  if (devicename[0]) {
    for (dev = first_dev; dev; dev = dev->next) {
      if (strcmp(dev->sane.name, devicename) == 0)
	break;
    }
    if (!dev) {  /* not in list, try manually... */
      status = attach_scanner(devicename, &dev);
      if (status != SANE_STATUS_GOOD) return status;
    }
  } else {  /* no device specified, so use first */
    dev = first_dev;
  }
  if (!dev) return SANE_STATUS_INVAL;

  /* create a scanner... */
  scanner = malloc(sizeof(*scanner));
  if (!scanner) return SANE_STATUS_NO_MEM;

  memset(scanner, 0, sizeof(*scanner));

  /* initialize scanner dependent stuff */

  scanner->unit_type = 
    (dev->info.unit_type & MI_UNIT_PIXELS) ? MS_UNIT_PIXELS : MS_UNIT_18INCH;
  /*
  scanner->unit_type = 
    (dev->info.unit_type & MI_UNIT_8TH_INCH) ? MS_UNIT_18INCH : MS_UNIT_PIXELS;
    */
  scanner->res_type =
    (dev->info.res_step & MI_RESSTEP_1PER) ? MS_RES_1PER : MS_RES_5PER;
  scanner->midtone_support = 
    (dev->info.enhance_cap & MI_ENH_CAP_MIDTONE) ? SANE_TRUE : SANE_FALSE;
  scanner->paper_length = 
    (scanner->unit_type == MS_UNIT_PIXELS) ? 
    (SANE_Int)(SANE_UNFIX(dev->info.max_y) * dev->info.base_resolution) :
    (SANE_Int)(SANE_UNFIX(dev->info.max_y) * 8);
  
  scanner->bright_r = 0;
  scanner->bright_g = 0;
  scanner->bright_b = 0;
  scanner->allow_calibrate = SANE_TRUE;
  scanner->onepass = (dev->info.modes & MI_MODES_ONEPASS);
  scanner->useADF = SANE_FALSE;
  /*  scanner->prescan = SANE_FALSE;*/
  scanner->allowbacktrack = SANE_TRUE;  /* ??? XXXXXXX */
  scanner->reversecolors = SANE_FALSE;
  scanner->fastprescan = SANE_FALSE;
  /*  scanner->filter = MS_FILT_CLEAR;*/
  scanner->bits_per_color = 8;

  /* init gamma tables */
  {
    int j;
    for (j=0; j<256; j++) {
      scanner->gamma_table[0][j] = j;
      scanner->gamma_table[1][j] = j;
      scanner->gamma_table[2][j] = j;
      scanner->gamma_table[3][j] = j;
    }
  }

  scanner->scanning = SANE_FALSE;
  scanner->this_pass = 0;
  scanner->sfd = -1;
  scanner->dev = dev;

  scanner->sense_flags = 0; /* ZZZZZZZZ */

  DBG(23, "sane_open:  gonna init opts:  ");
  /* init other stuff... XXXXXXXX */
  if ((status = init_options(scanner)) != SANE_STATUS_GOOD)
    return status;

  scanner->next = first_handle;
  first_handle = scanner;
  *handle = scanner;

  return SANE_STATUS_GOOD;
}




void
sane_close (SANE_Handle handle)
{
  Microtek_Scanner *ms = handle;

  DBG(10, "sane_close...\n");
  /* free malloc'ed stuff (strdup counts too!) */
  free((void *) ms->sod[OPT_MODE].constraint.string_list);
  free((void *) ms->sod[OPT_RESOLUTION].constraint.range);
  free((void *) ms->sod[OPT_SOURCE].constraint.string_list);
  free(ms->val[OPT_MODE].s);
  free(ms->val[OPT_HALFTONE_PATTERN].s);
  free(ms->val[OPT_SOURCE].s);
  /* remove Scanner from linked list */
  if (first_handle == ms)
    first_handle = ms->next;
  else {
    Microtek_Scanner *ts = first_handle;
    while ((ts != NULL) && (ts->next != ms)) ts = ts->next;
    ts->next = ts->next->next; /* == ms->next */
  }
  /* finally, say goodbye to the Scanner */
  free(ms);
}




/***** get_option_descriptor *************************/
const SANE_Option_Descriptor *
sane_get_option_descriptor (SANE_Handle handle,
			    SANE_Int option)
{
  Microtek_Scanner *scanner = handle;

  DBG(12, "sane_get_option_descriptor (%d)...\n", option);
  if ((unsigned)option >= NUM_OPTIONS)
    return NULL;
  return &(scanner->sod[option]);
}




/***** control_option ********************************/
SANE_Status 
sane_control_option (SANE_Handle handle,
		     SANE_Int option,
		     SANE_Action action,
		     void *value,
		     SANE_Int *info)
{
  Microtek_Scanner *scanner = handle;
  SANE_Option_Descriptor *sod;
  Microtek_Option_Value  *val;
  SANE_Status status;

  DBG(12, "sane_control_option (opt=%d,act=%d,val=%p,info=%p)\n",
      option, action, value, info);

  sod = scanner->sod;
  val = scanner->val;

  /* no changes while in mid-pass! */
  if (scanner->scanning) return SANE_STATUS_DEVICE_BUSY;
  /* and... no changes while in middle of three-pass series! */
  if (scanner->this_pass != 0) return SANE_STATUS_DEVICE_BUSY;

  if ((option >= NUM_OPTIONS) || (option < 0))
    return SANE_STATUS_INVAL;
  if (!SANE_OPTION_IS_ACTIVE(scanner->sod[option].cap))
    return SANE_STATUS_INVAL;

  if (info) *info = 0;

  /* choose by action */
  switch (action) {

  case SANE_ACTION_GET_VALUE:
    switch (option) {
      /* word options... */
    case OPT_RESOLUTION:
    case OPT_SPEED:
    case OPT_BACKTRACK:
    case OPT_NEGATIVE:
    case OPT_PREVIEW:
    case OPT_TL_X:
    case OPT_TL_Y:
    case OPT_BR_X:
    case OPT_BR_Y:
    case OPT_EXPOSURE:
    case OPT_BRIGHTNESS:
    case OPT_CONTRAST:
    case OPT_HIGHLIGHT:
    case OPT_SHADOW:
    case OPT_MIDTONE:
    case OPT_GAMMA_BIND:
    case OPT_ANALOG_GAMMA:
    case OPT_ANALOG_GAMMA_R:
    case OPT_ANALOG_GAMMA_G:
    case OPT_ANALOG_GAMMA_B:
    case OPT_EXP_RES:
    case OPT_FORCE_3PASS:
      *(SANE_Word *)value = val[option].w;
      return SANE_STATUS_GOOD;

      /* word-array options... */
      /*    case OPT_HALFTONE_PATTERN:*/
    case OPT_GAMMA_VECTOR:
    case OPT_GAMMA_VECTOR_R:
    case OPT_GAMMA_VECTOR_G:
    case OPT_GAMMA_VECTOR_B:
      memcpy(value, val[option].wa, sod[option].size);
      return SANE_STATUS_GOOD;
      
      /* string options... */
    case OPT_MODE:
    case OPT_HALFTONE_PATTERN:
    case OPT_CUSTOM_GAMMA:
    case OPT_SOURCE:
      strcpy(value, val[option].s);
      return SANE_STATUS_GOOD;

      /* others.... */
    case OPT_NUM_OPTS:
      *(SANE_Word *) value = NUM_OPTIONS;
      return SANE_STATUS_GOOD;

    default:
      return SANE_STATUS_INVAL;
    }
    break;
    
  case SANE_ACTION_SET_VALUE: {
    status = sanei_constrain_value(sod + option, value, info);
    if (status != SANE_STATUS_GOOD)
      return status;
    
    switch (option) {
      /* set word options... */
    case OPT_RESOLUTION:
    case OPT_TL_X:
    case OPT_TL_Y:
    case OPT_BR_X:
    case OPT_BR_Y:
    case OPT_FORCE_3PASS:
      if (info)
	*info |= SANE_INFO_RELOAD_PARAMS;
    case OPT_SPEED:
    case OPT_PREVIEW:
    case OPT_BACKTRACK:
    case OPT_NEGATIVE:
    case OPT_EXPOSURE:
    case OPT_BRIGHTNESS:
    case OPT_CONTRAST:
    case OPT_HIGHLIGHT:
    case OPT_SHADOW:
    case OPT_MIDTONE:
    case OPT_ANALOG_GAMMA:
    case OPT_ANALOG_GAMMA_R:
    case OPT_ANALOG_GAMMA_G:
    case OPT_ANALOG_GAMMA_B:
      val[option].w = *(SANE_Word *)value;
      return SANE_STATUS_GOOD;

    case OPT_EXP_RES:
      if (val[option].w != *(SANE_Word *) value) {
	val[option].w = *(SANE_Word *)value;
	if (info) *info |= SANE_INFO_RELOAD_OPTIONS | SANE_INFO_RELOAD_PARAMS;
	if (val[OPT_EXP_RES].w) {
	  sod[OPT_RESOLUTION].constraint.range = &(scanner->exp_res_range);
	  val[OPT_RESOLUTION].w *= 2;
	} else {
	  sod[OPT_RESOLUTION].constraint.range = &(scanner->res_range);
	  val[OPT_RESOLUTION].w /= 2;
	}
      }
      return SANE_STATUS_GOOD;

    case OPT_GAMMA_BIND:
    case OPT_CUSTOM_GAMMA:
      if (option == OPT_GAMMA_BIND) {
	if (val[option].w != *(SANE_Word *) value)
	  if (info) *info |= SANE_INFO_RELOAD_OPTIONS;
	val[option].w = *(SANE_Word *) value;
      } else if (option == OPT_CUSTOM_GAMMA) {
	if (val[option].s) {
	  if (strcmp(value, val[option].s)) 
	    if (info) *info |= SANE_INFO_RELOAD_OPTIONS;
	  free(val[option].s);
	}
	val[option].s = strdup(value);
      }
      if ( !(strcmp(val[OPT_CUSTOM_GAMMA].s, M_NONE)) ||
	   !(strcmp(val[OPT_CUSTOM_GAMMA].s, M_SCALAR)) ) {
	sod[OPT_GAMMA_VECTOR].cap |= SANE_CAP_INACTIVE;
	sod[OPT_GAMMA_VECTOR_R].cap |= SANE_CAP_INACTIVE;
	sod[OPT_GAMMA_VECTOR_G].cap |= SANE_CAP_INACTIVE;
	sod[OPT_GAMMA_VECTOR_B].cap |= SANE_CAP_INACTIVE;
      } 
      if ( !(strcmp(val[OPT_CUSTOM_GAMMA].s, M_NONE)) ||
	   !(strcmp(val[OPT_CUSTOM_GAMMA].s, M_TABLE)) ) {
	sod[OPT_ANALOG_GAMMA].cap |= SANE_CAP_INACTIVE;
	sod[OPT_ANALOG_GAMMA_R].cap |= SANE_CAP_INACTIVE;
	sod[OPT_ANALOG_GAMMA_G].cap |= SANE_CAP_INACTIVE;
	sod[OPT_ANALOG_GAMMA_B].cap |= SANE_CAP_INACTIVE;
      }
      if (!(strcmp(val[OPT_CUSTOM_GAMMA].s, M_SCALAR))) {
	if (val[OPT_GAMMA_BIND].w == SANE_TRUE) {
	  sod[OPT_ANALOG_GAMMA].cap &= ~SANE_CAP_INACTIVE;
	  sod[OPT_ANALOG_GAMMA_R].cap |= SANE_CAP_INACTIVE;
	  sod[OPT_ANALOG_GAMMA_G].cap |= SANE_CAP_INACTIVE;
	  sod[OPT_ANALOG_GAMMA_B].cap |= SANE_CAP_INACTIVE;
	} else {
	  sod[OPT_ANALOG_GAMMA].cap |= SANE_CAP_INACTIVE;
	  sod[OPT_ANALOG_GAMMA_R].cap &= ~SANE_CAP_INACTIVE;
	  sod[OPT_ANALOG_GAMMA_G].cap &= ~SANE_CAP_INACTIVE;
	  sod[OPT_ANALOG_GAMMA_B].cap &= ~SANE_CAP_INACTIVE;
	}
      }
      if (!(strcmp(val[OPT_CUSTOM_GAMMA].s, M_TABLE))) {
	if (val[OPT_GAMMA_BIND].w == SANE_TRUE) {
	  sod[OPT_GAMMA_VECTOR].cap &= ~SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_R].cap |= SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_G].cap |= SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_B].cap |= SANE_CAP_INACTIVE;
	} else {
	  sod[OPT_GAMMA_VECTOR].cap |= SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_R].cap &= ~SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_G].cap &= ~SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_B].cap &= ~SANE_CAP_INACTIVE;
	}
      }
      if (!(strcmp(val[OPT_CUSTOM_GAMMA].s, M_NONE)))
	sod[OPT_GAMMA_BIND].cap |= SANE_CAP_INACTIVE;
      else
	sod[OPT_GAMMA_BIND].cap &= ~SANE_CAP_INACTIVE;
      return SANE_STATUS_GOOD;
      

    case OPT_MODE:  
      if (val[option].s) {
	if (strcmp(val[option].s, value))
	  if (info) 
	    *info |= SANE_INFO_RELOAD_OPTIONS | SANE_INFO_RELOAD_PARAMS;
	free(val[option].s);
      }
      val[option].s = strdup(value);
      /* disable highlight/etc if not color or gray mode */
      if (strcmp(val[option].s, M_COLOR) &&
	  strcmp(val[option].s, M_GRAY)) {
	DBG(23, "sane_control_option:  hi dis\n");
	sod[OPT_HIGHLIGHT].cap |= SANE_CAP_INACTIVE;
	sod[OPT_SHADOW].cap |= SANE_CAP_INACTIVE;
	sod[OPT_MIDTONE].cap |= SANE_CAP_INACTIVE;
      } else {
	DBG(23, "sane_control_option:  hi en\n");
	sod[OPT_HIGHLIGHT].cap &= ~SANE_CAP_INACTIVE;
	sod[OPT_SHADOW].cap &= ~SANE_CAP_INACTIVE;
	sod[OPT_MIDTONE].cap &= ~SANE_CAP_INACTIVE;
      }
      if (strcmp(val[option].s, M_HALFTONE)) {
	sod[OPT_HALFTONE_PATTERN].cap |= SANE_CAP_INACTIVE;
      } else {
	sod[OPT_HALFTONE_PATTERN].cap &= ~SANE_CAP_INACTIVE;
      }
      if (strcmp(val[option].s, M_COLOR)) { /* not color */
	sod[OPT_GAMMA_BIND].cap |= SANE_CAP_INACTIVE;
        val[OPT_GAMMA_BIND].w = SANE_TRUE;
	sod[OPT_FORCE_3PASS].cap |= SANE_CAP_INACTIVE;
      } else {
	sod[OPT_GAMMA_BIND].cap &= ~SANE_CAP_INACTIVE;
	/*	if (scanner->dev->info.modes & MI_MODES_ONEPASS)
	  sod[OPT_FORCE_3PASS].cap &= ~SANE_CAP_INACTIVE;*/
      }
      return SANE_STATUS_GOOD;

    case OPT_HALFTONE_PATTERN:
    case OPT_SOURCE:
      if (val[option].s) free(val[option].s);
      val[option].s = strdup(value);
      return SANE_STATUS_GOOD;

    case OPT_GAMMA_VECTOR:
    case OPT_GAMMA_VECTOR_R:
    case OPT_GAMMA_VECTOR_G:
    case OPT_GAMMA_VECTOR_B:
      memcpy(val[option].wa, value, sod[option].size);
      return SANE_STATUS_GOOD;

    default:
      return SANE_STATUS_INVAL;
    }
  }
  break;
  
  case SANE_ACTION_SET_AUTO:
    return SANE_STATUS_UNSUPPORTED;	/* We are DUMB */
  }
  return SANE_STATUS_GOOD;
}




SANE_Status
sane_get_parameters (SANE_Handle handle,
		     SANE_Parameters *params)
{
  Microtek_Scanner *s = handle;

  DBG(23, "sane_get_parameters...\n");

  if (!s->scanning) {

    /* decipher scan mode */
    if (!(strcmp(s->val[OPT_MODE].s, M_LINEART)))
      s->mode = MS_MODE_LINEART;
    else if (!(strcmp(s->val[OPT_MODE].s, M_HALFTONE)))
      s->mode = MS_MODE_HALFTONE;
    else if (!(strcmp(s->val[OPT_MODE].s, M_GRAY)))
      s->mode = MS_MODE_GRAY;
    else if (!(strcmp(s->val[OPT_MODE].s, M_COLOR)))
      s->mode = MS_MODE_COLOR;

    if (s->mode == MS_MODE_COLOR) {
      if (s->onepass) {
	if (s->val[OPT_FORCE_3PASS].w) {
	  DBG(23, "sane_get_parameters:  forcing 3-pass color\n");
	  s->threepasscolor = SANE_TRUE;
	  s->onepasscolor = SANE_FALSE;
	  s->color_seq = MI_COLSEQ_PLANE; /* XXXXXXX */
	} else { /* regular one-pass */
	  DBG(23, "sane_get_parameters:  regular 1-pass color\n");
	  s->threepasscolor = SANE_FALSE;
	  s->onepasscolor = SANE_TRUE;
	  s->color_seq = s->dev->info.color_sequence;
	}
      } else { /* 3-pass scanner */
	DBG(23, "sane_get_parameters:  regular 3-pass color\n");
	s->threepasscolor = SANE_TRUE;
	s->onepasscolor = SANE_FALSE;
	s->color_seq = s->dev->info.color_sequence;
      } 
    } else { /* not color! */
	DBG(23, "sane_get_parameters:  non-color\n");
	s->threepasscolor = SANE_FALSE;
	s->onepasscolor = SANE_FALSE;
	s->color_seq = s->dev->info.color_sequence;
    }

    s->transparency = !(strcmp(s->val[OPT_SOURCE].s, M_TRANS));
    /* disallow exp. res. during preview scan XXXXXXXXXXX */
    /*s->expandedresolution = 
      (s->val[OPT_EXP_RES].w) && !(s->val[OPT_PREVIEW].w);*/
    s->expandedresolution = 
      (s->val[OPT_EXP_RES].w);

    if (s->res_type == MS_RES_1PER) {
      s->resolution = (SANE_Int)(SANE_UNFIX(s->val[OPT_RESOLUTION].w));
      s->resolution_code = 
	0xFF & ((s->resolution * 100) / 
		s->dev->info.base_resolution /
		(s->expandedresolution ? 2 : 1));
      DBG(23, "sane_get_parameters:  res_code = %d (%2x)\n", 
	      s->resolution_code, s->resolution_code);
    } else {
      DBG(23, "sane_get_parameters:  5 percent!!!\n");
      /* XXXXXXXXXXXXX */
    }

    s->prescan = s->val[OPT_PREVIEW].w;
    /* What are the units/ranges on exposure and contrast? YYYY*/
    s->exposure = (s->val[OPT_EXPOSURE].w / 3) + 7;
    s->contrast = (s->val[OPT_CONTRAST].w / 7) + 7;
    s->velocity  = s->val[OPT_SPEED].w;
    s->shadow    = s->val[OPT_SHADOW].w;
    s->highlight = s->val[OPT_HIGHLIGHT].w;
    s->midtone   = s->val[OPT_MIDTONE].w;
    if (SANE_OPTION_IS_ACTIVE(s->sod[OPT_BRIGHTNESS].cap)) {
      /*if (s->val[OPT_BRIGHTNESS].w >= 0) */
      s->bright_r = (SANE_Byte) (s->val[OPT_BRIGHTNESS].w);
	/*else
	s->bright_r = (SANE_Byte) (0x80 | (- s->val[OPT_BRIGHTNESS].w));*/
      s->bright_g = s->bright_b = s->bright_r;
    } else {
      s->bright_r = s->bright_g = s->bright_b = 0;
    }
    /* figure out halftone pattern selection... */
    if (s->mode == MS_MODE_HALFTONE) {
      int i = 0;
      while ((halftone_mode_list[i] != NULL) &&
	     (strcmp(halftone_mode_list[i], s->val[OPT_HALFTONE_PATTERN].s)))
	i++;
      s->pattern = ((i < s->dev->info.pattern_count) ? i : 0);
    } else
      s->pattern = 0;
      
    {
      /* need to 'round' things properly!  XXXXXXXX */
      double widthmm, heightmm;
      double dots_per_mm = s->resolution / MM_PER_INCH;
      double units_per_mm = 
      	(s->unit_type == MS_UNIT_18INCH) ? 
	(8.0 / MM_PER_INCH) :                       /* 1/8 inches */
	(s->dev->info.base_resolution / MM_PER_INCH);   /* pixels     */
      
      DBG(23, "sane_get_parameters:  dots_per_mm:  %f\n", dots_per_mm);
      DBG(23, "sane_get_parameters:  units_per_mm:  %f\n", units_per_mm);

      /* calculate frame coordinates...
       *  scanner coords are in 'units' -- pixels or 1/8"
       *  option coords are MM
       */
      s->x1 = (SANE_Int)(SANE_UNFIX(s->val[OPT_TL_X].w) * units_per_mm + 0.5);
      s->y1 = (SANE_Int)(SANE_UNFIX(s->val[OPT_TL_Y].w) * units_per_mm + 0.5);
      s->x2 = (SANE_Int)(SANE_UNFIX(s->val[OPT_BR_X].w) * units_per_mm + 0.5);
      s->y2 = (SANE_Int)(SANE_UNFIX(s->val[OPT_BR_Y].w) * units_per_mm + 0.5);

      /* these are just an estimate...
       * real values come from scanner after sane_start.
       */
      widthmm = SANE_UNFIX(s->val[OPT_BR_X].w - s->val[OPT_TL_X].w);
      heightmm = SANE_UNFIX(s->val[OPT_BR_Y].w - s->val[OPT_TL_Y].w);
      widthmm = (widthmm < 0.0) ? (- widthmm) : widthmm;
      heightmm = (heightmm < 0.0) ? (- heightmm) : heightmm;
      s->params.pixels_per_line = (SANE_Int)(widthmm * dots_per_mm + 0.5);
      s->params.lines = (SANE_Int)(heightmm * dots_per_mm + 0.5);
    }

    DBG(23, "sane_get_parameters:  lines:  %d\n", s->params.lines);

    switch (s->mode) {
    case MS_MODE_LINEART:
    case MS_MODE_HALFTONE:
      s->multibit = SANE_FALSE;
      s->params.format = SANE_FRAME_GRAY;
      s->params.bytes_per_line = (s->params.pixels_per_line + 7) / 8;
      s->params.depth = 1;
      s->filter = MS_FILT_CLEAR;
      break;
    case MS_MODE_GRAY:
      s->multibit = SANE_TRUE;
      s->params.format = SANE_FRAME_GRAY;
      s->params.bytes_per_line = s->params.pixels_per_line;
      s->params.depth = s->bits_per_color;
      s->filter = MS_FILT_CLEAR;
      break;
    case MS_MODE_COLOR:
      s->multibit = SANE_TRUE;
      if (s->onepasscolor) { /* a single-pass color scan */
	s->params.format = SANE_FRAME_RGB;
	s->params.bytes_per_line = s->params.pixels_per_line * 3;
	s->params.depth = s->bits_per_color;
   	s->filter = MS_FILT_CLEAR;
      } else { /* a three-pass color scan */
	s->params.bytes_per_line = s->params.pixels_per_line;
	s->params.depth = s->bits_per_color;
	/* this will be correctly set in sane_start */
	s->params.format = SANE_FRAME_RED; 
      }
      break;
    }

    /* also fixed in sane_start for multi-pass scans */
    s->params.last_frame = SANE_TRUE;  /* ?? XXXXXXXX */
  }

  if (params)
    *params = s->params;

  return SANE_STATUS_GOOD;
}




/*********************************************************************/

SANE_Status
sane_start (SANE_Handle handle)
{
  Microtek_Scanner *s = handle;
  SANE_Status status;
  SANE_Int busy, linewidth;
  
  DBG(10, "sane_start...\n");

  if (s->sfd < 0) {

    status = sane_get_parameters(s, 0);
    if (status != SANE_STATUS_GOOD) return status;
    
    /* set parameters that vary between multi passes */
    if (s->threepasscolor) {
      s->this_pass += 1;
      DBG(23, "sane_start:  three-pass, on %d\n", s->this_pass);
      switch (s->this_pass) {
      case 1:
	s->filter = MS_FILT_RED;
	s->params.format = SANE_FRAME_RED; 
	s->params.last_frame = SANE_FALSE;
	break;
      case 2:
	s->filter = MS_FILT_RED;
	s->params.format = SANE_FRAME_GREEN;
	s->params.last_frame = SANE_FALSE;
	break;
      case 3: 
	s->filter = MS_FILT_RED;
	s->params.format = SANE_FRAME_BLUE;
	s->params.last_frame = SANE_TRUE;
	break;
      default:
	s->filter = MS_FILT_CLEAR;
	DBG(23, "sane_start:  pass %d = filter?\n", s->this_pass);
	break;
      }
    } else
      s->this_pass = 0;
      
    status = sanei_scsi_open(s->dev->sane.name,
			     &(s->sfd),
			     sense_handler,
			     &(s->sense_flags));
    if (status != SANE_STATUS_GOOD) {
      DBG(10, "sane_start: open of %s failed: %s\n",
	  s->dev->sane.name, sane_strstatus (status));
      return status;
    }

    if ((status = wait_ready(s)) != SANE_STATUS_GOOD) return status;
    if (s->dev->info.does_mode1) {
      if ((status = mode_select_1(s)) != SANE_STATUS_GOOD) return status;
      if ((status = mode_sense_1(s))  != SANE_STATUS_GOOD) return status;
    }
    if (s->dev->info.source_options & (MI_SRC_FEED_BT | MI_SRC_HAS_TRANS)) {
      if ((status = accessory(s)) != SANE_STATUS_GOOD) return status;
      /* if SWslct ????  XXXXXXXXXXXXXXX */
    }
    if ((status = mode_sense(s))     != SANE_STATUS_GOOD) return status;
    if ((status = scanning_frame(s)) != SANE_STATUS_GOOD) return status;
    if ((status = download_gamma(s)) != SANE_STATUS_GOOD) return status;
    if ((status = mode_select(s))    != SANE_STATUS_GOOD) return status;
    if ((status = mode_sense(s))     != SANE_STATUS_GOOD) return status;
if ((status = wait_ready(s)) != SANE_STATUS_GOOD) return status;
    if ((status = start_scan(s))     != SANE_STATUS_GOOD) return status;

    s->sense_flags |= MS_SENSE_IGNORE; /* ignore vapid CMD ERROR */
    if ((status = wait_ready(s)) != SANE_STATUS_GOOD) return status;
    s->sense_flags &= ~MS_SENSE_IGNORE;

    if ((status = get_scan_status(s, &busy, 
				  &linewidth, &(s->undelivered_lines))) !=
	SANE_STATUS_GOOD) {
      DBG(10, "sane_start:  get_scan_status fails\n");
      end_scan(s);
      return status;
    }
 s->sense_flags |= MS_SENSE_IGNORE; /* ignore vapid CMD ERROR */
 if ((status = wait_ready(s)) != SANE_STATUS_GOOD) return status;
 s->sense_flags &= ~MS_SENSE_IGNORE;
    /* check for a bizarre linecount */
    if ((s->undelivered_lines < 0) || 
	(s->undelivered_lines > 
	 (s->params.lines * 2 * (s->expandedresolution ? 2 : 1)))) {
      DBG(10, "sane_start:  get_scan_status returns weird line count %d\n",
	  s->undelivered_lines);
      end_scan(s);
      return SANE_STATUS_DEVICE_BUSY;
    }

    /*    s->undelivered_lines = s->lines;*/

    /* figure out image format parameters */
    switch (s->mode) {
    case MS_MODE_LINEART:
    case MS_MODE_HALFTONE:
      s->pixel_bpl = linewidth;
      s->header_bpl = 0;
      s->ppl = linewidth * 8;
      s->planes = 1;
      s->line_format = MS_LNFMT_FLAT;
      break;
    case MS_MODE_GRAY:
      if (s->bits_per_color < 8) {
	s->pixel_bpl = linewidth;
	s->ppl = linewidth * (8 / s->bits_per_color);
      }	else {
	s->pixel_bpl = linewidth * ((s->bits_per_color + 7) / 8);
	s->ppl = linewidth;
      }
      s->header_bpl = 0;
      s->planes = 1;
      s->line_format = MS_LNFMT_FLAT;
      break;
    case MS_MODE_COLOR:
      /*      switch (s->dev->info.color_sequence) {*/
      switch (s->color_seq) {
      case MI_COLSEQ_PLANE:
	s->pixel_bpl = linewidth * ((s->bits_per_color + 7) / 8);
	s->ppl = linewidth;
	s->header_bpl = 0;
	s->planes = 1;
	s->line_format = MS_LNFMT_FLAT;
	break;
      case MI_COLSEQ_NONRGB:
	s->pixel_bpl = (linewidth - 2) * 3 * ((s->bits_per_color + 7) / 8);
	s->ppl = linewidth - 2;
	s->header_bpl = 2 * 3;
	s->planes = 3;
	s->line_format = MS_LNFMT_GOOFY_RGB;
	break;
      case MI_COLSEQ_PIXEL:
	s->pixel_bpl = linewidth * 3 * ((s->bits_per_color + 7) / 8);
	s->ppl = linewidth;
	s->header_bpl = 0;
	s->planes = 3;
	s->line_format = MS_LNFMT_FLAT;
	break;
      case MI_COLSEQ_RGB:
	s->pixel_bpl = linewidth * 3 * ((s->bits_per_color + 7) / 8);
	s->ppl = linewidth;
	s->header_bpl = 0;
	s->planes = 3;
	s->line_format = MS_LNFMT_SEQ_RGB;
	break;
      default:
	DBG(10, "sane_start:  Unknown color_sequence: %d\n",
	    s->dev->info.color_sequence);
	end_scan(s);
	return SANE_STATUS_INVAL;
      }
      break;
    default:
      DBG(10, "sane_start:  Unknown scan mode: %d\n", s->mode);
      end_scan(s);
      return SANE_STATUS_INVAL;
    }

    if (s->expandedresolution) {
      if (s->resolution > s->dev->info.base_resolution)
	s->exp_aspect = 
	  (double) s->dev->info.base_resolution / (double) s->resolution;
      else
	s->exp_aspect = 1.0;
      /* s->dest_pixel_bpl = s->pixel_bpl / s->exp_aspect;
      s->dest_ppl = s->ppl / s->exp_aspect;*/
      s->dest_ppl = s->ppl / s->exp_aspect;
      s->dest_pixel_bpl = (int) ceil((double)s->dest_ppl * (double)s->pixel_bpl / (double)s->ppl);
    } else {
      s->exp_aspect = 1.0;
      s->dest_pixel_bpl = s->pixel_bpl;
      s->dest_ppl = s->ppl;
    }
      
    s->params.lines = s->undelivered_lines;
    s->params.pixels_per_line = s->dest_ppl; /*s->ppl;*/
    s->params.bytes_per_line = s->dest_pixel_bpl; /*s->pixel_bpl;*/

    /* calculate maximum line capacity of SCSI buffer */
    s->max_scsi_lines = SCSI_BUFF_SIZE / (s->pixel_bpl + s->header_bpl);
    if (s->max_scsi_lines < 1) {
      DBG(10, "sane_start:  SCSI buffer smaller that one scan line!\n");
      end_scan(s);
      return SANE_STATUS_NO_MEM;
    }

    s->buffer = (u_int8_t *) malloc(SCSI_BUFF_SIZE * sizeof(u_int8_t));
    if (s->buffer == NULL) return SANE_STATUS_NO_MEM;

    /* ZZZZZZ throw dest_ stuff in here!!!!! */

    if (s->line_format == MS_LNFMT_GOOFY_RGB) {
      /* use some empirical heuristics to make up buffer sizes */
      /* ...we need to be able to hold all the rollover lines */
      int spacing = s->resolution / 37;  /* 37 -- the magic number */
      int holdsize = MAX(SCSI_BUFF_SIZE, spacing * s->dest_pixel_bpl);
      DBG(23, "sane_start:  spacing = %d,  holdsize = %d\n",
	  spacing, holdsize);

      s->pre_hold = (u_int8_t *) malloc(holdsize * sizeof(u_int8_t));
      if (s->pre_hold == NULL) {
	end_scan(s);
	return SANE_STATUS_NO_MEM;
      }
      s->post_hold = (u_int8_t *) malloc(holdsize * sizeof(u_int8_t));
      if (s->post_hold == NULL)	{
	end_scan(s);
	return SANE_STATUS_NO_MEM;
      }
      s->pre_lines = 0;
      s->post_lines = 0;
      s->extra_r_seg = 0;
      s->extra_g_seg = 0;
      s->extra_b_seg = 0;
    } else {
      s->pre_hold = NULL;
      s->post_hold = NULL;
    }

    s->scanning = SANE_TRUE;
    s->cancel = SANE_FALSE;

    DBG(23, "Scan Param:\n");
    DBG(23, "pix bpl: %d    hdr bpl: %d   ppl: %d\n",
	s->pixel_bpl, s->header_bpl, s->ppl);
    DBG(23, "lines: %d   planes: %d\n",
	s->undelivered_lines, s->planes);
    DBG(23, "dest bpl: %d   dest ppl: %d  aspect: %f\n",
	s->dest_pixel_bpl, s->dest_ppl, s->exp_aspect);

  } else { /* if s->sfd is already assigned... */
    return SANE_STATUS_INVAL;
  }

  return SANE_STATUS_GOOD;
}



SANE_Status
sane_read (SANE_Handle handle, SANE_Byte *dest_buffer,
	   SANE_Int dest_length, SANE_Int *ret_length)
{
  Microtek_Scanner *s = handle;
  SANE_Status status;
  SANE_Int busy, linewidth, remaining;
  int nlines;
  size_t buffsize;
  int max_dest_lines; /*, max_scsi_lines;*/
  SANE_Int ret_lines;

  DBG(10, "sane_read...\n");

  if (!(s->scanning))
    return SANE_STATUS_INVAL;

  if ((s->undelivered_lines <= 0) || (s->cancel)) {
    /* we are done scanning here */
    end_scan(s);
    if (s->this_pass == 3) s->this_pass = 0;
    if (s->cancel) {
      s->this_pass = 0; /* reset multi-pass counter */
      return SANE_STATUS_CANCELLED;
    } else
      return SANE_STATUS_EOF;
  }

  do { /* while (ret_lines <= 0) -- we need to return *something* */
 s->sense_flags |= MS_SENSE_IGNORE; /* ignore vapid CMD ERROR */
    if ((status = wait_ready(s)) != SANE_STATUS_GOOD) {
      DBG(18, "sane_read:  bad wait_ready!\n");
      end_scan(s);
      return status;
    }
 s->sense_flags &= ~MS_SENSE_IGNORE; 
    status = get_scan_status(s, &busy, &linewidth, &remaining);
    if (status != SANE_STATUS_GOOD) {
      DBG(18, "sane_read:  bad get_scan_status!\n");
      end_scan(s);
      return SANE_STATUS_IO_ERROR;
    }
    DBG(18, "sane_read: gss busy, linewidth, remaining:  %d, %d, %d\n",
	busy, linewidth, remaining);
    
    /* calculate maximum line capacity of destination buffer */
    max_dest_lines = dest_length / s->dest_pixel_bpl; /*s->pixel_bpl; */
    /* calculate maximum line capacity of SCSI buffer */
    /*max_scsi_lines = SCSI_BUFF_SIZE / (s->pixel_bpl + s->header_bpl);*/
    /* lines to get is minimum of max buffer sizes and remaining line count */
    nlines = MIN(remaining, MIN(s->max_scsi_lines, max_dest_lines));
    DBG(18, "sane_read:  max_dest: %d, max_scsi: %d, rem: %d, nlines: %d\n",
	max_dest_lines, s->max_scsi_lines, remaining, nlines);
    
    /* grab them bytes! (only if the scanner still has bytes to give...) */
    if (nlines > 0) {
      buffsize = nlines * (s->pixel_bpl + s->header_bpl); /* == "* linewidth" */
      status = read_scan_data(s, nlines, s->buffer, &buffsize);
      if (status != SANE_STATUS_GOOD) {
	DBG(18, "sane_read:  bad read_scan_data!\n");
	end_scan(s);
	return SANE_STATUS_IO_ERROR;
      }
      DBG(18, "sane_read:  dest_length: %d,  buffsize: %lu\n",
	  dest_length, (u_long) buffsize);
    } else {
      buffsize = 0;
    }
    
    switch (s->line_format) {
    case MS_LNFMT_FLAT:
      process_flat_data(s, nlines, dest_buffer, buffsize,
			ret_length, &ret_lines, s->multibit);
      break;
    case MS_LNFMT_SEQ_RGB: 
      process_seqrgb_data(s, nlines, dest_buffer, buffsize,
			  ret_length, &ret_lines);
      break;
    case MS_LNFMT_GOOFY_RGB:
      process_goofyrgb_data(s, nlines, dest_buffer, max_dest_lines, buffsize,
			    ret_length, &ret_lines);
      break;
    }
    
  } while (ret_lines <= 0);
  
  s->undelivered_lines -= ret_lines;
  
  return SANE_STATUS_GOOD;
}



/***** sane_exit() *******************************************/
void
sane_exit (void)
{
  Microtek_Device *next;

  DBG(10, "sane_exit...\n");
  /* close all leftover Scanners */
  /*(beware of how sane_close interacts with linked list) */
  while (first_handle != NULL)
    sane_close(first_handle);
  /* free up device list */
  while (first_dev != NULL) {
    next = first_dev->next;
    free((void *) first_dev->sane.name); 
    free((void *) first_dev->sane.model);
    free(first_dev);
    first_dev = next;
  }
  DBG(10, "sane_exit:  MICROTEK says goodbye.\n");
}





/* XXXXXXXXX */
void
sane_cancel (SANE_Handle handle)
{
  Microtek_Scanner *ms = handle;
  DBG(10, "sane_cancel...\n");
  ms->cancel = SANE_TRUE;
}



SANE_Status
sane_set_io_mode (SANE_Handle handle, SANE_Bool non_blocking)
{
  DBG(10, "sane_set_io_mode...\n");
  return SANE_STATUS_UNSUPPORTED;
}


SANE_Status
sane_get_select_fd (SANE_Handle handle, SANE_Int * fd)
{
  DBG(10, "sane_get_select_fd...\n");
  return SANE_STATUS_UNSUPPORTED;
}
