// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
extern "C" {
#include <stdio.h>
#include <stdlib.h>
}
#include "StyleParser.h"

StyleParser::StyleParser( StyleSheet* styleSheet )
        : _styleSheet( styleSheet ),
          _style( 0 ),
          _displayProto( 0 ),
          _state( Selector )
{
    StyleLexer* lexer = new StyleLexer;
    connect( this , SIGNAL( fwdData( const char*, int ) )        , lexer, SLOT( data( const char*, int ) ) );
    connect( this , SIGNAL( fwdEndOfData() )                     , lexer, SLOT( endOfData() ) );
    connect( lexer, SIGNAL( token( StyleLexer::Token, QString ) ), this , SLOT( token( StyleLexer::Token, QString ) ) );
    connect( lexer, SIGNAL( done() )                             , this , SLOT( done() ) );
}

StyleParser::~StyleParser()
{
}

void StyleParser::data( const char* bytes, int length )
{
    emit fwdData( bytes, length );
}

void StyleParser::endOfData()
{
    emit fwdEndOfData();
}

void StyleParser::token( StyleLexer::Token token, QString text )
{
    switch ( _state ) {
        case Selector:
            stateSelector( token, text );
            break;

        case SelectorAttrs:
            stateSelectorAttrs( token, text );
            break;

        case DeclAttr:
            stateDeclAttr( token, text );
            break;

        case DeclValue:
            stateDeclValue( token, text );
            break;

        default:
//            printf( "StyleParser::token() -- ignoring token = %d, text = '%s'\n", token, (const char*)text );
            break;
    }
}

void StyleParser::done()
{
    emit done( _styleSheet );
    delete this;
}

void StyleParser::stateSelector( StyleLexer::Token token, QString text )
{
    switch ( token ) {
        case StyleLexer::Identifier:
            _style = new Style( text );
            _state = SelectorAttrs;
            break;

        default:
//            printf( "StyleParser::stateSelector() -- ignoring token = %d, text = '%s'\n", token, (const char*)text );
            break;
    }
}

void StyleParser::stateSelectorAttrs( StyleLexer::Token token, QString text )
{
    switch ( token ) {
        case StyleLexer::OpenParen:
            _attrName = "";
            _attrValue = "";
            break;

        case StyleLexer::CloseParen:
            if ( _attrName != "" ) {
                _style->_sgmlAttrs.append( new Style::Attr( _attrName, _attrValue ) );
//                printf( "StyleParser::stateSelectorAttrs() -- name = '%s', value = '%s'\n", _attrName.data(), _attrValue.data() );
                _attrName = "";
                _attrValue = "";
            }
            break;

        case StyleLexer::Identifier:
            if ( _attrName == "" ) {
                _attrName = text;
            } else {
                _attrValue = text;
            }
            break;

        case StyleLexer::Integer:
            _attrValue = text;
            break;

        case StyleLexer::EqualSign:
            break;

        case StyleLexer::Comma:
            if ( _attrName != "" ) {
                _style->_sgmlAttrs.append( new Style::Attr( _attrName, _attrValue ) );
//                printf( "StyleParser::stateSelectorAttrs() -- name = '%s', value = '%s'\n", _attrName.data(), _attrValue.data() );
                _attrName = "";
                _attrValue = "";
            }
            break;

        case StyleLexer::OpenBrace:
            _state = DeclAttr;
            break;
            
        default:
            break;
    }
}

void StyleParser::stateDeclAttr( StyleLexer::Token token, QString text )
{
    switch ( token ) {
        case StyleLexer::Identifier:
            _attrName = text.copy();
            _state = DeclValue;
            break;

        case StyleLexer::CloseBrace:
            _style = 0;
            _state = Selector;
            break;
            
        default:
            break;
    }
}

void StyleParser::stateDeclValue( StyleLexer::Token token, QString text )
{
    switch ( token ) {
        case StyleLexer::Identifier:
        {
            if ( _attrName == "display" ) {
                _displayProto = styleSheetProto->findDisplay( text );
                if ( !_displayProto ) {
                    printf( "StyleParser::stateDeclValue() -- unknown display type '%s', ignoring declaration.\n", text.data() );
                } else {
                    _style->setDisplay( _displayProto );
                    _styleSheet->addStyle( _style );
                }
            } else {
                if ( !_style ) {
                    printf( "StyleParser::stateDeclValue() -- unknown display type, ignoring attr '%s'.\n", _attrName.data() );
                } else {
//                    printf( "StyleParser::stateDeclValue() -- attr = '%s', value = '%s'\n", _attrName.data(), text.data() );
                    StyleAttrProto* proto = styleSheetProto->findProto( _attrName );
                    if ( proto ) {
                        StyleAttr* attr = 0;
                        switch ( proto->type() ) {
                            case StyleAttrProto::String:
                                attr = new StringStyleAttr( proto, text );
                                break;
                                
                            case StyleAttrProto::Number:
                                attr = new NumberStyleAttr( proto, text.toInt(), FALSE );
                                break;
                                
                            case StyleAttrProto::Enumerated:
                                attr = new EnumeratedStyleAttr( proto, proto->tokenToValue( text ) );
                                break;
                                
                            case StyleAttrProto::List:
                                attr = new ListStyleAttr( proto, text );
                                break;
                        }
                        if ( attr ) {
                            _style->_attrs.append( attr );
                        }
                    } else {
                        printf( "StyleParser::stateDeclValue() -- unknown style '%s', ignoring.\n", _attrName.data() );
                    }
                }
            }
            _state = DeclAttr;
        }
        break;

        case StyleLexer::Integer:
            if ( !_style ) {
                printf( "StyleParser::stateDeclValue() -- unknown display type, ignoring attr '%s'.\n", _attrName.data() );
            } else {
//                printf( "StyleParser::stateDeclValue() -- attr = '%s', value = '%s'\n", _attrName.data(), text.data() );
                StyleAttrProto* proto = styleSheetProto->findProto( _attrName );
                if ( proto ) {
                    StyleAttr* attr = 0;
                    switch ( proto->type() ) {
                        case StyleAttrProto::String:
                            attr = new StringStyleAttr( proto, text );
                            break;
                            
                        case StyleAttrProto::Number:
                            attr = new NumberStyleAttr( proto, text.toInt(), FALSE );
                            break;
                            
                        case StyleAttrProto::Enumerated:
                            attr = new EnumeratedStyleAttr( proto, proto->tokenToValue( text ) );
                            break;
                            
                        case StyleAttrProto::List:
                            attr = new ListStyleAttr( proto, text );
                            break;
                    }
                    if ( attr ) {
                        _style->_attrs.append( attr );
                    }
                } else {
                    printf( "StyleParser::stateDeclValue() -- unknown style '%s', ignoring.\n", _attrName.data() );
                }
            }
            _state = DeclAttr;
            break;

        case StyleLexer::Colon:
            break;

        default:
            break;
    }
}
