/*
  Copyright (C) 1997  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

#include <stdio.h>
#include <stdlib.h>
#include <varargs.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <pwd.h>
#include <signal.h>
#include "common.h"

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                         Global program storage                          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

char *pipefname;							/* named pipe report file */
int pipeFD = 0;								/* report file descriptor */
char *scriptfname;							/* login script filename */
char *envfname;								/* dialing environment filename */
int childPID;								/* PID of running child */

int maxAttempts = MAXNUM_RETRY;				/* max dialing attempts */
int sleepDelay = MAXSEC_DELAY;				/* post-dial sleep delay */
int connectWait = MAXSEC_CNWAIT;			/* maximum wait for connection */
char account[MAXLEN_ACCOUNT+2];				/* account name */
char passwd[MAXLEN_PASSWD+2];				/* password */
char **phone;								/* phone numbers */
int numPhones;								/* how many there are to dial */
char **sline;								/* script lines */
int numSlines;								/* how many there are to include */
char **CBsln;								/* call-back script lines */
int numCBSlns;								/* how many there are to include */
int CBDelay;								/* time wait for call-back */
char *modemInit;							/* modem init string */				
char modemUInit[MAXLEN_MODEM+2];			/* user specified modem init str */
char modemUReset[MAXLEN_MODEM+2];			/* user specified reset string */				
char dialType;								/* dialing type character */
int autologin;								/* true for automatic login */
int callback;								/* true when call-back desired */
char dispname[MAXLEN_TTPARM+2];				/* X11 display for auto login */
char winpos[MAXLEN_TTPARM+2];				/* and terminal window position */
int authlogin;								/* true for PAP/PAPS/CHAPS login */
char bgcol[16] = {0};						/* form-background color string */


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                             Utility routines                            |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Print message together with system error message and exit. Note the
   implicit total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 256

void doErr(char *msg)
{
	char emsg[MSGLEN_ERR];

	if (errno < sys_nerr)
		sprintf(emsg, "xispdial: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xispdial: %s: error #%d\n", msg, errno);
	fputs(emsg, stderr);
	if (pipeFD) close(pipeFD);
	exit(1);
}


/* Open a named pipe for writing only */

int namedPipe(char *fname)
{
	struct stat st;
	int fd;

	if (access(fname, F_OK) == -1)				/* check to see if it exists */
		doErr("namedPipe: access");				/* nope, this is not right! */
	else {
		stat(fname, &st);						/* yes, get the node status */
		if (!S_ISFIFO(st.st_mode))				/* is it a FIFO? */
			doErr("namedPipe: stat");			/* nope, still not right! */
	}
	fd = open(fname, O_WRONLY|O_NDELAY);		/* yes, open it for writing */
	if (fd < 0)									/* error means no process has */
		doErr("namedPipe: open");				/* opened it for reading */
	return fd;									/* return the descriptor */
}


/* Write messages printf() style to the named pipe file descriptor. Note
   the implicit total length of MSGLEN_PIPE bytes for the resulting string. */

#define MSGLEN_PIPE 256

int pprintf(va_alist) va_dcl
{
	int iw, bw;
	va_list ap;
	char *fmt, msg[MSGLEN_PIPE];

	va_start(ap);								/* start variable arg list */
	fmt = va_arg(ap, char*);					/* first string is format */
	iw = vsprintf(msg, fmt, ap);				/* pass rest to vsprintf() */
	va_end(ap);									/* end variable arg list */
	bw = write(pipeFD, msg, strlen(msg));		/* write buffer to pipe */
	if (bw < strlen(msg))						/* all bytes written? */
		doErr("xispdial: pprintf");				/* nope, bail out */
	return iw;									/* else return items written */
}


/* Execute a child process, attaching its stderr to a named pipe. The
   pipe output is read by xisp and displayed on its browser window.   */

#define MAXARGS_CHILD 16

int child(va_alist) va_dcl
{
	int i, stat;
	char *args[MAXARGS_CHILD+1];
	va_list ap;

	childPID = fork();							/* fork to create child */
	if (childPID < 0)							/* ret < 0 : fork failed */
		doErr("child: fork");
	if (childPID) {								/* in parrent process */
		i = waitpid(childPID, &stat, 0);		/* get child return status */
		if (i < 0)								/* ret < 0 : wait failed */
			doErr("child: waitpid");
		childPID = 0;							/* OK, indicate child done */
	}
	else {										/* in child process */
		dup2(pipeFD, 2);						/* tack stderr on pipe input */
		va_start(ap);							/* start variable arg list */
		for (									/* parse program arguments */
			i=0; i < MAXARGS_CHILD &&			/* to a max of MAXARGS_CHILD */
			(args[i]=va_arg(ap, char*)) !=
				(char *)0;
			i++
		);
		va_end(ap);								/* end variable arg list */
		execv(args[0], args);					/* exec child */
		doErr("child: execv");					/* return here means error */
	}
	if (WIFEXITED(stat))						/* parent returns child stat */
		return(WEXITSTATUS(stat));				/* unless something funny */
	else										/* has happened, in which */
		return -1;								/* case -1 is returned */
}

/* Prints out of memory error and exits */

void outOfMem(char *msg)
{
	fprintf(stderr, "xispdial: %s: out of memory!\n", msg);
	if (pipeFD) close(pipeFD);
	exit(1);
}

/* Initializes the dialer file names using the user home directory. */

void initFnames(void)
{
	struct passwd *user;

	user = getpwuid(getuid());
	pipefname = (char *)malloc(6+strlen(user->pw_name)+strlen(PIPEFNAME)+1);
	scriptfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(SCRIPTFNAME)+1);
	envfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(ENVFNAME)+1);
	if (pipefname != NULL && scriptfname != NULL && envfname != NULL) {
		strcpy(pipefname, "/tmp/"); strcat(pipefname, PIPEFNAME);
		strcat(pipefname, "."); strcat(pipefname, user->pw_name);
		strcpy(scriptfname, user->pw_dir); strcat(scriptfname, "/");
		strcat(scriptfname, SCRIPTFNAME);
		strcpy(envfname, user->pw_dir); strcat(envfname, "/");
		strcat(envfname, ENVFNAME);
	}
	else outOfMem("initFnames");
}


/* Reads the environment file created by xISP and then deletes it */

void uEOF(void)
{
	pprintf("%s: %s!\n", "xispdial",			/* notify xisp */
			"environ premature EOF");
	close(pipeFD);								/* close the report pipe */
	exit(1);									/* and exit with error */
}

FILE *envfp;  /* environment file pointer common to readLine(), getISPenv() */

void readLine(char *param, int len)
{
	if (fgets(param, len+1,						/* read environment line */
			  envfp) == NULL)
		uEOF();									/* bail out on error */
	param[strlen(param)-1]=0;					/* kill the ending '\n' */
}

void getISPenv(void)
{
	int ir, i;
	char LM[16] = {0};

	envfp = fopen(envfname, "r");				/* open temp environment file */
	if (envfp == NULL)							/* if not there bail out */
		doErr("getISPenv: fopen");

	ir = fscanf(envfp,"%d ",&maxAttempts);		/* read all needed dialing */
	ir += fscanf(envfp,"%d ",&sleepDelay);		/* environment variables */
	ir += fscanf(envfp,"%d ",&connectWait);
	ir += fscanf(envfp,"%d ",&numPhones);		/* read # of phone numbers */
	if (ir < 4)									/* all read so far? */
		uEOF();									/* nope, bail out */
	phone = (char **)							/* allocate space for */
			malloc(numPhones*sizeof(char *));	/* numPhones phone numbers */
	for (i=0; i<numPhones && phone!=NULL;		/* which follow */
		 i++)
	{
		phone[i] = (char *)						/* allocate phone number */
					malloc(MAXLEN_PHONE+2);
		if (phone[i] != NULL)
			*(phone[i]) = 0;					/* and initialize it */
		else
			phone = NULL;
	}
	if (phone == NULL)							/* allocation completed? */
		outOfMem("getISPenv");					/* nope, bail out */
	for (i=0; i<numPhones; i++)
		readLine(phone[i], MAXLEN_PHONE);		/* read telephone numbers */
	readLine(LM, sizeof(LM)-2);					/* read login method */
	autologin = (strstr(LM, "AUTO") != NULL);	/* set autologin flag */
	if (autologin)
		callback = !strcmp(LM, "AUTO-CB");		/* set call-back flag */
	else
		authlogin = !strcmp(LM, "AUTH");		/* set authlogin flag */

	if (autologin) {							/* if login is indeed auto */
		readLine(account, MAXLEN_ACCOUNT);		/* read account name */
		readLine(passwd, MAXLEN_PASSWD);		/* read password and */
		ir = fscanf(envfp,"%d ",&numSlines);	/* number of script lines */
		if (! ir)								/* bail out on error */
			uEOF();
		if (numSlines) {						/* 0 lines is possible */
			sline = (char **)					/* allocate space for */
			 malloc(numSlines*sizeof(char *));	/* numSlines scipt lines */
			for (i=0; i<numSlines&&sline!=NULL;	/* which follow */
				 i++)
			{
				sline[i] = (char *)				/* allocate script line */
					malloc(2*MAXLEN_SLINE+5+2);
				if (sline[i] != NULL)
					*(sline[i]) = 0;			/* and initialize it */
				else
					sline = NULL;
			}
			if (sline == NULL)					/* allocation completed? */
				outOfMem("getISPenv");			/* nope, bail out */
			for (i=0; i<numSlines; i++)
				readLine(sline[i],				/* read script lines */
						 2*MAXLEN_SLINE+5);
		}
		if (callback) {
			ir= fscanf(envfp,"%d ",&CBDelay);	/* time wait for call-back */
			ir+=fscanf(envfp,"%d ",&numCBSlns);	/* number of call-back lines */
			if (ir < 2)							/* bail out on error */
				uEOF();
			if (numCBSlns) {					/* 0 lines is possible */
				CBsln = (char **)				/* allocate space for */
				 malloc(numSlines*				/* numCBSlns call-back lines */
						sizeof(char *));
				for (i=0; i<numCBSlns &&		/* which follow */
						  CBsln!=NULL;
					 i++)
				{
					CBsln[i] = (char *)			/* allocate call-back line */
						malloc(2*MAXLEN_SLINE
							   +5+2);
					if (CBsln[i] != NULL)
						*(CBsln[i]) = 0;		/* and initialize it */
					else
						CBsln = NULL;
				}
				if (CBsln == NULL)				/* allocation completed? */
					outOfMem("getISPenv");		/* nope, bail out */
				for (i=0; i<numCBSlns; i++)
					readLine(CBsln[i],			/* read script lines */
							 2*MAXLEN_SLINE+5);
			}
		}
	}
	else if (!authlogin) {						/* !authlogin->manual login */
		readLine(dispname, MAXLEN_TTPARM);		/* read the X11 display */
		readLine(bgcol, sizeof(bgcol)-2);		/* read background color */
		readLine(winpos, MAXLEN_TTPARM);		/* read window position */
	}

	readLine(modemUReset, MAXLEN_MODEM);		/* modem reset string */
	readLine(modemUInit, MAXLEN_MODEM);			/* modem init string */
	if (fscanf(envfp,"%c ",&dialType) < 1)		/* and dialing type */
		uEOF();									/* bail out on error */

	fclose(envfp);								/* close and remove the */
	unlink(envfname);							/* temp environment file */
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                             Dialer routines                             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

#ifdef _SLOW_MODEM_
#define MODEM_SLEEP 2
#else
#define MODEM_SLEEP 1
#endif

int init_modem(void)							/* initialize modem */
{
	int rcode;

	rcode = child(CHAT,"-V","TIMEOUT","3",
				  "",modemUReset,"OK-+++\\c-OK",
				  modemInit,(char *)0);
	sleep(MODEM_SLEEP);
	return rcode;
}

int re_init_modem(void)							/* re-initialize modem */
{
	int rcode;

	rcode = child(CHAT,"-V","TIMEOUT","3",
				  "",modemUReset,"OK-+++\\c-OK",
				  modemInit,(char *)0);
	if (rcode) {
		sleep(MODEM_SLEEP);
		rcode = child(CHAT,"-V","TIMEOUT","3",
					  "","AT","OK-AT-OK",
					  modemInit,(char *)0);
	}
	sleep(MODEM_SLEEP);
	return rcode;
}

#define MAXLEN_PREAMBLE 256
#define MAXLEN_CBTRAIL  128

int callnumber(char *number)					/* dial the number given */
{
	int i, rcode;
	FILE *scriptF;
	char preambleD[MAXLEN_PREAMBLE+1] =
			"TIMEOUT\t\t3\nABORT\t\tBUSY\nABORT\t\t'NO CARRIER'\n"
			"ABORT\t\tenied\nABORT\t\timeout\n''\t\t\t'AT D%c %s'\n"
			"TIMEOUT\t\t%d\nCONNECT\t\t''\nTIMEOUT\t\t5\n\\r\t\t''\n",
		 preambleCB[MAXLEN_PREAMBLE+1] =
			"TIMEOUT\t\t3\nABORT\t\tBUSY\n"
			"ABORT\t\tenied\nABORT\t\timeout\n''\t\t\t'AT D%c %s'\n"
			"TIMEOUT\t\t%d\nCONNECT\t\t''\nTIMEOUT\t\t5\n\\r\t\t''\n",
		 cbtrail[MAXLEN_CBTRAIL+1] =
			"TIMEOUT\t\t%d\nCONNECT\t\t''\nTIMEOUT\t\t5\n\\r\t\t''\n",
		 script[MAXNUM_SLINES*(2*MAXLEN_SLINE+5)+MAXLEN_PREAMBLE+1];

	umask(077);									/* read/write only by owner */
	scriptF = fopen(scriptfname, "w");			/* open the script file */
	if (scriptF == NULL)						/* failed to do so? */
		doErr("callnumber: fopen");				/* yup, bail out */
	if (autologin) {							/* if automatic login */
		if (callback)							/* and if call-back enabled */
			strcpy(script, preambleCB);			/* copy call-back preamble */
		else									/* otherwise copy */
			strcpy(script, preambleD);			/* the dialing preamble */
		for (i=0; i<numSlines; i++) {			/* append the script lines */
			strcat(script, sline[i]);
			strcat(script, "\n");
		}
		fprintf(scriptF, script, dialType,		/* using the script as fmt */
				number, connectWait,			/* stick all the rest in */
				account, passwd);
		if (callback) {							/* if call-back is desired */
			strcpy(script, cbtrail);			/* to the script trail add */
			for (i=0; i<numCBSlns; i++) {		/* the call-back lines */
				strcat(script, CBsln[i]);
				strcat(script, "\n");
			}
			fprintf(scriptF, script, CBDelay,	/* using script as fmt stick */
					account, passwd);			/* delay/account/password in */
		}
	}
	else {										/* PAP/PAPS/CHAPS or manual */
		strcpy(script, preambleD);				/* copy the dialing preamble */
		fprintf(scriptF, script, dialType,		/* and print the script out */
				number);
	}
	fclose(scriptF);							/* close the script file */
	rcode = child(CHAT,"-V","-f",				/* hand it over to chat */
				  scriptfname,(char *)0);
	unlink(scriptfname);						/* done with script, delete */
	if (rcode == 0) {							/* chat terminated OK? */
		pprintf("\n%s: %s %s\n", "xispdial",	/* yes, send "done" message */
				"done dialing", number);
		if (!autologin && !authlogin)			/* start manual login window */
			rcode = child(TERMINAL, "-bgcol",	/* if it is so desired */
						bgcol, "-display",
					  	dispname, "-geometry",
						winpos, (char *)0);
		close(pipeFD);							/* close the report pipe */
		exit(rcode);							/* and exit */
	}
	if (rcode < 0) {							/* nope, chat problem */
		pprintf("\n%s: %s\n", "callnumber",		/* report it */
				"chat execution failed");
		close(pipeFD);							/* and bail out */
		exit(1);							
	}
	return rcode;								/* else valid report code */
}

void callall(int sn)							/* call the list of numbers */
{
	int i, rcode;

	pprintf("%s: %s #%d\n", "xispdial",			/* report dialing attempt */
			"dialing attempt", sn+1);
	for (i=0; i<numPhones; i++) {				/* try all phones given */
		pprintf("xispdial: dialing: %s\n",		/* report dialed number */
				phone[i]);
		if ((rcode = callnumber(phone[i])))		/* dial it */
			pprintf("\n");						/* not successful, print NL */
		if (rcode == 3) {						/* did chat timeout? */
			pprintf("TIMEOUT\n");				/* yup, report it */
			pprintf("%s: %s %s\n",
				"xispdial", "timeout dialing",
				phone[i]);
			child(CHAT,"-V",""," ",(char *)0);	/* and hang-up */
		}
		sleep(sleepDelay);						/* sleep a while */
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                         SIGTERM trap and Main                           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void Term(int signal)
{
	switch (signal) {
		case SIGTERM: {							/* SIGTERM signal received */
			pprintf("\nxispdial: SIGTERM\n");	/* return status via pipe */
			if (childPID)						/* if a child is in progress */
				kill(childPID, SIGTERM);		/* send it SIGTERM also */
			unlink(envfname);					/* delete environment file */
			unlink(scriptfname);				/* and script file */
			re_init_modem();					/* reset the modem */
			close(pipeFD);						/* close the report pipe */
			exit(1);							/* and exit */
			break;
		}
		default: exit(1);
	}
}

int main()
{
	int rcode, attempt;

	signal(SIGTERM, Term);			/* register SIGTERM handler */
	initFnames();					/* assemble file names used */
	pipeFD = namedPipe(pipefname);	/* initialize named pipe to xISP */
	getISPenv();					/* get info via dialing environment file */

	pprintf("xispdial: PID=%d.\n", getpid());	/* the '.' is important! */
	pprintf("xispdial: dialing ISP: user: %s\n", account);

/* Prepare the modem init string
   ----------------------------- */
	modemInit = malloc(strlen(modemUInit)+4+1);
	if (modemInit == NULL)
		outOfMem("main");
	strcpy(modemInit, modemUInit);
	strcat(modemInit, " H0");

/* Initialize the modem for dialing
   -------------------------------- */
	if ((rcode=init_modem())) {
		pprintf("xispdial: chat returned %d!\n", rcode);
		pprintf("xispdial: modem init failed!\n");
		return 1;
	}

/* Loop dialing the numbers given maxAttempts times
   ------------------------------------------------ */
	for (attempt=0; attempt<maxAttempts; attempt++) {
		pprintf("\n");
		callall(attempt);
		if ((rcode=re_init_modem())) {
			pprintf("xispdial: chat returned %d!\n", rcode);
			pprintf("xispdial: modem re-init failed!\n");
			return 1;
		}
	}
	close(pipeFD);
	return 1;
}

