/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: search.c,v 1.2 94/06/19 11:22:12 nau Exp $";

/* search routines
 */

#include "global.h"

#include "data.h"
#include "search.h"

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	LineTypePtr		SearchLineOnLayer(LayerTypePtr, Position, Position);
static	RectTypePtr		SearchRectOnLayer(LayerTypePtr, Position, Position);
static	TextTypePtr		SearchTextOnLayer(LayerTypePtr, Position, Position);

/* ---------------------------------------------------------------------------
 * searches a via
 */
PinTypePtr SearchVia(Position X, Position Y)
{
	PinTypePtr	via = PCB->Via +PCB->ViaN -1;
	Cardinal	n = PCB->ViaN;

	if (!PCB->ViaOn)
		return(NULL);

	for(; n; n--, via--)
		if (abs(via->X - X) <= via->Thickness/4 &&
		    abs(via->Y - Y) <= via->Thickness/4 )
			return(via);
	return(NULL);
}

/* ---------------------------------------------------------------------------
 * searches a pin
 */
Boolean SearchPin(ElementTypePtr *ElementPtrPtr, PinTypePtr *PinPtrPtr, Position X, Position Y)
{
	ElementTypePtr	element = PCB->Element +PCB->ElementN -1;
	PinTypePtr		pin;
	Cardinal		elementN = PCB->ElementN,
					n;

	if (!PCB->PinOn)
		return(False);

	for(; elementN; elementN--, element--)
		for(pin = element->Pin, n = element->PinN; n; n--, pin++)
			if (abs(pin->X - X) <= pin->Thickness/4 &&
		    	abs(pin->Y - Y) <= pin->Thickness/4 )
		    {
		    	*PinPtrPtr = pin;
		    	*ElementPtrPtr = element;
		    	return(True);
		    }
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches line on specified layer
 * the search starts with the last line and goes back to the beginning
 * We wont process an exact match because this means to use sqare root
 * calculations.
 */
static LineTypePtr SearchLineOnLayer(LayerTypePtr Layer, Position X, Position Y)
{
	LineTypePtr	line = Layer->Line +Layer->LineN -1;
	Cardinal	n = Layer->LineN;

	for(; n; n--, line--)
		switch(line->Direction)
		{
			case 0:				/* line goes straight up */
				if (line->Y1 <= Y && line->Y2 >= Y &&
					abs(line->X1 -X) <= line->Thickness/2)
					return(line);
				break;

			case 1:				/* line goes up and right */
				if (X >= line->X1 && X <= line->X2 &&
					Y >= line->Y1 && Y <= line->Y2 &&
					abs((Y -line->Y1) -(X -line->X1)) <= line->Thickness/2)
					return(line);
				break;

			case 2:				/* line goes straight right */
				if (line->X1 <= X &&
					line->X2 >= X &&
					abs(line->Y1 -Y) <= line->Thickness/2)
					return(line);
				break;

			case 3:				/* line goes down and right */
				if (X >= line->X1 && X <= line->X2 &&
					Y <= line->Y1 && Y >= line->Y2 &&
					abs((line->Y1 -Y) -(X -line->X1)) <= line->Thickness/2)
					return(line);
				break;

			default:
				break;
		}

	return(NULL);
}

/* ---------------------------------------------------------------------------
 * searches line on all layers that are switched on
 * the values are returned with pointers to a pointer
 */
Boolean SearchLine(LayerTypePtr *LayerPtrPtr, LineTypePtr *LinePtrPtr, Position X, Position Y)
{
	Cardinal	i;

	for (i = 0; i < MAX_LAYER; i++)
	{
		*LayerPtrPtr = &PCB->Layer[LayerStack[i]];
		if ((*LayerPtrPtr)->On &&
			(*LinePtrPtr = SearchLineOnLayer(*LayerPtrPtr, X, Y)) != NULL)
			return(True);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches rectangle on specified layer
 * the search starts with the last rectangle and goes back to the beginning
 */
static RectTypePtr SearchRectOnLayer(LayerTypePtr Layer, Position X, Position Y)
{
	RectTypePtr	rect = Layer->Rect +Layer->RectN -1;
	Cardinal	n = Layer->RectN;

	for(; n; n--, rect--)
		if (X >= rect->X && X <= rect->X +rect->Width &&
			Y >= rect->Y && Y <= rect->Y +rect->Height)
			return(rect);
	return(NULL);
}

/* ---------------------------------------------------------------------------
 * searches rectangle on all layers that are switched on
 * the values are returned with pointers to a pointer
 */
Boolean SearchRect(LayerTypePtr *LayerPtrPtr, RectTypePtr *RectPtrPtr, Position X, Position Y)
{
	Cardinal	i;

	for (i = 0; i < MAX_LAYER; i++)
	{
		*LayerPtrPtr = &PCB->Layer[LayerStack[i]];
		if ((*LayerPtrPtr)->On &&
			(*RectPtrPtr = SearchRectOnLayer(*LayerPtrPtr, X, Y)) != NULL)
			return(True);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches text on specified layer
 * the search starts with the last text and goes back to the beginning
 */
static TextTypePtr SearchTextOnLayer(LayerTypePtr Layer, Position X, Position Y)
{
	TextTypePtr	text = Layer->Text +Layer->TextN -1;
	Cardinal	n = Layer->TextN;

	for(; n; n--, text--)
		if (X >= text->Rect.X && X <= text->Rect.X +text->Rect.Width &&
			Y >= text->Rect.Y && Y <= text->Rect.Y +text->Rect.Height)
			return(text);
	return(NULL);
}

/* ---------------------------------------------------------------------------
 * searches text on all layers that are switched on
 * the values are returned with pointers to a pointer
 */
Boolean SearchText(LayerTypePtr *LayerPtrPtr, TextTypePtr *TextPtrPtr, Position X, Position Y)
{
	Cardinal	i;

	for (i = 0; i < MAX_LAYER; i++)
	{
		*LayerPtrPtr = &PCB->Layer[LayerStack[i]];
		if ((*LayerPtrPtr)->On &&
			(*TextPtrPtr = SearchTextOnLayer(*LayerPtrPtr, X, Y)) != NULL)
			return(True);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * searches an element
 * the search starts with the last element and goes back to the beginning
 */
ElementTypePtr SearchElement(Position X, Position Y)
{
	ElementTypePtr	element = PCB->Element+PCB->ElementN -1,
					save = NULL;
	Cardinal		n = PCB->ElementN,
					area = 0;

	if (!PCB->ElementOn)
		return(NULL);

	for(; n; n--, element--)
		if (X >= element->Rect.X && X <= element->Rect.X +element->Rect.Width &&
			Y >= element->Rect.Y && Y <= element->Rect.Y +element->Rect.Height)
		{
				/* use the element with the smallest bounding box */
			if (!save || element->Rect.Width *element->Rect.Y < area)
			{
				area = element->Rect.Width *element->Rect.Y;
				save = element;
			}
		}
	return(save);
}

