/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: menu.c,v 1.5 94/07/19 14:08:20 nau Exp $";

/* initializes menus and handles callbacks
 */

#include <ctype.h>

#include "global.h"

#include "control.h"
#include "create.h"
#include "cursor.h"
#include "data.h"
#include "dialog.h"
#include "draw.h"
#include "file.h"
#include "error.h"
#include "memory.h"
#include "menu.h"
#include "misc.h"
#include "remove.h"

#include <X11/Xaw/Form.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>

/* ---------------------------------------------------------------------------
 * some menu types
 */
typedef struct						/* a single popup menu entry */
{
	String			Name,			/* the widgets name */
					Label;			/* menu item label or '---' for a seperator */
	XtCallbackProc	Callback;
	XtPointer		ClientData;
	Widget			W;
} PopupEntryType, *PopupEntryTypePtr;

typedef struct						/* a popup menu */
{
	String				Name,		/* the widgets name */
						Title;		/* the menu title */
	PopupEntryTypePtr	Entries;	/* pointer to popup menu */
	XtCallbackProc		CB_Popup,	/* called on popup */
						CB_Popdown;	/* and popdown */
} PopupMenuType, *PopupMenuTypePtr;

typedef struct
{
	String				Name,		/* the widgets name */
						Label;		/* the buttontext */
	PopupMenuTypePtr	PopupMenu;	/* pointer to popup menu */
} MenuButtonType, *MenuButtonTypePtr;

typedef struct
{
	String			Name,			/* the widhets name */
					Label;			/* the buttontext */
	XtCallbackProc	Callback;		/* called on selection */
	XtPointer		ClientData;		/* data passed to callback */
} CommandButtonType, *CommandButtonTypePtr;

/* ---------------------------------------------------------------------------
 * some local defines
 */
#define	FULLSIZE	0
#define	SHRINK		1

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	CB_Redraw(Widget, XtPointer, XtPointer);
static	void	CB_Size(Widget, XtPointer, XtPointer);
static	void	CB_Zoom(Widget, XtPointer, XtPointer);
static	void	CB_Grid(Widget, XtPointer, XtPointer);
static	void	CB_Linesize(Widget, XtPointer, XtPointer);
static	void	CB_Viasize(Widget, XtPointer, XtPointer);
static	void	CBPopup_Center(Widget, XtPointer, XtPointer);
static	void	CB_Mode(Widget, XtPointer, XtPointer);
static	void	InitPopupTree(Widget, PopupEntryTypePtr);
static	void	InitPopupMenu(Widget, PopupMenuTypePtr);
static	Widget	InitCommandButton(Widget, CommandButtonTypePtr, Widget, Widget);

/* ---------------------------------------------------------------------------
 * menu definitions
 */
static	PopupEntryType	ModeMenuEntries[] = {
							{ "line", "---", NULL, NULL, NULL },
							{ "noMode", "reset all modes", CB_Mode, (XtPointer) MODE_NONE, NULL },
							{ "lineMode", "line mode", CB_Mode, (XtPointer) MODE_LINE, NULL },
							{ "blockMode", "block mode", CB_Mode, (XtPointer) MODE_BLOCK, NULL },
							{ "elementMode", "element mode", CB_Mode, (XtPointer) MODE_ELEMENT, NULL },
							{ "linestackMode", "linestack mode", CB_Mode, (XtPointer) MODE_LINESTACK, NULL },
							{ NULL, NULL, NULL, NULL }},

						ControlMenuEntries[] = {
							{ "line", "---", NULL, NULL, NULL },
							{ "header", "viasize", NULL, NULL, NULL },
							{ "viasize40", "40 mil", CB_Viasize, (XtPointer) 40, NULL },
							{ "viasize60", "60 mil", CB_Viasize, (XtPointer) 60, NULL },
							{ "viasize80", "80 mil", CB_Viasize, (XtPointer) 80, NULL },
							{ "line", "---", NULL, NULL, NULL },
							{ "header", "linewidth", NULL, NULL, NULL },
							{ "linesize10", "10 mil", CB_Linesize, (XtPointer) 10, NULL },
							{ "linesize20", "20 mil", CB_Linesize, (XtPointer) 20, NULL },
							{ "linesize50", "50 mil", CB_Linesize, (XtPointer) 50, NULL },
							{ "linesize100", "100 mil", CB_Linesize, (XtPointer) 100, NULL },
							{ "line", "---", NULL, NULL, NULL },
							{ "header", "grid", NULL, NULL, NULL },
							{ "grid10", "10 mil", CB_Grid, (XtPointer) 10, NULL },
							{ "grid20", "20 mil", CB_Grid, (XtPointer) 20, NULL },
							{ "grid50", "50 mil", CB_Grid, (XtPointer) 50, NULL },
							{ "grid100", "100 mil", CB_Grid, (XtPointer) 100, NULL },
							{ "line", "---", NULL, NULL, NULL },
							{ "header", "zoom", NULL, NULL, NULL },
							{ "zoom1", "1 : 1 ", CB_Zoom, (XtPointer) 0, NULL },
							{ "zoom2", "1 : 2 ", CB_Zoom, (XtPointer) 1, NULL },
							{ "zoom4", "1 : 4 ", CB_Zoom, (XtPointer) 2, NULL },
							{ "zoom8", "1 : 8 ", CB_Zoom, (XtPointer) 3, NULL },
							{ "zoom16", "1 :16 ", CB_Zoom, (XtPointer) 4, NULL },
							{ "line", "---", NULL, NULL, NULL },
							{ "header", "display size", NULL, NULL, NULL },
							{ "fullsize", "full size", CB_Size, (XtPointer) FULLSIZE, NULL },
							{ "shrink", "shrink to fit", CB_Size, (XtPointer) SHRINK, NULL },
							{ NULL, NULL, NULL, NULL }};

static	PopupMenuType	ControlMenu = { "controlMenu", "Control Menu", ControlMenuEntries, CBPopup_Center, NULL },
						ModeMenu = { "modeMenu", "Mode Menu", ModeMenuEntries, CBPopup_Center, NULL };

static	CommandButtonType	RedrawButton = { "redraw", "Redraw", CB_Redraw, NULL };

/* ---------------------------------------------------------------------- 
 * callback routine used by redraw button
 */
static void CB_Redraw(Widget W, XtPointer ClientData, XtPointer CallData)
{
	RedrawOutput();
}

/* ---------------------------------------------------------------------- 
 * callback routine used by shrink and fullsize button
 */
static void CB_Size(Widget W, XtPointer ClientData, XtPointer CallData)
{
	Position	minx, miny,
				maxx, maxy;

	switch((int) ClientData)
	{
		case FULLSIZE:			/* convert viewport to fullsize */
			Settings.Width = Settings.MaxPCBWidth;
			Settings.Height = Settings.MaxPCBHeight;
			break;

		case SHRINK:			/* convert viewport to pcb size */
			GetMinMaxCoordinates(PCB, &minx, &miny, &maxx, &maxy);
			if (maxx <= 100 || maxy <= 100)
				return;
			Settings.Width = MIN(maxx, Settings.MaxPCBWidth);
			Settings.Height = MIN(maxy, Settings.MaxPCBHeight);
			break;
	}
	XtVaSetValues(Output.Output,
		XtNwidth, TO_SCREEN(Settings.Width),
		XtNheight, TO_SCREEN(Settings.Height),
		NULL);
}

/* ---------------------------------------------------------------------- 
 * callback routine to select a new zoom factor
 */
static void CB_Zoom(Widget W, XtPointer ClientData, XtPointer CallData)
{
	SetZoom((int) ClientData);
}

/* ---------------------------------------------------------------------- 
 * callback routine to select a new grid
 */
static void CB_Grid(Widget W, XtPointer ClientData, XtPointer CallData)
{
	SetGrid((int) ClientData);
}

/* ---------------------------------------------------------------------- 
 * callback routine to select a new linesize
 */
static void CB_Linesize(Widget W, XtPointer ClientData, XtPointer CallData)
{
	SetLineSize((int) ClientData);
}

/* ---------------------------------------------------------------------- 
 * callback routine to select a new viasize
 */
static void CB_Viasize(Widget W, XtPointer ClientData, XtPointer CallData)
{
	SetViaSize((int) ClientData);
}

/* ---------------------------------------------------------------------- 
 * callback routine called when menu is popped up to center it at 
 * the current pointer location
 */
static void CBPopup_Center(Widget W, XtPointer ClientData, XtPointer CallData)
{
		/* widget has to be realized to call CenterWidgetAtPointerLocation() */
	XtRealizeWidget(W);
	CenterWidgetAtPointerLocation(W);
}

/* ---------------------------------------------------------------------- 
 * callback routine called by mode menu selectio
 */
static void CB_Mode(Widget W, XtPointer ClientData, XtPointer CallData)
{
	SetMode((int) ClientData);
}

/* ---------------------------------------------------------------------------
 * initializes a command button
 */
static Widget InitCommandButton(Widget Parent, CommandButtonTypePtr CommandButtonPtr, Widget Top, Widget Left)
{
	Widget	command;

	command = XtVaCreateManagedWidget(CommandButtonPtr->Name, commandWidgetClass,
		Parent,
		XtNlabel, CommandButtonPtr->Label,
		XtNfromHoriz, Left,
		XtNfromVert, Top,
		XtNleft, XtChainLeft,
		XtNright, XtChainLeft,
		XtNtop, XtChainTop,
		XtNbottom, XtChainTop,
		NULL);
	XtAddCallback(command, XtNcallback,
		CommandButtonPtr->Callback, CommandButtonPtr->ClientData);

		/* return the created button widget to position some others */
	return(command);
}

/* ---------------------------------------------------------------------------
 * initializes a menu tree
 */
static void InitPopupTree(Widget Parent, PopupEntryTypePtr EntryPtr)
{
	for (; EntryPtr->Label; EntryPtr++)
	{
			/* check if it's only a seperator */
		if (strncmp(EntryPtr->Label, "---", 3))
		{
			EntryPtr->W = XtVaCreateManagedWidget(EntryPtr->Name, smeBSBObjectClass,
				Parent,
				XtNlabel, EntryPtr->Label,
				XtNjustify, XtJustifyCenter,
				XtNleftMargin, 12, 
				NULL);
			if (EntryPtr->Callback)
				XtAddCallback(EntryPtr->W, XtNcallback,
					EntryPtr->Callback, EntryPtr->ClientData);
			else
					/* entry is not selectable */
				XtVaSetValues(EntryPtr->W, XtNsensitive, False, NULL);
		}
		else
			XtVaCreateManagedWidget("menuLine", smeLineObjectClass, Parent, NULL);
	}
}

/* ---------------------------------------------------------------------------
 * initializes one popup menu
 */
static void InitPopupMenu(Widget Parent, PopupMenuTypePtr MenuPtr)
{
	Widget	shell;

	shell = XtVaCreatePopupShell(MenuPtr->Name, simpleMenuWidgetClass,
		Parent,
		XtNlabel, MenuPtr->Title,
		NULL);
	InitPopupTree(shell, MenuPtr->Entries);

		/* install popup and popdown callbacks */
	if (MenuPtr->CB_Popup)
		XtAddCallback(shell, XtNpopupCallback, MenuPtr->CB_Popup, NULL);
	if (MenuPtr->CB_Popdown)
		XtAddCallback(shell, XtNpopupCallback, MenuPtr->CB_Popdown, NULL);
}

/* ---------------------------------------------------------------------------
 * initializes button related menus
 * also initializes control panel
 */
Widget InitMenu(Widget Parent, Widget Top, Widget Left)
{
	Widget	redraw;

	redraw = InitCommandButton(Parent, &RedrawButton, Top, Left);
	InitPopupMenu(Parent, &ControlMenu);
	InitPopupMenu(Parent, &ModeMenu);
	return(redraw);
}

