/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 *  RCS: $Header: global.h,v 1.5 94/07/13 14:19:36 nau Exp $
 */

/* definition of types and macros used by all files
 */

#ifndef	__GLOBAL_INCLUDED__
#define	__GLOBAL_INCLUDED__

#include <limits.h>

#include "macro.h"

#include <X11/Xmd.h>
#include <X11/Xlib.h>
#include <X11/Xresource.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>

/* ---------------------------------------------------------------------------
 * some file-, directory- and environment names
 */
#define	FONTFILENAME		"default_font"

/* ---------------------------------------------------------------------------
 * modes
 */
#define	MODE_NONE				0
#define	MODE_LINE				1
#define	MODE_BLOCK				2
#define	MODE_ELEMENT			3
#define	MODE_LINESTACK			4

/* ---------------------------------------------------------------------------
 * some limit specifications
 */
#define	MAX_LAYER			8		/* max number of layer, check source */
									/* code for more changes */
#define	MIN_LINESIZE		1		/* thickness of lines in 1/1000'' */
#define	MAX_LINESIZE		250
#define	MIN_TEXTSIZE		5		/* thickness of text lines in 1/1000'' */
#define	MAX_TEXTSIZE		250
#define	MIN_PINORVIASIZE	20		/* size of a pin or via in */
#define	MAX_PINORVIASIZE	250
#define	MAX_GRID			200		/* grid in 1/1000'' */
#define	MIN_ZOOM			0		/* min zoom (shift operations) */
#define	MAX_ZOOM			4		/* max zoom (shift operations) */
#define	MAX_FONTPOSITION	127		/* upper limit of characters in my font */

									/* the following limits are PCB and Screen
									 * limits so no convertion has to be done
									 */
#define	MIN_COORD			0
#define	MAX_COORD			20000 

#define	STEP_VIA			500		/* block size for realloc() calls */
#define	STEP_PIN			30
#define	STEP_LINE			500
#define	STEP_ELEMENTLINE	20
#define	STEP_ELEMENTARC		5
#define	STEP_MARK			5
#define	STEP_RECT			100
#define	STEP_TEXT			50
#define	STEP_ELEMENT		50
#define	STEP_SYMBOLLINE		10

#define	MARKED_STATE_RESET		0	/* states for 'marked block' */
#define	MARKED_STATE_STARTED	1
#define	MARKED_STATE_ENDED		2

#define	REDRAW_NONE				0	/* set by control panel */
#define	REDRAW_CURRENT			1
#define	REDRAW_ALL				2

#define	DEFAULT_CELLSIZE		50	/* default cell size for symbols */

#define	MARK_SIZE				50	/* size of a mark for elements */

/* ---------------------------------------------------------------------------
 * some useful values of our widgets
 */
typedef struct						/* holds information about output window */
{
	Widget			Toplevel,		/* toplevel widget */
					StatusLine,		/* label widget of statusline */
					PCBname,		/* current layouts name */
					Elementname,	/* name of currently loaded element */
					CursorPosition,	/* current cursor position */
					ScrollbarLeft,	/* viewport scrollbars */
					ScrollbarBottom,
					Control,		/* control widget */
					Menu,			/* popup menue */
					InputText,		/* fields for user input */
					InputField,
					MasterForm,		/* the widgets thats hols all the others */
					Viewport,		/* viewport widget to scoll output */
					Output;			/* simple widget for drawing */
	Window			OutputWindow;	/* window ID of simple widget */
	GC				bgGC,			/* background */
					fgGC;			/* and foreground (changed from some routines) */
	Position		OffsetX,		/* origin offset */
					OffsetY;
	Dimension		Width,			/* sizes of output window */
					Height;
} OutputType, *OutputTypePtr;

/* ----------------------------------------------------------------------
 * layer group. A layer group identifies layers which are always switched
 * on/off together.
 */
typedef struct
{
	Cardinal		Number[MAX_LAYER],		/* number of entries per groups */
					Entries[MAX_LAYER][MAX_LAYER];
} LayerGroupType, *LayerGroupTypePtr;

/* ---------------------------------------------------------------------------
 * the basic type supported by PCB
 * there are only four combination of coordinates supported:
 * line straight up, right and up, straight right and down and right
 * check 'draw.c' and 'create.c' for details
 */
typedef struct						/* holds information about one line */
{
	Position		X1,				/* start and endposition, */
					Y1,				/* direction and thickness */
					X2,
					Y2;
	Dimension		Thickness;
	BYTE			Direction;
	short			Flags;
} LineType, *LineTypePtr;

typedef struct
{
	Position		X,				/* upper left corner */
					Y;
	Dimension		Width,			/* dimensions */
					Height;
	short			Flags;
} RectType, *RectTypePtr;

typedef struct
{
	Position		X,				/* origin */
					Y;
	RectType		Rect;			/* surrounding rectangle */
	BYTE			Direction;
	short			Flags;
	char			*TextString;	/* string */
} TextType, *TextTypePtr;

typedef struct						/* holds information about one layer */
{
	char			*Name;			/* layer name */
	Cardinal		LineN,			/* number of lines */
					RectN,			/* rectangles */
					TextN,			/* and labels */
					LineMax,		/* max number from malloc() */
					RectMax,		/* max number from malloc() */
					TextMax;		/* max number from malloc() */
	LineTypePtr		Line;			/* pointer to additional structures */
	RectTypePtr		Rect;
	TextTypePtr		Text;
	Boolean			On;				/* visible flag */
	Pixel			Color;			/* color */
} LayerType, *LayerTypePtr;

typedef struct
{
	Position		X,				/* start-, endposition and thickness */
					Y;
	Dimension		Thickness;
	char			*Name;				
	short			Flags;
} PinType, *PinTypePtr;

typedef struct						/* used for arcs */
{
	Position		X,				/* center coordinates */
					Y;
	Dimension		Width,			/* length of axis */
					Height,
					Thickness;
	int				StartAngle,		/* the two limiting angles in degrees */
					Delta;	
} ArcType, *ArcTypePtr;

typedef struct						/* additional marks for elements */
{
	Position	X,
				Y;
} MarkType, *MarkTypePtr;

typedef struct
{

	Position		TextX,			/* position of description string */
					TextY;
	BYTE			Direction;		/* and its direction */
	RectType		Rect;			/* surrounding rectangle */
	Cardinal		PinN,			/* number of pins, lines and arcs */
					PinMax,
					LineN,
					LineMax,
					MarkN,
					MarkMax,
					ArcN,
					ArcMax;
	PinTypePtr		Pin;			/* pointer to pin description */
	LineTypePtr		Line;
	ArcTypePtr		Arc;
	MarkTypePtr		Mark;
	char			*NameOnPCB,		/* the elements names */
					*CanonicalName;
} ElementType, *ElementTypePtr;

typedef struct						/* a single symbol */
{
	LineTypePtr		Line;
	Boolean			Valid;
	Cardinal		LineN,			/* number of lines */
					LineMax;
	Dimension		Width,			/* size of cell */
					Height,
					Delta;			/* distance to next symbol in 0.001'' */
} SymbolType, *SymbolTypePtr;

typedef struct						/* complete set of symbols */
{
	Dimension		MaxHeight,		/* maximum cell width and height */
					MaxWidth;
	RectType		DefaultSymbol;	/* the default symbol is a rectangle */
	SymbolType		Symbol[MAX_FONTPOSITION+1];
	Boolean			Valid;
} FontType, *FontTypePtr;

typedef struct						/* holds information about board layout */
{									/* most of the information is saved with layout */
									/* and initialized from resources when */
									/* a new board design is started */
	char			*Name,			/* name of board */
					*Filename;		/* name of file (from load) */
	Cardinal		ViaN,			/* number of vias */
					ViaMax,			/* max number from malloc() */
					ElementN,		/* and elements */
					ElementMax;		/* max number from malloc() */
	Boolean			Changed,		/* layout has been changed */
					ViaOn,			/* visibility flags */
					ElementOn,
					PinOn;
	short			Flags;			/* see macro.h */
	Pixel			ViaColor,		/* some colors */
					PinColor,
					ConnectedColor,
					ElementColor;
	int				Zoom,			/* zoom factor (factor = 1 << zoom) */
					CursorX,		/* cursor position as saved with layout */
					CursorY,
					Grid,			/* used grid with offsets */
					GridOffsetX,	/* as saves with layout */
					GridOffsetY;
	PinTypePtr		Via;			/* pointer to object data */
	ElementTypePtr	Element;
	LayerType		Layer[MAX_LAYER];
	FontType		Font;
	LayerGroupType	LayerGroups;
} PCBType, *PCBTypePtr;

/* ---------------------------------------------------------------------------
 * stack type, used to simplify drawing of many parallel lines like
 * data- or addressbuses
 */
typedef struct
{
	Cardinal		LineN,			/* lines currently on stack */
					LineMax;		/* maximum number of lines on stack */
	LineTypePtr		Line;			/* data */
	Position		OriginX,		/* origin of stack coordinate system */
					OriginY;
	Boolean			On;
} LineStackType, *LineStackTypePtr;

/* ---------------------------------------------------------------------------
 * some types for cursor drawing, setting of block and lines
 * as well as for merging of elements
 */
typedef struct						/* current marked line */
{
	Position		X1,				/* start- and endposition */
					Y1,
					X2,
					Y2;
	BYTE			Direction,
					State;			/* flags: first call, ... */
} MarkedLineType, *MarkedLineTypePtr;

typedef struct						/* currently marked block */
{
	Position		X,				/* origin of marked block, used for */
					Y,				/* translations and rotations */
					X1,				/* upper left and lower right corner */
					Y1,
					X2,
					Y2;
	BYTE			State;			/* state flag (one, both corners set, off) */
} MarkedBlockType, *MarkedBlockTypePtr;

typedef struct						/* used for merging of elements */
{
	ElementType		Element;		/* the data itself */
	Boolean			Valid;			/* 'holds valid data' */
} MarkedElementType, *MarkedElementTypePtr;

typedef struct						/* holds cursor information */
{
	GC					GC,			/* GC for cursor drawing */
						MarkedGC;	/* and for marked lines and rectangles */
	Position			X,			/* position in PCB coordinates */
						Y;
	Boolean				On,			/* flag for 'is visible' */
						ElementOn;	/* flag 'draw merged element' */
	int					Mode;		/* display mode for lines, blocks... */
	MarkedLineType		MarkedLine;	/* marked line, block ... */
	MarkedBlockType		MarkedBlock;
	MarkedElementType	MarkedElement;
	LineStackType		LineStack;
} CursorType, *CursorTypePtr;

/* ---------------------------------------------------------------------------
 * our resources
 * most of them is used as default when a new design is started
 */
typedef struct						/* some resources... */
{
	Pixel			bgColor,		/* background and cursor color ... */
					CursorColor,
					ViaColor,		/* different object colors */
					PinColor,
					ElementColor,
					ConnectedColor,
					LayerColor[MAX_LAYER];
	Dimension		ViaThickness,	/* some preset values */
					LineThickness,
					Width,			/* width and height of visible pcb */
					Height,
					MaxPCBWidth,	/* maximum allowed size of layout */
					MaxPCBHeight;
	int				Grid,			/* grid 0.001'' */
					Zoom,			/* number of shift operations for zooming */
					PinoutZoom,		/* same for pinout windows */
					PinoutNameLength,	/* maximum displayed length of a pinname */
					CharPerLine;	/* width of an output line in characters */
	long			BackupInterval;	/* time between two backups in seconds */
	String			FontCommand,	/* commands for file loading */
					FileCommand,
					ElementCommand,
					SaveCommand,	/* and saving */
					FontFile,		/* name of default font file */
					Groups,			/* string with layergroups */
					FilePath,		/* colon seperated search paths */
					FontPath,
					ElementPath;
	LayerGroupType	LayerGroups;	/* default layer groups */
	Boolean			UsePolygonLines,/* see PCBType */
					UseBackingStore,/* use backing store for main window,
									 * costs lots of memory
									 */
					SaveLastCommand,/* save the last command entered by user */
					SaveInTMP,				/* always save data in /tmp */
					ResetAfterElement,		/* reset connections after */
											/* each element */
					RingBellWhenFinished;	/* flag if a signal should be produced
											 * when searching of connections is done
											 */
	XFontStruct		*PinoutFont;	/* font ID used for pin names */
} SettingType, *SettingTypePtr;

#endif
