/* This file is part of
 * ======================================================
 * 
 *           LyX, the High Level Word Processor
 * 	 
 *	  Copyright (C) 1995, 1996 Matthias Ettrich
 *
 *======================================================*/

#include "config.h"
#include "table.h"

#define WIDTH_OF_LINE 5

/* konstruktor */
LyXTable::LyXTable(int rows_arg, int columns_arg) {
   Init(rows_arg, columns_arg);
}

LyXTable::LyXTable(FILE* file) {
  Read(file);
}


LyXTable::~LyXTable() {
   int i;
   delete[] alignment;
   delete[] top_line;
   delete[] bottom_line;
   delete[] left_line;
   delete[] right_line;
   for (i=0; i<rows; i++) {
	   delete[] cell_info[i]; // verify that this shoudn't be freed with delete
   }
   delete[] cell_info;
   delete[] width_of_column;
}


LyXTable* LyXTable::Clone(){
  LyXTable *result = new LyXTable(rows, columns);
  int row, column;;

  for (row=0; row<rows; row++){
    for (column=0; column<columns;column++){
      result->cell_info[row][column] = cell_info[row][column];
    }
  }

  for (row=0; row<rows; row++){
    result->top_line[row] = top_line[row];
    result->bottom_line[row] = bottom_line[row];
  }

  for (column=0; column<columns; column++){
    result->left_line[column] = left_line[column];
    result->right_line[column] = right_line[column];
    result->alignment[column] = alignment[column];
  }
  
  return result;
}


/* activates all lines and sets all widths to 0 */ 
void LyXTable::Init(int rows_arg, int columns_arg) {
  int i,j;
  rows = rows_arg;
  columns = columns_arg;
  alignment = new char[columns];
  top_line = new char[rows];
  bottom_line = new char[rows];
  left_line = new char[columns];
  right_line = new char[columns];
  width_of_column = new int[columns];
  cell_info = new cellstruct*[rows];
  for (i=0; i<rows; i++) {
    cell_info[i] = new cellstruct[columns];
  }
  
   for (i=0; i<rows;i++) {
      top_line[i] = 1;
      bottom_line[i] = 0;
   }
   bottom_line[i-1] = 1;
   bottom_line[0] = 1;
   
   for (i=0; i<rows;i++) {
      for (j=0; j<columns; j++) {
	 cell_info[i][j].width_of_cell = 0;
	 cell_info[i][j].multicolumn = LYX_TABLE_CELL_NORMAL;
	 cell_info[i][j].alignment = LYX_ALIGN_CENTER;
	 cell_info[i][j].top_line = top_line[i];
	 cell_info[i][j].bottom_line = bottom_line[i];
      }
   }

   for (i=0; i<columns;i++) {
      left_line[i] = 1;
      right_line[i] = 0;
   }
   right_line[i-1] = 1;

   for (i=0; i<columns;i++) {
      alignment[i] = LYX_ALIGN_CENTER;
      // set width_of_column to zero before it is used in
      // calculate_width_of_column() (thornley)
      width_of_column[i] = 0;
      calculate_width_of_column(i);
   }
   
   calculate_width_of_table();
}

void LyXTable::AppendRow(int cell) {
   int row = row_of_cell(cell);
   char* top_line2 = new char[rows+1];
   char* bottom_line2 = new char[rows+1];
   cellstruct** cell_info2 = new cellstruct*[rows+1];
   int i;
   for (i=0; i <= row; i++) {
      top_line2[i] = top_line[i];
      bottom_line2[i] = bottom_line[i];
      cell_info2[i] = cell_info[i];
   }
   for (i=rows-1; i >= row; i--) {
      top_line2[i+1] = top_line[i];
      bottom_line2[i+1] = bottom_line[i];
      cell_info2[i+1] = cell_info[i];
   }

   cell_info2[row+1] = new cellstruct[columns];
   for (i=0; i<columns; i++) {
      cell_info2[row+1][i].width_of_cell = 0;
      cell_info2[row+1][i] = cell_info2[row][i];
   }
   
   delete[] top_line;
   top_line = top_line2;
   delete[] bottom_line;
   bottom_line = bottom_line2;
   delete[] cell_info;
   cell_info = cell_info2;
   
   rows++;
   
   Reinit();
}

void LyXTable::DeleteRow(int cell) {
   int row = row_of_cell(cell);
   char* top_line2 = new char[rows-1];
   char* bottom_line2 = new char[rows-1];
   cellstruct** cell_info2 = new cellstruct*[rows-1];
   int i;
   delete[] cell_info[row];
   for (i=0; i < row; i++) {
      top_line2[i] = top_line[i];
      bottom_line2[i] = bottom_line[i];
      cell_info2[i] = cell_info[i];
   }
   for (i=row; i < rows - 1; i++) {
      top_line2[i] = top_line[i+1];
      bottom_line2[i] = bottom_line[i+1];
      cell_info2[i] = cell_info[i+1];
   }
   

   delete[] top_line;
   top_line = top_line2;
   delete[] bottom_line;
   bottom_line = bottom_line2;
   delete[] cell_info;
   cell_info = cell_info2;
   
   rows--;

   Reinit();
}



void LyXTable::AppendColumn(int cell) {
   int i,j;
   char* alignment2 = new char[columns+1];
   char* left_line2 = new char[columns+1];
   char* right_line2 = new char[columns+1];
   
   
   int column = right_column_of_cell(cell);
   
   for (i=0; i<=column; i++){
      alignment2[i] = alignment[i];
      left_line2[i] = left_line[i];
      right_line2[i] = right_line[i];
   }
   for (i=columns-1; i>=column; i--){
      alignment2[i+1] = alignment[i];
      left_line2[i+1] = left_line[i];
      right_line2[i+1] = right_line[i];
   }
   
   
   
   delete[] alignment;
   alignment = alignment2;
   delete[] left_line;
   left_line = left_line2;
   delete[] right_line;
   right_line = right_line2;


   for (i=0; i<rows;i++){
     cellstruct* tmp = cell_info[i];
     cell_info[i] = new cellstruct[columns+1];
     for (j=0; j<=column; j++){
       cell_info[i][j]=tmp[j];
     }
     for (j=column; j<columns; j++){
       cell_info[i][j+1]=tmp[j];
     }
     // care about multicolumns
     if (cell_info[i][column+1].multicolumn==LYX_TABLE_CELL_BEGIN_OF_MULTICOLUMN){
       cell_info[i][column+1].multicolumn = LYX_TABLE_CELL_PART_OF_MULTICOLUMN;
     }
     if (column + 1 == columns ||
	 cell_info[i][column+2].multicolumn!=LYX_TABLE_CELL_PART_OF_MULTICOLUMN){
       cell_info[i][column+1].multicolumn = LYX_TABLE_CELL_NORMAL;
     }
     
     delete[] tmp;
   }

   
   columns++;
   
   delete[] width_of_column;
   width_of_column = new int[columns];

   Reinit();
}

void LyXTable::Reinit(){   
  int i,j;
  for (i=0; i<rows;i++) {
    for (j=0; j<columns; j++) {
      cell_info[i][j].width_of_cell = 0;
    }
  }
  
  for (i=0; i<columns;i++) {
    calculate_width_of_column(i);
  }
  calculate_width_of_table();
}

void LyXTable::DeleteColumn(int cell) {
   
   int column1 = column_of_cell(cell);
   int column2 = right_column_of_cell(cell);
   int column;
   
   if (column1 == 0 && column2 == columns - 1)
     return;
   
   for (column = column1; column <= column2;column++){
     delete_column(column1);
   }
}


int LyXTable::GetNumberOfCells(){
  int result = 0;
  int row, column;
  
  for (row=0; row<rows; row++){
    for (column=0; column<columns; column++){
      if (cell_info[row][column].multicolumn != LYX_TABLE_CELL_PART_OF_MULTICOLUMN)
	result++;
    }
  }
  return result;
}

int LyXTable::NumberOfCellsInRow(int cell)
{
  int row = row_of_cell(cell);
  int i=0;
  int result = 0;
  for (i=0; i<columns; i++){
    if (cell_info[row][i].multicolumn != LYX_TABLE_CELL_PART_OF_MULTICOLUMN)
      result++;
  }
  return result;
}



int LyXTable::AppendCellAfterCell(int append_cell, int question_cell){
   return (right_column_of_cell(append_cell) == 
	   right_column_of_cell(question_cell));
}

int LyXTable::DeleteCellIfColumnIsDeleted(int cell, int delete_column_cell){
   if (column_of_cell(delete_column_cell) == 0 && 
       right_column_of_cell(delete_column_cell) == columns - 1)
     return 0;
   else
     return (
	     column_of_cell(cell) >= column_of_cell(delete_column_cell)
	     &&
	     column_of_cell(cell) <= right_column_of_cell(delete_column_cell)
	     );
}

/* returns 1 if there is a topline, returns 0 if not */ 
char LyXTable::TopLine(int cell){
  if (IsMultiColumn(cell))
    return cellinfo_of_cell(cell)->top_line;
  else
    return top_line[row_of_cell(cell)];
}

char LyXTable::BottomLine(int cell){
  if (IsMultiColumn(cell))
    return cellinfo_of_cell(cell)->bottom_line;
  else
    return bottom_line[row_of_cell(cell)];
}

char LyXTable::LeftLine(int cell){
   return left_line[column_of_cell(cell)];
}

char LyXTable::RightLine(int cell){
   return right_line[right_column_of_cell(cell)];
}


char LyXTable::TopAlreadyDrawed(int cell){
  if (AdditionalHeight(cell))
     return 0;
  int row = row_of_cell(cell);
  if (row > 0){
    int column = column_of_cell(cell);
    while (column && cell_info[row-1][column].multicolumn == LYX_TABLE_CELL_PART_OF_MULTICOLUMN)
      column--;
    if (cell_info[row-1][column].multicolumn == LYX_TABLE_CELL_NORMAL)
      return bottom_line[row-1];
    else
      return cell_info[row-1][column].bottom_line;
  }
  return 0;
}


char LyXTable::VeryLastRow(int cell){
//   if (row_of_cell(cell) == rows-1)
//     return 1;
//   cell++;
//   while (!IsFirstCell(cell))
//     cell++;
//   return AdditionalHeight(cell);
  return (row_of_cell(cell) == rows-1);
}


// int LyXTable::AdditionalHeight(int cell){
//    int row = row_of_cell(cell);
//    int top = 0;
//    int bottom = 0;
//    int column;
//    if (row){
//      for (column=0;column < columns-1 && !bottom;column++){
//        switch (cell_info[row-1][column].multicolumn){
//        case LYX_TABLE_CELL_BEGIN_OF_MULTICOLUMN:
// 	 bottom = cell_info[row-1][column].bottom_line;
// 	 break;
//        case LYX_TABLE_CELL_NORMAL:
// 	 bottom = bottom_line[row-1];
//        }
//      }
//      for (column=0;column < columns-1 && !top;column++){
//        switch (cell_info[row][column].multicolumn){
//        case LYX_TABLE_CELL_BEGIN_OF_MULTICOLUMN:
// 	 top = cell_info[row][column].top_line;
// 	 break;
//        case LYX_TABLE_CELL_NORMAL:
// 	 top = top_line[row];
//        }
//      }
//    }
//    if (top && bottom)
//      return WIDTH_OF_LINE;
//    else
//      return 0;
// }

int LyXTable::AdditionalHeight(int cell){
   int row = row_of_cell(cell);
   int top = 1;
   int bottom = 1;
   int column;
   if (row){
     for (column=0;column < columns-1 && bottom;column++){
       switch (cell_info[row-1][column].multicolumn){
       case LYX_TABLE_CELL_BEGIN_OF_MULTICOLUMN:
	 bottom = cell_info[row-1][column].bottom_line;
	 break;
       case LYX_TABLE_CELL_NORMAL:
	 bottom = bottom_line[row-1];
       }
     }
     for (column=0;column < columns-1 && top;column++){
       switch (cell_info[row][column].multicolumn){
       case LYX_TABLE_CELL_BEGIN_OF_MULTICOLUMN:
	 top = cell_info[row][column].top_line;
	 break;
       case LYX_TABLE_CELL_NORMAL:
	 top = top_line[row];
       }
     }
     if (top && bottom)
       return WIDTH_OF_LINE;
   }
   return 0;
}



int LyXTable::AdditionalWidth(int cell){
  // internally already set in SetWidthOfCell
  // used to get it back in text.C
  int col = right_column_of_cell(cell);
  if (col < columns - 1 && right_line[col] && left_line[col+1])
    return WIDTH_OF_LINE;
  else
    return 0;
}


/* returns the maximum over all rows */ 
int LyXTable::WidthOfColumn(int cell){
  int column1 = column_of_cell(cell);
  int column2 = right_column_of_cell(cell);
  int i;
  int result=0;
  for (i=column1; i<=column2;i++){
    result += width_of_column[i];
  }
  return result;
}
int LyXTable::WidthOfTable(){
   return width_of_table;
}

/* returns 1 if a complete update is necessary, otherwise 0 */ 
int LyXTable::SetWidthOfCell(int cell, int new_width){
   int row = row_of_cell(cell);
   int column1 = column_of_cell(cell);
   int column2 = right_column_of_cell(cell);
   int tmp=0;
   int i;
   int width = (new_width + 2*WIDTH_OF_LINE) / (column2 - column1 + 1);
   
   for (i=column1; i<=column2;i++){
     cell_info[row][i].width_of_cell = width;
     if (i==column2 && right_line[i] && i< columns-1 && left_line[i+1])
       cell_info[row][i].width_of_cell += WIDTH_OF_LINE; // additional width
     tmp += calculate_width_of_column(i);
   }
   
   if (tmp){
     calculate_width_of_table();
     return 1;
   }
   else
     return 0;
}

int LyXTable::SetAlignment(int cell, char align){
  if (!IsMultiColumn(cell))
    alignment[column_of_cell(cell)] = align;
  cellinfo_of_cell(cell)->alignment = align;
  return 1;
}

int LyXTable::SetTopLine(int cell, char line){
  if (!IsMultiColumn(cell))
    top_line[row_of_cell(cell)] = line;
  else
    cellinfo_of_cell(cell)->top_line = line;
  return 1;
}

int LyXTable::SetBottomLine(int cell, char line){
  if (!IsMultiColumn(cell))
    bottom_line[row_of_cell(cell)] = line;
  else
    cellinfo_of_cell(cell)->bottom_line = line;
  return 1;
}

int LyXTable::SetLeftLine(int cell, char line){
   left_line[column_of_cell(cell)] = line;
   return 1;
}

int LyXTable::SetRightLine(int cell, char line){
  right_line[right_column_of_cell(cell)] = line;
  return 1;
}


char LyXTable::GetAlignment(int cell){
  if (IsMultiColumn(cell))
    return cellinfo_of_cell(cell)->alignment;
  else
    return alignment[column_of_cell(cell)];
}

int LyXTable::GetWidthOfCell(int cell){
  int row = row_of_cell(cell);
  int column1 = column_of_cell(cell);
  int column2 = right_column_of_cell(cell);
  int i;
  int result=0;
  for (i=column1; i<=column2;i++){
    result += cell_info[row][i].width_of_cell;
  }
  
  result += AdditionalWidth(cell);
  
   return result;
}



int LyXTable::GetBeginningOfTextInCell(int cell){
   
   int x = 0;
   
   switch (GetAlignment(cell)){
   case LYX_ALIGN_CENTER:
      x += (WidthOfColumn(cell) - GetWidthOfCell(cell)) / 2;
      break;
   case LYX_ALIGN_RIGHT:
      x += WidthOfColumn(cell) - GetWidthOfCell(cell) - AdditionalWidth(cell);
      break;
    default: /* LYX_ALIGN_LEFT: nothing :-) */ 
      break;
   }

   // the LaTeX Way :-(
   x += WIDTH_OF_LINE;
   return x;
}



char LyXTable::IsFirstCell(int cell) {
   return (column_of_cell(cell) == 0);
}



char LyXTable::calculate_width_of_column(int column) {
   int i, max;
   int old_column_width = width_of_column[column];
   max = 0;
   for (i=0; i<rows; i++) {
     if (cell_info[i][column].width_of_cell > max) {
       max = cell_info[i][column].width_of_cell;
     }
   }
   width_of_column[column] = max;
   return (width_of_column[column] != old_column_width);
}

void LyXTable::calculate_width_of_table() {
   int i;
   width_of_table = 0;
   for (i=0; i<columns;i++) {
      width_of_table += width_of_column[i];
   }
}

int LyXTable::row_of_cell(int cell) {
  // return cell/columns;

  int row = 0;
  int c= 0;
  int column = 0;
  
  while (c < cell && row < rows && column < columns)
    {
      c++;
      do{
	column++;
      } while (column < columns &&
	       cell_info[row][column].multicolumn == LYX_TABLE_CELL_PART_OF_MULTICOLUMN);
      if (column == columns){
	column = 0;
	row++;
      }
    }
 return row; 
}

int LyXTable::column_of_cell(int cell) {
  // return cell - (row_of_cell(cell) * columns);
  int row = 0;
  int c= 0;
  int column = 0;
  
  while (c < cell && row < rows && column < columns)
    {
      c++;
      do{
	column++;
      } while (column < columns &&
	       cell_info[row][column].multicolumn == LYX_TABLE_CELL_PART_OF_MULTICOLUMN);
      if (column == columns){
	column = 0;
	row++;
      }
    }
 return column; 
}

int LyXTable::right_column_of_cell(int cell) {
  int row = row_of_cell(cell);
  int column = column_of_cell(cell);
  while (column < columns - 1 &&
	 cell_info[row][column+1].multicolumn == LYX_TABLE_CELL_PART_OF_MULTICOLUMN)
    column++;
  return column;
}


void LyXTable::Write(FILE* file){
   int i,j;
   fprintf(file, "multicol2\n");
   fprintf(file, "%d %d\n", rows, columns);
   for (i=0; i<rows; i++){
      fprintf(file, "%d %d\n", top_line[i], bottom_line[i]);
   }
   for (i=0; i<columns; i++){
      fprintf(file, "%d %d %d\n", alignment[i], left_line[i], right_line[i]);
   }

   for (i=0; i<rows;i++){
     for (j=0;j<columns;j++){
       fprintf(file, "%d %d %d %d\n", cell_info[i][j].multicolumn, cell_info[i][j].alignment,
	      cell_info[i][j].top_line, cell_info[i][j].bottom_line );
     }
   }
     
}

void LyXTable::Read(FILE* file){
   int i,j;
   int rows_arg = 0;
   int columns_arg = 0;
   char s[100];
   int a = 0;
   int b = 0;
   int c = 0;
   int d = 0;
   fscanf(file, "%s", s);
   fscanf(file, "%d %d\n", &rows_arg, &columns_arg);
   Init(rows_arg, columns_arg);
   for (i=0; i<rows; i++){
     fscanf(file, "%d %d\n", &a, &b);
     top_line[i] = (char) a;
     bottom_line[i] = (char) b;
   }
   for (i=0; i<columns; i++){
     fscanf(file, "%d %d %d\n", &a, &b, &c);
     alignment[i] = (char) a;
     left_line[i] = (char)b;
     right_line[i] = (char)c;
   }

   if (StringEqual(s, "multicol")){
     for (i=0; i<rows;i++){
       for (j=0;j<columns;j++){
	 fscanf(file, "%d %d\n", &a, &b);
	 cell_info[i][j].multicolumn = (char) a;
	 cell_info[i][j].alignment = (char) b;
       }
     }
   }
   if (StringEqual(s, "multicol2")){
     for (i=0; i<rows;i++){
       for (j=0;j<columns;j++){
	 fscanf(file, "%d %d %d %d\n", &a, &b, &c, &d);
	 cell_info[i][j].multicolumn = (char) a;
	 cell_info[i][j].alignment = (char) b;
	 cell_info[i][j].top_line = (char) c;
	 cell_info[i][j].bottom_line = (char) d;
       }
     }
   }
}

// cell <0 will tex the preamble
// returns the number of printed newlines
int LyXTable::TexEndOfCell(FILE* file, int cell){
  int i;
  int ret = 0;
  int tmp, tmp2;
  int fcell;
  if (cell == GetNumberOfCells() - 1){
    // the very end at the very beginning
    if (IsMultiColumn(cell)){
      fprintf(file, "}");
    }
    fprintf(file, "\\\\\n");
    ret++;
    
    tmp = 0;
    fcell = cell; 
    while (!IsFirstCell(fcell))fcell--;
    for (i=0; i < NumberOfCellsInRow(fcell); i++){
      if (BottomLine(fcell+i))
	tmp++;
    }
    if (tmp == NumberOfCellsInRow(fcell)){
      fprintf(file, "\\hline ");
    }
    else {
      tmp = 0;
      for (i=0; i < NumberOfCellsInRow(fcell); i++){
	if (BottomLine(fcell+i)){
	  fprintf(file, "\\cline{%d-%d} ", column_of_cell(fcell+i)+1, 
		  right_column_of_cell(fcell+i)+1);
	  tmp = 1;
	}
      }
    }
    if (tmp){
      fprintf(file, "\n");
      ret++;
    }
    fprintf(file, "\\end{tabular}");
  }
  else {  
    if (cell < 0){
      // preamble
      fprintf(file, "\\begin{tabular}{");
      for (i=0; i<columns;i++){
	if (left_line[i])
	  fprintf(file, "|");
	switch (alignment[i]) {
	case LYX_ALIGN_LEFT: fprintf(file, "l"); break;
	case LYX_ALIGN_RIGHT: fprintf(file, "r"); break;
	default:  fprintf(file, "c"); break;
	}
      if (right_line[i])
	fprintf(file, "|");
      }
      fprintf(file, "}\n");
      ret++;
      tmp = 0;
      if (GetNumberOfCells()){
	fcell = 0;
	for (i=0; i < NumberOfCellsInRow(fcell); i++){
	  if (TopLine(fcell+i))
	    tmp++;
	}
	if (tmp == NumberOfCellsInRow(fcell)){
	  fprintf(file, "\\hline ");
	}
	else {
	  tmp = 0;
	  for (i=0; i < NumberOfCellsInRow(fcell); i++){
	    if (TopLine(fcell+i)){
	      fprintf(file, "\\cline{%d-%d} ", column_of_cell(fcell+i)+1, 
		      right_column_of_cell(fcell+i)+1);
	      tmp = 1;
	    }
	  }
	}
	if (tmp){
	  fprintf(file, "\n");
	  ret++;
	}
      }
    }
    else{
      // usual cells
      if (IsMultiColumn(cell)){
	fprintf(file, "}");
      }
      
      if (right_column_of_cell(cell) == columns -1){
	fprintf(file, "\\\\\n");
	ret++;

	tmp = 0;
	fcell = cell;
	while (!IsFirstCell(fcell))fcell--;
	for (i=0; i < NumberOfCellsInRow(cell); i++){
	  if (BottomLine(fcell+i))
	    tmp++;
	}
	if (tmp == NumberOfCellsInRow(cell)){
	  fprintf(file, "\\hline ");
	}
	else {
	  tmp = 0;
	  for (i=0; i < NumberOfCellsInRow(fcell); i++){
	    if (BottomLine(fcell+i)){
	      fprintf(file, "\\cline{%d-%d} ", column_of_cell(fcell+i)+1, 
		      right_column_of_cell(fcell+i)+1);
	      tmp = 1;
	    }
	  }
	}
	if (tmp){
	  fprintf(file, "\n");
	  ret++;
	}
	tmp = 0;
	if (cell < GetNumberOfCells()-1){
	  fcell = cell+1;
	  while (!IsFirstCell(fcell))fcell--;
	  for (i=0; i < NumberOfCellsInRow(fcell); i++){
	    if (TopLine(fcell+i))
	      tmp++;
	  }
	  if (tmp == NumberOfCellsInRow(fcell)){
	    fprintf(file, "\\hline ");
	  }
	  else {
	    tmp = 0;
	    for (i=0; i < NumberOfCellsInRow(fcell); i++){
	      if (TopLine(fcell+i)){
		fprintf(file, "\\cline{%d-%d} ", column_of_cell(fcell+i)+1, 
			right_column_of_cell(fcell+i)+1);
		tmp = 1;
	      }
	    }
	  }
	  if (tmp){
	    fprintf(file, "\n");
	    ret++;
	  }
	}
      }
      else
	fprintf(file, "&");
    }

    if (IsMultiColumn(cell+1)){
      fprintf(file, "\\multicolumn{%d}{", cells_in_multicolumn(cell+1));
      if (LeftLine(cell+1))
	fprintf(file, "|");
      switch (GetAlignment(cell+1)) {
      case LYX_ALIGN_LEFT: fprintf(file, "l"); break;
      case LYX_ALIGN_RIGHT: fprintf(file, "r"); break;
      default:  fprintf(file, "c"); break;
      }
      if (RightLine(cell+1))
	fprintf(file, "|");
      if (column_of_cell(cell+2)!=0 && LeftLine(cell+2))
	fprintf(file, "|");
      fprintf(file, "}{");
    }
  }
  return ret;
}


char LyXTable::IsMultiColumn(int cell){
  return (cellinfo_of_cell(cell)->multicolumn != LYX_TABLE_CELL_NORMAL);
}

LyXTable::cellstruct* LyXTable::cellinfo_of_cell(int cell){
  int row = row_of_cell(cell);
  int column = column_of_cell(cell);
  return  &cell_info[row][column];
}
   
void LyXTable::SetMultiColumn(int cell, int number){
  cellinfo_of_cell(cell)->multicolumn = LYX_TABLE_CELL_BEGIN_OF_MULTICOLUMN;
  cellinfo_of_cell(cell)->alignment = alignment[column_of_cell(cell)];
  cellinfo_of_cell(cell)->top_line = top_line[row_of_cell(cell)];
  cellinfo_of_cell(cell)->bottom_line = bottom_line[row_of_cell(cell)];
  for (number--;number>0;number--){
    cellinfo_of_cell(cell+1)->multicolumn = LYX_TABLE_CELL_PART_OF_MULTICOLUMN;
  }
}

int LyXTable::cells_in_multicolumn(int cell){
  int row = row_of_cell(cell);
  int column = column_of_cell(cell);
  int result = 1;
  column++;
  while (column < columns && cell_info[row][column].multicolumn
	 == LYX_TABLE_CELL_PART_OF_MULTICOLUMN){
    result++;
    column++;
  }
  return result;
}



int  LyXTable::UnsetMultiColumn(int cell){
  int row = row_of_cell(cell);
  int column = column_of_cell(cell);

  int result = 0;

  if (cell_info[row][column].multicolumn == LYX_TABLE_CELL_BEGIN_OF_MULTICOLUMN){
    cell_info[row][column].multicolumn = LYX_TABLE_CELL_NORMAL;
    column++;
    while (column < columns &&
	   cell_info[row][column].multicolumn == LYX_TABLE_CELL_PART_OF_MULTICOLUMN){
      cell_info[row][column].multicolumn = LYX_TABLE_CELL_NORMAL;
      column++;
      result++;
    }
  }
  return result;
}

void LyXTable::delete_column(int column) {
   
   int i,j;
   char* alignment2 = new char[columns-1];
   char* left_line2 = new char[columns-1];
   char* right_line2 = new char[columns-1];
   
   for (i=0; i<column; i++){
      alignment2[i] = alignment[i];
      left_line2[i] = left_line[i];
      right_line2[i] = right_line[i];
   }
   for (i=column; i<columns-1; i++){
      alignment2[i] = alignment[i+1];
      left_line2[i] = left_line[i+1];
      right_line2[i] = right_line[i+1];
   }
   
   delete[] alignment;
   alignment = alignment2;
   delete[] left_line;
   left_line = left_line2;
   delete[] right_line;
   right_line = right_line2;

   for (i=0; i<rows;i++){
     cellstruct* tmp = cell_info[i];
     cell_info[i] = new cellstruct[columns-1];
     for (j=0; j<column; j++){
       cell_info[i][j]=tmp[j];
     }
     for (j=column; j<columns-1; j++){
       cell_info[i][j]=tmp[j+1];
     }
     delete[] tmp;
   }


   
   columns--;
   
   delete[] width_of_column;
   width_of_column = new int[columns];

   Reinit();
}
