#ident "@(#) $Id: PgSQL.py,v 1.4 2001/11/03 18:36:50 ballie01 Exp $"
# vi:set sw=4 ts=8 showmode ai:
#--(H+)-----------------------------------------------------------------+
# Name:		PgSQL.py						|
#									|
# Description:	This file implements a Python DB-API 2.0 interface to	|
#		PostgreSQL.						|
#=======================================================================|
# Copyright 2000 by Billy G. Allie.					|
# All rights reserved.							|
#									|
# Permission to use, copy, modify, and distribute this software and its	|
# documentation for any purpose and without fee is hereby granted, pro-	|
# vided that the above copyright notice appear in all copies and that	|
# both that copyright notice and this permission notice appear in sup-	|
# porting documentation, and that the copyright owner's name not be	|
# used in advertising or publicity pertaining to distribution of the	|
# software without specific, written prior permission.			|
#									|
# THE AUTHOR(S) DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,	|
# INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS.  IN	|
# NO EVENT SHALL THE AUTHOR(S) BE LIABLE FOR ANY SPECIAL, INDIRECT OR	|
# CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS	|
# OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE	|
# OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE	|
# USE OR PERFORMANCE OF THIS SOFTWARE.					|
#=======================================================================|
# Revision History:							|
#									|
# Date      Ini Description						|
# --------- --- ------------------------------------------------------- |
# 03NOV2001 bga	It appears that under certain circumstances, PostgreSQL	|
#		will not return a precision/scale value for a numeric	|
#		column when no result rows are returned [Bug #477792].	|
#		The problem is fixed by using a precision of (30,6) in	|
#		case this condition occurs.				|
# 21OCT2001 gh  Change the import of DateTime to avoid conflicts with	|
# 		ZOPE's builtin DateTime module.				|
# 17OCT2001 bga Added code to ensure that creation of a binary object	|
#		via the binary() method always occurs within the con-	|
#		text of a transaction.					|
# 27SEP2001 bga	Change code so that the escaping/quoting is different	|
#		if the result is to be used to build PostgreSQL arrays.	|
#	    ---	Modified all the Object._quote methods so that they now	|
#		accept an option argument, forArray, which if set to 1	|
#		will cause escaping/quoting for PostgreSQL array use.	|
#	    --- Re-organized the code use to build PostgreSQL arrays.	|
# 26SEP2001 bga Added additional PostgreSQL types to BINARY, ROWID, etc	|
#	    --- Fixed problems associated with type casting.		|
# 24SEP2001 bga Changed the method used to quote strings.  This new way	|
#		does not require the services of pgFixEsc, which was	|
#		removed from pgconnection.c.				|
#	    --- Added support for the PostgreSQL type BYTEA.		|
#	    --- Improved support for the BYTEA type.			|
# 19SEP2001 bga Cleaned up the logic in hadleArray().			|
#	    --- Updated the PgSQL.__doc__ string.			|
# 18SEP2001 bga	[Feature Request #462588] Cursor.close() no longer ends	|
#		(rollback) the open transaction when the last cursor is	|
#		closed.							|
#	    --- On Connection.close(), reset all the attributes to a	|
#		sane state (i.e. None or 0, as appropiate).		|
#	    --- [Bug #462589] In a fit of optimization, I had intro-	|
#		duced a bug that caused comparisons of PgResultSets to	|
#		anything to fail.  This bug has been squashed.		|
# 03SEP2001 gh  Fixed several cases of references to non-existing	|
#		globals in PgMoney.					|
# 30AUG2001 bga	Previously, I was not building up the description	|
#		attibute if no rows were returned.  The description	|
#		attribute is now built on the first successful select	|
#		or cursor fetch.					|
# 23AUG2001 bga [Bug #454653] Change the code to rollback an open trans-|
#		action, if any, when the last cursor of a connection is	|
#		closed.							|
#	    --- Added code that will, if weak references are not avail-	|
#		able, remove any cursors that are only referenced from	|
#		the connection.cursors list.  This is in effect garbage	|
#		collection for deleted cursors.  The gabage collection	|
#		of orphaned cursor will occur at the following points:	|
#		1.  After a new cursor is created.			|
#		2.  When a cursor is closed.				|
#		3.  When the connection.autocommit value is changed.	|
#	    --- Changed cursor.isClosed to cursor.closed.  Why, closed	|
#		is used in other object (file and PgLargeObject, for	|
#		example) and I thought I would follow the trend (i.e.	|
#		for no good reason).					|
#	    --- Change from the use of hash() to generate the portal	|
#		name to the use of id().				|
# 10AUG2001 bga [Bug #449743]Added code to trap a failure to connect to	|
#		a database and delete the Connection object on failure.	|
#		This resolves the problem reported by Adam Buraczewski.	|
#	    --- Fixed a bug in fetchmany() that could return more than	|
#		the requested rows if PostgreSQL Portals aren't used in	|
#		the query.						|
# 03AUG2001 bga The PgVer object is now implemented in C (as PgVersion)	|
#		and been moved to the libpq module.  This was done to	|
#		support handling of large objects in libpq based on the	|
#		PostgreSQL version.					|
# 01AUG2001 bga Changed the code so that execute will not attempt to do	|
#		parameter subsitution on the query string if no addi-	|
#		tional parameters are passed to execute.		|
#		Modify PgSQL to reflect that fact.			|
# --------- bga Remove prior comments to reduce the size of the flower	|
#		box.  See revision 1.19 for earlier comments.		|
#--(H-)-----------------------------------------------------------------+
"""
    PgSQL - A PyDB-SIG 2.0 compliant module for PostgreSQL.

    Copyright 2000 by Billy G. Allie <Bill.Allie@mug.org>
    See package documentation for further information on copyright.

    Inline documentation is sparse.
    See the Python DB-SIG 2.0 specification for usage information.

    basic usage:

	PgSQL.connect(connect_string) -> connection
	    connect_string = 'host:port:database:user:password:options:tty'
	    All parts are optional. You may also pass the information in as
	    keyword arguments with the following keywords: 'host', 'port',
	    'database', 'user', 'password', 'options', and 'tty'.  The port
	    may also be passed in as part of the host keyword parameter,
	    ie. host='localhost:5432'.

	connection.cursor() -> cursor
	    Create a new cursor object.  A connection can support multiple
	    cursors at the same time.

	connection.close()
	    Closes the connection now (instead of when __del__ is called).
	    The connection will be unusable from this point forward.
	    NOTE: Any uncommited transactions will be rolled back and any
	          open cursors for this connection will be closed.

	connection.commit()
	    Commit any pending transactions for this connection.

	    NOTE: This will reset any open cursors for this connection to their
		  inital state.  Any PostgreSQL portals in use by cursors will
		  be closed and any remaining query results will be discarded.

	connection.rollback()
	    Rollback any pending transactions for this connection.

	    NOTE: This will reset any open cursors for this connection to their
		  inital state.  Any PostgreSQL portals in use by cursors will
		  be closed and any remaining query results will be discarded.

	connection.binary(string) -> PgLargeObject
	    Create a new PostgreSQL large object.  If string is present, it is
	    written to the new large object.  The returned large object will
	    not be opened (i.e. it will be closed).
	
	connection.un_link(OID|PgLargeObject)
	    Un-links (removes) the PgLargeObject from the database.

	    NOTE: This is a PostgreSQL extension to the Connection object.
		  It is not safe to un-link PgLargeObjects while in a trans-
		  action (in versions prior to 7.1) since this action can not
		  be rollbacked completely.  Therefore, an attempt to un-link
		  while in a transaction (in versions prior to 7.1) will raise
		  an exception.

        connection.version
            This instance of the PgVersion class contains information about
            the version of the PostgreSQL backend to which the connection
	    object is connected to.

            NOTE: This is a PgSQL extension to the Connection object.

	cursor.execute(query[, param1[, param2, ..., paramN])
	    Execute a query, binding the parameters if they are passed. The
	    binding syntax is the same as the '%' operator except that only %s
	    [or %(name)s] should be used and the %s [or %(name)s] should not be
	    quoted.  Any necessary quoting will be performed by the execute
	    method.

	cursor.execute(query[, sequence])
	    Execute a query, binding the contents of the sequence as parameters.
	    The binding syntax is the same as the '%' operator except that only
	    %s should be used and the %s should not be quoted.  Any necessary
	    quoting will be performed by the execute method.

	cursor.execute(query[, dictionary])
	    Execute a query, binding the contents of the dictionary as para-
	    meters.  The binding syntax is the same as the '%' operator except
	    that only %s [or %(name)s] should be used and the %s [or %(name)s]
	    should not be quoted.  Any necessary quoting will be performed by
	    the execute method.

	NOTE: In order to use a PostgreSQL portal (i.e. DECLARE ... CURSOR
              FOR ...), the word SELECT must be the first word in the query,
              and can only be be proceeded by spaces and tabs.  If this is not
              the case, a cursor will be simulated by reading the entire result
	      set into memory and handing out the results as needed.

	NOTE: PostgreSQL cursors are read-only.  SELECT ... FOR UPDATE queries
	      will not use PostgreSQL cursors, but will simulate a cursor by
	      reading the entire result set into memory and handing out the
	      results as needed.

        NOTE: Setting the variable, PgSQL.noPostgresCursor, to 1 will cause
              PgSQL to NOT use PostgreSQL cursor (DECLARE ... CURSOR FOR ...),
              even if all the conditions for doing so are met.  PgSQL will
              simulate a cursor by reading the entire result set into memory
              and handing out the results as needed.
              
	cursor.executemany(query, sequence_of_params)
	    Execute a query many times, once for each params in the sequence.

	NOTE: The restriction on the use of PostgreSQL cursors described in
	      the cursor.execute() note also applies to cursor.executemany().
	      The params in the sequence of params can be a list, tuple or
	      dictionary.

	cursor.fetchone() -> PgResultSet
	    Fetch the next row of the query result set as a single PgResultSet
	    containing the column data.  Returns None when no more rows are
	    available.  A PgResultSet is a sequence that can be indexed by
	    a column name in addition to an integer.

	cursor.fetchmany([size]) -> [PgResultSet, ...]
	    Fetch the next set of rows from the query result, returning a
	    sequence of PgResultSets.  An empty sequence is returned when
	    no more rows are available.

	    The number of rows returned is determined by the size parameter.
	    If the size parameter is ommited, the value of cursor.arraysize
	    is used.  If size is given, cursor.arraysize is set to that value.

	cursor.fetchall() -> [PgResultSet, ...]
	    Fetch all (remaining) rows from the query result, returning a
	    sequence of PgResultSets.  An empty sequence is returned when
	    no more rows are available.

	cursor.description -> [(column info), ... ]
	    Returns a sequence of 8-item tuples.  Each tuple describes one
	    column of the result: (name, type code, display size, internal
	    size, precision, scale, null_ok, isArray).

	    NOTE: null_ok is not implemented.
	          isArray is a PostgreSQL specific extension.

	cursor.rowcount
	    The number of rows the last execute produced (for DQL statements)
	    or affected (for DML statement).

        cursor.oidValue
	    The object ID of the inserted record, if the last SQL command
	    was an INSERT, otherwise it returns 0 (aka. InvalidOid)

            NOTE: oidValue is a PostgreSQL specific extension.

	cursor.close()
	    Closes the cursor now (instead of when __del__ is called).  The
	    cursor will be unusable from this point forward.

	cursor.rewind()
	    Moves the cursor back to the beginning of the query result.
	    This is a PgSQL extension to the PyDB 2.0 API.

	PgResultSet.description() -> [(column info), ... ]
	    Returns a sequence of 8-item tuples.  Each tuple describes one
	    column of the result: (name, type code, display size, internal
	    size, precision, scale, null_ok. isArray).

	    NOTE: null_ok is not implemented.
	          isArray is a PostgreSQL specific extension.

	PgResultSet.<column name> -> value
	    Column names are attributes to the PgResultSet.
	
	Note:	Setting the variable, PgSQL.fetchReturnsList = 1 will cause
		the fetch*() methods to return a list instead of a PgResultSet.

        PgSQL.version
            This string object contains the version number of PgSQL.
"""

from types import *
from sys import getrefcount
import string
import re

try:
    import weakref
    noWeakRef = 0
except:
    noWeakRef = 1

try:
    from mx import DateTime
except ImportError:
    import DateTime
from libpq import *

version = '$Revision: 1.4 $'

apilevel = '2.0'
threadsafety = 1
paramstyle = 'pyformat'

# Setting this variable to 1 will cause the fetch*() methods to return
# a list instead of a PgResultSet

fetchReturnsList = 0

# Setting this variable to 1 will prevent the use of a PostgreSQL Cursor
# (via the "DECLARE ... CURSOR FOR ..." syntax).  A cursor will be simulated
# by reading all of the query result into memory and doling out the results
# as needed.

noPostgresCursor = 0

re_DQL = re.compile('^[\s]*SELECT[\s]', re.I)
re_DRT = re.compile('[\s]*DROP[\s]+TABLE[\s]', re.I)
re_DRI = re.compile('[\s]*DROP[\s]+INDEX[\s]', re.I)
re_4UP = re.compile('[\s]FOR[\s]+UPDATE', re.I)

replace = string.replace

#-----------------------------------------------------------------------+
# Make the required Date/Time constructor visable in the PgSQL module.	|
#-----------------------------------------------------------------------+

Date = DateTime.Date
Time = DateTime.Time
Timestamp = DateTime.Timestamp
DateFromTicks = DateTime.DateFromTicks
TimeFromTicks = DateTime.TimeFromTicks
TimestampFromTicks = DateTime.TimestampFromTicks

#-------------------------------+
# And also the DateTime types	|
#-------------------------------+

DateTimeType = DateTime.DateTimeType
DateTimeDeltaType = DateTime.DateTimeDeltaType

#-----------------------------------------------------------------------+
# Name:		DBAPITypeObject						|
#									|
# Description:  The DBAPITypeObject class allows implementing the re-	|
#		quired DP-API 2.0 type objects even if their are multi-	|
#		ple database types for the DP-API 2.0 types.		|
#									|
# Note:		This object is taken from the Python DP-API 2.0 imple-	|
#		mentation hints.					|
#-----------------------------------------------------------------------+

class DBAPITypeObject:
    def __init__(self, name, *values):
	self.name = name
	self.values = values

    def __repr__(self):
	return self.name

    def __cmp__(self, other):
	if other in self.values:
	    return 0
	elif other < self.values:
	    return 1
	return -1

# Define the object types required by the DB-API 2.0 specification.

BINARY	 = DBAPITypeObject('BINARY', PG_OID, PG_BLOB, PG_BYTEA)

DATETIME = DBAPITypeObject('DATETIME', PG_DATE, PG_TIME, PG_TIMESTAMP,
				       PG_ABSTIME, PG_RELTIME, PG_INTERVAL,
				       PG_TINTERVAL)

NUMBER	 = DBAPITypeObject('NUMBER', PG_INT8, PG_INT2, PG_INT4, PG_FLOAT4,
				     PG_FLOAT8, PG_MONEY, PG_NUMERIC)

ROWID	 = DBAPITypeObject('ROWID', PG_OID, PG_ROWID, PG_CID, PG_TID, PG_XID)

STRING	 = DBAPITypeObject('STRING', PG_CHAR, PG_BPCHAR, PG_TEXT, PG_VARCHAR,
				     PG_NAME)

# BOOLEAN is the PostgreSQL boolean type.

BOOLEAN  = DBAPITypeObject('BOOLEAN', PG_BOOL)

# OTHER is for PostgreSQL types that don't fit in the standard Objects.

OTHER	 = DBAPITypeObject('OTHER', PG_POINT, PG_LSEG, PG_PATH, PG_BOX,
				    PG_POLYGON, PG_LINE, PG_CIDR, PG_CIRCLE,
				    PG_INET)

#-----------------------------------------------------------------------+
# Name:		PgTypes							|
#									|
# Description:	PgTypes is an object wrapper for the type OID's used by	|
#		PostgreSQL.  It is used to display a meaningful text	|
#		description of the type while still allowing it to be	|
#		compared as a numeric value.				|
#-----------------------------------------------------------------------+

class PgTypes:
    def __init__(self, value):
	self.value = value

    def __coerce__(self, other):
	if type(other) in [IntType, LongType, FloatType]:
	    return (self.value, int(other))
	return None

    def __cmp__(self, other):
	if self.value < other:
	    return -1
	elif self.value == other:
	    return 0
	else:
	    return 1

    def __repr__(self):
	return PQftypeName(self.value)

    def __prt__(self):
	return PQftypeName(self.value)

    def __int__(self):
	return int(self.value)

    def __long__(self):
	return long(self.value)

    def __float__(self):
	return float(self.value)

    def __complex__(self):
	return complex(self.value)

#-----------------------------------------------------------------------+
# Name:		TypeCache						|
#									|
# Description:	TypeCache is an object that defines methods to:		|
#		1.  Cast PostgreSQL result strings into the appropiate	|
#		    Python type or object [typecast()].			|
#		2.  Retrieve addition information about a type from the	|
#		    PostgreSQL system catalogs [getTypeInfo()].  This	|
#		    type information is maintained in a local cache so	|
#		    that subsequent request for the same information do	|
#		    not require a database query to fulfill it.		|
#-----------------------------------------------------------------------+

class TypeCache:
    """Type cache -- used to cache postgreSQL data type information."""

    def __init__(self, conn):
	if noWeakRef:
	    self.__conn = conn
	else:
	    self.__conn = weakref.proxy(conn, self.__callback)
	self.__type_cache = {}
	self.__lo_cache = {}
    
    def __callback(self, o):
	self.__conn = None

    def typecast(self, colinfo, value):
	"""
	typecast(rowinfo, value)
	    Convert certain postgreSQL data types into the appropiate python
	    object."""
    
        if value == None:
	    return value

        _fn, _ft, _ds, _is, _p, _s, _nu, _ia = colinfo

	_ftv = _ft.value
	
	if _ia:
	    # Convert string representation of the array into list.
	    exec "_list = %s" % replace(replace(value, '{', '['), '}', ']')
	    return self.handleArray(colinfo, _list)

	if _ftv == PG_INT2:
	    if type(value) == PgInt2Type:
		return value
	    else:
		return PgInt2(value)
	elif _ftv == PG_INT4 or _ftv == ROWID:
	    if type(value) == IntType:
		return value
	    else:
		return int(value)
	elif _ftv == PG_INT8:
	    if type(PgInt8) == ClassType:
		if isinstance(value, PgInt8):
		    return value
	    else:
		if type(value) == PgInt8Type:
		    return value
	    return PgInt8(value)
	elif _ftv == PG_NUMERIC:
	    if isinstance(value, PgNumeric):
		return value
	    else:
		return PgNumeric(value, _p, _s)
	elif _ftv == PG_MONEY:
	    if isinstance(value, PgMoney):
		return value
	    else:
		return PgMoney(value)
	elif _ftv == DATETIME:
	    if type(value) in [ DateTimeType, DateTimeDeltaType ]:
		return value
	    else:
		return DateTime.ISO.ParseAny(value)
	elif _ftv == BINARY:
	    if isinstance(value, PgBytea) or type(value) == PgLargeObjectType:
		return value
	    elif type(value) == IntType:
		return PgLargeObject(self.conn, value)
	    else:
		return PgBytea(value)
	elif _ftv == OTHER:
	    if isinstance(value, PgOther):
		return value
	    else:
		return PgOther(value)

	# Other typecasting is not needed.  It will be once support for
	# the other built-in types (ie. box, line, inet, cidr, etc) are added.
	
	return value

    def handleArray(self, colinfo, lst):
	# If the list is empty, just return the empty list.
	if len(lst) == 0:
	    return lst

        _fn, _ft, _ds, _is, _p, _s, _nu, _ia = colinfo

	_ftv = _ft.value

	for _i in range(len(lst)):
	    if type(lst[_i]) == ListType:
		lst[_i] = self.handleArray(colinfo, lst[_i])
	    elif _ftv == PG_INT4 or _ftv == ROWID:
		lst[_i] = int(lst[_i])
	    elif _ftv == PG_INT8:
		lst[_i] = PgInt8(lst[_i])
	    elif _ftv == PG_NUMERIC:
		lst[_i] = PgNumeric(lst[_i], _p, _s)
	    elif _ftv == PG_INT2:
		lst[_i] = PgInt2(lst[_i])
	    elif _ftv == DATETIME:
		lst[_i] = DateTime.ISO.ParseAny(lst[_i])
	    elif _ftv == PG_MONEY:
		if lst[_i][0] == '(':
		    lst[_i] = PgMoney(-float(replace(lst[_i][2:-1], ',', '')))
		elif lst[_i][0] == '-':
		    lst[_i] = PgMoney(-float(replace(lst[_i][2:], ',', '')))
		else:
		    lst[_i] = PgMoney(float(replace(lst[_i][1:], ',', '')))
	    elif _ftv == BINARY:
		if _ftv == PG_BYTEA:
		    # There is no need to un-escape lst[_i], it's already been
		    # done when the PostgreSQL array was converted to a list
		    # via 'exec'.
		    lst[_i] = PgBytea(lst[_i])
		else:
		    lst[_i] = PgLargeObject(self.conn, int(lst[_i]))
		    
	return lst

    def getTypeInfo(self, pgtype):
	try:
	    return self.__type_cache[pgtype.value]
	except KeyError:
	    _nl = len(self.__conn.notices)
	    _res = self.__conn.conn.query("SELECT typname, typprtlen, typelem "
					  "FROM pg_type "
					  "WHERE oid = %s" % pgtype.value)

	    if len(self.__conn.notices) != _nl:
		raise Warning, self.__conn.notices.pop()

	    _n = _res.getvalue(0, 0)
	    _p = _res.getvalue(0, 1)
	    _b = _res.getvalue(0, 2)
	    if _n[0] == '_':
		_n = _n[1:]
		_i = 1
	    else:
		_i = 0

	    self.__type_cache[pgtype.value] = (_n, _p, _i, PgTypes(_b))
	
	return self.__type_cache[pgtype.value]

#-----------------------------------------------------------------------+
# Name:		PgOther							|
#									|
# Description:	A Python wrapper class for the PostgreSQL types that do	|
#		not (yet) have an implementation in python.  The number	|
#		of types in this catagory will shrink as more wrappers	|
#		are implementated.					|
#									|
# Note:		A Python String is used to store the PostgreSQL type in	|
#		class.							|
#-----------------------------------------------------------------------+

class PgOther:
    def __init__(self, value):
	if type(value) != StringType:
	    raise TypeError, "argument must be a string."

	self.value = value

        if hasattr(value, '__methods__'):
            for i in self.value.__methods__:
                exec 'self.%s = self.value.%s' % (i, i)

    # This definition of __coerce__ will cause Python to always call the
    # (existing) arithmatic operators for this class.  We can the perform the
    # appropiate operation on the base type, letting it decide what to do.
    def __coerce__(self, other):
            return (self, other)

    # NOTE: A string is being concatenated to a PgOther, so the result type
    #       is a PgOther
    def __add__(self, other):
        return PgOther((self.value + other))

    # NOTE: A PgOther is being concatenated to a string, so the result type
    #       is a string.
    def __radd__(self, other):
        return (other + self.value)

    def __mul__(self, other):
        return PgOther((self.value * other))

    def __rmul__(self, other):
        return PgOther((self.value * other))

    def __repr__(self):
        return repr(self.value)
    
    def __str__(self):
	return str(self.value)

    # NOTE: A PgOther object will use the PgQuoteString() function in libpq.
    def _quote(self, forArray=0):
	if self.value:
	    return PgQuoteString(self.value, forArray)
	return 'NULL'

#-----------------------------------------------------------------------+
# Name:		PgBytea							|
#									|
# Description:	A Python wrapper class for the PostgreSQL BYTEA type.	|
#									|
# Note:		A Python String is used to store the PostgreSQL type in	|
#		class.							|
#-----------------------------------------------------------------------+

class PgBytea:
    def __init__(self, value):
	if type(value) != StringType:
	    raise TypeError, "argument must be a string."

	self.value = value

        if hasattr(value, '__methods__'):
            for i in self.value.__methods__:
                exec 'self.%s = self.value.%s' % (i, i)

    # This definition of __coerce__ will cause Python to always call the
    # (existing) arithmatic operators for this class.  We can the perform the
    # appropiate operation on the base type, letting it decide what to do.
    def __coerce__(self, other):
            return (self, other)

    def __add__(self, other):
        return PgBytea((self.value + other))

    def __radd__(self, other):
        return PgBytea(other + self.value)

    def __mul__(self, other):
        return PgBytea((self.value * other))

    def __rmul__(self, other):
        return PgBytea((self.value * other))

    def __repr__(self):
        return repr(self.value)
    
    def __str__(self):
	return str(self.value)

    # NOTE: A PgBytea object will use the PgQuoteBytea() function in libpq
    def _quote(self, forArray=0):
	if self.value:
	    return PgQuoteBytea(self.value, forArray)
	return 'NULL'

#-----------------------------------------------------------------------+
# Name:		PgNumeric						|
#									|
# Description:	A Python wrapper class for the PostgreSQL numeric type.	|
#		It implements addition, subtraction, mulitplcation, and	|
#		division of scaled, fixed precision numbers.		|
#									|
# Note:		The PgNumeric class uses a Python Long type to store	|
#		the PostgreSQL numeric type.				|
#-----------------------------------------------------------------------+
    
class PgNumeric:
    def __init__(self, value, prec=None, scale=None):
        if type(value) == LongType or type(value) == IntType  or value == None:
            if prec == None or scale == None:
                raise TypeError, \
                      "you must supply precision and scale when value is a " \
	              "integer, long, or None"
	    if value == None:
		self.__v = value
	    else:
		self.__v = long(value)
            self.__p = prec
            self.__s = scale
            return
        
 	if type(value) != StringType:
	    raise TypeError, "value must be a string."

        _v = value.split()
        if len(_v) == 0 or len(_v) > 1:
            raise ValueError, \
                "invalid literal for PgNumeric: %s" % value
        _v = _v[0]

        # At this point _v is value with leading and trailing blanks
        # removed.  Initalize the precision and scale values.  They will be
        # determined from the input string (value) is prec and scale are
        # None.

        _vs = value.rfind('.')  # Get the location of the decimal point.
                                # It's used to determine the precision and
        if prec:		# scale if they aren't passed in, and to
            self.__p = prec     # adjust the input string to match the passed
        else:                   # in scale.
            self.__p = len(value)
            if _vs >= 0:
                self.__p = self.__p - 1

        # Calculate the scale of the passed in string.  _vs will contain the
        # calulated scale.

        if _vs >= 0:
            _vs = len(value) - _vs - 1
        else:
            _vs = 0

        if scale:
            self.__s = scale
        else:
            self.__s = _vs

        # Calculate the number of character to add/remove from the end of
        # the input string in order to have it match the given scale.  _sd
        # will contain the number of characters to add (>0) or remove (<0).

        _sd = self.__s - _vs

        if _sd == 0:
            pass			# No change to value needed.
        elif _sd > 0:
            _v = _v + ('0' * _sd)	# Add needed zeros to the end of value
        else:
            _v = _v[:_sd]		# Remove excess digits from the end.

        if self.__s:
            _s = _v[:-(self.__s + 1)] + _v[-self.__s:]
        else:
            _s = _v
            
        try:
            self.__v = long(_s)
        except:
            raise ValueError, \
                "invalid literal for PgNumeric: %s" % value

    def __fmtNumeric(self, value=None):
	# Check to see if the string representation of the python long has
	# a trailing 'L', if so, remove it.  Python 1.5 has the trailing 'L',
	# Python 1.6 does not.
	if value == None:
	    _v = str(self.__v)
	else:
	    _v = str(value)
	if _v[-1:] == 'L':
	    _v = _v[:-1]

	# Check to see if the numeric is less than zero and fix string if so.
	if len(_v) <= self.__s:
            _v = ("0" * (self.__s - len(_v) + 1)) + _v

	if self.__s:
	    _s = "%s.%s" % (_v[:-(self.__s)], _v[-(self.__s):])
	else:
	    _s = "%s" % _v
	return _s

    def __repr__(self):
	return "<PgNumeric instance - precision: %d scale: %d value: %s" % \
	       (self.__p, self.__s, self.__fmtNumeric())

    def __str__(self):
	return self.__fmtNumeric()

    def getScale(self):
	return self.__s

    def getPrecision(self):
	return self.__p

    def __coerce__(self, other):
	_s = None
	if type(other) in [IntType, LongType]:
	    _s = str(other)
	elif type(other) == FloatType:
	    _s = str(long(((other * (10.0 ** self.__s)) + 0.5)))
	elif type(other) == type(self):
	    return (self, other)
	else:
	    _s = None

	if _s:
	    if _s[-1:] == 'L':
		_s = _s[:-1] # Work around v1.5/1.6 differences
	    _s = "%s.%s" % (_s, ("0" * self.__s))
	    if len(_s) > self.__p:
		return None
	    else:
		return (self, PgNumeric(_s, self.__p, self.__s))

	return None
    
    def _round(self, value, drop):
	if drop > 0:
	    return (((value / (10L ** (drop - 1))) + 5L) / 10L)

	return value

    def __mul__(self, other):
	_p = self._round((self.__v * other.__v), other.__s )
	return PgNumeric(_p, self.__p, self.__s)

    def __rmul__(other, self):
	_p = self._round((self.__v * other.__v), other.__s )
	return PgNumeric(_p, self.__p, self.__s)


    def __div__(self, other):
	_n = (self.__v * (10L ** (other.__s + 1)))
	_d = other.__v
	_q = (_n / _d)
	return PgNumeric(self._round(_q, 1), self.__p, self.__s)

    def __rdiv__(other, self):
	_n = (self.__v * (10L ** (other.__s + 1)))
	_d = other.__v
	_q = (_n / _d)
	return PgNumeric(self._round(_q, 1), self.__p, self.__s)

    def __add__(self, other):
        if self.__s < other.__s:
            _d = other.__s - self.__s
            _s = self._round(((self.__v * (10L ** _d)) + other.__v), _d)
        elif self.__s > other.__s:
            _d = self.__s - other.__s
            _s = self._round((self.__v + (other.__v * (10L ** _d))), _d)
        else:
            _s = self.__v + other.__v
    	return PgNumeric(_s, self.__p, self.__s)

    def __radd__(self, other):
    	return __add__(other, self)

    def __sub__(self, other):
        if self.__s < other.__s:
            _d = other.__s - self.__s
            _s = self._round(((self.__v * (10L ** _d)) - other.__v), _d)
        elif self.__s > other.__s:
            _d = self.__s - other.__s
            _s = self._round((self.__v - (other.__v * (10L ** _d))), _d)
        else:
            _s = self.__v - other.__v
    	return PgNumeric(_s, self.__p, self.__s)

    def __rsub__(self, other):
    	return __sub__(other, self)

    def _quote(self, forArray=0):
	if self.__v:
	    if forArray:
		return '"%s"' % self.__fmtNumeric()
	    else:
		return "'%s'" % self.__fmtNumeric()
	return 'NULL'

    def __int__(self):
	return int(self.__v / (10L ** self.__s))

    def __long__(self):
	return self.__v / (10L ** self.__s)

    def __float__(self):
	v = self.__v
	s = 10L ** self.__s
	return (float(v / s) + (float(v % s) / float(s)))

    def __complex__(self):
	return complex(self.__float__())

#-----------------------------------------------------------------------+
# Name:		PgMoney							|
#									|
# Description:	A Python wrapper class for the PostgreSQL Money type.	|
#		It's primary purpose it to check for overflow during	|
#		calulations and to provide formatted output.		|
#									|
# Note:		The PgMoney class uses a Python Floating point number	|
#		represent a PostgreSQL money type.			|
#-----------------------------------------------------------------------+

class PgMoney:
    def __init__(self, value):
	if value == None:
	    self.value = value
	    return

	self.value = float(value)
	if self.value < -21474836.48 or self.value > 21474836.47:
	    raise OverflowError, 'money initialization'
	

    def __checkresult(self, value, op):
	if value < -21474836.48 or value > 21474836.47:
	    raise OverflowError, 'money %s' % op
	return PgMoney(value)

    def __coerce__(self, other):
	if other == None:
	    return None
        res = coerce(self.value, other)
        if res == None:
            return None
        _s, _o = res
	return (self, _o)

    def __hash__(self):
        return hash(self.value)

    def __cmp__(self, other):
        if self.value < other:
            return -1
        elif self.value == other:
            return 0
        else:
            return 1

    def __add__(self, other):
	return self.__checkresult(self.value + other, "addition")

    def __sub__(self, other):
 	return self.__checkresult(self.value - other, "subtraction")

    def __mul__(self, other):
	return self.__checkresult(self.value * other, "mulitplication")

    def __div__(self, other):
	return self.__checkresult(self.value / other, "division")

    def __divmod__(self, other):
	_a, _b = divmod(self.value, other)
	return (self.__checkresult(_a, "divmod"), _b)

    def __pow__(self, other, modulus=None):
	return self.__checkresult(pow(self.value, other, modulus), "pow")

    def __radd__(self, other):
	return self.__checkresult(other + self.value, "addition")

    def __rsub__(self, other):
	return self.__checkresult(other - self.value, "subtraction")

    def __rmul__(self, other):
	return self.__checkresult(other * self.value, "multiplication")

    def __rdiv__(self, other):
	return self.__checkresult(other / self.value, "division")

    def __rdivmod__(self, other):
	_a, _b = divmod(other, self.value)
	return (self.__checkresult(_a, "divmod"), _b)

    def __rpow__(self, other, modulus=None):
	return self.__checkresult(pow(other, self.value, modulus), "pow")

    def __neg__(self):
	return self.__checkresult(self.value * -1, "negation")

    def __pos__(self):
	return self.value

    def __abs__(self):
	return self.__checkresult(abs(self.value), "abs")

    def __complex__(self):
	return complex(self.value)

    def __int__(self):
	return int(self.value)

    def __long__(self):
	return long(self.value)

    def __float__(self):
	return self.value	# PgMoney is already a float :-)

    def __repr__(self):
	return '%.2f' % self.value

    def __str__(self):
	_s = '%.2f' % abs(self.value)
	_i = string.rfind(_s, '.')
	_c = (_i - 1) / 3
	for _j in range(_c):
	    _i = _i - 3
	    _s = '%s,%s' % (_s[:_i], _s[_i:])
	if self.value < 0.0:
	    return '($%s)' % _s
	else:
	    return '$%s' % _s

    def _quote(self, forArray=0):
	if self.value:
	    if forArray:
		return '"%s"' % str(self.value)
	    else:
		return "'%s'" % str(self.value)
	return 'NULL'

#-----------------------------------------------------------------------+
# Name:		PgInt8							|
#									|
# Description:	A Python wrapper class for the PostgreSQL int8 type.	|
#		It's primary purpose it to check for overflow during	|
#		calulations.						|
#									|
# Note:		The PgInt8 class uses a Python Long Integer to hold	|
#		the PostgreSQL int8 type.				|
#									|
# Note:		This class will only be defined if the C implementation	|
#		of the PgInt8 object was not imported with/from libpq.	|
#-----------------------------------------------------------------------+

if dir().count('PgInt8') == 0:	# Only define this class is PgInt8 wasn't
				# brought in via libpq.
    class PgInt8:
	def __init__(self, value):
	    if value == None:
		self.value = value
		return

	    self.value = long(value)
	    if self.value < -9223372036854775808L or \
	       self.value > 9223372936854775807L:
		raise OverflowError, 'int8 initialization'

	def __checkresult(self, value, op):
	    if value < -9223372036854775808L or value > 9223372936854775807L:
		raise OverflowError, 'int8 %s' % op
	    return PgInt8(value)

	def __coerce__(self, other):
	    if other == None:
		return None
	    res = coerce(self.value, other)
            if res == None:
                return None
            _s, _o = res
	    return (self, _o)

	def __hash__(self):
	    return hash(self.value)

	def __cmp__(self, other):
	    if self.value < other:
		return -1
	    elif self.value == other:
		return 0
	    else:
		return 1

	def __nonzero__(self):
	    return self.value != 0

	def __add__(self, other):
	    return self.__checkresult(self.value + other, "addition")

	def __sub__(self, other):
	    return self.__checkresult(self.value - other, "subtraction")

	def __mul__(self, other):
	    return self.__checkresult(self.value * other, "mulitplication")

	def __div__(self, other):
	    return self.__checkresult(self.value / other, "division")

	def __divmod__(self, other):
	    _a, _b = divmod(self.value, other)
	    return (self.__checkresult(_a, "divmod"), _b)

	def __pow__(self, other, modulus=None):
	    return self.__checkresult(pow(self.value, other, modulus), "pow")

	def __lshift__(self, other):
	    return self.__checkresult(self.value << other, 'lshift')

	def __rshift__(self, other):
	    return self.__checkresult(self.value >> other, 'rshift')

	def __and__(self, other):
	    return self.__checkresult(self.value & other, 'and')

	def __xor__(self, other):
	    return self.__checkresult(self.value ^ other, 'xor')

	def __or__(self, other):
	    return self.__checkresult(self.value | other, 'or')

	def __radd__(self, other):
	    return self.__checkresult(other + self.value, "addition")

	def __rsub__(self, other):
	    return self.__checkresult(other - self.value, "subtraction")

	def __rmul__(self, other):
	    return self.__checkresult(other * self.value, "mulitplication")

	def __rdiv__(self, other):
	    return self.__checkresult(other / self.value, "division")

	def __rdivmod__(self, other):
	    _a, _b = divmod(other, self.value)
	    return (self.__checkresult(_a, "divmod"),
		    self.__checkresult(_b, "divmod"))

	def __rpow__(self, other, modulus=None):
	    return self.__checkresult(pow(other, self.value, modulus), "pow")

	def __rlshift__(self, other):
	    return self.__checkresult(other << self.value, 'lshift')

	def __rrshift__(self, other):
	    return self.__checkresult(other >> self.value, 'rshift')

	def __rand__(self, other):
	    return self.__checkresult(other & self.value, 'and')

	def __rxor__(self, other):
	    return self.__checkresult(other ^ self.value, 'xor')

	def __ror__(self, other):
	    return self.__checkresult(other | self.value, 'or')

	def __neg__(self):
	    return self.__checkresult(neg(self.value), 'neg')

	def __pos__(self):
	    return self.__checkresult(pos(self.value), 'pos')

	def __abs__(self):
	    return self.__checkresult(abs(self.value), 'abs')

	def __complex__(self):
	    return complex(self)

	def __int__(self):
	    return int(self.value)

	def __long__(self):
	    return self.value	# PgInt8 is already a Long.

	def __float__(self):
	    return float(self.value)

	def __complex__(self):
	    return complex(self.value)

	def __hex__(self):
	    return hex(self.value)

	def __oct__(self):
	    return oct(self.value)

	def __repr__(self):
	    return repr(self.value)

	def __str__(self):
	    return str(self.value)

	def _quote(self, forArray=0):
	    if self.value:
		s = str(self.value)
		if s[-1:] == "L":
		    s = s[:-1]
		return "%s" % s
	    return 'NULL'

#-----------------------------------------------------------------------+
# Name:		PgResultSet						|
#									|
# Description:	This class defines the DB-API query result set for a	|
#		single row.  It emulates a sequence  with the added	|
#	        feature of being able to reference an attribute by	|
#		column name in addition to a zero-based numeric index.	|
#-----------------------------------------------------------------------+

class PgResultSet:

    def __init__(self, value, description, mapname=None):
	self.__dict__['baseObj'] = value
	self.__dict__['_desc_'] = description

	# Now we set up attributes based on the column names in the result set
        self.__dict__['_xlatkey'] = {}
        if mapname == None:
            for _i in range(len(description)):
                self.__dict__['_xlatkey'][description[_i][0]] = _i
        else:
            self.__dict__['_xlatkey'].update(mapname)
	return

    def __getattr__(self, key):
	if self._xlatkey.has_key(key):
	    return self.baseObj[self._xlatkey[key]]
	raise AttributeError, key

    # We define a __setattr__ routine that will only allow the attributes that
    # are the column names to be updated.  All other attributes are read-only.
    def __setattr__(self, key, value):
	if key in ('baseObj', '_xlatkey', '_desc_'):
	    raise AttributeError, "%s is read-only." % key

	if self._xlatkey.has_key(key):
	    self.__dict__['baseObj'][self._xlatkey(key)] = value
	else:
	    raise AttributeError, key

    def __len__(self):
	return len(self.baseObj)

    def __getitem__(self, key):
	if type(key) == StringType:
	    key = self._xlatkey[key]
	return self.baseObj[key]

    def __setitem__(self, key, value):
	if type(key) == StringType:
	    key = self._xlatkey[key]
	self.baseObj[key] = value
	return

    def __getslice__(self, i, j):
	return PgResultSet(self.baseObj[i:j], self._desc_[i:j])

    def __setslice__(self, i, j, value):
	# If we are passed a PgResultSet object, convert it to the base 
	# sequence object for the result set. Also update the description from
	# the PgResultSet.
	if i < 0:
	    i = len(self.baseObj) + i
	if j < 0:
	    j = len(self.baseObj) + j
	elif j > len(self.baseObj):
	    j = len(self.baseObj)
	if isinstance(value, PgResultSet):
	    self.__dict__['_desc_'][i:j] = value._desc_
	    self.__dict__['_xlatkey'] = {}
	    for _i in range(len(self._desc_)):
		self.__dict__['_xlatkey'][self._desc_[_i][0]] = _i
	    value = value.baseObj
	else:
	    try:
		if len(value) != (j - i):
		    raise ValueError, 'slice size mis-match'
	    except:
		raise TypeError, 'illegal argument type for built-in operation'

	self.baseObj[i:j] = value
	return

    def __repr__(self):
	return repr(self.baseObj)

    def __str__(self):
	return str(self.baseObj)

    def __cmp__(self, other):
	return cmp(self.baseObj, other)

    def description(self):
	return self._desc_

    def keys(self):
	_k = []
	for _i in self._desc_:
	    _k.append(_i[0])
	return _k

    def values(self):
	return self.baseObj[:]

    def items(self):
	_items = []
	for i in range(len(self.baseObj)):
	    _items.append((self._desc_[i][0], self.baseObj[i]))
	return _items

    def has_key(self, key):
	return self._xlatkey.has_key(key)

    def get(self, key):
	return self[key]

#-----------------------------------------------------------------------+
# Define the PgSQL function calls:					|
#									|
#	connect()      -- connecnt to a PostgreSQL database.		|
#	_handleArray() -- Transform a Python list into a string repre-	|
#			  senting a PostgreSQL array.			|
#	_quote()       -- Transform a Python object representing a	|
#			  PostgreSQL type into a appropiately quoted	|
#			  string that can be sent to the database in a	|
#			  UPDATE/INSERT statement.  _quote() calls the	|
#			  _handleArray() function to quote arrays.	|
#       _quoteall()    -- transforms all elements of a list or diction-	|
#			  ary using _quote.				|
#-----------------------------------------------------------------------+

def connect(dsn=None, user=None, password=None, host=None, database=None,
		port=None, options=None, tty=None):
    """
connection =  PgSQL.connect(dsn[, user, password, host, database, port,
				  options, tty])
    Opens a connection to a PostgreSQL database."""
	
    _d = {}

    # Try getting values from the DSN first.
    if dsn != None:
	try:
	    params = string.split(dsn, ":")
	    if params[0] != '':	_d["host"]     = params[0]
	    if params[1] != '':	_d["port"]     = params[1]
	    if params[2] != '':	_d["dbname"]   = params[2]
	    if params[3] != '':	_d["user"]     = params[3]
	    if params[4] != '':	_d["password"] = params[4]
	    if params[5] != '':	_d["options"]  = params[5]
	    if params[6] != '':	_d["tty"]      = params[6]
	except:
	    pass

    # Override from the keyword arguments, if needed.
    if (user != None):	   _d["user"]	  = user
    if (password != None): _d["password"] = password
    if (host != None):
	_d["host"] = host
	try:
	    params = string.split(host, ":")
	    _d["host"] = params[0]
	    _d["port"] = params[1]
	except:
	    pass
    if (database != None): _d["dbname"]	 = database
    if (port != None):	   _d["port"]	 = port
    if (options != None):  _d["options"] = options
    if (tty != None):	   _d["tty"]	 = tty

    # Build up the connection info string passed to PQconnectdb
    # via the constructor to Connection.

    connInfo = ""
    for i in _d.keys():
	connInfo = "%s%s=%s " % (connInfo, i, _d[i])
	
    return Connection(connInfo)

def _handleArray(value):
    """
_handleArray(list) -> string
    This function handle the transformation of a Python list into a string that
    can be used to update a PostgreSQL array attribute."""

    #Check for, and handle an empty list.
    if len(value) == 0:
	return '{}'

    _j = "'{"
    for _i in value:
	if type(_i) in [ListType, TupleType]:
	    _v = list(_i)
	    _j = _j + _handleArray(_v) + ','
	elif hasattr(_i, '_quote'):
	    _j = '%s%s,' % (_j, _i._quote(1))
	elif type(_i) in [DateTime.DateTimeType, DateTime.DateTimeDeltaType]:
	    _j = '%s"%s",' % (_j, _i)
	elif type(_i) == PgInt2Type or type(_i) == PgInt8Type:
	    _j = '%s%s,' % (_j, str(_i))
	else:
	    _j = '%s%s,' % (_j, PgQuoteString(_i, 1))

    return _j[:-1] + "}'"

def _quote(value):
    """
_quote(value) -> string
    This function transforms the Python value into a string suitable to send
    to the PostgreSQL database in a insert or update statement.  This function
    is automatically applied to all parameter sent vis an execute() call.
    Because of this an update/insert statement string in an execute() call
    should only use '%s' [or '%(name)s'] for variable subsitution without any
    quoting."""

    if value == None:
	return 'NULL'

    if type(value) in [ListType, TupleType]:
	return _handleArray(list(value))

    if hasattr(value, '_quote'):
	return value._quote()

    if type(value) in [DateTimeType, DateTimeDeltaType]:
        return "'%s'" % value

    if type(value) == StringType:
	return PgQuoteString(value)

    return repr(value)

def _quoteall(vdict):
    """
_quoteall(vdict)->dict
    Quotes all elements in a list or dictionary to make them suitable for
    insertion."""

    if type(vdict) == DictType or isinstance(vdict, PgResultSet):
        t = {}
        for k, v in vdict.items():
	    t[k]=_quote(v)
    elif type(vdict) == StringType:
	# Note: a string is a SequenceType, but is treated as a single
	#	entity, not a sequence of characters.
	t = (_quote(vdict), )
    elif type(vdict)in [ListType, TupleType]:
	t = tuple(map(_quote, vdict))
    else:
	raise TypeError, \
	      "argument to _quoteall must be a sequence or dictionary!"

    return t

#-----------------------------------------------------------------------+
# Name:		Connection						|
#									|
# Description:	Connection defines the Python DB-API 2.0 connection	|
#		object.  See the DB-API 2.0 specifiaction for details.	|
#-----------------------------------------------------------------------+

class Connection:
    """Python DB-API 2.0 Connection Object."""

    def __init__(self, connInfo):
	try:
	    self.__dict__["conn"] = PQconnectdb(connInfo)
	except Exception, m:
	    # The connection to the datadata failed.
	    # Clean up the Connection object that was created.
	    # Note: _isOpen must be defined for __del__ to work.
	    self.__dict__["_isOpen"] = None
	    del(self)
	    raise DatabaseError, m
	    
	self.__dict__["autocommit"] = 0
	self.__dict__["notices"] = self.conn.notices
	self.__dict__["inTransaction"] = 0
	self.__dict__["version"] = self.conn.version
	self.__dict__["_isOpen"] = 1
	self.__dict__["_cache"] = TypeCache(self)
	if noWeakRef:
	    self.__dict__["cursors"] = []
	else:
	    self.__dict__["cursors"] = weakref.WeakValueDictionary()

    def __del__(self):
	if self._isOpen:
	    self.close()	# Ensure that the connection is closed.

    def __setattr__(self, name, value):
	if name == "autocommit":
	    if value == None:
		raise InterfaceError, \
		    "Can't delete the autocommit attribute."
            # Don't allow autocommit to change if there are any opened cursor
            # associated with this connection.
	    if self.__anyLeft():
		if noWeakRef:
		    # If the are cursors left, but there are no weak references
		    # available, garbage collect any cursors that are only
		    # referenced in self.cursors.

		    self.__gcCursors()

		    if len(self.cursors) > 0:
			raise AttributeError, \
			    "Can't change autocommit when a cursor is active."
		else:
		    raise AttributeError, \
			"Can't change autocommit when a cursor is active."

	    # It's possible that the connection can still have an open
	    # transaction, even though there are no active cursors.

	    if self.inTransaction:
		self.rollback()

	    if value:
		self.__dict__[name] = 1
	    else:
		self.__dict__[name] = 0
        elif self.__dict__.has_key(name):
	    raise AttributeError, "%s is read-only." % name
	else:
            raise AttributeError, name

    def __closeCursors(self, flag=0):
	"""
	__closeCursors() - closes all cursors associated with this connection"""
	if self.__anyLeft():
	    if noWeakRef:
		curs = self.cursors[:]
	    else:
		curs = map(lambda x: x(), self.cursors.data.values())

	    for i in curs:
		if flag:
		    i.close()
		else:
		    i._Cursor__reset()
                    
	return self.inTransaction

    def __anyLeft(self):
	if noWeakRef:
	    return len(self.cursors) > 0

	return len(self.cursors.data.keys()) > 0

    def __gcCursors(self):
	# This routine, which will be called only if weak references are not
	# available, will check the reference counts of the cursors in the
	# connection.cursors list and close any that are only referenced
	# from that list.  This will clean up deleted cursors.

	for i in self.cursors[:]:
	    # Check the reference count.  It will be 4 if it only exists in
	    # self.cursors.  The magic number for is derived from the fact
	    # that there will be 1 reference count for each of the follwoing:
	    #     self.cursors, self.cursors[:], i, and as the argument to
	    #     getrefcount(),

	    if getrefcount(i) < 5:
		i.close()


    def close(self):
	"""
    close()
	Close the connection now (rather than whenever __del__ is called).
	Any active cursors for this connection will be closed and the connection
	will be unusable from this point forward.\n"""

	if not self._isOpen:
	    raise InterfaceError, "Connection is already closed."

	if self.__closeCursors(1):
	    try:
		_nl = len(self.conn.notices)
		self.conn.query("ROLLBACK WORK")
		if len(self.conn.notices) != _nl:
		    raise Warning, self.__conn.notices.pop()
	    except:
		pass

	self.__dict__["_cache"] = None
	self.__dict__["_isOpen"] = 0
	self.__dict__["autocommit"] = None
	self.__dict__["conn"] = None
	self.__dict__["cursors"] = None
	self.__dict__["inTransaction"] = 0
	self.__dict__["version"] = None
        self.__dict__["notices"] = None

    def commit(self):
	"""
    commit()
	Commit any pending transactions to the database.\n"""

	if not self._isOpen:
	    raise InterfaceError, "Commit failed - Connection is not open."

	if self.autocommit:
	    raise InterfaceError, "Commit failed - autocommit is on."

	if self.__closeCursors():
	    self.__dict__["inTransaction"] = 0
	    _nl = len(self.conn.notices)
	    res = self.conn.query("COMMIT WORK")
	    if len(self.conn.notices) != _nl:
		raise Warning, self.conn.notices.pop()
	    if res.resultStatus != COMMAND_OK:
		raise InternalError, "Commit failed - reason unknown."

    def rollback(self):
	"""
    rollback()
	Rollback to the start of any pending transactions.\n"""

	if not self._isOpen:
	    raise InterfaceError, "Rollback failed - Connection is not open."

	if self.autocommit:
	    raise InterfaceError, "Rollback failed - autocommit is on."

	if self.__closeCursors():
	    self.__dict__["inTransaction"] = 0
	    _nl = len(self.conn.notices)
	    res = self.conn.query("ROLLBACK WORK")
	    if len(self.conn.notices) != _nl:
		raise Warning, self.conn.notices.pop()
	    if res.resultStatus != COMMAND_OK:
		raise InternalError, \
		      	"Rollback failed - %s" % res.resultErrorMessage

    def cursor(self, name=None):
	"""
    cursor([name])
	Returns a new 'Cursor Object' (optionally named 'name')."""

	if not self._isOpen:
	    raise InterfaceError, \
		  "Create cursor failed - Connection is not open."

	return Cursor(self, name)

    def binary(self, string=None):
	"""
    binary([string])
	Returns a new 'Large Object'.  If sting is present, it is used to
	initialize the large object."""

	if not self._isOpen:
	    raise InterfaceError, \
		  "Creation of large object failed - Connection is not open."

	_nl = len(self.conn.notices)

	# Ensure that we are in a transaction for working with large objects
	if not self.inTransaction:
	    conn.conn.query("BEGIN WORK")

	_lo = self.conn.lo_creat(INV_READ | INV_WRITE)

	if len(self.conn.notices) != _nl:
	    raise Warning, self.__conn.notices.pop()

	if string:
	    _lo.open("w")
	    _lo.write(string)
	    _lo.close()
	
	    if len(self.conn.notices) != _nl:
		if not self.inTransaction:
		    conn.conn.query("ROLLBACK WORK")
		raise Warning, self.conn.notices.pop()

	if not self.inTransaction:
	    conn.conn.query("COMMIT WORK")

	return _lo

    def unlink(self, lobj):
	"""
    unlink(OID|PgLargeObject)
	Remove a large object from the database inversion file syste."""

	if not self._isOpen:
	    raise InterfaceError, \
		  "Unlink of large object failed - Connection is not open."

	if self.version.post70 or not self.inTransaction:
	    raise NotSupportedError, \
		  "unlink of a PostgreSQL Large Object in a transaction"

	if type(lobj) == IntType:
	    oid = lobj
	elif type(lobj) == PgLargeObjectType:
	    oid = lobj.oid
	    
	_nl = len(self.conn.notices)
	res = self.conn.lo_unlink(oid)
	if len(self.conn.notices) != _nl:
	    raise Warning, self.conn.notices.pop()

	return res


#-----------------------------------------------------------------------+
# Name:		Cursor							|
#									|
# Description:	Cursor defines the Python DB-API 2.0 cursor object.	|
#		See the DB-API 2.0 specification for details.		|
#-----------------------------------------------------------------------+

class Cursor:
    """Python DB-API 2.0 Cursor Object."""

    def __init__(self, conn, name):
        if not isinstance(conn, Connection):
            raise TypeError, "Cursor requires a connection."

        # Generate a unique name for the cursor is one is not given.
        if name == None:
            name = "PgSQL_%08X" % id(self)
        elif type(name) != StringType:
            raise TypeError, "Cursor name must be a string."
        
	# Define the public variables for this cursor.
	self.__dict__["arraysize"] = 1

	# Define the private variables for this cursor.
	if noWeakRef:
	    self.__dict__["conn"]    = conn
	else:
	    self.__dict__["conn"]    = weakref.proxy(conn)
	self.__dict__["name"]        = name

        # This needs to be defined here sot that the initial call to __reset()
        # will work.
	self.__dict__["closed"]    = None

        self.__reset()
        
	# _varhdrsz is the length (in bytes) of the header for variable
	# sized postgreSQL data types.

	self.__dict__["_varhdrsz"] = 4

	# Add ourselves to the list of cursors for our owning connection.
	if noWeakRef:
	    self.conn.cursors.append(self)
	    if len(self.conn.cursors) > 1:
		# We have additional cursors, garbage collect them.
		self.conn._Connection__gcCursors()
	else:
	    self.conn.cursors[id(self)] = self

	if not conn.autocommit:
	    # Only the first created cursor begins the transaction.
 	    if not self.conn.inTransaction:
		conn.conn.query("BEGIN WORK")
		self.conn.__dict__["inTransaction"] = 1

    def __del__(self):
	# Ensure that the cursor is closed when it is deleted.  This takes
	# care of some actions that needs to be completed when a cursor is
	# deleted, such as disassociating the cursor from the connection
	# and closing an open transaction if this is the last cursor for
	# the connection.
	if not self.closed:
	    self.close()

    def __reset(self):
	try:
	    if (self.closed == 0) and self.conn.inTransaction:
		try:
		    self.conn.conn.query("CLOSE %s" % self.name)
		except:
		    pass
	except:
	    pass

	self.__dict__["res"]	     = None
        # closed is a trinary variable:
        #     == None => Cursor has not been opened.
        #     ==    0 => Cursor is open.
        #     ==    1 => Curosr is closed.
	self.__dict__["closed"]    = None
	self.__dict__["description"] = None
	self.__dict__["oidValue"]    = None
	self.__dict__["_mapname"]    = None
	self.__dict__["_rows_"]      = 0
	self.__dict__["_idx_"]       = 1
	self.__dict__["rowcount"]    = -1

    def __setattr__(self, name, value):
        if self.closed:
            raise InterfaceError, "Operation failed - the cursor is closed."
        
	if name in ["rowcount", "oidValue", "description"]:
	    raise AttributeError, "%s is read-only." % name
	elif self.__dict__.has_key(name):
	    self.__dict__[name] = value
        else:
            raise AttributeError, name
        
    def __fetchOneRow(self):
	if self._idx_ >= self._rows_:
            self.__dict__['rowcount'] = 0
	    return None

	_j = []
	_r = self.res
	_c = self.conn._cache
	for _i in range(self.res.nfields):
	    _j.append(_c.typecast(self.description[_i],
                                  _r.getvalue(self._idx_, _i)))
	
	self._idx_ = self._idx_ + 1

        self.__dict__['rowcount'] = 1
        
	# Return a new PgResultSet.  Note that we pass a copy of the descrip-
	# tion to PgResultSet.

	if fetchReturnsList:
	    # Return a list (This is the minimum required by DB-API 2.0
	    # compliance).
	    return _j
	else:
	    # Return a new PgResultSet.  Note that we pass a copy of the
	    # description to PgResultSet.
	    return PgResultSet(_j, self.description[:], self._mapname)

    def __fetchManyRows(self, count, iList=[]):
	_many = iList
	if count < 0:
	    while 1:
		_j = self.__fetchOneRow()
		if _j is not None:
		    _many.append(_j)
		else:
		    break
	elif count > 0:
	    for _i in range(count):
		_j = self.__fetchOneRow()
		if _j is not None:
		    _many.append(_j)
		else:
		    break

        self.__dict__['rowcount'] = len(_many)
        
	return _many

    def __makedesc__(self):
        # Since __makedesc__ will only be called after a successful query or
        # fetch, self.res will contain the information needed to build the
        # description attribute even if no rows were returned.  So, we always
        # build up the description.
        self.__dict__['description'] = []
        self._mapname = {}
        _res = self.res
        _cache = self.conn._cache
        for _i in range(_res.nfields):
            _j = []

            _j.append(_res.fname(_i))

            _typ = PgTypes(_res.ftype(_i))
            _mod = _res.fmod(_i)
            _tn, _pl, _ia, _bt = _cache.getTypeInfo(_typ)
            if _ia:
                _s, _pl, _s, _s = _cache.getTypeInfo(_bt)
                if _bt == PG_OID:
                    _bt = PgTypes(PG_BLOB)
                _typ = _bt
            elif _typ.value == PG_OID:
                _p = _res.getvalue(0, _i)
                if type(_p) in [PgLargeObjectType, NoneType]:
                    _typ = PgTypes(PG_BLOB)
                else:
                    _typ = PgTypes(PG_ROWID)

            _j.append(_typ)

            # Calculate Display size, Internal size, Precision and Scale.
            # Note: Precision and Scale only have meaning for PG_NUMERIC
            # columns.
            if _typ.value == PG_NUMERIC:
                if _mod == -1:
                    # We have a numeric with no scale/precision.
                    # Get them from by converting the string to a PgNumeric
                    # and pulling them form the PgNumeric object.  If that
		    # fails default to a precision of 30 with a scale of 6.
		    try:
			nv = PgNumeric(_res.getvalue(0, _i))
			_p = nv.getPrecision()
			_s = nv.getScale()
		    except ValueError, m:
			_p = 30
			_s = 6
                else:
                    # We hava a valid scale/precision value.  Use them.
                    _s = _mod - self._varhdrsz
                    _p = (_s >> 16) & 0xffff
                    _s = _s & 0xffff
                _j.append(_p + 3)		# Display size
                _j.append(_p)		# Internal (storage) size
                _j.append(_p)		# Precision
                _j.append(_s)		# Scale
            else:
                if _pl == -1:
                    _pl = _res.fsize(_i)
                    if _pl == -1:
                        _pl = _mod - self._varhdrsz
                _j.append(_pl)		# Display size
                _s = _res.fsize(_i)
                if _s == -1:
                    _s = _mod
                _j.append(_s)		# Internal (storage) size
                if _typ.value == PG_MONEY:
                    _j.append(9)		# Presicion and Scale (from
                    _j.append(2)		# the PostgreSQL doco.)
                else:
                    _j.append(None)		# Preision
                    _j.append(None)		# Scale

            _j.append(None)	# nullOK is not implemented (yet)
            _j.append(_ia)	# Array indicator (PostgreSQL specific)

            self.__dict__["description"].append(_j)

            # Add the fieldname:fieldindex to the _mapname dictionary
            self._mapname[_j[0]] = _i

    def callproc(self, proc, *args):
        if self.closed:
            raise InterfaceError, "callproc failed - the cursor is closed."
        
	if self.conn == None:
	    raise Error, "connection is closed."

	if self.closed == 0:
	    raise InterfaceError, "callproc() failed - cursor is active."
		    
	if self.conn.autocommit:
	    pass
	else:
	    if not self.conn.inTransaction:
		_nl = len(self.conn.notices)
		self.conn.conn.query('BEGIN WORK')
		if len(self.conn.notices) != _nl:
		    raise Warning, self.__conn.notices.pop()
		self.conn.__dict__["inTransaction"] = 1

	_qstr = "select %s(" % proc
	for _i in range(len(args)):
	    _qstr = '%s%s, ' % (_qstr, _quote(args[_i]))
	if len(args) == 0:
	    _qstr = '%s)' % _qstr
	else:
	    _qstr = '%s)' % _qstr[:-2]

	_nl = len(self.conn.notices)

	try:
	    self.res = self.conn.conn.query(_qstr)
	    self._rows_ = self.res.ntuples
	    self._idx_  = 0
	    if type(self.res) != PgResultType:
		self.__dict__['rowcount'] = -1
	    else:
                self.__dict__['oidValue'] =  self.res.oidValue
                if self.res.resultType == RESULT_DQL:
                    pass
                elif self.res.resultType == RESULT_DML:
                    self.__dict__['rowcount'] = self.res.cmdTuples
                else:
                    self.__dict__['rowcount'] = -1
	except OperationalError, msg:
	    # Uh-oh.  A fatal error occurred.  This means the current trans-
	    # action has been aborted.  Try to recover to a sane state.
	    if self.conn.inTransaction:
		self.conn.conn.query('END WORK')
		self.conn.__dict__["inTransaction"] = 0
		self.conn._Connection__closeCursors()
	    raise OperationalError, msg
	except InternalError, msg:
	    # An internal error occured.  Try to get to a sane state.
	    self.conn.__dict__["inTransaction"] = 0
	    self.conn.Connection__closeCursors_()
	    self.conn.close()
	    raise InternalError, msg

	if len(self.conn.notices) != _nl:
	    _drop = self.conn.notices[-1]
	    if _drop.find('transaction is aborted') > 0:
		raise Warning, self.conn.notices.pop()

	self._rows_ = self.res.ntuples
	self._idx_ = 0
	self.__dict__['rowcount'] = -1    # New query - no fetch occured yet.
	self.__makedesc__()

	return None

    def close(self):
        if self.closed:
            raise InterfaceError, "The cursor is already closed."

        # Dis-associate ourselves from our cursor.
        self.__reset()
        try:
            _cc = self.conn.cursors
            if noWeakRef:
                _cc.remove(self)
		if (len(_cc) > 0):
		    # We have additional cursors, garbage collect them.
		    _cc._Connection__gcCursors()
            else:
                del _cc.data[id(self)]
        except:
            pass
        self.conn = None
	self.closed = 1

    def execute(self, query, *parms):
        if self.closed:
            raise InterfaceError, "execute failed - the cursor is closed."

	if self.conn == None:
	    raise Error, "connection is closed."

	if self.closed == 0:
	    if re_DQL.search(query):
                # A SELECT has already been executed with this cursor object,
                # which means a PostgreSQL portal (may) have been opened.
                # Trying to open another portal will cause an error to occur,
                # so we asusme that the developer is done with the previous
                # SELECT and reset the cursor object to it's inital state.
		self.__reset()
		    
	_qstr = query
	if self.conn.autocommit:
	    pass
	else:
            _badQuery = (self.conn.version < 70100) and \
			(re_DRT.search(query) or re_DRI.search(query))
	    if not self.conn.inTransaction:
		if _badQuery:
		    pass # PostgreSQL version < 7.1 and not in transaction,
			 # so DROP TABLE/INDEX is ok.
		else:
		    _nl = len(self.conn.notices)
		    self.conn.conn.query('BEGIN WORK')
		    if len(self.conn.notices) != _nl:
			raise Warning, self.conn.notices.pop()
		    self.conn.__dict__["inTransaction"] = 1

	    if re_DQL.search(query) and \
               not (noPostgresCursor or re_4UP.search(query)):
		_qstr = "DECLARE %s CURSOR FOR %s" % (self.name, query)
		self.closed = 0
	    elif _badQuery and self.conn.inTransaction:
		raise NotSupportedError, \
		      "DROP [TABLE|INDEX] within a transaction"
	    if not self.conn.inTransaction:
		if _badQuery:
		    pass # not in transaction so DROP TABLE/INDEX is ok.
		else:
		    _nl = len(self.conn.notices)
		    self.conn.conn.query('BEGIN WORK')
		    if len(self.conn.notices) != _nl:
			raise Warning, self.conn.notices.pop()
		    self.conn.__dict__["inTransaction"] = 1

	_nl = len(self.conn.notices)

	try:
	    if len(parms) == 0:
		# If there are no paramters, just execute the query.
		self.res = self.conn.conn.query(_qstr)
	    else:
		if len(parms) == 1 and \
		   (type(parms[0]) in [DictType, ListType, TupleType] or \
					    isinstance(parms[0], PgResultSet)):
		    parms = _quoteall(parms[0])
		else:
		    parms = tuple(map(_quote, parms));
		self.res = self.conn.conn.query(_qstr % parms)
	    self._rows_ = self.res.ntuples
	    self._idx_  = 0
	    self.__dict__['rowcount'] = -1 # New query - no fetch occured yet.
	    if type(self.res) != PgResultType:
		self.__dict__['rowcount'] = -1
	    else:
                self.__dict__['oidValue'] =  self.res.oidValue
                if self.res.resultType == RESULT_DQL:
                    pass
                elif self.res.resultType == RESULT_DML:
                    self.__dict__['rowcount'] = self.res.cmdTuples
                else:
                    self.__dict__['rowcount'] = -1
	except OperationalError, msg:
	    # Uh-oh.  A fatal error occurred.  This means the current trans-
	    # action has been aborted.  Try to recover to a sane state.
	    if self.conn.inTransaction:
		_n = len(self.conn.notices)
		self.conn.conn.query('ROLLBACK WORK')
		if len(self.conn.notices) != _n:
		    raise Warning, self.__conn.notices.pop()
		self.conn.__dict__["inTransaction"] = 0
		self.conn._Connection__closeCursors()
	    raise OperationalError, msg
	except InternalError, msg:
	    # An internal error occured.  Try to get to a sane state.
	    self.conn.__dict__["inTransaction"] = 0
	    self.conn.Connection__closeCursors_()
	    self.conn.close()
	    raise InternalError, msg

	if len(self.conn.notices) != _nl:
	    _drop = self.conn.notices[-1]
	    if _drop.find('transaction is aborted') > 0:
		raise Warning, self.conn.notices.pop()

	if self.res.resultType == RESULT_DQL:
	    self.__makedesc__()
	elif _qstr[:8] == 'DECLARE ':
	    # Ok -- we've created a cursor, we will pre-fetch the first row in
	    # order to make the description array.  Note: the first call to
	    # fetchXXX will return the pre-fetched record.
	    self.res = self.conn.conn.query("FETCH 1 FROM %s" % self.name)
	    self._rows_ = self.res.ntuples
	    self._idx_ = 0
	    self.__makedesc__()

	if len(self.conn.notices) != _nl:
	    _drop = self.conn.notices[-1]
	    if _drop.find('transaction is aborted') > 0:
		raise Warning, self.conn.notices.pop()
	    
    def executemany(self, query, parm_sequence):
        if self.closed:
            raise InterfaceError, "executemany failed - the cursor is closed."

	if self.conn == None:
	    raise Error, "connection is closed."

	for _i in parm_sequence:
	    self.execute(query, _i)

    def fetchone(self):
        if self.closed:
            raise InterfaceError, "fetchone failed - the cursor is closed."

	if self.conn == None:
	    raise Error, "connection is closed."

	if self.res == None:
	    raise Error, \
                  "fetchone() failed - cursor does not contain a result."
	elif self.res.resultType != RESULT_DQL:
	    if self.closed == None:
		raise Error, \
		      "fetchone() Failed - cursor does not contain any rows."

	if self._idx_ < self._rows_:
	    pass	# Still data in result buffer, use it.
	elif self.closed == 0:
	    _nl = len(self.conn.notices)
	    self.res = self.conn.conn.query("FETCH 1 FROM %s" % self.name)
	    self._rows_ = self.res.ntuples
	    self._idx_ = 0

	    if len(self.conn.notices) != _nl:
		_drop = self.conn.notices[-1]
		if _drop.find('transaction is aborted') > 0:
		    raise Warning, self.conn.notices.pop()

	return self.__fetchOneRow()

    def fetchmany(self, sz=None):
        if self.closed:
            raise InterfaceError, "fetchmany failed - the cursor is closed."

	if self.conn == None:
	    raise Error, "connection is closed."

	if self.res == None:
	    raise Error, \
		  "fetchmany() failed - cursor does not contain a result."
	elif self.res.resultType != RESULT_DQL:
	    if self.close == None:
		raise Error, \
		      "fetchmany() Failed - cursor does not contain any rows."
	
	if sz == None:
	    sz = self.arraysize
	else:
	    self.__dict__["arraysize"] = abs(sz)

	if sz < 0:
	    return self.fetchall()
	    
	_list = []

        # While there are still results in the PgResult object, append them
        # to the list of results.
	while self._idx_ < self._rows_ and sz > 0:
	    _list.append(self.__fetchOneRow())
	    sz = sz - 1

        # If still need more results to fullfill the request, fetch them from
        # the PostgreSQL portal.
	if self.closed == 0 and sz > 0:
	    _nl = len(self.conn.notices)
	    self.res = self.conn.conn.query("FETCH %d FROM %s" %
					    (sz, self.name))
	    self._rows_ = self.res.ntuples
	    self._idx_ = 0

	    if len(self.conn.notices) != _nl:
		_drop = self.conn.notices[-1]
		if _drop.find('transaction is aborted') > 0:
		    raise Warning, self.conn.notices.pop()

	return self.__fetchManyRows(sz, _list)

    def fetchall(self):
        if self.closed:
            raise InterfaceError, "fetchall failed - the cursor is closed."

	if self.conn == None:
	    raise Error, "connection is closed."

	if self.res == None:
	    raise Error, \
                  "fetchall() failed - cursor does not contain a result."
	elif self.res.resultType != RESULT_DQL:
	    if self.closed == None:
		raise Error, \
		      "fetchall() Failed - cursor does not contain any rows."
	    
	_list = []

        # While there are still results in the PgResult object, append them
        # to the list of results.
	while self._idx_ < self._rows_:
	    _list.append(self.__fetchOneRow())

        # Fetch the remaining results from the PostgreSQL portal.
	if self.closed == 0:
	    _nl = len(self.conn.notices)
	    self.res = self.conn.conn.query("FETCH ALL FROM %s" % self.name)
	    self._rows_ = self.res.ntuples
	    self._idx_ = 0

	    if len(self.conn.notices) != _nl:
		_drop = self.conn.notices[-1]
		if _drop.find('transaction is aborted') > 0:
		    raise Warning, self.conn.notices.pop()

	return self.__fetchManyRows(self._rows_, _list)

    def rewind(self):
        if self.closed:
            raise InterfaceError, "rewind failed - the cursor is closed."

	if self.conn == None:
	    raise Error, "connection is closed."

	if self.res == None:
	    raise Error, "rewind() failed - cursor does not contain a result."
	elif self.res.resultType != RESULT_DQL:
	    if self.closed == None:
		raise Error, \
		          "rewind() Failed - cursor does not contain any rows."
	    
	if self.closed == 0:
	    _nl = len(self.conn.notices)
	    self.res = self.conn.conn.query("MOVE BACKWARD ALL IN  %s" %
					    self.name)
	    self._rows_ = 0
	    if len(self.conn.notices) != _nl:
		_drop = self.conn.notices[-1]
		if _drop.find('transaction is aborted') > 0:
		    raise Warning, self.conn.notices.pop()

	self.__dict__["rowcount"] = -1
	self._idx_ = 0

    def setinputsizes(self, sizes):
        if self.closed:
            raise InterfaceError, "setinputsize failed - the cursor is closed."

    def setoutputsize(self, size, column=None):
        if self.closed:
            raise InterfaceError, "setoutputsize failed - the cursor is closed."
