#!/bin/bash
# This file contains the test framework, that is common code
# used by all tests.
# Copyright (C) 2018 by Rainer Gerhards

# "config settings" for the testbench
TB_TIMEOUT_STARTUP=400  # 40 seconds - Solaris sometimes needs this...
export TESTPORT=31514
export OUTFILE=librelp.out.log
export valgrind="valgrind --malloc-fill=ff --free-fill=fe --log-fd=1"
#export OPT_VERBOSE=-v # uncomment for debugging 

######################################################################
# functions
######################################################################

# $1 is name of pidfile to wait for
wait_process_startup_via_pidfile() {
	i=0
	while test ! -f $1 ; do
		sleep .100
		(( i++ ))
		if test $i -gt $TB_TIMEOUT_STARTUP
		then
		   printf "ABORT! Timeout waiting on startup\n"
		   exit 1
		fi
	done
	printf "program started up, pidfile $1 contains $(cat $1)\n"
}

# start receiver WITH valgrind, add receiver command line parameters after function name
startup_receiver_valgrind() {
	printf 'Starting Receiver...\n'
	$valgrind ./receive -p $TESTPORT --outfile $OUTFILE -F receive.pid $OPT_VERBOSE $* &
	export RECEIVE_PID=$!
	printf "got receive pid $RECEIVE_PID\n"
	wait_process_startup_via_pidfile receive.pid
	printf 'Receiver running\n'
}

# start receiver, add receiver command line parameters after function name
startup_receiver() {
	printf 'Starting Receiver...\n'
	./receive -p $TESTPORT -F receive.pid --outfile $OUTFILE $OPT_VERBOSE $* &
	export RECEIVE_PID=$!
	printf "got receive pid $RECEIVE_PID\n"
	wait_process_startup_via_pidfile receive.pid
	printf 'Receiver running\n'
}

stop_receiver() {
	if [ "$RECEIVE_PID" == "" ]; then
		printf 'oops - receiver pid not found in stop_receiver\n'
		return
	fi
	kill $RECEIVE_PID &> /dev/null
	wait $RECEIVE_PID
	printf 'receiver %d stopped\n' $RECEIVE_PID
}

abort_receiver() {
	if [ "$RECEIVE_PID" == "" ]; then
		printf 'oops - receiver pid not found in abort_receiver\n'
		return
	fi
set -x; set +v
	printf 'stopping receiver %d via abort method\n' $RECEIVE_PID
	kill -USR1 $RECEIVE_PID # &> /dev/null
	wait $RECEIVE_PID
	printf 'receiver %d stopped via abort method\n' $RECEIVE_PID
set +x; set +v
}

# $1 is the value to check for
# $2 (optinal) is the file to check
check_output() {
	EXPECTED="$1"
	if [ "$2" == "" ] ; then
		FILE_TO_CHECK="$OUTFILE"
	else
		FILE_TO_CHECK="$2"
	fi
	grep $3 "$EXPECTED" $FILE_TO_CHECK > /dev/null
	if [ $? -ne 0 ]; then
		printf "\nFAIL: expected message not found. Expected:\n"
		printf "%s\n" "$EXPECTED"
		printf "\n$FILE_TO_CHECK actually is:\n"
		cat $FILE_TO_CHECK
		exit 1
	fi
}


# wait until $TESTPORT is no longer bound, e.g. for session closure
# TODO: evaluate if this function is really used and, if not,
# TODO: remove again - 2018-11-19 rgerhards
wait_testport_available() {
	while true; do
		printf 'checking NETSTAT\n'
		if ! netstat -tp | grep -q $TESTPORT; then
			break
		fi
		sleep 1
	done
}


# $1 is the value to check for
# $2 (optinal) is the file to check
check_output_only() {
	EXPECTED="$1"
	if [ "$2" == "" ] ; then
		FILE_TO_CHECK="$OUTFILE"
	else
		FILE_TO_CHECK="$2"
	fi
#	printf "\ncheck_output_only on $FILE_TO_CHECK with '$EXPECTED'\n"
	grep -q "$EXPECTED" $FILE_TO_CHECK;
	if [ $? -ne 0 ]; then
		# False
#		printf "\ncheck_output_only FALSE \n";
		return 1;
	else
		# true
#		printf "\ncheck_output_only TRUE \n";
		return 0;
	fi
}

# cleanup temporary
# note: on solaris,
# get full command line: /usr/ucb/ps awwx
# find who listens on port:
# netstat -an | grep $TESTPORT
# ./CI/solaris-findport.sh $TESTPORT
cleanup() {
	if [ "$(uname)" == "SunOS" ] ; then
		pkill -x receive
		echo pkill result $?
	fi

	if [ -f receive.pid ]; then
		kill -9 $(cat receive.pid) &> /dev/null
	fi

	rm -f -- receive.pid $OUTFILE *.err.log error.out.log
}

# cleanup at end of regular test run
terminate() {
	cleanup
	printf "$0 SUCCESS\n"
}

# check that the output file contains correct number of messages
# Works on $OUTFILE
# TODO: check sequence, so that we do not have duplicates...
check_msg_count() {
	printf 'We have %s lines in %s\n' $(wc -l < $OUTFILE) $OUTFILE
	if ! ./chkseq -s1 -e$NUMMESSAGES -f$OUTFILE -d ; then
		exit 1
	fi
	return
	lines=$(wc -l < $OUTFILE)
	if [ "$lines" -ne $NUMMESSAGES ]; then
		printf 'FAIL: message count not correct for %s\n' $OUTFILE
		printf 'Have %s lines, expected %d\n' "$lines" $NUMMESSAGES
		exit 1
	fi
}

######################################################################
# testbench initialization code - do this LAST here in the file
######################################################################
printf "============================================================\n"
printf "Test: $0\n"
printf "============================================================\n"

# on Solaris we still have some issues sometimes. Please keep this
# informational info inside the framework until this can be totally
# considered revolved - rgerhards, 2018-04-17
ps -ef|grep receive
if [ "$(uname)" == "XXSunOS" ] ; then
	/usr/ucb/ps awwx
	echo pgrep
	pgrep receive
	echo next
	ps -efl
	netstat -an | grep $TESTPORT
	CI/solaris-findport.sh $TESTPORT
fi

cleanup # we do cleanup in case it was not done previously
