/* -*- mode: c; c-file-style: "gnu" -*-
 * os_sendfile.c -- OS-dependent sendfile() wrappers.
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file os_sendfile.c
 * OS-dependent sendfile() wrappers.
 *
 * For every different sendfile() or look-a-like call, a wrapper is
 * placed into this file. If there is support to send a buffer using
 * that call (like on Solaris), another wrapper is put here for that.
 */

#include "system.h"

#include <stddef.h>
#include <sys/types.h>
#ifdef HAVE_SYS_SENDFILE_H
#  ifdef _FILE_OFFSET_BITS
#    define __OS_S _FILE_OFFSET_BITS
#    undef _FILE_OFFSET_BITS
#    undef __USE_FILE_OFFSET64
#  endif
#include <sys/sendfile.h>
#  if __OS_S
#    define _FILE_OFFSET_BITS __OS_S
#    define __USE_FILE_OFFSET64
#    undef __OS_S
#  endif
#endif
#include <sys/socket.h>
#include <sys/uio.h>
#include <unistd.h>

#include "options.h"
#include "os.h"
#include "thy.h"

#if HAVE_SENDFILE_LINUX
int
thy_sendfile (int fd, int s, off_t *offset, size_t count)
{
  ssize_t sent;
  off_t pos;

  pos = lseek (s, *offset, SEEK_SET);
  if (pos == (off_t) - 1)
    return -1;

  sent = sendfile (fd, s, NULL, count);
  if (sent != -1)
    {
      *offset += sent;
      return 0;
    }
  return -1;
}
#endif

#if HAVE_SENDFILE_FREEBSD
int
thy_sendfile (int fd, int s, off_t *offset, size_t count)
{
  off_t sbytes = 0;
  int rval = sendfile (s, fd, *offset, count, NULL, &sbytes, 0);

  if (rval != -1 || errno == EAGAIN)
    *offset += sbytes;
  return rval;
}
#endif

#if HAVE_SENDFILE_HPUX
#ifndef sbsize_t
#define sbsize_t ssize_t
#endif

int
thy_sendfile (int fd, int s, off_t *offset, size_t count)
{
  sbsize_t sbytes = sendfile (fd, s, *offset, count, NULL, 0);

  if (sbytes != -1)
    *offset += sbytes;
  return (sbytes == -1) ? sbytes : 0;
}
#endif

#if HAVE_SENDFILE_AIX
int
thy_sendfile (int fd, int s, off_t *offset, size_t count)
{
  struct sf_parms sf;
  int rval;

  sf.header_data = NULL;
  sf.header_length = 0;
  sf.file_descriptor = s;
  sf.file_offset = *offset;
  sf.file_bytes = count;
  sf.trailer_data = NULL;
  sf.trailer_length = 0;

  rval = send_file (&fd, &sf, 0);

  if (rval != -1)
    *offset += sf.bytes_sent;
  return rval;
}
#endif

#if NEED_SENDFILE_EMU
int
thy_sendfile (int fd, int s, off_t *offset, size_t count)
{
  char buffer[_THY_BUFSIZE];
  ssize_t rbytes;
  int sbytes;
  size_t len;

  if (count < sizeof (buffer))
    len = count;
  else
    len = sizeof (buffer);

  lseek (s, *offset, SEEK_SET);
  rbytes = read (s, buffer, len);
  sbytes = send (fd, buffer, rbytes, 0);
  if (sbytes < 0)
    return -1;
  *offset += sbytes;
  return 0;
}
#endif

#if HAVE_DECL_SFV_FD_SELF
int
thy_sendbuffer (int fd, void *buffer, size_t size)
{
  off_t baddr = (off_t)buffer;

  return sendfile (fd, SFV_FD_SELF, &baddr, size);
}
#else
int
thy_sendbuffer (int fd, void *buffer, size_t size)
{
  return write (fd, buffer, size);
}
#endif
