// sffview.cpp
//
// This file is part of sffview, a program to view
// structured fax files (sff).

// Copyright (C) 2000 Peter Schaefer
//
// Permission to use, copy, modify, and distribute this software and its
// documentation for any purpose and without fee is hereby granted, provided
// that the above copyright notice appear in all copies. This software
// is provided "as is" without express or implied warranty.
//
// You can contact the author by email at peter.schaefer@gmx.de.
//

// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
#pragma hdrstop
#endif

#ifndef WX_PRECOMP
#include "wx/wx.h"
#endif

#include "wx/image.h"
#include "wx/wfstream.h"
#include "wx/toolbar.h"
#include "inc/sfftobmp.h"

#ifndef __WXMSW__
#include "bitmaps/open.xpm"
#include "bitmaps/help.xpm"
#include "bitmaps/prev.xpm"
#include "bitmaps/next.xpm"
#endif

// ----------------------------------------------------------------------------
// Class declarations
// ----------------------------------------------------------------------------

// forwards

class MyFrame;
class MyApp;

// MyCanvas

class MyCanvas: public wxScrolledWindow
{
public:
    MyCanvas() {};
    MyCanvas( wxWindow *parent, wxWindowID, const wxPoint &pos, const wxSize &size );
    ~MyCanvas();

    void OnPaint( wxPaintEvent &event );
    bool OpenFile(const wxString& strFilename);

    void DisplayPage(int nPage);

    int  GetPageCount();

    wxBitmap  *my_bmp;
    CSffFile  *my_file;

    DECLARE_DYNAMIC_CLASS(MyCanvas)
    DECLARE_EVENT_TABLE()
};

// MyFrame

class MyFrame: public wxFrame
{
public:
    MyFrame();
		
		void RecreateToolbar();
		void UpdateUI();

    void OnAbout( wxCommandEvent &event );
    void OnQuit( wxCommandEvent &event );
    void OnFileOpen( wxCommandEvent &event );

    void OnPrevPage( wxCommandEvent &event );
    void OnNextPage( wxCommandEvent &event );

    int               m_nCurrentPage;
    int               m_nPageCount;
    MyCanvas         *m_canvas;

    DECLARE_DYNAMIC_CLASS(MyFrame)
    DECLARE_EVENT_TABLE()
};

// MyApp

class MyApp: public wxApp
{
public:
    virtual bool OnInit();
};

// ----------------------------------------------------------------------------
// wxSFFHandler (not used currently)
// ----------------------------------------------------------------------------

#if defined(wxSFFHandler)

#define wxBITMAP_TYPE_SFF 49

class WXDLLEXPORT wxSFFHandler: public wxImageHandler
{
  DECLARE_DYNAMIC_CLASS(wxSFFHandler)

public:

  inline wxSFFHandler()
  {
      m_name = "SFF file";
      m_extension = "sff";
      m_type = wxBITMAP_TYPE_SFF;
      m_mime = "image/sff";
  };

  virtual bool LoadFile( wxImage *image, wxInputStream& stream, bool verbose=TRUE  );
  virtual bool SaveFile( wxImage *image, wxOutputStream& stream, bool verbose=TRUE );
  virtual bool DoCanRead( wxInputStream& stream );
};

#if !USE_SHARED_LIBRARIES
IMPLEMENT_DYNAMIC_CLASS(wxSFFHandler,wxImageHandler)
#endif

#if wxUSE_STREAMS

bool wxSFFHandler::DoCanRead( wxInputStream& stream )
{
	TSFFFileHeader dh;
  unsigned char SFFID[4] = { 0x53, 0x66, 0x66, 0x66 };

  stream.Read(&dh, sizeof(dh));
  if (stream.LastError() == wxStream_NOERROR) {
    if (::memcmp(&dh.sff_id, &SFFID, sizeof(SFFID)) == 0) {
      if (dh.version > 1) {
        return true;
      }
    }
  }
  return false;
}

bool wxSFFHandler::SaveFile( wxImage *image, wxOutputStream& stream, bool verbose )
{
  return false;
}

bool wxSFFHandler::LoadFile( wxImage *image, wxInputStream& stream, bool verbose )
{
	static wxByte acOutBuffer[5120];
	TSFFRecord rec;
  int i,j;

  try
  {
      CSffFile sff(stream);

      image->Destroy();
   
			long dwHeight = sff.GetPageHeight(1);
			long dwWidth  = sff.GetPageWidth(1);

      image->Create( dwWidth, dwHeight );

      /* alloc ram */
      wxUint8 *ptr = image->GetData();
      if (!ptr)
      {
          wxLogError( "Cannot allocate RAM for RGB data in file\n" );
          return FALSE;
      }

     /*
      * Decode SFF Records ...
      */

			sff.SeekPage(1);
      long dwBytesPerLine = dwWidth>>3;
      int bit, val;
      wxUint8 *data;
      long line = 0;
      while (sff.GetRecord(rec)) {
        switch(rec.type) {
		      case NORMAL :
			      if (sff.DecodeRecord(rec, acOutBuffer, sizeof(acOutBuffer))) {
              data = ptr;
              for (i = 0; i < dwBytesPerLine; i++) {
                for (bit = 0; bit < 8; bit++) {
                  val = ((acOutBuffer[i] & (0x80 >> bit)) ? 0x00 : 0xFF);
                  *data++ = val; *data++ = val; *data++ = val;
                }
              }
              ptr += dwWidth; ptr += dwWidth; ptr += dwWidth;
	            line++;
			      }
			      if (rec.pData != 0) free(rec.pData);
			      break;
		      case USERINFO :
			      // momentan unbercksichtigt
			      if (rec.pData != 0) free(rec.pData);
			      break;
		      case BADLINE :
            // Puffer wiederholen
            data = ptr;
            for (i = 0; i < dwBytesPerLine; i++) {
              for (bit = 0; bit < 8; bit++) {
                val = ((acOutBuffer[i] & (0x80 >> bit)) ? 0x00 : 0xFF);
                *data++ = val; *data++ = val; *data++ = val;
              }
            }
            ptr += dwWidth; ptr += dwWidth; ptr += dwWidth;
	          line++;
			      break;
		      case WHITESKIP :
            data = ptr;
			      for (j=0; j < rec.cb; ++j) {
              for (i = 0; i < dwBytesPerLine; i++) {
                for (bit = 0; bit < 8; bit++) {
                  *data++ = 0xFF; *data++ = 0xFF; *data++ = 0xFF;
                }
              }
              ptr += dwWidth; ptr += dwWidth; ptr += dwWidth;
              line++;
			      }
			      break;
		      default :
			      break;
        }
      }
	  }
	  catch (CSimpleException e) {
      wxLogError( e.GetReason() );
      image->Destroy();
      return FALSE;
	  }		
     
    image->SetMask( FALSE );
    return TRUE;
}
#endif // wxUSE_STREAMS

#endif // wxSFFHandler

// ----------------------------------------------------------------------------
// Implementation
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// constants
// ----------------------------------------------------------------------------

const int ID_QUIT  = 108;
const int ID_ABOUT = 109;
const int ID_OPEN  = 110;
const int ID_PREVPAGE = 111;
const int ID_NEXTPAGE  = 112;

const int ID_TOOLBAR = 500;

IMPLEMENT_APP(MyApp)

// ----------------------------------------------------------------------------
// MyCanvas
// ----------------------------------------------------------------------------

IMPLEMENT_DYNAMIC_CLASS(MyCanvas, wxScrolledWindow)

BEGIN_EVENT_TABLE(MyCanvas, wxScrolledWindow)
  EVT_PAINT(MyCanvas::OnPaint)
END_EVENT_TABLE()

MyCanvas::MyCanvas( wxWindow *parent, wxWindowID id,
                    const wxPoint &pos, const wxSize &size )
        : wxScrolledWindow( parent, id, pos, size, wxSUNKEN_BORDER )
{
  my_bmp = (wxBitmap*) NULL;
  my_file = (CSffFile *) NULL;
  SetBackgroundColour(*wxWHITE);
}

MyCanvas::~MyCanvas()
{
  if (my_bmp) delete my_bmp;
  if (my_file) delete my_file;
}

bool MyCanvas::OpenFile(const wxString& strFilename)
{
  bool rc = false;

  if (wxFileExists(strFilename))
  {
      try
      {
        my_file = new CSffFile(strFilename);
        rc = true;
      }
	    catch (CSimpleException e) {
        wxLogError( e.GetReason() );
	    }		
  }
  return rc;
}

void MyCanvas::DisplayPage(int nPage)
{
	if (my_file == NULL)
		return;

  if (my_bmp) delete my_bmp;
  my_bmp = (wxBitmap*) NULL;

  try
  {
		TSFFRecord rec;
	  long dwHeight = my_file->GetPageHeight(1);
	  long dwWidth  = my_file->GetPageWidth(1);
    wxUint8 *bitmap = (wxUint8 *)malloc(dwWidth*dwHeight);
   /*
    * Decode SFF Records ...
    */

	  my_file->SeekPage(nPage);
    long dwBytesPerLine = dwWidth>>3;
    wxUint8 *ptr = bitmap;
    while (my_file->GetRecord(rec)) {
      switch(rec.type) {
        case NORMAL :
	        my_file->DecodeRecord(rec, ptr, dwBytesPerLine);
          ptr += dwBytesPerLine;
	        if (rec.pData != 0) free(rec.pData);
	        break;
        case USERINFO :
	        // momentan unbercksichtigt
	        if (rec.pData != 0) free(rec.pData);
	        break;
        case BADLINE :
          // Puffer wiederholen
          memset(ptr, 0x00, dwBytesPerLine);
          ptr += dwBytesPerLine;
	        break;
        case WHITESKIP :
          memset(ptr, 0x00, dwBytesPerLine*rec.cb);
          ptr += dwBytesPerLine*rec.cb;
	        break;
        default :
	        break;
      }
    }
    my_bmp = new wxBitmap((const char *)bitmap, dwWidth, dwHeight, 1);
    free(bitmap);
    SetScrollbars( 1, 1, dwWidth, dwHeight);
    Refresh();
  }
  catch (CSimpleException e) {
    wxLogError( e.GetReason() );
  }		
}

int  MyCanvas::GetPageCount()
{
	if (my_file == NULL)
		return -1;
	return my_file->GetPageCount();
}

void MyCanvas::OnPaint( wxPaintEvent &WXUNUSED(event) )
{
  wxPaintDC dc( this );
  PrepareDC( dc );
  if (my_bmp && my_bmp->Ok()) dc.DrawBitmap( *my_bmp, 0, 0 );
}

// ----------------------------------------------------------------------------
// MyFrame
// ----------------------------------------------------------------------------

IMPLEMENT_DYNAMIC_CLASS( MyFrame, wxFrame )

BEGIN_EVENT_TABLE(MyFrame,wxFrame)
  EVT_MENU    (ID_OPEN,  MyFrame::OnFileOpen)
  EVT_MENU    (ID_PREVPAGE, MyFrame::OnPrevPage)
  EVT_MENU    (ID_NEXTPAGE, MyFrame::OnNextPage)
  EVT_MENU    (ID_ABOUT, MyFrame::OnAbout)
  EVT_MENU    (ID_QUIT,  MyFrame::OnQuit)
END_EVENT_TABLE()

MyFrame::MyFrame()
       : wxFrame( (wxFrame *)NULL, -1, "SFF Viewer",
                  wxPoint(20,20), wxSize(800,600) )
{
  wxMenu *file_menu = new wxMenu();
  file_menu->Append( ID_OPEN, "&ffnen..");
  file_menu->Append( ID_ABOUT, "Info..");
  file_menu->Append( ID_QUIT, "B&eenden");

  wxMenuBar *menu_bar = new wxMenuBar();
  menu_bar->Append(file_menu, "&Datei");

  SetMenuBar( menu_bar );

  CreateStatusBar(2);
  int widths[] = { -1, 100 };
  SetStatusWidths( 2, widths );

  m_canvas = new MyCanvas( this, -1, wxPoint(0,0), wxSize(800,600) );
  m_canvas->SetScrollbars( 1, 1, 800, 600);

  m_nCurrentPage = 1;
  m_nPageCount = 1;

	RecreateToolbar();
	
  SetStatusText("Bereit.");
}

void MyFrame::RecreateToolbar()
{
    // delete and recreate the toolbar
    wxToolBarBase *toolBar = GetToolBar();
    delete toolBar;

    SetToolBar(NULL);

    long style = wxNO_BORDER | wxTB_FLAT | wxTB_DOCKABLE | wxTB_HORIZONTAL;

    toolBar = CreateToolBar(style, ID_TOOLBAR);
    toolBar->SetMargins( 4, 4 );

    // Set up toolbar
    wxBitmap toolBarBitmaps[4];

    toolBarBitmaps[0] = wxBITMAP(open);
    toolBarBitmaps[1] = wxBITMAP(prev);
    toolBarBitmaps[2] = wxBITMAP(next);
    toolBarBitmaps[3] = wxBITMAP(help);

#ifdef __WXMSW__
    int width = 24;
#else
    int width = 16;
#endif

    int currentX = 5;

    toolBar->AddTool(ID_OPEN, toolBarBitmaps[0], wxNullBitmap, FALSE, currentX, -1, (wxObject *) NULL, "Datei ffnen");
    currentX += width + 5;
    toolBar->AddSeparator();
    toolBar->AddTool(ID_PREVPAGE, toolBarBitmaps[1], wxNullBitmap, FALSE, currentX, -1, (wxObject *) NULL, "vorherige Seite");
    currentX += width + 5;
    toolBar->AddTool(ID_NEXTPAGE, toolBarBitmaps[2], wxNullBitmap, FALSE, currentX, -1, (wxObject *) NULL, "nchste Seite");
    currentX += width + 5;
    toolBar->AddSeparator();
    toolBar->AddTool(ID_ABOUT, toolBarBitmaps[3], wxNullBitmap, FALSE, currentX, -1, (wxObject *) NULL, "Info");

    toolBar->Realize();

//    toolBar->SetRows(1);
   	
   	toolBar->EnableTool( ID_PREVPAGE, FALSE );
   	toolBar->EnableTool( ID_NEXTPAGE, FALSE );
}

void MyFrame::UpdateUI()
{
  char acBuf[20];
  wxToolBarBase *tb = GetToolBar();
 	tb->EnableTool( ID_PREVPAGE, (m_nPageCount > 1) && (m_nCurrentPage > 1));
 	tb->EnableTool( ID_NEXTPAGE, (m_nPageCount > 1) && (m_nCurrentPage < m_nPageCount));
 	sprintf(acBuf, "%d/%d", m_nCurrentPage, m_nPageCount);
 	SetStatusText(acBuf, 1);
 	
}

void MyFrame::OnQuit( wxCommandEvent &WXUNUSED(event) )
{
  Close( TRUE );
}

void MyFrame::OnPrevPage( wxCommandEvent &WXUNUSED(event) )
{
	if (m_nCurrentPage > 1) {
		m_canvas->DisplayPage(--m_nCurrentPage);
	}
	UpdateUI();
}

void MyFrame::OnNextPage( wxCommandEvent &WXUNUSED(event) )
{
	if (m_nCurrentPage < m_nPageCount) {
	  m_canvas->DisplayPage(++m_nCurrentPage);
	}
	UpdateUI();
}

void MyFrame::OnAbout( wxCommandEvent &WXUNUSED(event) )
{
  (void)wxMessageBox(
		"This is sffview, a program to view structured fax files (sff)\n\n"
		"This software and its documentation is\n"
		"Copyright (C) 2000 Peter Schaefer\n\n"
		"Permission to use, copy, modify, and distribute this software and its\n"
		"documentation for any purpose and without fee is hereby granted, provided\n"
		"that the above copyright notice appear in all copies. This software\n"
		"is provided 'as is' without express or implied warranty.\n\n"
		"You can contact the author by email at peter.schaefer@gmx.de",
		"ber SFF-Viewer", wxICON_INFORMATION | wxOK );
}

void MyFrame::OnFileOpen( wxCommandEvent& WXUNUSED(event) )
{
    wxFileDialog dialog(this, "Datei ffnen", "", "", "Faxdateien (*.sff)|*.sff", 0);

    if (dialog.ShowModal() == wxID_OK)
    {
       	m_nCurrentPage = 1;
      	m_nPageCount = 1;
        if (m_canvas->OpenFile(dialog.GetPath())) {
	       	m_nCurrentPage = 1;
        	m_nPageCount=m_canvas->GetPageCount();
        	m_canvas->DisplayPage(m_nCurrentPage);
					SetStatusText("Datei erfolgreich geladen.");
        } else {
					SetStatusText("Datei konnte nicht geladen werden.");
        }
				UpdateUI();
    }
}

//-----------------------------------------------------------------------------
// MyApp
//-----------------------------------------------------------------------------

bool MyApp::OnInit()
{
  wxFrame *frame = new MyFrame();
  frame->Show( TRUE );

  return TRUE;
}

