#
# Copyright (c) 2002, 2003, 2004 Art Haas
#
# This file is part of PythonCAD.
# 
# PythonCAD is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# PythonCAD is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with PythonCAD; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# available units
#

import types

MILLIMETERS = 0
MICROMETERS = 1
METERS = 2
KILOMETERS = 3
INCHES = 4
FEET = 5
YARDS = 6
MILES = 7

def get_all_units():
    unitlist = []
    unitlist.append('Millimeters')
    unitlist.append('Micrometers')
    unitlist.append('Meters')
    unitlist.append('Kilometers')
    unitlist.append('Inches')
    unitlist.append('Feet')
    unitlist.append('Yards')
    unitlist.append('Miles')
    return unitlist

class Unit(object):
    """A class defining a unit for linear dimensions.

The Unit class is used to establish what unit to
assign distances between points. The class contains
two methods used to scale distance value to and from
an equivalent distance in millimeters.

toMillimeters(): Convert a distance to millimeters
fromMillimeters(): Convert a distance from millimeters
    """
    def __init__(self, unit=None):
        _unit = unit
        if _unit is None:
            _unit = MILLIMETERS
        if (_unit != MILLIMETERS and
            _unit != MICROMETERS and
            _unit != METERS and
            _unit != KILOMETERS and
            _unit != INCHES and
            _unit != FEET and
            _unit != YARDS and
            _unit != MILES):
            raise ValueError, "Invalid unit choice: " + str(unit)
        self.__unit = _unit

    def getUnit(self):
        return self.__unit

    def getStringUnit(self):
        _u = self.__unit
        if _u == MILLIMETERS:
            _str = 'millimeters'
        elif _u == MICROMETERS:
            _str = 'micrometers'
        elif _u == METERS:
            _str = 'meters'
        elif _u == KILOMETERS:
            _str = 'kilometers'
        elif _u == INCHES:
            _str = 'inches'
        elif _u == FEET:
            _str = 'feet'
        elif _u == YARDS:
            _str = 'yards'
        elif _u == MILES:
            _str = 'miles'
        else:
            raise ValueError, "Unexpected unit: %d" % _u
        return _str

    def setStringUnit(self, unit):
        if not isinstance(unit, types.StringTypes):
            raise TypeError, "Unexpected unit string: " + str(unit)
        _ul = str(unit.lower())
        if _ul == 'millimeters':
            _unit = MILLIMETERS
        elif _ul == 'micrometers':
            _unit = MICROMETERS
        elif _ul == 'meters':
            _unit = METERS
        elif _ul == 'kilometers':
            _unit = KILOMETERS
        elif _ul == 'inches':
            _unit = INCHES
        elif _ul == 'feet':
            _unit = FEET
        elif _ul == 'yards':
            _unit = YARDS
        elif _ul == 'miles':
            _unit = MILES
        else:
            raise ValueError, "Unexpected unit string: %s" % unit
        self.__unit = _unit
            
    def setUnit(self, unit):
        if (unit != MILLIMETERS and
            unit != MICROMETERS and
            unit != METERS and
            unit != KILOMETERS and
            unit != INCHES and
            unit != FEET and
            unit != YARDS and
            unit != MILES):
            raise ValueError, "Invalid unit choice: " + str(unit)
        self.__unit = unit

    unit = property(getUnit, setUnit, None, "Basic unit.")

    def toMillimeters(self, value):
        """Scale a value to the equivalent distance in millimeters.

toMillimeters(value)
        """
        _v = value
        if not isinstance(_v, float):
            _v = float(value)
        if self.__unit == MILLIMETERS:
            _sv = _v
        elif self.__unit == MICROMETERS:
            _sv = _v * 1e-3
        elif self.__unit == METERS:
            _sv = _v * 1e3
        elif self.__unit == KILOMETERS:
            _sv = _v * 1e6
        elif self.__unit == INCHES:
            _sv = _v * 25.4
        elif self.__unit == FEET:
            _sv = _v * 304.8
        elif self.__unit == YARDS:
            _sv = _v * 914.4
        elif self.__unit == MILES:
            _sv = _v * 1609344.4
        else:
            raise ValueError, "Undefined unit value! " + str(self.__unit)
        return _sv

    def fromMillimeters(self, value):
        """Scale a value from an equivalent distance in millimeters.

fromMillimeters(value)
        """
        _v = value
        if not isinstance(_v, float):
            _v = float(value)
        if self.__unit == MILLIMETERS:
            _sv = _v
        elif self.__unit == MICROMETERS:
            _sv = _v * 1e3
        elif self.__unit == METERS:
            _sv = _v * 1e-3
        elif self.__unit == KILOMETERS:
            _sv = _v * 1e-6
        elif self.__unit == INCHES:
            _sv = _v / 25.4
        elif self.__unit == FEET:
            _sv = _v / 304.8
        elif self.__unit == YARDS:
            _sv = _v / 914.4
        elif self.__unit == MILES:
            _sv = _v / 1609344.4
        else:
            raise ValueError, "Undefined unit value! " + str(self.__unit)
        return _sv

def unit_string(value):
    """Return a text string for the integer unit value.

unit_string(value)
    """
    if value == MILLIMETERS:
        _str = 'millimeters'
    elif value == MICROMETERS:
        _str = 'micrometers'
    elif value == METERS:
        _str = 'meters'
    elif value == KILOMETERS:
        _str = 'kilometers'
    elif value == INCHES:
        _str = 'inches'
    elif value == FEET:
        _str = 'feet'
    elif value == YARDS:
        _str = 'yards'
    elif value == MILES:
        _str = 'miles'
    else:
        raise ValueError, "Unexpected unit: " + str(value)
    return _str
        
        
