
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stddef.h>
#include <string.h>
#include <pcap.h>
#include <event.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>

#include <debug/log.h>
#include <debug/memory.h>

#include "capture.h"
#include "config.h"
#include "link.h"
#include "flow.h"

struct capture
{
   const struct link *link;
   char errbuf[PCAP_ERRBUF_SIZE];
   pcap_t *pcap;
   struct event event;
   uint64_t runt_errors;
};

static struct capture capture;

static int packet_to_flowrec (struct flow *flow,const void *buf,size_t len)
{
   const struct iphdr *ip = buf;

   if (!len || len < (ip->ihl << 2))
	 {
		capture.runt_errors++;
		return (-1);
	 }

   buf += ip->ihl << 2;
   len -= ip->ihl << 2;

   flow->proto = ip->protocol;
   flow->saddr = ntohl (ip->saddr);
   flow->daddr = ntohl (ip->daddr);
   flow->tos = ip->tos;
   flow->id = ntohs (ip->id);

   /*
	* Since we don't know the order in which we will receive
	* fragments and only the first fragment contains the header
	* information we first create a flow without all the upper
	* layer info.
	*
	* When the time comes and we receive the first fragment, we
	* fill in the blanks and aggregate the flow.
	*
	* If either the fragment flow or the real flow expires before
	* this process is complete (or during a fragment flow), two
	* different flows will be exported. There is not much we can
	* do about situation though.
	*
	* All the juicy bits of how fragments are matched and flows
	* aggregated can be found in flow.c
	*/

   if (ntohs (ip->frag_off) & (IP_OFFMASK | IP_MF))
	 {
		flow->flags |= FLOW_FRAG;

		if (ntohs (ip->frag_off) & IP_OFFMASK)
		  return (0);
	 }

   switch (flow->proto)
	 {
	  case IPPROTO_UDP:
		  {
			 const struct udphdr *udp = buf;

			 if (len < sizeof (struct udphdr))
			   {
				  capture.runt_errors++;
				  return (-1);
			   }

			 flow->sport = ntohs (udp->source);
			 flow->dport = ntohs (udp->dest);
		  }
		break;

	  case IPPROTO_TCP:
		  {
			 const struct tcphdr *tcp = buf;

			 if (len < sizeof (struct tcphdr))
			   {
				  capture.runt_errors++;
				  return (-1);
			   }

			 flow->sport = ntohs (tcp->source);
			 flow->dport = ntohs (tcp->dest);

			 if (tcp->fin)
			   flow->flags |= FLOW_FIN;

			 /*
			  * We're only interested in ACKs after we received
			  * a FIN. If this is the side who closed the connection,
			  * the FIN would've been saved earlier on, else we
			  * would've gotten the FIN now.
			  */

			 if ((flow->flags & FLOW_FIN) && tcp->ack)
			   flow->flags |= FLOW_ACK;

			 if (tcp->rst)
			   flow->flags |= FLOW_RST;
		  }
		break;
	 }

   return (0);
}

static void capture_process (u_char *user,const struct pcap_pkthdr *h,const u_char *sp)
{
   int len;

   if ((len = capture.link->decode (sp,h->caplen)) >= 0)
	 {
		struct pcap_pkthdr hdr = *h;
		struct flow flow;

		hdr.caplen -= len;
		hdr.len -= len;
		sp += len;

		memset (&flow,0L,sizeof (struct flow));

		flow.timestamp = h->ts.tv_sec;
		flow.octets = h->len;
		flow.packets = 1;

		if (packet_to_flowrec (&flow,sp,h->caplen))
		  return;

		if ((flow.flags & (FLOW_FIN | FLOW_ACK)) == (FLOW_FIN | FLOW_ACK) ||
			(flow.flags & FLOW_RST))
		  {
			 struct flow *tmp;

			 if ((tmp = flow_find (&flow)) != NULL)
			   flow_remove (tmp);

			 return;
		  }

		flow_insert (&flow);
	 }
}

static void capture_dispatch (int fd,short event,void *arg)
{
   if (pcap_dispatch (capture.pcap,-1,capture_process,NULL) < 0)
	 {
		log_printf (LOG_ERROR,"%s\n",capture.errbuf);
		capture_close ();
	 }
}

int capture_open (const struct config *config)
{
   int type;

   if ((capture.pcap = pcap_open_live (config->iface,config->snaplen,1,1000,capture.errbuf)) == NULL)
	 {
		log_printf (LOG_ERROR,"%s\n",capture.errbuf);
		return (-1);
	 }

   if (pcap_setnonblock (capture.pcap,1,capture.errbuf))
	 {
		log_printf (LOG_ERROR,"%s\n",capture.errbuf);
		pcap_close (capture.pcap);
		return (-1);
	 }

   type = pcap_datalink (capture.pcap);

   if ((capture.link = link_find (type)) == NULL)
	 {
		log_printf (LOG_ERROR,"unsupported link layer: %d\n",type);
		pcap_close (capture.pcap);
		return (-1);
	 }

   if (config->expr != NULL)
	 {
		bpf_u_int32 net,mask;
		struct bpf_program bpf;

		if (pcap_lookupnet (config->iface,&net,&mask,capture.errbuf))
		  {
			 log_printf (LOG_WARNING,"%s\n",capture.errbuf);
			 net = mask = 0;
		  }

		if (pcap_compile (capture.pcap,&bpf,config->expr,1,mask))
		  {
			 log_printf (LOG_ERROR,"%s\n",pcap_geterr (capture.pcap));
			 pcap_close (capture.pcap);
			 return (-1);
		  }

		if (pcap_setfilter (capture.pcap,&bpf))
		  {
			 log_printf (LOG_ERROR,"%s\n",pcap_geterr (capture.pcap));
			 pcap_freecode (&bpf);
			 pcap_close (capture.pcap);
			 return (-1);
		  }

		pcap_freecode (&bpf);
	 }

   if (flow_open (config->flows,config->active,config->inactive))
	 {
		log_printf (LOG_ERROR,"failed to allocate memory: %m\n");
		pcap_close (capture.pcap);
		return (-1);
	 }

   event_set (&capture.event,
			  pcap_fileno (capture.pcap),
			  EV_READ | EV_PERSIST,
			  capture_dispatch,NULL);

   if (event_add (&capture.event,NULL))
	 {
		log_printf (LOG_ERROR,"failed to add event handler: %m\n");
		flow_close ();
		pcap_close (capture.pcap);
		return (-1);
	 }

   return (0);
}

void capture_close (void)
{
   static volatile int called = 0;

   if (!called)
	 {
		called = 1;
		event_del (&capture.event);
		pcap_close (capture.pcap);
		flow_close ();
	 }
}

