#define PHIDGETS_INTERNAL

#include <phidgets/phidgets.h>

#include <debug.h>
#include <assert.h>

static bool initialised = false;

static bool match_serial_number(struct usb_dev_handle const* usbdev,
    void* custom, unsigned int len)
{
  char buffer[6];
  struct usb_dev_handle* usbdev_i = (struct usb_dev_handle*)usbdev;
  usb_get_string_simple(usbdev_i,
      usb_device(usbdev_i)->descriptor.iSerialNumber, buffer, 6);
  return atoi(buffer) == *(unsigned int*)custom;
}

Phidget* phidget_new_Phidget()
{
  TRACE("creating a new Phidget instance...");

  Phidget* ret = (Phidget*)malloc(sizeof(Phidget));
  if (!ret) {
    ERROR("could not allocate memory for Phidget instance.");
    return 0;
  }

  ret->hid_iface = hid_new_HIDInterface();

  phidget_reset_Phidget(ret);
  return ret;
}

void phidget_delete_Phidget(Phidget** const phid)
{
  if (!phid || !*phid) {
    ERROR("cannot delete NULL HIDInterface.");
    return;
  }

  free(*phid);
  *phid = 0;
}

void phidget_reset_Phidget(Phidget* const phid)
{
  if (!phid) {
    ERROR("cannot reset NULL Phidget.");
    return;
  }

  hid_reset_HIDInterface(phid->hid_iface);
  phid->hid_error = HID_RET_SUCCESS;
  phid->id[0] = '\0';
}

phidget_return phidget_init()
{
  if (phidget_is_initialised()) {
    ERROR("cannot initialise already initialised Phidgets library.");
    return PHIDGET_RET_ALREADY_INITIALISED;
  }
  
  if (hid_is_initialised()) {
    WARNING("HID subsystem already initialised.");
  }
  else {
    TRACE("initialising HID subsystem...");
    if (hid_init() != HID_RET_SUCCESS) {
      ERROR("failed to initialised HID subsystem.");
      return PHIDGET_RET_HID_ERROR;
    }
  }

  initialised = true;
  NOTICE("successfully initialised Phidgets library.");
  return PHIDGET_RET_SUCCESS;
}

phidget_return phidget_cleanup()
{
  if (!phidget_is_initialised()) {
    ERROR("cannot deinitialise uninitialised Phidgets library.");
    return PHIDGET_RET_NOT_INITIALISED;
  }

  if (!hid_is_initialised()) {
    WARNING("HID subsystem already deinitialised.");
  }
  else {
    TRACE("deinitialising HID subsystem...");
    if (hid_cleanup() != HID_RET_SUCCESS) {
      ERROR("failed to deinitialised HID subsystem.");
      return PHIDGET_RET_HID_ERROR;
    }
  }
  
  initialised = false;
  NOTICE("successfully deinitialised Phidgets library.");
  return PHIDGET_RET_SUCCESS;
}

bool phidget_is_initialised()
{
  return hid_is_initialised() && initialised;
}

phidget_return phidget_open(Phidget* const phid, int const interface,
    HIDInterfaceMatcher* const matcher, unsigned int serial,
    unsigned short retries)
{
  if (!phidget_is_initialised()) {
    ERROR("cannot open Phidget when Phidgets library has not been initialised.");
    return PHIDGET_RET_NOT_INITIALISED;
  }

  if (!phid) {
    ERROR("cannot open NULL Phidget.");
    return PHIDGET_RET_INVALID_PARAMETER;
  }

  if (phidget_is_opened(phid)) {
    ERROR("cannot open already opened Phidget.");
    return PHIDGET_RET_DEVICE_ALREADY_OPENED;
  }

  if (!matcher) {
    ERROR("cannot match against NULL HIDInterfaceMatcher.");
    return PHIDGET_RET_INVALID_PARAMETER;
  }

  TRACE("opening Phidget device with serial number %d...", serial);

  matcher->matcher_fn = match_serial_number;
  matcher->custom_data = &serial;
  matcher->custom_data_length = sizeof(serial);

  hid_return ret;
  if (retries) 
    ret = hid_force_open(phid->hid_iface, interface, matcher, retries);
  else 
    ret = hid_open(phid->hid_iface, interface, matcher);

  if (ret != HID_RET_SUCCESS) {
    phid->hid_error = ret;
    ERROR("failed to open Phidget device with serial number %d.", serial);
    return PHIDGET_RET_HID_ERROR;
  }

  snprintf(phid->id, sizeof(phid->id), "#%d (%s)", serial, phid->hid_iface->id);

  NOTICE("successfully opened Phidget %s.", phid->id);
  return PHIDGET_RET_SUCCESS;
}

phidget_return phidget_close(Phidget* const phid)
{
  if (!phid) {
    ERROR("cannot close NULL Phidget.");
    return PHIDGET_RET_INVALID_PARAMETER;
  }

  TRACE("closing Phidget %s...", phid->id);

  if (phidget_is_opened(phid)) {

    hid_return ret = hid_close(phid->hid_iface);
    if (ret != HID_RET_SUCCESS) {
      phid->hid_error = ret;
      ERROR("failed to close Phidget %s.", phid->id);
      return PHIDGET_RET_HID_ERROR;
    }
  }
  else WARNING("attempt to close unopened Phidget.");

  NOTICE("successfully closed Phidget %s.", phid->id);
  return HID_RET_SUCCESS;
}

bool phidget_is_opened(Phidget const* const phid)
{
  if (!phid) WARNING("attempt to query open status of NULL Phidget.");
  return phid && hid_is_opened(phid->hid_iface);
}

/* COPYRIGHT --
 *
 * This file is part of libphidgets, a user-space library for phidgets.
 * libphidgets is (c) 2003-2005 Martin F. Krafft <krafft@ailab.ch>
 * and distributed under the terms of the Artistic Licence.
 * See the ./COPYING file in the source tree root for more information.
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES
 * OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
