/***************************************************************************
                          c_time.h  -  description
                             -------------------
    begin                : Sat 19 Feb 2000
    copyright            : (C) 2000 by Martin Preuss
    email                : martin@aquamaniac.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/



#ifndef C_TIME_H
#define C_TIME_H 1


#include <time.h>
#include <errno.h>
#include <string>

#define LANG_en

namespace HBCI {

/** @short This class provides an operating system independant
 * interface to the time.
 *
 * @author Martin Preuss<martin@aquamaniac.de> */
class DLLIMPORT DateTime {
private:
    static const char _daynames[][16];
    static const char _monthnames[][16];
    static const char _shortdaynames[][4];
    static const char _shortmonthnames[][4];

    static void  _appendNum(std::string &b, int num, const char *fs);
    void _importTime(const struct tm &tin);

    unsigned short _seconds;      // (0-59)
    unsigned short _minutes;      // (0-59)
    unsigned short _hours;        // (0-23)
    unsigned short _year;         // (0-4095)
    unsigned short _month;        // (1-12)
    unsigned short _day;          // (1-31)
    unsigned short _weekday;      // (0-6, with 0 being Sunday)

public:
    unsigned short seconds() const { return _seconds;};      // (0-59)
    unsigned short minutes() const { return _minutes;};      // (0-59)
    unsigned short hours()   const { return _hours;};        // (0-23)
    unsigned short year()    const { return _year;};      // (0-4095)
    unsigned short month()   const { return _month;};     // (1-12)
    unsigned short day()     const { return _day;};       // (1-31)
    unsigned short weekday() const { return _weekday;};      // (0-6, with 0 being Sunday)

    DateTime(unsigned short year,
         unsigned short month,
         unsigned short day,
         unsigned short hour=0,
         unsigned short minute=0,
         unsigned short second=0);
    DateTime(const time_t &t);
    DateTime(const struct tm &t);
    DateTime();
    ~DateTime();

    /**
     * Checks the given time for correct values.
     * @return true if time is ok, false on error.
     */
    bool isValid() const;

    /**
     * Compares to times.
     * @return t1 smaller than t2: -1, t1 equals t2: 0, t1 greater than t2: 1
     * @param t2 time to compare against
     */
    int compareTime(const DateTime &t2) const;
    
    /**
     * Exports a time from this class to the os.
     * This method is only an internal one, only to be used by other
     * interfacing classes (like c_file).
     */
    time_t exportTime() const;

    /**
     * Exports a time from this class to the os.
     * This method is only an internal one, only to be used by other
     * interfacing classes (like c_file).
     */
    struct tm exportTimeTM() const;

    /**
     * Gets the current time.
     * @param gmt if true then gmt time will be returned instead of
     * localtime.
     */
    static DateTime currentTime(bool gmt=false);

    /**
     * Converts a time struct into a string according to the given
     * format string.
     * @return true if all went fine, false on error
     * @param fs the format string. This string may contain any character.
     * Some strings are escape sequences, which are replaced by values:
     * %s=seconds, %m=minutes, %h=hours, %D=day of the month,
     * %M=name of the month, %S short name of the month,
     * %N=number of the month, %Y=year,
     * %W=the weekdays name, %w=short weekday name.
     * Please note that these escape sequences are
     * case sensitive !
     */
    std::string timeString(std::string fs) const;

    /**
     * Adds the specified number of seconds to the given time.
     * You can also add negative values, thus in result subtracting
     * seconds ;-)
     * @author Martin Preuss<martin@aquamaniac.de>
     * @param seconds number of seconds to add.
     */
    DateTime addSeconds(long seconds) const ;
    
};


} /* namespace HBCI */


#endif
