/* 
 * $Id: ctkclist.c,v 1.54 2000/09/03 21:52:08 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include "ctk.h"
#include "ctkcolor.h"
#include "ctkutil.h"

/* Create a CList Widget */
CtkWidget* ctk_clist_new(gint columns)
{
	CtkCList *clist;
	
	clist = g_malloc(sizeof(CtkCList));
	
	ctk_clist_init(clist, columns);

	return ((CtkWidget *)clist);
}

/* CList Unselect Row */
void ctk_clist_unselect_row(CtkCList *clist, gint row, gint column)
{
      CtkTable* table = CTK_TABLE(clist);
      
      if (!clist)
	    return;
      
      if (clist->selected_row == row &&
	  clist->selected_col == column) 
      {
	      if (row != -1)
	      {
		    ctk_table_colour_row(table, row+1, CTK_WIDGET(table)->main_col);
	            ctk_signal_emit_by_name(CTK_OBJECT(clist), "unselect_row");
	      }
	      clist->cursor_row = -1;
	      clist->selected_row = -1;
	      clist->selected_col = -1;
      }
      
      ctk_draw_mark_changed(CTK_WIDGET(clist));
}

void ctk_clist_select_row(CtkCList* clist, gint row, gint column) 
{
      CtkTable*          self     = CTK_TABLE(clist);
      CtkWidget*         widget   = CTK_WIDGET(clist);
      CtkWidget*         parent;
      CtkScrolledWindow* swparent;
      gint               i;
      
      if (!clist)
	    return;
      
      /* Sanitize selection */
      if (column < 0) column = 0;
      if (row    < 0) row    = 0;
      
      if (column >= self->cols)
	    column = self->cols - 1;
      
      if (row >= self->rows - 1)
	    row = self->rows - 2;
	    
      /* Remove old selection */
      ctk_clist_unselect_row(clist, clist->selected_row, clist->selected_col);

      /* Set the user's selection */
      clist->cursor_row   = row;
      clist->selected_row = row;
      clist->selected_col = column;
	
      if (clist->cursor_row != -1)
      {
	    ctk_table_colour_row(self, clist->cursor_row+1, ((CtkWidget*)self)->selected_col);
      }
	
      /* If we're not packed, don't bother fixing scroll state */
      if (widget->node->parent &&
	  CTK_CHECK_TYPE(widget->node->parent->data, CtkTypeScrolledWindow))
      {
	    gint bottom;
	    gint y_of_row;
		
	    swparent = CTK_SCROLLED_WINDOW(widget->node->parent->data);
	    parent   = CTK_WIDGET(swparent);

	    /* We have to deal with the bottom bar hardcoded style */
	    bottom = parent->row + parent->height - swparent->hscrollbar_shown;
	    y_of_row = ctk_table_get_row_row(CTK_TABLE(clist), clist->cursor_row + 1);
	    
	    if (y_of_row)
	    {
	    	  for (i = 0; i <= y_of_row - bottom; i++)
	    	  {
	    	        ctk_scrolled_window_scroll(parent, CTK_SCROLL_DOWN);
	    	  }
	
		  for (i = 0; i < parent->row - y_of_row + clist->column_show; i++)
		  {
		        ctk_scrolled_window_scroll(parent, CTK_SCROLL_UP);
		  }
	    }
      }

      if (clist->selected_row >= 0 && clist->cursor_row >= 0)
            ctk_signal_emit_by_name(CTK_OBJECT(clist), "select_row");
      
      ctk_draw_mark_changed(widget);
}

/* CList mouse handle events */
gboolean ctk_clist_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      gint x   = event->x;
      gint y   = event->y;
      gint row;
      gint col;
      
      ctk_table_get_click_cell(CTK_TABLE(widget), &row, &col, x, y);
      ctk_clist_select_row(CTK_CLIST(widget), row-1, col);
      
      return TRUE;
}

gboolean ctk_clist_drag(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      CtkCList* clist;
      clist = CTK_CLIST(widget);
      ctk_clist_select_row(clist, clist->selected_row+event->dy, clist->selected_col);
      
      return TRUE;
}

void ctk_clist_move_cursor(CtkCList* clist, gint amount)
{
      ctk_clist_select_row(clist, clist->selected_row + amount, clist->selected_col);
}

/* CList key handle events */
gboolean ctk_clist_key_press(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
	CtkCList*  clist;
	CtkWidget* parent;
	gint       ch = event->keyval;

	if (!widget)
	    return FALSE;

	if (!widget->node->parent) 
		return FALSE;
	
	clist  = CTK_CLIST(widget);
	parent = CTK_WIDGET(widget->node->parent->data);

	if (!CTK_CHECK_TYPE(parent, CtkTypeScrolledWindow))
	{
		parent = NULL;
	}
	
	/* Down Key */
	if (ch == AK_ARROW_DOWN) {
	      ctk_clist_select_row(clist,
				       clist->selected_row + 1,
				       clist->selected_col);
	}
	/* Up Key */
	else if (ch == AK_ARROW_UP) {
	      ctk_clist_select_row(clist, 
				       clist->selected_row - 1,
				       clist->selected_col);
	} 
	/* Right Key */
	else if (ch == AK_ARROW_RIGHT)
	{
	      if (parent)
	      {
	      	ctk_scrolled_window_scroll(parent, CTK_SCROLL_RIGHT);
	      }
	}
	/* Left Key */
	else if (ch == AK_ARROW_LEFT)
	{
	      if (parent)
	      {
	      	ctk_scrolled_window_scroll(parent, CTK_SCROLL_LEFT);
	      }
	}
	/* Page Up */
	else if (ch == AK_PAGEUP)
	{
	      if (parent)
	      {
	      	ctk_clist_select_row(clist,
				       clist->selected_row - parent->height / 2,
				       clist->selected_col);
		  }
	}
	/* Page Down */
	else if (ch == AK_PAGEDOWN) 
	{
	      if (parent)
	      {
	      	ctk_clist_select_row(clist,
				       clist->selected_row + parent->height / 2,
				       clist->selected_col);
		  }
	}
	/* Enter Key */
	if ((ch == AK_ENTER) || (ch == ' '))
	{
	      if (clist->selected_col == -1)
		    clist->selected_col = 0;
	      ctk_clist_select_row(clist,clist->cursor_row,clist->selected_col);
	}

	return TRUE;
}

gboolean ctk_clist_focus_in_event(CtkWidget* object, 
				  gpointer event, gpointer data)
{
      CtkCList* clist = CTK_CLIST(object);

      object->selected_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_RED);
      
      /* Set Selection if it is not set */
      if (((CtkCList *)object)->selected_row == -1)
	    ((CtkCList *)object)->selected_row = 0;
      if (((CtkCList *)object)->cursor_row == -1)
	    ((CtkCList *)object)->cursor_row = 0;
      
      ctk_table_colour_row(CTK_TABLE(clist), clist->cursor_row+1, object->selected_col);
      
/*      Removed b/c this is not gtk compliant and causes trouble in SAS
 *	modules.
      ctk_clist_select_row(CTK_CLIST(object), 
			       clist->selected_row, 
			       clist->selected_col);
*/
      
      ctk_draw_mark_changed(object);
      
      return FALSE;
}

gboolean ctk_clist_focus_out_event(CtkWidget *object, 
				   gpointer event, gpointer data)
{
      CtkCList* clist = CTK_CLIST(object);

      object->selected_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_BLUE);
      
      ctk_table_colour_row(CTK_TABLE(clist), clist->cursor_row+1, object->selected_col);

/*      Removed b/c this is not gtk compliant and causes trouble in SAS
 *	modules.
      ctk_clist_select_row(CTK_CLIST(object), 
			   clist->selected_row, 
			   clist->selected_col);
*/      
      return FALSE;
}

/* Pack a widget into a clist column */
void ctk_clist_set_column_widget(CtkCList* clist, gint column, CtkWidget* widget)
{
	CtkTable* table = CTK_TABLE(clist);
	
	// FIXME: assumption that there is no column label already
	ctk_table_attach(table, widget, column*2, column*2+1, 0, 1,
		CTK_FILL|CTK_EXPAND, CTK_FILL, 0, 0);

	if (column != (table->cols-1)/2)
	{
		CtkWidget* vbar = ctk_vseparator_new();
		if (clist->column_show)	ctk_widget_show(vbar);
		else			ctk_widget_hide(vbar);
		ctk_table_attach(table, vbar, column*2+1, column*2+2, 0, 1,
			CTK_FILL, CTK_FILL, 0, 0);
	}
	
	ctk_table_colour_row(table, 0, ((CtkWidget*)table)->title_col);
	
	if (clist->column_show)	ctk_widget_show(widget);
	else					ctk_widget_hide(widget);
	
	ctk_size_mark_changed(CTK_WIDGET(clist));
}

/* Set clist column title */
void ctk_clist_set_column_title(CtkCList* clist, gint column, const gchar* text)
{
	CtkWidget *label;
	
	if (!text)
	    return;
	
	label = ctk_label_new(text);
	ctk_misc_set_alignment(CTK_MISC(label), 0, 0.5);
	
	ctk_clist_set_column_widget(clist, column, label);
}

/* Set the column width */
void ctk_clist_set_column_width(CtkCList *clist, gint column, gint width)
{
	if (width < 0)
	    width = 0;

	ctk_table_column_hard_size(CTK_TABLE(clist), column*2, width);
	
	ctk_size_mark_changed(CTK_WIDGET(clist));
}

/* Set the title columns show to TRUE */
void ctk_clist_column_titles_show(CtkCList* clist)
{
	GNode* scan;
	
	clist->column_show = TRUE;
	ctk_size_mark_changed(CTK_WIDGET(clist));
	
	if (!CTK_WIDGET(clist)->node)
		return;
	
	for (scan = CTK_WIDGET(clist)->node->children; scan; scan = scan->next)
	{
		CtkTableChild* child = CTK_TABLE_CHILD(scan->data);
		
		if (child->top_attach == 0)
		{
			ctk_widget_show(CTK_WIDGET(((CtkWidget*)child)->node->children->data));
		}
	}
}

/* Set the title columns show to FALSE */
void ctk_clist_column_titles_hide(CtkCList* clist)
{
	GNode* scan;
	
	clist->column_show = FALSE;
	ctk_size_mark_changed(CTK_WIDGET(clist));
	
	if (!CTK_WIDGET(clist)->node)
		return;

	for (scan = ((CtkWidget*)clist)->node->children; scan; scan = scan->next)
	{
		CtkTableChild* child = CTK_TABLE_CHILD(scan->data);
		
		if (child->top_attach == 0)
		{
			ctk_widget_hide(CTK_WIDGET(((CtkWidget*)child)->node->children->data));
		}
	}
}

void ctk_clist_insert_widget(CtkCList* clist, gint row, CtkWidget* widgets[])
{
	CtkTable* table = CTK_TABLE(clist);
	gint      i;
	
	ctk_table_insert_row(table, row+1);
	for (i = 0; i < (table->cols+1)/2; i++)
	{
		ctk_table_attach(table, widgets[i], i*2, i*2+1, row+1, row+2,
			CTK_FILL, CTK_FILL, 0, 0);
			
		if (i != (table->cols-1)/2)
		{
			CtkWidget* vbar = ctk_vseparator_new();
			ctk_widget_show(vbar);
			ctk_table_attach(table, vbar, i*2+1, i*2+2, row+1, row+2,
				CTK_FILL, CTK_FILL, 0, 0);
		}
	}
	
	if (clist->cursor_row > row)
	{
		clist->cursor_row++;
		clist->selected_row++;
	}
	
	if (clist->cursor_row == -1)
	{
		clist->cursor_row   = 0;
		clist->selected_row = 0;
		clist->selected_col = 0;
		ctk_table_colour_row(CTK_TABLE(clist), clist->cursor_row+1, ((CtkWidget*)clist)->selected_col);
	}
	
	ctk_size_mark_changed(CTK_WIDGET(clist));
}

/* Insert a row in a clist */
void ctk_clist_insert(CtkCList *clist, gint row, gchar *text[])
{	
      CtkWidget** widgets;
      gint        cols;
      gint        i;
	
      cols = (CTK_TABLE(clist)->cols+1)/2;
	
      widgets = (CtkWidget**)g_malloc(sizeof(CtkWidget*) * cols);
      for (i = 0; i < cols; i++)
      {
            if (((CtkTable*)clist)->layout.hetero.x[i*2].hard_size)
            {
                  gchar* foo = g_strdup(text[i]);
                  if (strlen(foo) > ((CtkTable*)clist)->layout.hetero.x[i*2].hard_size)
                  {
                  	foo[((CtkTable*)clist)->layout.hetero.x[i*2].hard_size] = 0;
                  }
                  widgets[i] = ctk_label_new(foo);
                  g_free(foo);
            }
            else
            {
	          widgets[i] = ctk_label_new(text[i]);
	    }
	    
	    ctk_misc_set_alignment(CTK_MISC(widgets[i]), 0, 0.5);
	    ctk_widget_show(widgets[i]);
      }
	
      ctk_clist_insert_widget(clist, row, widgets);
      g_free(widgets);
}

/* Append a row to a clist */
void ctk_clist_append(CtkCList* clist, gchar* text[])
{
	ctk_clist_insert(clist, ((CtkTable*)clist)->rows-1, text);
}
void ctk_clist_append_widget(CtkCList* clist, CtkWidget* widgets[])
{
	ctk_clist_insert_widget(clist, CTK_TABLE(clist)->rows-1, widgets);
}

/* Prepend a row to a clist */
void  ctk_clist_prepend(CtkCList* clist, gchar* text[])
{	
	ctk_clist_insert(clist, 0, text);
}
void ctk_clist_prepend_widget(CtkCList* clist, CtkWidget* widgets[])
{
	ctk_clist_insert_widget(clist, 0, widgets);
}

/* Clear all of the CLists rows */
void ctk_clist_clear(CtkCList* clist)
{
	CtkTable* table = CTK_TABLE(clist);
	
	if (!clist)
	    return;
	
	clist->selected_row = -1;
	clist->selected_col = -1;
	clist->cursor_row = -1;
	
	ctk_table_delete_below_row(table, 1);
	ctk_size_mark_changed(CTK_WIDGET(clist));
}

/**** FIXME: freeze&thaw are worthless right now */

/* Freeze CList */
void  ctk_clist_freeze(CtkCList *clist)
{
	if (!clist)
	    return;
	
	clist->frozen = TRUE;
}

/* Thaw CList */
void
    ctk_clist_thaw(CtkCList *clist)
{
	if (!clist)
	    return;
	
	clist->frozen = FALSE;
}

/* Get the widget in that cell */
gint ctk_clist_get_widget(CtkCList *clist, gint row, gint col, CtkWidget** widget)
{
	*widget = ctk_table_find_child(CTK_TABLE(clist), col*2, col*2+1, row+1, row+2);
	return 1;
}

/* Get Text from a spec */
gint ctk_clist_get_text(CtkCList *clist, gint row, gint col, gchar **text)
{
	CtkLabel* label;
	
	label = CTK_LABEL(ctk_table_find_child(CTK_TABLE(clist), col*2, col*2+1, row+1, row+2));
	
	if (label)
	{
		*text = label->text;
		return 1;
	}
	else
	{
		*text = "";
		return 0;
	}
}

/* Set the text of a specific row and col */
void ctk_clist_set_text(CtkCList* clist, gint row, gint col, const gchar* text)
{
	CtkLabel* label;
	
	label = CTK_LABEL(ctk_table_find_child(CTK_TABLE(clist), col, col+1, row, row+1));
	
	if (label)
	{
		ctk_label_set_text(label, text);
		ctk_size_mark_changed(CTK_WIDGET(clist));
	}
}

/* Remove a row from a clist */
void ctk_clist_remove(CtkCList *clist, gint row)
{
	ctk_table_delete_row(CTK_TABLE(clist), row + 1);
}

/* Initialize the CList Widget */
void ctk_clist_init(CtkCList* clist, gint columns)
{
	ctk_table_init(&clist->table, 1, columns*2 - 1, FALSE);
	((CtkObject *)clist)->type = CtkTypeCList;
	
	CTK_WIDGET(clist)->sensitive = TRUE;
	
	CTK_WIDGET(clist)->set_min_size  = &ctk_clist_min_size;
	CTK_WIDGET(clist)->set_real_size = &ctk_clist_real_size;
	
	clist->column_show  = FALSE;
	clist->selected_row = -1;
	clist->selected_col = -1;
	clist->cursor_row   = -1;
	clist->frozen       = FALSE;	
	
	ctk_signal_new("select_row",   CtkTypeCList);
	ctk_signal_new("unselect_row", CtkTypeCList);
	ctk_signal_new("click_column", CtkTypeCList);

	clist->selected_row = 0;
	clist->selected_col = 0;

	clist->column_show = FALSE;
	
	ctk_signal_connect(CTK_OBJECT(clist), "focus_in_event",
			   CTK_SIGNAL_FUNC(&ctk_clist_focus_in_event),  NULL);
	ctk_signal_connect(CTK_OBJECT(clist), "focus_out_event", 
			   CTK_SIGNAL_FUNC(&ctk_clist_focus_out_event), NULL);
	ctk_signal_connect(CTK_OBJECT(clist), "button_press_event",
			   CTK_SIGNAL_FUNC(&ctk_clist_button_press),    NULL);
	ctk_signal_connect(CTK_OBJECT(clist), "key_press_event",
			   CTK_SIGNAL_FUNC(&ctk_clist_key_press),       NULL);
	ctk_signal_connect(CTK_OBJECT(clist), "ctk_drag",
			   CTK_SIGNAL_FUNC(&ctk_clist_drag),            NULL);
}

void ctk_clist_min_size(CtkWidget* widget)
{
	/* We simulate column spacing with the extra col of space */
	ctk_table_min_size(widget);
}

void ctk_clist_real_size(CtkWidget* widget)
{
	CtkWidget*     sw;
	GNode*         scan;
	gboolean       moved;
	CtkTableChild* tchild;
	CtkWidget*     child;
	
	ctk_table_real_size(widget);
	
	/* Now, for the hack: If we're in a scrolled window, reposition widgets
	 * so that the title is at the top.
	 */
	if (CTK_CLIST(widget)->column_show && widget->node->parent &&
		CTK_CHECK_TYPE(widget->node->parent->data, CtkTypeScrolledWindow))
	{
		sw = CTK_WIDGET(widget->node->parent->data);
		
		moved = FALSE;
		for (scan = widget->node->children; scan; scan = scan->next)
		{
			tchild = CTK_TABLE_CHILD(scan->data);
			if (tchild->top_attach == 0)
			{
				/* This is a title, move it to the top */
				child = CTK_WIDGET(tchild);
				child->row = sw->row;
				moved = TRUE;
			}
		}
		
		if (moved)
		{
			for (scan = widget->node->children; scan; scan = scan->next)
			{
				tchild = CTK_TABLE_CHILD(scan->data);
				child = CTK_WIDGET(scan->data);
				if (tchild->top_attach != 0 &&
					child->row <= sw->row && 
					child->row+child->height > sw->row)
				{
					/* Was at the top - move it */
					child->row -= child->height;
				}
			}
		}
	}
}
