/***************************************************************************
 $RCSfile: readertest.c,v $
                             -------------------
    cvs         : $Id: readertest.c,v 1.11 2003/05/08 11:01:23 aquamaniac Exp $
    begin       : Thu Dec 12 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : martin@libchipcard.de

 ****************************************************************************
 * This program is free software; you can redistribute it and/or modify     *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation; either version 2 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA *
 ****************************************************************************/



#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

/* Internationalization */
#ifdef HAVE_GETTEXT_ENVIRONMENT
# include <libintl.h>
# include <locale.h>
# define I18N(m) gettext(m)
#else
# define I18N(m) m
#endif
#define I18NT(m) m


#include <chipcard.h>
#include <reader/ctcore.h>
#include <reader/ctcore_public.h>
#include <chameleon/chameleon.h>
#include <chameleon/debug.h>
#include <chameleon/conf.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>


#define k_PRG "readertest"
#define k_PRG_VERSION_INFO \
  "readertest v0.3.1 (part of libchipcard v"k_CHIPCARD_VERSION_STRING")\n"\
  "(c) 2003 Martin Preuss<martin@libchipcard.de>\n"\
  "This program is free software licensed under GPL.\n"\
  "See COPYING for details.\n"


#define RT_RESULT_OK               0
#define RT_RESULT_BAD_PARAMS       1
#define RT_RESULT_INTERNAL         2
#define RT_RESULT_NO_READER        3
#define RT_RESULT_BAD_DESCRIPTIONS 4


typedef struct _S_PARAM FREEPARAM;
typedef struct _S_ARGS ARGUMENTS;

struct _S_PARAM {
  FREEPARAM  *next;
  const char *param;
};



FREEPARAM *FreeParam_new(const char *s) {
  FREEPARAM *fr;

  fr=(FREEPARAM*)malloc(sizeof(FREEPARAM));
  assert(fr);
  memset(fr, 0, sizeof(FREEPARAM));
  fr->param=s;
  return fr;
}


void FreeParam_free(FREEPARAM *fr) {
  if (fr)
    free(fr);
}


struct _S_ARGS {
  FREEPARAM *params;
  int verbous;
  const char *driver;
  const char *drivertype;
  const char *readertype;
  const char *port;
  const char *readername;
};


ARGUMENTS *Arguments_new() {
  ARGUMENTS *ar;

  ar=(ARGUMENTS*)malloc(sizeof(ARGUMENTS));
  assert(ar);
  memset(ar, 0, sizeof(ARGUMENTS));
  ar->verbous=0;
  ar->port="COM1";
  ar->drivertype="ctapi";
  ar->readertype="towitoko";
  ar->readername="TestReader";
  return ar;
}


void Arguments_free(ARGUMENTS *ar) {
  if (ar) {
    FREEPARAM *fr;
    FREEPARAM *next;

    fr=ar->params;
    while(fr) {
      next=fr->next;
      FreeParam_free(fr);
      fr=next;
    } /* while */
  }
}


void usage(const char *name) {
  fprintf(stdout,
	  I18N("ReaderTest - A tool to check a reader configuration\n"
	       "                 Part of LibChipCard "k_CHIPCARD_VERSION_STRING"\n"
	       "(c) 2003 Martin Preuss<martin@libchipcard.de>\n"
	       "This library is free software; you can redistribute it and/or\n"
	       "modify it under the terms of the GNU Lesser General Public\n"
	       "License as published by the Free Software Foundation; either\n"
	       "version 2.1 of the License, or (at your option) any later version.\n"
	       "\n"
	       "Usage:\n"
	       k_PRG" [OPTIONS] \n"
	       "\nOptions:\n"
	       " -h            - show this help\n"
	       " -V            - show version information\n"
	       " -v            - be more verbous\n"
	       " -p PORT       - Where is the reader connected ? \n"
	       " -r READERTYPE - What's the type of the reader ? \n"
	       " -d DRIVERTYPE - What's the type of the driver ? \n"
	       " -D DRIVER     - Path and name of the driver to load \n"
	       " -n NAME       - name of the reader (needed for PC/SC)\n")
	 );
}


void Arguments_AddParam(ARGUMENTS *ar, const char *pr) {
  FREEPARAM *curr;
  FREEPARAM *nfp;

  DBG_ENTER;
  assert(ar);
  assert(pr);

  nfp=FreeParam_new(pr);

  curr=ar->params;
  if (!curr) {
    ar->params=nfp;
  }
  else {
    /* find last */
    while(curr->next) {
      curr=curr->next;
    } /* while */
    curr->next=nfp;
  }
  DBG_LEAVE;
}




int checkArgs(ARGUMENTS *args, int argc, char **argv) {
  int i;

  i=1;
  while (i<argc){
    if (strcmp(argv[i],"-h")==0 || strcmp(argv[i],"--help")==0) {
      usage(argv[0]);
      return -1;
    }
    else if (strcmp(argv[i],"-V")==0 || strcmp(argv[i],"--version")==0) {
      fprintf(stdout,k_PRG_VERSION_INFO);
      return -1;
    }
    else if (strcmp(argv[i],"-v")==0) {
      args->verbous=1;
    }
    else if (strcmp(argv[i],"-p")==0) {
      i++;
      if (i>=argc)
	return 1;
      args->port=argv[i];
    }
    else if (strcmp(argv[i],"-r")==0) {
      i++;
      if (i>=argc)
	return 1;
      args->readertype=argv[i];
    }
    else if (strcmp(argv[i],"-d")==0) {
      i++;
      if (i>=argc)
	return 1;
      args->drivertype=argv[i];
    }
    else if (strcmp(argv[i],"-D")==0) {
      i++;
      if (i>=argc)
	return 1;
      args->driver=argv[i];
    }
    else if (strcmp(argv[i],"-n")==0) {
      i++;
      if (i>=argc)
	return 1;
      args->readername=argv[i];
    }
    else
      // otherwise add param
      Arguments_AddParam(args, argv[i]);
    i++;
  } // while
  // that's it
  return 0;
}


int checkItNow(ARGUMENTS *args) {
  CTCORETABLE *ct;
  int cid;
  int tid;
  CTREADERDESCRIPTION *rd;
  CTREADERDESCRIPTION *rd2;
  ERRORCODE err;
  int status;
  char atrbuffer[256];
  int atrlen;
  CONFIGGROUP *pseudoconf;
  CONFIGGROUP *drivers;

  drivers=Config_new();
  if (CTCore_ReadDriverDescriptions(CHIPCARD_DRIVERS, drivers)) {
    Config_free(drivers);
    DBG_ERROR("Error loading driver descriptions");
    return RT_RESULT_BAD_DESCRIPTIONS;
  }

  printf(I18N("Initializing core\n"));
  ct=CTCore_new();
  /* init core */
  err=CTCore_Init(ct, drivers);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTCore_free(ct);
    Config_free(drivers);
    return RT_RESULT_INTERNAL;
  }

  if (!args->driver) {
    DBG_ERROR("No driver given.");
    CTCore_Fini(ct);
    CTCore_free(ct);
    return RT_RESULT_BAD_PARAMS;
  }

  printf(I18N("Adding reader\n"));
  /* setup a reader description */

  pseudoconf = Config_new();
  Config_SetValue(pseudoconf, CONFIGMODE_VARIABLE |
                  CONFIGMODE_NAMECREATE_VARIABLE |
                  CONFIGMODE_OVERWRITE_VARS, /* more modes */
                  "port", args->port);
  Config_SetValue(pseudoconf, CONFIGMODE_VARIABLE |
                  CONFIGMODE_NAMECREATE_VARIABLE |
                  CONFIGMODE_OVERWRITE_VARS, /* more modes */
		  "readertype", args->readertype);
  Config_SetValue(pseudoconf, CONFIGMODE_VARIABLE |
                  CONFIGMODE_NAMECREATE_VARIABLE |
                  CONFIGMODE_OVERWRITE_VARS, /* more modes */
                  "drivertype", args->drivertype);
  Config_SetValue(pseudoconf, CONFIGMODE_VARIABLE |
                  CONFIGMODE_NAMECREATE_VARIABLE |
                  CONFIGMODE_OVERWRITE_VARS, /* more modes */
                  "driver", args->driver);
  Config_SetValue(pseudoconf, CONFIGMODE_VARIABLE |
		  CONFIGMODE_NAMECREATE_VARIABLE |
		  CONFIGMODE_OVERWRITE_VARS, /* more modes */
		  "name", args->readername);
  rd=CTCore_ReadReaderDescr(pseudoconf);
  Config_free(pseudoconf);

  if (! rd) {
    DBG_ERROR("Invalid driver configuation parameters passed");
    CTCore_Fini(ct);
    CTCore_free(ct);
    return RT_RESULT_BAD_PARAMS;
  }

  err=CTCore_AddReader(ct, rd);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    CTCore_ReaderDescr_free(rd);
    CTCore_Fini(ct);
    CTCore_free(ct);
    return RT_RESULT_BAD_PARAMS;
  }

  /* register client */
  printf(I18N("Registering client, allocating reader\n"));
  cid=CTCore_RegisterClient(ct);
  tid=CTCore_AllocTerminal(ct, cid,rd->id, &rd2);
  if (tid==-1) {
    fprintf(stderr,I18N("No reader at this port.\n"));
    CTCore_Fini(ct);
    CTCore_free(ct);
    return RT_RESULT_NO_READER;
  }
  printf(I18N("Reader seems to be available ;-)\n"));

  /* check reader status */
  printf(I18N("Checking reader status\n"));
  atrlen=sizeof(atrbuffer);
  err=CTCore_CheckReaderStatus(ct,
			       cid,
			       tid,
			       &status,
			       atrbuffer,
			       &atrlen);
  if (!Error_IsOk(err)) {
    DBG_ERROR("Internal error allocating reader");
    CTCore_Fini(ct);
    CTCore_free(ct);
    return RT_RESULT_INTERNAL;
  }
  if (status & CHIPCARD_STATUS_INSERTED) {
    printf(I18N("- Card is inserted\n"));
    if (status & CHIPCARD_STATUS_PROCESSOR)
      printf(I18N("- Card might be a processor card\n"));
    else
      printf(I18N("- Card might be a memory card\n"));
  }
  else
    printf(I18N("- No card\n"));

  /* release terminal */
  printf(I18N("Releasing reader, unregistering client\n"));
  CTCore_ReleaseTerminal(ct,cid,tid);

  /* unregister client, clean up */
  CTCore_UnregisterClient(ct, cid);
  printf(I18N("Deinititializing Core\n"));
  CTCore_Fini(ct);
  CTCore_free(ct);
  printf(I18N("\nCongratulations, the reader is available ;-)\n"));
  return RT_RESULT_OK;
}




int main(int argc, char **argv) {
  ERRORCODE err;
  ARGUMENTS *args;
  int rv;

#ifdef HAVE_GETTEXT_ENVIRONMENT
  setlocale(LC_ALL,"");
  if (bindtextdomain("readertest", I18N_PATH)==0) {
    fprintf(stderr," Error bindtextdomain()\n");
  }
  if (textdomain("readertest")==0) {
    fprintf(stderr," Error textdomain()\n");
  }
#endif

  args=Arguments_new();
  rv=checkArgs(args,argc,argv);
  if (rv==-1) {
    Arguments_free(args);
    return 0;
  }
  else if (rv) {
    Arguments_free(args);
    return rv;
  }

  if (Logger_Open("rt",
		  0,
		  LoggerTypeConsole,
		  LoggerFacilityUser)) {
    fprintf(stderr,I18N("Could not setup logging, aborting.\n"));
    return 2;
  }
  Logger_SetLevel(LoggerLevelInfo);

  /* init all modules */
  err=Chameleon_Init();
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    Arguments_free(args);
    return 2;
  }
  err=CTCore_ModuleInit();
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    Arguments_free(args);
    return 2;
  }

  /* perform check */
  rv=checkItNow(args);

  /* deinitialize modules */
  err=CTCore_ModuleFini();
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
  }
  err=Chameleon_Fini();
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
  }

  Arguments_free(args);

  return rv;
}

