/***************************************************************************
 $RCSfile: command.c,v $
                             -------------------
    cvs         : $Id: command.c,v 1.14 2003/05/07 22:27:22 aquamaniac Exp $
    begin       : Tue Dec 03 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif

#include "command.h"
#include "chameleon/conf.h"
#include "chameleon/debug.h"

#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <ctype.h>
#include <stdarg.h>

#define CTCOMMAND__CHECK_TYPE_ASCII    1
#define CTCOMMAND__CHECK_TYPE_ALPHA    2
#define CTCOMMAND__CHECK_TYPE_NUM      3
#define CTCOMMAND__CHECK_TYPE_ALPHANUM 4



int CTCommand__GetHexString(const char *param,
			    unsigned char *buffer,
			    int *len) {
  int i;
  unsigned char c;
  unsigned char n;
  unsigned char t;

  /* skip blanks */
  while (*param && *param<33)
    param++;
  i=0;
  c=0;
  while(*param && i<(*len*2)) {
    /* skip spaces */
    while(*param && isspace(*param))
      param++;
    if (*param==0)
      break;

    t=toupper(*param);
    if (!isspace(t)) {
      if (!((t>='0' && t<='9') ||
	    (t>='A' && t<='F'))) {
	DBG_ERROR("Only hex digits allowed");
	return CTCOMMAND_RESULT_BAD_ARGUMENT;
      }
      n=t-'0';
      if (n>9)
	n-=7;
      c=c<<4;
      c|=(n&0xf);
    } /* if !isspace() */
    if ((i && (i & 1)) || isspace(t)) {
      *buffer=c;
      buffer++;
      c=0;
    }
    i++;
    param++;
  } /* while */
  if (i>=(*len*2) && *param) {
    DBG_ERROR("Parameter too long (limit is %d)",*len);
    return CTCOMMAND_RESULT_BAD_ARGUMENT;
  }
  if (i&1) {
    DBG_ERROR("Odd number of digits");
    return CTCOMMAND_RESULT_BAD_ARGUMENT;
  }
  *len=i/2;
  return CTCOMMAND_RESULT_OK;
}


int CTCommand__GetString(const unsigned char *param,
			 unsigned char *buffer,
			 int *len,
			 unsigned int typ) {
  int i;
  unsigned char c;
  int rv;

  i=0;
  c=0;
  while(*param && i<*len) {
    c=*param;
    switch(typ) {
    case CTCOMMAND__CHECK_TYPE_ASCII:
      rv=1;
      break;
    case CTCOMMAND__CHECK_TYPE_ALPHA:
      rv=isalpha(c);
      break;
    case CTCOMMAND__CHECK_TYPE_NUM:
      rv=isdigit(c);
      break;
    case CTCOMMAND__CHECK_TYPE_ALPHANUM:
      rv=isalnum(c);
      break;
    default:
      DBG_ERROR("Bad check mode (%d)",typ);
      return CTCOMMAND_RESULT_INTERNAL;
    }
    if (!rv) {
      DBG_ERROR("Bad character");
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }
    *buffer=c;
    buffer++;
    i++;
    param++;
  } /* while */
  if (i>=*len && *param) {
    DBG_ERROR("Param too long (limit is %d)",*len);
    return CTCOMMAND_RESULT_BAD_ARGUMENT;
  }
  *len=i;
  return CTCOMMAND_RESULT_OK;
}



int CTCommand__MakeFPIN2(const char *src,
			 char *buffer) {
  int pinlen;
  int i;
  int j;
  int k;

  pinlen=strlen(src);

  /* preset */
  for (i=0; i<8; i++) buffer[i]=0xff;
  /* set C to "2", set length */
  buffer[0]=pinlen + 0x20;

  /* now transform pin */
  for (i=0; i<pinlen; i++) {
    k=i/2+1;
    j=src[i]-'0';
    if (j>9) {
      DBG_ERROR("Only digits allowed");
      return CTCOMMAND_RESULT_BAD_CHAR;
    }

    if (i & 1){
      // right digit
      buffer[k]&=0xf0; // erase right digit
      buffer[k]+=j;    // set right digit
    }
    else {
      // left digit
      buffer[k]&=0x0f; // erase left digit
      buffer[k]+=j<<4; // set left digit
    }
  } // for

  return 0;
}


int CTCommand__CheckArg(CONFIGGROUP *grp,
			int argc,
			char **argv,
			char *buffer,
			int *bufferlen) {
  const char *type;
  int argnum;
  const char *arg;
  int minv;
  int maxv;
  int ttype;

  assert(grp);
  assert(buffer);
  assert(*bufferlen>3);
  argnum=Config_GetIntValue(grp,
			    "argument",
			    -1,
			    0);
  if (argnum==-1 || argnum>=argc) {
    DBG_ERROR("Argument index (%d) out of range", argnum);
    return CTCOMMAND_RESULT_BAD_ARGNUM;
  }

  type=Config_GetValue(grp,
		       "type",
		       "",
		       0);
  minv=Config_GetIntValue(grp,
			  "min",
			  -1,
			  0);
  maxv=Config_GetIntValue(grp,
			  "max",
			  -1,
			  0);
  ttype=-1;
  arg=argv[argnum];
  assert(arg);

  if (strcmp(type,"char")==0) {
    /* get a single character from the param */
    int val;

    if (sscanf(arg,"%i",&val)!=1) {
      DBG_ERROR("Char value expected");
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }

    if (*bufferlen<1) {
      DBG_ERROR("Buffer too small");
      return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
    }
    if ((minv!=-1 && val<minv) ||
	(maxv!=-1 && val>maxv) ||
	((unsigned int)val)>255) {
      DBG_ERROR("Param \"%s\": Argument out of boundaries", grp->name);
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }
    buffer[0]=val&0xff;
    *bufferlen=1;
    return CTCOMMAND_RESULT_OK;
  }

  else if (strcmp(type,"uchar")==0) {
    /* get a single unsigned character from the param */
    unsigned int val;

    if (sscanf(arg,"%i",&val)!=1) {
      DBG_ERROR("Char value expected");
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }

    if (*bufferlen<1) {
      DBG_ERROR("Buffer too small");
      return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
    }
    if ((minv!=-1 && val<(unsigned int)minv) ||
	(maxv!=-1 && val>(unsigned int)maxv) ||
	val>255) {
      DBG_ERROR("Param \"%s\": Argument out of boundaries", grp->name);
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }
    buffer[0]=val&0xff;
    *bufferlen=1;
    return CTCOMMAND_RESULT_OK;
  }

  else if (strcmp(type,"word")==0) {
    /* get a single word from the param and store it in the buffer using
     * BIG_ENDIAN (68000er notation: HiByte first) */
    int val;

    if (*bufferlen<2) {
      DBG_ERROR("Buffer too small");
      return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
    }

    if (sscanf(arg,"%i",&val)!=1) {
      DBG_ERROR("Integer value expected");
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }
    if ((minv!=-1 && val<minv) ||
	(maxv!=-1 && val>maxv) ||
	((unsigned int)val)>65535) {
      DBG_ERROR("Argument outside limits");
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }
    buffer[0]=(unsigned char)((val>>8)&0xff);
    buffer[1]=(unsigned char)((val)&0xff);
    *bufferlen=2;
    return CTCOMMAND_RESULT_OK;
  }

  else if (strcmp(type,"uword")==0) {
    /* get a single unsigned word from the param and store it in the buffer
     * using BIG_ENDIAN (68000er notation: HiByte first) */
    unsigned int val;

    if (*bufferlen<2) {
      DBG_ERROR("Buffer too small");
      return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
    }

    if (sscanf(arg,"%i",&val)!=1) {
      DBG_ERROR("Integer value expected");
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }
    if ((minv!=-1 && val<(unsigned int)minv) ||
	(maxv!=-1 && val>(unsigned int)maxv) ||
	(val>65535)) {
      DBG_ERROR("Argument outside limits");
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }
    buffer[0]=(unsigned char)((val>>8)&0xff);
    buffer[1]=(unsigned char)((val)&0xff);
    *bufferlen=2;
    return CTCOMMAND_RESULT_OK;
  }

  else if (strcmp(type,"bin")==0) {
    int rv;
    int ll;
    int filler;
    int j;

    filler=Config_GetIntValue(grp,
			      "fillwith",
			      -1,
			      0);
    if (maxv!=-1) {
      if (maxv>*bufferlen) {
	DBG_ERROR("Buffer smaller than \"max\"");
	return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
      }
      ll=maxv;
    }
    else
      ll=*bufferlen;

    if (filler!=-1) {
      if (maxv==-1) {
	DBG_ERROR("Filler given, but no \"max\"");
	return CTCOMMAND_RESULT_BAD_CONFIG;
      }
      /* fill buffer */
      for (j=0; j<maxv; j++)
	buffer[j]=filler;
    }

    rv=CTCommand__GetHexString(arg, buffer, &ll);
    if (rv!=CTCOMMAND_RESULT_OK)
      return rv;
    if (minv!=-1 && ll<minv) {
      DBG_ERROR("Argument too short (limit is %d, size is %d)", minv, ll);
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }
    if (filler!=-1)
      *bufferlen=maxv;
    else
      *bufferlen=ll;
    return CTCOMMAND_RESULT_OK;
  }

  else if (strcmp(type,"fpin2")==0) {
    int rv;
    int pinlen;

    if (*bufferlen<8) {
      DBG_ERROR("Buffer too small");
      return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
    }

    pinlen=strlen(arg);
    if (maxv!=-1) {
      if (pinlen>maxv) {
	DBG_ERROR("Argument too long (limit is %d, size is %d)", maxv, pinlen);
	return CTCOMMAND_RESULT_BAD_ARGUMENT;
      }
    }
    if (minv!=-1) {
      if (pinlen<minv) {
	DBG_ERROR("Argument too short (limit is %d, size is %d)", minv, pinlen);
	return CTCOMMAND_RESULT_BAD_ARGUMENT;
      }
    }

    /* transform data to fpin2 */
    rv=CTCommand__MakeFPIN2(arg, buffer);
    if (rv!=CTCOMMAND_RESULT_OK)
      return rv;
    *bufferlen=8;
    return CTCOMMAND_RESULT_OK;
  }

  else if (strcmp(type,"ascii")==0)
    ttype=CTCOMMAND__CHECK_TYPE_ASCII;
  else if (strcmp(type,"alpha")==0)
    ttype=CTCOMMAND__CHECK_TYPE_ALPHA;
  else if (strcmp(type,"num")==0)
    ttype=CTCOMMAND__CHECK_TYPE_NUM;
  else if (strcmp(type,"alphanum")==0)
    ttype=CTCOMMAND__CHECK_TYPE_ALPHANUM;
  else {
    DBG_ERROR("Bad type");
    return CTCOMMAND_RESULT_BAD_TYPE;
  }

  /* really get the data */
  if (ttype!=-1){
    int rv;
    int ll;
    int filler;
    int j;

    filler=Config_GetIntValue(grp,
			      "fillwith",
			      -1,
			      0);
    if (maxv!=-1) {
      if (maxv>*bufferlen) {
	DBG_ERROR("Buffer smaller than \"max\"");
	return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
      }
      ll=maxv;
    }
    else
      ll=*bufferlen;

    if (filler!=-1) {
      if (maxv==-1) {
	DBG_ERROR("Filler given, but no \"max\"");
	return CTCOMMAND_RESULT_BAD_CONFIG;
      }
      /* fill buffer */
      for (j=0; j<maxv; j++)
	buffer[j]=filler;
    }

    rv=CTCommand__GetString(arg,
			    buffer,
			    &ll,
			    ttype);
    if (rv!=CTCOMMAND_RESULT_OK)
      return rv;
    if (minv!=-1 && ll<minv) {
      DBG_ERROR("Argument too short (limit is %d)", minv);
      return CTCOMMAND_RESULT_BAD_ARGUMENT;
    }
    if (filler!=-1)
      *bufferlen=maxv;
    else
      *bufferlen=ll;
    return CTCOMMAND_RESULT_OK;
  }

  DBG_ERROR("Huh ? We should never reach this point :-O");
  return CTCOMMAND_RESULT_INTERNAL;
}


int CTCommand__GetParam(CONFIGGROUP *cmd,
			CONFIGGROUP *apdu,
			const char *name,
			int argc,
			char **argv,
			int *arg) {
  int rv;
  CONFIGGROUP *param;
  const char *p;
  char vbuffer[300];
  int vbufferlen;

  p=Config_GetValue(apdu, name, 0, 0);
  if (!p) {
    DBG_ERROR("\"%s\" required in ADPU \"%s\"",name, apdu->name);
    return CTCOMMAND_RESULT_BAD_CONFIG;
  }
  while(*p && isspace(*p))
    p++;
  if (*p==0) {
    DBG_ERROR("Argument required");
    return CTCOMMAND_RESULT_BAD_CONFIG;
  }
  if (*p=='$') {
    p++;
    param=Config_GetGroup(cmd,
			  p,
			  CONFIGMODE_PATHMUSTEXIST |
			  CONFIGMODE_NAMEMUSTEXIST);
    if (!param) {
      DBG_ERROR("Param definition for \"%s\" not found",p);
      return CTCOMMAND_RESULT_BAD_CONFIG;
    }
    vbufferlen=sizeof(vbuffer);
    rv=CTCommand__CheckArg(param,
			   argc,
			   argv,
			   vbuffer,
			   &vbufferlen);
    if (rv!=CTCOMMAND_RESULT_OK) {
      DBG_ERROR("Error in param \"%s\"",p);
      return rv;
    }
    if (vbufferlen<1 || vbufferlen>2) {
      DBG_ERROR("Bad parameter size in param \"%s\"",p);
      return CTCOMMAND_RESULT_BAD_CONFIG;
    }
    /* store var */
    *arg=(unsigned char)(vbuffer[0]);
    if (vbufferlen==2) {
      *arg=*arg<<8;
      *arg+=(unsigned char)(vbuffer[1]);
    }
  }
  else {
    if (sscanf(p,"%i",arg)!=1) {
      DBG_ERROR("Bad value");
      return CTCOMMAND_RESULT_BAD_CONFIG;
    }
  }

  return CTCOMMAND_RESULT_OK;
}


int CTCommand__TranslateAPDU(CONFIGGROUP *cmd,
			     CONFIGGROUP *apdu,
			     int argc,
			     char **argv,
			     unsigned char *buffer,
			     int *bufferlen) {
  int rv;
  int val;
  int i;
  const char *p;
  int vbufferlen;
  char databuffer[300];
  int databufferpos;
  int idx;
  int firstEmpty;

  i=0;
  /* CLA */
  rv=CTCommand__GetParam(cmd,
			 apdu,
			 "cla",
			 argc,
			 argv,
			 &val);
  if (rv!=CTCOMMAND_RESULT_OK) {
    return rv;
  }
  if (*bufferlen<i) {
    DBG_ERROR("Command buffer full (limit is %d)",*bufferlen);
    return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
  }
  *buffer=(unsigned char)(val & 0xff);
  buffer++;
  i++;

  /* INS */
  rv=CTCommand__GetParam(cmd,
			 apdu,
			 "ins",
			 argc,
			 argv,
			 &val);
  if (rv!=CTCOMMAND_RESULT_OK) {
    return rv;
  }
  if (*bufferlen<i) {
    DBG_ERROR("Command buffer full (limit is %d)",*bufferlen);
    return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
  }
  *buffer=(unsigned char)(val & 0xff);
  buffer++;
  i++;

  /* P1 and P2 */
  if (Config_GetValue(apdu, "p12", 0, 0)==0) {
    /* get p1 and p2 in two steps */
    /* P1 */
    rv=CTCommand__GetParam(cmd,
			   apdu,
			   "p1",
			   argc,
			   argv,
			   &val);
    if (rv!=CTCOMMAND_RESULT_OK) {
      return rv;
    }
    if (*bufferlen<i) {
      DBG_ERROR("Command buffer full (limit is %d)",*bufferlen);
      return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
    }
    *buffer=(unsigned char)(val & 0xff);
    buffer++;
    i++;

    /* P2 */
    rv=CTCommand__GetParam(cmd,
			   apdu,
			   "p2",
			   argc,
			   argv,
			   &val);
    if (rv!=CTCOMMAND_RESULT_OK) {
      return rv;
    }
    if (*bufferlen<i) {
      DBG_ERROR("Command buffer full (limit is %d)",*bufferlen);
      return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
    }
    *buffer=(unsigned char)(val & 0xff);
    buffer++;
    i++;
  }
  else {
    /* get p1 and p2 in a single step */
    rv=CTCommand__GetParam(cmd,
			   apdu,
			   "p12",
			   argc,
			   argv,
			   &val);
    if (rv!=CTCOMMAND_RESULT_OK) {
      return rv;
    }
    if (*bufferlen<i+1) {
      DBG_ERROR("Command buffer full (limit is %d)",*bufferlen);
      return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
    }
    *buffer=(unsigned char)((val>>8) & 0xff);
    buffer++;
    i++;
    *buffer=(unsigned char)( val & 0xff);
    buffer++;
    i++;
  }

  /* data */
  databufferpos=0;
  idx=0;
  firstEmpty=0;
  while ((p=Config_GetValue(apdu, "data", 0, idx))) {
    while(*p && isspace(*p))
      p++;
    if (*p) {
      if (firstEmpty) {
	DBG_ERROR("Error in DATA (first param is empty while next are not)");
	return CTCOMMAND_RESULT_BAD_CONFIG;
      }
      if (*p=='$') {
	CONFIGGROUP *param;

	p++;
	param=Config_GetGroup(cmd,
			      p,
			      CONFIGMODE_PATHMUSTEXIST |
			      CONFIGMODE_NAMEMUSTEXIST);
	if (!param) {
	  DBG_ERROR("Param definition for \"%s\" not found",p);
	  return CTCOMMAND_RESULT_BAD_CONFIG;
	}
	vbufferlen=sizeof(databuffer)-databufferpos;
	rv=CTCommand__CheckArg(param,
			       argc,
			       argv,
			       &(databuffer[databufferpos]),
			       &vbufferlen);
      } /* if var */
      else {
	if (rv!=CTCOMMAND_RESULT_OK) {
	  DBG_ERROR("Error in param \"%s\"",p);
	  return rv;
	}
	vbufferlen=sizeof(databuffer)-databufferpos;
	if (vbufferlen<1) {
	  DBG_ERROR("Command buffer full (limit is %d)",*bufferlen);
	  return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
	}
	rv=CTCommand__GetHexString(p,
				   &(databuffer[databufferpos]),
				   &vbufferlen);
	if (rv!=CTCOMMAND_RESULT_OK) {
          DBG_ERROR("Error %d",rv);
	  return rv;
	}
      }

      /* check data size */
      if (vbufferlen+databufferpos>sizeof(databuffer)) {
	DBG_ERROR("Command buffer full (limit is %d)",*bufferlen);
	return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
      }
      databufferpos+=vbufferlen;
    } /* if data not empty */
    else {
      /* data is empty, so add null bytes (size indicator is 0) */
      if (idx==0) {
	firstEmpty=1;
      }
      else {
	DBG_ERROR("Empty argument found in DATA");
	return CTCOMMAND_RESULT_BAD_CONFIG;
      }
    }
    idx++;
  } /* while */

  /* store data */
  if (databufferpos || firstEmpty) {
    if (*bufferlen<i+databufferpos+1) {
      DBG_ERROR("Command buffer full (limit is %d)",*bufferlen);
      return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
    }
    *buffer=(unsigned char)databufferpos;
    buffer++;
    i++;
    if (databufferpos) {
      memmove(buffer, databuffer, databufferpos);
      buffer+=databufferpos;
      i+=databufferpos;
    }
  }

  /* Lr */
  rv=CTCommand__GetParam(cmd,
			 apdu,
			 "lr",
			 argc,
			 argv,
			 &val);
  if (rv!=CTCOMMAND_RESULT_OK) {
    return rv;
  }
  if (val) {
    if (*bufferlen<i) {
      DBG_ERROR("Command buffer full (limit is %d)",*bufferlen);
      return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
    }
    if (val>255)
      *buffer=0;
    else
      *buffer=(unsigned char)(val & 0xff);
    buffer++;
    i++;
  }

  *bufferlen=i;
  return CTCOMMAND_RESULT_OK;
}


int CTCommand_MakeAPDU(CONFIGGROUP *root,
		       const char *command,
		       int argc,
		       char **argv,
		       unsigned char *buffer,
		       int *bufferlen) {
  CONFIGGROUP *grp;
  CONFIGGROUP *apdu;
  int arguments;
  const char *apduname;
  int rv;

  assert(root);
  assert(argv);
  assert(buffer);
  assert(bufferlen);
  assert(*bufferlen>3);

  grp=Config_GetGroup(root,
		      command,
		      CONFIGMODE_PATHMUSTEXIST |
		      CONFIGMODE_NAMEMUSTEXIST);
  if (!grp) {
    DBG_ERROR("Command \"%s\" not found",command);
    return CTCOMMAND_RESULT_COMMMAND_NOT_FOUND;
  }
  arguments=Config_GetIntValue(grp,
			       "arguments",
			       -1,
			       0);
  if (arguments==-1) {
    DBG_ERROR("\"arguments\" not found in command \"%s\"",command);
    return CTCOMMAND_RESULT_BAD_CONFIG;
  }
  if (argc!=arguments) {
    DBG_ERROR("Command \"%s\" needs %d arguments (we have %d)",
	      command,
	      arguments, argc);
    return CTCOMMAND_RESULT_COMMMAND_NOT_FOUND;
  }
  apduname=Config_GetValue(grp,
			   "apdu",
			   0,
			   0);
  if (!apduname) {
    DBG_ERROR("No APDU for command \"%s\"",command);
    return CTCOMMAND_RESULT_BAD_CONFIG;
  }
  apdu=Config_GetGroup(grp,
		       apduname,
		       CONFIGMODE_PATHMUSTEXIST |
		       CONFIGMODE_NAMEMUSTEXIST);
  if (!apdu) {
    DBG_ERROR("APDU \"%s\" for command \"%s\" not found",
	      apduname,command);
    return CTCOMMAND_RESULT_BAD_CONFIG;
  }

  /* now finally translate the APDU */
  rv=CTCommand__TranslateAPDU(grp,
			      apdu,
			      argc,
			      argv,
			      buffer,
			      bufferlen);
  DBG_LEAVE;
  return rv;
}

int CTCommand_Locate(CONFIGGROUP *root,
		     const char *readertype,
		     const char *cardtype,
		     const char *command,
		     char *buffer,
		     int bufferlen) {
  CONFIGGROUP *grp;

  assert(root);
  assert(command);
  assert(*command);
  assert(buffer);
  assert(bufferlen);

  if (cardtype) {
    DBG_INFO("Locating path for card \"%s/%s/%s\"",
	     readertype,cardtype,command);
  }

  /* check reader/card/command */
  if (readertype && cardtype) {
    if (*readertype && *cardtype) {
      if ((strlen(readertype)+strlen(cardtype)+
	   strlen(command)+3)>=bufferlen) {
	DBG_ERROR("Buffer too small (limit is %d)", bufferlen);
	return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
      }
      buffer[0]=0;
      strcpy(buffer, readertype);
      strcat(buffer, "/");
      strcat(buffer, cardtype);
      strcat(buffer, "/");
      strcat(buffer, command);
      DBG_DEBUG("Trying command \"%s\"",buffer);
      grp=Config_GetGroup(root,
			  buffer,
			  CONFIGMODE_PATHMUSTEXIST |
			  CONFIGMODE_NAMEMUSTEXIST);
      if (grp) {
	DBG_VERBOUS("Found command \"%s\"",buffer);
	return CTCOMMAND_RESULT_OK;
      }
    }
  }

  /* check card/command */
  if (cardtype) {
    if (*cardtype) {
      if ((strlen(cardtype)+
	   strlen(command)+2)>=bufferlen) {
	DBG_ERROR("Buffer too small (limit is %d)", bufferlen);
	return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
      }
      buffer[0]=0;
      strcpy(buffer, cardtype);
      strcat(buffer, "/");
      strcat(buffer, command);
      DBG_DEBUG("Trying command \"%s\"",buffer);
      grp=Config_GetGroup(root,
			  buffer,
			  CONFIGMODE_PATHMUSTEXIST |
			  CONFIGMODE_NAMEMUSTEXIST);
      if (grp) {
	DBG_VERBOUS("Found command \"%s\"",buffer);
	return CTCOMMAND_RESULT_OK;
      }
    }
  }

  /* check reader/all/command */
  if (readertype) {
    if (*readertype) {
      if ((strlen(readertype)+strlen("all")+
	   strlen(command)+3)>=bufferlen) {
	DBG_ERROR("Buffer too small (limit is %d)", bufferlen);
	return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
      }
      buffer[0]=0;
      strcpy(buffer, readertype);
      strcat(buffer, "/all/");
      strcat(buffer, command);
      DBG_DEBUG("Trying command \"%s\"",buffer);
      grp=Config_GetGroup(root,
			  buffer,
			  CONFIGMODE_PATHMUSTEXIST |
			  CONFIGMODE_NAMEMUSTEXIST);
      if (grp) {
	DBG_VERBOUS("Found command \"%s\"",buffer);
	return CTCOMMAND_RESULT_OK;
      }
    }
  }

  /* check all/command */
  if ((strlen("all")+
       strlen(command)+3)>=bufferlen) {
    DBG_ERROR("Buffer too small (limit is %d)", bufferlen);
    return CTCOMMAND_RESULT_BUFFER_TOO_SMALL;
  }
  buffer[0]=0;
  strcpy(buffer, "all/");
  strcat(buffer, command);
  DBG_DEBUG("Trying command \"%s\"",buffer);
  grp=Config_GetGroup(root,
		      buffer,
		      CONFIGMODE_PATHMUSTEXIST |
		      CONFIGMODE_NAMEMUSTEXIST);
  if (grp) {
    DBG_DEBUG("Found command \"%s\"",buffer);
    return CTCOMMAND_RESULT_OK;
  }

  buffer[0]=0;
  DBG_DEBUG("Command \"%s\" not found",command);
  return CTCOMMAND_RESULT_COMMMAND_NOT_FOUND;
}


int CTCommand_Check(CONFIGGROUP *root,
		    const char *command) {
  CONFIGGROUP *grp;

  assert(root);
  assert(command);
  assert(*command);

  grp=Config_GetGroup(root,
		      command,
		      CONFIGMODE_PATHMUSTEXIST |
		      CONFIGMODE_NAMEMUSTEXIST);
  if (grp) {
    DBG_VERBOUS("Found command \"%s\"",command);
    return CTCOMMAND_RESULT_OK;
  }
  return CTCOMMAND_RESULT_COMMMAND_NOT_FOUND;
}


int CTCommand_Add(CONFIGGROUP *root,
		  CONFIGGROUP *cfg) {
  assert(root);
  assert(cfg);
  assert(cfg->groups);

  Config_ImportGroup(root, cfg);
  return CTCOMMAND_RESULT_OK;

  return CTCOMMAND_RESULT_OK;
}


int CTCommand_MakeAPDU2(CONFIGGROUP *root,
			const char *command,
			unsigned char *buffer,
			int *bufferlen,
			int argc, ...){
  int rv;
  va_list arguments;
  char **argv;
  int i;

  assert(root);
  assert(buffer);
  assert(bufferlen);
  assert(*bufferlen>3);

  va_start(arguments,argc);
  argv=(char**)malloc(sizeof(char*)*argc);
  assert(argv);
  for (i=0; i<argc; i++)
    argv[i]=va_arg(arguments, char*);
  rv=CTCommand_MakeAPDU(root,
			command,
			argc,
			argv,
			buffer,
			bufferlen);
  free(argv);
  va_end(arguments);
  return rv;
}






