/***************************************************************************
 $RCSfile: hbcicard.cpp,v $
                             -------------------
    cvs         : $Id: hbcicard.cpp,v 1.15 2003/05/07 20:14:59 aquamaniac Exp $
    begin       : Tue Aug 28 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif


#include <stdio.h> // DEBUG

#include "cterror.h"
#include "ctpointer.h"
#include "ctmisc.h"
#include "ctcommand.h"
#include "ctcard.h"
#include "ctprocessorcard.h"
#include "cttlv.h"
#include "hbcicard.h"
#include <engine/chameleon/error.h>
#include <engine/chameleon/debug.h>




/*___________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * HBCICard::cardData
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


HBCICard::CardData::CardData()
    :_cardtype(0)
    ,_industrialkey(0)
    ,_shortinstcode(0)
    ,_bestuntil_year(0)
    ,_bestuntil_month(0)
    ,_active_year(0)
    ,_active_month(0)
    ,_active_day(0)
    ,_countrycode(280)
{
}


HBCICard::CardData::CardData(const string &data, int cardtype)
:_cardtype(0)
,_industrialkey(0)
,_shortinstcode(0)
,_bestuntil_year(0)
,_bestuntil_month(0)
,_active_year(0)
,_active_month(0)
,_active_day(0)
,_countrycode(280)
{

  if (data.length()<22)
    throw CTError("HBCICard::CardData::CardData()",
		  k_CTERROR_INVALID,0,0,
		  "data too small");

  // check the rules according to HBCI
  if (cardtype==k_HBCICARD_TYPE_0) {
    if (data.length()>22) {
      DBG_VERBOUS("LIBCHIPCARD: TYPE0: data too small.");
      throw CTError("HBCICard::CardData::CardData()",
		    k_CTERROR_INVALID,0,0,
		    "type 0: data too long");
    }
    if (data.substr(17,3)=="EUR") {
      DBG_VERBOUS("LIBCHIPCARD: TYPE0: EUR not allowed.");
      throw CTError("HBCICard::CardData::CardData()",
		    k_CTERROR_INVALID,0,0,
		    "type 0: EUR not allowed");
    }
  }
  else if (cardtype==k_HBCICARD_TYPE_1) {
    DBG_VERBOUS("LIBCHIPCARD: Your card is type1.");
    if (data.length()<24) {
      DBG_VERBOUS("LIBCHIPCARD: TYPE1: data too small.");
      throw CTError("HBCICard::CardData::CardData()",
		    k_CTERROR_INVALID,0,0,
		    "type 1: data too small");
    }
    if ((data.substr(17,3)=="DEM") ||
	(data.at(23)==0)) {
      if (data.at(23)==0) {
	DBG_VERBOUS("LIBCHIPCARD: TYPE1: byte 23 is zero.");
      }
      else {
	DBG_VERBOUS("LIBCHIPCARD: TYPE1: DEM not allowed.");
      }
      throw CTError("HBCICard::CardData::CardData()",
		    k_CTERROR_INVALID,0,0,
		    "type 1: byte 23 is zero OR value==DEM");
    }
  }
  if (data.at(20)!=1){ // check value of currency (should be 1)
    DBG_VERBOUS("LIBCHIPCARD: byte 20 is not 1.");
    throw CTError("HBCICard::CardData::CardData()",
		  k_CTERROR_INVALID,0,0,
		  "byte 20 is not 1");
  }

  // copy data in
  _cardtype=cardtype;
  _industrialkey=CTMisc::bsd2int(data.at(0));
  _shortinstcode=(CTMisc::bsd2int(data.at(1))*10000)+
    (CTMisc::bsd2int(data.at(2))*100)+
    CTMisc::bsd2int(data.at(3));
  _cardnumber.erase();
  _cardnumber=CTMisc::bsd2string(data.substr(4,5));
  _bestuntil_year=CTMisc::bsd2int(data.at(10));
  _bestuntil_month=CTMisc::bsd2int(data.at(11));
  _active_year=CTMisc::bsd2int(data.at(12));
  _active_month=CTMisc::bsd2int(data.at(13));
  _active_day=CTMisc::bsd2int(data.at(14));
  _countrycode=(CTMisc::bsd2int(data.at(15))*100)+
    CTMisc::bsd2int(data.at(16));
  _currency=data.substr(17,3);
  CTMisc::removeBlanks(_currency);
}


HBCICard::CardData::~CardData(){
}






/*___________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * HBCICard::instituteData
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


HBCICard::instituteData::instituteData()
    :_service(0)
    ,_country(0)
{
}

HBCICard::instituteData::instituteData(const string &data)
:_service(0)
,_country(0)
{
  CTError err;

  err=fromString(data);
  //fprintf(stderr,"Error is: %s\n",err.errorString().c_str());
  if (!err.isOk())
    throw CTError("HBCICard::instituteData::instituteData",err);
}


HBCICard::instituteData::~instituteData(){
}


CTError HBCICard::instituteData::fromString(const string &data){
  int i;

  // reset data
  _name.erase();
  _code.erase();
  _service=0;
  _addr.erase();
  _addr_suffix.erase();
  _country=0;
  _user.erase();


  // check if data size is ok
  if (data.length()<88)
    return CTError("HBCICard::instituteData::fromString()",
		   k_CTERROR_INVALID,0,0,
		   "wrong length of data.\n");

  // check if empty data
  for (i=0; i<(int)data.length(); i++)
    if (data[i]!=32)
      break;
  if (i>=(int)data.length())
    return CTError("HBCICard::instituteData::fromString()",
		   k_CTERROR_INVALID,0,0,
		   "no information in data");

  _name=data.substr(0,20);
  CTMisc::removeBlanks(_name);
  _code.erase();
  _code=CTMisc::bsd2string(data.substr(20,4));
  _service=data.at(24);
  _addr=data.substr(25,28);
  CTMisc::removeBlanks(_addr);
  _addr_suffix=data.substr(53,2);
  CTMisc::removeBlanks(_addr_suffix);
  // the following line caused some trouble, but seems to be ok now.
  _country=CTMisc::string2num(data.substr(55,3));
  _user=data.substr(58,30);
  CTMisc::removeBlanks(_user);
  return CTError();
}


string HBCICard::instituteData::toString() const {
    string result;
    string tmp;
    char buffer[88];

    memset(buffer,32,sizeof(buffer));
    result.assign(buffer,sizeof(buffer));

    // name
    if (_name.length()>20)
        throw CTError("HBCICard::instituteData::toString()",
                      k_CTERROR_INVALID,0,0,
                      "name too long");
    result.replace(0,_name.length(),_name);

    // institute code
    tmp=CTMisc::string2bsd(_code);
    if (tmp.length()>4)
        throw CTError("HBCICard::instituteData::toString()",
                      k_CTERROR_INVALID,0,0,
                      "institute code too long");
    result.replace(20,tmp.length(),tmp);

    // service
    result[24]=_service;

    // address
    if (_addr.length()>28)
        throw CTError("HBCICard::instituteData::toString()",
                      k_CTERROR_INVALID,0,0,
                      "addr too long");
    result.replace(25,_addr.length(),_addr);

    // address suffix
    if (_addr_suffix.length()>2)
        throw CTError("HBCICard::instituteData::toString()",
                      k_CTERROR_INVALID,0,0,
                      "addr too long");
    result.replace(53,_addr_suffix.length(),_addr_suffix);

    // country
    tmp=CTMisc::num2string(_country);
    if (tmp.length()>3)
        throw CTError("HBCICard::instituteData::toString()",
                      k_CTERROR_INVALID,0,0,
                      "country code too long");
    result.replace(55,tmp.length(),tmp);

    // user
    if (_user.length()>30)
        throw CTError("HBCICard::instituteData::toString()",
                      k_CTERROR_INVALID,0,0,
                      "user too long");
    result.replace(58,_user.length(),_user);

    // done.
    return result;
}


string HBCICard::instituteData::dump(){
    string result;

    result+="Country       : ";
    result+=CTMisc::num2string(_country)+"\n";
    result+="Institute Name: ";
    result+=_name+"\n";
    result+="Institute Code: ";
    result+=_code+"\n";
    result+="Service Type  : ";
    if (_service==1)
        result+="CEPT\n";
    else if (_service==2)
        result+="TCP\n";
    else
        result+=CTMisc::num2string(_service)+" (unknown)\n";
    result+="IP Address    : ";
    result+=_addr+"\n";
    result+="IP Port       : ";
    result+=_addr_suffix+"\n";
    result+="User ID       : ";
    result+=_user+"\n";

    return result;
}




/*___________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * HBCICard
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


HBCICard::HBCICard(const CTCard &c)
:CTProcessorCard(c)
,_type(k_HBCICARD_TYPE_UNKNOWN)
{
}


HBCICard::~HBCICard(){
}


CTError HBCICard::openCard(){
  CTError err;

  err=CTProcessorCard::openCard();
  if (!err.isOk())
    return err;
  err=reopenCard();
  if (!err.isOk()) {
    CTProcessorCard::closeCard();
    return err;
  }
  return CTError();
}


CTError HBCICard::closeCard(bool force){
    CTError err;

    err=CTProcessorCard::closeCard(force);
    _type=k_HBCICARD_TYPE_UNKNOWN;
    return err;
}


CTError HBCICard::_checkType(){
    string id;
    string fcp;
    CTError err;
    unsigned char id1[9]={0xd2,0x76,0x00,0x00,0x25,0x48,0x42,0x01,0x00};
    unsigned char id2[9]={0xd2,0x76,0x00,0x00,0x25,0x48,0x42,0x02,0x00};

    _type=k_HBCICARD_TYPE_UNKNOWN;

    // first try version 2
    id.assign((char*)id2,9);
    err=selectById(fcp,id);
    // ok ?
    if (err.isOk()) {
        // yes, so it is a type 1 card
        _type=k_HBCICARD_TYPE_1;
        return err;
    }
    // was it a HW error ?
    if (err.code()!=k_CTERROR_OK)
        // yes, then return this error
        return err;

    // then try version 1
    id.assign((char*)id1,9);
    err=selectById(fcp,id);
    if (err.isOk()) {
        _type=k_HBCICARD_TYPE_0;
    }
    else {
      DBG_VERBOUS("LIBCHIPCARD: Card is not a DDV card");
    }
    return err;
}


CTError HBCICard::_getCID(string &cid){
  CTCommand cmd;
  CTError err;

  // select appropriate banking DF
  if (_type==k_HBCICARD_TYPE_UNKNOWN) {
    err=_checkType();
    if (!err.isOk())
      return err;
  }

  if (_type==k_HBCICARD_TYPE_0) {
    DBG_VERBOUS("LIBCHIPCARD: Card type is 0");
  }
  else if (_type==k_HBCICARD_TYPE_1) {
    DBG_VERBOUS("LIBCHIPCARD: Card type is 1");
  }

  // read record 1 of EF_ID
  cmd.setCla(0x00);      // chip card command
  cmd.setIns(0xb2);      // read record
  cmd.setP1(0x01);       // first record
  cmd.setP2(0xcc);       // directly select SFI=19
  cmd.setLr(255);
  cmd.setData(""); // data to send
  err=execCommand(cmd);
  if (!err.isOk()) {
    DBG_VERBOUS("LIBCHIPCARD: Could not read CID (IO).");
    return err;
  }
  cid=cmd.data();
  return err;
}


CTError HBCICard::getCID(string &cid){
  if (_cid.empty())
    return CTError("HBCICard::getCID()",
		   k_CTERROR_INVALID,
		   0,
		   0,
		   "card is not open");
  cid=_cid;
  return CTError();
}


HBCICard::CardData HBCICard::getCardId(){
  string cid;
  CTError err;

  err=getCID(cid);
  if (!err.isOk())
    throw err;
  return CardData(cid, _type);
}


HBCICard::instituteData HBCICard::getInstituteData(int num){
  CTCommand cmd;
  CTError err;

  // read record n of EF_BNK
  cmd.setCla(0x00);      // chip card command
  cmd.setIns(0xb2);      // read record
  cmd.setP1(num);        // record number
  cmd.setP2(0xd4);       // directly select SFI=1ah
  cmd.setLr(255);
  cmd.setData("");   // data to send
  err=execCommand(cmd);
  if (!err.isOk())
    throw err;
  // return a new instituteData
  return instituteData(cmd.data());
}


CTError HBCICard::getInstituteData(int num,
				   HBCICard::instituteData &idata){
  CTCommand cmd;
  CTError err;

  // read record n of EF_BNK
  cmd.setCla(0x00);      // chip card command
  cmd.setIns(0xb2);      // read record
  cmd.setP1(num);        // record number
  cmd.setP2(0xd4);       // directly select SFI=1ah
  cmd.setLr(255);
  cmd.setData("");   // data to send
  err=execCommand(cmd);
  if (!err.isOk())
    return CTError("HBCICard::getInstituteData", err);

  // return a new instituteData
  return idata.fromString(cmd.data());
}



CTError HBCICard::putInstituteData(int num, const instituteData &d){
  string response;

  return execCommand("put_inst_data",
		   _cmdPutInstData,
		   response,
		   CTMisc::num2string(num),
		   CTMisc::bin2hex(d.toString()));
}


CTError HBCICard::verifyPin(const string &pin){
  CTError err;
  string response;

  DBG_INFO(
	   "----------------- WARNING -----------\n"
	   "Your Pin may be shown in the following lines, even if you\n"
	   "don't see it. When submitting this output for bug reporting,\n"
	   "PLEASE REMOVE the following area of your output file !\n");

  err=execCommand("verify_pin",
		  _cmdVerifyPin,
		  response,
		  "0x81",
		  pin);
  DBG_INFO(
	   "------------ END OF PIN OUTPUT ------\n"
	   "This is the end of the area you should remove from the\n"
	   "debugging output prior to submitting it.\n");

  if (!err.isOk())
    return CTError("HBCICard::verifyPin",err);
  return CTError();
}


CTError HBCICard::verifyPin(){
  CTError err;
  string response;
  int to;

  to=timeout();
  setTimeout(60);
  err=execCommand("secure_verify_pin",
		_cmdSecureVerifyPin,
		response,
		"0x81");
  setTimeout(to);
  if (!err.isOk())
    return CTError("HBCICard::verifyPin",err);
  return CTError();
}


bool HBCICard::_hash2mac0(string &hash, string &mac){
  CTCommand cmd;
  CTError err;
  string hashL;
  string hashR;

  if (hash.length()!=20)
    return false;
  hashL=hash.substr(0,8);
  hashR=hash.substr(8,12);

  // write HashR to EF_MAC
  cmd.setCla(0x00);      // chip card command
  cmd.setIns(0xdc);      // update record
  cmd.setP1(0x01);       // record 1
  cmd.setP2(0xdc);       // SFI 1B (EF_MAC)
  cmd.setLr(0);
  cmd.setData(hashR); // data to send
  err=execCommand(cmd);
  if (!err.isOk()) {
    DBG_INFO("LIBCHIPCARD: %s (1)",err.errorString().c_str());
    return false;
  }
  // put data HashL
  cmd.setCla(0x00);      // chip card command
  cmd.setIns(0xda);      // put data
  cmd.setP1(0x01);
  cmd.setP2(0x00);
  cmd.setLr(0);
  cmd.setData(hashL); // data to send
  err=execCommand(cmd);
  if (!err.isOk()) {
    DBG_INFO("LIBCHIPCARD: %s (2)",err.errorString().c_str());
    return false;
  }
  // now read back HashR with secure messaging to get the MAC
  cmd.setCla(0x04);      // chip card command with secure messaging
  cmd.setIns(0xb2);      // read record
  cmd.setP1(0x01);       // record 1
  cmd.setP2(0xdc);       // SFI 1B (EF_MAC)
  cmd.setLr(255);
  cmd.setData("");  // data to send
  err=execCommand(cmd);
  if (!err.isOk()) {
    DBG_INFO("LIBCHIPCARD: %s (3)",err.errorString().c_str());
    return false;
  }
  if (cmd.data().length()<20) {
    DBG_INFO("LIBCHIPCARD: MAC too small.\n");
    return false;
  }
  mac=cmd.data().substr(12,8);
  return true;
}


bool HBCICard::_hash2mac1(string &hash, string &mac){
  CTCommand cmd;
  CTError err;
  string hashL;
  string hashR;

  if (hash.length()!=20){
    DBG_INFO("LIBCHIPCARD: Bad Hash length (%d bytes). (1)",
	     hash.length());
    return false;
  }
  hashL=hash.substr(0,8);
  hashR=hash.substr(8,12);

  // write HashR to EF_MAC
  cmd.setCla(0x00);      // chip card command
  cmd.setIns(0xdc);      // update record
  cmd.setP1(0x01);       // record 1
  cmd.setP2(0xdc);       // SFI 1B (EF_MAC)
  cmd.setLr(0);          // expected size of card's response
  cmd.setData(hashR); // data to send
  err=execCommand(cmd);
  if (!err.isOk()) {
    DBG_INFO("LIBCHIPCARD: %s (2)",err.errorString().c_str());
    return false;
  }

  // send hashL and read the mac
  cmd.setCla(0x08);         // chip card command with secure messaging
  cmd.setIns(0xb2);         // read record
  cmd.setP1(0x01);          // record 1
  cmd.setP2(0xdc);          // SFI 1B (EF_MAC)
  cmd.setLr(256);           // this means send "0" as Lr
  cmd.setData((char)0xba);  // tag and length of response descriptor
  cmd.addData((char)0xc);
  cmd.addData((char)0xb4); // tag and length of cct (whatever that is)
  cmd.addData((char)0x0a);
  cmd.addData((char)0x87); // tag and length of random number
  cmd.addData((char)0x08);
  cmd.addData(hashL);      // left hash (as random number)
  cmd.addData((char)0x96); // tag and length of le (whatever that means)
  cmd.addData((char)0x01);
  cmd.addData((char)0x00);
  err=execCommand(cmd);
  if (!err.isOk()){
    DBG_INFO("LIBCHIPCARD: %s (3)",err.errorString().c_str());
    return false;
  }

  if (cmd.data().length()<24) {
    DBG_INFO("LIBCHIPCARD: EF_MAC too small (%d bytes). (4)",
	     cmd.data().length());
    return false;
  }
  mac=cmd.data().substr(16,8);
  return true;
}


bool HBCICard::hash2MAC(string &hash, string &mac){
  switch(_type) {
  case k_HBCICARD_TYPE_0:
    return _hash2mac0(hash,mac);
  case k_HBCICARD_TYPE_1:
    return _hash2mac1(hash,mac);
  default:
    DBG_INFO("LIBCHIPCARD: unknown chip card type (%8x) (%d).",
	     (unsigned int)this,
	    _type);
    return false;
  }
  // I know, we never reach this point, but let's keep GCC happy ;-)
  return false;
}


bool HBCICard::_getKeyVersion0(int key, int &kv){
  string data;
  CTError err;

  // first select EF_KEYD
  err=selectEF(data,0x0013);
  if (!err.isOk())
    return false;

  // read record "key" of EF_AUTD
  err=readRecord(data,key);
  if (!err.isOk())
    return false;
  if (data.length()<5)
    return false;
  kv=(unsigned char)(data.at(4));
  return true;
}


bool HBCICard::_getKeyVersion1(int key,int &kv){
  CTCommand cmd;
  CTError err;

  // GET_KEYINFO
  cmd.setCla(0xb0);       // special chip card command
  cmd.setIns(0xee);       // GET_KEYINFO
  cmd.setP1(0x80);        // DF-spezifisch
  cmd.setP2(key);         // key number
  cmd.setLr(255);
  cmd.setData("");   // data to send
  err=execCommand(cmd);
  if (!err.isOk())
    return false;
  kv=(unsigned char)(cmd.data().at(0));
  return true;
}


bool HBCICard::getKeyVersion(int key, int &kv){
  switch(_type) {
  case k_HBCICARD_TYPE_0:
    return _getKeyVersion0(key,kv);
  case k_HBCICARD_TYPE_1:
    return _getKeyVersion1(key,kv);
  default:
    return false;
  }
  // I know, we never reach this point, but let's keep GCC happy ;-)
  return false;
}


bool HBCICard::getRandom(string &d) {
  CTCommand cmd;
  CTError err;

  // get challenge
  cmd.setCla(0x00);      // chip card command
  cmd.setIns(0x84);      // get challenge
  cmd.setP1(0x00);
  cmd.setP2(0x00);
  cmd.setLr(255);
  cmd.setData("");  // send no data
  err=execCommand(cmd);
  if (!err.isOk())
    return false;
  if (cmd.data().length()!=8) {
    return false;
  }
  // store left
  d=cmd.data();
  return true;
}


bool HBCICard::cryptBlock(string &src, string &dst){
  CTCommand cmd;
  CTError err;

  // check size (must be 8)
  if (src.length()!=8)
    return false;

  // internal authenticate
  cmd.setCla(0x00);      // chip card command
  cmd.setIns(0x88);      // internal authenticate
  cmd.setP1(0x00);
  if (_type==k_HBCICARD_TYPE_0)
    cmd.setP2(0x80);
  else if (_type==k_HBCICARD_TYPE_1)
    cmd.setP2(0x83);
  else return false;
  cmd.setLr(255);
  cmd.setData(src);
  err=execCommand(cmd);
  if (!err.isOk())
    return false;
  if (cmd.data().length()!=8)
    return false;
  dst=cmd.data();
  return true;
}


bool HBCICard::readSEQ(unsigned int &d){
  CTCommand cmd;
  CTError err;
  int i;

  // read record 1 of EF_SEQ
  cmd.setCla(0x00);      // chip card command
  cmd.setIns(0xb2);      // read record
  cmd.setP1(0x01);       // first record
  cmd.setP2(0xe4);       // directly select SFI=1c
  cmd.setLr(255);
  cmd.setData("");       // data to send
  err=execCommand(cmd);
  if (!err.isOk())
    return false;
  i=cmd.data().at(0) & 0xff;
  d=i*256;
  i=cmd.data().at(1) & 0xff;
  d+=i;
  return true;
}


bool HBCICard::writeSEQ(unsigned int d){
  CTCommand cmd;
  CTError err;

  // update record 1 of EF_SEQ
  cmd.setCla(0x00);      // chip card command
  cmd.setIns(0xdc);      // update record
  cmd.setP1(0x01);       // first record
  cmd.setP2(0xe4);       // directly select SFI=1c
  cmd.setLr(0);
  cmd.setData((char)(d >> 8));
  cmd.addData((char) (d & 0xff));
  err=execCommand(cmd);
  if (!err.isOk())
    DBG_INFO("LIBCHIPCARD: %s",err.errorString().c_str());
  return err.isOk();
}


bool HBCICard::getCryptKeyNumber(int &kn){
  if (_type==k_HBCICARD_TYPE_0) {
    kn=1;
    return true;
  }
  else if (_type==k_HBCICARD_TYPE_1) {
    kn=3;
    return true;
  }
  return false;
}


bool HBCICard::getSignKeyNumber(int &kn){
  if (_type==k_HBCICARD_TYPE_0) {
    /* according to HBCI-spec it should be key 2, but that does not
     * work. So I return key 1 instead, which seems to work */
    kn=1;
    return true;
  }
  else if (_type==k_HBCICARD_TYPE_1) {
    kn=2;
    return true;
  }
  return false;
}


CTError HBCICard::reopenCard(){
  CTError err;
  string fcp;

  _type=k_HBCICARD_TYPE_UNKNOWN;
  if (!isProcessorCard())
    return CTError("CTCard::reopenCard()",
		   k_CTERROR_INVALID,
		   0,
		   0,
		   "not a HBCI card");
  err=selectMF(fcp);
  if (!err.isOk())
    return err;

  return _getCID(_cid);
}


string HBCICard::cardType(){
  return "HBCICard";
}


string HBCICard::cardTypes(){
  return CTProcessorCard::cardTypes()+",HBCICard";
}



string HBCICard::cardNumber() {
  CardData cid;

  if (_cid.empty())
    return "";
  cid=CardData(_cid, _type);
  return cid.cardNumber();
}



