package DBIShell::UTIL;

#  dbishell: A generic database shell based on the Perl DBI layer
#  Copyright (C) 2000  Vivek Dasmohapatra (vivek@etla.org)

#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.

#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

use 5.004;
use strict;
use DBI      qw/:sql_types/;
use Exporter qw();
use Term::ReadKey;

use constant FALSE     => (1 == 0);
use constant TRUE      => (1 == 1);
use constant ZERO_TRUE => '0E0';
use constant DEOL      => '/';       # default EOL sequence

use constant SQL_VARCHAR_T => { TYPE => SQL_VARCHAR };
use constant SQL_INTEGER_T => { TYPE => SQL_INTEGER };
use constant SQL_NUMERIC_T => { TYPE => SQL_NUMERIC };

use constant PWENT_NAME    => 0;
use constant PWENT_PASSWD  => 1;
use constant PWENT_UID     => 2;
use constant PWENT_GID     => 3;
use constant PWENT_QUOTA   => 4;
use constant PWENT_COMMENT => 5;
use constant PWENT_GCOS    => 6;
use constant PWENT_DIR     => 7;
use constant PWENT_SHELL   => 8;
use constant PWENT_EXPIRE  => 9;

use constant READMODE_REVERT => 0; # restore to last seen mode.
use constant READMODE_NORMAL => 1; # standard 'sane' mode.
use constant READMODE_NOECHO => 2; # good for passwords.
use constant READMODE_CBREAK => 3; # fnord.
use constant READMODE_RAW    => 4; # no translation except lf<->crlf.
use constant READMODE_RAW_LF => 5; # no translation.

use constant STAT_DEV     => 0;
use constant STAT_INO     => 1;
use constant STAT_MODE    => 2;
use constant STAT_NLINK   => 3;
use constant STAT_UID     => 4;
use constant STAT_GID     => 5;
use constant STAT_RDEV    => 6;
use constant STAT_SIZE    => 7;
use constant STAT_ATIME   => 8;
use constant STAT_MTIME   => 9;
use constant STAT_CTIME   => 10;
use constant STAT_BLKSIZE => 11;
use constant STAT_BLOCKS  => 12;

use constant STAT_PERM_MASK =>  07777;
use constant STAT_TYPE_MASK => ~07777;

use constant PARAM_IN  => 0;
use constant PARAM_OUT => 1;

use constant CONTEXT_NIL  => 0x00;  # ignore.
use constant CONTEXT_HEAD => 0x01;  # header row
use constant CONTEXT_DATA => 0x02;  # data   row
use constant CONTEXT_ITEM => 0x03;  # member of a list
use constant CONTEXT_LIST => 0x04;  # whole (simple) list
use constant CONTEXT_META => 0x05;  # meta data [no lines, etc]
use constant CONTEXT_HELP => 0x06;  # help text
use constant CONTEXT_INFO => 0x07;  # Info
use constant CONTEXT_WARN => 0x08;  # Warning
use constant CONTEXT_ERR  => 0x09;  # Error
use constant CONTEXT_TAB  => 0x0a;  # Completion choices

use constant CACHEABLE_STH_ATTR =>
  qw(NUM_OF_FIELDS NULLABLE  NAME NAME_lc Statement  SCALE
     NUM_OF_PARAMS PRECISION TYPE NAME_uc CursorName RowsInCache);

use constant DBI_ATTR_NAMES =>
  qw(Warn
     Active
     Kids
     ActiveKids
     PrintError
     RaiseError
     ChopBlanks
     LongTruncOk
     AutoCommit
     RowCacheSize
    );

use vars qw(%_sql_types *term_size);

BEGIN { *term_size = *Term::ReadKey::GetTerminalSize; }

use constant SQLT_OFFSET =>
  eval
  {
      my $O = 0;
      foreach (@{ $DBI::EXPORT_TAGS{sql_types} })
      {
	  $_sql_types{$_} = 1;
	  my $X = $::{'DBI::'}{$_}->();
	  ($X < $O) && ($O = $X);
      }
      return $O;
  };

#warn("SQLT_OFFSET = ",SQLT_OFFSET,"\n");

use constant UNSLASH_MAP => {
			     t    => "\t",
			     n    => "\n",
			     r    => "\r"
			    };

use constant GPL_LICENSE => <<'GplLicense';
		    GNU GENERAL PUBLIC LICENSE
		       Version 2, June 1991

 Copyright (C) 1989, 1991 Free Software Foundation, Inc.
                       59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 Everyone is permitted to copy and distribute verbatim copies
 of this license document, but changing it is not allowed.

			    Preamble

  The licenses for most software are designed to take away your
freedom to share and change it.  By contrast, the GNU General Public
License is intended to guarantee your freedom to share and change free
software--to make sure the software is free for all its users.  This
General Public License applies to most of the Free Software
Foundation's software and to any other program whose authors commit to
using it.  (Some other Free Software Foundation software is covered by
the GNU Library General Public License instead.)  You can apply it to
your programs, too.

  When we speak of free software, we are referring to freedom, not
price.  Our General Public Licenses are designed to make sure that you
have the freedom to distribute copies of free software (and charge for
this service if you wish), that you receive source code or can get it
if you want it, that you can change the software or use pieces of it
in new free programs; and that you know you can do these things.

  To protect your rights, we need to make restrictions that forbid
anyone to deny you these rights or to ask you to surrender the rights.
These restrictions translate to certain responsibilities for you if you
distribute copies of the software, or if you modify it.

  For example, if you distribute copies of such a program, whether
gratis or for a fee, you must give the recipients all the rights that
you have.  You must make sure that they, too, receive or can get the
source code.  And you must show them these terms so they know their
rights.

  We protect your rights with two steps: (1) copyright the software, and
(2) offer you this license which gives you legal permission to copy,
distribute and/or modify the software.

  Also, for each author's protection and ours, we want to make certain
that everyone understands that there is no warranty for this free
software.  If the software is modified by someone else and passed on, we
want its recipients to know that what they have is not the original, so
that any problems introduced by others will not reflect on the original
authors' reputations.

  Finally, any free program is threatened constantly by software
patents.  We wish to avoid the danger that redistributors of a free
program will individually obtain patent licenses, in effect making the
program proprietary.  To prevent this, we have made it clear that any
patent must be licensed for everyone's free use or not licensed at all.

  The precise terms and conditions for copying, distribution and
modification follow.

		    GNU GENERAL PUBLIC LICENSE
   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION

  0. This License applies to any program or other work which contains
a notice placed by the copyright holder saying it may be distributed
under the terms of this General Public License.  The "Program", below,
refers to any such program or work, and a "work based on the Program"
means either the Program or any derivative work under copyright law:
that is to say, a work containing the Program or a portion of it,
either verbatim or with modifications and/or translated into another
language.  (Hereinafter, translation is included without limitation in
the term "modification".)  Each licensee is addressed as "you".

Activities other than copying, distribution and modification are not
covered by this License; they are outside its scope.  The act of
running the Program is not restricted, and the output from the Program
is covered only if its contents constitute a work based on the
Program (independent of having been made by running the Program).
Whether that is true depends on what the Program does.

  1. You may copy and distribute verbatim copies of the Program's
source code as you receive it, in any medium, provided that you
conspicuously and appropriately publish on each copy an appropriate
copyright notice and disclaimer of warranty; keep intact all the
notices that refer to this License and to the absence of any warranty;
and give any other recipients of the Program a copy of this License
along with the Program.

You may charge a fee for the physical act of transferring a copy, and
you may at your option offer warranty protection in exchange for a fee.

  2. You may modify your copy or copies of the Program or any portion
of it, thus forming a work based on the Program, and copy and
distribute such modifications or work under the terms of Section 1
above, provided that you also meet all of these conditions:

    a) You must cause the modified files to carry prominent notices
    stating that you changed the files and the date of any change.

    b) You must cause any work that you distribute or publish, that in
    whole or in part contains or is derived from the Program or any
    part thereof, to be licensed as a whole at no charge to all third
    parties under the terms of this License.

    c) If the modified program normally reads commands interactively
    when run, you must cause it, when started running for such
    interactive use in the most ordinary way, to print or display an
    announcement including an appropriate copyright notice and a
    notice that there is no warranty (or else, saying that you provide
    a warranty) and that users may redistribute the program under
    these conditions, and telling the user how to view a copy of this
    License.  (Exception: if the Program itself is interactive but
    does not normally print such an announcement, your work based on
    the Program is not required to print an announcement.)

These requirements apply to the modified work as a whole.  If
identifiable sections of that work are not derived from the Program,
and can be reasonably considered independent and separate works in
themselves, then this License, and its terms, do not apply to those
sections when you distribute them as separate works.  But when you
distribute the same sections as part of a whole which is a work based
on the Program, the distribution of the whole must be on the terms of
this License, whose permissions for other licensees extend to the
entire whole, and thus to each and every part regardless of who wrote it.

Thus, it is not the intent of this section to claim rights or contest
your rights to work written entirely by you; rather, the intent is to
exercise the right to control the distribution of derivative or
collective works based on the Program.

In addition, mere aggregation of another work not based on the Program
with the Program (or with a work based on the Program) on a volume of
a storage or distribution medium does not bring the other work under
the scope of this License.

  3. You may copy and distribute the Program (or a work based on it,
under Section 2) in object code or executable form under the terms of
Sections 1 and 2 above provided that you also do one of the following:

    a) Accompany it with the complete corresponding machine-readable
    source code, which must be distributed under the terms of Sections
    1 and 2 above on a medium customarily used for software interchange; or,

    b) Accompany it with a written offer, valid for at least three
    years, to give any third party, for a charge no more than your
    cost of physically performing source distribution, a complete
    machine-readable copy of the corresponding source code, to be
    distributed under the terms of Sections 1 and 2 above on a medium
    customarily used for software interchange; or,

    c) Accompany it with the information you received as to the offer
    to distribute corresponding source code.  (This alternative is
    allowed only for noncommercial distribution and only if you
    received the program in object code or executable form with such
    an offer, in accord with Subsection b above.)

The source code for a work means the preferred form of the work for
making modifications to it.  For an executable work, complete source
code means all the source code for all modules it contains, plus any
associated interface definition files, plus the scripts used to
control compilation and installation of the executable.  However, as a
special exception, the source code distributed need not include
anything that is normally distributed (in either source or binary
form) with the major components (compiler, kernel, and so on) of the
operating system on which the executable runs, unless that component
itself accompanies the executable.

If distribution of executable or object code is made by offering
access to copy from a designated place, then offering equivalent
access to copy the source code from the same place counts as
distribution of the source code, even though third parties are not
compelled to copy the source along with the object code.

  4. You may not copy, modify, sublicense, or distribute the Program
except as expressly provided under this License.  Any attempt
otherwise to copy, modify, sublicense or distribute the Program is
void, and will automatically terminate your rights under this License.
However, parties who have received copies, or rights, from you under
this License will not have their licenses terminated so long as such
parties remain in full compliance.

  5. You are not required to accept this License, since you have not
signed it.  However, nothing else grants you permission to modify or
distribute the Program or its derivative works.  These actions are
prohibited by law if you do not accept this License.  Therefore, by
modifying or distributing the Program (or any work based on the
Program), you indicate your acceptance of this License to do so, and
all its terms and conditions for copying, distributing or modifying
the Program or works based on it.

  6. Each time you redistribute the Program (or any work based on the
Program), the recipient automatically receives a license from the
original licensor to copy, distribute or modify the Program subject to
these terms and conditions.  You may not impose any further
restrictions on the recipients' exercise of the rights granted herein.
You are not responsible for enforcing compliance by third parties to
this License.

  7. If, as a consequence of a court judgment or allegation of patent
infringement or for any other reason (not limited to patent issues),
conditions are imposed on you (whether by court order, agreement or
otherwise) that contradict the conditions of this License, they do not
excuse you from the conditions of this License.  If you cannot
distribute so as to satisfy simultaneously your obligations under this
License and any other pertinent obligations, then as a consequence you
may not distribute the Program at all.  For example, if a patent
license would not permit royalty-free redistribution of the Program by
all those who receive copies directly or indirectly through you, then
the only way you could satisfy both it and this License would be to
refrain entirely from distribution of the Program.

If any portion of this section is held invalid or unenforceable under
any particular circumstance, the balance of the section is intended to
apply and the section as a whole is intended to apply in other
circumstances.

It is not the purpose of this section to induce you to infringe any
patents or other property right claims or to contest validity of any
such claims; this section has the sole purpose of protecting the
integrity of the free software distribution system, which is
implemented by public license practices.  Many people have made
generous contributions to the wide range of software distributed
through that system in reliance on consistent application of that
system; it is up to the author/donor to decide if he or she is willing
to distribute software through any other system and a licensee cannot
impose that choice.

This section is intended to make thoroughly clear what is believed to
be a consequence of the rest of this License.

  8. If the distribution and/or use of the Program is restricted in
certain countries either by patents or by copyrighted interfaces, the
original copyright holder who places the Program under this License
may add an explicit geographical distribution limitation excluding
those countries, so that distribution is permitted only in or among
countries not thus excluded.  In such case, this License incorporates
the limitation as if written in the body of this License.

  9. The Free Software Foundation may publish revised and/or new versions
of the General Public License from time to time.  Such new versions will
be similar in spirit to the present version, but may differ in detail to
address new problems or concerns.

Each version is given a distinguishing version number.  If the Program
specifies a version number of this License which applies to it and "any
later version", you have the option of following the terms and conditions
either of that version or of any later version published by the Free
Software Foundation.  If the Program does not specify a version number of
this License, you may choose any version ever published by the Free Software
Foundation.

  10. If you wish to incorporate parts of the Program into other free
programs whose distribution conditions are different, write to the author
to ask for permission.  For software which is copyrighted by the Free
Software Foundation, write to the Free Software Foundation; we sometimes
make exceptions for this.  Our decision will be guided by the two goals
of preserving the free status of all derivatives of our free software and
of promoting the sharing and reuse of software generally.

			    NO WARRANTY

  11. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED
OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS
TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE
PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,
REPAIR OR CORRECTION.

  12. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,
INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING
OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED
TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY
YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER
PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE
POSSIBILITY OF SUCH DAMAGES.

		     END OF TERMS AND CONDITIONS
GplLicense

use vars qw(@SQL_NTYPES @SQL_CTYPES @SQL_DTYPES @SQL_TTYPES);
use vars qw($P $VERSION @EXPORT @EXPORT_OK %EXPORT_TAGS @ISA);
use vars qw(@PWENT @RMODE @SQL_T @PARAM @STAT @CONTEXT);

#        SQL_CHAR SQL_NUMERIC SQL_DECIMAL SQL_INTEGER SQL_SMALLINT
#        SQL_FLOAT SQL_REAL SQL_DOUBLE SQL_VARCHAR
#        SQL_DATE SQL_TIME SQL_TIMESTAMP
#        SQL_LONGVARCHAR SQL_BINARY SQL_VARBINARY SQL_LONGVARBINARY
#        SQL_BIGINT SQL_TINYINT
#        SQL_WCHAR SQL_WVARCHAR SQL_WLONGVARCHAR
#        SQL_BIT

foreach(
	qw(SQL_NUMERIC
	   SQL_DECIMAL
	   SQL_INTEGER
	   SQL_SMALLINT
	   SQL_BIGINT
	   SQL_FLOAT
	   SQL_TINYINT
	   SQL_DOUBLE
	   SQL_REAL
          )
       )
{
    $_sql_types{$_} || next;
    $SQL_NTYPES[$::{'DBI::'}{$_}->() - SQLT_OFFSET] = 1;
}

#SQL_NOT_A_DEFINED_CONSTANT_BUT_THIS_IS_OK

foreach(
	qw(SQL_WLONGVARCHAR
	   SQL_VARCHAR
	   SQL_BINARY
	   SQL_LONGVARCHAR
	   SQL_VARBINARY
	   SQL_CHAR
	   SQL_LONGVARBINARY
	   SQL_WVARCHAR
	   SQL_WCHAR
          )
       )
{
    $_sql_types{$_} || (warn("DBI $DBI::VERSION does not define $_\n"),next);
    $SQL_CTYPES[$::{'DBI::'}{$_}->() - SQLT_OFFSET] = 1;
}

foreach (qw(SQL_DATE))
{
    $_sql_types{$_} || (warn("DBI $DBI::VERSION does not define $_\n"),next);
    $SQL_DTYPES[$::{'DBI::'}{$_}->() - SQLT_OFFSET] = 1;
}

foreach (qw(SQL_TIME SQL_TIMESTAMP))
{
    $_sql_types{$_} || (warn("DBI $DBI::VERSION does not define $_\n"),next);
    $SQL_TTYPES[$::{'DBI::'}{$_}->() - SQLT_OFFSET] = 1;
}


@RMODE = qw(READMODE_REVERT
            READMODE_NORMAL
            READMODE_NOECHO
            READMODE_CBREAK
            READMODE_RAW
            READMODE_RAW_LF
           );

@PWENT = qw(PWENT_NAME
	    PWENT_PASSWD
	    PWENT_UID
	    PWENT_GID
	    PWENT_QUOTA
	    PWENT_COMMENT
	    PWENT_GCOS
	    PWENT_DIR
	    PWENT_SHELL
	    PWENT_EXPIRE
	   );

@STAT  = qw(
            STAT_DEV
            STAT_INO
            STAT_MODE
            STAT_NLINK
            STAT_UID
            STAT_GID
            STAT_RDEV
            STAT_SIZE
            STAT_ATIME
            STAT_MTIME
            STAT_CTIME
            STAT_BLKSIZE
            STAT_BLOCKS
            STAT_PERM_MASK
            STAT_TYPE_MASK
           );

@SQL_T = qw(
            SQL_VARCHAR_T
            SQL_INTEGER_T
            SQL_NUMERIC_T
           );

@PARAM = qw(PARAM_IN PARAM_OUT);

@CONTEXT = qw(
              CONTEXT_NIL
              CONTEXT_HEAD
              CONTEXT_DATA
              CONTEXT_ITEM
              CONTEXT_LIST
              CONTEXT_META
              CONTEXT_HELP
              CONTEXT_INFO
              CONTEXT_WARN
              CONTEXT_ERR
              CONTEXT_TAB
             );

$VERSION     = 0.01_21;
@EXPORT      = qw(IFDEBUG TRUE FALSE ZERO_TRUE IS_TRUE DEOL);
@EXPORT_OK   = ('stringify'  ,
		'kfmt'       ,
		'get_param'  ,
		'unslash'    ,
		'_NULLS'     ,
		'IS_TRUE'    ,
		'IS_NUMERIC' ,
		'IS_STRING'  ,
		'IS_DATE'    ,
		'IS_TIME'    ,
		'GPL_LICENSE',
		'PARAM_IN'   ,
		'PARAM_OUT'  ,
		@PWENT       ,
		@RMODE       ,
		@SQL_T       ,
		@STAT        ,
		@CONTEXT     ,
	       );



%EXPORT_TAGS = (
		stat      => \ @STAT    ,
		pwent     => \ @PWENT   ,
		readmode  => \ @RMODE   ,
		sql_types => \ @SQL_T   ,
		param     => \ @PARAM   ,
		context   => \ @CONTEXT ,
	       );

@ISA         = qw(Exporter);

sub kfmt           ($);
sub stringify      ($$;@);
sub get_param      ($;$);
sub unslash        ($$);
#sub prescan_format ($$);

sub IS_NUMERIC ($);
sub IS_STRING  ($);
sub IS_DATE    ($);
sub IS_TIME    ($);
sub IS_TRUE    ($);

sub IS_NUMERIC ($) { $SQL_NTYPES[$_[0] + SQLT_OFFSET] }
sub IS_STRING  ($) { $SQL_CTYPES[$_[0] + SQLT_OFFSET] }
sub IS_DATE    ($) { $SQL_DTYPES[$_[0] + SQLT_OFFSET] }
sub IS_TIME    ($) { $SQL_TTYPES[$_[0] + SQLT_OFFSET] }

sub IS_TRUE    ($) { ($_[0] =~ /^(?:no|false|NULL|off)$/i || !$_[0]) ? 0 : 1 }

sub IFDEBUG    (@) { $ENV{DEBUG_DBISHELL} && warn(caller(),"\n",@_) }

sub cache_sth_attr ($)
{
    my $x    = 0;
    my $sth  = shift(@_);
    my $csth = {};

    foreach my $a (CACHEABLE_STH_ATTR)
    {
	eval { if(exists($sth->{$a})) { $csth->{$a} = $sth->{$a} } };
    }

    eval {  $x++ }; # clear any errors

    return $csth;
}

sub _NULLS (@) { map { defined($_) ? $_ : 'NULL' } @_ }

sub unslash ($$)
{
    my $umap = $_[0] || UNSLASH_MAP;

    exists($umap->{$_[1]}) ? $umap->{$_[1]} : $_[1];
}

# Read in a specific parameter, in the terminal mode specified:
sub get_param ($;$)
{
    my $lf;
    my $param;
    my ($prompt,$mode) = @_;

    length($mode) || ($mode = READMODE_NORMAL);

    Term::ReadKey::ReadMode($mode, *STDIN);
    print(STDERR $prompt);
    chomp($param = readline(*STDIN));
    if($mode == READMODE_NOECHO){ print(STDERR $/) }
    Term::ReadKey::ReadMode(READMODE_REVERT, *STDIN);

    return $param;
}

# Yes, yes, I know: Data::Dumper can do this, but I like this format
# better.
sub stringify ($$;@)
{
    my @buf;
    my $S = shift(@_);
    my $N = shift(@_);
    local($P) = $P.$S;

    foreach my $arg (@_)
    {
	my $rtype;

	if    ( !defined($arg)                  ) { push(@buf, 'undef()') }
	elsif ( $arg =~ /^[\-\+]?\d+(?:.\d+)?$/ ) { push(@buf, $arg)      }
	elsif ( $rtype = ref($arg)              )
	{
	    if ($rtype eq 'HASH' || $arg =~ /HASH/)
	    {
		my @k;
		my @f;
		my @s;
		my $maxklen;
		foreach my $k (keys(%$arg))
		{
		    push(@k, $k);
		    push(@f, kfmt($k));
		    ($maxklen < length($f[$#f]))
		      && ($maxklen = length($f[$#f]));
		}
		for (my $x  = 0; $x < @k; $x++)
		{
		    my $f = sprintf("%-$maxklen.${maxklen}s", $f[$x]);
		    local($P) = $S x ($maxklen + length($P) + length($S)*4);
		    push(@s,
			 join(' => ',$f,stringify($S,$N,$arg->{$k[$x]}))
			);
		}
		push(@buf,
		     join(join(",$N", map{"$P$_"}@s),
			  "{$N",join('',"$N",substr($P,1),"}")
			 )
		    );
	    }
	    elsif ($rtype eq 'ARRAY' || $arg =~ /ARRAY/)
	    {
		push(@buf,
		     join(join(', ', map { stringify($S,$N,$_) } @$arg),
			  '[',']'
			 )
		    );
	    }
	    else { push(@buf, $arg) }
	}
	else { push(@buf, join(quotemeta($arg),"'","'")) }
    }
    return join(', ', @buf);
}

# aux func for stringify:
sub kfmt ($) { $_[0] =~/^\w+$/ ? $_[0] : join(quotemeta($_[0]),'"','"') }

sub dynaload ($)
{
    my $target = $_[0];
    my $load   = 0;
    my $symbol_table = \%::;

    #warn("dynaload :: @{[caller()]}\n===================\n");

    my @plist = split(/::/,$target);

    # we _could_ check %INC instead, but that only shows loaded files,
    # not loaded packages: there is a difference. This way we check the
    # symbol table itself, which has to be right, unless something really
    # funky happened.
    foreach my $p (@plist)
    {
	$symbol_table->{"${p}::"} || ($load = 1,last);
	$symbol_table = $symbol_table->{"${p}::"};
    }

    if($load)
    {
	my $source;
	$source = join('/',@plist);
	$source = join('.',$source,'pm');

	eval { require $source; };

	if ($@) { return undef() }
	else    { return $target }
    }

    return $target;
}

__END__
#TLF: Nikola Tesla dies for you...
=pod

=head1 NAME

DBIShell::UTIL - general utility routines and data

=head1 SYNOPSIS

  use DBIShell::UTIL;

=head1 DESCRIPTION

This module provides a set of general utility routines and data for
dbishell.

=head1 EXPORTED SYMBOLS

=head2 :readmode

    READMODE_REVERT
    READMODE_NORMAL
    READMODE_NOECHO
    READMODE_CBREAK
    READMODE_RAW
    READMODE_RAW_LF

Constants for use w. Term::ReadKey to specify terminal modes.

=head2 :pwent

    PWENT_NAME
    PWENT_PASSWD
    PWENT_UID
    PWENT_GID
    PWENT_QUOTA
    PWENT_COMMENT
    PWENT_GCOS
    PWENT_DIR
    PWENT_SHELL
    PWENT_EXPIRE

Constants defining the positions of fields in getpwent* style lists.

=head2 :sql_types

    SQL_VARCHAR_T
    SQL_INTEGER_T
    SQL_NUMERIC_T

Constants for use in ->bind_param_inout(PNUM, SCALAR_REF, MAXLEN, TYPE)
as the TYPE parameter.

=head2 :param

    Two values:

    PARAM_IN
    PARAM_OUT

Should be used as flags to indicate whether parameters are in or inout
parameters anywhere that dbishell code needs to do so.

=head2 DBIShell:UTIL::GPL_LICENSE

Constant containing the GPL

=head1 FUNCTIONS

=head2 DBIShell:UTIL::stringify(INDENT_CHAR, SEPARATOR_STR, LIST...)

Render a multilevel hash/array structure into a pretty printed string.
INDENT_CHAR is used for indentation [' ' is a good value] SEPARATOR_STR
ise used, together with a comma, to separate individual elements.

=head2 DBIShell:UTIL::kfmt(STRING)

Formats a hash key for stringify, puts quotes around it if required.

=head2 DBIShell:UTIL::get_param(PROMPT [, MODE])

Print out PROMPT to STDERR, and read the users response withthe terminal in
mode MODE. MODE should be a member of the :readmode group.

[ See L</:readmode> ]

=head2 DBIShell:UTIL::unslash(MAP, THING)

Use map [hash reference] MAP to turn THING into it's unescaped form.
If MAP is false, the default MAP is used.

=head2 DBIShell:UTIL::_NULLS(LIST)

Return LIST, but with all the undef() elemnts replaced by 'NULL'

=head2 DBIShell:UTIL::IS_TRUE(THING)

Interpret THING and return true or false:

'no', 'false', 'NULL', 'off' [case insensitive] and anything perl would
normally consider false generate a false return, everything else generates
a true return value.

=head2 DBIShell:UTIL::IS_NUMERIC(THING)

Is THING a SQL Type that is a number?

=head2 DBIShell:UTIL::IS_STRING(THING)

Is THING a SQL Type that is a string?

=head2 DBIShell:UTIL::IS_DATE(THING)

Is THING a SQL Type that is a date?

=head2 DBIShell:UTIL::IS_TIME(THING)

Is THING a SQL Type that is a time?

=head2 DBIShell:UTIL::cache_sth_attr(STH)

Take a dbi statement handle and cache the most useful attributes in a hash,
return a reference to said hash.

=head2 DBIShell:UTIL::dynaload(PACKAGE)

Attempt to load package PACKAGE from the perl library path:
If there is already a symbol table entry for said package, return true
immediately. If not, attempt to find and load it, and return undef()
on error, or true [PACKAGE] on success

=head1 SEE ALSO

dbishell.

=head1 AUTHOR

Vivek Dasmohapatra (vivek@etla.org)

=cut
