#!/usr/bin/env python
#
# Copyright (C) 2002-2003 Mark Ferrell <xrxgrok@yahoo.com>
#
# -----------------------------------------------------------------------
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
# -----------------------------------------------------------------------

import os, os.path, sys, commands, random, re

synonyms = ['2tla', 'c2t']
summary = """Export CVS revision(s) to a Tla project tree"""
usage = """usage totla: [-dicsS] [-T <version>] [<branch>.]<changesets> <destdir>
	-d		Override tla's Date entry
	-i		Import each changeset into the Tla archive
	-c		Commit each changeset to the Tla archive
	-s		Seal each changeset in the Tla archive
	-S		Attempt to generate Summary information from log
	-l		Use a file list to (potentially) speed tla's operation
			May cause problems if the tagging method causes some
			files stored in the CVS tree to be ignored in arch;
			non-default for that reason.
	-T <version>	Tag the changeset in the arch archive
	-r <integer>	Number of times to retry failed "cvs update" commands.
	-n <directory>  New files reuse explicit id from this Arch source tree.
        -N		Stop when a new explicit id would be required.
	-m              Stop when the same changeset requires the creation AND
	                the deletion of explicit id. This is probably a rename
			(a "move") and one may want to fix the ids manually.
	-D <root>	If your repository is mounted locally, specify the root
			of the local repository module to get with raw RCS.
			This is much faster than using CVS locally.  This
			should point at the module in the CVS root, not the CVS
			root itself.
	-A <file>	Translate CVS authors to Arch authors by way of the
			specified file.  The file should contain lines like
			"cvs_name    Arch Name <archname@example.com>". The
			amount of whitespace in between the CVS name and the
			Arch name is not important.
			THIS FEATURE CHANGES my-id DURING OPERATION. BE CAREFUL
			WITH ITS USE.
			
	<destdir>	path to destination Tla repository to generate The
			destination path must be an existing Tla archive path,
			this script will not initiate a new Tla archive
	<branch>	The branch to pull the following changesets from.
			Does not apply if using the branch or branch.
			forms of the changeset arguments.
	<changesets>	List of changesets to apply
	 cset1:cset2	Between set1 and set2, including set1 and set2.
	 cset1::cset2	Between set1 and set2, excluding set1 and set2.
	 cset:		set and following changesets on the same branch.
	 set::		After set on the same branch.
	 :cset		set and previous changesets on the same branch.
	 ::cset		Before set on the same branch.
	 cset		Just set
	 branch		All changesets on the branch.
	 branch.	The last changeset on the branch."""

save_tags_in = ',cscvs-totla-removed-tags'

class TlaError(Exception):
	"""TLA has returned an error value other than success"""

class CvsError(Exception):
	"""CVS has returned an error value other than success"""

author_translations = {}
author_re = re.compile('^(\S+)\s+(.*)$')

def load_author_translations(filename):
	file = open(filename, "r");
	line = file.readline()
	while line:
		match = author_re.match(line)
		if match:
			author_translations[match.group(1).strip()] = match.group(2).strip()
		line = file.readline()

def runTla_simple_output(command):
	pipe = os.popen('tla %s' % command)
	output = pipe.read().strip()
	ret = pipe.close()
	if ret: raise TlaError('"tla %s" failed with exit status %s' % (command, ret))
	return output
	
def runTla(command, user=None):
	"""Invoke tla; throw an appropriate TlaError in the event of an unsuccesful return"""
	old_user = ''
	try:
		if user:			# A horrible KLUDGE
			old_user = runTla_simple_output('my-id')
			runTla("my-id '%s'" % user)
		ret = os.system('tla %s' % command)
	finally:
		if user:
			try: runTla("my-id '%s'" % old_user)
			except TlaError, e: sys.stderr.write("caught '%s' resetting my-id; attempting to continue anyhow\n")
	if ret: raise TlaError('"tla %s" failed with exit status %s' % (command, ret))

def cvs_get(apply, rev, retries=None):
	if retries is None: retries = apply.retries
	command = "cvs -q update -p -r%s '%s'" % (rev.revision, rev.filename)
	pipe = os.popen(command)
	file = pipe.read()
	ret = pipe.close()
	if ret:
		if retries > 0: return cvs_get(apply, rev, retries - 1)
		else: raise CvsError('"%s" failed with exit status %s'
				     % (command, ret))
	else:
		return file

def safe_get_mode(filename):
	try:
		return os.lstat(filename).st_mode
	except OSError, e:
		return 0644		# KLUDGE

def rcs_get_with_perms(apply, rev):
	mask = os.umask(0022);
	os.umask(mask);			# set it back, yuck.
	filename = os.path.join(apply.local_repository, "%s,v" % rev.filename)
	attic_filename = os.path.join(os.path.dirname(filename), "Attic", os.path.basename(filename))
	if os.path.exists(attic_filename):
		filename = attic_filename
	mode = safe_get_mode(filename)
	mode |= 0222			# set globally writable, then...
	mode &= ~mask			# hack off the umask
	command = "co -q -p%s '%s'" % (rev.revision, filename)
	pipe = os.popen(command)
	file = pipe.read()
	ret = pipe.close()
	if ret: raise CvsError('"%s" failed with exit status %s' % (command, ret))
	else: return [file, mode]

def cvs_get_with_perms(apply, rev, retries=None):
	if apply.local_repository:
		return rcs_get_with_perms(apply, rev)
	
	if retries is None: retries = apply.retries

	cachefilename = os.path.join(",,mode-cache", rev.filename)
	if os.path.exists(rev.filename):
		mode = safe_get_mode(rev.filename)
		return [cvs_get(apply, rev, retries), mode]
	elif os.path.exists(cachefilename):
		mode = safe_get_mode(cachefilename)
		return [cvs_get(apply, rev, retries), mode]

	command = "cvs -Q update -r%s '%s'" % (rev.revision, rev.filename)
	undocommand = "cvs -Q update -A '%s'" % rev.filename
	ret = os.system(command)
	if ret:
		if retries > 0:
			# Make sure the file is gone so we don't hit the
			# os.path.exists test above
			try: os.unlink(rev.filename)
			except OSError, e: pass
			return cvs_get_perms(apply, rev, retries - 1)
		else: raise CvsError('"%s" failed with exit status %s'
				     % (command, ret))
	else:
		mode = safe_get_mode(rev.filename)
		file = open(rev.filename, "r")
		delta = file.read()
		file.close()
		if not os.path.exists(os.path.dirname(cachefilename)):
			os.makedirs(os.path.dirname(cachefilename))
		os.rename(rev.filename, cachefilename)
		file = open(cachefilename, "w")	# zero out file, leaving mode
		os.system(undocommand)
		return [delta, mode]

def write_file_breaking_links(rev, data, mode):
	# FIXME - I think this is better now - it breaks hard links
	newname = os.path.join(os.path.dirname(rev.filename),
			       ",,new.%s.%s" % (random.random(), os.path.basename(rev.filename)))
	try: os.unlink(newname)
	except OSError, e: pass
	file = open(newname, "w")
	file.write(data)
	file.close()
	os.chmod(newname, mode)
	os.rename(newname, rev.filename)
	

def add(apply, config, rev):
	sys.stdout.write("A %s [%s]\n" % (rev.filename, rev.revision))

	delta, mode = cvs_get_with_perms(apply, rev)

	oldwd = os.getcwd()

	try:
		os.chdir(apply.destdir)

		dirname = os.path.dirname(rev.filename)
		if len(dirname) and not os.path.exists(dirname):
			base = ''
			for dir in dirname.split('/'):
				base = os.path.join(base, dir)
				if os.path.exists(base): continue
				os.mkdir(base, 0755)

		write_file_breaking_links(rev, delta, mode)
		
		os.utime(rev.filename, (rev.time, rev.time))

		dirname = os.path.dirname(rev.filename)
		while dirname:
			os.utime(dirname, (rev.time, rev.time))
			dirname = os.path.dirname(dirname)
	finally:
		os.chdir(oldwd)


def remove(apply, config, rev):
	sys.stdout.write("R %s [%s]\n" % (rev.filename, rev.revision))

	oldwd = os.getcwd()
	os.chdir(apply.destdir)
	try: os.unlink(rev.filename)
	except OSError, e:
		if e.errno == 2:
			pass ## file already removed, harmless.
		else:
			raise e

	try:
		dirname = os.path.dirname(rev.filename)
		while dirname:
			os.utime(dirname, (rev.time, rev.time))
			dirname = os.path.dirname(dirname)
	finally:
		os.chdir(oldwd)

def dir_is_empty(dir):
    for entry in os.listdir(dir):
        if entry == '.arch-ids': continue
        dir_entry = os.path.join(dir, entry)
        if not os.path.isdir(dir_entry): return False
        if not dir_is_empty(dir_entry): return False
    return True

def remove_empty_dirs_recursively(apply, dir, removed_tags):
	for entry in os.listdir(dir):
		if entry == '.arch-ids':
			# Save and remove explicit tags
			archids = os.path.join(dir, '.arch-ids')
			for entry in os.listdir(archids):
				tagfile = os.path.join(archids, entry)
				if entry == '=id' or entry.endswith('.id'):
					remove_explicit_tags(apply, [tagfile], removed_tags)
				else:
					# Sanity, in case there are non-id files in .arch-ids
					os.unlink(tagfile)
			os.rmdir(archids)
		else:
			# All other entries must be "empty" directories (modulo .arch-id dirs)
			remove_empty_dirs_recursively(apply, os.path.join(dir, entry), removed_tags)
	os.rmdir(dir)

def prunedirs(apply, config, rev, removed_tags):
	class DontPrune: pass

	dir =  os.path.dirname(rev.filename)
	oldwd = os.getcwd()
	os.chdir(apply.destdir)

	try:
		while dir:
			# See if the path is actually empty
			try:
				if not dir_is_empty(dir): raise DontPrune
			except OSError, e:
				if e.errno == 2:
					dir = os.path.dirname(dir)
					if not len(dir): raise DontPrune
					continue # already deleted, harmless

			# Remove the entry
			sys.stdout.write("R %s [empty]\n" % dir)
			try:
				remove_empty_dirs_recursively(apply, dir, removed_tags)
				
				# Fix timestamps below this directory to match the
				# revision that did the removal
				dirname = os.path.dirname(dir)
				while dirname:
					os.utime(dirname, (rev.time, rev.time))
					dirname = os.path.dirname(dirname)
			finally:
				os.chdir(oldwd)

			dir = os.path.dirname(dir)
	except DontPrune: pass
	os.chdir(oldwd)


def patch(apply, config, rev):
	sys.stdout.write("U %s [%s]\n" % (rev.filename, rev.revision))

	delta, mode = cvs_get_with_perms(apply, rev)

	oldwd = os.getcwd()
	try:
		os.chdir(apply.destdir)

		dirname = os.path.dirname(rev.filename)
		if len(dirname) and not os.path.exists(dirname):
			base = ''
			for dir in dirname.split('/'):
				base = os.path.join(base, dir)
				if os.path.exists(base): continue
				os.mkdir(base, 0755)

		write_file_breaking_links(rev, delta, mode)

		os.utime(rev.filename, (rev.time, rev.time))

		dirname = os.path.dirname(rev.filename)
		while dirname:
			os.utime(dirname, (rev.time, rev.time))
			dirname = os.path.dirname(dirname)
	finally:
		os.chdir(oldwd)

def make_log(apply, config, changeset):
	import re

	oldwd = os.getcwd()
	try:
		os.chdir(apply.destdir)
		pipe = os.popen("tla make-log", "r")
		logfile = pipe.read()
		logfile = logfile.strip()
		pipe.close()
		file = open(logfile, "w")
		if apply.summary:
			if not changeset.log.count('\n'):
				line = changeset.log
			else:
				line, junk = changeset.log.split('\n',1)
				match = re.compile(r"^([^;:]+).*").match(line)
				if match: line = match.groups()[0]
				else: line = "%s..." % line
		else:
			line = "cscvs to tla changeset %d" % changeset.index
		file.write("Summary: %s\n" % line)
		file.write("CSCVSID: %s.%d\n" % (changeset.branchName, changeset.index))
		file.write("Keywords: cscvs:%s-%d\n\n" % (changeset.branch.branchName, changeset.index))
		file.write("Author: %s\n" % (changeset.author))
		stime, etime = changeset.time
		import CVS
		file.write("Date: %s\n" % CVS.time(stime))
		file.write("%s\n" % changeset.log)
		file.close()
	finally:
		os.chdir(oldwd)


def undo(apply, config, changeset, error = None):
	if error is not None:
		print "Performing undo due to exception: %s" % error
	os.chdir(apply.destdir)
	runTla("undo")
	sys.exit(1)

class CreateIdStop(Exception):
    """We need to create an explicit id but the user gave an option forbidding it."""

def add_explicit_tag(apply, file):
        if not apply.create_ids: raise CreateIdStop
	(s,o) = commands.getstatusoutput('tla add ' + file)
	if s != 0: raise TlaError('tla add %s failed with exit code %s: %s' % (file, s, o))
	# if we stop on add+remove, print names of files we are tagging
	if apply.no_add_remove:
		sys.stdout.write('T %s\n' % file)

def new_explicit_tags(apply, o):
	files = filter(None, o.split('\n'))
	for f in files:
		add_explicit_tag(apply, f)
	return files

import shutil
def cp_a(src, dst):
	shutil.copy2(src, dst)
	s = os.stat(src)
	os.chown(dst, s.st_uid, s.st_gid)

def reuse_explicit_tags(apply, o):
	retval = []
	files_to_tag = filter(None, o.split('\n'))
	for f in files_to_tag:
		dir = os.path.dirname(f)
		base = os.path.basename(f)
		if os.path.isdir(os.path.join(apply.destdir, f)):
			idpath = os.path.join(f, '.arch-ids', '=id')
		else:
			idpath = os.path.join(dir, '.arch-ids', base + '.id')
                dest_iddir = os.path.dirname(os.path.join(apply.destdir, idpath))
                if not os.path.isdir(dest_iddir): os.mkdir(dest_iddir)
                if os.path.isfile(os.path.join(apply.ids_from_dir, idpath)):
                    cp_a(os.path.join(apply.ids_from_dir, idpath),
                         os.path.join(apply.destdir, idpath))
                else:
		    add_explicit_tag(apply, f)
		    retval.append(f)
	return retval

def remove_explicit_tags(apply, tagfiles, removed_tags):
	# if we stop on add+remove, print explicit tags we are deleting
	if apply.no_add_remove:
		for t in tagfiles:
			sys.stdout.write('D %s\n' % t)
	# Remove any explicit tags that tree-lint says are superfluous
	for t in tagfiles:
		removed_tags[t] = open(t).read()
		os.unlink(t)

def save_removed_tags(apply, tags):
	base = os.path.join(apply.destdir, save_tags_in)
	for name, data  in tags.iteritems():
		name = os.path.join(base, name)
		dir = os.path.dirname(name)
		if not os.path.isdir(dir): os.makedirs(dir)
		open(name, 'w').write(data)

class AddRemoveStop(Exception):
 	"""Explicit tags were created AND deleted in this changeset and the
 	configuration says to stop processing whenever this happens."""

def fix_explicit_tags(apply, oldwd, removed_tags={}, safe=False):
	# Add any explicit tags that tree-lint says are missing
	os.chdir(apply.destdir)
	try:
		# Add missing tags. We need to iterate until no explicit tag
		# is missing so explicit tags for new files inside new
		# directories are properly created.
		has_new_tag = []
		while True:
			(s,o) = commands.getstatusoutput('tla tree-lint -t --strict')
			if s == 0: break		# no tags needed adding
			elif s != 256: raise TlaError('tree-lint -t failed with exit code %s: %s' % (s, o))
			elif not apply.ids_from_dir:
				has_new_tag.append(new_explicit_tags(apply, o))
			else:
				has_new_tag.append(reuse_explicit_tags(apply, o))

		# Remove superfluous explicit tag files
		(s,o) = commands.getstatusoutput('tla tree-lint -m --strict')
		if s == 0: pass # no superfluous tag
		elif s != 256: raise TlaError ('tla tree-lint -m failed with exit code %s: %s' % (s, o))
		else:
			remove_explicit_tags(apply, filter(None, o.split('\n')), removed_tags)

		if not safe and apply.no_add_remove and has_new_tag and removed_tags:
			save_removed_tags(apply, removed_tags)
			raise AddRemoveStop
	finally:
		os.chdir(oldwd)
 
def apply_changeset(apply, config, changeset):
	import time
	import StorageLayer
	SCM = StorageLayer.convertingImport('SCM')

	date_string=''
	oldwd = os.getcwd()
	try:
		sys.stdout.write("N changeset %d\n" % changeset.index)

		# unless tla restricts us to working with the full tree only (as in the
		# case where  there are any additions or moves scheduled), we can use a
		# list of files to dramatically speed the commit operation
		try: 
			revisions = None
			if apply.post == 'commit':
				revisions = changeset.getAllNewRevisionsIterator()
			elif apply.post == 'import':
				revisions = changeset.getAllRevisionsIterator()
			elif apply.post == '':
				revisions = changeset.getAllNewRevisionsIterator()

			prune = {}
			for rev in revisions:
				filename = os.path.basename(rev.filename)
				if filename == '.cvsignore': continue
				if rev.type == SCM.Revision.Revision.PLACEHOLDER:
					pass
				elif rev.type == SCM.Revision.Revision.REMOVE:
					remove(apply, config, rev)
					dirname = os.path.dirname(rev.filename)
                                        while dirname:
                                            prune[dirname] = rev
                                            dirname = os.path.dirname(dirname)

				elif rev.type == SCM.Revision.Revision.ADD or \
				    (rev.type == SCM.Revision.Revision.CHANGE and
				     rev.predecessor != None and
				     rev.predecessor.type in (SCM.Revision.Revision.PLACEHOLDER,
				                              SCM.Revision.Revision.REMOVE)):
					add(apply, config, rev)
				elif rev.type == SCM.Revision.Revision.CHANGE:
					patch(apply, config, rev)

			# Prune empty directories..  We do this after we have processed
			# all the revisions incase a different revision adds a new
			# file, we don't want to go removing and adding a directory
			# w/in the same changeset.
			removed_tags = {}
			for dir in prune.keys(): prunedirs(apply, config, prune[dir], removed_tags)

			fix_explicit_tags(apply, oldwd, removed_tags)
				
		except (CvsError, TlaError), e:
			if isinstance(e, CvsError): fix_explicit_tags(apply, oldwd, safe=True)
			undo(apply, config, changeset, e)
			raise e

		except (AddRemoveStop, CreateIdStop):
			make_log(apply, config, changeset)
			raise

		make_log(apply, config, changeset)

		if apply.date:
			stime, etime = changeset.time
			date = time.gmtime(stime)
			date_string = time.strftime("--date '%a %b %d %H:%M:%S %Y'", date)

		if len(apply.post):
			os.chdir(apply.destdir)
			try:
				if apply.author_translations:
					runTla("%s %s %s" % (apply.post, date_string, apply.seal),
					       author_translations[changeset.author])
				else:
					runTla("%s %s %s" % (apply.post, date_string, apply.seal))
			except TlaError, e: undo(apply, config, changeset, e)

		if apply.tag:
			os.chdir(apply.destdir)
			runTla("tag -A %s %s %s %s" % (apply.archive, date_string, apply.version, apply.tag))
	finally:
		os.chdir(oldwd)

def totla(config):
	import StorageLayer
	from RangeArgParser import RangeArgParser
	import getopt
	SCM = StorageLayer.convertingImport('SCM')
	CVS = StorageLayer.convertingImport('CVS')

	catalog = SCM.Catalog.Catalog(config)

	class ApplyConfig(RangeArgParser):
		def __init__(self, catalog):
			super(ApplyConfig, self).__init__(catalog)
			self.archive = ''
			self.version = ''
			self.post = ''
			self.destdir = ''
			self.summary = 0
			self.date = 0
			self.seal = ''
			self.tag = None
			self.retries = 0
			self.ids_from_dir = ''
                        self.create_ids = True
			self.no_add_remove = False
			self.local_repository = ''
			self.author_translations = ''

	apply = ApplyConfig(catalog)

	opts, args = getopt.getopt(config.args, 'dicsST:r:n:NmD:A:')

	for opt, val in opts:
		if opt == '-d':
			apply.date = 1
		elif opt == '-i':
			apply.post = 'import'
		elif opt == '-c':
			apply.post = 'commit'
		elif opt == '-s':
			apply.seal = '--seal'
		elif opt == '-S':
			apply.summary = 1
		elif opt == '-T':
			if os.system("tla valid-package-name --no-archive --vsn %s" % val):
				sys.stderr.write("cscvs [toarch aborted]: %s: Invalid version\n" % val)
				sys.exit(1)
			apply.tag = val
		elif opt == '-r':
			apply.retries = int(val)
		elif opt == '-n':
                	apply.ids_from_dir = val
                elif opt == '-N':
                        apply.create_ids = False
		elif opt == '-m':
			apply.no_add_remove = True
		elif opt == '-D':
			apply.local_repository = val
		elif opt == '-A':
			apply.author_translations = val
		else: raise CVS.Usage

	if len(args) > 2 or len(args) <= 1:
		raise CVS.Usage

	if not apply.parseArg(args[0]):
		raise CVS.Usage

	if apply.author_translations: load_author_translations(apply.author_translations)
	
	destdir = os.path.realpath(args[1])
	os.chdir(destdir) # throw an exception if destdir is not correct
	apply.destdir = destdir = os.path.realpath(destdir)
	os.chdir(apply.destdir)
	try: runTla("tree-root")
	except TlaError: sys.exit(1)
	line = os.popen("tla tree-version").readline().strip()
	apply.archive, apply.version = line.split('/')
	if not apply.archive and not apply.version: sys.exit(1)
	os.chdir(config.topdir)

	try:
		for set in apply.getChangesetIterator():
			apply_changeset(apply, config, set)
	except AddRemoveStop:
		print "Suspected move, aborting."
		print "Deleted tags are saved in " + save_tags_in
        except CreateIdStop:
        	print "New explicit id required, aborting."
	catalog.close()


if __name__ == "__main__":
	sys.path.append('modules')
	import StorageLayer
	CVS = StorageLayer.convertingImport('CVS')
	config = CVS.Config(os.path.curdir())
	config.cmd = 'annotate'
	config.argv = sys.argv[1:]
	config.cat_path = os.path.join(config.topdir, "CVS/Catalog.sqlite")

	try: totla(config)
	except CVS.Usage:
		sys.stderr.write("%s\n" % usage)
		sys.exit(1)

# tag: Mark Ferrell Sat Jun 28 11:43:40 CDT 2003 (cmds/totla.py)
#
