"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const language_server_1 = require("@volar/language-server");
const simpleProject_1 = require("@volar/language-server/lib/project/simpleProject");
const node_1 = require("@volar/language-server/node");
const language_core_1 = require("@vue/language-core");
const language_service_1 = require("@vue/language-service");
const ts = require("typescript");
const vscode_uri_1 = require("vscode-uri");
const reactivityAnalyze_1 = require("./lib/reactivityAnalyze");
const reactivityAnalyzeLS_1 = require("./lib/reactivityAnalyzeLS");
const connection = (0, node_1.createConnection)();
const server = (0, node_1.createServer)(connection);
const tsserverRequestHandlers = new Map();
let tsserverRequestId = 0;
connection.listen();
connection.onNotification('tsserver/response', ([id, res]) => {
    tsserverRequestHandlers.get(id)?.(res);
    tsserverRequestHandlers.delete(id);
});
connection.onInitialize(params => {
    const tsconfigProjects = (0, language_service_1.createUriMap)();
    const file2ProjectInfo = new Map();
    server.fileWatcher.onDidChangeWatchedFiles(({ changes }) => {
        for (const change of changes) {
            const changeUri = vscode_uri_1.URI.parse(change.uri);
            if (tsconfigProjects.has(changeUri)) {
                tsconfigProjects.get(changeUri).dispose();
                tsconfigProjects.delete(changeUri);
                file2ProjectInfo.clear();
            }
        }
    });
    let simpleLanguageService;
    return server.initialize(params, {
        setup() { },
        async getLanguageService(uri) {
            if (uri.scheme === 'file') {
                const fileName = uri.fsPath.replace(/\\/g, '/');
                let projectInfoPromise = file2ProjectInfo.get(fileName);
                if (!projectInfoPromise) {
                    projectInfoPromise = sendTsServerRequest('_vue:' + ts.server.protocol.CommandTypes.ProjectInfo, {
                        file: fileName,
                        needFileNameList: false,
                    });
                    file2ProjectInfo.set(fileName, projectInfoPromise);
                }
                const projectInfo = await projectInfoPromise;
                if (projectInfo) {
                    const { configFileName } = projectInfo;
                    let languageService = tsconfigProjects.get(vscode_uri_1.URI.file(configFileName));
                    if (!languageService) {
                        languageService = createProjectLanguageService(server, configFileName);
                        tsconfigProjects.set(vscode_uri_1.URI.file(configFileName), languageService);
                    }
                    return languageService;
                }
            }
            return simpleLanguageService ??= createProjectLanguageService(server, undefined);
        },
        getExistingLanguageServices() {
            return Promise.all([
                ...tsconfigProjects.values(),
                simpleLanguageService,
            ].filter(promise => !!promise));
        },
        reload() {
            for (const languageService of tsconfigProjects.values()) {
                languageService.dispose();
            }
            tsconfigProjects.clear();
            if (simpleLanguageService) {
                simpleLanguageService.dispose();
                simpleLanguageService = undefined;
            }
        },
    }, (0, language_service_1.createVueLanguageServicePlugins)(ts, {
        collectExtractProps(...args) {
            return sendTsServerRequest('_vue:collectExtractProps', args);
        },
        getComponentDirectives(...args) {
            return sendTsServerRequest('_vue:getComponentDirectives', args);
        },
        getComponentEvents(...args) {
            return sendTsServerRequest('_vue:getComponentEvents', args);
        },
        getComponentNames(...args) {
            return sendTsServerRequest('_vue:getComponentNames', args);
        },
        getComponentProps(...args) {
            return sendTsServerRequest('_vue:getComponentProps', args);
        },
        getComponentSlots(...args) {
            return sendTsServerRequest('_vue:getComponentSlots', args);
        },
        getElementAttrs(...args) {
            return sendTsServerRequest('_vue:getElementAttrs', args);
        },
        getElementNames(...args) {
            return sendTsServerRequest('_vue:getElementNames', args);
        },
        getImportPathForFile(...args) {
            return sendTsServerRequest('_vue:getImportPathForFile', args);
        },
        getPropertiesAtLocation(...args) {
            return sendTsServerRequest('_vue:getPropertiesAtLocation', args);
        },
        getDocumentHighlights(fileName, position) {
            return sendTsServerRequest('_vue:documentHighlights-full', {
                file: fileName,
                ...{ position },
                filesToSearch: [fileName],
            });
        },
        getEncodedSemanticClassifications(fileName, span) {
            return sendTsServerRequest('_vue:encodedSemanticClassifications-full', {
                file: fileName,
                ...span,
                format: ts.SemanticClassificationFormat.TwentyTwenty,
            });
        },
        async getQuickInfoAtPosition(fileName, { line, character }) {
            const result = await sendTsServerRequest('_vue:' + ts.server.protocol.CommandTypes.Quickinfo, {
                file: fileName,
                line: line + 1,
                offset: character + 1,
            });
            return result?.displayString;
        },
    }));
    async function sendTsServerRequest(command, args) {
        return await new Promise(resolve => {
            const requestId = ++tsserverRequestId;
            tsserverRequestHandlers.set(requestId, resolve);
            connection.sendNotification('tsserver/request', [requestId, command, args]);
        });
    }
    function createProjectLanguageService(server, tsconfig) {
        const commonLine = tsconfig && !ts.server.isInferredProjectName(tsconfig)
            ? (0, language_core_1.createParsedCommandLine)(ts, ts.sys, tsconfig)
            : (0, language_core_1.createParsedCommandLineByJson)(ts, ts.sys, ts.sys.getCurrentDirectory(), {});
        const language = (0, language_core_1.createLanguage)([
            {
                getLanguageId: uri => server.documents.get(uri)?.languageId,
            },
            (0, language_core_1.createVueLanguagePlugin)(ts, commonLine.options, commonLine.vueOptions, uri => uri.fsPath.replace(/\\/g, '/')),
        ], (0, language_service_1.createUriMap)(), uri => {
            const document = server.documents.get(uri);
            if (document) {
                language.scripts.set(uri, document.getSnapshot(), document.languageId);
            }
            else {
                language.scripts.delete(uri);
            }
        });
        return (0, language_service_1.createLanguageService)(language, server.languageServicePlugins, (0, simpleProject_1.createLanguageServiceEnvironment)(server, [...server.workspaceFolders.all]), {});
    }
});
connection.onInitialized(server.initialized);
connection.onShutdown(server.shutdown);
connection.onRequest('vue/interpolationRanges', async (params) => {
    const uri = vscode_uri_1.URI.parse(params.textDocument.uri);
    const languageService = await server.project.getLanguageService(uri);
    if (languageService) {
        const sourceFile = languageService.context.language.scripts.get(uri);
        if (sourceFile?.generated) {
            const ranges = [];
            for (const code of (0, language_core_1.forEachEmbeddedCode)(sourceFile.generated.root)) {
                const codeText = code.snapshot.getText(0, code.snapshot.getLength());
                if ((code.id.startsWith('template_inline_ts_')
                    && codeText.startsWith('0 +')
                    && codeText.endsWith('+ 0;'))
                    || (code.id.startsWith('style_') && code.id.endsWith('_inline_ts'))) {
                    for (const mapping of code.mappings) {
                        for (let i = 0; i < mapping.sourceOffsets.length; i++) {
                            ranges.push([
                                mapping.sourceOffsets[i],
                                mapping.sourceOffsets[i] + mapping.lengths[i],
                            ]);
                        }
                    }
                }
            }
            return ranges;
        }
    }
    return [];
});
const cacheDocuments = new Map();
connection.onRequest('vue/reactivityAnalyze', async (params) => {
    if (params.syncDocument) {
        const document = language_server_1.TextDocument.create(params.textDocument.uri, params.syncDocument.languageId, 0, params.syncDocument.content);
        const snapshot = ts.ScriptSnapshot.fromString(params.syncDocument.content);
        cacheDocuments.set(params.textDocument.uri, [document, snapshot]);
    }
    const uri = vscode_uri_1.URI.parse(params.textDocument.uri);
    const languageService = await server.project.getLanguageService(uri);
    const sourceScript = languageService.context.language.scripts.get(uri);
    let document;
    let snapshot;
    if (sourceScript) {
        document = languageService.context.documents.get(sourceScript.id, sourceScript.languageId, sourceScript.snapshot);
        snapshot = sourceScript.snapshot;
    }
    else if (cacheDocuments.has(params.textDocument.uri)) {
        const [doc, snap] = cacheDocuments.get(params.textDocument.uri);
        document = doc;
        snapshot = snap;
    }
    if (!document || !snapshot) {
        return;
    }
    let offset = document.offsetAt(params.position);
    if (sourceScript?.generated) {
        const serviceScript = sourceScript.generated.languagePlugin.typescript?.getServiceScript(sourceScript.generated.root);
        if (!serviceScript) {
            return;
        }
        const map = languageService.context.language.maps.get(serviceScript.code, sourceScript);
        let embeddedOffset;
        for (const [mapped, mapping] of map.toGeneratedLocation(offset)) {
            if ((0, language_core_1.isReferencesEnabled)(mapping.data)) {
                embeddedOffset = mapped;
                break;
            }
        }
        if (embeddedOffset === undefined) {
            return;
        }
        offset = embeddedOffset;
        const embeddedUri = languageService.context.encodeEmbeddedDocumentUri(sourceScript.id, serviceScript.code.id);
        document = languageService.context.documents.get(embeddedUri, serviceScript.code.languageId, serviceScript.code.snapshot);
        snapshot = serviceScript.code.snapshot;
    }
    const { languageService: tsLs, fileName } = (0, reactivityAnalyzeLS_1.getLanguageService)(ts, snapshot, document.languageId);
    const result = (0, reactivityAnalyze_1.analyze)(ts, tsLs, fileName, offset);
    if (!result) {
        return;
    }
    const subscribers = [];
    const dependencies = [];
    if (sourceScript?.generated) {
        const serviceScript = sourceScript.generated.languagePlugin.typescript?.getServiceScript(sourceScript.generated.root);
        if (!serviceScript) {
            return;
        }
        const docs = [
            languageService.context.documents.get(sourceScript.id, sourceScript.languageId, sourceScript.snapshot),
            document,
            languageService.context.language.maps.get(serviceScript.code, sourceScript),
        ];
        for (const dependency of result.dependencies) {
            let start = document.positionAt(dependency.getStart(result.sourceFile));
            let end = document.positionAt(dependency.getEnd());
            if (ts.isBlock(dependency) && dependency.statements.length) {
                const { statements } = dependency;
                start = document.positionAt(statements[0].getStart(result.sourceFile));
                end = document.positionAt(statements[statements.length - 1].getEnd());
            }
            const sourceRange = (0, language_service_1.getSourceRange)(docs, { start, end });
            if (sourceRange) {
                dependencies.push(sourceRange);
            }
        }
        for (const subscriber of result.subscribers) {
            if (!subscriber.sideEffectInfo) {
                continue;
            }
            let start = document.positionAt(subscriber.sideEffectInfo.handler.getStart(result.sourceFile));
            let end = document.positionAt(subscriber.sideEffectInfo.handler.getEnd());
            if (ts.isBlock(subscriber.sideEffectInfo.handler) && subscriber.sideEffectInfo.handler.statements.length) {
                const { statements } = subscriber.sideEffectInfo.handler;
                start = document.positionAt(statements[0].getStart(result.sourceFile));
                end = document.positionAt(statements[statements.length - 1].getEnd());
            }
            const sourceRange = (0, language_service_1.getSourceRange)(docs, { start, end });
            if (sourceRange) {
                subscribers.push(sourceRange);
            }
        }
    }
    else {
        for (const dependency of result.dependencies) {
            let start = document.positionAt(dependency.getStart(result.sourceFile));
            let end = document.positionAt(dependency.getEnd());
            if (ts.isBlock(dependency) && dependency.statements.length) {
                const { statements } = dependency;
                start = document.positionAt(statements[0].getStart(result.sourceFile));
                end = document.positionAt(statements[statements.length - 1].getEnd());
            }
            dependencies.push({ start, end });
        }
        for (const subscriber of result.subscribers) {
            if (!subscriber.sideEffectInfo) {
                continue;
            }
            let start = document.positionAt(subscriber.sideEffectInfo.handler.getStart(result.sourceFile));
            let end = document.positionAt(subscriber.sideEffectInfo.handler.getEnd());
            if (ts.isBlock(subscriber.sideEffectInfo.handler) && subscriber.sideEffectInfo.handler.statements.length) {
                const { statements } = subscriber.sideEffectInfo.handler;
                start = document.positionAt(statements[0].getStart(result.sourceFile));
                end = document.positionAt(statements[statements.length - 1].getEnd());
            }
            subscribers.push({ start, end });
        }
    }
    return {
        subscribers,
        dependencies,
    };
});
//# sourceMappingURL=index.js.map