use std::fmt::Write;

use arrayvec::ArrayString;
use glob::glob;
use tokio::join;
use udisks2::block::BlockProxy;
use udisks2::drive::{DriveProxy, RotationRate};
use udisks2::filesystem::FilesystemProxy;
use udisks2::partition::PartitionProxy;
use udisks2::partitiontable::PartitionTableProxy;
use udisks2::{Client, Object};
use zbus::proxy::PropertyStream;
use zbus::zvariant;
use zbus::zvariant::OwnedObjectPath;

use magpie_platform::disks::Disk;

use crate::disks::stats::Stats;
use crate::disks::util;
use crate::util::{read_i64, read_u64};

/// Just contains all the PropertyStream's to declutter the DiskWrapper struct
#[derive(Default)]
pub struct StreamListener {
    pub ata_temp_change_stream: Option<PropertyStream<'static, f64>>,

    pub partition_table_change_stream: Option<PropertyStream<'static, Vec<OwnedObjectPath>>>,

    pub capacity_change_stream: Option<PropertyStream<'static, u64>>,
    pub drive_wwn_change_stream: Option<PropertyStream<'static, String>>,
    pub nvme_wwn_change_stream: Option<PropertyStream<'static, String>>,
    pub drive_change_stream: Option<PropertyStream<'static, OwnedObjectPath>>,
    pub ejectable_change_stream: Option<PropertyStream<'static, bool>>,
    pub size_change_stream: Option<PropertyStream<'static, u64>>,
    pub rotation_rate_change_stream: Option<PropertyStream<'static, RotationRate>>,
    pub serial_change_stream: Option<PropertyStream<'static, String>>,
    pub model_change_stream: Option<PropertyStream<'static, String>>,
    pub vendor_change_stream: Option<PropertyStream<'static, String>>,
}

impl StreamListener {
    /// Joins streams from another StreamListener into this one only if the fields are None
    /// Returns itself for daisy-chaining Java style
    pub fn join(&mut self, other: StreamListener) -> &mut StreamListener {
        if self.ata_temp_change_stream.is_none() {
            self.ata_temp_change_stream = other.ata_temp_change_stream;
        }

        if self.partition_table_change_stream.is_none() {
            self.partition_table_change_stream = other.partition_table_change_stream;
        }

        if self.capacity_change_stream.is_none() {
            self.capacity_change_stream = other.capacity_change_stream;
        }

        if self.drive_wwn_change_stream.is_none() {
            self.drive_wwn_change_stream = other.drive_wwn_change_stream;
        }

        if self.nvme_wwn_change_stream.is_none() {
            self.nvme_wwn_change_stream = other.nvme_wwn_change_stream;
        }

        if self.drive_change_stream.is_none() {
            self.drive_change_stream = other.drive_change_stream;
        }

        if self.ejectable_change_stream.is_none() {
            self.ejectable_change_stream = other.ejectable_change_stream;
        }

        if self.size_change_stream.is_none() {
            self.size_change_stream = other.size_change_stream;
        }

        if self.rotation_rate_change_stream.is_none() {
            self.rotation_rate_change_stream = other.rotation_rate_change_stream;
        }

        if self.serial_change_stream.is_none() {
            self.serial_change_stream = other.serial_change_stream;
        }

        if self.model_change_stream.is_none() {
            self.model_change_stream = other.model_change_stream;
        }

        if self.vendor_change_stream.is_none() {
            self.vendor_change_stream = other.vendor_change_stream;
        }

        self
    }
}

#[derive(Default)]
pub struct PartitionGrouper {
    pub partition_object: Option<PartitionProxy<'static>>,
    pub partition_size_stream: Option<PropertyStream<'static, u64>>,
    pub filesystem_object: Option<FilesystemProxy<'static>>,
    pub filesystem_size_stream: Option<PropertyStream<'static, u64>>,
    pub block_object: Option<BlockProxy<'static>>,
    pub block_size_stream: Option<PropertyStream<'static, u64>>,
}

#[derive(Default)]
pub struct DiskWrapper {
    pub disk_id: String,
    pub root_object: Option<Object>,
    pub partition_table: Option<PartitionTableProxy<'static>>,
    pub block_proxy: Option<BlockProxy<'static>>,
    pub drive_proxy: Option<DriveProxy<'static>>,
    pub drive_ata: Option<udisks2::ata::AtaProxy<'static>>,
    pub nvme_controller: Option<udisks2::nvme::controller::ControllerProxy<'static>>,
    pub nvme_namespace: Option<udisks2::nvme::namespace::NamespaceProxy<'static>>,
    pub partition_groups: Vec<PartitionGrouper>,
    pub is_system: bool,
    stream_listeners: StreamListener,
}

#[macro_export]
macro_rules! poll_wrapper {
    ($base_obj_option: expr, $stream_option: expr, $cached_function_call: ident, $function_call: ident) => {
        if let Some(base_obj) = $base_obj_option.as_ref() {
            if let Some(ref mut stream) = $stream_option {
                match stream_has_contents(stream).await {
                    Some(Ok(value)) => Some(value),
                    Some(Err(_)) => base_obj.$function_call().await.ok(),
                    None => match base_obj.$cached_function_call() {
                        Ok(value) => value,
                        Err(_) => base_obj.$function_call().await.ok(),
                    },
                }
            } else {
                base_obj.$function_call().await.ok()
            }
        } else {
            None
        }
    };
}

impl DiskWrapper {
    pub async fn new(disk_id: &str, udisks2: &Client, object: Object) -> Self {
        let mut wrapper = Self {
            disk_id: disk_id.to_string(),
            root_object: Some(object),
            partition_table: None,
            block_proxy: None,
            drive_proxy: None,
            drive_ata: None,
            nvme_controller: None,
            nvme_namespace: None,
            partition_groups: Vec::new(),
            is_system: false,
            stream_listeners: Default::default(),
        };

        // Initialize the wrapper with data from udisks2
        wrapper.initialize_from_udisks(udisks2).await;

        wrapper
    }

    async fn initialize_from_udisks(&mut self, udisks2: &Client) {
        if let Some(object) = self.root_object.as_ref() {
            // Get block proxy
            self.block_proxy = util::block(object, &self.disk_id).await;

            // Get drive proxy and ata data
            if let Some(block_proxy) = self.block_proxy.as_ref() {
                if let Some(drive_obj) = util::drive(udisks2, block_proxy, &self.disk_id).await {
                    let (drive_proxy, drive_ata, nvme_controller) = join!(
                        drive_obj.drive(),
                        drive_obj.drive_ata(),
                        drive_obj.nvme_controller()
                    );

                    if let Ok(drive_proxy) = drive_proxy {
                        self.drive_proxy = Some(drive_proxy);
                    }

                    if let Ok(drive_ata) = drive_ata {
                        self.drive_ata = Some(drive_ata);
                    }

                    // Get nvme controller if available
                    if let Ok(nvme_controller) = nvme_controller {
                        self.nvme_controller = Some(nvme_controller);
                    }
                }
            }

            let (nvme_namespace, partition_table) =
                join!(object.nvme_namespace(), object.partition_table());

            self.nvme_namespace = nvme_namespace.ok();
            self.partition_table = partition_table.ok();

            // Determine if this is a system disk
            self.is_system = false; //util::is_system(&self.disk_id, rt, &self.filesystems);
        }

        if let Some(partiton_groups) = self.load_partition_groups(udisks2).await {
            self.partition_groups = partiton_groups;

            'outer: for fs in self
                .partition_groups
                .iter()
                .filter_map(|group| group.filesystem_object.as_ref())
            {
                let Ok(mount_points) = fs.mount_points().await else {
                    continue;
                };

                for mount_point in mount_points {
                    let Ok(pt) = String::from_utf8(mount_point) else {
                        continue;
                    };

                    if pt == "/\0" {
                        self.is_system = true;
                        break 'outer;
                    }
                }
            }
        }

        let (atas, nvmes, drives, partitions, blocks) = join!(
            self.initialize_ata_listeners(),
            self.initialize_nvme_listeners(),
            self.initialize_drive_listeners(),
            self.initialize_partition_listeners(),
            self.initialize_block_proxy_listeners()
        );

        self.stream_listeners = atas;

        self.stream_listeners
            .join(nvmes)
            .join(drives)
            .join(partitions)
            .join(blocks);
    }

    async fn get_block_for_partition(partition: &Object) -> Option<BlockProxy<'static>> {
        partition.block().await.ok()
    }

    async fn get_filesystem_for_partition(
        partition: &Object,
        udisks2: &Client,
    ) -> Option<FilesystemProxy<'static>> {
        let mut object = partition.clone();

        if let Ok(encrypted) = object.encrypted().await {
            if let Ok(cleartext) = encrypted.cleartext_device().await {
                let Ok(obj) = udisks2.object(cleartext);

                object = obj;
            }
        }

        object.filesystem().await.ok()
    }

    async fn load_partition_groups(&mut self, udisks2: &Client) -> Option<Vec<PartitionGrouper>> {
        // already inited and no cheap way to look for changes, so we just wont.
        if !self.partition_groups.is_empty()
            && self
                .stream_listeners
                .partition_table_change_stream
                .is_none()
        {
            return None;
        }

        // do NOT use the nice macro. we DONT want a new value if there isnt a new value!

        if let Some(partition_table) = self.partition_table.as_ref() {
            if let Some(ref mut partition_listener) =
                self.stream_listeners.partition_table_change_stream
            {
                match stream_has_contents(partition_listener).await {
                    None => {
                        // no update, just quietly move on
                        None
                    }
                    Some(partition_result) => match partition_result {
                        Ok(mut partitions) => {
                            Some(Self::extract_partition_groups(udisks2, &mut partitions).await)
                        }
                        Err(_) => None,
                    },
                }
            } else if let Ok(mut partitions) = partition_table.partitions().await {
                Some(Self::extract_partition_groups(udisks2, &mut partitions).await)
            } else {
                None
            }
        } else {
            None
        }
    }

    async fn extract_partition_groups(
        udisks2: &Client,
        partitions: &mut Vec<OwnedObjectPath>,
    ) -> Vec<PartitionGrouper> {
        let mut out = Vec::new();

        for partition in partitions.drain(..) {
            let Some(obj) = udisks2.object(partition).ok() else {
                continue;
            };

            let block_obj = Self::get_block_for_partition(&obj).await;
            let block_resize = if let Some(block_obj) = block_obj.as_ref() {
                Some(block_obj.receive_size_changed().await)
            } else {
                None
            };

            let filesystem_obj = Self::get_filesystem_for_partition(&obj, udisks2).await;
            let filesystem_resize = if let Some(filesystem) = filesystem_obj.as_ref() {
                Some(filesystem.receive_size_changed().await)
            } else {
                None
            };

            let partition_obj = obj.partition().await.ok();
            let partition_resize = if let Some(partition) = partition_obj.as_ref() {
                Some(partition.receive_size_changed().await)
            } else {
                None
            };

            out.push(PartitionGrouper {
                partition_object: partition_obj,
                partition_size_stream: partition_resize,
                block_object: block_obj,
                block_size_stream: block_resize,
                filesystem_object: filesystem_obj,
                filesystem_size_stream: filesystem_resize,
            })
        }

        out
    }

    async fn initialize_block_proxy_listeners(&self) -> StreamListener {
        let mut out = StreamListener::default();

        if let Some(block_proxy) = self.block_proxy.as_ref() {
            let (capacity_change_stream, drive_change_stream) = join!(
                block_proxy.receive_size_changed(),
                block_proxy.receive_drive_changed()
            );

            out.capacity_change_stream = Some(capacity_change_stream);
            out.drive_change_stream = Some(drive_change_stream);
        }

        out
    }

    async fn initialize_drive_listeners(&self) -> StreamListener {
        let mut out = StreamListener::default();

        if let Some(drive) = self.drive_proxy.as_ref() {
            let (
                drive_wwn_change_stream,
                ejectable_change_stream,
                size_change_stream,
                rotation_rate_change_stream,
                serial_change_stream,
                model_change_stream,
                vendor_change_stream,
            ) = join!(
                drive.receive_wwn_changed(),
                drive.receive_ejectable_changed(),
                drive.receive_size_changed(),
                drive.receive_rotation_rate_changed(),
                drive.receive_serial_changed(),
                drive.receive_model_changed(),
                drive.receive_vendor_changed()
            );

            out.drive_wwn_change_stream = Some(drive_wwn_change_stream);
            out.ejectable_change_stream = Some(ejectable_change_stream);
            out.size_change_stream = Some(size_change_stream);
            out.rotation_rate_change_stream = Some(rotation_rate_change_stream);
            out.serial_change_stream = Some(serial_change_stream);
            out.model_change_stream = Some(model_change_stream);
            out.vendor_change_stream = Some(vendor_change_stream);
        }

        out
    }

    async fn initialize_nvme_listeners(&self) -> StreamListener {
        let mut out = StreamListener::default();

        if let Some(nvme_namespace) = self.nvme_namespace.as_ref() {
            out.nvme_wwn_change_stream = Some(nvme_namespace.receive_wwn_changed().await);
        }

        out
    }

    async fn initialize_partition_listeners(&self) -> StreamListener {
        let mut out = StreamListener::default();

        if let Some(partition_table) = self.partition_table.as_ref() {
            out.partition_table_change_stream =
                Some(partition_table.receive_partitions_changed().await);
        }

        out
    }

    async fn initialize_ata_listeners(&self) -> StreamListener {
        let mut out = StreamListener::default();

        if let Some(drive_ata) = self.drive_ata.as_ref() {
            out.ata_temp_change_stream = Some(drive_ata.receive_smart_temperature_changed().await);
        }

        out
    }

    pub async fn get_wwn(&mut self) -> Option<String> {
        if let Some(wwn) = poll_wrapper!(
            self.nvme_namespace,
            self.stream_listeners.nvme_wwn_change_stream,
            cached_wwn,
            wwn
        ) {
            return Some(wwn);
        }

        poll_wrapper!(
            self.drive_proxy,
            self.stream_listeners.drive_wwn_change_stream,
            cached_wwn,
            wwn
        )
    }

    pub async fn get_serial(&mut self) -> Option<String> {
        poll_wrapper!(
            self.drive_proxy,
            self.stream_listeners.serial_change_stream,
            cached_serial,
            serial
        )
    }

    fn map_rotation_rate(rate: Option<RotationRate>) -> Option<u64> {
        rate.map(|r| match r {
            RotationRate::Unknown => None,
            // a special case where Rotation is known, but zero
            RotationRate::NonRotating => None,
            RotationRate::Rotating(rate) => Some(rate as u64),
        })
        .flatten()
    }

    pub async fn get_rotation_rate(&mut self) -> Option<u64> {
        Self::map_rotation_rate(poll_wrapper!(
            self.drive_proxy,
            self.stream_listeners.rotation_rate_change_stream,
            cached_rotation_rate,
            rotation_rate
        ))
    }

    pub async fn get_formatted_bytes(&mut self) -> Option<u64> {
        let mut out = 0;

        for partition_group in self.partition_groups.iter_mut() {
            let mut this_amt = 0;

            this_amt = this_amt.max(
                poll_wrapper!(
                    partition_group.partition_object,
                    partition_group.partition_size_stream,
                    cached_size,
                    size
                )
                .unwrap_or(0),
            );
            this_amt = this_amt.max(
                poll_wrapper!(
                    partition_group.filesystem_object,
                    partition_group.filesystem_size_stream,
                    cached_size,
                    size
                )
                .unwrap_or(0),
            );
            this_amt = this_amt.max(
                poll_wrapper!(
                    partition_group.block_object,
                    partition_group.block_size_stream,
                    cached_size,
                    size
                )
                .unwrap_or(0),
            );

            out += this_amt;
        }

        Some(out)
    }

    pub async fn get_temperature(&mut self) -> Option<u32> {
        let mut hwmon_dirs = ArrayString::<256>::new();
        let device_id = &self.disk_id;
        match write!(
            &mut hwmon_dirs,
            "/sys/block/{device_id}/device/hwmon[0-9]*/temp[0-9]*_input"
        ) {
            Err(e) => {
                log::warn!("Failed to format hwmon dirs: {e:?}");
                return None;
            }
            _ => {}
        }

        let glob = match glob(hwmon_dirs.as_str()) {
            Ok(glob) => glob,
            Err(e) => {
                log::warn!("Failed to glob hwmon dirs: {e:?}");
                return None;
            }
        };

        if let Some(temperature) = glob
            .filter_map(Result::ok)
            .filter_map(|f| read_i64(f.as_os_str().to_string_lossy().as_ref(), "temperature"))
            .map(|i| (i + util::MK_TO_0_C) as u32)
            .next()
        {
            return Some(temperature);
        }

        poll_wrapper!(
            self.drive_ata,
            self.stream_listeners.ata_temp_change_stream,
            cached_smart_temperature,
            smart_temperature
        )
        .map(|f| (f * 1000.) as u32)
        .map(|v| if v == 0 { None } else { Some(v) })
        .flatten()
    }

    pub fn create_stats(&self) -> Stats {
        Stats::load(&self.disk_id)
    }

    pub async fn update_disk_obj(&mut self, disk: &mut Disk, client: Option<&Client>) {
        disk.capacity_bytes = self.get_capacity().await.unwrap_or(0);
        disk.temperature_milli_k = self.get_temperature().await;

        if let Some(client) = client {
            self.load_partition_groups(client).await;
        }

        disk.formatted_bytes = self.get_formatted_bytes().await;
    }

    pub async fn create_disk_obj(&mut self) -> Disk {
        Disk {
            id: self.disk_id.clone(),
            model: util::model(&self.disk_id),
            kind: util::kind(&self.disk_id, self.drive_proxy.as_ref())
                .await
                .map(|kind| kind.into()),
            smart_interface: util::smart_interface(
                self.drive_ata.as_ref(),
                self.nvme_controller.as_ref(),
            )
            .map(|kind| kind.into()),
            capacity_bytes: self.get_capacity().await.unwrap_or(0),
            formatted_bytes: self.get_formatted_bytes().await,
            is_system: self.is_system,
            busy_percent: 0.0,
            response_time_ms: 0.0,
            rx_speed_bytes_ps: 0,
            rx_bytes_total: 0,
            tx_speed_bytes_ps: 0,
            tx_bytes_total: 0,
            ejectable: self.get_ejectable().await,
            temperature_milli_k: self.get_temperature().await,
            serial_number: self.get_serial().await,
            world_wide_name: self.get_wwn().await,
            rotation_rate: self.get_rotation_rate().await,
            sector_size: 512,
        }
    }

    async fn get_capacity(&mut self) -> Option<u64> {
        if let Some(cap) = poll_wrapper!(
            self.block_proxy,
            self.stream_listeners.capacity_change_stream,
            cached_size,
            size
        ) {
            return Some(cap);
        }

        //backup method

        let disk_id = &self.disk_id;
        let mut size_file = ArrayString::<256>::new();
        if let Err(e) = write!(&mut size_file, "/sys/block/{disk_id}/size") {
            log::warn!("Failed to format disk size file: {e:?}");
            return None;
        }

        read_u64(size_file.as_str(), "disk size").and_then(
            |cap| {
                if cap == 0 {
                    None
                } else {
                    Some(cap)
                }
            },
        )
    }

    async fn get_ejectable(&mut self) -> bool {
        poll_wrapper!(
            self.drive_proxy,
            self.stream_listeners.ejectable_change_stream,
            cached_ejectable,
            ejectable
        )
        .unwrap_or(false)
    }
}

async fn stream_has_contents<T>(
    stream: &mut PropertyStream<'_, T>,
) -> Option<Result<T, zbus::Error>>
where
    T: TryFrom<zvariant::OwnedValue>,
    T::Error: Into<zbus::Error>,
    T: Unpin,
{
    use futures::stream::Stream;
    use std::task::{Context, Poll};

    // Create a dummy context for polling
    let mut stream = std::pin::pin!(stream);

    // Poll the stream for a next value without blocking
    match stream
        .as_mut()
        .poll_next(&mut Context::from_waker(&futures::task::noop_waker()))
    {
        Poll::Ready(Some(str)) => Some(str.get().await),
        Poll::Ready(None) => None, // Stream ended without values
        Poll::Pending => None,     // No immediate value available
    }
}
