/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.apache.streampipes.client.http;

import org.apache.streampipes.client.model.StreamPipesClientConfig;
import org.apache.streampipes.client.serializer.Serializer;
import org.apache.streampipes.client.util.StreamPipesApiPath;

import org.apache.http.client.fluent.Request;
import org.apache.http.entity.ContentType;

public abstract class PostRequest<K, V, T> extends HttpRequest<K, V, T> {

  private K body;
  private final boolean withBody;

  public PostRequest(StreamPipesClientConfig clientConfig,
                     StreamPipesApiPath apiPath,
                     Serializer<K, V, T> serializer,
                     K body) {
    super(clientConfig, apiPath, serializer);
    this.body = body;
    this.withBody = true;
  }

  public PostRequest(StreamPipesClientConfig clientConfig,
                     StreamPipesApiPath apiPath,
                     Serializer<K, V, T> serializer) {
    super(clientConfig, apiPath, serializer);
    this.withBody = false;
  }

  @Override
  protected Request makeRequest(Serializer<K, V, T> serializer) {
    Request request = Request
        .Post(makeUrl())
        .setHeaders(standardPostHeaders());

    if (withBody) {
      addBody(request, serializer);
    }

    return request;
  }

  protected void addBody(Request request, Serializer<K, V, T> serializer) {
    request.bodyString(serializer.serialize(body), ContentType.APPLICATION_JSON);
  }

}
