/*
 * Decompiled with CFR 0.152.
 */
package org.apache.amoro.server.table;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.amoro.AmoroTable;
import org.apache.amoro.ServerTableIdentifier;
import org.apache.amoro.TableFormat;
import org.apache.amoro.TableIDWithFormat;
import org.apache.amoro.api.CatalogMeta;
import org.apache.amoro.config.Configurations;
import org.apache.amoro.config.TableConfiguration;
import org.apache.amoro.exception.ObjectNotExistsException;
import org.apache.amoro.server.AmoroManagementConf;
import org.apache.amoro.server.catalog.CatalogManager;
import org.apache.amoro.server.catalog.ExternalCatalog;
import org.apache.amoro.server.catalog.InternalCatalog;
import org.apache.amoro.server.catalog.ServerCatalog;
import org.apache.amoro.server.manager.MetricManager;
import org.apache.amoro.server.optimizing.OptimizingStatus;
import org.apache.amoro.server.persistence.PersistentBase;
import org.apache.amoro.server.persistence.TableRuntimeMeta;
import org.apache.amoro.server.persistence.mapper.TableMetaMapper;
import org.apache.amoro.server.table.RuntimeHandlerChain;
import org.apache.amoro.server.table.TableRuntime;
import org.apache.amoro.server.table.TableService;
import org.apache.amoro.shade.guava32.com.google.common.annotations.VisibleForTesting;
import org.apache.amoro.shade.guava32.com.google.common.base.MoreObjects;
import org.apache.amoro.shade.guava32.com.google.common.base.Objects;
import org.apache.amoro.shade.guava32.com.google.common.base.Preconditions;
import org.apache.amoro.shade.guava32.com.google.common.collect.Lists;
import org.apache.amoro.shade.guava32.com.google.common.collect.Sets;
import org.apache.amoro.shade.guava32.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.amoro.utils.TablePropertyUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultTableService
extends PersistentBase
implements TableService {
    public static final Logger LOG = LoggerFactory.getLogger(DefaultTableService.class);
    private final long externalCatalogRefreshingInterval;
    private final Map<Long, TableRuntime> tableRuntimeMap = new ConcurrentHashMap<Long, TableRuntime>();
    private final ScheduledExecutorService tableExplorerScheduler = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("table-explorer-scheduler-%d").setDaemon(true).build());
    private final CompletableFuture<Boolean> initialized = new CompletableFuture();
    private final Configurations serverConfiguration;
    private final CatalogManager catalogManager;
    private RuntimeHandlerChain headHandler;
    private ExecutorService tableExplorerExecutors;

    public DefaultTableService(Configurations configuration, CatalogManager catalogManager) {
        this.catalogManager = catalogManager;
        this.externalCatalogRefreshingInterval = ((Duration)configuration.get(AmoroManagementConf.REFRESH_EXTERNAL_CATALOGS_INTERVAL)).toMillis();
        this.serverConfiguration = configuration;
    }

    @Override
    public void onTableCreated(InternalCatalog catalog, ServerTableIdentifier identifier) {
        this.triggerTableAdded(catalog, identifier);
    }

    @Override
    public void onTableDropped(InternalCatalog catalog, ServerTableIdentifier identifier) {
        Optional.ofNullable(this.tableRuntimeMap.get(identifier.getId())).ifPresent(tableRuntime -> {
            try {
                if (this.headHandler != null) {
                    this.headHandler.fireTableRemoved((TableRuntime)tableRuntime);
                }
                tableRuntime.dispose();
                this.tableRuntimeMap.remove(identifier.getId());
            }
            catch (Exception e) {
                LOG.error("Error occurred while removing tableRuntime of table {}", (Object)identifier.getId(), (Object)e);
            }
        });
    }

    @Override
    public void addHandlerChain(RuntimeHandlerChain handler) {
        this.checkNotStarted();
        if (this.headHandler == null) {
            this.headHandler = handler;
        } else {
            this.headHandler.appendNext(handler);
        }
    }

    @Override
    public void handleTableChanged(TableRuntime tableRuntime, OptimizingStatus originalStatus) {
        if (this.headHandler != null) {
            this.headHandler.fireStatusChanged(tableRuntime, originalStatus);
        }
    }

    @Override
    public void handleTableChanged(TableRuntime tableRuntime, TableConfiguration originalConfig) {
        if (this.headHandler != null) {
            this.headHandler.fireConfigChanged(tableRuntime, originalConfig);
        }
    }

    @Override
    public void initialize() {
        this.checkNotStarted();
        List tableRuntimeMetaList = this.getAs(TableMetaMapper.class, TableMetaMapper::selectTableRuntimeMetas);
        ArrayList<TableRuntime> tableRuntimes = new ArrayList<TableRuntime>(tableRuntimeMetaList.size());
        tableRuntimeMetaList.forEach(tableRuntimeMeta -> {
            TableRuntime tableRuntime = new TableRuntime((TableRuntimeMeta)tableRuntimeMeta, this);
            this.tableRuntimeMap.put(tableRuntimeMeta.getTableId(), tableRuntime);
            tableRuntime.registerMetric(MetricManager.getInstance().getGlobalRegistry());
            tableRuntimes.add(tableRuntime);
        });
        if (this.headHandler != null) {
            this.headHandler.initialize(tableRuntimes);
        }
        if (this.tableExplorerExecutors == null) {
            int threadCount = this.serverConfiguration.getInteger(AmoroManagementConf.REFRESH_EXTERNAL_CATALOGS_THREAD_COUNT);
            int queueSize = this.serverConfiguration.getInteger(AmoroManagementConf.REFRESH_EXTERNAL_CATALOGS_QUEUE_SIZE);
            this.tableExplorerExecutors = new ThreadPoolExecutor(threadCount, threadCount, 0L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(queueSize), new ThreadFactoryBuilder().setNameFormat("table-explorer-executor-%d").setDaemon(true).build());
        }
        this.tableExplorerScheduler.scheduleAtFixedRate(this::exploreTableRuntimes, 0L, this.externalCatalogRefreshingInterval, TimeUnit.MILLISECONDS);
        this.initialized.complete(true);
    }

    private TableRuntime getAndCheckExist(ServerTableIdentifier tableIdentifier) {
        Preconditions.checkArgument((tableIdentifier != null ? 1 : 0) != 0, (Object)"tableIdentifier cannot be null");
        TableRuntime tableRuntime = this.getRuntime(tableIdentifier.getId());
        if (tableRuntime == null) {
            throw new ObjectNotExistsException(tableIdentifier);
        }
        return tableRuntime;
    }

    @Override
    public TableRuntime getRuntime(Long tableId) {
        this.checkStarted();
        return this.tableRuntimeMap.get(tableId);
    }

    @Override
    public boolean contains(Long tableId) {
        this.checkStarted();
        return this.tableRuntimeMap.containsKey(tableId);
    }

    @Override
    public AmoroTable<?> loadTable(ServerTableIdentifier identifier) {
        return this.catalogManager.loadTable(identifier.getIdentifier());
    }

    @Override
    public void dispose() {
        this.tableExplorerScheduler.shutdown();
        if (this.tableExplorerExecutors != null) {
            this.tableExplorerExecutors.shutdown();
        }
        if (this.headHandler != null) {
            this.headHandler.dispose();
        }
    }

    @VisibleForTesting
    void exploreTableRuntimes() {
        if (!this.initialized.isDone()) {
            throw new IllegalStateException("TableService is not initialized");
        }
        long start = System.currentTimeMillis();
        List<ServerCatalog> externalCatalogs = this.catalogManager.getServerCatalogs();
        List externalCatalogNames = externalCatalogs.stream().map(ServerCatalog::name).collect(Collectors.toList());
        LOG.info("Syncing server catalogs: {}", (Object)String.join((CharSequence)",", externalCatalogNames));
        for (ServerCatalog serverCatalog : externalCatalogs) {
            try {
                if (serverCatalog.isInternal()) {
                    this.exploreInternalCatalog((InternalCatalog)serverCatalog);
                    continue;
                }
                this.exploreExternalCatalog((ExternalCatalog)serverCatalog);
            }
            catch (Throwable e) {
                LOG.error("TableExplorer error when explore table runtimes for catalog:{}", (Object)serverCatalog.name(), (Object)e);
            }
        }
        Set catalogNames = this.catalogManager.listCatalogMetas().stream().map(CatalogMeta::getCatalogName).collect(Collectors.toSet());
        for (TableRuntime tableRuntime : this.tableRuntimeMap.values()) {
            if (catalogNames.contains(tableRuntime.getTableIdentifier().getCatalog())) continue;
            this.disposeTable(tableRuntime.getTableIdentifier());
        }
        long l = System.currentTimeMillis();
        LOG.info("Syncing external catalogs took {} ms.", (Object)(l - start));
    }

    @VisibleForTesting
    public void exploreExternalCatalog(ExternalCatalog externalCatalog) {
        ArrayList tableIdentifiersFutures = Lists.newArrayList();
        externalCatalog.listDatabases().forEach(database -> {
            try {
                tableIdentifiersFutures.add(CompletableFuture.supplyAsync(() -> {
                    try {
                        return externalCatalog.listTables((String)database).stream().map(TableIdentity::new).collect(Collectors.toSet());
                    }
                    catch (Exception e) {
                        LOG.error("TableExplorer list tables in database {} error", database, (Object)e);
                        return new HashSet();
                    }
                }, this.tableExplorerExecutors));
            }
            catch (RejectedExecutionException e) {
                LOG.error("The queue of table explorer is full, please increase the queue size or thread count.");
            }
        });
        Set tableIdentifiers = tableIdentifiersFutures.stream().map(CompletableFuture::join).reduce((a, b) -> {
            a.addAll(b);
            return a;
        }).orElse(Sets.newHashSet());
        LOG.info("Loaded {} tables from external catalog {}.", (Object)tableIdentifiers.size(), (Object)externalCatalog.name());
        Map<TableIdentity, ServerTableIdentifier> serverTableIdentifiers = this.getAs(TableMetaMapper.class, mapper -> mapper.selectTableIdentifiersByCatalog(externalCatalog.name())).stream().collect(Collectors.toMap(TableIdentity::new, tableIdentifier -> tableIdentifier));
        LOG.info("Loaded {} tables from Amoro server catalog {}.", (Object)serverTableIdentifiers.size(), (Object)externalCatalog.name());
        ArrayList taskFutures = Lists.newArrayList();
        Sets.difference((Set)tableIdentifiers, serverTableIdentifiers.keySet()).forEach(tableIdentity -> {
            try {
                taskFutures.add(CompletableFuture.runAsync(() -> {
                    try {
                        this.syncTable(externalCatalog, (TableIdentity)tableIdentity);
                    }
                    catch (Exception e) {
                        LOG.error("TableExplorer sync table {} error", (Object)tableIdentity.toString(), (Object)e);
                    }
                }, this.tableExplorerExecutors));
            }
            catch (RejectedExecutionException e) {
                LOG.error("The queue of table explorer is full, please increase the queue size or thread count.");
            }
        });
        Sets.difference(serverTableIdentifiers.keySet(), (Set)tableIdentifiers).forEach(tableIdentity -> {
            try {
                taskFutures.add(CompletableFuture.runAsync(() -> {
                    try {
                        this.disposeTable((ServerTableIdentifier)serverTableIdentifiers.get(tableIdentity));
                    }
                    catch (Exception e) {
                        LOG.error("TableExplorer dispose table {} error", (Object)tableIdentity.toString(), (Object)e);
                    }
                }, this.tableExplorerExecutors));
            }
            catch (RejectedExecutionException e) {
                LOG.error("The queue of table explorer is full, please increase the queue size or thread count.");
            }
        });
        taskFutures.forEach(CompletableFuture::join);
    }

    private void exploreInternalCatalog(InternalCatalog internalCatalog) {
        LOG.info("Start explore internal catalog {}", (Object)internalCatalog.name());
        List identifiers = this.getAs(TableMetaMapper.class, m -> m.selectTableIdentifiersByCatalog(internalCatalog.name()));
        AtomicInteger addedCount = new AtomicInteger();
        identifiers.stream().filter(i -> !this.tableRuntimeMap.containsKey(i.getId())).peek(i -> LOG.info("Found new table {} in internal catalog {}, create table runtime for it.", i, (Object)internalCatalog.name())).peek(i -> addedCount.incrementAndGet()).forEach(i -> this.triggerTableAdded(internalCatalog, (ServerTableIdentifier)i));
        Set tableIds = identifiers.stream().map(ServerTableIdentifier::getId).collect(Collectors.toSet());
        HashSet tablesToBeDisposed = Sets.newHashSet();
        this.tableRuntimeMap.forEach((id, tableRuntime) -> {
            if (tableRuntime.getTableIdentifier().getCatalog().equals(internalCatalog.name()) && !tableIds.contains(id)) {
                LOG.info("Found table {} in internal catalog {} is removed, dispose table runtime for it.", id, (Object)internalCatalog.name());
                tablesToBeDisposed.add(tableRuntime.getTableIdentifier());
            }
        });
        tablesToBeDisposed.forEach(this::disposeTable);
        LOG.info("Explore internal catalog {} finished, {} tables are added, {} tables are disposed.", new Object[]{internalCatalog.name(), addedCount.get(), tablesToBeDisposed.size()});
    }

    private void checkStarted() {
        try {
            this.initialized.get();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void checkNotStarted() {
        if (this.initialized.isDone()) {
            throw new IllegalStateException("Table service has started.");
        }
    }

    private void syncTable(ExternalCatalog externalCatalog, TableIdentity tableIdentity) {
        AtomicBoolean tableRuntimeAdded = new AtomicBoolean(false);
        try {
            this.doAsTransaction(() -> externalCatalog.syncTable(tableIdentity.getDatabase(), tableIdentity.getTableName(), tableIdentity.getFormat()), () -> {
                ServerTableIdentifier tableIdentifier = externalCatalog.getServerTableIdentifier(tableIdentity.getDatabase(), tableIdentity.getTableName());
                tableRuntimeAdded.set(this.triggerTableAdded(externalCatalog, tableIdentifier));
            });
        }
        catch (Throwable t) {
            if (tableRuntimeAdded.get()) {
                this.revertTableRuntimeAdded(externalCatalog, tableIdentity);
            }
            throw t;
        }
    }

    private boolean triggerTableAdded(ServerCatalog catalog, ServerTableIdentifier serverTableIdentifier) {
        AmoroTable<?> table = catalog.loadTable(serverTableIdentifier.getDatabase(), serverTableIdentifier.getTableName());
        if (TableFormat.ICEBERG.equals((Object)table.format()) && TablePropertyUtil.isMixedTableStore((Map)table.properties())) {
            return false;
        }
        TableRuntime tableRuntime = new TableRuntime(serverTableIdentifier, this, table.properties());
        this.tableRuntimeMap.put(serverTableIdentifier.getId(), tableRuntime);
        tableRuntime.registerMetric(MetricManager.getInstance().getGlobalRegistry());
        if (this.headHandler != null) {
            this.headHandler.fireTableAdded(table, tableRuntime);
        }
        return true;
    }

    private void revertTableRuntimeAdded(ExternalCatalog externalCatalog, TableIdentity tableIdentity) {
        ServerTableIdentifier tableIdentifier = externalCatalog.getServerTableIdentifier(tableIdentity.getDatabase(), tableIdentity.getTableName());
        if (tableIdentifier != null) {
            this.tableRuntimeMap.remove(tableIdentifier.getId());
        }
    }

    private void disposeTable(ServerTableIdentifier tableIdentifier) {
        TableRuntime existedTableRuntime = this.tableRuntimeMap.get(tableIdentifier.getId());
        try {
            this.doAsTransaction(() -> Optional.ofNullable(existedTableRuntime).ifPresent(tableRuntime -> {
                try {
                    if (this.headHandler != null) {
                        this.headHandler.fireTableRemoved((TableRuntime)tableRuntime);
                    }
                    tableRuntime.dispose();
                    this.tableRuntimeMap.remove(tableIdentifier.getId());
                }
                catch (Exception e) {
                    LOG.error("Error occurred while disposing table {}", (Object)tableIdentifier, (Object)e);
                }
            }), () -> this.doAs(TableMetaMapper.class, mapper -> mapper.deleteTableIdByName(tableIdentifier.getCatalog(), tableIdentifier.getDatabase(), tableIdentifier.getTableName())));
        }
        catch (Throwable t) {
            this.tableRuntimeMap.putIfAbsent(tableIdentifier.getId(), existedTableRuntime);
            throw t;
        }
    }

    private static class TableIdentity {
        private final String database;
        private final String tableName;
        private final TableFormat format;

        protected TableIdentity(TableIDWithFormat idWithFormat) {
            this.database = idWithFormat.getIdentifier().getDatabase();
            this.tableName = idWithFormat.getIdentifier().getTableName();
            this.format = idWithFormat.getTableFormat();
        }

        protected TableIdentity(ServerTableIdentifier serverTableIdentifier) {
            this.database = serverTableIdentifier.getDatabase();
            this.tableName = serverTableIdentifier.getTableName();
            this.format = serverTableIdentifier.getFormat();
        }

        protected TableIdentity(String database, String tableName, TableFormat format) {
            this.database = database;
            this.tableName = tableName;
            this.format = format;
        }

        public String getDatabase() {
            return this.database;
        }

        public String getTableName() {
            return this.tableName;
        }

        public TableFormat getFormat() {
            return this.format;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            TableIdentity that = (TableIdentity)o;
            return Objects.equal((Object)this.database, (Object)that.database) && Objects.equal((Object)this.tableName, (Object)that.tableName);
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.database, this.tableName});
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("database", (Object)this.database).add("tableName", (Object)this.tableName).toString();
        }
    }
}

