/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hertzbeat.alert.calculate.realtime.window;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class TimeService {
    private static final Logger log = LoggerFactory.getLogger(TimeService.class);
    static final long DEFAULT_WATERMARK_DELAY_MS = 30000L;
    private static final long WATERMARK_BROADCAST_INTERVAL_MS = 5000L;
    private static final long MAX_FUTURE_TIME_MS = 60000L;
    private static final long MAX_PAST_TIME_MS = 86400000L;
    private final AtomicLong maxTimestamp = new AtomicLong(0L);
    private final AtomicLong currentWatermark = new AtomicLong(0L);
    private final CopyOnWriteArrayList<WatermarkListener> listeners = new CopyOnWriteArrayList();
    private ScheduledExecutorService scheduler;

    public TimeService(List<WatermarkListener> initialListeners) {
        this.listeners.addAll(initialListeners);
    }

    @PostConstruct
    public void start() {
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setUncaughtExceptionHandler((thread, throwable) -> {
            log.error("TimeService scheduler has uncaughtException.");
            log.error(throwable.getMessage(), throwable);
        }).setDaemon(true).setNameFormat("timeservice-scheduler-%d").build();
        this.scheduler = Executors.newSingleThreadScheduledExecutor(threadFactory);
        this.scheduler.scheduleAtFixedRate(this::broadcastWatermark, 0L, 5000L, TimeUnit.MILLISECONDS);
        log.info("TimeService started with watermark delay: {}ms", (Object)30000L);
    }

    @PreDestroy
    public void stop() {
        if (this.scheduler != null && !this.scheduler.isShutdown()) {
            log.info("Shutting down TimeService scheduler...");
            this.scheduler.shutdown();
            try {
                if (!this.scheduler.awaitTermination(10L, TimeUnit.SECONDS)) {
                    log.warn("TimeService scheduler did not terminate within 10 seconds, forcing shutdown");
                    this.scheduler.shutdownNow();
                    if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                        log.error("TimeService scheduler did not terminate");
                    }
                }
            }
            catch (InterruptedException e) {
                log.warn("Interrupted while waiting for TimeService scheduler to terminate");
                this.scheduler.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        log.info("TimeService stopped");
    }

    public boolean isValidTimestamp(long timestamp) {
        long currentTime = System.currentTimeMillis();
        return timestamp >= currentTime - 86400000L && timestamp <= currentTime + 60000L;
    }

    public boolean isLateData(long timestamp) {
        return timestamp < this.getCurrentWatermark();
    }

    public void updateMaxTimestamp(long timestamp) {
        this.maxTimestamp.getAndUpdate(current -> Math.max(current, timestamp));
    }

    public void addWatermarkListener(WatermarkListener listener) {
        this.listeners.add(listener);
    }

    public void removeWatermarkListener(WatermarkListener listener) {
        this.listeners.remove(listener);
    }

    public long getCurrentWatermark() {
        return this.currentWatermark.get();
    }

    public long getMaxTimestamp() {
        return this.maxTimestamp.get();
    }

    private void broadcastWatermark() {
        try {
            long maxTs = this.maxTimestamp.get();
            if (maxTs <= 0L) {
                return;
            }
            long newWatermark = maxTs - 30000L;
            long currentWm = this.currentWatermark.get();
            if (newWatermark <= currentWm) {
                return;
            }
            if (this.currentWatermark.compareAndSet(currentWm, newWatermark)) {
                Watermark watermark = new Watermark(newWatermark);
                for (WatermarkListener listener : this.listeners) {
                    try {
                        listener.onWatermark(watermark);
                    }
                    catch (Exception e) {
                        log.error("Error notifying watermark listener: {}", (Object)e.getMessage(), (Object)e);
                    }
                }
                log.debug("Broadcast watermark: {} (maxTimestamp: {})", (Object)newWatermark, (Object)maxTs);
            }
        }
        catch (Exception e) {
            log.error("Error in watermark broadcast: {}", (Object)e.getMessage(), (Object)e);
        }
    }

    public static class Watermark {
        private final long timestamp;

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof Watermark)) {
                return false;
            }
            Watermark other = (Watermark)o;
            if (!other.canEqual(this)) {
                return false;
            }
            return this.getTimestamp() == other.getTimestamp();
        }

        protected boolean canEqual(Object other) {
            return other instanceof Watermark;
        }

        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            long $timestamp = this.getTimestamp();
            result = result * 59 + (int)($timestamp >>> 32 ^ $timestamp);
            return result;
        }

        public String toString() {
            return "TimeService.Watermark(timestamp=" + this.getTimestamp() + ")";
        }

        public Watermark(long timestamp) {
            this.timestamp = timestamp;
        }

        public long getTimestamp() {
            return this.timestamp;
        }
    }

    public static interface WatermarkListener {
        public void onWatermark(Watermark var1);
    }
}

