/*  $Header: /home/mwicks/Repository/dvipdfm/dvi.c,v 1.39 1998/12/30 19:36:10 mwicks Exp $

    This is dvipdf, a DVI to PDF translator.
    Copyright (C) 1998  by Mark A. Wicks

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    
    The author may be contacted via the e-mail address

	mwicks@kettering.edu
*/

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "error.h"
#include "numbers.h"
#include "mfileio.h"
#include "pdflimits.h"
#include "pdfdev.h"
#include "pdfdoc.h"
#include "tfm.h"
#include "mem.h"
#include "dvi.h"
#include "vf.h"

#include "dvicodes.h"


/* Interal Variables */

static FILE *dvi_file;
static debug = 0;
static unsigned numfonts = 0, stackdepth;
static unsigned long *page_loc = NULL;
static long max_pages = 0;
static unsigned long post_location, dvi_file_size;
static UNSIGNED_PAIR numpages = 0;
static UNSIGNED_QUAD media_width, media_height;
static UNSIGNED_QUAD dvi_unit_num, dvi_unit_den, dvi_mag;


#define PHYSICAL 1
#define VIRTUAL 2
#define DVI 1
#define VF 2
struct font_def {
  int type;  /* Type is physical or virtual */
  int font_id;  /* id returned by dev (for PHYSICAL fonts)
		   or by vf module for (VIRTUAL fonts) */
  int tfm_id;
  mpt_t size;
  char *name;
  int source;  /* Source is either DVI or VF */
  signed long tex_id /* id used internally by TeX */;
} font_def[MAX_FONTS];

static unsigned char verbose = 0;

void dvi_set_verbose(void)
{
  if (verbose < 255) {
  verbose += 1;
  }
}

void dvi_set_debug(void)
{
  debug = 1;
}

int dvi_npages (void)
{
  return numpages;
}

static void invalid_signature()
{
  ERROR ("dvi_init:  Something is wrong.  Are you sure this is a DVI file?\n");
}

#define range_check_loc(loc) {if ((loc) > dvi_file_size) invalid_signature();}


static void find_post (void)
{
  long current;
  int read_byte;

  /* First find end of file */  
  dvi_file_size = file_size (dvi_file);
  current = dvi_file_size;
 
  /* Scan backwards through PADDING */  
  do {
     current -= 1;
     seek_absolute (dvi_file, current);

  } while ((read_byte = fgetc(dvi_file)) == PADDING &&
	   current > 0);

  /* file_position now points to last non padding character or beginning of file */
  if (dvi_file_size - current < 4 ||
      current == 0 || read_byte != DVI_ID) {
    fprintf (stderr, "DVI ID = %d\n", read_byte);
    invalid_signature();
  } 

  /* Make sure post_post is really there */
  current = current - 5;
  seek_absolute (dvi_file, current);
  if ((read_byte = fgetc(dvi_file)) != POST_POST) {
     fprintf (stderr, "Found %d where post_post opcode should be\n", read_byte);
     invalid_signature();
  }
  current = get_signed_quad (dvi_file);
  seek_absolute (dvi_file, current);
  if ((read_byte = fgetc(dvi_file)) != POST) {
     fprintf (stderr, "Found %d where post_post opcode should be\n", read_byte);
     invalid_signature();
  }
  post_location = current;
}

static void get_page_info (void) 
{
  int i;
  seek_absolute (dvi_file, post_location+27);
  numpages = get_unsigned_pair (dvi_file);
  if (verbose > 2) {
    fprintf (stderr, "Page count:\t %4d\n", numpages);
  }
  if (numpages == 0) {
    ERROR ("dvi_init:  Page count is 0!");
  }
  max_pages = numpages;
  page_loc = NEW (max_pages, unsigned long);
  seek_absolute (dvi_file, post_location+1);
  page_loc[numpages-1] = get_unsigned_quad(dvi_file);
  range_check_loc(page_loc[numpages-1]+41);
  for (i=numpages-2; i>=0; i--) {
    seek_absolute (dvi_file, page_loc[i+1]+41);
    page_loc[i] = get_unsigned_quad(dvi_file);
    range_check_loc(page_loc[numpages-1]+41);
  }
}

/* Following are computed "constants" used for unit conversion */
static double dvi2pts = 0.0, total_mag = 1.0;

double dvi_tell_mag (void)
{
  return total_mag;
}

static void do_scales (double mag)
{
  total_mag = (double) dvi_mag / 1000.0 * mag;
  dvi2pts = (double) dvi_unit_num / (double) dvi_unit_den;
  dvi2pts *= (72.0)/(254000.0);
  dvi2pts *= total_mag;
}


static void get_dvi_info (void)
{
  seek_absolute (dvi_file, post_location+5);
  dvi_unit_num = get_unsigned_quad(dvi_file);
  dvi_unit_den = get_unsigned_quad(dvi_file);
  dvi_mag = get_unsigned_quad(dvi_file);
  media_height = get_unsigned_quad(dvi_file);
  media_width = get_unsigned_quad(dvi_file);
  stackdepth = get_unsigned_pair(dvi_file);
  if (stackdepth > DVI_MAX_STACK_DEPTH) {
    fprintf (stderr, "DVI needs stack depth of %d,", stackdepth);
    fprintf (stderr, "but MAX_DVI_STACK_DEPTH is %d", DVI_MAX_STACK_DEPTH);
    ERROR ("Capacity exceeded.");
  }

  if (verbose > 2) {
    fprintf (stderr, "DVI File Info\n");
    fprintf (stderr, "Unit: %ld / %ld\n", dvi_unit_num, dvi_unit_den);
    fprintf (stderr, "Mag: %ld\n", dvi_mag);
    fprintf (stderr, "Media Height: %ld\n", media_height);
    fprintf (stderr, "Media Width: %ld\n", media_width);
    fprintf (stderr, "Stack Depth: %d\n", stackdepth);
  }
  
}

static void dump_font_info (void)
{
  unsigned i;
  fprintf (stderr, "\nFont info\n");
  for (i=0; i<numfonts; i++) {
    fprintf (stderr, "name: %10s, ", font_def[i].name);
    fprintf (stderr, "TeX/DVI ID: %5ld, ", font_def[i].tex_id);
    fprintf (stderr, "dev/vf ID: %5d, ", font_def[i].font_id);
    fprintf (stderr, "size: %5.2f pt, ", font_def[i].size*dvi2pts);
    switch (font_def[i].type) {
    case PHYSICAL:
      fprintf (stderr, "Type: PHYSICAL, ");
      break;
    case VIRTUAL:
      fprintf (stderr, "Type: VIRTUAL, ");
      break;
    }
    switch (font_def[i].source) {
    case DVI:
      fprintf (stderr, "Source: DVI file\n");
      break;
    case VF:
      fprintf (stderr, "Source: VF file\n");
      break;
    }
  }
}

static void get_a_font_record (SIGNED_QUAD tex_id)
{
  UNSIGNED_BYTE dir_length, name_length;
  UNSIGNED_QUAD checksum, size, design_size;
  char *directory, *name;
  int font_id;
  if (debug) {
    fprintf (stderr, "get_a_font_record: tex_id = %ld\n", tex_id);
  }
  checksum = get_unsigned_quad (dvi_file);
  size = get_unsigned_quad (dvi_file);
  design_size = get_unsigned_quad (dvi_file);
  dir_length = get_unsigned_byte (dvi_file);
  name_length = get_unsigned_byte (dvi_file);
  directory = NEW (dir_length+1, char);
  if (fread (directory, 1, dir_length, dvi_file) !=
      dir_length) {
    invalid_signature();
  }
  name = NEW (name_length+1, char);
  if (fread (name, 1, name_length, dvi_file) !=
      name_length) {
    invalid_signature();
  }
  directory[dir_length] = 0;
  name[name_length] = 0;
  font_id = dvi_locate_font (name, size);
  font_def[font_id].source = DVI;
  font_def[font_id].tex_id = tex_id;
  RELEASE (directory);
  RELEASE (name);
  return;
}

static void get_dvi_fonts (void)
{
  UNSIGNED_BYTE code;
  SIGNED_QUAD tex_id;
  seek_absolute (dvi_file, post_location+29);
  while (numfonts < MAX_FONTS && (code = get_unsigned_byte(dvi_file)) != POST_POST) {
    switch (code)
      {
      case FNT_DEF1:
	tex_id = get_unsigned_byte (dvi_file);
	break;
      case FNT_DEF2:
	tex_id = get_unsigned_pair (dvi_file);
	break;
      case FNT_DEF3:
	tex_id = get_unsigned_triple (dvi_file);
	break;
      case FNT_DEF4:
	tex_id = get_signed_quad (dvi_file);
	break;
      default:
	fprintf (stderr, "Unexpected op code: %3d\n", code);
	invalid_signature();
      }
    get_a_font_record(tex_id);
  }
  if (verbose>2) {
    dump_font_info();
  }
}


void get_comment(void)
{
  UNSIGNED_BYTE length;
  static char dvi_comment[257];
  seek_absolute (dvi_file, 14);
  length = get_unsigned_byte(dvi_file);
  if (fread (dvi_comment, 1, length, dvi_file) != length) {
    invalid_signature();
  }
  dvi_comment[length] = 0;
  if (verbose) {
    fprintf (stderr, "DVI Comment: %s\n", dvi_comment);
  }
  dev_add_comment (dvi_comment);
}


/* The section below this line deals with the actual processing of the
   dvi file.

   The dvi file processor state is contained in the following
   variables: */

struct dvi_registers {
  SIGNED_QUAD h, v, w, x, y, z;
};

static struct dvi_registers dvi_state;
static struct dvi_registers dvi_stack[DVI_MAX_STACK_DEPTH];
static int current_font;
static dvi_stack_depth = 0;  
static int processing_page = 0;

static void clear_state (void)
{
  dvi_state.h = 0; dvi_state.v = 0; dvi_state.w = 0;
  dvi_state.x = 0; dvi_state.y = 0; dvi_state.z = 0;
  dvi_stack_depth = 0;
  current_font = -1;
}


double dvi_unit_size(void)
{
  return dvi2pts;
}

int dvi_locate_font (char *tex_name, mpt_t ptsize)
{
  int thisfont;
  if (debug) {
    fprintf (stderr, "dvi_locate_font: fontname: (%s) ptsize: %ld, dvi_id: %d\n",
	     tex_name, ptsize, numfonts);
  }
  if (verbose)
    fprintf (stderr, "<%s@%.2fpt", tex_name, ptsize*dvi2pts);
  if (numfonts == MAX_FONTS)
    ERROR ("dvi_locate_font:  Tried to load too many fonts\n");
  /* This routine needs to be recursive/reentrant.  Load current high water
     mark into an automatic variable  */
  thisfont = numfonts++;
  font_def[thisfont].tfm_id = tfm_open (tex_name);
  font_def[thisfont].source = VF; /* This will be reset later if 
				     it was really generated by the
				     dvi file */
  /* type1_font_resource on next line always returns an *indirect*
     obj */ 
  font_def[thisfont].font_id = dev_locate_font (tex_name, ptsize);
  if (font_def[thisfont].font_id >= 0) {
    font_def[thisfont].type = PHYSICAL;
  } else {
    if (verbose)
      fprintf (stderr, "(VF)");
    font_def[thisfont].type = VIRTUAL;
    font_def[thisfont].font_id = vf_locate_font (tex_name, ptsize);
    if (font_def[thisfont].font_id < 0) {
      fprintf (stderr, "%s: Can't locate an AFM or VF file\n", tex_name);
      ERROR ("Not sure how to proceed.  For now this is fatal\n\
Maybe in the future, I'll substitute some other font.");
    }
  }
  font_def[thisfont].size = ptsize;
  font_def[thisfont].name = NEW (strlen(tex_name)+1, char);
  strcpy (font_def[thisfont].name, tex_name);
  if (verbose)
    fprintf (stderr, ">");
  return (thisfont);
}


double dvi_dev_xpos(void) {
  return dvi_state.h*dvi2pts;
}

double dvi_dev_ypos (void)
{
  return -(dvi_state.v*dvi2pts);
}

static void do_moveto (SIGNED_QUAD x, SIGNED_QUAD y)
{
  dvi_state.h = x;
  dvi_state.v = y;
}

void dvi_right (SIGNED_QUAD x)
{
  dvi_state.h += x;
}

void dvi_down (SIGNED_QUAD y)
{
  dvi_state.v += y;
}

static void do_string (unsigned char *s, int len)
{
  mpt_t width = 0;
  int i;
  struct font_def *p;
  if (debug) {
    int i;
    fprintf (stderr, "do_string: (font: %d)\n", current_font);
    for (i=0; i<len; i++) fputc (s[i], stderr);
    fputc ('\n', stderr);
  }
  
  if (current_font < 0) {
    ERROR ("do_string:  No font selected");
  }
  p = font_def+current_font;
  for (i=0; i<len; i++) {
    width += tfm_get_fw_width(p->tfm_id, s[i]);
  }
  width = sqxfw (p->size, width);
  switch (p->type) {
  case PHYSICAL:
    dev_set_string (dvi_state.h, -dvi_state.v, s, len,
		    width, p->font_id);
    break;
  case VIRTUAL:
    dvi_push();
    for (i=0; i<len; i++) {
      dvi_set (s[i]);
    }
    dvi_pop();
  }
  dvi_state.h += width;
}

void dvi_set (SIGNED_QUAD ch)
{
  mpt_t width;
  struct font_def *p;
  unsigned char lch = (unsigned char) ch;
  if (current_font < 0) {
    ERROR ("dvi_set:  No font selected");
  }
  /* The division by dvi2pts seems strange since we actually know the
     "dvi" size of the fonts contained in the DVI file.  In other
     words, we converted from DVI units to pts and back again!
     The problem comes from fonts defined in VF files where we don't know the DVI
     size.  It's keeping me sane to keep *point sizes* of *all* fonts in
     the dev.c file and convert them back if necessary */ 
  p = font_def+current_font;
  width = tfm_get_fw_width (p->tfm_id, ch);
  width = sqxfw (p->size, width);
  switch (p->type) {
  case PHYSICAL:
    dev_set_string (dvi_state.h, -dvi_state.v, &lch, 1, width,
		  p->font_id);
    break;
  case VIRTUAL:    
    vf_set_char (ch, p->font_id);
    break;
  }
  dvi_state.h += width;
}

void dvi_put (SIGNED_QUAD ch)
{
  mpt_t width;
  struct font_def *p;
  unsigned char lch = (unsigned char) ch;
  if (current_font < 0) {
    ERROR ("dvi_put:  No font selected");
  }
  p = font_def+current_font;
  switch (p->type) {
  case PHYSICAL:
    width = tfm_get_fw_width (p->tfm_id, ch);
    width = sqxfw (p->size, width);
    dev_set_string (dvi_state.h, -dvi_state.v, &lch, 1, width,
		    p->font_id);
    break;
  case VIRTUAL:    
    vf_set_char (ch, p->font_id);
    break;
  }
  return;
}


void dvi_rule (SIGNED_QUAD width, SIGNED_QUAD height)
{
  do_moveto (dvi_state.h, dvi_state.v);
  dev_rule (dvi_state.h, -dvi_state.v,
	    width, height);
}

static void do_set1(void)
{
  dvi_set (get_unsigned_byte(dvi_file));
}

static void do_setrule(void)
{
  SIGNED_QUAD width, height;
  height = get_signed_quad (dvi_file);
  width = get_signed_quad (dvi_file);
  if (width > 0 && height > 0) {
    dvi_rule (width, height);
  }
  dvi_right (width);
}

static void do_putrule(void)
{
  SIGNED_QUAD width, height;
  height = get_signed_quad (dvi_file);
  width = get_signed_quad (dvi_file);
  if (width > 0 && height > 0) {
    dvi_rule (width, height);
  }
}

static void do_put1(void)
{
  dvi_put (get_unsigned_byte(dvi_file));
}

void dvi_push (void) 
{
  if (debug) {
    fprintf (stderr, "Pushing onto stack of depth %d\n",
	     dvi_stack_depth);
  }
  dvi_stack[dvi_stack_depth++] = dvi_state;
}

void dvi_pop (void)
{
  if (debug) {
    fprintf (stderr, "Popping off stack of depth %d\n",
	     dvi_stack_depth);
  }
  if (dvi_stack_depth > 0) {
    dvi_state = dvi_stack[--dvi_stack_depth];
  } else
    ERROR ("dvi_pop: Tried to pop an empty stack");
  do_moveto (dvi_state.h, dvi_state.v);
}


static void do_right1(void)
{
  dvi_right (get_signed_byte(dvi_file));
}

static void do_right2(void)
{
  dvi_right (get_signed_pair(dvi_file));
}

static void do_right3(void)
{
  dvi_right (get_signed_triple(dvi_file));
}

static void do_right4(void)
{
  dvi_right (get_signed_quad(dvi_file));
}

void dvi_w (SIGNED_QUAD ch)
{
  dvi_state.w = ch;
  dvi_right (ch);
}

void dvi_w0(void)
{
  dvi_right (dvi_state.w);
}

static void do_w1(void)
{
  dvi_w (get_signed_byte(dvi_file));
}

static void do_w2(void)
{
  dvi_w (get_signed_pair(dvi_file));
}

static void do_w3(void)
{
  dvi_w (get_signed_triple(dvi_file));
}

static void do_w4(void)
{
  dvi_w (get_signed_quad(dvi_file));
}

void dvi_x (SIGNED_QUAD ch)
{
  dvi_state.x = ch;
  dvi_right (ch);
}

void dvi_x0(void)
{
  dvi_right (dvi_state.x);
}

static void do_x1(void)
{
  dvi_x (get_signed_byte(dvi_file));
}

static void do_x2(void)
{
  dvi_x (get_signed_pair(dvi_file));
}

static void do_x3(void)
{
  dvi_x (get_signed_triple(dvi_file));
}

static void do_x4(void)
{
  dvi_x (get_signed_quad(dvi_file));
}

static void do_down1(void)
{
  dvi_down (get_signed_byte(dvi_file));
}

static void do_down2(void)
{
  dvi_down (get_signed_pair(dvi_file));
}

static void do_down3(void)
{
  dvi_down (get_signed_triple(dvi_file));
}

static void do_down4(void)
{
  dvi_down (get_signed_quad(dvi_file));
}

void dvi_y (SIGNED_QUAD ch)
{
  dvi_state.y = ch;
  dvi_down (ch);
}

void dvi_y0(void)
{
  dvi_down (dvi_state.y);
}

static void do_y1(void)
{
  dvi_y (get_signed_byte(dvi_file));
}

static void do_y2(void)
{
  dvi_y (get_signed_pair(dvi_file));
}

static void do_y3(void)
{
  dvi_y (get_signed_triple(dvi_file));
}

static void do_y4(void)
{
  dvi_y (get_signed_quad(dvi_file));
}

void dvi_z (SIGNED_QUAD ch)
{
  dvi_state.z = ch;
  dvi_down (ch);
}

void dvi_z0(void)
{
  dvi_down (dvi_state.z);
}

static void do_z1(void)
{
  dvi_z (get_signed_byte(dvi_file));
}

static void do_z2(void)
{
  dvi_z (get_signed_pair(dvi_file));
}

static void do_z3(void)
{
  dvi_z (get_signed_triple(dvi_file));
}

static void do_z4(void)
{
  dvi_z (get_signed_quad(dvi_file));
}

static void do_fntdef(void)
{
  int area_len, name_len, i;
  get_signed_quad(dvi_file);
  get_signed_quad(dvi_file);
  get_signed_quad(dvi_file);
  area_len = get_unsigned_byte(dvi_file);
  name_len = get_unsigned_byte(dvi_file);
  for (i=0; i<area_len+name_len; i++) {
    get_unsigned_byte (dvi_file);
  }
}

static void do_fntdef1(void)
{
  get_unsigned_byte(dvi_file);
  do_fntdef();
}

static void do_fntdef2(void)
{
  get_unsigned_pair(dvi_file);
  do_fntdef();
}

static void do_fntdef3(void)
{
  get_unsigned_triple(dvi_file);
  do_fntdef();
}

static void do_fntdef4(void)
{
  get_signed_quad(dvi_file);
  do_fntdef();
}


void dvi_set_font (int font_id)
{
  current_font = font_id;
}

static void do_fnt (SIGNED_QUAD font_id)
{
  int i;
  for (i=0; i<numfonts; i++) {
    if (font_def[i].source == DVI && font_def[i].tex_id == font_id) break;
  }
  if (i == numfonts) {
    fprintf (stderr, "fontid: %ld\n", font_id);
    ERROR ("dvi_do_fnt:  Tried to select a font that hasn't been defined");
  }
  current_font = i;
}

static void do_fnt1(void)
{
  SIGNED_QUAD font;
  font = get_unsigned_byte(dvi_file);
  do_fnt(font);
}

static void do_fnt2(void)
{
  SIGNED_QUAD font;
  font = get_unsigned_pair(dvi_file);
  do_fnt(font);
}

static void do_fnt3(void)
{
  SIGNED_QUAD font;
  font = get_unsigned_triple(dvi_file);
  do_fnt(font);
}

static void do_fnt4(void)
{
  SIGNED_QUAD font;
  font = get_signed_quad(dvi_file);
  do_fnt(font);
}

static void do_xxx(UNSIGNED_QUAD size) 
{
  UNSIGNED_QUAD i;
  Ubyte *buffer;
  buffer = NEW (size+1, Ubyte);
  for (i=0; i<size; i++) {
    buffer[i] = get_unsigned_byte(dvi_file);
  }
  if (debug)
    fprintf (stderr, "Special: %s\n", buffer);
  dev_do_special (buffer, size, dvi_dev_xpos(), dvi_dev_ypos());
  RELEASE (buffer);
}

static void do_xxx1(void)
{
  SIGNED_QUAD size;
  if (debug)
    fprintf (stderr, "(xxx1)");
  size = get_unsigned_byte(dvi_file);
  do_xxx(size);
}

static void do_xxx2(void)
{
  SIGNED_QUAD size;
  size = get_unsigned_pair(dvi_file);
  do_xxx(size);
}

static void do_xxx3(void)
{
  SIGNED_QUAD size;
  size = get_unsigned_triple(dvi_file);
  do_xxx(size);
}

static void do_xxx4(void)
{
  SIGNED_QUAD size;
  size = get_unsigned_quad(dvi_file);
  do_xxx(size);
}

static void do_bop(void)
{
  int i;
  if (processing_page) 
    ERROR ("dvi_do_bop:  Got a bop inthe middle of a page");
  /* For now, ignore TeX's count registers */
  for (i=0; i<10; i++) {
    get_signed_quad (dvi_file);
  }
/*  Ignore previous page pointer since we have already saved this
    information */
  get_signed_quad (dvi_file);
  clear_state();
  processing_page = 1;
  dev_bop();
}

static void do_eop(void)
{
  processing_page = 0;
  if (dvi_stack_depth != 0) {
    ERROR ("do_eop:  stack_depth is not zero at end of page");
  }
  dev_eop();
}

#define S_BUFFER_SIZE 1024
static unsigned char s_buffer[S_BUFFER_SIZE];
static s_len = 0;

void dvi_do_page(int n)  /* Most of the work of actually interpreting
			    the dvi file is here. */
{
  unsigned char opcode;
  /* Position to beginning of page */
  if (debug) fprintf (stderr, "Seeking to page %d @ %ld\n", n,
			  page_loc[n]);
  seek_absolute (dvi_file, page_loc[n]);
  dvi_stack_depth = 0;
  while (1) {
    /* The most like opcodes are individual setchars.  These are
       buffered for speed */
    s_len = 0;
    while (s_len < S_BUFFER_SIZE && (opcode = fgetc (dvi_file)) <=
	   SET_CHAR_127) {
      s_buffer[s_len++] = opcode;
    }
    if (s_len > 0) {
      do_string (s_buffer, s_len);
    }
    if (s_len == S_BUFFER_SIZE)
      continue;
    /* If we are here, we have an opcode that is something
       other than SET_CHAR */
    if (opcode >= FNT_NUM_0 && opcode <= FNT_NUM_63) {
      do_fnt (opcode - FNT_NUM_0);
      continue;
    }
    switch (opcode)
      {
      case SET1:
	do_set1();
	break;
      case SET2:
      case SET3:
      case SET4:
	ERROR ("dvi_do_page: Multibyte byte character in DVI file.  I can't handle this!");
	break;
      case SET_RULE:
	do_setrule();
	break;
      case PUT1:
	do_put1();
	break;
      case PUT2:
      case PUT3:
      case PUT4:
	ERROR ("dvi_do_page: Multibyte byte character in DVI file.  I can't handle this!");
	break;
      case PUT_RULE:
	do_putrule();
	break;
      case NOP:
	break;
      case BOP:
	do_bop();
	break;
      case EOP:
	do_eop();
	return;
      case PUSH:
	dvi_push();
	break;
      case POP:
	dvi_pop();
	break;
      case RIGHT1:
	do_right1();
	break;
      case RIGHT2:
	do_right2();
	break;
      case RIGHT3:
	do_right3();
	break;
      case RIGHT4:
	do_right4();
	break;
      case W0:
	dvi_w0();
	break;
      case W1:
	do_w1();
	break;
      case W2:
	do_w2();
	break;
      case W3:
	do_w3();
	break;
      case W4:
	do_w4();
	break;
      case X0:
	dvi_x0();
	break;
      case X1:
	do_x1();
	break;
      case X2:
	do_x2();
	break;
      case X3:
	do_x3();
	break;
      case X4:
	do_x4();
	break;
      case DOWN1:
	do_down1();
	break;
      case DOWN2:
	do_down2();
	break;
      case DOWN3:
	do_down3();
	break;
      case DOWN4:
	do_down4();
	break;
      case Y0:
	dvi_y0();
	break;
      case Y1:
	do_y1();
	break;
      case Y2:
	do_y2();
	break;
      case Y3:
	do_y3();
	break;
      case Y4:
	do_y4();
	break;
      case Z0:
	dvi_z0();
	break;
      case Z1:
	do_z1();
	break;
      case Z2:
	do_z2();
	break;
      case Z3:
	do_z3();
	break;
      case Z4:
	do_z4();
	break;
      case FNT1:
	do_fnt1();
	break;
      case FNT2:
	do_fnt2();
	break;
      case FNT3:
	do_fnt3();
	break;
      case FNT4:
	do_fnt4();
	break;
      case XXX1:
	do_xxx1();
	break;
      case XXX2:
	do_xxx2();
	break;
      case XXX3:
	do_xxx3();
	break;
      case XXX4:
	do_xxx4();
	break;
      case FNT_DEF1:
	do_fntdef1();
	break;
      case FNT_DEF2:
	do_fntdef2();
	break;
      case FNT_DEF3:
	do_fntdef3();
	break;
      case FNT_DEF4:
	do_fntdef4();
	break;
      case PRE:
      case POST:
      case POST_POST:
	ERROR("Unexpected preamble or postamble in dvi file");
	break;
      default:
	ERROR("Unexpected opcode or DVI file ended prematurely");
      }
  }
}

error_t dvi_init (char *dvi_filename, char *pdf_filename, double mag, double x_offset, double
		  y_offset)
{
  if (!(dvi_file = fopen (dvi_filename, FOPEN_RBIN_MODE))) {
    ERROR ("dvi_init:  Specified DVI file doesn't exist");
    return (FATAL_ERROR);
  }
  /* DVI files are most easily read backwards by searching
     for post_post and then post opcode */
  find_post ();
  get_dvi_info();
  do_scales(mag);
  dev_init(dvi2pts, x_offset, y_offset);
  get_page_info();
  pdf_doc_init (pdf_filename);
  get_comment();
  get_dvi_fonts();
  clear_state();
  return (NO_ERROR);
}

void dvi_close (void)
{
  int i;
  /* We add comment in dvi_close instead of dvi_init so user has
     a change to overwrite it.  The docinfo dictionary is
     treated as a write-once record */

  /* Do some house cleaning */
  fclose (dvi_file);
  for (i=0; i<numfonts; i++) {
    RELEASE (font_def[i].name);
  }
  RELEASE (page_loc);
  numfonts = 0;
  numpages = 0;
  dvi_file = NULL;
  dev_close_all_fonts();
  vf_close_all_fonts();
  tfm_close_all();
  pdf_doc_close();
}

/* The following are need to implement virtual fonts
   According to documentation, the vf "subroutine"
   must have state pushed and must have
   w,v,y, and z set to zero.  The current font
   is determined by the virtual font header, which
   may be undefined */

  static int saved_dvi_font;

void dvi_vf_init (int dev_font_id)
{
  dvi_push ();
  dvi_state.w = 0; dvi_state.x = 0;
  dvi_state.y = 0; dvi_state.z = 0;
  saved_dvi_font = current_font;
  current_font = dev_font_id;
}

/* After VF subroutine is finished, we simply pop the DVI stack */
void dvi_vf_finish (void)
{
  dvi_pop();
  current_font = saved_dvi_font;
}

